# MD013 - Keep lines short for better readability

## What this rule does

Checks that lines don't exceed a maximum length to ensure your content is easy to read on all devices.

## Why this matters

- **Improves readability**: Shorter lines are easier to scan and understand quickly
- **Works everywhere**: Content displays properly on mobile devices, terminals, and narrow windows
- **Better for code reviews**: Side-by-side comparisons work better with reasonable line lengths
- **Accessibility**: Screen readers and assistive technologies handle shorter lines more effectively

## Examples

<!-- rumdl-disable MD013 -->

### ✅ Correct

```markdown
This line is a reasonable length that's easy to read
and displays well on all devices.

Even with links, you can keep lines manageable by using
[reference-style links][1] instead of inline URLs.

[1]: https://example.com/very-long-url-that-would-make-the-line-too-long
```

### ❌ Incorrect

```markdown
This is an extremely long line that goes on and on and makes it difficult to read the content, especially on mobile devices or when viewing files in split-screen editors or during code reviews where horizontal space is limited.
```

### 🔧 Fixed

```markdown
This is a line that has been wrapped to stay within
the maximum length, making it much easier to read
and work with in various contexts.
```

<!-- rumdl-enable MD013 -->

## Configuration

```toml
[MD013]
line-length = 100  # Maximum characters per line (default: 80)
code-blocks = false  # Don't check code blocks (default: true)
tables = false  # Don't check tables (default: false)
headings = true  # Check headings (default: true)
paragraphs = true  # Check paragraph/regular text (default: true)
strict = false  # Disables exceptions for URLs, etc. (default: false)
reflow = false  # Enable automatic text reflow/wrapping (default: false)
reflow-mode = "default"  # Reflow mode: "default", "normalize", or "sentence-per-line" (default: "default")
```

### Configuration options explained

- `line-length`: The maximum number of characters allowed per line (set to `0` to disable all line length checks)
- `code-blocks`: Whether to check line length in code blocks (default: `true`)
- `tables`: Whether to check line length in tables (default: `false`)
- `headings`: Whether to check line length in headings (default: `true`)
- `paragraphs`: Whether to check line length in regular text/paragraphs (default: `true`). When false, `line-length` is still used for reflow but no warnings are reported
- `strict`: When true, disables exceptions for URLs and other special content (default: `false`)
- `reflow`: When true, enables automatic text reflow to wrap long lines intelligently (default: `false`)
- `reflow-mode`: Controls how text is reflowed when `reflow` is true (default: `"default"`, see Reflow Modes section below)

## Automatic fixes

When `reflow` is set to `true`, this rule can automatically wrap long lines while preserving Markdown formatting:

- Intelligently breaks lines at appropriate points
- Preserves bold, italic, links, code spans, and other Markdown elements
- Maintains proper list continuation indentation
- Preserves hard line breaks (two trailing spaces)
- Does not wrap code blocks, tables, headings, or reference definitions

### Reflow Modes

The `reflow-mode` option controls how text is reformatted when `reflow` is true:

#### `default` mode

Standard text wrapping that breaks lines at word boundaries to fit within the configured line length.

```toml
[MD013]
line-length = 80
reflow = true
reflow-mode = "default"
```

#### `normalize` mode

Normalizes all paragraph text to consistently wrap at the configured line length, removing irregular line breaks.

```toml
[MD013]
line-length = 80
reflow = true
reflow-mode = "normalize"
```

This mode is useful for standardizing documents with inconsistent line wrapping.

#### `sentence-per-line` mode

Enforces one sentence per line, making diffs cleaner and easier to review. This mode:

- Detects sentence boundaries (periods, exclamation marks, question marks)
- Handles common abbreviations (Mr., Dr., etc.) without breaking
- Preserves decimal numbers and ellipses
- Works with markdown formatting

```toml
[MD013]
line-length = 80
reflow = true
reflow-mode = "sentence-per-line"
```

Example transformation:

```markdown
# Before
This is the first sentence. This is the second sentence. And this is the third.

# After
This is the first sentence.
This is the second sentence.
And this is the third.
```

This mode is particularly useful for:

- Technical documentation where each sentence often contains a single concept
- Documents maintained in version control where sentence-level diffs are clearer
- Collaborative writing where different authors work on different sentences

#### Sentence-per-line without line length warnings

If you want to use sentence-per-line mode for formatting but don't want warnings about long sentences, you can disable paragraph checking while keeping code blocks and tables checked:

```toml
[MD013]
line-length = 80
paragraphs = false  # Don't warn about long paragraphs
code-blocks = true  # Still check code blocks
tables = true  # Still check tables
reflow = true
reflow-mode = "sentence-per-line"
```

This configuration is useful when:

- You want automatic sentence-per-line formatting without validation noise
- You care about line length in code blocks and tables but not in regular text
- You're using semantic line breaks where sentence length is determined by content, not arbitrary limits

#### Disabling all line length checks

If you want to completely disable all line length checking (for paragraphs, headings, code blocks, and tables), set `line-length` to `0`:

```toml
[MD013]
line-length = 0  # Disable all line length checks
reflow = true
reflow-mode = "sentence-per-line"
```

When `line-length` is set to `0`, no line length warnings will be reported for any content type. This is particularly useful for:

- Projects using semantic line breaks where lines break at logical boundaries regardless of length
- Sentence-per-line workflows where line length is not a concern
- Documentation where long lines (e.g., URLs, technical terms) are unavoidable and acceptable

**Note**: With `line-length: 0` and `reflow-mode: "sentence-per-line"`, rumdl will:

- Split multiple sentences that share a line into separate lines
- Join single sentences that span multiple lines into one line (since there's no length constraint)
- Provide consistent "one sentence = one line" formatting throughout your document

### Example with automatic reflow

```toml
[MD013]
line-length = 80
reflow = true
```

With this configuration, long lines will be automatically wrapped to fit within 80 characters while maintaining proper Markdown formatting.

**Note**: When `reflow` is `false` (default), automatic fixes are not available and you'll need to manually wrap long lines.

## Learn more

- [Line length best practices](https://en.wikipedia.org/wiki/Line_length)
- [Readability and line length](https://baymard.com/blog/line-length-readability)

## Related rules

- [MD009](md009.md): Remove trailing spaces at line ends
- [MD010](md010.md): Use spaces instead of tabs
- [MD047](md047.md): End files with a single newline
