# sphinx conf
# scikit-plots documentation build configuration file, created by
# sphinx-quickstart on August 16th, 2024 20:37.

# Authors: The scikit-plots developers
# SPDX-License-Identifier: BSD-3-Clause

# pylint: disable=C0413
# pylint: disable=import-error
# pylint: disable=wrong-import-order
# pylint: disable=line-too-long

"""
scikit-plots documentation build config file, created by sphinx-quickstart.

This file is executed with the current directory set to its containing dir
by ``execfile()``, e.g. the working directory will be scikit-plots/docs.
Ensure that all specified paths relative to the docs directory are made
absolute by using ``os.path.abspath``.

Note that not all possible configuration values are present in this
autogenerated file.

All configuration values have a default; values that are commented out
serve to show the default.

See: https://www.sphinx-doc.org/en/master/usage/configuration.html
for more details on configuring the documentation build.
"""

# %%
#
##########################################################################
## Imports
##########################################################################
# Python's standard library
import os
import sys

import datetime
import importlib
import shutil
import textwrap
import warnings
from pathlib import Path

import jinja2

# Set up sphinx
from sphinx.util import logging

# If extensions (or modules to document with autodoc) are in another directory,
# add these directories to sys.path here. If the directory is relative to the
# documentation root, use os.path.abspath to make it absolute, like shown here.
# sys.path.insert(0, os.path.join(os.path.dirname(__file__), "_sphinx_ext/sklearn_ext"))
sys.path.insert(0, os.path.abspath("."))

logger = logging.getLogger(__name__)

try:
    # Set the backend of matplotlib to prevent build errors.
    import matplotlib

    # Use Non-GUI non-interactive backend before pyplot is imported
    matplotlib.use("Agg")

    # Configure plotly to integrate its output into the HTML pages generated by
    # sphinx-gallery.
    import plotly.io as pio

    pio.renderers.default = "sphinx_gallery"
except ImportError:
    # Make it possible to render the doc when not running the examples
    # that need plotly.
    pass

# Use datetime.UTC if available (Python 3.11+), else fallback to datetime.timezone.utc
UTC = getattr(datetime, "UTC", datetime.timezone.utc)  # datetime.datetime.now(tz=UTC)

# %%
#
##########################################################################
## Project information
# https://www.sphinx-doc.org/en/master/usage/configuration.html#project-information
##########################################################################

# import requests
# _metadata_info = requests.get(
#     "https://pypi.org/pypi/scikit-plots/json", timeout=10
# ).json()["info"]
# _last_version = _metadata_info["version"]

# General information about the project.
project = "scikit-plots"
# author = u'Reiichiro S. Nakano'
author = "scikit-plots developers"
# A copyright statement. Permitted styles are as follows,
# where ‘YYYY’ represents a four-digit year.
# The project_copyright alias.
# copyright = u'2017, Reiichiro S. Nakano'
copyright = f"2024 - {datetime.datetime.now().year} {author} (BSD-3 Clause License)"

# %%
#
##########################################################################
## version
##########################################################################

import switcher

switcher.main()  # switcher.json

# Import scikitplot information.
import scikitplot as sp

# from sklearn.externals._packaging.version import parse
from scikitplot.externals._packaging.version import parse

# Possible Doc Version Syntax expected after releases:
# 0.4.0+git.20250114.96321ef       : <Version('0.4.0+git.20250114.96321ef')>
# 0.4.0.post0+git.20250114.96321ef : <Version('0.4.0.post0+git.20250114.96321ef')>
# 0.5.dev0+git.20250114.96321ef    : <Version('0.5.dev0+git.20250114.96321ef')>
_version_raw = sp.version.full_version
## Version should follow PEP440
_version_parsed = parse(_version_raw)  # <Version('0.5.dev0+git.20250114.96321ef')>

# Short X.Y version
# The version info for the project you're documenting, acts as replacement for
# |version| and |release|, also used in various other places throughout the
# built documents.
# https://www.sphinx-doc.org/en/master/usage/configuration.html#project-tags
# https://www.sphinx-doc.org/en/master/usage/restructuredtext/roles.html#substitutions
# The short X.Y version.
# Base version: X.Y.Z
# (this is the main version number, excluding pre-release and post-release tags)
# _version_parsed.base_version = '0.4.0' or '0.5'
# _version_parsed.release      = (0, 4, 0) or (0, 5)
version = ".".join(_version_parsed.base_version.split(".")[:2])  # '0.4' or '0.5'
print("Major.Minor Version:", version)

# Full version
# The full version, including alpha/beta/rc tags.
# Removes post from release name
# _is_postrelease = _version_parsed.is_postrelease
if _version_parsed.is_postrelease:
    release = _version_parsed.base_version  # '0.4.0'
else:
    # for dev sp.version.full_version or sp.version.short_version
    release = _version_raw.split("+")[0].strip()  # without git section, If any

_is_devrelease = _version_parsed.is_devrelease

# debug that building expected version
logger.info("scikit-plots raw          : %s " % _version_raw)
logger.info("scikit-plots version      : %s " % version)
logger.info("scikit-plots release      : %s " % release)
logger.info("scikit-plots is_devrelease: %s " % _is_devrelease)
if release is None:
    raise ValueError(
        f"Ill-formed version: {_version_raw!r}. Version should follow PEP440"
    )

# Release mode enables optimizations and other related options.
# is_release_build = tags.has('release')

# %%
#
##########################################################################
## binder_branch = gh_branch
##########################################################################

# binder_branch
if _is_devrelease:
    gh_branch = "main"
else:
    major, minor = version.split(".")
    gh_branch = f"maintenance/{major}.{minor}.X"

logger.info(f"gh_branch: {gh_branch}")

# %%
#
##########################################################################
## General configuration
# https://www.sphinx-doc.org/en/master/usage/configuration.html#general-configuration
##########################################################################

# If your documentation needs a minimal Sphinx version, state it here.
# needs_sphinx = '1.0'

# Add any Sphinx extension module names here, as strings. They can be
# extensions coming with Sphinx (named 'sphinx.ext.*') or your custom
# ones.
#
# Note on Extension Order:
# -------------------------
# The order of extensions in the `extensions` list may affect the behavior of Sphinx.
# - Extensions that modify or depend on others should be listed after the ones they depend on.
# - Built-in Sphinx extensions (e.g., 'sphinx.ext.*') should generally come first.
# - Custom or third-party extensions should be added after all core extensions.
# - If you encounter issues, consider changing the order to resolve potential conflicts.
extensions = [
    # Core extensions
    #
    "sphinx.ext.autodoc",  # Include documentation from docstrings
    "sphinx.ext.autosummary",  # Generate autodoc summaries
    "numpydoc",  # Support for NumPy-style docstrings, Needs to be loaded *after* autodoc.
    "sphinx.ext.napoleon",  # Support for NumPy and Google style docstrings
    #
    # Built-in extensions (load early)
    #
    "sphinx.ext.ifconfig",  # Include content based on configuration
    "sphinx.ext.extlinks",  # Markup to shorten external links by extlinks
    "sphinx.ext.intersphinx",  # Link to other projects’ documentation
    "sphinx.ext.linkcode",  # Add external links to source code
    # 'sphinx.ext.viewcode',          # Add links to highlighted source code
    # "sphinx.ext.graphviz",          # Add Graphviz graphs
    "sphinx.ext.imgconverter",  # A reference image converter using Imagemagick
    "sphinx.ext.inheritance_diagram",  # Include inheritance diagrams
    # "sphinx.ext.todo",              # Support for todo items
    # "sphinx.ext.autosectionlabel",  # Allow reference sections using its title
    # "sphinx.ext.duration",          # Measure durations of Sphinx processing
    # "sphinx.ext.coverage",          # Collect doc coverage stats
    # "sphinx.ext.githubpages",       # Publish HTML docs in GitHub Pages
    "sphinx.ext.doctest",  # Test snippets in the documentation
    #
    # Matplotlib extensions (load after built-ins)
    #
    "matplotlib.sphinxext.figmpl_directive",
    "matplotlib.sphinxext.mathmpl",
    "matplotlib.sphinxext.plot_directive",  # Integrate Matplotlib plots into documentation.
    # 'matplotlib.sphinxext.roles',  # required matplotlib 3.9.1
    "scikitplot.externals._sphinxext.roles",  # If matplotlib 3.9.1
    # IPython extensions (load after built-ins and matplotlib)
    "IPython.sphinxext.ipython_console_highlighting",
    "IPython.sphinxext.ipython_directive",
    #
    # Third-party extensions (load after built-ins and matplotlib/IPython)
    #
    # https://sphinx-extensions.readthedocs.io/en/latest/
    # https://sphinx-extensions.readthedocs.io/en/latest/sphinx-prompt.html
    # "sphinx-prompt",  # Add prompts and outputs to your documentation
    "sphinx_prompt",  # Add prompts and outputs to your documentation
    "sphinxext.opengraph",  # Add OpenGraph metadata for better sharing of documentation.
    "sphinx_copybutton",  # Add a "copy" button to code blocks in the documentation
    "sphinx_design",  # Add design components and elements to documentation.
    "sphinx_tabs.tabs",  # Tabbed content extension
    # 'nbsphinx',                     # to publish Jupyter notebooks as documentation pages.
    # 'myst_parser',                  # Markdown parser (.md files) with Sphinx
    # gallery extensions (these affect layout and content presentation)
    "sphinx_gallery.gen_gallery",  # Generate galleries of example scripts and figures.
    "sphinxcontrib.sass",  # Support for SASS stylesheets in Sphinx documentation.
    "sphinxcontrib.inkscapeconverter",  # Convert SVGs created by Inkscape.
    #
    # Custom extensions (these should be placed last to avoid conflicts)
    # See _sphinx_ext/
    #
    # local matplotlib: Custom extensions
    "_sphinx_ext.mpl_ext.redirect_from",
    "_sphinx_ext.mpl_ext.github",
    # local sklearn: Custom extensions
    "_sphinx_ext.sklearn_ext.allow_nan_estimators",  # Custom extension for handling NaN values in estimators.
    "_sphinx_ext.sklearn_ext.autoshortsummary",  # Custom extension for generating short summaries.
    "_sphinx_ext.sklearn_ext.doi_role",  # Custom extension for handling DOI references.
    "_sphinx_ext.sklearn_ext.dropdown_anchors",  # Custom extension for dropdown navigation anchors.
    "_sphinx_ext.sklearn_ext.override_pst_pagetoc",  # Custom extension for overriding page TOC in certain cases.
    "_sphinx_ext.sklearn_ext.sphinx_issues",  # Custom extension for managing and displaying issues.
    "_sphinx_ext.sklearn_ext.move_gallery_links",  # Custom extension for rearranging gallery links.
    # sp: Custom extensions
    "_sphinx_ext.skplt_ext.url_extension",  # URL, REPLite extension
    "_sphinx_ext.skplt_ext.version_info_extension",  # version_info_extension
    # Tags and other utility extensions (load last if they depend on others)
    # "sphinx_remove_toctrees",       # Remove certain TOC trees from specific documentation pages.
    "sphinx_tags",  # Needs to be loaded *after* autodoc.
    "_sphinx_ext.sklearn_ext.search_filter",  # Custom extension
    "_sphinx_ext.sklearn_ext.add_js_css_files",  # Custom extension
]

# %%
#
##########################################################################
## Add extension mathjax or imgmath for maths
##########################################################################

# For maths, use mathjax by default and svg if NO_MATHJAX env variable is set
# (useful for viewing the doc offline)
if os.environ.get("NO_MATHJAX"):
    extensions.append("sphinx.ext.imgmath")
    imgmath_image_format = "svg"
    mathjax_path = ""
else:
    # Render mathematical expressions using MathJax.
    extensions.append("sphinx.ext.mathjax")
    mathjax_path = "https://cdn.jsdelivr.net/npm/mathjax@3/es5/tex-chtml.js"

# %%
#
##########################################################################
## Add extension jupyterlite_sphinx and jupyter_sphinx
## https://sphinx-gallery.github.io/stable/configuration.html#generate-jupyterlite-links-for-gallery-notebooks-experimental
##########################################################################

try:
    import jupyter_sphinx  # noqa: F401  # pylint: disable=W0611

    extensions.append("jupyter_sphinx")

    import jupyterlite_sphinx  # noqa: F401  # pylint: disable=W0611

    extensions.append("jupyterlite_sphinx")
    with_jupyterlite = True
except ImportError:
    # In some cases we don't want to require jupyterlite_sphinx to be installed,
    # e.g. the doc-min-dependencies build
    warnings.warn(
        "jupyterlite_sphinx is not installed, you need to install it "
        "if you want JupyterLite links to appear in each example"
    )
    with_jupyterlite = False

# %%
#
##########################################################################
## check dependencies
##########################################################################


def _check_dependencies():
    module_names = {
        **{e: e.split(".")[0] for e in extensions},
        # Explicitly list deps that are not extensions, or whose PyPI package
        # name does not match the (toplevel) module name.
        "colorspacious": "colorspacious",
        "sphinxcontrib.inkscapeconverter": "sphinxcontrib-svg2pdfconverter",
        "pydata_sphinx_theme": "pydata_sphinx_theme",
        # "mpl_sphinx_theme": 'mpl_sphinx_theme',
    }
    for module_name in module_names:
        try:
            # module = __import__(module_name)
            module = importlib.import_module(module_name)
            print(module)
        except Exception as e:
            raise ImportError(
                "The following dependencies are missing to build the "
                f"documentation: {module_name}"
            ) from e

    # debug sphinx-pydata-theme and mpl-theme-version
    import pydata_sphinx_theme

    print(f"pydata sphinx theme : {pydata_sphinx_theme.__version__}")
    # import mpl_sphinx_theme
    # print(f"mpl sphinx theme: {mpl_sphinx_theme.__version__}")

    if shutil.which("dot") is None:
        raise OSError(
            "No binary named dot - graphviz must be installed to build the "
            "documentation"
        )
    # if shutil.which('latex') is None:
    #     raise OSError(
    #         "No binary named latex - a LaTeX distribution must be installed to build "
    #         "the documentation")


_check_dependencies()

# %%
#
##########################################################################
## READ_THE_DOCS configuration
##########################################################################

# -- Sitemap ---------------------------------------------------------
# ReadTheDocs has its own way of generating sitemaps, etc.
# if not os.environ.get("READTHEDOCS"):
#     extensions += ["sphinx_sitemap"]
#     html_baseurl = os.environ.get("SITEMAP_URL_BASE", "http://127.0.0.1:8000/")
#     sitemap_locales = [None]
#     sitemap_url_scheme = "{link}"

# -- Build the latest JS for local preview -----------------------------
# import subprocess import

# -- Run JS Build if needed ------------------------------------------
# Note: this will be a one off run on RTD but may be run multiple times locally
# during development and testing
# if os.environ.get('READ_THE_DOCS'):
#     ## setup the build environment on RTD
#     node_modules_bin = f'{path_root}/node_modules/.bin/';
#     os.environ['PATH'] = f'{node_modules_bin}:' + os.environ["PATH"]
#     subprocess.run(["npm", "install", "jsdoc", "typedoc"], cwd=path_root)
#     print("Local `thebe` not found, building...")
#     subprocess.run(["npm", "install"], cwd=path_root)
#     subprocess.run(["npm", "run", "build"], cwd=path_root)
#     if not Path("_static/lib").exists():
#         print("Local `thebe` not found, building...")
#         subprocess.run(["npm", "install"], cwd=path_root)
#         subprocess.run(["npm", "run", "build"], cwd=path_root)

# %%
#
##########################################################################
## configuration
##########################################################################

## .today() Equivalent to datetime.datetime.now() with no timezone info, Local or tz-aware.
# now_utc = datetime.datetime.now(datetime.timezone.utc)
now_utc = datetime.datetime.today()  # Local time, naive

## Human-readable formats (e.g., "April 14, 2025 11:30 UTC")
_human_today_fmt = "%B %d, %Y %H:%M UTC"
_human_utc = now_utc.strftime(_human_today_fmt)
print("Human-readable UTC     :", _human_utc)

## ISO 8601 formats (e.g., "2025-04-14T11:30Z", "2025-04-14T11:30+03:00")
_iso_today_fmt = "%Y-%m-%dT%H:%MZ"
_iso_utc = now_utc.strftime(_iso_today_fmt)
print("ISO 8601 UTC           :", _iso_utc)

# There are two options for replacing |today|: either, you set today to some
# non-false value, then it is used:
# today = ''
# Else, today_fmt is used as the format for a strftime call.
# today_fmt = '%B %d, %Y'
#
# Option 2: Use today's date in the specified format
# %B → Full month name (e.g., March)
# %d → Day of the month (e.g., 30)
# %Y → Full year (e.g., 2025)
# %H → Hour (24-hour format)
# %M → Minute
# today_fmt = "%B %d, %Y %H:%M"
# today = datetime.datetime.today().strftime(today_fmt)

today_fmt = _human_today_fmt
today = _human_utc

copyright += f" {_version_raw} {_iso_utc}"

# %%
#
##########################################################################
## Options for highlighting
##########################################################################

# The name of the Pygments (syntax highlighting) style to use.
# pygments_style = "sphinx"

# %%
#
##########################################################################
## Options for HTTP requests
##########################################################################

# Use a browser-like user agent to avoid some "403 Client Error: Forbidden for
# url" errors. This is taken from the variable navigator.userAgent inside a
# browser console.
user_agent = (
    "Mozilla/5.0 (X11; Ubuntu; Linux x86_64; rv:100.0) Gecko/20100101 Firefox/100.0"
)

##########################################################################
## Options for internationalisation
##########################################################################

# The language for content autogenerated by Sphinx. Refer to documentation
# for a list of supported languages.
#
# This is also used if you do content translation via gettext catalogs.
# Usually you set "language" from the command line for these cases.
language = "en"  # default

##########################################################################
## Options for markup
##########################################################################

# https://www.sphinx-doc.org/en/master/usage/restructuredtext/basics.html#roles
# The reST default role (used for this markup: `text`) for all docs.
# default_role = None
# documenting code snippets or textual data that should appear exactly as written, use this.
# Docutils supports the following roles:
#   - 'emphasis' – equivalent of *emphasis*
#   - 'strong' – equivalent of **strong**
#   - 'literal' – equivalent of ``literal``
#   - 'subscript' – subscript text
#   - 'superscript' – superscript text
#   - 'title-reference' – for titles of books, periodicals, and other materials
#
# https://www.sphinx-doc.org/en/master/usage/restructuredtext/roles.html#roles
# want references to Python objects (like classes or functions), then use this role.
# default_role = 'obj'  # 'any' 'py:obj'
default_role = "literal"

# If true, keep warnings as "system message" paragraphs in the built documents.
# keep_warnings = False

# If true, sectionauthor and moduleauthor directives will be shown in the
# output. They are ignored by default.
# show_authors = False

# rst_prolog for docs text substitution
# will be included at the beginning of every source file that is read.
# Purpose:
# Injects reStructuredText substitution definitions into every RST source file.
# This lets you write |project| in any .rst file and have it replaced with the value.
# ✅ This affects RST parsing, and therefore any output format (HTML, PDF via LaTeX, etc.).
# rst_prolog = """
# .. |project| replace:: {project}
# .. include:: /global/roles.rst
# """.format(project=project)
#
# We use |scikit-learn| for classical ML, and |PyTorch| or |TensorFlow| for deep learning.
# To track experiments, `wandb`_ or `MLflow`_ can be used alongside `Optuna`_ for hyperparameter tuning.
# .. |PEP8| replace:: `PEP8 <https://www.python.org/dev/peps/pep-0008>`_
rst_prolog = f"""
.. |psf| replace:: Python Software Foundation
.. |full_version| replace:: {_version_raw}

.. _Python: https://www.python.org/
.. _PEP8: https://peps.python.org/pep-0008/
.. _conda: https://docs.conda.io/en/latest/
.. _pytest: https://docs.pytest.org/en/stable/
.. _NumPy: https://numpy.org/
.. _SciPy: https://scipy.org/
.. _pandas: https://pandas.pydata.org/
.. _Matplotlib: https://matplotlib.org/
.. _seaborn: https://seaborn.pydata.org/
.. _scikit-learn: https://scikit-learn.org/stable/
.. _xgboost: https://xgboost.readthedocs.io/
.. _lightgbm: https://lightgbm.readthedocs.io/
.. _catboost: https://catboost.ai/
.. _tensorflow: https://www.tensorflow.org/
.. _keras: https://keras.io/
.. _pytorch: https://pytorch.org/
.. _jax: https://jax.readthedocs.io/
.. _optuna: https://optuna.org/
.. _mlflow: https://mlflow.org/
.. _transformers: https://huggingface.co/docs/transformers/
.. _datasets: https://huggingface.co/docs/datasets/
.. _skorch: https://skorch.readthedocs.io/
.. _onnx: https://onnx.ai/
.. _wandb: https://wandb.ai/
.. _Weights & Biases: https://wandb.ai/

.. |Python| replace:: Python_
.. |PEP8| replace:: PEP8_
.. |conda| replace:: conda_
.. |pytest| replace:: pytest_
.. |NumPy| replace:: NumPy_
.. |SciPy| replace:: SciPy_
.. |pandas| replace:: pandas_
.. |Matplotlib| replace:: Matplotlib_
.. |seaborn| replace:: seaborn_
.. |scikit-learn| replace:: scikit-learn_
.. |xgboost| replace:: XGBoost_
.. |lightgbm| replace:: LightGBM_
.. |catboost| replace:: CatBoost_
.. |tensorflow| replace:: TensorFlow_
.. |keras| replace:: Keras_
.. |pytorch| replace:: PyTorch_
.. |jax| replace:: JAX_
.. |optuna| replace:: Optuna_
.. |mlflow| replace:: MLflow_
.. |transformers| replace:: 🤗 Transformers_
.. |datasets| replace:: 🤗 Datasets_
.. |skorch| replace:: skorch_
.. |onnx| replace:: ONNX_
.. |wandb| replace:: `Weights & Biases`_

.. |emoji| unicode:: U+1F680
   :trim:
"""

# will be included at the end of every source file that is read.
# rst_epilog = """
# .. |footer_note| replace:: Built with ♥ by SuperTeam.
# """

##########################################################################
### Options for object signatures
##########################################################################

# If true, '()' will be appended to :func: etc. cross-reference text.
# add_function_parentheses = True
add_function_parentheses = False

##########################################################################
## Options for source files
##########################################################################

# List of patterns, relative to source directory, that match files and
# directories to ignore when looking for source files.
# This patterns also effect to html_static_path and html_extra_path
# Exclude patterns for files and directories to ignore when looking for source files
exclude_patterns = [
    "**/.git",  # Ignores all .git directories
    "**.ipynb_checkpoints",
    "**/sg_execution_times.rst",
    "build",  # Ignore the _build directory where generated files are stored
    "_build",  # Ignore the _build directory where generated files are stored
    "_templates",
    "Thumbs.db",  # Ignore Thumbs.db (used by Windows Explorer)
    ".DS_Store",  # Ignore .DS_Store (used by macOS Finder)
    "venv",  # Ignore virtual environment directory (if applicable)
    "env",  # Ignore virtual environment directory (if applicable)
]

# The master toctree document.
# Added in version 4.0: The root_doc alias.
master_doc = "index"

# The encoding of source files.
# source_encoding = 'utf-8-sig'  # paste Unicode (⚠️, ✅, 🐢, ⚡️ etc.) directly in the .rst file. ✅ ✅ ❌ ⚠️ 🐢 ⚡ 🐍 🪶 🧪 🚀 ⏱️ 💡 ⚖️ 🧪 🧠 🚫

# The suffix(es) of source filenames.
# You can specify multiple suffix as a list of string:
# source_suffix = '.rst'
# source_suffix = {
#     '.rst': 'restructuredtext',
#     '.md': 'markdown',
#     '.txt': 'markdown',
# }

##########################################################################
## Options for templating
##########################################################################

# Add any paths that contain templates here, relative to this directory.
templates_path = [
    "_templates",
    # 'templates',
]

##########################################################################
## Builder options
# https://www.sphinx-doc.org/en/master/usage/configuration.html#builder-options
##########################################################################

##########################################################################
## Options for HTML output
##########################################################################

# The theme to use for HTML and HTML Help pages.  Major themes that come with
# Sphinx are currently 'default' and 'sphinxdoc'.
#
# https://sphinx-themes.org/sample-sites/default-alabaster/
# html_theme = "alabaster"  # default
# https://sphinx-themes.org/sample-sites/sphinx-rtd-theme/
# html_theme = "sphinx_rtd_theme"  # yellowbrick
# https://sphinx-themes.org/sample-sites/pydata-sphinx-theme/
# https://pydata-sphinx-theme.readthedocs.io/en/latest/examples/gallery.html

# https://pydata-sphinx-theme.readthedocs.io/en/stable/examples/gallery.html#other-projects-using-this-theme
# html_theme = "furo"
html_theme = "pydata_sphinx_theme"  # scikit-learn

# Theme options are theme-specific and customize the look and feel of a theme
# further.  For a list of options available for each theme, see the
# documentation.
html_theme_options = {
    # -- General configuration ------------------------------------------------
    "logo": {
        "alt_text": "scikit-plots homepage",
        "image_relative": "logos/scikit-plots-logo-small.png",
        "image_light": "logos/scikit-plots-logo-small.png",
        "image_dark": "logos/scikit-plots-logo-small.png",
    },
    # -- Search and Edit ------------------------------------------------------
    "search_bar_text": "Search the docs ...",
    "use_edit_page_button": True,
    # -- Appearance Settings --------------------------------------------------
    "pygments_light_style": "tango",
    "pygments_dark_style": "monokai",
    # If the version compares greater than the preferred version
    # (or if the version match contains the strings "dev", "rc" or "pre"),
    # the announcement will say they are viewing an unstable development version instead.
    "show_version_warning_banner": True,  # ("dev" in _version_raw) or _is_devrelease,
    "surface_warnings": True,
    # The switcher requires a JSON file with the list of documentation versions, which
    # is generated by the script `build_tools/circle/list_versions.py` and placed under
    # the `js/` static directory; it will then be copied to the `_static` directory in
    # the built documentation
    # https://pydata-sphinx-theme.readthedocs.io/en/latest/user_guide/version-dropdown.html#configure-switcher-json-url
    "switcher": {
        # Add a unique query to the switcher.json url.  This will be ignored by
        # the server, but will be used as part of the key for caching by browsers
        # so when we do a new meso release the switcher will update "promptly" on
        # the stable and devdocs.
        # "json_url": "https://scikit-plots.github.io/dev/_static/versions.json",
        "json_url": "https://scikit-plots.github.io/dev/_static/switcher.json",
        "version_match": release,  # without git section choose by release
    },
    # check_switcher may be set to False if docbuild pipeline fails. See
    # https://pydata-sphinx-theme.readthedocs.io/en/stable/user_guide/version-dropdown.html#configure-switcher-json-url
    "check_switcher": True,
    # -- External and Icon Links ----------------------------------------------
    "external_links": [],
    "icon_links_label": "Icon Links",
    "icon_links": [
        {
            "name": "GitHub",
            "url": "https://github.com/scikit-plots/scikit-plots",
            # "icon": "fa-brands fa-github",
            "icon": "fa-brands fa-square-github",
            "type": "fontawesome",
        },
        {
            "name": "PyPI",
            "url": "https://pypi.org/project/scikit-plots",
            "icon": "fa-brands fa-python",
            "type": "fontawesome",
        },
    ],
    # alternative way to set twitter and github header icons
    # "github_url": "https://github.com/scikit-plots/scikit-plots",
    # "repository_url": "https://github.com/jupyter/jupyter-sphinx",
    # "use_repository_button": True,
    # "repository_branch": "main",
    # "use_issues_button": True,
    # "use_fullscreen_button": False,
    # -- Header and Footer Settings -------------------------------------------
    "header_links_before_dropdown": 5,
    "header_dropdown_text": "More",
    "navbar_align": "left",
    "navbar_start": ["navbar-logo"],
    # Note that the alignment of navbar_center is controlled by navbar_align
    "navbar_center": ["navbar-nav"],
    "navbar_end": ["theme-switcher", "version-switcher", "navbar-icon-links"],
    # navbar_persistent is persistent right (even when on mobiles)
    # "navbar_persistent": ["search-button"],
    "content_footer_items": [],
    "footer_start": ["copyright"],
    "footer_center": [],
    "footer_end": [],
    # -- Article and Content Settings -----------------------------------------
    "article_header_start": ["breadcrumbs"],
    "article_header_end": [],
    "article_footer_items": ["prev-next"],
    # -- Navigation Settings --------------------------------------------------
    "sidebar_includehidden": True,
    "collapse_navigation": False,
    "navigation_depth": 2,
    "show_toc_level": 1,
    "show_nav_level": 1,
    # If "prev-next" is included in article_footer_items, then setting show_prev_next
    # to True would repeat prev and next links. See
    # https://github.com/pydata/pydata-sphinx-theme/blob/b731dc230bc26a3d1d1bb039c56c977a9b3d25d8/src/pydata_sphinx_theme/theme/pydata_sphinx_theme/layout.html#L118-L129
    "show_prev_next": False,
    "navigation_with_keys": False,
    # Use html_sidebars that map page patterns to list of sidebar templates
    "primary_sidebar_end": [],
    # When specified as a dictionary, the keys should follow glob-style patterns, as in
    # https://www.sphinx-doc.org/en/master/usage/configuration.html#confval-exclude_patterns
    # In particular, "**" specifies the default for all pages
    # Use :html_theme.sidebar_secondary.remove: for file-wide removal
    # "secondary_sidebar_items": {"**": ["page-toc", "sourcelink"]},
    "secondary_sidebar_items": {
        "**": [
            # "edit-this-page",
            "page-toc",
            "sourcelink",
            # "sg_download_links",
            # "sg_launcher_links",
        ],
    },
    # -- Announcement ---------------------------------------------------------
    # We override the announcement template from pydata-sphinx-theme, where
    # this special value indicates the use of the unreleased banner. If we need
    # an actual announcement, then just place the text here as usual.
    "announcement": None,  # "unreleased" if not is_release_build else None,
    # "analytics": {
    #     "plausible_analytics_domain": "scikit-plots.github.io",
    #     "plausible_analytics_url": "https://views.scientific-python.org/js/script.js",
    # },
}
_html_secondary_sidebars = {
    "introduction/index": [],
    "user_guide/index": [],
    "user_guide/api/index": [],
    "user_guide/decile_wise_perf/index": [],
    "whats_new/index": [],
    "_tags/tagsindex": [],
}
for k, v in _html_secondary_sidebars.items():
    html_theme_options["secondary_sidebar_items"][k] = v

# Add any paths that contain custom themes here, relative to this directory.
# html_theme_path = []

# The name for this set of Sphinx documents.
# "<project> v<release> documentation" by default.
# html_title = 'scikitplot v0.1'

# A shorter title for the navigation bar.  Default is the same as html_title.
# html_short_title = None

# The name of an image file (relative to this directory) to place at the top
# of the sidebar.
# html_logo = None

# The name of an image file (relative to this directory) to use as a favicon of
# the docs.  This file should be a Windows icon file (.ico) being 16x16 or 32x32
# pixels large.
html_favicon = "logos/scikit-plots-favicon.ico"

# Additional CSS files, should be subset of the values of `sass_targets`
html_css_files = [
    "styles/colors.css",
    "styles/custom.css",
]

# Additional JS files
html_js_files = [
    "scripts/dropdown.js",
    "scripts/version-switcher.js",
]

# Add any paths that contain custom static files (such as style sheets) here,
# relative to this directory. They are copied after the builtin static files,
# so a file named "default.css" will overwrite the builtin "default.css".
# Any folder listed here is accessible via `pathto` or use Relative Path
html_static_path = [
    "_static",
    # 'images',
    "css",
    "js",
]

# Add any extra paths that contain custom files (such as robots.txt or
# .htaccess) here, relative to this directory. These files are copied
# directly to the root of the documentation.
# html_extra_path = []

# If not None, a 'Last updated on:' timestamp is inserted at every page
# bottom, using the given strftime format.
# The empty string is equivalent to '%b %d, %Y'.
# html_last_updated_fmt = None

# Custom sidebar templates, maps document names to template names.
# html_sidebars = {}
# Custom sidebar templates, maps document names to template names.
# Workaround for removing the left sidebar on pages without TOC
# A better solution would be to follow the merge of:
# https://github.com/pydata/pydata-sphinx-theme/pull/1682
html_sidebars = {
    # "governance": [],
    "project/glossary/index": [],
    "project/glossary/sklearn_glossary": [],
    "project/faq": [],
    # "roadmap": [],
    # "related_projects": [],
    # "support": [],
    # "teams/about": [],
}

# Additional templates that should be rendered to pages, maps page names to
# template names.
html_additional_pages = {"index": "index.html"}

# If false, no module index is generated.
# html_domain_indices = True
# html_domain_indices = False

# If false, no index is generated.
# html_use_index = True
# html_use_index = False

# If true, the index is split into individual pages for each letter.
# html_split_index = False

# If true, the reST sources are included in the HTML build as _sources/name.
# html_copy_source = True

# If true, links to the reST sources are added to the pages.
# html_show_sourcelink = True

# If true, an OpenSearch description file will be output, and all pages will
# contain a <link> tag referring to it.  The value of this option must be the
# base URL from which the finished HTML is served.
# html_use_opensearch = ''

# If nonempty, this is the file name suffix for HTML files (e.g. ".xhtml").
# html_file_suffix = '.html'

# If true, "(C) Copyright ..." is shown in the HTML footer. Default is True.
# html_show_copyright = True

# If true, "Created using Sphinx" is shown in the HTML footer. Default is True.
# html_show_sphinx = True

# Language to be used for generating the HTML full-text search index.
# Sphinx supports the following languages:
#   'da', 'de', 'en', 'es', 'fi', 'fr', 'h', 'it', 'ja'
#   'nl', 'no', 'pt', 'ro', 'r', 'sv', 'tr', 'zh'
# html_search_language = 'en'

# A dictionary with options for the search language support, empty by default.
# 'ja' uses this config value.
# 'zh' user can custom change `jieba` dictionary path.
# html_search_options = {'type': 'default'}

# The name of a javascript file (relative to the configuration directory) that
# implements a search results scorer. If empty, the default will be used.
# html_search_scorer = 'scorer.js'

##########################################################################
## Options for HTML help output
##########################################################################

# Output file base name for HTML help builder.
# The default is the project name with spaces removed and doc appended.
# htmlhelp_basename = 'scikit-plotsdoc'

##########################################################################
## Options for LaTeX output
##########################################################################

# latex_elements = {
#     # The paper size ('letterpaper' or 'a4paper').
#     'papersize': 'letterpaper',

#     # The font size ('10pt', '11pt' or '12pt').
#     'pointsize': '10pt',

#     # Additional stuff for the LaTeX preamble.
#     'preamble': '',

#     # Latex figure (float) alignment
#     'figure_align': 'htbp',
# }

# Grouping the document tree into LaTeX files. List of tuples.
# latex_documents = [
#     (
#         master_doc,  # source start file
#         "scikitplot.tex",  # target name
#         "{} Documentation".format(project),  # title
#         author,  # author
#         "manual",  # documentclass [howto,manual, or own class]
#     )
# ]

# The name of an image file (relative to this directory) to place at the top of
# the title page.
# latex_logo = None

# Documents to append as an appendix to all manuals.
# latex_appendices = []

# If false, no module index is generated.
# latex_domain_indices = True

# If true, show page references after internal links.
# latex_show_pagerefs = False

# If true, show URL addresses after external links.
# latex_show_urls = False

##########################################################################
## Options for manual page output
##########################################################################

# One entry per manual page. List of tuples
# (source start file, name, description, authors, manual section).
# man_pages = [
#     (
#         master_doc,
#         project,
#         "{} Documentation".format(project),
#         [author],
#         1
#     )
# ]

# If true, show URL addresses after external links.
# man_show_urls = False

##########################################################################
## Options for Texinfo output
##########################################################################

# Grouping the document tree into Texinfo files. List of tuples
# (source start file, target name, title, author,
#  dir menu entry, description, category)
# texinfo_documents = [
#     (
#         master_doc,
#         "scikitplot",
#         "{} Documentation".format(project),
#         author,
#         "scikitplot",
#         "Machine Learning Visualization with Python",
#         "Scientific Visualization",
#     )
# ]

# Documents to append as an appendix to all manuals.
# texinfo_appendices = []

# If false, no module index is generated.
# texinfo_domain_indices = True

# If true, do not generate a @detailmenu in the "Top" node's menu.
# texinfo_no_detailmenu = False

# How to display URL addresses: 'footnote', 'no', or 'inline'.
# texinfo_show_urls = 'footnote'

##########################################################################
## Options for the linkcheck builder
##########################################################################

# Config for linkcheck that checks the documentation for broken links
# ignore all links in 'whats_new' to avoid doing many github requests and
# hitting the github rate threshold that makes linkcheck take a lot of time
linkcheck_exclude_documents = [r"whats_new/.*"]

# default timeout to make some sites links fail faster
linkcheck_timeout = 10

# Allow redirects from doi.org
linkcheck_allowed_redirects = {r"https://doi.org/.+": r".*"}
linkcheck_ignore = [
    # ignore links to local html files e.g. in image directive :target: field
    r"^..?/",
    # ignore links to specific pdf pages because linkcheck does not handle them
    # ('utf-8' codec can't decode byte error)
    # r"http://www.utstat.toronto.edu/~rsalakhu/sta4273/notes/Lecture2.pdf#page=.*",
    # (
    #     "https://www.fordfoundation.org/media/2976/roads-and-bridges"
    #     "-the-unseen-labor-behind-our-digital-infrastructure.pdf#page=.*"
    # ),
    # links falsely flagged as broken
    # (
    #     "https://www.researchgate.net/publication/"
    #     "233096619_A_Dendrite_Method_for_Cluster_Analysis"
    # ),
    # (
    #     "https://www.researchgate.net/publication/221114584_Random_Fourier"
    #     "_Approximations_for_Skewed_Multiplicative_Histogram_Kernels"
    # ),
    # (
    #     "https://www.researchgate.net/publication/4974606_"
    #     "Hedonic_housing_prices_and_the_demand_for_clean_air"
    # ),
    # (
    #     "https://www.researchgate.net/profile/Anh-Huy-Phan/publication/220241471_Fast_"
    #     "Local_Algorithms_for_Large_Scale_Nonnegative_Matrix_and_Tensor_Factorizations"
    # ),
    # "https://doi.org/10.13140/RG.2.2.35280.02565",
    # (
    #     "https://www.microsoft.com/en-us/research/uploads/prod/2006/01/"
    #     "Bishop-Pattern-Recognition-and-Machine-Learning-2006.pdf"
    # ),
    # "https://www.microsoft.com/en-us/research/wp-content/uploads/2016/02/tr-99-87.pdf",
    # "https://microsoft.com/",
    # "https://www.jstor.org/stable/2984099",
    # "https://stat.uw.edu/sites/default/files/files/reports/2000/tr371.pdf",
    # # Broken links from testimonials
    # "http://www.bestofmedia.com",
    # "http://www.data-publica.com/",
    # "https://livelovely.com",
    # "https://www.mars.com/global",
    # "https://www.yhat.com",
    # # Ignore some dynamically created anchors. See
    # # https://github.com/sphinx-doc/sphinx/issues/9016 for more details about
    # # the github example
    # r"https://github.com/conda-forge/miniforge#miniforge",
    # r"https://github.com/joblib/threadpoolctl/"
    # "#setting-the-maximum-size-of-thread-pools",
    # r"https://stackoverflow.com/questions/5836335/"
    # "consistently-create-same-random-numpy-array/5837352#comment6712034_5837352",
]

# Use Github token from environment variable to avoid Github rate limits when
# checking Github links
github_token = os.getenv("GITHUB_TOKEN")

if github_token is None:
    linkcheck_request_headers = {}
else:
    linkcheck_request_headers = {
        "https://github.com/": {"Authorization": f"token {github_token}"},
    }

##########################################################################
## Domain options
# https://www.sphinx-doc.org/en/master/usage/configuration.html#domain-options
##########################################################################

##########################################################################
## Options for the Python domain
##########################################################################

# If true, the current module name will be prepended to all description
# unit titles (such as .. function::).
# add_module_names = True

# A list of ignored prefixes for module index sorting.
# modindex_common_prefix = []

##########################################################################
## Extension options
# https://www.sphinx-doc.org/en/master/usage/configuration.html#extension-options
##########################################################################

##########################################################################
## Extension: autosectionlabel
##########################################################################

autosectionlabel_prefix_document = True  # default: False

##########################################################################
## Extension: *.rst sphinx-remove-toctrees, or use :orphan:
##########################################################################

# remove_from_toctrees = [
#   'README',
#   'community/sklearn_glossary',
#   # 'devel/index.rst',
# ]

##########################################################################
## Extension: Sphinx tags configuration
##########################################################################

tags_create_tags = True
# tags_output_dir = "_tags"  # default: "_tags"
tags_overview_title = "Tags overview"  # default: "Tags overview"
tags_extension = ["rst"]  # default: ["rst"]
tags_intro_text = "Tags:"  # default: "Tags:"
tags_page_title = "My tags"  # default: "My tags"
tags_page_header = "With this tag"  # default: "With this tag"
tags_index_head = "Tags"  # default: "Tags"

tags_create_badges = True  # default: False
tags_badge_colors = {
    "t0": None,  # white-plain
    "model-type:*": "primary",  # seablue
    "model-workflow:*": "secondary",  # purple
    "plot-type:*": "success",  # green
    "level:*": "info",  # blue
    "event-handling:*": "warning",  # orange
    "domain:*": "danger",  # red
    "*": "light",  # Used as a default value # gray
    "purpose:*": "dark",  # black
}

##########################################################################
## Extension: intersphinx
##########################################################################

# intersphinx configuration
# Locations of objects.inv files for intersphinx extension that auto-links
# to external api docs.
intersphinx_mapping = {
    # Build
    "python": (f"https://docs.python.org/{sys.version_info.major}", None),
    "pip": ("https://pip.pypa.io/en/stable/", None),
    "meson-python": ("https://mesonbuild.com/meson-python/", None),
    "setuptools": ("https://setuptools.pypa.io/en/stable", None),
    "packaging": ("https://packaging.python.org/en/latest", None),
    "virtualenv": ("https://virtualenv.pypa.io/en/stable", None),
    "IPython": ("https://ipython.readthedocs.io/en/stable/", None),
    "ipykernel": ("https://ipykernel.readthedocs.io/en/latest/", None),
    # Test
    "pytest": ("https://pytest.org/en/stable/", None),
    "tox": ("https://tox.wiki/en/stable", None),
    # Docs
    "numpydoc": ("https://numpydoc.readthedocs.io/en/latest", None),
    # Data Manuplation
    "numpy": ("https://numpy.org/doc/stable", None),
    "scipy": ("https://docs.scipy.org/doc/scipy/", None),
    "pandas": ("https://pandas.pydata.org/pandas-docs/stable/", None),
    "dateutil": ("https://dateutil.readthedocs.io/en/stable/", None),
    "xarray": ("https://docs.xarray.dev/en/stable/", None),
    # Visualization
    "matplotlib": ("https://matplotlib.org/stable/", None),
    "cycler": ("https://matplotlib.org/cycler/", None),
    "seaborn": ("https://seaborn.pydata.org/", None),
    "astropy": ("https://docs.astropy.org/en/stable/", None),
    # ML
    "statsmodels": ("https://www.statsmodels.org/stable/", None),
    "sklearn": ("https://scikit-learn.org/stable/", None),
    # DL
    "tensorflow": (
        "https://www.tensorflow.org/api_docs/python",
        "https://raw.githubusercontent.com/GPflow/tensorflow-intersphinx/master/tf2_py_objects.inv",
    ),
    # 'keras': ('https://keras.io/api/', None),  # Add Keras API documentation
    "pytorch": (
        "https://pytorch.org/docs/stable/",
        None,
        # "https://github.com/pytorch/docs/blob/main/stable/objects.inv",
    ),
    "datasets": ("https://huggingface.co/docs/datasets/main/en/", None),
    "transformers": ("https://huggingface.co/docs/transformers/main/en/", None),
    # image
    "aggdraw": ("https://aggdraw.readthedocs.io/en/stable/", None),
    "pillow": ("https://pillow.readthedocs.io/en/stable/", None),
    "imageio": ("https://imageio.readthedocs.io/en/stable", None),
    "skimage": ("https://scikit-image.org/docs/stable", None),
    # Misc
    "joblib": ("https://joblib.readthedocs.io/en/latest/", None),
    "skops": ("https://skops.readthedocs.io/en/stable/", None),
    "pluggy": ("https://pluggy.readthedocs.io/en/stable", None),
    "flask": ("https://flask.palletsprojects.com/en/stable/", None),
}

##########################################################################
## Extension: linkcode
##########################################################################

from _sphinx_ext.skplt_ext.github_link import make_linkcode_resolve

# The following is used by sphinx.ext.linkcode to provide links to github
# Create the partial (as you probably already do)
_linkcode_resolver_partial = make_linkcode_resolve(
    "scikitplot",
    (
        "https://github.com/scikit-plots/"
        "scikit-plots/blob/{revision}/"
        "{package}/{path}#L{lineno}"
    ),
)


# linkcode_resolve = _linkcode_resolver_partial
# Wrap in a real function so Sphinx doesn't complain
def linkcode_resolve(domain, info):
    """
    Resolve the GitHub source code URL for a given Python object.

    This function is used by `sphinx.ext.linkcode` to provide
    "View source" links in the documentation.

    Parameters
    ----------
    domain : str
        The documentation domain (e.g., "py" for Python).
    info : dict
        A dictionary containing information about the object.
        Typically includes keys like "module" and "fullname".

    Returns
    -------
    url : str or None
        The URL to the source code on GitHub, or None if it cannot be resolved.
    """
    return _linkcode_resolver_partial(domain, info)


##########################################################################
## Extension: copybutton
##########################################################################

# Specify how to identify the prompt when copying code snippets
copybutton_prompt_text = r">>> | \.\.\."
copybutton_prompt_is_regexp = True
copybutton_exclude = "style"

##########################################################################
## Extension: todo
##########################################################################

# If true, `todo` and `todoList` produce output, else they produce nothing.
todo_include_todos = False

##########################################################################
## Extension: autodoc
##########################################################################

## option without needing to specify it each time.
# autodoc_default_options = {
#     # 'members': True,               # Include member functions and attributes
#     # 'member-order': 'bysource',
#     # 'undoc-members': True,
#     # 'inherited-members': True,     # Show inherited members
#     # 'exclude-members': '__weakref__',
#     # 'special-members': '__init__',
#     # 'show-inheritance': True,      # Show inheritance by default for all classes
#     # 'private-members': True,
# }

## Show both class-level and __init__ docstrings together
## Specifies what content to display for class-level docstrings.
# 'both': Both the class and __init__ docstrings are combined. 'class' (default)
# autoclass_content = 'class'
## Use detailed parameter descriptions instead of inline type hints
## 'description': Type hints are displayed in the parameter and return value descriptions
## instead of the signature. "signature" (default)
autodoc_typehints = "description"
## Automatically inherit docstrings from parent classes
## False: Inherited members will appear undocumented unless
## the subclass explicitly provides a docstring. True (default)
# autodoc_inherit_docstrings = True
## Preserve the source code order for members
## Determines the order in which members (methods, attributes)
## are displayed in the documentation. 'bysource' (default)
# autodoc_member_order = "bysource"
## Mock external imports to avoid build errors
## aren’t available during the documentation build,
## this option mocks those imports to avoid build failures.
autodoc_mock_imports = [
    # "scikitplot",
    "passlib",
    "numpy",
    "pandas",
    "scipy",
    "sklearn",
    "xgboost",
    "catboost",
    "tensorflow",
    "pytorch",
    "flask_sqlalchemy",
    "mlflow",
]

##########################################################################
## Extension: autosummary
##########################################################################

# Enable generate autosummary even if no references
# Automatically generate summary tables and individual rst files
autosummary_generate = True

# maps functions with a class name that is indistinguishable when case is
# ignore to another filename
# Map fully qualified names of modules, classes, or functions to custom stub filenames.
# This is useful to resolve naming conflicts or control the generated filenames for `autosummary`.
autosummary_filename_map = {
    # "sklearn.cluster.dbscan": "dbscan-function",
    "scikitplot.modelplotpy.ModelPlotPy": "modelplotpy-cls",
    "scikitplot.sp_logging.critical": "critical-func",
    "scikitplot.sp_logging.debug": "debug-func",
    "scikitplot.sp_logging.error": "error-func",
    "scikitplot.sp_logging.fatal": "fatal-func",
    "scikitplot.sp_logging.warn": "warn-func",
    "scikitplot.sp_logging.warning": "warning-func",
}

##########################################################################
## Extension: numpydoc
##########################################################################

# Autodoc requires numpy to skip class members otherwise we get an exception:
# toctree contains reference to nonexisting document
# See: https://github.com/phn/pytpm/issues/3#issuecomment-12133978
# We do not need the table of class members because `sphinxext/override_pst_pagetoc.py`
# will show them in the secondary sidebar
# only the class-level docstring is shown, and methods/attributes will not appear
numpydoc_show_class_members = False  # You don't need to write :members:, If True

# members (methods/attributes) inherited from parent classes are shown in the class docs automatically
# only show methods defined directly in that class, and will skip inherited ones
numpydoc_show_inherited_class_members = False  # :inherited-members:

# We want in-page toc of class members instead of a separate page for each entry
# Each class method or attribute appears on a separate page in the HTML documentation.
numpydoc_class_members_toctree = False  # create separate pages for methods

# Produce `plot::` directives for examples that contain `import matplotlib` or
# `from matplotlib import`.
# Any plots generated (e.g., via matplotlib.pyplot.plot) are automatically rendered and shown in the built docs.
numpydoc_use_plots = True

##########################################################################
## Extension: napoleon
##########################################################################

# napoleon_google_docstring = False  # if you use numpy-style only
# napoleon_numpy_docstring = True

##########################################################################
## Extension: sphinxext_plot_directive_api
##########################################################################

# Options for the `::plot` directive:
# https://matplotlib.org/stable/api/sphinxext_plot_directive_api.html
plot_formats = [
    # ('png', 100),
    # ('hires.png', 350),
    "png",
    # "svg",
    # "pdf",
]
# By default, include the source code generating plots in documentation
plot_include_source = True
# Whether to show links to the files in HTML.
plot_html_show_formats = True
# Whether to show a link to the source in HTML.
plot_html_show_source_link = True

# A dictionary containing any non-standard rcParams that should be applied before each plot.
plot_rcparams = {
    "figure.figsize": (10, 4),
    "figure.dpi": 101,
}

# Code that should be executed before each plot.
# Default also includes a numpy and matplotlib import
# 🧹 Reset-related:
plot_pre_code = """
import gc; gc.collect()
import importlib
import sys

# List of modules to reload to reset state
for mod in ['matplotlib.pyplot', 'scikitplot']:
    if mod in sys.modules:
        importlib.reload(sys.modules[mod])
"""

##########################################################################
### Extension: matplotlib GitHub extension
##########################################################################

github_project_url = "https://github.com/scikit-plots/scikit-plots"

##########################################################################
## Additional Options for HTML output
##########################################################################

# Compile scss files into css files using sphinxcontrib-sass
sass_src_dir, sass_out_dir = "scss", "css/styles"
sass_targets = {
    f"{file.stem}.scss": f"{file.stem}.css"
    for file in Path(sass_src_dir).glob("*.scss")
}

# --------------------------------------------------------------------
# HTML context populated by the extension
# --------------------------------------------------------------------

# html_context for UI and templates
# Adds Configuration Variables Directly from conf.py into templates
# Pass the `repl_url` to the HTML context so it can be used in templates
# These will be populated by the extension
# Purpose:
# Used to pass Python variables into HTML templates (Jinja2-based)
# — like layout.html, page.html, or theme-specific templates
# such as alabaster, sphinx_rtd_theme, etc.
# ✅ This only affects HTML output, not RST content or jinja template.
html_context = {
    # "github_user": "scikit-plots",
    # "github_repo": "scikit-plots",
    # "github_version": "main",
    # "doc_path": "docs",
    # 'repl_url': repl_url,  # populated by the extension
    # "shell": {
    #   "intro": [
    #     {"title": "Title 1", "text": "Description 1"},
    #     {"title": "Title 2", "text": "Description 2"},
    #     # Add more entries as needed
    #   ]
    # },
    # version_info_extension.py 'releaselevel'
    # See https://github.com/scikit-learn/scikit-learn/pull/22550
    "is_devrelease": _is_devrelease,
}

# redirects dictionary maps from old links to new links
old_links_dict = {
    # "documentation": "index",
}
html_context["redirects"] = old_links_dict
for old_link in old_links_dict:
    html_additional_pages[old_link] = "redirects.html"

##########################################################################
## Extension: Jupyter Sphinx options
##########################################################################

jupyter_sphinx_thebelab_config = {
    # Binder integration
    "binderOptions": {
        "org": "scikit-plots",
        "repo": "scikit-plots",
        "branch": (
            gh_branch
        ),  # Can be any branch, tag, or commit hash. Use a branch that hosts your docs.
        "dependencies": "./binder/requirements.txt",
    }
}

##########################################################################
## Extension: jupyterlite-sphinx options
##########################################################################

# jupyterlite_build_command_options = {
#   "XeusAddon.environment_file": "environment.yml",  # default
# }
# jupyterlite_config = "jupyterlite_config.json"
# jupyterlite_contents = "./jupyterlite_contents"
# jupyterlite_bind_ipynb_suffix = False
# strip_tagged_cells = True

##########################################################################
## Extension: sphinx_gallery_conf
##########################################################################


# Import your custom sorting classes (uncomment if needed)
# from _sphinx_ext.skplt_ext.sg_custom_sorting import (
#   SubSectionTitleOrder, SKExampleTitleSortKey)

from sphinx_gallery.sorting import (
    # ExampleTitleSortKey,
    # ExplicitOrder,
    FileNameSortKey,
    # FileSizeSortKey,
    # FunctionSortKey,
    # NumberOfCodeLinesSortKey,
)

# Paths for examples and gallery directories
examples_dirs = ["../../galleries/examples"]
gallery_dirs = ["auto_examples"]

# Sphinx Gallery Configuration
# https://github.com/sphinx-gallery/sphinx-gallery/blob/master/sphinx_gallery/gen_gallery.py#L81
sphinx_gallery_conf = {
    # Backreferences and linking to function docs
    # Links examples to APIs documentation
    "backreferences_dir": os.path.join("modules", "generated"),
    # Specify the module to document
    "doc_module": ("scikitplot", "sklearn"),
    # Linking to external packages for reference URLs
    "reference_url": {
        "scikitplot": None,  # Will link to local module documentation
        # Example linking to external package docs
        "numpy": "https://numpy.org/doc/stable/",
        "matplotlib": "https://matplotlib.org/stable/",
    },
    "show_memory": False,  # Set to True if memory_profiler is available and needed
    # Directories for input example scripts and generated output galleries
    # Directory where your examples are stored
    "examples_dirs": examples_dirs,  # List of folders with example scripts
    # Directory where the output gallery will be saved
    "gallery_dirs": gallery_dirs,  # List of folders where galleries will be generated
    # "ignore_pattern": r'__init__\\.py',
    # Only run files with filenames matching this pattern (default is ".py")
    # "filename_pattern": r'/plot',  # re.escape(os.sep) + "plot" gives: "/plot" or gives: "\\\\plot"
    # 'filename_pattern': r'^((?!(sg|skip)).)*$',
    # Binder integration
    # https://sphinx-gallery.github.io/stable/configuration.html#generate-binder-links-for-gallery-notebooks-experimental
    "binder": {
        # Required keys
        "org": "scikit-plots",
        "repo": "scikit-plots",
        # Can be any branch, tag, or commit hash. Use a branch that hosts your docs.
        "branch": gh_branch,
        # '<list_of_paths_to_dependency_files>'
        "dependencies": "./binder/requirements.txt",
        "binderhub_url": "https://mybinder.org",  # URL of a binderhub deployment
        # Optional keys
        # Whether Binder links should start Jupyter Lab instead of the Jupyter Notebook interface.
        "use_jupyter_lab": True,
        # Jupyter notebooks for Binder will be copied to this directory (relative to built documentation root).
        # 'notebooks_dir': '<notebooks-directory-name>'
        # A prefix to prepend to any filepaths in Binder links.
        # 'filepath_prefix': '<prefix>'
    },
    # Optional Additional options
    "line_numbers": True,  # sphinx_gallery_line_numbers = True
    "first_notebook_cell": (
        "# This cell is added by Sphinx-Gallery\n"
        "# It can be customized to whatever you like\n"
    ),
    "last_notebook_cell": "# This is the last cell",
    # Optional
    "inspect_global_variables": False,  # Avoid generating too many cross-links
    # <<-- THIS hides config-only comments in the rendered gallery
    "remove_config_comments": True,
    "recommender": {
        "enable": True,
        "n_examples": 4,
        "min_df": 12,
    },
    "reset_modules": (  # reset loaded modules between examples
        "_sphinx_ext.skplt_ext.sg_doc_build.reset_others",
        "matplotlib",
        "seaborn",
        # "scikitplot.reset",  # (sklearn, matplotlib, seaborn, numpy)
    ),
    # By default, Sphinx-Gallery will reset modules before each example is run.
    "reset_modules_order": "both",  # 'before' or 'after'
    # Optionally sort the examples within subsections (uncomment if needed)
    # Optional sorting: sorts subsections based on titles
    # Options: 'NumberOfCodeLinesSortKey' (default), 'FileNameSortKey', 'FileSizeSortKey', 'ExampleTitleSortKey'
    # Use an instance of FileNameSortKey
    "subsection_order": FileNameSortKey(examples_dirs),
    # Options: 'NumberOfCodeLinesSortKey' (default), 'FileNameSortKey', 'FileSizeSortKey', 'ExampleTitleSortKey'
    "within_subsection_order": "FileNameSortKey",
    # Making cell magic executable in notebooks
    # https://sphinx-gallery.github.io/stable/configuration.html#making-cell-magic-executable-in-notebooks
    "promote_jupyter_magic": True,
    # Build examples in parallel
    "parallel": 2,
}
# Add JupyterLite configuration if enabled
if with_jupyterlite:
    sphinx_gallery_conf["jupyterlite"] = {
        "notebook_modification_function": (
            "_sphinx_ext.skplt_ext.sg_doc_build.notebook_modification_function"
        ),
    }

# Secondary sidebar configuration for pages generated by sphinx-gallery

# For the index page of the gallery and each nested section, we hide the secondary
# sidebar by specifying an empty list (no components), because there is no meaningful
# in-page toc for these pages, and they are generated so "sourcelink" is not useful
# either.

# For each example page we keep default ["page-toc", "sourcelink"] specified by the
# "**" key. "page-toc" is wanted for these pages. "sourcelink" is also necessary since
# otherwise the secondary sidebar will degenerate when "page-toc" is empty, and the
# script `sphinxext/move_gallery_links.py` will fail (it assumes the existence of the
# secondary sidebar). The script will remove "sourcelink" in the end.

for examples_dir, gallery_dir in zip(examples_dirs, gallery_dirs):
    html_theme_options["secondary_sidebar_items"][f"{gallery_dir}/index"] = []
    for sub_dir in (Path(examples_dir)).iterdir():
        if sub_dir.is_dir():
            html_theme_options["secondary_sidebar_items"][
                f"{gallery_dir}/{sub_dir.name}/index"
            ] = []

##########################################################################
## Extension: sphinx_gallery_conf
##########################################################################

# Config for sphinxext.opengraph
ogp_site_url = "https://scikit-plots.github.io/stable/"
ogp_image = "https://scikit-plots.github.io/stable/_static/scikit-plots-logo-small.png"
ogp_use_first_image = True
ogp_site_name = "scikit-plots"

##########################################################################
## Extension: _sphinxext
##########################################################################

# Config for sphinx_issues
# we use the issues path for PRs since the issues URL will forward
issues_github_path = "scikit-plots/scikit-plots"

##########################################################################
## Extension: extlinks
##########################################################################

extlinks = {
    "issue": ("https://github.com/scikit-plots/scikit-plots/issues/%s", "issue %s")
}

##########################################################################
## Extension: _sphinx_ext skplt_ext infer_next_release_versions
## Convert .rst.template files to .rst
##########################################################################

# add rst templates kwargs variables
release_versions_rst_templates = [
    (
        "devel/index",  # rst_template_name
        "devel/index",  # rst_target_name
        {  # kwargs
            "is_devrelease": _is_devrelease,
        },
    ),
    (
        "devel/guide_maintainer",  # rst_template_name
        "devel/guide_maintainer",  # rst_target_name
        {},  # kwargs
    ),
    # Add more default templates here
    # (
    #     "min_dependency_table",
    #     "min_dependency_table",
    #     {"dependent_packages": {} },
    # ),
    # (
    #     "min_dependency_substitutions",
    #     "min_dependency_substitutions",
    #     {"dependent_packages": {} },
    # ),
]

##########################################################################
## Extension: _sphinx_ext skplt_ext url_extension
## Convert .rst.template files to .rst
##########################################################################

# If development build, link to local page in the top navbar; otherwise link to the
# development version; see https://github.com/scikit-learn/scikit-learn/pull/22550
development_link = (
    "devel/index"
    if _is_devrelease
    else "https://scikit-plots.github.io/dev/devel/index.html"
)

url_rst_templates = [
    (
        "index",  # rst_template_name
        "index",  # rst_target_name
        {  # kwargs
            "development_link": development_link,
        },
    )
]

##########################################################################
## Convert .rst.template files to .rst
##########################################################################

from apis_reference import APIS_REFERENCE, DEPRECATED_APIS_REFERENCE

# Define the templates and target files for conversion
# Each entry is in the format (template name, file name, kwargs for rendering)
rst_templates: list[tuple[str, str, dict[str, any]]] = [
    (
        "apis/index",
        "apis/index",
        {
            "APIS_REFERENCE": sorted(APIS_REFERENCE.items(), key=lambda x: x[0]),
            "DEPRECATED_APIS_REFERENCE": sorted(
                DEPRECATED_APIS_REFERENCE.items(), key=lambda x: x[0], reverse=True
            ),
        },
    ),
]

# Convert each module APIs reference page
for module in APIS_REFERENCE:
    rst_templates.append(
        (
            "apis/module",
            f"apis/{module}",
            {"module": module, "module_info": APIS_REFERENCE[module]},
        )
    )

# Convert the deprecated APIs reference page (if there exists any)
if DEPRECATED_APIS_REFERENCE:
    rst_templates.append(
        (
            "apis/deprecated",
            "apis/deprecated",
            {
                "DEPRECATED_APIS_REFERENCE": sorted(
                    DEPRECATED_APIS_REFERENCE.items(), key=lambda x: x[0], reverse=True
                )
            },
        )
    )

######################################################################
## jinja2 Template Renderer
## https://jinja.palletsprojects.com/en/stable/templates/#import
## https://ttl255.com/jinja2-tutorial-part-6-include-and-import/
######################################################################

# Step 1: Create a Jinja environment instance
jinja_env = jinja2.Environment(extensions=["jinja2.ext.i18n"])
# Step 2: Register constants and functions globally
jinja_env.globals["imp0rt"] = importlib.import_module  # Make available in all templates

# Load your template from a file
for rst_template_name, rst_target_name, kwargs in rst_templates:
    # Read the corresponding template file into jinja2
    with (Path(f"{rst_template_name}.rst.template")).open("r", encoding="utf-8") as f:
        # t = jinja2.Template(f.read())
        t = jinja_env.from_string(f.read())  # Use from_string to create the template

    # Render the template with kwargs variables and write to the target
    with (Path(f"{rst_target_name}.rst")).open("w", encoding="utf-8") as f:
        f.write(t.render(**kwargs))
