#!/usr/bin/env python
"""Generate requirements/*.txt files from pyproject.toml."""

# Authors: The scikit-plots developers
# SPDX-License-Identifier: BSD-3-Clause

# import os
import sys
import logging
from pathlib import Path
from datetime import datetime

# import pytz

try:  # standard module since Python 3.11
    import tomllib as toml
except (ImportError, ModuleNotFoundError):
    try:  # available for older Python via pip
        import tomli as toml
    except (ImportError, ModuleNotFoundError):
        sys.exit("Please install `tomli` first: `{mamba, pip} install tomli`")

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

script_pth = Path(__file__)
repo_dir = script_pth.parent.parent.parent

# Use logging instead of print
# print(repo_dir)
logger.info(repo_dir)

script_relpth = script_pth.relative_to(repo_dir)

header = [
    # f"# Created at: {time.ctime(time.time())}",
    f"# Created at: {datetime.now()}",
    f"# Generated via {script_relpth.as_posix()}.",
    "# Do not edit this file; modify `pyproject.toml` instead "
    "and run `python tools/generate_requirements.py`.",
]


def generate_requirement_file(name, req_list, *, extra_list=None):
    req_fname = repo_dir / "requirements" / f"{name}.txt"

    # Use Pathlib instead of os.makedirs and os.path.dirname
    # os.makedirs(os.path.dirname(req_fname), exist_ok=True)
    Path(req_fname).parent.mkdir(parents=True, exist_ok=True)

    comment = {
        "scikit-plots[core]": "-r core.txt",
        "scikit-plots[cpu]": "-r cpu.txt",
        "scikit-plots[gpu]": "-r gpu.txt",
        "scikit-plots[legacy]": "-r legacy.txt",
        "scikit-plots[tpu]": "-r tpu.txt",
        "scikit-plots[ci32_requirements]": "-r ci32_requirements.txt",
        # TODO:remove once scikit-umfpack issues are resolved
        "scikit-umfpack": "# scikit-umfpack  # circular dependency issues",
    }

    req_list = [(comment[x] if x in comment else x) for x in req_list]

    ## filter build
    # if name == "build":
    #     req_list = [x for x in req_list if "numpy" not in x]
    #     req_list.append("ninja")

    if extra_list:
        req_list += extra_list

    req_fname.write_text("\n".join(header + req_list) + "\n")
    return req_list


def main():
    pyproject = toml.loads((repo_dir / "pyproject.toml").read_text())

    default = generate_requirement_file(
        "default",
        pyproject["project"]["dependencies"],
        extra_list=["-r legacy.txt"],
    )
    # generate_requirement_file(
    #     "build",
    #     pyproject["build-system"]["requires"],
    #     extra_list=default,
    # )
    for key, opt_list in pyproject["project"]["optional-dependencies"].items():
        generate_requirement_file(key, opt_list)

    # generate requirements/all.txt
    all_path = repo_dir / "requirements" / "all.txt"
    files = ["build", "dev", "test", "doc"]
    reqs = [f"-r {x}.txt" for x in files]
    all_path.write_text("\n".join(header + reqs) + "\n")


if __name__ == "__main__":
    main()
