"""
Django settings for codex project.

Generated by 'django-admin startproject' using Django 3.0.3.

For more information on this file, see
https://docs.djangoproject.com/en/dev/topics/settings/

For the full list of settings and their values, see
https://docs.djangoproject.com/en/dev/ref/settings/
"""

from os import environ
from pathlib import Path
from types import MappingProxyType

from comicbox.config import get_config
from loguru import logger

from codex.settings.hypercorn import load_hypercorn_config
from codex.settings.secret_key import get_secret_key
from codex.settings.timezone import get_time_zone
from codex.settings.whitenoise import immutable_file_test

######################################
# Undocumented Environment Variables #
######################################
# SECURITY WARNING: don't run with debug turned on in production!
FALSY = {None, "", "false", "0", False}


def not_falsy_env(name):
    """Return a boolean environment envs mindful of falsy values."""
    return bool(environ.get(name, "").lower() not in FALSY)


DEBUG = not_falsy_env("DEBUG")
BUILD = not_falsy_env("BUILD")
# Slow query middleware
# limit in seconds
SLOW_QUERY_LIMIT = float(environ.get("CODEX_SLOW_QUERY_LIMIT", "0.5"))
LOG_RESPONSE_TIME = not_falsy_env("CODEX_LOG_RESPONSE_TIME")
# Search indexing memory controls
MMAP_RATIO = int(environ.get("CODEX_MMAP_RATIO", "240"))
WRITER_MEMORY_PERCENT = float(environ.get("CODEX_WRITER_MEMORY_PERCENT", "0.6"))
CPU_MULTIPLIER = float(environ.get("CODEX_CPU_MULTIPLIER", "1.25"))
CHUNK_PER_GB = int(environ.get("CODEX_CHUNK_PER_GB", "250"))
MAX_CHUNK_SIZE = int(environ.get("CODEX_MAX_CHUNK_SIZE", "1000"))
# sqlite parser breaks with more than 1000 variables in a query and
# django only fixes this in the bulk_create & bulk_update functions.
# So for complicated queries I gotta batch them myself. These batch sizes
# are only a proxy for query terms, but it works.
# FILTER_BATCH_SIZE of 990 errors sometimes.
FILTER_BATCH_SIZE = int(environ.get("CODEX_FILTER_BATCH_SIZE", "900"))
LINK_FK_BATCH_SIZE = int(environ.get("CODEX_LINK_FK_BATCH_SIZE", "20000"))
LINK_M2M_BATCH_SIZE = int(environ.get("CODEX_LINK_M2M_BATCH_SIZE", "20000"))
VITE_HOST = environ.get("VITE_HOST")
SEARCH_INDEX_BATCH_SIZE = int(environ.get("CODEX_SEARCH_INDEX_BATCH_SIZE", "10000"))
LOG_RETENTION = environ.get("LOG_RETENTION", "6 months")
MAX_OBJ_PER_PAGE = int(environ.get("CODEX_BROWSER_MAX_OBJ_PER_PAGE", "100"))


####################################
# Documented Environment Variables #
####################################
LOGLEVEL = environ.get("LOGLEVEL", "TRACE" if DEBUG else "INFO")
TZ = environ.get("TIMEZONE", environ.get("TZ"))
CONFIG_PATH = Path(environ.get("CODEX_CONFIG_DIR", Path.cwd() / "config"))
RESET_ADMIN = not_falsy_env("CODEX_RESET_ADMIN")
LOG_DIR = Path(environ.get("CODEX_LOG_DIR", CONFIG_PATH / "logs"))
LOG_TO_CONSOLE = environ.get("CODEX_LOG_TO_CONSOLE") != "0"
LOG_TO_FILE = environ.get("CODEX_LOG_TO_FILE") != "0"
THROTTLE_ANON = int(environ.get("CODEX_THROTTLE_ANON", "0"))
THROTTLE_USER = int(environ.get("CODEX_THROTTLE_USER", "0"))
THROTTLE_OPDS = int(environ.get("CODEX_THROTTLE_OPDS", "0"))
THROTTLE_OPENSEARCH = int(environ.get("CODEX_THROTTLE_OPENSEARCH", "0"))
FIX_FOREIGN_KEYS = not_falsy_env("CODEX_FIX_FOREIGN_KEYS")
INTEGRITY_CHECK = not_falsy_env("CODEX_INTEGRITY_CHECK")
FTS_INTEGRITY_CHECK = not_falsy_env("CODEX_FTS_INTEGRITY_CHECK")
FTS_REBUILD = not_falsy_env("CODEX_FTS_REBUILD")

# Base paths
BASE_DIR = Path(__file__).resolve().parent.parent.parent
CODEX_PATH = BASE_DIR / "codex"
CUSTOM_COVERS_SUBDIR = "custom-covers"
CUSTOM_COVERS_DIR = CONFIG_PATH / CUSTOM_COVERS_SUBDIR
CUSTOM_COVERS_GROUP_DIRS = ("publishers", "imprints", "series", "volumes", "story-arcs")
for group_dir in CUSTOM_COVERS_GROUP_DIRS:
    custom_cover_group_dir = CUSTOM_COVERS_DIR / group_dir
    custom_cover_group_dir.mkdir(exist_ok=True, parents=True)

# SECURITY WARNING: keep the secret key used in production secret!
SECRET_KEY = get_secret_key(CONFIG_PATH)


def _get_logging():
    loggers = {}
    if LOGLEVEL != "TRACE":
        loggers.update(
            {
                "asyncio": {
                    "level": "INFO",
                },
            }
        )
    if not DEBUG:
        loggers.update(
            {
                "urllib3.connectionpool": {"level": "INFO"},
                "watchdog": {
                    "level": "INFO",
                },
                "PIL": {
                    "level": "INFO",
                },
            }
        )
    return {"version": 1, "loggers": loggers}


LOG_PATH = LOG_DIR / "codex.log"
LOG_ROTATION = "10 MB"
LOGGING = _get_logging()

ALLOWED_HOSTS = ["*"]


# Application definition
def _get_installed_apps():
    installed_apps = [
        "django.contrib.auth",
        "django.contrib.contenttypes",
        "django.contrib.sessions",
        "django.contrib.messages",
    ]

    if DEBUG:
        # comes before static apps
        installed_apps += ["nplusone.ext.django", "schema_graph"]

    installed_apps += [
        "whitenoise.runserver_nostatic",
        "django.contrib.staticfiles",
        "rest_framework",
        "rest_registration",
        "corsheaders",
    ]
    if not BUILD:
        installed_apps += [
            "django_vite",
        ]
    installed_apps += [
        "codex",
        "cachalot",
        "drf_spectacular",
    ]
    return tuple(installed_apps)


INSTALLED_APPS = _get_installed_apps()


def _get_middleware():
    middleware = [
        "corsheaders.middleware.CorsMiddleware",
        "django.middleware.security.SecurityMiddleware",
        "whitenoise.middleware.WhiteNoiseMiddleware",
        "django.contrib.sessions.middleware.SessionMiddleware",
        "django.middleware.common.CommonMiddleware",
        "django.middleware.csrf.CsrfViewMiddleware",
        "django.contrib.auth.middleware.AuthenticationMiddleware",
        "django.contrib.messages.middleware.MessageMiddleware",
        "django.middleware.clickjacking.XFrameOptionsMiddleware",
        "codex.middleware.TimezoneMiddleware",
    ]
    if DEBUG:
        middleware += [
            "nplusone.ext.django.NPlusOneMiddleware",
        ]

    if LOG_RESPONSE_TIME:
        middleware += [
            "codex.middleware.LogResponseTimeMiddleware",
        ]
    return tuple(middleware)


MIDDLEWARE = _get_middleware()

if DEBUG:
    NPLUSONE_LOGGER = logger
    NPLUSONE_LOG_LEVEL = "WARNING"

ROOT_URLCONF = "codex.urls.root"

CODEX_TEMPLATES = CODEX_PATH / "templates"
TEMPLATES = [
    {
        "BACKEND": "django.template.backends.django.DjangoTemplates",
        "DIRS": [CODEX_TEMPLATES],
        "APP_DIRS": True,
        "OPTIONS": {
            "context_processors": [
                "django.template.context_processors.debug",
                "django.template.context_processors.request",
                "django.contrib.auth.context_processors.auth",
                "django.contrib.messages.context_processors.messages",
            ],
        },
    },
]

WSGI_APPLICATION = "codex.wsgi.application"


# Database
# https://docs.djangoproject.com/en/dev/ref/settings/#databases

DB_PATH = CONFIG_PATH / "codex.sqlite3"

# Move old DB_PATH
OLD_DB_PATH = CONFIG_PATH / "db.sqlite3"
if not DB_PATH.exists() and OLD_DB_PATH.exists():
    OLD_DB_PATH.rename(DB_PATH)

BACKUP_DB_DIR = CONFIG_PATH / "backups"
BACKUP_DB_PATH = (BACKUP_DB_DIR / DB_PATH.stem).with_suffix(DB_PATH.suffix + ".bak")

DATABASES = {
    "default": {
        "ENGINE": "django.db.backends.sqlite3",
        "NAME": DB_PATH,
        "CONN_MAX_AGE": 600,
        "OPTIONS": {
            "init_command": "PRAGMA journal_mode=wal;",
            "timeout": 120,
        },
    },
}
SESSION_ENGINE = "django.contrib.sessions.backends.cached_db"
# The new DEFAULT_AUTO_FIELD in Django 3.2 is BigAutoField (64 bit),
#   but it can't be auto migrated. Automigration has been punted to
#   Django 4.0 at the earliest:
#   https://code.djangoproject.com/ticket/32674
DEFAULT_AUTO_FIELD = "django.db.models.AutoField"


# Password validation
# https://docs.djangoproject.com/en/3.0/ref/settings/#auth-password-validators

AUTH_PASSWORD_VALIDATORS = [
    {
        "NAME": (
            "django.contrib.auth.password_validation.UserAttributeSimilarityValidator"
        )
    },
    {
        "NAME": "django.contrib.auth.password_validation.MinimumLengthValidator",
        "OPTIONS": {"min_length": 4},
    },
]


# Internationalization
# https://docs.djangoproject.com/en/dev/topics/i18n/
LANGUAGE_CODE = "en-us"
USE_I18N = True
TIME_ZONE = get_time_zone(TZ)

# Hypercorn
HYPERCORN_CONFIG_TOML = CONFIG_PATH / "hypercorn.toml"
HYPERCORN_CONFIG_TOML_DEFAULT = CODEX_PATH / "settings/hypercorn.toml.default"
HYPERCORN_CONFIG = load_hypercorn_config(
    HYPERCORN_CONFIG_TOML, HYPERCORN_CONFIG_TOML_DEFAULT, DEBUG
)
PORT = int(HYPERCORN_CONFIG.bind[0].split(":")[1])


# Static files (CSS, JavaScript, Images)
# https://docs.djangoproject.com/en/dev/howto/static-files/
# WHITENOISE_KEEP_ONLY_HASHED_FILES is still not usable with vite chunking
WHITENOISE_STATIC_PREFIX = "static/"
WHITENOISE_IMMUTABLE_FILE_TEST = immutable_file_test
STATIC_ROOT = CODEX_PATH / "static_root"
ROOT_PATH = HYPERCORN_CONFIG.root_path
STATIC_URL = (
    ROOT_PATH + "/" + WHITENOISE_STATIC_PREFIX
    if ROOT_PATH
    else WHITENOISE_STATIC_PREFIX
)
STORAGES = {
    "staticfiles": {
        "BACKEND": "whitenoise.storage.CompressedManifestStaticFilesStorage"
    }
}
STATICFILES_DIRS = (
    [CODEX_PATH / "static_src", CODEX_PATH / "static_build"] if (DEBUG or BUILD) else []
)
for path in STATICFILES_DIRS:
    path.mkdir(exist_ok=True, parents=True)

SESSION_COOKIE_AGE = 60 * 60 * 24 * 60  # 60 days

# Setup support for proxy headers
USE_X_FORWARDED_HOST = True
SECURE_PROXY_SSL_HEADER = ("HTTP_X_FORWARDED_PROTO", "https")

_THROTTLE_MAP = MappingProxyType(
    {
        "anon": ("rest_framework.throttling.AnonRateThrottle", THROTTLE_ANON),
        "user": ("rest_framework.throttling.UserRateThrottle", THROTTLE_USER),
        "opds": ("rest_framework.throttling.ScopedRateThrottle", THROTTLE_OPDS),
        "opensearch": (
            "rest_framework.throttling.ScopedRateThrottle",
            THROTTLE_OPENSEARCH,
        ),
    }
)
_THROTTLE_CLASSES = set()
_THROTTLE_RATES = {}
for scope, value in _THROTTLE_MAP.items():
    classname, rate_value = value
    if rate_value or classname == "rest_framework.throttling.ScopedRateThrottle":
        _THROTTLE_CLASSES.add(classname)
        rate = f"{rate_value}/min" if value[1] else None
        _THROTTLE_RATES[scope] = rate

REST_FRAMEWORK = {
    "DEFAULT_AUTHENTICATION_CLASSES": (
        "rest_framework.authentication.SessionAuthentication",
    ),
    "DEFAULT_RENDERER_CLASSES": (
        "djangorestframework_camel_case.render.CamelCaseJSONRenderer",
        "djangorestframework_camel_case.render.CamelCaseBrowsableAPIRenderer",
    ),
    "DEFAULT_PARSER_CLASSES": (
        "djangorestframework_camel_case.parser.CamelCaseJSONParser",
        # "djangorestframework_camel_case.parser.CamelCaseFormParser",
        # "djangorestframework_camel_case.parser.CamelCaseMultiPartParser",
    ),
    "DEFAULT_SCHEMA_CLASS": "drf_spectacular.openapi.AutoSchema",
    "EXCEPTION_HANDLER": "codex.views.error.codex_exception_handler",
    "DEFAULT_THROTTLE_CLASSES": tuple(_THROTTLE_CLASSES),
    "DEFAULT_THROTTLE_RATES": _THROTTLE_RATES,
}

REST_REGISTRATION = {
    "REGISTER_VERIFICATION_ENABLED": False,
    "REGISTER_EMAIL_VERIFICATION_ENABLED": False,
    "RESET_PASSWORD_VERIFICATION_ENABLED": False,
    "USER_HIDDEN_FIELDS": (
        # DEFAULT
        "last_login",
        "is_active",
        "user_permissions",
        "groups",
        "date_joined",
        # SHOWN
        # "is_staff", "is_superuser",
        # HIDDEN
        "email",
        "first_name",
        "last_name",
    ),
}


SPECTACULAR_SETTINGS = {
    "TITLE": "Codex API",
    "DESCRIPTION": "Comic Library Browser and Reader",
    "OAS_VERSION": "3.1.0",
    "VERSION": "3.0.0",
    "CONTACT": {
        "name": "Support",
        "url": "https://github.com/ajslater/codex?tab=readme-ov-file#-support",
    },
    "PREPROCESSING_HOOKS": ["codex.urls.spectacular.allow_list"],
    "SERVE_PERMISSIONS": ["rest_framework.permissions.IsAdminUser"],
    "EXTERNAL_DOCS": {
        "url": "https://github.com/ajslater/codex/",
        "description": "Codex Docs",
    },
}

CORS_ALLOW_CREDENTIALS = True

ROOT_CACHE_PATH = CONFIG_PATH / "cache"
DEFAULT_CACHE_PATH = ROOT_CACHE_PATH / "default"
DEFAULT_CACHE_PATH.mkdir(exist_ok=True, parents=True)
CACHES = {
    "default": {
        "BACKEND": "django.core.cache.backends.filebased.FileBasedCache",
        "LOCATION": str(DEFAULT_CACHE_PATH),
    },
}

INTERNAL_IPS = ("127.0.0.1",)

CHANNEL_LAYERS = {
    "default": {"BACKEND": "channels.layers.InMemoryChannelLayer"},
}

if DEBUG and not BUILD:
    import socket

    DEV_SERVER_HOST = VITE_HOST if VITE_HOST else socket.gethostname()
    DJANGO_VITE = {
        "default": {
            "dev_mode": DEBUG,
            "dev_server_host": DEV_SERVER_HOST,
        }
    }

CACHALOT_UNCACHABLE_TABLES = frozenset(
    {"django_migrations", "django_session", "codex_useractive"}
)

COMICBOX_CONFIG = get_config(
    {
        "loglevel": LOGLEVEL,
        "delete_keys": frozenset({"pages", "reprints"}),
    }
)
