"""OAuth routers for Google and GitHub authentication."""

import logging
import time
from urllib.parse import quote

import httpx
from fastapi import APIRouter, Request, Query
from fastapi.responses import RedirectResponse

from api.auth.users import (
    jwt_authentication,
    google_oauth_client,
    github_oauth_client,
    UserManager,
)
from api.auth.database import MongoDBUserDatabase
from api.database.async_mongodb import async_mongodb_adapter
from api.config import settings

logger = logging.getLogger(__name__)


def get_oauth_backend_callback_url(provider: str) -> str:
    """Get backend OAuth callback URL for provider.

    This is where the OAuth provider (Google/GitHub) redirects to after authorization.
    Uses development URL when DEBUG=true, otherwise uses production URL.
    Make sure this URI is registered in the OAuth provider's console.

    Args:
        provider: OAuth provider name (google, github)

    Returns:
        Backend callback URL
    """
    if settings.debug:
        base_url = settings.oauth_backend_callback_url_dev
    else:
        base_url = settings.oauth_backend_callback_url_prod

    callback_url = str(base_url).format(provider=provider)
    logger.info(
        "Using backend OAuth callback URL for %s: %s (debug=%s). "
        "Ensure this URI is registered in %s OAuth console.",
        provider,
        callback_url,
        settings.debug,
        provider.capitalize(),
    )
    return callback_url


def get_oauth_frontend_redirect_url(provider: str) -> str:
    """Get frontend redirect URL for provider.

    This is where the backend redirects to after processing OAuth callback.
    Uses development URL when DEBUG=true, otherwise uses production URL.

    Args:
        provider: OAuth provider name (google, github)

    Returns:
        Frontend redirect URL
    """
    if settings.debug:
        base_url = settings.oauth_frontend_redirect_url_dev
    else:
        base_url = settings.oauth_frontend_redirect_url_prod

    return str(base_url).format(provider=provider)


def create_oauth_router(provider: str, oauth_client):
    """Create OAuth router with custom frontend redirect.

    Args:
        provider: OAuth provider name (google, github)
        oauth_client: OAuth client instance

    Returns:
        APIRouter with OAuth routes
    """
    router = APIRouter()

    @router.get("/callback")
    async def custom_callback(
        code: str = Query(...),
        state: str = Query(None),
        request: Request = None,
    ) -> RedirectResponse:
        """Custom OAuth callback that redirects to frontend with token.
        
        Handles two flows:
        1. Main OAuth flow (login/signup) - no state or state without encoded user_id
        2. V1 OAuth flow (connecting GitHub to existing user) - state contains encoded user_id
        
        Args:
            code: OAuth authorization code
            state: OAuth state parameter (may contain encoded user_id for v1 flow)
            request: FastAPI request object
        """
        try:
            logger.info(
                "OAuth callback received for %s",
                provider,
                extra={"event": "oauth_callback_start", "provider": provider, "has_state": bool(state)},
            )
            
            if state and provider == "github":
                from api.routers.v1.oauth import decode_user_state
                user_id, _ = decode_user_state(state)
                if user_id:
                    logger.info(
                        "Detected v1 OAuth flow for user %s, routing to v1 handler",
                        user_id,
                        extra={"event": "oauth_v1_flow_detected", "provider": provider, "user_id": str(user_id)},
                    )
                    from api.routers.v1.oauth import github_oauth_callback as v1_callback
                    return await v1_callback(code=code, state=state)
            
            redirect_uri = get_oauth_backend_callback_url(provider)
            logger.debug("Getting access token with redirect_uri: %s", redirect_uri)
            
            access_token_response = await oauth_client.get_access_token(code, redirect_uri)
            logger.debug("Access token received for %s", provider)
            
            if provider == "google":
                try:
                    user_info = await oauth_client.get_id_email(access_token_response["access_token"])
                except Exception as google_error:
                    logger.warning("Google People API failed, trying userinfo endpoint: %s", google_error)
                    async with httpx.AsyncClient() as client:
                        response = await client.get(
                            "https://www.googleapis.com/oauth2/v2/userinfo",
                            headers={"Authorization": f"Bearer {access_token_response['access_token']}"},
                        )
                        response.raise_for_status()
                        user_data = response.json()
                        user_id = user_data.get("id")
                        user_email = user_data.get("email")
                        if not user_id or not user_email:
                            raise ValueError("Missing id or email in Google userinfo response") from google_error
                        user_info = (user_id, user_email)
            else:
                user_info = await oauth_client.get_id_email(access_token_response["access_token"])
            
            logger.debug("User info received: %s", user_info)
            
            if isinstance(user_info, tuple) and len(user_info) == 2:
                user_id, user_email = user_info
            else:
                logger.error("Unexpected user_info format from %s: %s", provider, type(user_info))
                raise ValueError(f"Unexpected user_info format: {user_info}")

            await async_mongodb_adapter.connect()
            db = async_mongodb_adapter.get_database()
            collection = db.users
            user_db = MongoDBUserDatabase(collection)
            user_manager = UserManager(user_db)

            expires_at = None
            if "expires_in" in access_token_response:
                expires_at = int(time.time()) + access_token_response["expires_in"]

            logger.debug("Calling oauth_callback for %s with account_id: %s, email: %s", provider, user_id, user_email)
            try:
                user = await user_manager.oauth_callback(
                    oauth_name=provider,
                    access_token=access_token_response["access_token"],
                    account_id=str(user_id),
                    account_email=user_email,
                    expires_at=expires_at,
                    refresh_token=access_token_response.get("refresh_token"),
                    request=request,
                )
            except ValueError as e:
                error_msg = str(e)
                if "invitation" in error_msg.lower() or "admin" in error_msg.lower():
                    logger.warning(
                        "Admin signup blocked for %s: %s",
                        user_email,
                        error_msg,
                    )
                    frontend_url = get_oauth_frontend_redirect_url(provider)
                    error_url = f"{frontend_url}?error=admin_signup_blocked&message={error_msg.replace(' ', '%20')}"
                    return RedirectResponse(url=error_url)
                raise
            logger.info(
                "SECURITY: User created/updated via OAuth: %s (email: %s)",
                user.id,
                user.email,
                extra={
                    "event": "oauth_user_created",
                    "provider": provider,
                    "user_id": str(user.id),
                    "email": user_email,
                },
            )

            from api.auth.admin import is_internal_admin_domain
            from api.database.mongodb import mongodb_manager
            from bson import ObjectId
            from datetime import datetime

            if is_internal_admin_domain(user_email):
                db = mongodb_manager.get_database()
                user_doc = db.users.find_one({"_id": ObjectId(str(user.id))})
                if user_doc:
                    update_fields = {}
                    if user_doc.get("plan") != "architect":
                        update_fields["plan"] = "architect"
                        logger.info(
                            "Updating plan to architect for internal admin: %s",
                            user_email,
                        )
                    if not user_doc.get("is_super_admin") and not user_doc.get("admin_role"):
                        if not db.users.find_one({"is_super_admin": True}):
                            update_fields["is_super_admin"] = True
                            update_fields["admin_role"] = "super_admin"
                            update_fields["admin_status"] = "active"
                            update_fields["admin_permissions"] = ["*"]
                            logger.info(
                                "Promoting to super admin (first admin): %s",
                                user_email,
                            )
                    
                    if update_fields:
                        update_fields["updated_at"] = datetime.utcnow()
                        db.users.update_one(
                            {"_id": ObjectId(str(user.id))},
                            {"$set": update_fields},
                        )
                        user_dict = user.to_dict()
                        user_dict.update(update_fields)
                        from api.auth.users import User
                        user = User.from_dict(user_dict)
                        logger.info(
                            "User updated with new fields: %s",
                            list(update_fields.keys()),
                        )

            jwt_strategy = jwt_authentication.get_strategy()
            jwt_token = await jwt_strategy.write_token(user)

            if not jwt_token or not isinstance(jwt_token, str):
                logger.error("Invalid token generated for OAuth user: %s", user.id)
                raise ValueError("Failed to generate authentication token")
            
            import jwt
            from datetime import datetime
            try:
                decoded = jwt.decode(jwt_token, options={"verify_signature": False})
                exp_timestamp = decoded.get("exp")
                if exp_timestamp:
                    exp_time = datetime.fromtimestamp(exp_timestamp)
                    now = datetime.utcnow()
                    lifetime_minutes = (exp_time - now).total_seconds() / 60
                    logger.info(
                        "JWT token created: exp=%s (UTC), lifetime_minutes=%.1f, configured_expire_minutes=%d",
                        exp_time,
                        lifetime_minutes,
                        settings.access_token_expire_minutes,
                    )
            except Exception as decode_error:
                logger.debug("Could not decode JWT for expiration logging: %s", decode_error)

            from api.models.audit_log import AuditEventType, AuditLogSeverity
            from api.services.audit_log_service import audit_log_service

            ip_address = request.client.host if request.client else None
            user_agent = request.headers.get("user-agent") if request else None

            from datetime import datetime
            is_new_user = not user.is_verified
            
            audit_log_service.log_event(
                event_type=AuditEventType.AUTHENTICATION_SUCCESS,
                severity=AuditLogSeverity.LOW,
                message=f"User {user.id} authenticated via OAuth ({provider})",
                success=True,
                user_id=str(user.id),
                organization_id=str(user.organization_id) if user.organization_id else None,
                ip_address=ip_address,
                user_agent=user_agent,
                endpoint=f"/oauth/{provider}/callback",
                method="GET",
                details={
                    "auth_method": "oauth",
                    "provider": provider,
                    "is_signup": is_new_user,
                    "token_size": len(jwt_token),
                },
                compliance_tags=["PCI-DSS-10", "SOC2"],
            )

            frontend_url = get_oauth_frontend_redirect_url(provider)
            
            from api.utils.cookies import set_auth_cookie_safe
            from urllib.parse import quote
            
            cookie_success = False
            if settings.debug:
                sync_url = f"{frontend_url}?token={quote(jwt_token)}"
                logger.info(
                    "Development mode: Redirecting to frontend with token for sync (user: %s)",
                    user.id,
                )
                response = RedirectResponse(url=sync_url)
                cookie_success = True
            else:
                response = RedirectResponse(url=frontend_url)
                cookie_success, _ = set_auth_cookie_safe(
                    response, jwt_token, fallback_to_header=False
                )
                
                if cookie_success:
                    logger.info(
                        "Cookie set successfully for OAuth callback (user: %s)",
                        user.id,
                    )
                else:
                    logger.error(
                        "Failed to set cookie for OAuth callback (user: %s). "
                        "User will need to authenticate again.",
                        user.id,
                    )
                    error_url = f"{frontend_url}?error=token_failed&message=Cookie%20authentication%20failed.%20Please%20try%20again."
                    response = RedirectResponse(url=error_url)

            logger.info(
                "OAuth callback successful for %s, redirecting to frontend",
                provider,
                extra={
                    "event": "oauth_callback_success",
                    "provider": provider,
                    "user_id": str(user.id),
                    "cookie_set": cookie_success,
                    "token_size": len(jwt_token),
                },
            )
            
            if not cookie_success:
                logger.warning(
                    "Cookie not set for OAuth callback - requests will fail without authentication",
                    extra={
                        "provider": provider,
                        "user_id": str(user.id),
                        "fallback_token_provided": fallback_token is not None,
                    },
                )
            
            return response
        except Exception as e:
            logger.error(
                "SECURITY: OAuth callback error for %s: %s",
                provider,
                e,
                exc_info=True,
                extra={"event": "oauth_callback_error", "provider": provider},
            )
            frontend_url = get_oauth_frontend_redirect_url(provider)
            error_message = quote(str(e))
            error_url = f"{frontend_url}?error=oauth_failed&message={error_message}"
            logger.error("Redirecting to frontend with error: %s", error_url)
            return RedirectResponse(url=error_url)

    @router.get("/authorize")
    async def oauth_authorize() -> RedirectResponse:
        """Get OAuth authorization URL."""
        redirect_uri = get_oauth_backend_callback_url(provider)
        authorization_url = await oauth_client.get_authorization_url(
            redirect_uri=redirect_uri,
            state=None,
        )
        return RedirectResponse(url=authorization_url)

    return router


google_oauth_router = create_oauth_router("google", google_oauth_client)
github_oauth_router = create_oauth_router("github", github_oauth_client)
