# Backend Token Storage Migration Plan

**Date:** 2024  
**Status:** Implementation Plan  
**Target:** Migrate from Authorization header tokens to httpOnly cookies

---

## Executive Summary

This document outlines the complete plan to migrate authentication token storage from Authorization headers (currently stored in frontend localStorage) to httpOnly cookies. This migration significantly improves security by protecting tokens from XSS attacks.

**Estimated Time:** 1-2 weeks  
**Risk Level:** Medium (requires careful testing)  
**Breaking Changes:** None (backward compatible during migration)

---

## Current State Analysis

### Current Implementation

#### Token Flow
1. **OAuth Callback** (`api/auth/oauth.py:232`)
   - Returns token in URL query parameter: `?access_token={jwt_token}&token_type=bearer`
   - Frontend extracts token and stores in `localStorage`

2. **Token Refresh** (`api/routers/v1/auth.py:263`)
   - Returns token in JSON response body
   - Frontend updates `localStorage`

3. **Token Reading** (`api/middleware/auth.py:167`)
   - Reads from `Authorization` header only
   - Checks API keys first, then JWT tokens

4. **CORS Configuration** (`api/middleware/cors.py:37`)
   - ✅ Already has `allow_credentials=True` (required for cookies)

### Security Concerns

- ❌ Tokens accessible via JavaScript (XSS vulnerability)
- ❌ Tokens persist across browser sessions
- ❌ No SameSite protection
- ❌ Tokens exposed in URL query parameters

---

## Target Implementation

### New Token Flow

1. **OAuth Callback**
   - Set httpOnly cookie with token
   - Redirect without token in URL
   - Frontend reads cookie automatically

2. **Token Refresh**
   - Update httpOnly cookie with new token
   - Return success status (no token in body)

3. **Token Reading**
   - Check cookies first
   - Fallback to Authorization header (for API keys and backward compatibility)

4. **Logout**
   - Clear httpOnly cookie
   - Return success status

---

## ⚠️ CRITICAL: Domain Configuration Verification

### Current Domain Setup

**Production:**
- Frontend: `https://wistx.ai` or `https://app.wistx.ai`
- Backend: `https://api.wistx.ai`

**Development:**
- Frontend: `http://localhost:3000`
- Backend: `http://localhost:8000`

### ⚠️ Problem: Different Domains

Cookies **DO NOT** work across different domains (`wistx.ai` vs `api.wistx.ai`).

### Solutions (Choose One)

#### Option 1: Same Domain (Recommended)
**Setup:**
- Frontend: `https://app.wistx.ai`
- Backend: `https://app.wistx.ai/api` (via reverse proxy/API gateway)

**Pros:**
- ✅ Cookies work automatically
- ✅ No CORS issues
- ✅ Best security

**Cons:**
- ❌ Requires infrastructure changes
- ❌ Need reverse proxy/API gateway

#### Option 2: Subdomain Cookies
**Setup:**
- Frontend: `https://app.wistx.ai`
- Backend: `https://api.wistx.ai`
- Set cookie `domain='.wistx.ai'` (note the leading dot)

**Pros:**
- ✅ Works with current infrastructure
- ✅ Cookies shared across subdomains

**Cons:**
- ⚠️ Less secure (cookies accessible to all subdomains)
- ⚠️ Requires careful subdomain management

**Implementation:**
```python
response.set_cookie(
    'auth_token',
    token,
    domain='.wistx.ai',  # Works for app.wistx.ai and api.wistx.ai
    httponly=True,
    secure=True,
    samesite='strict',
    path='/',
)
```

#### Option 3: Next.js API Routes Proxy (Recommended for Next.js)
**Setup:**
- Frontend: `https://app.wistx.ai`
- Backend: `https://api.wistx.ai`
- Next.js API routes (`/api/*`) proxy to backend
- Cookies set via Next.js API routes (same domain)

**Pros:**
- ✅ Works with current infrastructure
- ✅ Cookies work (same domain)
- ✅ No backend changes needed

**Cons:**
- ⚠️ Adds Next.js API route layer
- ⚠️ Slightly more complex

**Implementation:**
```typescript
// app/api/auth/[...path]/route.ts
export async function POST(request: Request, { params }: { params: { path: string[] } }) {
  const body = await request.json();
  const response = await fetch(`${process.env.API_URL}/v1/auth/${params.path.join('/')}`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      'Cookie': request.headers.get('Cookie') || '', // Forward cookies
    },
    body: JSON.stringify(body),
  });
  
  // Forward response cookies
  const setCookie = response.headers.get('Set-Cookie');
  if (setCookie) {
    return new Response(await response.text(), {
      status: response.status,
      headers: {
        'Set-Cookie': setCookie,
      },
    });
  }
  
  return response;
}
```

### Domain Verification Checklist

**Before Migration:**
- [ ] Determine which solution to use
- [ ] Test cookie setting/reading in chosen solution
- [ ] Verify CORS configuration
- [ ] Test in all browsers (Chrome, Firefox, Safari, Edge)
- [ ] Test in production-like environment
- [ ] Document domain configuration

**Testing Script:**
```python
# tests/integration/test_cookie_domains.py
def test_cookie_setting_across_domains():
    """Test cookie setting works with domain configuration."""
    # Test same domain
    # Test subdomain cookies
    # Test CORS with credentials
    pass
```

---

## Implementation Plan

### Phase 0: Domain Verification & Setup (Day 0 - CRITICAL)

#### 0.1 Verify Domain Configuration

**File:** `api/utils/domain_config.py` (NEW)

```python
"""Domain configuration utilities."""

from urllib.parse import urlparse
from api.config import settings


def get_cookie_domain() -> str | None:
    """Get cookie domain based on environment.
    
    Returns:
        Cookie domain string (e.g., '.wistx.ai') or None for same-domain
    """
    if settings.debug:
        return None  # localhost doesn't need domain
    
    # Extract domain from frontend URL
    frontend_url = settings.oauth_frontend_redirect_url_prod
    parsed = urlparse(frontend_url)
    domain = parsed.netloc
    
    # If using subdomain approach, return parent domain
    if domain.startswith('app.'):
        return '.wistx.ai'
    elif domain.startswith('www.'):
        return '.wistx.ai'
    
    return None  # Same domain, no domain needed


def should_use_subdomain_cookies() -> bool:
    """Check if subdomain cookies should be used.
    
    Returns:
        True if frontend and backend are on different domains
    """
    frontend_domain = urlparse(settings.oauth_frontend_redirect_url_prod).netloc
    backend_domain = urlparse(settings.oauth_backend_callback_url_prod).netloc
    
    # Extract base domains
    frontend_base = '.'.join(frontend_domain.split('.')[-2:])
    backend_base = '.'.join(backend_domain.split('.')[-2:])
    
    return frontend_base == backend_base and frontend_domain != backend_domain
```

#### 0.2 Create Domain Verification Tests

**File:** `tests/integration/test_domain_config.py` (NEW)

```python
"""Tests for domain configuration."""

import pytest
from api.utils.domain_config import get_cookie_domain, should_use_subdomain_cookies


def test_cookie_domain_production():
    """Test cookie domain in production."""
    # Test subdomain cookies
    pass


def test_cookie_domain_development():
    """Test cookie domain in development."""
    # Test localhost (no domain)
    pass


def test_cookie_setting_across_domains():
    """Test cookie setting works across domains."""
    # Test cookie is set correctly
    # Test cookie is readable
    pass
```

**Tasks:**
- [ ] Create domain configuration utilities
- [ ] Write domain verification tests
- [ ] Test cookie setting/reading
- [ ] Verify CORS configuration
- [ ] Document chosen approach

---

### Phase 1: Cookie Utilities (Day 1)

#### 1.1 Create Cookie Helper Module

**File:** `api/utils/cookies.py` (NEW)

```python
"""Cookie utilities for authentication tokens."""

from datetime import datetime, timedelta
from fastapi import Response
from api.config import settings


COOKIE_NAME = "auth_token"
COOKIE_MAX_AGE = 60 * 60 * 24 * 7  # 7 days
COOKIE_PATH = "/"
MAX_TOKEN_SIZE = 3500  # Leave buffer under 4KB cookie limit


def validate_token_size(token: str) -> None:
    """Validate token size fits in cookie.
    
    Args:
        token: JWT token string
        
    Raises:
        ValueError: If token is too large for cookie
    """
    if len(token) > MAX_TOKEN_SIZE:
        raise ValueError(
            f"Token size ({len(token)} bytes) exceeds cookie limit ({MAX_TOKEN_SIZE} bytes). "
            "Consider reducing token payload or using token references."
        )


def set_auth_cookie(
    response: Response,
    token: str,
    max_age: int = COOKIE_MAX_AGE,
    domain: str | None = None,
) -> None:
    """Set authentication token as httpOnly cookie.
    
    Args:
        response: FastAPI Response object
        token: JWT token string
        max_age: Cookie max age in seconds (default: 7 days)
        domain: Cookie domain (None for same-domain, '.wistx.ai' for subdomains)
    """
    validate_token_size(token)
    
    cookie_kwargs = {
        "key": COOKIE_NAME,
        "value": token,
        "max_age": max_age,
        "httponly": True,
        "secure": not settings.debug,  # HTTPS only in production
        "samesite": "strict",  # CSRF protection
        "path": COOKIE_PATH,
    }
    
    if domain:
        cookie_kwargs["domain"] = domain
    
    response.set_cookie(**cookie_kwargs)


def clear_auth_cookie(response: Response, domain: str | None = None) -> None:
    """Clear authentication cookie.
    
    Args:
        response: FastAPI Response object
        domain: Cookie domain (must match domain used when setting)
    """
    delete_kwargs = {
        "key": COOKIE_NAME,
        "path": COOKIE_PATH,
        "samesite": "strict",
    }
    
    if domain:
        delete_kwargs["domain"] = domain
    
    response.delete_cookie(**delete_kwargs)


def get_auth_token_from_cookie(request) -> str | None:
    """Get authentication token from cookie.
    
    Args:
        request: FastAPI Request object
        
    Returns:
        Token string or None if not found
    """
    return request.cookies.get(COOKIE_NAME)
```

**Tasks:**
- [ ] Create `api/utils/cookies.py`
- [ ] Add unit tests
- [ ] Test cookie setting/reading
- [ ] Test cookie clearing

---

### Phase 2: Update Authentication Middleware (Day 2)

#### 2.1 Update Token Reading Logic

**File:** `api/middleware/auth.py`

**Current Code (line 167):**
```python
authorization = request.headers.get("authorization", "")
```

**New Code:**
```python
# Check cookie first (for browser-based auth)
token = None
cookie_token = request.cookies.get("auth_token")
if cookie_token:
    token = cookie_token
    authorization = f"Bearer {token}"
else:
    # Fallback to Authorization header (for API keys and backward compatibility)
    authorization = request.headers.get("authorization", "")
```

**Update `_extract_jwt_user` method:**

```python
async def _extract_jwt_user(self, authorization: str | None = None, token: str | None = None) -> dict | None:
    """Extract user info from JWT token.
    
    Args:
        authorization: Authorization header value (optional)
        token: Direct token string (optional, takes precedence)
        
    Returns:
        User info dictionary or None if invalid
    """
    # Use direct token if provided, otherwise extract from authorization header
    if token:
        jwt_token = token
    elif authorization and authorization.startswith("Bearer "):
        jwt_token = authorization.replace("Bearer ", "").strip()
    else:
        return None
    
    if not jwt_token:
        return None
    
    # ... rest of existing JWT validation logic ...
```

**Update `dispatch` method:**

```python
async def dispatch(self, request: Request, call_next: Callable) -> Response:
    """Process request and extract authentication."""
    if self._is_excluded_path(request.url.path):
        return await call_next(request)

    # Check cookie first, then Authorization header
    cookie_token = request.cookies.get("auth_token")
    authorization = request.headers.get("authorization", "")
    
    user_info = None
    
    if cookie_token:
        # Try JWT token from cookie
        user_info = await self._extract_jwt_user(token=cookie_token)
    
    if not user_info and authorization:
        # Fallback to Authorization header (API keys or JWT)
        user_info = await self._extract_api_key_user(authorization)
        if not user_info:
            user_info = await self._extract_jwt_user(authorization=authorization)
    
    # ... rest of existing logic ...
```

**Tasks:**
- [ ] Update `_extract_jwt_user` to accept direct token
- [ ] Update `dispatch` to check cookies first
- [ ] Add logging for cookie-based auth
- [ ] Test backward compatibility (Authorization header still works)
- [ ] Add unit tests

---

### Phase 3: Update OAuth Callback (Day 3)

#### 3.1 Update OAuth Callback Endpoints

**File:** `api/auth/oauth.py`

**Current Code (line 232-233):**
```python
frontend_url = get_oauth_frontend_redirect_url(provider)
redirect_url = f"{frontend_url}?access_token={jwt_token}&token_type=bearer"
return RedirectResponse(url=redirect_url)
```

**New Code:**
```python
from api.utils.cookies import set_auth_cookie

frontend_url = get_oauth_frontend_redirect_url(provider)
response = RedirectResponse(url=frontend_url)
set_auth_cookie(response, jwt_token)
return response
```

**File:** `api/routers/v1/oauth.py`

**Update GitHub OAuth callback (line 242):**
```python
from api.utils.cookies import set_auth_cookie

# After storing GitHub token
frontend_url = get_oauth_frontend_redirect_url("github")
response = RedirectResponse(url=frontend_url)
# Set cookie if user is authenticated (check if user has auth token)
# Note: This endpoint is for connecting GitHub, not initial auth
# May need to check if user already has auth cookie
return response
```

**Tasks:**
- [ ] Update OAuth callback in `api/auth/oauth.py`
- [ ] Update GitHub OAuth callback in `api/routers/v1/oauth.py`
- [ ] Remove token from URL query parameters
- [ ] Test OAuth flow end-to-end
- [ ] Verify cookie is set correctly
- [ ] Test redirect works without token in URL

---

### Phase 4: Update Token Refresh (Day 4)

#### 4.1 Update Refresh Token Endpoint

**File:** `api/routers/v1/auth.py`

**Current Code (line 263-274):**
```python
@router.post("/refresh", response_model=TokenRefreshResponse)
async def refresh_token(
    request: Request,
    current_user: User = Depends(get_user_from_expired_token),
) -> TokenRefreshResponse:
    jwt_strategy = jwt_authentication.get_strategy()
    new_token = await jwt_strategy.write_token(current_user)
    # ... audit logging ...
    return TokenRefreshResponse(access_token=new_token, token_type="bearer")
```

**New Code:**
```python
from api.utils.cookies import set_auth_cookie
from fastapi import Response

@router.post("/refresh")
async def refresh_token(
    request: Request,
    response: Response,
    current_user: User = Depends(get_user_from_expired_token),
) -> dict[str, str]:
    """Refresh JWT access token.
    
    Sets new token in httpOnly cookie and returns success status.
    """
    jwt_strategy = jwt_authentication.get_strategy()
    new_token = await jwt_strategy.write_token(current_user)
    
    # Set cookie instead of returning token in body
    set_auth_cookie(response, new_token)
    
    # ... audit logging ...
    
    # Return success status (no token in body for security)
    return {"status": "success", "message": "Token refreshed"}
```

**Update Response Model:**

Remove `TokenRefreshResponse` model or make it optional for backward compatibility.

**Tasks:**
- [ ] Update refresh token endpoint
- [ ] Set cookie instead of returning token
- [ ] Update response model
- [ ] Test token refresh flow
- [ ] Verify cookie is updated
- [ ] Test backward compatibility (if needed)

---

### Phase 5: Add Logout Endpoint (Day 5)

#### 5.1 Create Logout Endpoint

**File:** `api/routers/v1/auth.py`

**New Code:**
```python
from api.utils.cookies import clear_auth_cookie
from fastapi import Response

@router.post("/logout")
async def logout(
    request: Request,
    response: Response,
    current_user: dict[str, Any] = Depends(get_current_user),
) -> dict[str, str]:
    """Logout user and clear authentication cookie.
    
    Args:
        request: FastAPI Request object
        response: FastAPI Response object
        current_user: Current authenticated user
        
    Returns:
        Success status
    """
    # Clear cookie
    clear_auth_cookie(response)
    
    # Audit logging
    from api.models.audit_log import AuditEventType, AuditLogSeverity
    from api.services.audit_log_service import audit_log_service
    
    ip_address = request.client.host if request.client else None
    user_agent = request.headers.get("user-agent")
    
    audit_log_service.log_event(
        event_type=AuditEventType.AUTHENTICATION_LOGOUT,
        severity=AuditLogSeverity.LOW,
        message=f"User {current_user.get('user_id')} logged out",
        success=True,
        user_id=current_user.get("user_id"),
        organization_id=current_user.get("organization_id"),
        ip_address=ip_address,
        user_agent=user_agent,
        endpoint="/v1/auth/logout",
        method="POST",
        compliance_tags=["PCI-DSS-10", "SOC2"],
    )
    
    return {"status": "success", "message": "Logged out successfully"}
```

**Tasks:**
- [ ] Create logout endpoint
- [ ] Clear authentication cookie
- [ ] Add audit logging
- [ ] Test logout flow
- [ ] Verify cookie is cleared
- [ ] Test logout with different browsers

---

### Phase 6: Update Dependencies (Day 6)

#### 6.1 Update Auth Dependencies

**File:** `api/dependencies/auth.py`

**Current Code (line 41-79):**
```python
async def get_current_user(
    request: Request,
    authorization: Annotated[str | None, Header()] = None,
) -> dict[str, Any]:
    user_info = getattr(request.state, "user_info", None)
    if user_info:
        return user_info
    
    if not authorization or not authorization.startswith("Bearer "):
        raise HTTPException(...)
    
    api_key_value = authorization.replace("Bearer ", "").strip()
    user_info = await get_user_from_api_key(api_key_value)
    # ...
```

**New Code:**
```python
from api.utils.cookies import get_auth_token_from_cookie

async def get_current_user(
    request: Request,
    authorization: Annotated[str | None, Header()] = None,
) -> dict[str, Any]:
    """Extract user information from cookie or Authorization header.
    
    Checks request.state first (set by middleware), then cookie, then header.
    """
    # Check request.state first (set by middleware)
    user_info = getattr(request.state, "user_info", None)
    if user_info:
        return user_info
    
    # Check cookie (for browser-based auth)
    cookie_token = get_auth_token_from_cookie(request)
    if cookie_token:
        # Middleware should have already set request.state.user_info
        # But if not, we can validate here as fallback
        from api.auth.users import jwt_authentication
        from api.database.async_mongodb import async_mongodb_adapter
        from api.auth.database import MongoDBUserDatabase
        from api.auth.users import UserManager
        
        try:
            await async_mongodb_adapter.connect()
            db = async_mongodb_adapter.get_database()
            collection = db.users
            user_db = MongoDBUserDatabase(collection)
            user_manager = UserManager(user_db)
            
            strategy = jwt_authentication.get_strategy()
            user = await strategy.read_token(cookie_token, user_manager)
            if user:
                # Build user_info dict (same as middleware)
                # ...
                return user_info
        except Exception:
            pass  # Fall through to Authorization header
    
    # Fallback to Authorization header (for API keys)
    if not authorization or not authorization.startswith("Bearer "):
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Authentication required",
        )
    
    api_key_value = authorization.replace("Bearer ", "").strip()
    user_info = await get_user_from_api_key(api_key_value)
    
    if not user_info:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Invalid or expired token",
        )
    
    return user_info
```

**Note:** In practice, the middleware should handle cookie-based auth, so this dependency mainly needs to handle the fallback case.

**Tasks:**
- [ ] Update `get_current_user` to check cookies
- [ ] Ensure backward compatibility
- [ ] Test with cookies
- [ ] Test with Authorization header
- [ ] Test with API keys

---

### Phase 7: Testing & Validation (Day 7-8)

#### 7.1 Unit Tests

**File:** `tests/unit/utils/test_cookies.py` (NEW)

```python
"""Tests for cookie utilities."""

import pytest
from fastapi import Response
from api.utils.cookies import (
    set_auth_cookie,
    clear_auth_cookie,
    get_auth_token_from_cookie,
    COOKIE_NAME,
)


def test_set_auth_cookie():
    """Test setting authentication cookie."""
    response = Response()
    token = "test-token-123"
    
    set_auth_cookie(response, token)
    
    # Check cookie is set
    assert COOKIE_NAME in response.headers.get("set-cookie", "")
    assert token in response.headers.get("set-cookie", "")
    assert "HttpOnly" in response.headers.get("set-cookie", "")
    assert "SameSite=Strict" in response.headers.get("set-cookie", "")


def test_clear_auth_cookie():
    """Test clearing authentication cookie."""
    response = Response()
    
    clear_auth_cookie(response)
    
    # Check cookie is deleted
    assert COOKIE_NAME in response.headers.get("set-cookie", "")
    assert "Max-Age=0" in response.headers.get("set-cookie", "")
```

**File:** `tests/unit/middleware/test_auth_cookies.py` (NEW)

```python
"""Tests for authentication middleware with cookies."""

import pytest
from fastapi import Request
from api.middleware.auth import AuthenticationMiddleware


@pytest.mark.asyncio
async def test_middleware_reads_cookie():
    """Test middleware reads token from cookie."""
    # Mock request with cookie
    # Test middleware extracts user info
    pass


@pytest.mark.asyncio
async def test_middleware_fallback_to_header():
    """Test middleware falls back to Authorization header."""
    # Mock request without cookie but with header
    # Test middleware extracts user info from header
    pass
```

#### 7.2 Integration Tests

**File:** `tests/integration/test_oauth_cookies.py` (NEW)

```python
"""Integration tests for OAuth with cookies."""

import pytest
from fastapi.testclient import TestClient


def test_oauth_callback_sets_cookie(client: TestClient):
    """Test OAuth callback sets httpOnly cookie."""
    # Mock OAuth callback
    # Verify cookie is set
    # Verify redirect doesn't include token in URL
    pass


def test_token_refresh_updates_cookie(client: TestClient):
    """Test token refresh updates cookie."""
    # Authenticate user
    # Call refresh endpoint
    # Verify cookie is updated
    pass


def test_logout_clears_cookie(client: TestClient):
    """Test logout clears cookie."""
    # Authenticate user
    # Call logout endpoint
    # Verify cookie is cleared
    pass
```

#### 7.3 E2E Tests

**File:** `tests/e2e/test_auth_flow.py` (UPDATE)

```python
"""E2E tests for authentication flow with cookies."""

def test_complete_auth_flow():
    """Test complete authentication flow with cookies."""
    # 1. OAuth callback sets cookie
    # 2. Subsequent requests use cookie
    # 3. Token refresh updates cookie
    # 4. Logout clears cookie
    pass
```

**Tasks:**
- [ ] Write unit tests for cookie utilities
- [ ] Write unit tests for middleware
- [ ] Write integration tests for OAuth
- [ ] Write integration tests for token refresh
- [ ] Write integration tests for logout
- [ ] Update E2E tests
- [ ] Test in different browsers
- [ ] Test with different cookie settings

---

### Phase 8: Documentation & Deployment (Day 9-10)

#### 8.1 Update API Documentation

**File:** `docs/api-reference/authentication.md` (UPDATE)

- Document cookie-based authentication
- Document logout endpoint
- Update OAuth flow documentation
- Add migration guide

#### 8.2 Update OpenAPI Spec

**File:** `docs/api-reference/openapi.json` (UPDATE)

- Update OAuth callback response (remove token from query)
- Add logout endpoint
- Update refresh token response

#### 8.3 Deployment Checklist

- [ ] All tests passing
- [ ] Code review completed
- [ ] Documentation updated
- [ ] Frontend ready (already done)
- [ ] Staging deployment
- [ ] Staging testing
- [ ] Production deployment
- [ ] Monitor error rates
- [ ] Monitor cookie usage

---

## Backward Compatibility

### Strategy

During migration, support both methods:

1. **Cookie-based auth** (new, preferred)
2. **Authorization header** (fallback, for API keys and compatibility)

### Implementation

- Middleware checks cookie first, then header
- Dependencies check cookie first, then header
- OAuth callback sets cookie (new) but can still return token in URL (if needed)
- Token refresh sets cookie (new) but can still return token in body (if needed)

### Migration Period

- **Week 1-2:** Both methods supported
- **Week 3:** Monitor usage, ensure all clients migrated
- **Week 4:** Remove Authorization header support for JWT tokens (keep for API keys)

---

## Security Considerations

### Cookie Settings

- ✅ `httponly=True` - Prevents JavaScript access
- ✅ `secure=True` (production) - HTTPS only
- ✅ `samesite="strict"` - CSRF protection
- ✅ `path="/"` - Available site-wide
- ✅ `max_age=7 days` - Reasonable expiration

### CSRF Protection

- ✅ SameSite=strict prevents CSRF
- ✅ CSRF tokens already implemented (frontend)
- ✅ State-changing operations require CSRF token

### Token Security

- ✅ Tokens not accessible via JavaScript
- ✅ Tokens not exposed in URLs
- ✅ Tokens automatically sent with requests
- ✅ Tokens cleared on logout

---

## Rollback Plan

### If Issues Occur

1. **Feature Flag**: Add feature flag to disable cookie auth
2. **Fallback**: Authorization header still works
3. **Quick Fix**: Revert OAuth callback to return token in URL
4. **Monitoring**: Monitor error rates and user sessions

### Rollback Steps

1. Disable cookie setting in OAuth callback
2. Revert middleware changes (if needed)
3. Frontend falls back to localStorage
4. Monitor and fix issues
5. Re-enable when ready

---

## Success Criteria

### Phase 1 Complete
- [ ] Cookie utilities implemented and tested
- [ ] Unit tests passing

### Phase 2 Complete
- [ ] Middleware reads cookies
- [ ] Backward compatibility maintained
- [ ] Tests passing

### Phase 3 Complete
- [ ] OAuth callback sets cookies
- [ ] No tokens in URLs
- [ ] E2E tests passing

### Phase 4 Complete
- [ ] Token refresh updates cookies
- [ ] Tests passing

### Phase 5 Complete
- [ ] Logout endpoint implemented
- [ ] Cookie cleared on logout
- [ ] Tests passing

### Phase 6 Complete
- [ ] Dependencies updated
- [ ] All tests passing

### Phase 7 Complete
- [ ] All tests passing
- [ ] E2E tests passing
- [ ] Browser compatibility verified

### Phase 8 Complete
- [ ] Documentation updated
- [ ] Deployed to production
- [ ] Monitoring in place

---

## Timeline

| Phase | Days | Description |
|-------|------|-------------|
| 1 | 1 | Cookie utilities |
| 2 | 1 | Update middleware |
| 3 | 1 | Update OAuth callbacks |
| 4 | 1 | Update token refresh |
| 5 | 1 | Add logout endpoint |
| 6 | 1 | Update dependencies |
| 7 | 2 | Testing & validation |
| 8 | 2 | Documentation & deployment |
| **Total** | **10** | **~2 weeks** |

---

## Risk Assessment

### Low Risk
- Cookie utilities (isolated, well-tested)
- Logout endpoint (new, doesn't affect existing flow)

### Medium Risk
- Middleware changes (affects all requests)
- OAuth callback changes (affects user signup/login)
- Token refresh changes (affects session management)

### Mitigation
- Feature flags for gradual rollout
- Comprehensive testing
- Backward compatibility maintained
- Monitoring and alerting
- Rollback plan ready

---

## Next Steps

1. **Review this plan** with team
2. **Assign tasks** to developers
3. **Set up feature flags** for gradual rollout
4. **Begin Phase 1** implementation
5. **Daily standups** to track progress
6. **Weekly reviews** to assess readiness

---

## Questions & Answers

### Q: Will this break existing API clients?
**A:** No, Authorization header still works for API keys and backward compatibility.

### Q: What about mobile apps?
**A:** Mobile apps can continue using Authorization headers. Cookies are for browser-based auth.

### Q: How do we test this?
**A:** Comprehensive unit, integration, and E2E tests. Also test in staging environment.

### Q: What if cookies are disabled?
**A:** Fallback to Authorization header. Users can enable cookies or use API keys.

### Q: How long is the migration period?
**A:** 2-4 weeks with both methods supported, then gradual removal of header support for JWT.

---

---

## Phase 2: Refresh Token Pattern (Post-Migration - Weeks 3-4)

### Overview

After initial cookie migration is stable, implement refresh token pattern for enhanced security.

### Benefits

- ✅ Shorter access token lifetime (15 min vs 30 min)
- ✅ Revocable refresh tokens
- ✅ Token rotation capability
- ✅ Better security posture

### Implementation

#### 2.1 Database Schema Changes

**File:** `api/database/migrations/add_refresh_tokens.py` (NEW)

```python
"""Migration to add refresh tokens collection."""

from pymongo import IndexModel


def upgrade(db):
    """Add refresh_tokens collection."""
    refresh_tokens = db.refresh_tokens
    
    # Create indexes
    indexes = [
        IndexModel([("user_id", 1), ("token_hash", 1)]),
        IndexModel([("expires_at", 1)], expireAfterSeconds=0),
        IndexModel([("created_at", 1)]),
    ]
    refresh_tokens.create_indexes(indexes)


def downgrade(db):
    """Remove refresh_tokens collection."""
    db.refresh_tokens.drop()
```

#### 2.2 Refresh Token Model

**File:** `api/models/refresh_token.py` (NEW)

```python
"""Refresh token model."""

from datetime import datetime, timedelta
from typing import Optional
from bson import ObjectId
from pymongo import MongoClient


class RefreshToken:
    """Refresh token document."""
    
    def __init__(
        self,
        user_id: str,
        token_hash: str,
        expires_at: datetime,
        created_at: datetime,
        last_used_at: Optional[datetime] = None,
        rotated_from: Optional[str] = None,
        revoked: bool = False,
    ):
        self.user_id = user_id
        self.token_hash = token_hash
        self.expires_at = expires_at
        self.created_at = created_at
        self.last_used_at = last_used_at
        self.rotated_from = rotated_from
        self.revoked = revoked
    
    @classmethod
    def create(cls, user_id: str, token: str) -> "RefreshToken":
        """Create new refresh token."""
        import hashlib
        
        token_hash = hashlib.sha256(token.encode()).hexdigest()
        expires_at = datetime.utcnow() + timedelta(days=30)
        
        return cls(
            user_id=user_id,
            token_hash=token_hash,
            expires_at=expires_at,
            created_at=datetime.utcnow(),
        )
    
    def is_valid(self) -> bool:
        """Check if token is valid."""
        return not self.revoked and datetime.utcnow() < self.expires_at
```

#### 2.3 Update Token Refresh Endpoint

**File:** `api/routers/v1/auth.py`

```python
@router.post("/refresh")
async def refresh_token(
    request: Request,
    response: Response,
    current_user: User = Depends(get_user_from_expired_token),
) -> dict[str, str]:
    """Refresh access token using refresh token.
    
    Implements token rotation:
    1. Validates refresh token
    2. Issues new access token (15 min)
    3. Issues new refresh token (30 days)
    4. Revokes old refresh token
    """
    from api.utils.cookies import get_auth_token_from_cookie
    from api.models.refresh_token import RefreshToken
    from api.database.mongodb import mongodb_manager
    import hashlib
    
    # Get refresh token from cookie
    refresh_token_cookie = request.cookies.get("refresh_token")
    if not refresh_token_cookie:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Refresh token required",
        )
    
    # Validate refresh token
    token_hash = hashlib.sha256(refresh_token_cookie.encode()).hexdigest()
    db = mongodb_manager.get_database()
    refresh_token_doc = db.refresh_tokens.find_one({
        "user_id": str(current_user.id),
        "token_hash": token_hash,
        "revoked": False,
    })
    
    if not refresh_token_doc:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Invalid refresh token",
        )
    
    refresh_token = RefreshToken(**refresh_token_doc)
    if not refresh_token.is_valid():
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Refresh token expired",
        )
    
    # Issue new tokens
    jwt_strategy = jwt_authentication.get_strategy()
    new_access_token = await jwt_strategy.write_token(current_user)
    
    # Generate new refresh token
    import secrets
    new_refresh_token = secrets.token_urlsafe(32)
    new_refresh_token_hash = hashlib.sha256(new_refresh_token.encode()).hexdigest()
    
    # Revoke old refresh token
    db.refresh_tokens.update_one(
        {"token_hash": token_hash},
        {
            "$set": {
                "revoked": True,
                "rotated_to": new_refresh_token_hash,
                "revoked_at": datetime.utcnow(),
            }
        }
    )
    
    # Store new refresh token
    new_refresh_token_doc = RefreshToken.create(
        str(current_user.id),
        new_refresh_token
    )
    db.refresh_tokens.insert_one(new_refresh_token_doc.__dict__)
    
    # Set cookies
    from api.utils.cookies import set_auth_cookie, get_cookie_domain
    
    cookie_domain = get_cookie_domain()
    set_auth_cookie(response, new_access_token, max_age=60 * 15, domain=cookie_domain)  # 15 min
    set_auth_cookie(
        response,
        new_refresh_token,
        max_age=60 * 60 * 24 * 30,
        domain=cookie_domain,
        cookie_name="refresh_token"
    )  # 30 days
    
    return {"status": "success", "message": "Token refreshed"}
```

**Tasks:**
- [ ] Create refresh token model
- [ ] Create database migration
- [ ] Update refresh endpoint
- [ ] Implement token rotation
- [ ] Add token revocation
- [ ] Update tests

---

## Additional Considerations

### Error Handling

#### Cookie Setting Failures

**File:** `api/utils/cookies.py`

```python
def set_auth_cookie_safe(
    response: Response,
    token: str,
    max_age: int = COOKIE_MAX_AGE,
    domain: str | None = None,
    fallback_to_header: bool = True,
) -> tuple[bool, str | None]:
    """Safely set auth cookie with fallback.
    
    Returns:
        Tuple of (success, fallback_token)
        If cookie setting fails and fallback_to_header=True, returns token for header
    """
    try:
        set_auth_cookie(response, token, max_age, domain)
        return True, None
    except Exception as e:
        logger.warning("Failed to set auth cookie: %s", e)
        if fallback_to_header:
            return False, token
        raise
```

#### Cookie Reading Failures

**File:** `api/middleware/auth.py`

```python
async def dispatch(self, request: Request, call_next: Callable) -> Response:
    """Process request with cookie fallback."""
    # Try cookie first
    cookie_token = request.cookies.get("auth_token")
    if cookie_token:
        try:
            user_info = await self._extract_jwt_user(token=cookie_token)
            if user_info:
                request.state.user_info = user_info
                return await call_next(request)
        except Exception as e:
            logger.warning("Cookie auth failed, falling back to header: %s", e)
    
    # Fallback to header
    # ... existing header logic ...
```

### Monitoring & Metrics

#### Cookie Acceptance Rate

**File:** `api/middleware/auth.py`

```python
# Track cookie vs header usage
if cookie_token:
    metrics.increment("auth.method.cookie")
else:
    metrics.increment("auth.method.header")
```

#### Token Refresh Metrics

**File:** `api/routers/v1/auth.py`

```python
# Track refresh success/failure
metrics.increment("auth.refresh.success")
metrics.histogram("auth.refresh.duration", duration_ms)
```

### Edge Cases

#### 1. Cookies Disabled

**Detection:**
```python
# Check if cookies are supported
cookie_supported = request.cookies.get("_cookie_check") is not None
if not cookie_supported:
    # Fallback to header
    pass
```

**User Experience:**
- Show warning if cookies disabled
- Provide instructions to enable cookies
- Fallback to Authorization header

#### 2. Browser Restrictions

**Safari ITP:**
- Test in Safari
- Monitor cookie acceptance
- Provide fallback

**Third-Party Cookies:**
- Use same-domain setup
- Avoid third-party cookie requirements

#### 3. Multi-Tab Scenarios

**Issue:** Token refresh in one tab should update all tabs

**Solution:**
- Use `BroadcastChannel` API (frontend)
- Or rely on cookie updates (automatic)

#### 4. Token Expiration During Request

**Issue:** Token expires while request is in flight

**Solution:**
- Retry with token refresh
- Already implemented in frontend

### Performance Considerations

#### Cookie Size Impact

- Monitor token size
- Consider token compression (if needed)
- Use token references (if tokens get large)

#### Database Queries

- Index refresh tokens properly
- Cache token validation (Redis)
- Batch token operations

### Security Enhancements

#### 1. Token Rotation

- Implemented in Phase 2
- Rotate refresh tokens on use
- Invalidate old tokens

#### 2. Token Revocation

**File:** `api/routers/v1/auth.py`

```python
@router.post("/revoke")
async def revoke_all_tokens(
    current_user: dict[str, Any] = Depends(get_current_user),
) -> dict[str, str]:
    """Revoke all tokens for current user."""
    from api.database.mongodb import mongodb_manager
    
    db = mongodb_manager.get_database()
    db.refresh_tokens.update_many(
        {"user_id": current_user["user_id"], "revoked": False},
        {"$set": {"revoked": True, "revoked_at": datetime.utcnow()}}
    )
    
    response = Response()
    clear_auth_cookie(response)
    clear_auth_cookie(response, cookie_name="refresh_token")
    
    return {"status": "success", "message": "All tokens revoked"}
```

#### 3. Suspicious Activity Detection

- Track token usage patterns
- Alert on unusual activity
- Auto-revoke suspicious tokens

### Testing Scenarios

#### Comprehensive Test Cases

1. **Cookie Setting**
   - [ ] Same domain
   - [ ] Subdomain cookies
   - [ ] Localhost (no domain)
   - [ ] Token size validation
   - [ ] Cookie expiration

2. **Cookie Reading**
   - [ ] Valid cookie
   - [ ] Expired cookie
   - [ ] Invalid cookie
   - [ ] Missing cookie (fallback to header)

3. **Token Refresh**
   - [ ] Successful refresh
   - [ ] Expired refresh token
   - [ ] Revoked refresh token
   - [ ] Token rotation

4. **Multi-Tab**
   - [ ] Token refresh in one tab
   - [ ] Cookie update in all tabs
   - [ ] Logout in one tab

5. **Error Cases**
   - [ ] Cookies disabled
   - [ ] Cookie size exceeded
   - [ ] Network failures
   - [ ] Server errors

6. **Browser Compatibility**
   - [ ] Chrome
   - [ ] Firefox
   - [ ] Safari
   - [ ] Edge
   - [ ] Mobile browsers

### Production Deployment Checklist

#### Pre-Deployment

- [ ] Domain configuration verified
- [ ] Cookie setting tested
- [ ] CORS configured correctly
- [ ] All tests passing
- [ ] Browser compatibility verified
- [ ] Monitoring in place
- [ ] Rollback plan ready

#### Deployment

- [ ] Deploy to staging first
- [ ] Test in staging environment
- [ ] Monitor error rates
- [ ] Gradual rollout (feature flag)
- [ ] Monitor cookie acceptance
- [ ] Monitor token refresh rates

#### Post-Deployment

- [ ] Monitor error rates
- [ ] Monitor cookie vs header usage
- [ ] Monitor token refresh success
- [ ] User feedback
- [ ] Performance metrics

### Rollback Procedures

#### Quick Rollback

1. **Disable Cookie Setting:**
   ```python
   # Feature flag
   USE_COOKIES = False
   
   if USE_COOKIES:
       set_auth_cookie(response, token)
   else:
       # Return token in response
       return {"access_token": token}
   ```

2. **Revert Middleware:**
   - Remove cookie reading logic
   - Use header only

3. **Frontend Fallback:**
   - Already supports localStorage
   - Automatic fallback

#### Full Rollback

1. Revert code changes
2. Deploy previous version
3. Clear cookies (if needed)
4. Monitor for issues

---

## Conclusion

This migration plan provides a comprehensive, step-by-step approach to implementing httpOnly cookie-based authentication. The plan:

- ✅ **Addresses domain configuration** (critical)
- ✅ **Maintains backward compatibility**
- ✅ **Includes thorough testing**
- ✅ **Provides clear rollback strategy**
- ✅ **Plans for refresh token pattern** (Phase 2)
- ✅ **Covers edge cases and error handling**
- ✅ **Includes monitoring and metrics**

**Ready to begin implementation when approved.**

**Next Steps:**
1. Review domain configuration options
2. Choose domain solution
3. Verify domain setup
4. Begin Phase 0 (Domain Verification)
5. Proceed with Phase 1 (Cookie Utilities)

