"""CSRF protection middleware using Double Submit Cookie pattern."""

import logging
from typing import Callable

from fastapi import Request, Response, HTTPException, status
from starlette.middleware.base import BaseHTTPMiddleware

from api.utils.csrf import (
    validate_csrf_token,
    is_state_changing_method,
    CSRF_COOKIE_NAME,
    CSRF_HEADER_NAME,
)
from api.config import settings

logger = logging.getLogger(__name__)


class CSRFProtectionMiddleware(BaseHTTPMiddleware):
    """CSRF protection middleware using Double Submit Cookie pattern.
    
    Validates CSRF tokens for state-changing HTTP methods (POST, PUT, PATCH, DELETE).
    Compares X-CSRF-Token header to csrf_token cookie value.
    
    Exemptions:
    - Safe methods (GET, HEAD, OPTIONS)
    - Health check endpoints
    - CSRF token endpoint itself
    - OAuth callback endpoints (handled separately)
    """

    def __init__(self, app):
        """Initialize CSRF protection middleware.

        Args:
            app: ASGI application
        """
        super().__init__(app)
        self.excluded_paths = {
            "/health",
            "/docs",
            "/redoc",
            "/openapi.json",
            "/v1/csrf-token",
            "/v1/auth/sync",
            "/v1/auth/refresh",  # Token refresh endpoint
            "/auth/",
        }

    def _is_excluded_path(self, path: str) -> bool:
        """Check if path is excluded from CSRF protection.
        
        Args:
            path: Request path
            
        Returns:
            True if path is excluded, False otherwise
        """
        return any(path.startswith(excluded) for excluded in self.excluded_paths)

    async def dispatch(self, request: Request, call_next: Callable) -> Response:
        """Process request and validate CSRF token.

        Args:
            request: FastAPI request object
            call_next: Next middleware/handler

        Returns:
            Response object

        Raises:
            HTTPException: If CSRF token is missing or invalid
        """
        # Disable CSRF protection in development mode
        if settings.debug:
            logger.debug("CSRF protection disabled in development mode")
            return await call_next(request)

        if request.method == "OPTIONS":
            return await call_next(request)

        if self._is_excluded_path(request.url.path):
            return await call_next(request)

        if not is_state_changing_method(request.method):
            return await call_next(request)
        
        header_token = request.headers.get(CSRF_HEADER_NAME) or request.headers.get(
            CSRF_HEADER_NAME.lower()
        )
        cookie_token = request.cookies.get(CSRF_COOKIE_NAME)
        
        logger.info(
            "CSRF validation check: method=%s, path=%s, has_header=%s, has_cookie=%s",
            request.method,
            request.url.path,
            bool(header_token),
            bool(cookie_token),
        )
        
        if not validate_csrf_token(header_token, cookie_token):
            from api.models.audit_log import AuditEventType, AuditLogSeverity
            from api.services.audit_log_service import audit_log_service
            
            ip_address = request.client.host if request.client else None
            user_agent = request.headers.get("user-agent")
            request_id = getattr(request.state, "request_id", None)
            
            audit_log_service.log_event(
                event_type=AuditEventType.AUTHENTICATION_FAILURE,
                severity=AuditLogSeverity.HIGH,
                message="CSRF token validation failed",
                success=False,
                ip_address=ip_address,
                user_agent=user_agent,
                request_id=request_id,
                endpoint=request.url.path,
                method=request.method,
                details={
                    "reason": "CSRF token missing or invalid",
                    "has_header": bool(header_token),
                    "has_cookie": bool(cookie_token),
                    "path": request.url.path,
                },
                compliance_tags=["PCI-DSS-10", "SOC2"],
            )
            
            logger.warning(
                "CSRF validation failed: method=%s, path=%s, has_header=%s, has_cookie=%s",
                request.method,
                request.url.path,
                bool(header_token),
                bool(cookie_token),
            )
            
            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail="CSRF token missing or invalid",
                headers={"X-CSRF-Required": "true"},
            )
        
        logger.info(
            "CSRF validation successful: method=%s, path=%s",
            request.method,
            request.url.path,
        )
        
        return await call_next(request)

