"""Resource type validation utilities."""

VALID_AWS_RESOURCE_TYPES = {
    "EC2",
    "RDS",
    "S3",
    "Lambda",
    "EKS",
    "ECS",
    "VPC",
    "IAM",
    "CloudFront",
    "Route53",
    "ElastiCache",
    "DynamoDB",
    "SQS",
    "SNS",
    "KMS",
    "SecretsManager",
    "CloudWatch",
    "ELB",
    "ALB",
    "NLB",
    "EFS",
    "EBS",
    "Redshift",
    "Aurora",
    "DocumentDB",
    "Neptune",
    "Elasticsearch",
    "OpenSearch",
    "MSK",
    "MQ",
    "EventBridge",
    "StepFunctions",
    "AppSync",
    "API Gateway",
    "Cognito",
    "WAF",
    "Shield",
    "GuardDuty",
    "SecurityHub",
    "Inspector",
    "Macie",
    "Config",
    "CloudTrail",
    "Systems Manager",
    "CodeBuild",
    "CodeDeploy",
    "CodePipeline",
    "CloudFormation",
    "Terraform",
}

VALID_GCP_RESOURCE_TYPES = {
    "GCE",
    "Cloud SQL",
    "GKE",
    "Cloud Storage",
    "Cloud Functions",
    "Cloud Run",
    "Cloud Build",
    "Cloud Pub/Sub",
    "Cloud Bigtable",
    "Cloud Spanner",
    "Cloud Firestore",
    "Cloud Datastore",
    "Cloud Memorystore",
    "Cloud Load Balancing",
    "Cloud CDN",
    "Cloud IAM",
    "Cloud KMS",
    "Cloud Secret Manager",
    "Cloud Monitoring",
    "Cloud Logging",
    "Cloud Trace",
    "Cloud Debugger",
    "Cloud Scheduler",
    "Cloud Tasks",
    "Cloud Endpoints",
    "Cloud Armor",
    "Cloud Identity",
    "Cloud Resource Manager",
    "Cloud Billing",
    "Cloud Asset",
    "Cloud Security Command Center",
    "Cloud DLP",
    "Cloud IAP",
    "Cloud VPN",
    "Cloud Interconnect",
    "Cloud NAT",
    "Cloud DNS",
    "Cloud Domains",
    "Cloud DNS",
}

VALID_AZURE_RESOURCE_TYPES = {
    "Virtual Machines",
    "VM Scale Sets",
    "SQL Database",
    "Cosmos DB",
    "Storage Account",
    "Blob Storage",
    "File Storage",
    "Queue Storage",
    "Table Storage",
    "AKS",
    "Container Instances",
    "App Service",
    "Functions",
    "Logic Apps",
    "Service Bus",
    "Event Hubs",
    "Event Grid",
    "Key Vault",
    "Active Directory",
    "Application Gateway",
    "Load Balancer",
    "Front Door",
    "CDN",
    "Traffic Manager",
    "DNS",
    "Virtual Network",
    "VPN Gateway",
    "ExpressRoute",
    "Network Security Group",
    "Application Security Group",
    "Firewall",
    "DDoS Protection",
    "Security Center",
    "Sentinel",
    "Defender",
    "Policy",
    "Resource Manager",
    "Monitor",
    "Log Analytics",
    "Application Insights",
    "Backup",
    "Site Recovery",
    "Cost Management",
}

VALID_RESOURCE_TYPES = VALID_AWS_RESOURCE_TYPES | VALID_GCP_RESOURCE_TYPES | VALID_AZURE_RESOURCE_TYPES


def validate_resource_types(resource_types: list[str]) -> tuple[list[str], list[str]]:
    """Validate resource types against known resource type lists.

    Args:
        resource_types: List of resource types to validate

    Returns:
        Tuple of (valid_types, invalid_types)
    """
    valid_types = []
    invalid_types = []

    for resource_type in resource_types:
        if not resource_type or not resource_type.strip():
            continue

        normalized = resource_type.strip().upper()

        if normalized in VALID_RESOURCE_TYPES:
            valid_types.append(resource_type.strip())
        else:
            invalid_types.append(resource_type.strip())

    return valid_types, invalid_types

