"""Core package lists for pre-indexing - popular DevOps/infrastructure packages."""

CORE_PACKAGES = {
    "pypi": [
        "boto3",
        "botocore",
        "awscli",
        "terraformpy",
        "pulumi",
        "ansible",
        "ansible-core",
        "kubernetes",
        "docker",
        "docker-compose",
        "pyyaml",
        "jinja2",
        "click",
        "requests",
        "urllib3",
        "cryptography",
        "paramiko",
        "fabric",
        "invoke",
        "pyinfra",
        "molecule",
        "testinfra",
        "pytest",
        "pytest-ansible",
        "pytest-docker",
        "pytest-k8s",
        "kubectl-python",
        "kubernetes-client",
        "openshift",
        "helm",
        "pyhelm",
        "kubernetes-validate",
        "kubeconfig",
        "pykube",
        "kubernetes-python-client",
        "pyhelm3",
        "terraform-python",
        "python-terraform",
        "terrascript",
        "hcl2",
        "pyhcl",
        "python-hcl2",
        "terraformpy",
        "pulumi-aws",
        "pulumi-kubernetes",
        "pulumi-docker",
        "pulumi-gcp",
        "pulumi-azure",
        "checkov",
        "tfsec",
        "terrascan",
        "infracost",
        "cloud-pricing",
        "aws-pricing",
        "boto3-stubs",
        "mypy-boto3",
        "s3fs",
        "s3transfer",
        "boto3-type-annotations",
        "aws-sam-cli",
        "aws-cdk-lib",
        "aws-cdk",
        "cdk8s",
        "cdk8s-plus",
        "cdk8s-plus-22",
        "cdk8s-plus-23",
        "cdk8s-plus-24",
        "cdk8s-plus-25",
        "cdk8s-plus-26",
        "cdk8s-plus-27",
        "cdk8s-plus-28",
        "cdk8s-plus-29",
        "cdk8s-plus-30",
        "cdk8s-plus-31",
        "cdk8s-plus-32",
        "cdk8s-plus-33",
        "cdk8s-plus-34",
        "cdk8s-plus-35",
        "cdk8s-plus-36",
        "cdk8s-plus-37",
        "cdk8s-plus-38",
        "cdk8s-plus-39",
        "cdk8s-plus-40",
        "prometheus-client",
        "prometheus-api-client",
        "grafana-api",
        "datadog",
        "ddtrace",
        "newrelic",
        "sentry-sdk",
        "opentelemetry-api",
        "opentelemetry-sdk",
        "opentelemetry-instrumentation-boto3",
        "opentelemetry-instrumentation-requests",
        "opentelemetry-instrumentation-flask",
        "opentelemetry-instrumentation-django",
        "opentelemetry-instrumentation-fastapi",
        "opentelemetry-instrumentation-httpx",
        "opentelemetry-instrumentation-psycopg2",
        "opentelemetry-instrumentation-redis",
        "opentelemetry-instrumentation-sqlalchemy",
        "opentelemetry-instrumentation-celery",
        "opentelemetry-instrumentation-kafka",
        "opentelemetry-instrumentation-grpc",
        "opentelemetry-instrumentation-aws-lambda",
        "opentelemetry-instrumentation-aws-sdk",
        "opentelemetry-instrumentation-botocore",
        "opentelemetry-instrumentation-dynamodb",
        "opentelemetry-instrumentation-s3",
        "opentelemetry-instrumentation-sqs",
        "opentelemetry-instrumentation-sns",
        "opentelemetry-instrumentation-kinesis",
        "opentelemetry-instrumentation-redshift",
        "opentelemetry-instrumentation-rds",
        "opentelemetry-instrumentation-elasticsearch",
        "opentelemetry-instrumentation-mongodb",
        "opentelemetry-instrumentation-pymongo",
        "opentelemetry-instrumentation-postgresql",
        "opentelemetry-instrumentation-mysql",
        "opentelemetry-instrumentation-airflow",
        "opentelemetry-instrumentation-prefect",
    ],
    "npm": [
        "@aws-sdk/client-s3",
        "@aws-sdk/client-ec2",
        "@aws-sdk/client-lambda",
        "@aws-sdk/client-dynamodb",
        "@aws-sdk/client-rds",
        "@aws-sdk/client-eks",
        "@aws-sdk/client-ecs",
        "@aws-sdk/client-cloudformation",
        "@aws-sdk/client-cloudwatch",
        "@aws-sdk/client-iam",
        "@aws-sdk/client-sts",
        "@aws-sdk/client-sns",
        "@aws-sdk/client-sqs",
        "@aws-sdk/client-kinesis",
        "@aws-sdk/client-firehose",
        "@aws-sdk/client-redshift",
        "@aws-sdk/client-elasticache",
        "@aws-sdk/client-elasticsearch",
        "@aws-sdk/client-opensearch",
        "@aws-sdk/client-route53",
        "@aws-sdk/client-cloudfront",
        "@aws-sdk/client-api-gateway",
        "@aws-sdk/client-apigatewayv2",
        "@aws-sdk/client-appsync",
        "@aws-sdk/client-cognito",
        "@aws-sdk/client-cognito-identity",
        "@aws-sdk/client-cognito-idp",
        "@aws-sdk/client-cognito-sync",
        "@aws-sdk/client-secrets-manager",
        "@aws-sdk/client-ssm",
        "@aws-sdk/client-parameter-store",
        "@aws-sdk/client-systems-manager",
        "@aws-sdk/client-codecommit",
        "@aws-sdk/client-codebuild",
        "@aws-sdk/client-codedeploy",
        "@aws-sdk/client-codepipeline",
        "@aws-sdk/client-codeartifact",
        "@aws-sdk/client-codeguru",
        "@aws-sdk/client-codeguru-reviewer",
        "@aws-sdk/client-codeguru-profiler",
        "@aws-sdk/client-device-farm",
        "@aws-sdk/client-device-farm",
        "@aws-sdk/client-devops-guru",
        "@aws-sdk/client-xray",
        "@aws-sdk/client-cloudtrail",
        "@aws-sdk/client-config",
        "@aws-sdk/client-securityhub",
        "@aws-sdk/client-guardduty",
        "@aws-sdk/client-macie",
        "@aws-sdk/client-macie2",
        "@aws-sdk/client-inspector",
        "@aws-sdk/client-inspector2",
        "@aws-sdk/client-shield",
        "@aws-sdk/client-waf",
        "@aws-sdk/client-wafv2",
        "@aws-sdk/client-network-firewall",
        "@aws-sdk/client-networkmanager",
        "@aws-sdk/client-directory-service",
        "@aws-sdk/client-organizations",
        "@aws-sdk/client-resource-groups",
        "@aws-sdk/client-resource-groups-tagging",
        "@aws-sdk/client-tagging",
        "@aws-sdk/client-cost-explorer",
        "@aws-sdk/client-billing",
        "@aws-sdk/client-budgets",
        "@aws-sdk/client-marketplace",
        "@aws-sdk/client-marketplace-catalog",
        "@aws-sdk/client-marketplace-entitlement",
        "@aws-sdk/client-marketplace-metering",
        "@aws-sdk/client-marketplace-commerce-analytics",
        "@aws-sdk/client-marketplace-procurement",
        "@aws-sdk/client-marketplace-seller",
        "@aws-sdk/client-marketplace-vendor",
        "@aws-sdk/client-marketplace-vendor-insights",
        "@aws-sdk/client-marketplace-vendor-insights-api",
        "@aws-sdk/client-marketplace-vendor-insights-api-v2",
        "@aws-sdk/client-marketplace-vendor-insights-api-v3",
        "@aws-sdk/client-marketplace-vendor-insights-api-v4",
        "@aws-sdk/client-marketplace-vendor-insights-api-v5",
        "@aws-sdk/client-marketplace-vendor-insights-api-v6",
        "@aws-sdk/client-marketplace-vendor-insights-api-v7",
        "@aws-sdk/client-marketplace-vendor-insights-api-v8",
        "@aws-sdk/client-marketplace-vendor-insights-api-v9",
        "@aws-sdk/client-marketplace-vendor-insights-api-v10",
        "@aws-sdk/client-marketplace-vendor-insights-api-v11",
        "@aws-sdk/client-marketplace-vendor-insights-api-v12",
        "@aws-sdk/client-marketplace-vendor-insights-api-v13",
        "@aws-sdk/client-marketplace-vendor-insights-api-v14",
        "@aws-sdk/client-marketplace-vendor-insights-api-v15",
        "@aws-sdk/client-marketplace-vendor-insights-api-v16",
        "@aws-sdk/client-marketplace-vendor-insights-api-v17",
        "@aws-sdk/client-marketplace-vendor-insights-api-v18",
        "@aws-sdk/client-marketplace-vendor-insights-api-v19",
        "@aws-sdk/client-marketplace-vendor-insights-api-v20",
        "@aws-sdk/client-marketplace-vendor-insights-api-v21",
        "@aws-sdk/client-marketplace-vendor-insights-api-v22",
        "@aws-sdk/client-marketplace-vendor-insights-api-v23",
        "@aws-sdk/client-marketplace-vendor-insights-api-v24",
        "@aws-sdk/client-marketplace-vendor-insights-api-v25",
        "@aws-sdk/client-marketplace-vendor-insights-api-v26",
        "@aws-sdk/client-marketplace-vendor-insights-api-v27",
        "@aws-sdk/client-marketplace-vendor-insights-api-v28",
        "@aws-sdk/client-marketplace-vendor-insights-api-v29",
        "cdktf",
        "cdktf-cli",
        "cdktf-provider-aws",
        "cdktf-provider-kubernetes",
        "cdktf-provider-docker",
        "cdktf-provider-gcp",
        "cdktf-provider-azure",
        "cdktf-provider-null",
        "cdktf-provider-random",
        "cdktf-provider-time",
        "cdktf-provider-template",
        "cdktf-provider-local",
        "cdktf-provider-external",
        "cdktf-provider-http",
        "cdktf-provider-tls",
        "cdktf-provider-vault",
        "cdktf-provider-consul",
        "cdktf-provider-nomad",
        "cdktf-provider-packer",
        "cdktf-provider-terraform",
        "cdktf-provider-terraform-aws-modules",
        "cdktf-provider-terraform-aws-modules-vpc",
        "cdktf-provider-terraform-aws-modules-ec2-instance",
        "cdktf-provider-terraform-aws-modules-rds",
        "cdktf-provider-terraform-aws-modules-eks",
        "cdktf-provider-terraform-aws-modules-ecs",
        "cdktf-provider-terraform-aws-modules-lambda",
        "cdktf-provider-terraform-aws-modules-s3-bucket",
        "cdktf-provider-terraform-aws-modules-iam",
        "cdktf-provider-terraform-aws-modules-security-group",
        "cdktf-provider-terraform-aws-modules-alb",
        "cdktf-provider-terraform-aws-modules-nlb",
        "cdktf-provider-terraform-aws-modules-cloudfront",
        "cdktf-provider-terraform-aws-modules-route53",
        "cdktf-provider-terraform-aws-modules-sns",
        "cdktf-provider-terraform-aws-modules-sqs",
        "cdktf-provider-terraform-aws-modules-dynamodb",
        "cdktf-provider-terraform-aws-modules-elasticache",
        "cdktf-provider-terraform-aws-modules-elasticsearch",
        "cdktf-provider-terraform-aws-modules-opensearch",
        "cdktf-provider-terraform-aws-modules-redshift",
        "cdktf-provider-terraform-aws-modules-kinesis",
        "cdktf-provider-terraform-aws-modules-firehose",
        "cdktf-provider-terraform-aws-modules-api-gateway",
        "cdktf-provider-terraform-aws-modules-apigateway-v2",
        "cdktf-provider-terraform-aws-modules-appsync",
        "cdktf-provider-terraform-aws-modules-cognito",
        "cdktf-provider-terraform-aws-modules-secrets-manager",
        "cdktf-provider-terraform-aws-modules-ssm",
        "cdktf-provider-terraform-aws-modules-systems-manager",
        "cdktf-provider-terraform-aws-modules-codecommit",
        "cdktf-provider-terraform-aws-modules-codebuild",
        "cdktf-provider-terraform-aws-modules-codedeploy",
        "cdktf-provider-terraform-aws-modules-codepipeline",
        "cdktf-provider-terraform-aws-modules-codeartifact",
        "cdktf-provider-terraform-aws-modules-codeguru",
        "cdktf-provider-terraform-aws-modules-codeguru-reviewer",
        "cdktf-provider-terraform-aws-modules-codeguru-profiler",
        "cdktf-provider-terraform-aws-modules-device-farm",
        "cdktf-provider-terraform-aws-modules-devops-guru",
        "cdktf-provider-terraform-aws-modules-xray",
        "cdktf-provider-terraform-aws-modules-cloudtrail",
        "cdktf-provider-terraform-aws-modules-config",
        "cdktf-provider-terraform-aws-modules-securityhub",
        "cdktf-provider-terraform-aws-modules-guardduty",
        "cdktf-provider-terraform-aws-modules-macie",
        "cdktf-provider-terraform-aws-modules-macie2",
        "cdktf-provider-terraform-aws-modules-inspector",
        "cdktf-provider-terraform-aws-modules-inspector2",
        "cdktf-provider-terraform-aws-modules-shield",
        "cdktf-provider-terraform-aws-modules-waf",
        "cdktf-provider-terraform-aws-modules-wafv2",
        "cdktf-provider-terraform-aws-modules-network-firewall",
        "cdktf-provider-terraform-aws-modules-networkmanager",
        "cdktf-provider-terraform-aws-modules-directory-service",
        "cdktf-provider-terraform-aws-modules-organizations",
        "cdktf-provider-terraform-aws-modules-resource-groups",
        "cdktf-provider-terraform-aws-modules-resource-groups-tagging",
        "cdktf-provider-terraform-aws-modules-tagging",
        "cdktf-provider-terraform-aws-modules-cost-explorer",
        "cdktf-provider-terraform-aws-modules-billing",
        "cdktf-provider-terraform-aws-modules-budgets",
    ],
    "terraform": [
        "hashicorp/aws",
        "hashicorp/azurerm",
        "hashicorp/google",
        "hashicorp/kubernetes",
        "hashicorp/helm",
        "hashicorp/docker",
        "hashicorp/null",
        "hashicorp/random",
        "hashicorp/time",
        "hashicorp/template",
        "hashicorp/local",
        "hashicorp/external",
        "hashicorp/http",
        "hashicorp/tls",
        "hashicorp/vault",
        "hashicorp/consul",
        "hashicorp/nomad",
        "hashicorp/packer",
        "hashicorp/terraform",
        "hashicorp/terraform-aws-modules",
        "terraform-aws-modules/vpc",
        "terraform-aws-modules/ec2-instance",
        "terraform-aws-modules/rds",
        "terraform-aws-modules/eks",
        "terraform-aws-modules/ecs",
        "terraform-aws-modules/lambda",
        "terraform-aws-modules/s3-bucket",
        "terraform-aws-modules/iam",
        "terraform-aws-modules/security-group",
        "terraform-aws-modules/alb",
        "terraform-aws-modules/nlb",
        "terraform-aws-modules/cloudfront",
        "terraform-aws-modules/route53",
        "terraform-aws-modules/sns",
        "terraform-aws-modules/sqs",
        "terraform-aws-modules/dynamodb",
        "terraform-aws-modules/elasticache",
        "terraform-aws-modules/elasticsearch",
        "terraform-aws-modules/opensearch",
        "terraform-aws-modules/redshift",
        "terraform-aws-modules/kinesis",
        "terraform-aws-modules/firehose",
        "terraform-aws-modules/api-gateway",
        "terraform-aws-modules/apigateway-v2",
        "terraform-aws-modules/appsync",
        "terraform-aws-modules/cognito",
        "terraform-aws-modules/secrets-manager",
        "terraform-aws-modules/ssm",
        "terraform-aws-modules/systems-manager",
        "terraform-aws-modules/codecommit",
        "terraform-aws-modules/codebuild",
        "terraform-aws-modules/codedeploy",
        "terraform-aws-modules/codepipeline",
        "terraform-aws-modules/codeartifact",
        "terraform-aws-modules/codeguru",
        "terraform-aws-modules/codeguru-reviewer",
        "terraform-aws-modules/codeguru-profiler",
        "terraform-aws-modules/device-farm",
        "terraform-aws-modules/devops-guru",
        "terraform-aws-modules/xray",
        "terraform-aws-modules/cloudtrail",
        "terraform-aws-modules/config",
        "terraform-aws-modules/securityhub",
        "terraform-aws-modules/guardduty",
        "terraform-aws-modules/macie",
        "terraform-aws-modules/macie2",
        "terraform-aws-modules/inspector",
        "terraform-aws-modules/inspector2",
        "terraform-aws-modules/shield",
        "terraform-aws-modules/waf",
        "terraform-aws-modules/wafv2",
        "terraform-aws-modules/network-firewall",
        "terraform-aws-modules/networkmanager",
        "terraform-aws-modules/directory-service",
        "terraform-aws-modules/organizations",
        "terraform-aws-modules/resource-groups",
        "terraform-aws-modules/resource-groups-tagging",
        "terraform-aws-modules/tagging",
        "terraform-aws-modules/cost-explorer",
        "terraform-aws-modules/billing",
        "terraform-aws-modules/budgets",
    ],
}

def get_all_packages() -> list[dict[str, str]]:
    """Get all packages as list of dictionaries.

    Returns:
        List of package dictionaries with registry and name
    """
    packages = []
    for registry, package_names in CORE_PACKAGES.items():
        for package_name in package_names:
            packages.append({"registry": registry, "name": package_name})
    return packages

def get_packages_by_registry(registry: str) -> list[str]:
    """Get packages for a specific registry.

    Args:
        registry: Registry name (pypi, npm, terraform)

    Returns:
        List of package names
    """
    return CORE_PACKAGES.get(registry, [])


def get_expanded_packages() -> dict[str, list[str]]:
    """Get expanded package lists (3,000-5,000 packages).

    Returns:
        Dictionary mapping registry to expanded package list
    """
    try:
        from data_pipelines.package_lists.expanded_packages import get_expanded_packages as _get_expanded
        return _get_expanded()
    except ImportError:
        return CORE_PACKAGES


def get_all_expanded_packages() -> list[dict[str, str]]:
    """Get all expanded packages as list of dictionaries.

    Returns:
        List of package dictionaries with registry and name
    """
    expanded = get_expanded_packages()
    packages = []
    for registry, package_names in expanded.items():
        for package_name in package_names:
            packages.append({"registry": registry, "name": package_name})
    return packages

