"""WISTX MCP Server main entry point."""

import asyncio
import hashlib
import hmac
import json
import logging
import re
import signal
import sys
import time
import uuid
from contextvars import ContextVar
from typing import Any
import httpx

from wistx_mcp.config import settings
from wistx_mcp.tools.lib.auth_context import AuthContext, get_auth_context, set_auth_context
from wistx_mcp.tools.lib.concurrent_limiter import get_concurrent_limiter
from wistx_mcp.tools.lib.input_sanitizer import sanitize_tool_arguments
from wistx_mcp.tools.lib.logging_utils import safe_json_dumps, sanitize_error_message, sanitize_arguments
from wistx_mcp.tools.lib.mcp_errors import MCPError, MCPErrorCode
from wistx_mcp.tools.lib.request_context import (
    get_request_context,
    set_request_context,
    update_request_context,
)
from wistx_mcp.tools.lib.rate_limiter import get_rate_limiter
from wistx_mcp.tools.lib.request_deduplicator import get_request_deduplicator
from wistx_mcp.tools.lib.resource_manager import get_resource_manager
from wistx_mcp.tools.lib.constants import (
    MAX_REQUEST_SIZE_BYTES,
    GLOBAL_TOOL_TIMEOUT_SECONDS,
    MAX_RATE_LIMIT_CALLS,
    RATE_LIMIT_WINDOW_SECONDS,
    MAX_CONCURRENT_TOOLS,
    REQUEST_DEDUPLICATION_TTL_SECONDS,
    MAX_REQUEST_ID_LENGTH,
    TOOL_TIMEOUTS,
)

request_id_var: ContextVar[str] = ContextVar("request_id", default="")

from wistx_mcp.tools.lib.protocol_handler import (
    SUPPORTED_PROTOCOL_VERSIONS,
    DEFAULT_PROTOCOL_VERSION,
)

# Phase 2 & Phase 3 Imports
from wistx_mcp.tools.lib.lazy_tool_loader import LazyToolLoader
from wistx_mcp.tools.lib.rate_limit_headers import RateLimitHeaders
from wistx_mcp.tools.lib.tool_analytics import ToolAnalytics
from wistx_mcp.tools.lib.distributed_tool_cache import DistributedToolCache
from wistx_mcp.tools.lib.comprehensive_audit_logger import ComprehensiveAuditLogger

VALID_RESOURCE_TYPES = {"repository", "documentation", "document", "health"}
RESOURCE_ID_PATTERN = re.compile(r"^[a-zA-Z0-9_-]+$")
REQUEST_ID_PATTERN = re.compile(r"^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$")

_inflight_requests: set[str] = set()
_inflight_lock = asyncio.Lock()

logging.basicConfig(
    level=getattr(logging, str(settings.log_level).upper(), logging.INFO),
    format="%(asctime)s - %(name)s - %(levelname)s - %(message)s",
    datefmt="%Y-%m-%d %H:%M:%S",
)

logger = logging.getLogger(__name__)

from wistx_mcp.tools.lib.logging_utils import SensitiveDataFilter

logging.getLogger().addFilter(SensitiveDataFilter())


async def main() -> None:
    """Main entry point for MCP server."""
    from mcp.server import Server
    from mcp.server.stdio import stdio_server
    from mcp.types import Tool, TextContent
    from wistx_mcp.tools import (
        mcp_tools,
        user_indexing,
        web_search,
        search_codebase,
        regex_search,
        package_search,
        design_architecture,
        troubleshoot_issue,
        generate_documentation,
        manage_integration,
        manage_infrastructure,
        infrastructure_context,
        code_examples,
    )

    logger.info("Starting WISTX MCP Server v%s", settings.server_version)
    logger.info("Server: %s", settings.server_name)

    resource_manager = await get_resource_manager()
    shutdown_event = resource_manager.get_shutdown_event()

    rate_limiter = None
    deduplicator = None
    concurrent_limiter = None

    # Phase 2 & Phase 3 Initialization
    lazy_tool_loader = LazyToolLoader()
    rate_limit_headers = RateLimitHeaders()
    tool_analytics = ToolAnalytics()

    # Phase 3: Initialize Redis client for distributed cache
    redis_client = None
    try:
        import redis
        redis_url = settings.redis_url or "redis://localhost:6379"
        redis_client = redis.from_url(redis_url, decode_responses=True)
        redis_client.ping()
        logger.info("Redis connection established: %s", redis_url)
    except Exception as e:
        logger.warning("Redis connection failed, using local cache fallback: %s", e)
        redis_client = None

    distributed_cache = DistributedToolCache(redis_client=redis_client, ttl=3600)
    audit_logger = ComprehensiveAuditLogger(retention_days=90)

    try:
        rate_limiter = await get_rate_limiter(
            max_calls=MAX_RATE_LIMIT_CALLS,
            window_seconds=RATE_LIMIT_WINDOW_SECONDS,
        )
        deduplicator = await get_request_deduplicator(
            ttl_seconds=REQUEST_DEDUPLICATION_TTL_SECONDS,
        )
        concurrent_limiter = await get_concurrent_limiter(
            max_concurrent=MAX_CONCURRENT_TOOLS,
        )
        logger.info("Background tasks started successfully")
    except Exception as e:
        logger.error("Failed to start background tasks: %s", e, exc_info=True)
        raise

    def signal_handler(signum: int, _frame: Any) -> None:
        """Handle shutdown signals."""
        logger.info("Received signal %d, initiating graceful shutdown...", signum)
        shutdown_event.set()

    if sys.platform != "win32":
        signal.signal(signal.SIGINT, signal_handler)
        signal.signal(signal.SIGTERM, signal_handler)
    else:
        logger.warning("Signal handlers not available on Windows")

    resource_manager.register_http_client(mcp_tools.api_client)

    from wistx_mcp.tools.lib.api_client import get_api_client
    try:
        global_api_client = await get_api_client()
        resource_manager.register_http_client(global_api_client)
    except Exception as e:
        logger.debug("Could not register global API client: %s", e)

    try:
        app = Server("wistx-mcp")
        logger.info("MCP Server instance created successfully")
    except (RuntimeError, ValueError, AttributeError) as e:
        logger.error("Failed to create MCP Server: %s", e, exc_info=True)
        raise
    except Exception as e:
        logger.error("Unexpected error creating MCP Server: %s", e, exc_info=True)
        raise RuntimeError(f"Failed to create MCP Server: {e}") from e

    if settings.api_key:
        try:
            auth_ctx = AuthContext(api_key=settings.api_key)
            await auth_ctx.validate()
            set_auth_context(auth_ctx)
            logger.info("Authentication successful for user: %s", auth_ctx.get_user_id())
        except Exception as e:
            logger.warning("Failed to validate API key from settings: %s", e)

    try:
        @app.on_initialize()
        async def on_initialize(params: dict[str, Any]) -> dict[str, Any]:
            """Handle MCP server initialization.

            Args:
                params: Initialization parameters from client

            Returns:
                Server capabilities and information

            Raises:
                MCPError: If protocol version is unsupported or initialization fails
            """
            request_id = str(uuid.uuid4())
            request_id_var.set(request_id)
            logger.info("MCP server initialization [request_id=%s]", request_id)

            from wistx_mcp.tools.lib.protocol_handler import validate_protocol_version

            client_protocol = params.get("protocolVersion", DEFAULT_PROTOCOL_VERSION)
            try:
                validated_version = validate_protocol_version(client_protocol)
            except ValueError as e:
                raise MCPError(
                    code=MCPErrorCode.INVALID_REQUEST,
                    message=str(e),
                ) from e

            supported_version = validated_version

            set_request_context({
                "request_id": request_id,
                "protocol_version": supported_version,
            })

            init_options = params.get("initializationOptions", {}) or {}
            api_key = init_options.get("api_key") or settings.api_key

            if api_key:
                try:
                    auth_ctx = AuthContext(api_key=api_key)
                    await auth_ctx.validate()
                    set_auth_context(auth_ctx)
                    logger.info(
                        "Authentication successful via initialization for user: %s [request_id=%s]",
                        auth_ctx.get_user_id(),
                        request_id,
                    )
                except ValueError as e:
                    logger.warning("Failed to validate API key from initialization: %s [request_id=%s]", e, request_id)
                    raise MCPError(
                        code=MCPErrorCode.INVALID_PARAMS,
                        message=f"Invalid API key: {sanitize_error_message(e)}",
                    ) from e
                except (RuntimeError, ConnectionError) as e:
                    logger.warning("Authentication service unavailable: %s [request_id=%s]", e, request_id)
                    raise MCPError(
                        code=MCPErrorCode.INTERNAL_ERROR,
                        message="Authentication service temporarily unavailable",
                    ) from e

            from wistx_mcp.tools.lib.protocol_handler import ensure_protocol_compliance

            response = {
                "protocolVersion": supported_version,
                "capabilities": {
                    "tools": {},
                    "resources": {
                        "subscribe": True,
                        "listChanged": True,
                    },
                    "prompts": {},
                },
                "serverInfo": {
                    "name": settings.server_name,
                    "version": settings.server_version,
                },
            }

            return ensure_protocol_compliance(response, protocol_version=supported_version)
    except AttributeError:
        logger.warning("MCP SDK version may not support on_initialize decorator, skipping initialization handler")

    @app.list_resources()
    async def list_resources() -> list[dict[str, Any]]:
        """List available MCP resources."""
        from mcp.types import Resource

        auth_ctx = get_auth_context()
        if not auth_ctx:
            return []

        user_id = auth_ctx.get_user_id() or "anonymous"
        rate_limiter_instance = await get_rate_limiter()
        if not await rate_limiter_instance.check_rate_limit(f"resource:{user_id}"):
            logger.warning("Rate limit exceeded for resource listing: user_id=%s", user_id)
            return []

        try:
            from wistx_mcp.tools import user_indexing
            from wistx_mcp.tools.lib.constants import RESOURCE_OPERATION_TIMEOUT_SECONDS

            result = await asyncio.wait_for(
                user_indexing.list_resources(api_key=auth_ctx.api_key),
                timeout=RESOURCE_OPERATION_TIMEOUT_SECONDS,
            )
            resources_list = result.get("resources", [])

            mcp_resources = []
            for resource in resources_list:
                resource_type = resource.get("resource_type", "unknown")
                resource_id = resource.get("resource_id", "")
                name = resource.get("name") or resource.get("repo_url") or resource.get("documentation_url") or resource_id

                uri = f"wistx://{resource_type}/{resource_id}"
                mcp_resources.append(
                    Resource(
                        uri=uri,
                        name=name,
                        description=resource.get("description", ""),
                        mimeType="application/json",
                    )
                )

            mcp_resources.append(
                Resource(
                    uri="wistx://health",
                    name="Server Health",
                    description="MCP server health status",
                    mimeType="application/json",
                )
            )

            mcp_resources.append(
                Resource(
                    uri="wistx://metrics",
                    name="Server Metrics",
                    description="MCP server metrics and performance statistics",
                    mimeType="application/json",
                )
            )

            # Phase 3: Add cache statistics resource
            mcp_resources.append(
                Resource(
                    uri="wistx://cache-stats",
                    name="Cache Statistics",
                    description="Distributed cache statistics and performance metrics",
                    mimeType="application/json",
                )
            )

            # Phase 3: Add audit logs resource
            mcp_resources.append(
                Resource(
                    uri="wistx://audit-logs",
                    name="Audit Logs",
                    description="Comprehensive audit logs for tool calls, errors, and authentication events",
                    mimeType="application/json",
                )
            )

            from wistx_mcp.tools.lib.protocol_handler import ensure_protocol_compliance

            protocol_version = get_request_context().get("protocol_version")
            compliant_resources = ensure_protocol_compliance(mcp_resources, protocol_version=protocol_version)

            return compliant_resources
        except asyncio.TimeoutError:
            logger.error("Timeout listing MCP resources")
            return []
        except (ValueError, RuntimeError, ConnectionError) as e:
            logger.error("Error listing MCP resources: %s", e, exc_info=True)
            return []
        except Exception as e:
            logger.error("Unexpected error listing MCP resources: %s", e, exc_info=True)
            return []

    def validate_resource_uri(uri: str) -> tuple[str, str]:
        """Validate and parse resource URI.

        Args:
            uri: Resource URI to validate

        Returns:
            Tuple of (resource_type, resource_id)

        Raises:
            ValueError: If URI is invalid
        """
        if not uri.startswith("wistx://"):
            raise ValueError(f"Invalid resource URI: {uri}")

        parts = uri.replace("wistx://", "").split("/", 1)
        if len(parts) != 2:
            raise ValueError(f"Invalid resource URI format: {uri}")

        resource_type, resource_id = parts

        if resource_type not in VALID_RESOURCE_TYPES:
            raise ValueError(f"Invalid resource type: {resource_type}. Valid types: {', '.join(VALID_RESOURCE_TYPES)}")

        if not RESOURCE_ID_PATTERN.match(resource_id):
            raise ValueError(f"Invalid resource ID format: {resource_id}")

        return resource_type, resource_id

    def validate_request_id(request_id: str) -> str:
        """Validate and normalize request ID.

        Args:
            request_id: Request ID to validate

        Returns:
            Validated request ID

        Raises:
            ValueError: If request ID is invalid
        """
        if not isinstance(request_id, str):
            raise ValueError("Request ID must be a string")

        if len(request_id) > MAX_REQUEST_ID_LENGTH:
            raise ValueError(f"Request ID too long (max {MAX_REQUEST_ID_LENGTH} characters)")

        if not (REQUEST_ID_PATTERN.match(request_id) or request_id.replace("-", "").replace("_", "").isalnum()):
            raise ValueError("Invalid request ID format")

        return request_id

    @app.read_resource()
    async def read_resource(uri: str) -> str:
        """Read resource content."""
        auth_ctx = get_auth_context()
        if auth_ctx:
            user_id = auth_ctx.get_user_id() or "anonymous"
            rate_limiter_instance = await get_rate_limiter()
            if not await rate_limiter_instance.check_rate_limit(f"resource:{user_id}"):
                raise MCPError(
                    code=MCPErrorCode.RATE_LIMIT_EXCEEDED,
                    message="Rate limit exceeded for resource operations",
                )

        if uri == "wistx://metrics":
            from wistx_mcp.tools.lib.metrics_export import export_json_metrics, export_prometheus_metrics

            format_type = uri.split("?format=")[1] if "?format=" in uri else "json"

            if format_type == "prometheus":
                return export_prometheus_metrics()
            else:
                metrics_data = export_json_metrics()
                return json.dumps(metrics_data, indent=2)

        if uri == "wistx://health":
            from wistx_mcp.tools.lib.mongodb_client import get_mongodb_client
            from wistx_mcp.tools.lib.protocol_handler import ensure_protocol_compliance

            health_status = {
                "status": "healthy",
                "server": settings.server_name,
                "version": settings.server_version,
                "timestamp": time.time(),
                "components": {},
            }

            try:
                from wistx_mcp.tools.lib.metrics_export import export_summary_metrics
                metrics_summary = export_summary_metrics()
                health_status["metrics"] = metrics_summary
            except Exception as e:
                logger.debug("Could not export metrics: %s", e)

            try:
                mongodb_client = await get_mongodb_client()
                mongodb_healthy = await mongodb_client.health_check()
                health_status["components"]["mongodb"] = {
                    "status": "healthy" if mongodb_healthy else "unhealthy",
                }
            except Exception as e:
                logger.warning("Error checking MongoDB health: %s", e)
                health_status["components"]["mongodb"] = {
                    "status": "error",
                    "error": str(e),
                }

            try:
                rate_limiter_instance = await get_rate_limiter()
                health_status["components"]["rate_limiter"] = {
                    "status": "operational",
                    "tracked_identifiers": len(rate_limiter_instance.entries),
                }
            except Exception as e:
                logger.warning("Error checking rate limiter health: %s", e)
                health_status["components"]["rate_limiter"] = {
                    "status": "error",
                    "error": str(e),
                }

            component_statuses = [c.get("status") for c in health_status["components"].values()]
            if "error" in component_statuses:
                health_status["status"] = "degraded"
            elif "unhealthy" in component_statuses:
                health_status["status"] = "degraded"

            protocol_version = get_request_context().get("protocol_version")
            compliant_health = ensure_protocol_compliance(health_status, protocol_version=protocol_version)

            return json.dumps(compliant_health, indent=2)

        # Phase 3: Cache statistics resource
        if uri == "wistx://cache-stats":
            try:
                cache_stats = distributed_cache.get_cache_stats()
                return json.dumps({
                    "cache_stats": cache_stats,
                    "timestamp": time.time(),
                    "cache_type": "distributed",
                    "redis_enabled": redis_client is not None,
                }, indent=2)
            except Exception as e:
                logger.error("Error reading cache stats: %s", e)
                return json.dumps({
                    "error": "Failed to retrieve cache statistics",
                    "timestamp": time.time(),
                }, indent=2)

        # Phase 3: Audit logs resource
        if uri == "wistx://audit-logs":
            try:
                # Get query parameters for filtering
                import urllib.parse
                query_params = {}
                if "?" in uri:
                    query_string = uri.split("?")[1]
                    query_params = urllib.parse.parse_qs(query_string)

                user_id = query_params.get("user_id", [None])[0]
                limit = int(query_params.get("limit", [100])[0])

                audit_logs = audit_logger.get_logs(user_id=user_id, limit=limit)
                return json.dumps({
                    "audit_logs": audit_logs,
                    "count": len(audit_logs),
                    "timestamp": time.time(),
                    "filtered_by_user": user_id is not None,
                }, indent=2)
            except Exception as e:
                logger.error("Error reading audit logs: %s", e)
                return json.dumps({
                    "error": "Failed to retrieve audit logs",
                    "timestamp": time.time(),
                }, indent=2)

        auth_ctx = get_auth_context()
        if not auth_ctx:
            raise MCPError(
                code=MCPErrorCode.INVALID_REQUEST,
                message="Authentication required to read resources",
            )

        try:
            _resource_type, resource_id = validate_resource_uri(uri)
        except ValueError as e:
            raise MCPError(
                code=MCPErrorCode.INVALID_PARAMS,
                message=str(e),
            ) from e

        from wistx_mcp.tools import user_indexing
        from wistx_mcp.tools.lib.constants import RESOURCE_OPERATION_TIMEOUT_SECONDS

        try:
            result = await asyncio.wait_for(
                user_indexing.check_resource_status(
                    resource_id=resource_id,
                    api_key=auth_ctx.api_key,
                ),
                timeout=RESOURCE_OPERATION_TIMEOUT_SECONDS,
            )

            from wistx_mcp.tools.lib.protocol_handler import ensure_protocol_compliance

            protocol_version = get_request_context().get("protocol_version")
            compliant_result = ensure_protocol_compliance(result, protocol_version=protocol_version)

            return json.dumps(compliant_result, indent=2)
        except asyncio.TimeoutError:
            logger.error("Timeout reading MCP resource %s", uri)
            raise MCPError(
                code=MCPErrorCode.TIMEOUT,
                message=f"Resource operation timed out after {RESOURCE_OPERATION_TIMEOUT_SECONDS} seconds",
            )
        except ValueError as e:
            logger.error("Error reading MCP resource %s: %s", uri, e, exc_info=True)
            raise MCPError(
                code=MCPErrorCode.INVALID_PARAMS,
                message=f"Failed to read resource: {sanitize_error_message(e)}",
            ) from e
        except (RuntimeError, ConnectionError) as e:
            logger.error("Error reading MCP resource %s: %s", uri, e, exc_info=True)
            raise MCPError(
                code=MCPErrorCode.INTERNAL_ERROR,
                message="Failed to read resource due to internal error",
            ) from e
        except Exception as e:
            logger.error("Unexpected error reading MCP resource %s: %s", uri, e, exc_info=True)
            raise MCPError(
                code=MCPErrorCode.INTERNAL_ERROR,
                message="An unexpected error occurred while reading resource",
            ) from e

    try:
        @app.subscribe()
        async def subscribe(uri: str) -> None:
            """Subscribe to resource change notifications.

            Args:
                uri: Resource URI to subscribe to
            """
            auth_ctx = get_auth_context()
            if not auth_ctx:
                raise MCPError(
                    code=MCPErrorCode.INVALID_REQUEST,
                    message="Authentication required to subscribe to resources",
                )

            try:
                validate_resource_uri(uri)

                user_id = auth_ctx.get_user_id() or "unknown"

                resource_manager_instance = await get_resource_manager()
                await resource_manager_instance.subscribe(uri, user_id)

                logger.info(
                    "User %s subscribed to resource %s [request_id=%s]",
                    user_id[:8] if len(user_id) > 8 else user_id,
                    uri,
                    request_id_var.get(),
                )
            except ValueError as e:
                raise MCPError(
                    code=MCPErrorCode.INVALID_PARAMS,
                    message=f"Invalid resource URI: {sanitize_error_message(e)}",
                ) from e
            except Exception as e:
                logger.error("Error subscribing to resource %s: %s", uri, e, exc_info=True)
                raise MCPError(
                    code=MCPErrorCode.INTERNAL_ERROR,
                    message="Failed to subscribe to resource",
                ) from e
    except AttributeError:
        logger.warning("MCP SDK version may not support subscribe decorator, skipping subscription handler")

    try:
        @app.list_prompts()
        async def list_prompts() -> list[dict[str, Any]]:
            """List available MCP prompts."""
            from mcp.types import Prompt

            prompts = [
                Prompt(
                    name="generate_compliance_report",
                    description="Generate a compliance report for infrastructure resources",
                    arguments=[
                        {
                            "name": "resource_types",
                            "description": "List of resource types (RDS, S3, EC2, etc.)",
                            "required": True,
                        },
                        {
                            "name": "standards",
                            "description": "Compliance standards (PCI-DSS, HIPAA, etc.)",
                            "required": False,
                        },
                    ],
                ),
                Prompt(
                    name="calculate_infrastructure_costs",
                    description="Calculate costs for infrastructure resources",
                    arguments=[
                        {
                            "name": "resources",
                            "description": "List of resource specifications",
                            "required": True,
                        },
                    ],
                ),
            ]

            from wistx_mcp.tools.lib.protocol_handler import ensure_protocol_compliance

            protocol_version = get_request_context().get("protocol_version")
            compliant_prompts = ensure_protocol_compliance(prompts, protocol_version=protocol_version)

            return compliant_prompts
    except AttributeError:
        logger.warning("MCP SDK version may not support list_prompts decorator, skipping prompts handler")

    try:
        @app.get_prompt()
        async def get_prompt(name: str, arguments: dict[str, Any]) -> list[TextContent]:
            """Execute a prompt template.

            Args:
                name: Prompt name
                arguments: Prompt arguments

            Returns:
                List of TextContent results
            """
            auth_ctx = get_auth_context()
            if not auth_ctx:
                raise MCPError(
                    code=MCPErrorCode.INVALID_REQUEST,
                    message="Authentication required to execute prompts",
                )

            request_id = request_id_var.get() or str(uuid.uuid4())
            request_id_var.set(request_id)

            try:
                if name == "generate_compliance_report":
                    resource_types = arguments.get("resource_types", [])
                    if not resource_types:
                        raise MCPError(
                            code=MCPErrorCode.INVALID_PARAMS,
                            message="resource_types is required",
                        )

                    standards = arguments.get("standards", [])

                    from wistx_mcp.tools import mcp_tools

                    result = await mcp_tools.get_compliance_requirements(
                        resource_types=resource_types,
                        standards=standards,
                        api_key=auth_ctx.api_key,
                        include_remediation=True,
                        include_verification=True,
                        generate_report=True,
                    )

                    from wistx_mcp.tools.lib.context_builder import ContextBuilder

                    resource_type = resource_types[0] if resource_types else None
                    controls = result.get("controls", []) or result.get("data", {}).get("controls", [])

                    if controls:
                        markdown_result = ContextBuilder.format_compliance_as_toon(controls, resource_type)
                    else:
                        markdown_result = "No compliance controls found for the specified criteria."

                    if "report_id" in result:
                        markdown_result += "\n\n---\n\n**Report Generated:**\n"
                        markdown_result += f"- Report ID: `{result['report_id']}`\n"
                        if result.get("report_view_url"):
                            markdown_result += f"- View Report: {result['report_view_url']}\n"
                        if result.get("report_download_url"):
                            markdown_result += f"- Download Report: {result['report_download_url']}\n"

                    from wistx_mcp.tools.lib.protocol_handler import ensure_protocol_compliance

                    protocol_version = get_request_context().get("protocol_version")
                    prompt_result = [TextContent(type="text", text=markdown_result)]
                    compliant_result = ensure_protocol_compliance(prompt_result, protocol_version=protocol_version)

                    return compliant_result

                elif name == "calculate_infrastructure_costs":
                    resources = arguments.get("resources", [])
                    if not resources:
                        raise MCPError(
                            code=MCPErrorCode.INVALID_PARAMS,
                            message="resources is required",
                        )

                    from wistx_mcp.tools import mcp_tools

                    result = await mcp_tools.calculate_infrastructure_cost(
                        resources=resources,
                        api_key=auth_ctx.api_key,
                    )

                    from wistx_mcp.tools.lib.context_builder import ContextBuilder

                    markdown_result = ContextBuilder.format_pricing_as_toon(result)

                    from wistx_mcp.tools.lib.protocol_handler import ensure_protocol_compliance

                    protocol_version = get_request_context().get("protocol_version")
                    prompt_result = [TextContent(type="text", text=markdown_result)]
                    compliant_result = ensure_protocol_compliance(prompt_result, protocol_version=protocol_version)

                    return compliant_result

                else:
                    raise MCPError(
                        code=MCPErrorCode.METHOD_NOT_FOUND,
                        message=f"Unknown prompt: {name}",
                    )

            except MCPError:
                raise
            except ValueError as e:
                raise MCPError(
                    code=MCPErrorCode.INVALID_PARAMS,
                    message=sanitize_error_message(e),
                ) from e
            except Exception as e:
                logger.error("Error executing prompt %s: %s [request_id=%s]", name, e, request_id, exc_info=True)
                raise MCPError(
                    code=MCPErrorCode.INTERNAL_ERROR,
                    message="Failed to execute prompt",
                ) from e
    except AttributeError:
        logger.warning("MCP SDK version may not support get_prompt decorator, skipping prompt execution handler")

    @app.list_tools()
    async def list_tools() -> list[Tool]:
        """List available MCP tools."""
        from mcp.types import Tool
        
        cached_definitions = distributed_cache.get_tool_definitions()
        if cached_definitions:
            try:
                tools = [Tool(**tool_dict) for tool_dict in cached_definitions]
                logger.debug("Loaded tool list from cache (%d tools)", len(tools))
                return tools
            except Exception as e:
                logger.warning("Failed to deserialize cached tools, rebuilding: %s", e)
        
        logger.info("Building tool list (cache miss)")
        try:
            from wistx_mcp.tools.lib.tool_permissions import get_tool_permissions
            from wistx_mcp.tools.lib.tool_descriptions import ToolDescriptionManager
            
            def build_tool_description(tool_name: str, base_description: str | None = None) -> str:
                """Build tool description with plan requirements, using short description from ToolDescriptionManager."""
                short_desc = ToolDescriptionManager.get_short_description(tool_name)
                description = base_description if base_description else short_desc
                
                perms = get_tool_permissions(tool_name)
                required_plan = perms.get("required_plan", "scout")
                required_permission = perms.get("required_permission")
                
                plan_note = ""
                if required_plan != "scout":
                    plan_note = f" [REQUIRES: {required_plan} plan or higher]"
                if required_permission:
                    plan_note += f" [REQUIRES PERMISSION: {required_permission}]"
                
                return f"{description}{plan_note}"
            
            tools = [
            Tool(
                name="wistx_get_recommended_tools",
                description=build_tool_description("wistx_get_recommended_tools"),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "task_description": {
                            "type": "string",
                            "description": "Description of the task or goal you want to accomplish",
                            "minLength": 5,
                            "maxLength": 1000,
                        },
                        "category": {
                            "type": "string",
                            "enum": ["compliance", "pricing", "code", "infrastructure", "knowledge", "indexing", "documentation", "packages", "integration"],
                            "description": "Optional category filter to narrow recommendations",
                        },
                        "max_results": {
                            "type": "integer",
                            "default": 10,
                            "minimum": 1,
                            "maximum": 50,
                            "description": "Maximum number of recommendations to return",
                        },
                    },
                    "required": ["task_description"],
                },
            ),
            Tool(
                name="wistx_list_tools_by_category",
                description=build_tool_description("wistx_list_tools_by_category"),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "category": {
                            "type": "string",
                            "enum": ["compliance", "pricing", "code", "infrastructure", "knowledge", "indexing", "documentation", "packages", "integration"],
                            "description": "Category name to list tools for",
                        },
                        "include_descriptions": {
                            "type": "boolean",
                            "default": True,
                            "description": "Whether to include short descriptions for each tool",
                        },
                    },
                    "required": ["category"],
                },
            ),
            Tool(
                name="wistx_get_tool_documentation",
                description=build_tool_description("wistx_get_tool_documentation"),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "tool_name": {
                            "type": "string",
                            "description": "Name of the tool to get documentation for (e.g., 'wistx_get_compliance_requirements')",
                            "minLength": 1,
                            "maxLength": 100,
                        },
                    },
                    "required": ["tool_name"],
                },
            ),
            Tool(
                name="wistx_research_knowledge_base",
                description=build_tool_description("wistx_research_knowledge_base"),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "query": {
                            "type": "string",
                            "description": "Research query in natural language",
                            "minLength": 10,
                            "maxLength": 10000,
                        },
                        "domains": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Filter by domains: compliance, finops, devops, infrastructure, security, architecture, cloud, automation, platform, sre",
                            "default": [],
                        },
                        "content_types": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Filter by content types: guide, pattern, strategy, checklist, reference, best_practice",
                            "default": [],
                        },
                        "include_cross_domain": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include cross-domain relationships and impacts",
                        },
                        "include_web_search": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include real-time web search results (Tavily) for comprehensive coverage",
                        },
                        "format": {
                            "type": "string",
                            "enum": ["structured", "markdown", "executive_summary"],
                            "default": "markdown",
                            "description": "Response format (default: markdown for optimal LLM consumption)",
                        },
                        "max_results": {
                            "type": "integer",
                            "default": 20,
                            "minimum": 1,
                            "maximum": 100,
                            "description": "Maximum number of results",
                        },
                    },
                    "required": ["query"],
                },
            ),
            Tool(
                name="wistx_web_search",
                description=build_tool_description("wistx_web_search"),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "query": {
                            "type": "string",
                            "description": "Search query",
                            "minLength": 3,
                            "maxLength": 1000,
                        },
                        "search_type": {
                            "type": "string",
                            "enum": ["general", "security"],
                            "default": "general",
                            "description": "Type of search (general includes web search, security focuses on CVEs/advisories)",
                        },
                        "resource_type": {
                            "type": "string",
                            "description": "Filter by resource type (RDS, S3, EKS, GKE, AKS, etc.)",
                        },
                        "cloud_provider": {
                            "type": "string",
                            "enum": ["aws", "gcp", "azure"],
                            "description": "Filter by cloud provider",
                        },
                        "severity": {
                            "type": "string",
                            "enum": ["CRITICAL", "HIGH", "MEDIUM", "LOW"],
                            "description": "Filter by severity (for security searches)",
                        },
                        "include_cves": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include CVE database results",
                        },
                        "include_advisories": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include security advisories",
                        },
                        "limit": {
                            "type": "integer",
                            "default": 20,
                            "minimum": 1,
                            "maximum": 100,
                            "description": "Maximum number of results",
                        },
                    },
                    "required": ["query"],
                },
            ),
            Tool(
                name="wistx_get_compliance_requirements",
                description=build_tool_description("wistx_get_compliance_requirements"),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "resource_types": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "List of resource types (RDS, S3, EC2, Lambda, EKS, etc.)",
                            "minItems": 1,
                        },
                        "standards": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Compliance standards (PCI-DSS, HIPAA, CIS, SOC2, NIST-800-53, ISO-27001, GDPR, FedRAMP, etc.)",
                            "default": [],
                        },
                        "severity": {
                            "type": "string",
                            "enum": ["CRITICAL", "HIGH", "MEDIUM", "LOW"],
                            "description": "Filter by severity level",
                        },
                        "include_remediation": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include remediation guidance and code snippets",
                        },
                        "include_verification": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include verification procedures",
                        },
                        "generate_report": {
                            "type": "boolean",
                            "default": True,
                            "description": "Automatically generate and store a compliance report",
                        },
                    },
                    "required": ["resource_types"],
                },
            ),
            Tool(
                name="wistx_research_knowledge_base",
                description=build_tool_description(
                    "wistx_research_knowledge_base",
                    "Deep research tool for DevOps, infrastructure, compliance, FinOps, and platform engineering knowledge. "
                    "Searches internal knowledge base and optionally includes real-time web search for comprehensive coverage. "
                    "Use this tool for comprehensive research queries about best practices, patterns, strategies, guides, "
                    "and cross-domain insights. Always use this tool for deep research queries."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "query": {
                            "type": "string",
                            "description": "Research query in natural language",
                            "minLength": 10,
                            "maxLength": 10000,
                        },
                        "domains": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Filter by domains: compliance, finops, devops, infrastructure, security, architecture, cloud, automation, platform, sre",
                            "default": [],
                        },
                        "content_types": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Filter by content types: guide, pattern, strategy, checklist, reference, best_practice",
                            "default": [],
                        },
                        "include_cross_domain": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include cross-domain relationships and impacts",
                        },
                        "include_web_search": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include real-time web search results (Tavily) for comprehensive coverage",
                        },
                        "format": {
                            "type": "string",
                            "enum": ["structured", "markdown", "executive_summary"],
                            "default": "markdown",
                            "description": "Response format (default: markdown for optimal LLM consumption)",
                        },
                        "max_results": {
                            "type": "integer",
                            "default": 20,
                            "minimum": 1,
                            "maximum": 100,
                            "description": "Maximum number of results",
                        },
                    },
                    "required": ["query"],
                },
            ),
            Tool(
                name="wistx_calculate_infrastructure_cost",
                description=build_tool_description(
                    "wistx_calculate_infrastructure_cost",
                    "Calculate infrastructure costs for cloud resources. Use this tool when asked about pricing, costs, or cost optimization for AWS/GCP/Azure resources. Returns monthly/annual costs, cost breakdown, and optimization suggestions."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "resources": {
                            "type": "array",
                            "items": {
                                "type": "object",
                                "properties": {
                                    "cloud": {"type": "string", "description": "Cloud provider (aws, gcp, azure)"},
                                    "service": {"type": "string", "description": "Service name (rds, ec2, s3, etc.)"},
                                    "instance_type": {"type": "string", "description": "Instance type (db.t3.medium, etc.)"},
                                    "quantity": {"type": "integer", "description": "Quantity", "default": 1},
                                },
                                "required": ["cloud", "service", "instance_type"],
                            },
                            "description": "List of resource specifications",
                            "minItems": 1,
                        },
                    },
                    "required": ["resources"],
                },
            ),
            Tool(
                name="wistx_index_repository",
                description=build_tool_description(
                    "wistx_index_repository",
                    "Index a GitHub repository for user-specific search. Use this tool when asked to index or search through a GitHub repository. Supports both public and private repositories. "
                    "For public repositories, no GitHub token is needed. For private repositories, the system automatically uses your connected GitHub OAuth token (set up during signup). "
                    "You can optionally provide a github_token parameter for backward compatibility, but it's not required if you've connected GitHub during signup."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "repo_url": {
                            "type": "string",
                            "description": "GitHub repository URL (e.g., https://github.com/owner/repo)",
                        },
                        "branch": {
                            "type": "string",
                            "default": "main",
                            "description": "Branch to index",
                        },
                        "name": {
                            "type": "string",
                            "description": "Custom name for the resource",
                        },
                        "description": {
                            "type": "string",
                            "description": "Resource description",
                        },
                        "tags": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Tags for categorization",
                        },
                        "github_token": {
                            "type": "string",
                            "description": "GitHub personal access token (optional - only needed if you haven't connected GitHub OAuth during signup. For private repos, OAuth token is used automatically if available).",
                        },
                        "include_patterns": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "File path patterns to include (glob patterns, e.g., ['**/terraform/**', '**/*.tf']). If not provided, defaults to DevOps-focused paths.",
                        },
                        "exclude_patterns": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "File path patterns to exclude (glob patterns, e.g., ['**/src/**', '**/app/**']). If not provided, defaults to excluding application code paths.",
                        },
                    },
                    "required": ["repo_url"],
                },
            ),
            Tool(
                name="wistx_index_resource",
                description=build_tool_description(
                    "wistx_index_resource",
                    "Index content (documentation website or document file) for user-specific search. "
                    "Unified tool that handles both website crawling and single file indexing. "
                    "Automatically detects content type based on URL/file extension or explicit content_type parameter. "
                    "Use this tool when asked to index documentation websites, PDFs, DOCX files, or other documents."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "content_url": {
                            "type": "string",
                            "description": "Content URL - can be documentation website URL (for crawling) or document URL/file path (for single file). "
                                          "Examples: 'https://docs.example.com' (website) or 'https://example.com/doc.pdf' (file)",
                        },
                        "file_path": {
                            "type": "string",
                            "description": "Local file path for direct upload (for single file, optional). "
                                          "Example: /Users/john/Documents/compliance.pdf",
                        },
                        "content_type": {
                            "type": "string",
                            "enum": ["documentation", "pdf", "docx", "markdown", "md", "txt"],
                            "description": "Content type: 'documentation' for website crawling, or file type (pdf, docx, etc.) for single files. "
                                        "Auto-detected from file_path or URL extension if not provided.",
                        },
                        "name": {
                            "type": "string",
                            "description": "Custom name for the resource",
                        },
                        "description": {
                            "type": "string",
                            "description": "Resource description",
                        },
                        "tags": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Tags for categorization",
                        },
                        "include_patterns": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "URL patterns to include (for documentation websites only, e.g., ['/docs/', '/api/'])",
                        },
                        "exclude_patterns": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "URL patterns to exclude (for documentation websites only, e.g., ['/admin/', '/private/'])",
                        },
                    },
                    "required": [],
                    "anyOf": [
                        {"required": ["content_url"]},
                        {"required": ["file_path"]},
                    ],
                },
            ),
            Tool(
                name="wistx_list_resources",
                description=build_tool_description(
                    "wistx_list_resources",
                    "List all indexed resources for the user. Use this tool to see what repositories, documentation, or documents have been indexed."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "resource_type": {
                            "type": "string",
                            "enum": ["repository", "documentation", "document"],
                            "description": "Filter by resource type",
                        },
                        "status": {
                            "type": "string",
                            "enum": ["pending", "indexing", "completed", "failed"],
                            "description": "Filter by status",
                        },
                        "include_ai_analysis": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include AI-analyzed insights about resource collection with patterns, health analysis, and recommendations",
                        },
                    },
                    "required": [],
                },
            ),
            Tool(
                name="wistx_check_resource_status",
                description=build_tool_description(
                    "wistx_check_resource_status",
                    "Check indexing status and progress for a specific resource. Use this tool to monitor indexing progress."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "resource_id": {
                            "type": "string",
                            "description": "Resource ID",
                        },
                    },
                    "required": ["resource_id"],
                },
            ),
            Tool(
                name="wistx_delete_resource",
                description=build_tool_description(
                    "wistx_delete_resource",
                    "Delete an indexed resource and all associated knowledge articles. Use this tool to remove indexed content."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "resource_type": {
                            "type": "string",
                            "enum": ["repository", "documentation", "document"],
                            "description": "Type of resource to delete",
                        },
                        "identifier": {
                            "type": "string",
                            "description": "Resource identifier - can be repository URL (e.g., 'owner/repo' or full URL), documentation URL, document URL, or resource_id (e.g., 'res_abc123')",
                        },
                    },
                    "required": ["resource_type", "identifier"],
                },
            ),
            Tool(
                name="wistx_web_search",
                description=build_tool_description(
                    "wistx_web_search",
                    "Web search for security information and general web content. "
                    "Focused on security searches (CVEs, advisories) and general web search via Tavily. "
                    "For compliance requirements, use wistx_get_compliance_requirements tool. "
                    "For deep research with web search, use wistx_research_knowledge_base tool."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "query": {
                            "type": "string",
                            "description": "Search query",
                            "minLength": 3,
                            "maxLength": 1000,
                        },
                        "search_type": {
                            "type": "string",
                            "enum": ["general", "security"],
                            "default": "general",
                            "description": "Type of search (general includes web search, security focuses on CVEs/advisories)",
                        },
                        "resource_type": {
                            "type": "string",
                            "description": "Filter by resource type (RDS, S3, EKS, GKE, AKS, etc.)",
                        },
                        "cloud_provider": {
                            "type": "string",
                            "enum": ["aws", "gcp", "azure"],
                            "description": "Filter by cloud provider",
                        },
                        "severity": {
                            "type": "string",
                            "enum": ["CRITICAL", "HIGH", "MEDIUM", "LOW"],
                            "description": "Filter by severity (for security searches)",
                        },
                        "include_cves": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include CVE database results",
                        },
                        "include_advisories": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include security advisories",
                        },
                        "limit": {
                            "type": "integer",
                            "default": 20,
                            "minimum": 1,
                            "maximum": 100,
                            "description": "Maximum number of results",
                        },
                    },
                    "required": ["query"],
                },
            ),
            Tool(
                name="wistx_search_codebase",
                description=build_tool_description(
                    "wistx_search_codebase",
                    "Search user's indexed codebase including repositories, documentation, "
                    "and documents. Use this tool when asked to search through user's own "
                    "code, documentation, or indexed resources."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "query": {
                            "type": "string",
                            "description": "Natural language search question",
                            "minLength": 3,
                            "maxLength": 1000,
                        },
                        "repositories": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "List of repositories to search (owner/repo format, e.g., ['owner/repo', 'another/org/repo'])",
                        },
                        "resource_ids": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Filter by specific indexed resources (alternative to repositories)",
                        },
                        "resource_types": {
                            "type": "array",
                            "items": {
                                "type": "string",
                                "enum": ["repository", "documentation", "document"],
                            },
                            "description": "Filter by resource type",
                        },
                        "file_types": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Filter by file extensions (.tf, .yaml, .py, .md, etc.)",
                        },
                        "code_type": {
                            "type": "string",
                            "enum": ["terraform", "kubernetes", "docker", "python", "javascript", "yaml"],
                            "description": "Filter by code type",
                        },
                        "cloud_provider": {
                            "type": "string",
                            "enum": ["aws", "gcp", "azure"],
                            "description": "Filter by cloud provider mentioned in code",
                        },
                        "include_sources": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include source code snippets in results",
                        },
                        "include_ai_analysis": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include AI-analyzed results with explanations, code snippets analysis, pattern detection, and recommendations",
                        },
                        "limit": {
                            "type": "integer",
                            "default": 20,
                            "minimum": 1,
                            "maximum": 100,
                            "description": "Maximum number of results",
                        },
                    },
                    "required": ["query"],
                },
            ),
            Tool(
                name="wistx_regex_search",
                description=build_tool_description(
                    "wistx_regex_search",
                    "Search user's indexed codebase using regex patterns. "
                    "Use this tool for exact pattern matching, security audits (finding secrets, API keys), "
                    "compliance checks (finding violations), and code analysis. "
                    "Supports pre-built templates (api_key, password, ip_address, etc.) or custom regex patterns."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "pattern": {
                            "type": "string",
                            "description": "Regular expression pattern to search for (required if template not provided)",
                            "maxLength": 10000,
                        },
                        "template": {
                            "type": "string",
                            "enum": [
                                "api_key",
                                "password",
                                "secret_key",
                                "token",
                                "credential",
                                "private_key",
                                "ssh_key",
                                "aws_access_key",
                                "aws_secret_key",
                                "github_token",
                                "slack_token",
                                "jwt_token",
                                "ip_address",
                                "email",
                                "credit_card",
                                "ssn",
                                "unencrypted_storage",
                                "public_access",
                                "publicly_accessible",
                                "missing_backup",
                                "no_encryption",
                                "no_versioning",
                                "no_logging",
                                "no_mfa",
                                "insecure_protocol",
                                "latest_tag",
                                "no_resource_limits",
                                "hardcoded_port",
                                "hardcoded_url",
                                "function_definition",
                                "class_definition",
                                "terraform_resource",
                                "terraform_data_source",
                                "terraform_variable",
                                "kubernetes_secret",
                                "kubernetes_configmap",
                                "dockerfile_from",
                                "import_statement",
                            ],
                            "description": "Use pre-built pattern template (alternative to custom pattern)",
                        },
                        "repositories": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "List of repositories to search (owner/repo format)",
                        },
                        "resource_ids": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Filter by specific indexed resources",
                        },
                        "resource_types": {
                            "type": "array",
                            "items": {
                                "type": "string",
                                "enum": ["repository", "documentation", "document"],
                            },
                            "description": "Filter by resource type",
                        },
                        "file_types": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Filter by file extensions (.tf, .yaml, .py, .md, etc.)",
                        },
                        "code_type": {
                            "type": "string",
                            "enum": ["terraform", "kubernetes", "docker", "python", "javascript", "yaml"],
                            "description": "Filter by code type",
                        },
                        "cloud_provider": {
                            "type": "string",
                            "enum": ["aws", "gcp", "azure"],
                            "description": "Filter by cloud provider mentioned in code",
                        },
                        "case_sensitive": {
                            "type": "boolean",
                            "default": False,
                            "description": "Case-sensitive matching",
                        },
                        "multiline": {
                            "type": "boolean",
                            "default": False,
                            "description": "Multiline mode (^ and $ match line boundaries)",
                        },
                        "dotall": {
                            "type": "boolean",
                            "default": False,
                            "description": "Dot matches newline",
                        },
                        "include_context": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include surrounding code context",
                        },
                        "context_lines": {
                            "type": "integer",
                            "default": 3,
                            "minimum": 0,
                            "maximum": 10,
                            "description": "Number of lines before/after match",
                        },
                        "limit": {
                            "type": "integer",
                            "default": 100,
                            "minimum": 1,
                            "maximum": 1000,
                            "description": "Maximum number of results",
                        },
                        "timeout": {
                            "type": "number",
                            "default": 30.0,
                            "minimum": 1.0,
                            "maximum": 300.0,
                            "description": "Maximum search time in seconds",
                        },
                    },
                    "anyOf": [
                        {"required": ["pattern"]},
                        {"required": ["template"]},
                    ],
                },
            ),
            Tool(
                name="wistx_design_architecture",
                description=build_tool_description(
                    "wistx_design_architecture",
                    "Design and initialize DevOps/infrastructure/SRE/platform engineering projects. "
                    "Use this tool to scaffold new projects with compliance and security built-in, "
                    "design architectures, review existing architectures, or optimize architectures."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "action": {
                            "type": "string",
                            "enum": ["initialize", "design", "review", "optimize"],
                            "description": "Action to perform",
                        },
                        "project_type": {
                            "type": "string",
                            "enum": ["terraform", "kubernetes", "devops", "platform"],
                            "description": "Type of project (required for initialize)",
                        },
                        "project_name": {
                            "type": "string",
                            "description": "Name of the project (required for initialize)",
                        },
                        "architecture_type": {
                            "type": "string",
                            "enum": ["microservices", "serverless", "monolith", "event-driven"],
                            "description": "Architecture pattern",
                        },
                        "cloud_provider": {
                            "type": "string",
                            "enum": ["aws", "gcp", "azure", "multi-cloud"],
                            "description": "Cloud provider",
                        },
                        "compliance_standards": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Compliance standards to include",
                        },
                        "requirements": {
                            "type": "object",
                            "description": "Project requirements (scalability, availability, security, cost)",
                        },
                        "existing_architecture": {
                            "type": "string",
                            "description": "Existing architecture code/documentation (for review/optimize)",
                        },
                        "output_directory": {
                            "type": "string",
                            "default": ".",
                            "description": "Directory to create project",
                        },
                    },
                    "required": ["action"],
                },
            ),
            Tool(
                name="wistx_troubleshoot_issue",
                description=build_tool_description(
                    "wistx_troubleshoot_issue",
                    "Diagnose and fix infrastructure/code issues. "
                    "Analyzes errors, logs, and code to identify root causes, "
                    "provides fix recommendations, and prevention strategies. "
                    "Use this tool when encountering errors or issues."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "issue_description": {
                            "type": "string",
                            "description": "Description of the issue",
                            "minLength": 10,
                        },
                        "infrastructure_type": {
                            "type": "string",
                            "enum": ["terraform", "kubernetes", "docker", "cloudformation", "ansible"],
                            "description": "Type of infrastructure",
                        },
                        "cloud_provider": {
                            "type": "string",
                            "enum": ["aws", "gcp", "azure"],
                            "description": "Cloud provider",
                        },
                        "error_messages": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "List of error messages",
                        },
                        "configuration_code": {
                            "type": "string",
                            "description": "Relevant configuration code",
                        },
                        "logs": {
                            "type": "string",
                            "description": "Log output",
                        },
                        "resource_type": {
                            "type": "string",
                            "description": "Resource type (RDS, S3, EKS, etc.)",
                        },
                    },
                    "required": ["issue_description"],
                },
            ),
            Tool(
                name="wistx_generate_documentation",
                description=build_tool_description(
                    "wistx_generate_documentation",
                    "Generate comprehensive documentation and reports. "
                    "Creates architecture docs, runbooks, compliance reports, "
                    "security reports, cost reports, API documentation, and deployment guides. "
                    "Use this tool when asked to create documentation or reports."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "document_type": {
                            "type": "string",
                            "enum": [
                                "architecture_diagram",
                                "runbook",
                                "compliance_report",
                                "cost_report",
                                "security_report",
                                "api_documentation",
                                "deployment_guide",
                            ],
                            "description": "Type of document to generate",
                        },
                        "subject": {
                            "type": "string",
                            "description": "Subject of the document (project name, resource, topic)",
                        },
                        "infrastructure_code": {
                            "type": "string",
                            "description": "Infrastructure code to document",
                        },
                        "configuration": {
                            "type": "object",
                            "description": "Configuration to document",
                        },
                        "include_compliance": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include compliance information",
                        },
                        "include_security": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include security information",
                        },
                        "include_cost": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include cost information",
                        },
                        "include_best_practices": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include best practices",
                        },
                        "resource_types": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "List of resource types (for compliance/security reports)",
                        },
                        "compliance_standards": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "List of compliance standards (for compliance report)",
                        },
                        "resources": {
                            "type": "array",
                            "items": {"type": "object"},
                            "description": "List of resource specifications (for cost report)",
                        },
                        "api_spec": {
                            "type": "object",
                            "description": "API specification (for api_documentation)",
                        },
                        "format": {
                            "type": "string",
                            "enum": ["markdown", "pdf", "html", "json"],
                            "default": "markdown",
                            "description": "Output format",
                        },
                    },
                    "required": ["document_type", "subject"],
                },
            ),
            Tool(
                name="wistx_manage_integration",
                description=build_tool_description(
                    "wistx_manage_integration",
                    "Analyze, generate, and validate infrastructure component integrations. "
                    "Handles networking, security, monitoring, and service integrations. "
                    "Use this tool when working with component integrations."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "action": {
                            "type": "string",
                            "enum": ["analyze", "generate", "validate"],
                            "description": "Action to perform (analyze, generate, validate)",
                        },
                        "infrastructure_code": {
                            "type": "string",
                            "description": "Infrastructure code to analyze (for analyze action)",
                        },
                        "components": {
                            "type": "array",
                            "items": {"type": "object"},
                            "description": "List of components to integrate (for generate action)",
                        },
                        "integration_type": {
                            "type": "string",
                            "enum": ["networking", "security", "monitoring", "service"],
                            "description": "Type of integration",
                        },
                        "cloud_provider": {
                            "type": "string",
                            "enum": ["aws", "gcp", "azure"],
                            "description": "Cloud provider",
                        },
                        "compliance_standards": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Compliance standards to consider",
                        },
                        "pattern_name": {
                            "type": "string",
                            "description": "Specific integration pattern to use (optional)",
                        },
                    },
                    "required": ["action"],
                },
            ),
            Tool(
                name="wistx_search_packages",
                description=build_tool_description(
                    "wistx_search_packages",
                    "Search DevOps/infrastructure packages across PyPI, NPM, Terraform Registry, Crates.io, Go Modules, Helm Charts, Ansible Galaxy, Maven Central, NuGet, and RubyGems. "
                    "Supports semantic search (natural language), regex search (pattern matching), and hybrid search. "
                    "Use this tool to find packages that implement compliance, cost optimization, infrastructure patterns, etc. "
                    "Packages are automatically filtered to DevOps/infrastructure domain."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "query": {
                            "type": "string",
                            "description": "Natural language search query (required for semantic/hybrid search)",
                        },
                        "pattern": {
                            "type": "string",
                            "description": "Regex pattern (required for regex/hybrid search)",
                            "maxLength": 10000,
                        },
                        "template": {
                            "type": "string",
                            "enum": [
                                "api_key", "password", "secret_key", "token", "credential",
                                "terraform_resource", "kubernetes_secret", "import_statement",
                            ],
                            "description": "Pre-built template name (alternative to pattern)",
                        },
                        "search_type": {
                            "type": "string",
                            "enum": ["semantic", "regex", "hybrid"],
                            "default": "semantic",
                            "description": "Search type: semantic (natural language), regex (pattern), hybrid (both)",
                        },
                        "registry": {
                            "type": "string",
                            "enum": ["pypi", "npm", "terraform", "crates_io", "golang", "helm", "ansible", "maven", "nuget", "rubygems"],
                            "description": "Filter by registry",
                        },
                        "domain": {
                            "type": "string",
                            "enum": ["devops", "infrastructure", "compliance", "finops", "platform", "sre"],
                            "description": "Filter by domain",
                        },
                        "category": {
                            "type": "string",
                            "description": "Filter by category (infrastructure-as-code, cloud-providers, kubernetes, etc.)",
                        },
                        "package_name": {
                            "type": "string",
                            "description": "Search specific package",
                        },
                        "limit": {
                            "type": "integer",
                            "default": 20,
                            "minimum": 1,
                            "maximum": 100,
                            "description": "Maximum results",
                        },
                    },
                    "required": [],
                    "anyOf": [
                        {"required": ["query", "search_type"]},
                        {"required": ["pattern", "search_type"]},
                        {"required": ["template", "search_type"]},
                    ],
                },
            ),
            Tool(
                name="wistx_search_packages_read_file",
                description=build_tool_description(
                    "wistx_search_packages_read_file",
                    "Read specific file sections from package source code using SHA256 hash. "
                    "Use this tool to get complete context around code snippets found in package searches. "
                    "No indexing required - fetches packages on-demand from registries."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "registry": {
                            "type": "string",
                            "enum": ["pypi", "npm", "terraform", "crates_io", "golang", "helm", "ansible", "maven", "nuget", "rubygems"],
                            "description": "Package registry",
                        },
                        "package_name": {
                            "type": "string",
                            "description": "Package name",
                        },
                        "filename_sha256": {
                            "type": "string",
                            "description": "SHA256 hash of filename (from search results)",
                        },
                        "start_line": {
                            "type": "integer",
                            "description": "Starting line (1-based)",
                            "minimum": 1,
                        },
                        "end_line": {
                            "type": "integer",
                            "description": "Ending line (max 200 lines from start_line)",
                            "minimum": 1,
                        },
                        "version": {
                            "type": "string",
                            "description": "Optional package version",
                        },
                    },
                    "required": ["registry", "package_name", "filename_sha256", "start_line", "end_line"],
                },
            ),
            Tool(
                name="wistx_manage_infrastructure",
                description=build_tool_description(
                    "wistx_manage_infrastructure",
                    "Manage infrastructure lifecycle for Kubernetes clusters and multi-cloud resources. "
                    "Handles creation, updates, upgrades, backups, monitoring, and optimization. "
                    "Use this tool for Kubernetes cluster and multi-cloud management."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "action": {
                            "type": "string",
                            "enum": ["create", "update", "upgrade", "backup", "restore", "monitor", "optimize"],
                            "description": "Action to perform",
                        },
                        "infrastructure_type": {
                            "type": "string",
                            "enum": ["kubernetes", "multi_cloud", "hybrid_cloud"],
                            "description": "Type of infrastructure",
                        },
                        "resource_name": {
                            "type": "string",
                            "description": "Name of the resource/cluster",
                        },
                        "cloud_provider": {
                            "type": "string",
                            "description": "Cloud provider(s) - single string or list for multi-cloud",
                        },
                        "configuration": {
                            "type": "object",
                            "description": "Infrastructure configuration",
                        },
                        "compliance_standards": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Compliance standards to enforce",
                        },
                        "current_version": {
                            "type": "string",
                            "description": "Current version (for upgrade action)",
                        },
                        "target_version": {
                            "type": "string",
                            "description": "Target version (for upgrade action)",
                        },
                        "backup_type": {
                            "type": "string",
                            "enum": ["full", "incremental", "selective"],
                            "default": "full",
                            "description": "Type of backup (for backup action)",
                        },
                    },
                    "required": ["action", "infrastructure_type", "resource_name"],
                },
            ),
            Tool(
                name="wistx_get_existing_infrastructure",
                description=build_tool_description(
                    "wistx_get_existing_infrastructure",
                    "Get existing infrastructure context for a repository. Use this tool when coding agents need to understand existing infrastructure before creating new resources. Returns cost analysis, compliance status, and recommendations based on indexed repository data."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "repository_url": {
                            "type": "string",
                            "description": "GitHub repository URL (e.g., https://github.com/owner/repo)",
                        },
                        "environment_name": {
                            "type": "string",
                            "description": "Environment name (dev, stage, prod, etc.)",
                        },
                        "include_compliance": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include compliance status analysis",
                        },
                        "include_costs": {
                            "type": "boolean",
                            "default": True,
                            "description": "Include cost analysis",
                        },
                    },
                    "required": ["repository_url"],
                },
            ),
            Tool(
                name="wistx_get_devops_infra_code_examples",
                description=build_tool_description(
                    "wistx_get_devops_infra_code_examples",
                    "Search infrastructure code examples from curated repositories. Use this tool when asked for code examples, reference implementations, or sample code for infrastructure resources. Supports filtering by code type (terraform, kubernetes, docker, etc.), cloud provider, services, and compliance standards. Returns production-ready code examples with metadata, compliance analysis, and cost estimates."
                ),
                inputSchema={
                    "type": "object",
                    "properties": {
                        "query": {
                            "type": "string",
                            "description": "Search query (e.g., 'RDS database with encryption', 'Kubernetes deployment with autoscaling')",
                            "minLength": 3,
                        },
                        "code_types": {
                            "type": "array",
                            "items": {
                                "type": "string",
                                "enum": [
                                    "terraform",
                                    "opentofu",
                                    "pulumi",
                                    "ansible",
                                    "cloudformation",
                                    "bicep",
                                    "arm",
                                    "cdk",
                                    "cdk8s",
                                    "kubernetes",
                                    "docker",
                                    "helm",
                                    "github_actions",
                                    "gitlab_ci",
                                    "jenkins",
                                    "circleci",
                                    "argo_workflows",
                                    "tekton",
                                    "argocd",
                                    "flux",
                                    "spinnaker",
                                    "prometheus",
                                    "grafana",
                                    "datadog",
                                    "opentelemetry",
                                    "crossplane",
                                    "karpenter",
                                    "backstage",
                                    "sam",
                                    "serverless",
                                    "bash",
                                    "powershell",
                                ],
                            },
                            "description": "Filter by code types (terraform, kubernetes, docker, etc.)",
                        },
                        "cloud_provider": {
                            "type": "string",
                            "enum": ["aws", "gcp", "azure", "oracle", "alibaba"],
                            "description": "Filter by cloud provider",
                        },
                        "services": {
                            "type": "array",
                            "items": {"type": "string"},
                            "description": "Filter by cloud services (rds, s3, ec2, etc.)",
                        },
                        "min_quality_score": {
                            "type": "integer",
                            "minimum": 0,
                            "maximum": 100,
                            "description": "Minimum quality score (0-100)",
                        },
                        "compliance_standard": {
                            "type": "string",
                            "enum": ["PCI-DSS", "HIPAA", "SOC2", "CIS", "NIST-800-53", "ISO-27001"],
                            "description": "Filter by compliance standard (returns only compliant examples)",
                        },
                        "limit": {
                            "type": "integer",
                            "default": 10,
                            "minimum": 1,
                            "maximum": 100,
                            "description": "Maximum number of results",
                        },
                    },
                    "required": ["query"],
                },
            ),
            ]
            logger.info("Registered %d MCP tools: %s", len(tools), [t.name for t in tools])

            from wistx_mcp.tools.lib.protocol_handler import ensure_protocol_compliance

            protocol_version = get_request_context().get("protocol_version")
            compliant_tools = ensure_protocol_compliance(tools, protocol_version=protocol_version)

            try:
                tool_dicts = []
                for tool in compliant_tools:
                    if hasattr(tool, 'model_dump'):
                        tool_dict = tool.model_dump()
                    elif hasattr(tool, 'dict'):
                        tool_dict = tool.dict()
                    elif isinstance(tool, dict):
                        tool_dict = tool
                    else:
                        tool_dict = {
                            "name": getattr(tool, 'name', 'unknown'),
                            "description": getattr(tool, 'description', ''),
                            "inputSchema": getattr(tool, 'inputSchema', {}),
                        }
                    tool_dicts.append(tool_dict)
                distributed_cache.set_tool_definitions(tool_dicts)
                logger.debug("Cached tool list (%d tools)", len(tool_dicts))
            except Exception as e:
                logger.warning("Failed to cache tool list: %s", e)

            return compliant_tools
        except (ValueError, RuntimeError, AttributeError) as e:
            logger.error("Error in list_tools(): %s", e, exc_info=True)
            raise
        except Exception as e:
            logger.error("Unexpected error in list_tools(): %s", e, exc_info=True)
            raise RuntimeError(f"Failed to list tools: {e}") from e

    def _format_existing_infrastructure_result(result: dict[str, Any]) -> str:
        """Format existing infrastructure result as markdown.
        
        Args:
            result: Infrastructure context result dictionary
            
        Returns:
            Formatted markdown string
        """
        from wistx_mcp.tools.lib.context_builder import MarkdownBuilder
        
        builder = MarkdownBuilder()
        builder.add_header("Existing Infrastructure Context", level=1)
        
        repository_url = result.get("repository_url", "")
        environment_name = result.get("environment_name")
        status = result.get("status", "unknown")
        resource_id = result.get("resource_id")
        
        builder.add_bold(f"Repository: {repository_url}")
        builder.add_line_break()
        if environment_name:
            builder.add_bold(f"Environment: {environment_name}")
            builder.add_line_break()
        builder.add_bold(f"Status: {status}")
        builder.add_line_break()
        
        if status == "not_indexed":
            builder.add_paragraph(result.get("message", "Repository not indexed."))
            if result.get("recommendations"):
                builder.add_header("Recommendations", level=2)
                for rec in result["recommendations"]:
                    builder.add_list_item(rec)
            return builder.build()
        
        if resource_id:
            builder.add_bold(f"Resource ID: {resource_id}")
            builder.add_line_break()
        
        resources_count = result.get("resources_count", 0)
        total_monthly = result.get("total_monthly_cost", 0.0)
        total_annual = result.get("total_annual_cost", 0.0)
        
        builder.add_header("Cost Summary", level=2)
        builder.add_list_item(f"**Total Resources**: {resources_count}")
        builder.add_list_item(f"**Monthly Cost**: ${total_monthly:.2f}")
        builder.add_list_item(f"**Annual Cost**: ${total_annual:.2f}")
        
        cost_breakdown = result.get("cost_breakdown", {})
        if cost_breakdown:
            builder.add_header("Cost Breakdown", level=3)
            for service, cost in cost_breakdown.items():
                builder.add_list_item(f"**{service}**: ${cost:.2f}/month")
        
        cost_optimizations = result.get("cost_optimizations", [])
        if cost_optimizations:
            builder.add_header("Cost Optimization Opportunities", level=2)
            for opt in cost_optimizations[:5]:
                builder.add_list_item(opt)
        
        compliance_status = result.get("compliance_status")
        compliance_summary = result.get("compliance_summary", {})
        
        if compliance_status and compliance_summary:
            builder.add_header("Compliance Status", level=2)
            builder.add_bold(f"Overall Status: {compliance_status.upper()}")
            builder.add_line_break()
            
            for standard, data in compliance_summary.items():
                rate = data.get("compliance_rate", 0)
                compliant_count = data.get("compliant_count", 0)
                total_components = data.get("total_components", 0)
                builder.add_list_item(
                    f"**{standard.upper()}**: {rate:.1f}% compliant "
                    f"({compliant_count}/{total_components} components)"
                )
        
        recommendations = result.get("recommendations", [])
        if recommendations:
            builder.add_header("Recommendations", level=2)
            for rec in recommendations:
                builder.add_list_item(rec)
        
        context_for_agents = result.get("context_for_agents")
        if context_for_agents:
            builder.add_header("Context for Coding Agents", level=2)
            builder.add_code_block(context_for_agents, language="text")
        
        return builder.build()

    _tools_cache: list[Tool] | None = None

    def _validate_tool_arguments(tool: Tool, arguments: dict) -> None:
        """Validate tool arguments against input schema with comprehensive error handling.

        Args:
            tool: Tool definition with input schema
            arguments: Arguments to validate

        Raises:
            MCPError: If validation fails with context
            RuntimeError: If jsonschema is not available
        """
        try:
            from jsonschema import validate, ValidationError, SchemaError
        except ImportError as e:
            raise RuntimeError(
                "jsonschema library is required for argument validation. "
                "Install it with: pip install jsonschema"
            ) from e

        try:
            validate(instance=arguments, schema=tool.inputSchema)
        except SchemaError as e:
            logger.error("Invalid tool schema for %s: %s", tool.name, e)
            raise MCPError(
                code=MCPErrorCode.INTERNAL_ERROR,
                message=f"Invalid tool schema: {e.message}",
                data={"tool": tool.name, "schema_error": str(e)},
            ) from e
        except ValidationError as e:
            error_path = ".".join(str(p) for p in e.path) if e.path else "root"
            error_message = e.message

            if e.validator == "required":
                missing = e.validator_value
                error_message = f"Missing required field(s): {', '.join(missing)}"
            elif e.validator == "type":
                expected = e.validator_value
                error_message = f"Expected type {expected}, got {type(e.instance).__name__}"
            elif e.validator == "enum":
                allowed = e.validator_value
                error_message = f"Value must be one of: {', '.join(str(v) for v in allowed)}"

            raise MCPError(
                code=MCPErrorCode.INVALID_PARAMS,
                message=f"Invalid argument '{error_path}': {error_message}",
                data={
                    "tool": tool.name,
                    "field": error_path,
                    "error": error_message,
                    "validator": e.validator,
                },
            ) from e
        except Exception as e:
            logger.error("Unexpected error validating arguments: %s", e, exc_info=True)
            raise MCPError(
                code=MCPErrorCode.INTERNAL_ERROR,
                message="Argument validation failed",
                data={"tool": tool.name, "error": str(e)},
            ) from e

    @app.call_tool()
    async def call_tool(name: str, arguments: dict) -> list[TextContent]:
        """Handle tool calls with authentication, rate limiting, and security checks."""
        from wistx_mcp.tools.lib.metrics import get_tool_metrics

        request_id = request_id_var.get() or str(uuid.uuid4())
        try:
            request_id = validate_request_id(request_id)
        except ValueError:
            request_id = str(uuid.uuid4())

        if not request_id_var.get():
            request_id_var.set(request_id)

        async with _inflight_lock:
            _inflight_requests.add(request_id)

        auth_ctx = None
        try:
            if "api_key" in arguments:
                raise MCPError(
                    code=MCPErrorCode.INVALID_PARAMS,
                    message="API key cannot be passed via tool arguments for security reasons. Use WISTX_API_KEY environment variable or provide via MCP initialization.",
                    data={"request_id": request_id, "tool": name},
                )

            try:
                raw_serialized = json.dumps(arguments)
                raw_size = len(raw_serialized.encode("utf-8"))
                if raw_size > MAX_REQUEST_SIZE_BYTES:
                    raise MCPError(
                        code=MCPErrorCode.INVALID_PARAMS,
                        message=f"Request too large: {raw_size} bytes (max {MAX_REQUEST_SIZE_BYTES})",
                    )
            except (TypeError, ValueError) as e:
                if isinstance(e, MCPError):
                    raise
                raise MCPError(
                    code=MCPErrorCode.INVALID_PARAMS,
                    message=f"Cannot serialize request arguments: {sanitize_error_message(e)}",
                ) from e

            sanitized_args = sanitize_tool_arguments(arguments)

            from wistx_mcp.tools.lib.tool_registry import (
                is_tool_deprecated,
                get_deprecation_warning,
                get_tool_version,
                resolve_tool_name,
            )
            from api.services.version_tracking_service import version_tracking_service

            resolved_name = resolve_tool_name(name)
            tools_list = await list_tools()
            tool = next((t for t in tools_list if t.name == resolved_name), None)
            if not tool:
                raise MCPError(
                    code=MCPErrorCode.METHOD_NOT_FOUND,
                    message=f"Unknown tool: {name}",
                    data={"request_id": request_id, "tool": name},
                )

            tool_version_info = get_tool_version(resolved_name)
            tool_version = None
            if tool_version_info:
                tool_version = tool_version_info.get("current_version", "v1")
            
            actual_tool_name = resolved_name

            if is_tool_deprecated(actual_tool_name):
                deprecation_warning = get_deprecation_warning(actual_tool_name)
                if deprecation_warning:
                    logger.warning(
                        "Deprecated tool called: %s [request_id=%s] - %s",
                        actual_tool_name,
                        request_id,
                        deprecation_warning,
                    )
                    from wistx_mcp.tools.lib.tool_registry import get_tool_version
                    version_info = get_tool_version(actual_tool_name)
                    if version_info:
                        logger.info(
                            "Tool version info for %s: current=%s, deprecated=%s",
                            actual_tool_name,
                            version_info.get("current_version"),
                            version_info.get("deprecated_versions", []),
                        )

            user_id = None
            api_key_id = None
            auth_ctx = get_auth_context()
            if auth_ctx:
                user_id = auth_ctx.get_user_id()
                api_key_id = auth_ctx.get_api_key_id()

                # Phase 3: Log authentication event
                audit_logger.log_authentication(
                    user_id=user_id or "unknown",
                    auth_method="api_key",
                    success=True,
                )

            try:
                version_tracking_service.track_mcp_tool_version_usage(
                    tool_name=actual_tool_name,
                    tool_version=tool_version,
                    user_id=user_id,
                    api_key_id=api_key_id,
                )
            except Exception as e:
                logger.debug("Failed to track MCP tool version usage: %s", e)

            _validate_tool_arguments(tool, sanitized_args)

            deduplicator_instance = await get_request_deduplicator(ttl_seconds=REQUEST_DEDUPLICATION_TTL_SECONDS)
            cached_result = await deduplicator_instance.check_duplicate(request_id, name, sanitized_args)
            if cached_result is not None:
                logger.info("Returning cached result for duplicate request [request_id=%s]", request_id)
                success = True
                return cached_result

            context = get_request_context()
            logger.info(
                "Tool called: %s [request_id=%s, user_id=%s] with arguments: %s",
                name,
                context.get("request_id", request_id),
                context.get("user_id", "unknown"),
                safe_json_dumps(sanitize_arguments(sanitized_args), indent=2),
            )

            set_request_context({
                "request_id": request_id,
                "user_id": None,
                "tool_name": name,
            })

            start_time = time.time()
            success = False
            error: Exception | None = None

            from wistx_mcp.tools.lib.security_logger import (
                SecurityEventType,
                get_security_logger,
            )

            security_logger = get_security_logger()

            existing_auth_ctx = get_auth_context()
            if not existing_auth_ctx:
                await security_logger.log_security_event(
                    event_type=SecurityEventType.AUTHENTICATION_FAILURE,
                    tool_name=name,
                    severity="HIGH",
                    details={"reason": "no_auth_context"},
                    alert=True,
                )
                raise MCPError(
                    code=MCPErrorCode.INVALID_REQUEST,
                    message="Authentication required. Provide API key via WISTX_API_KEY environment variable or MCP initialization.",
                    data={"request_id": request_id, "tool": name},
                )
            else:
                auth_ctx = existing_auth_ctx
                user_id = auth_ctx.get_user_id() or "unknown"
                update_request_context(user_id=user_id)
                await security_logger.log_security_event(
                    event_type=SecurityEventType.AUTHENTICATION_SUCCESS,
                    user_id=user_id,
                    tool_name=name,
                    severity="INFO",
                )

            from wistx_mcp.tools.lib.tool_permissions import (
                get_tool_permissions,
                check_plan_access,
            )

            tool_perms = get_tool_permissions(actual_tool_name)
            required_plan = tool_perms.get("required_plan", "scout")
            required_permission = tool_perms.get("required_permission")
            quota_required = tool_perms.get("quota_required", True)

            user_plan = auth_ctx.user_info.get("plan", "scout") if auth_ctx.user_info else "scout"
            user_permissions = auth_ctx.user_info.get("permissions", []) if auth_ctx.user_info else []

            if not check_plan_access(user_plan, required_plan):
                await security_logger.log_security_event(
                    event_type=SecurityEventType.AUTHORIZATION_DENIED,
                    user_id=user_id,
                    tool_name=actual_tool_name,
                    severity="HIGH",
                    details={
                        "reason": "insufficient_plan",
                        "required_plan": required_plan,
                        "user_plan": user_plan,
                    },
                    alert=True,
                )
                raise MCPError(
                    code=MCPErrorCode.INVALID_REQUEST,
                    message=f"Tool '{actual_tool_name}' requires {required_plan} plan or higher. Your current plan: {user_plan}",
                    data={
                        "request_id": request_id,
                        "tool": actual_tool_name,
                        "required_plan": required_plan,
                        "user_plan": user_plan,
                    },
                )

            if required_permission and required_permission not in user_permissions:
                await security_logger.log_security_event(
                    event_type=SecurityEventType.AUTHORIZATION_DENIED,
                    user_id=user_id,
                    tool_name=actual_tool_name,
                    severity="HIGH",
                    details={
                        "reason": "missing_permission",
                        "required_permission": required_permission,
                    },
                    alert=True,
                )
                raise MCPError(
                    code=MCPErrorCode.INVALID_REQUEST,
                    message=f"Tool '{actual_tool_name}' requires permission: {required_permission}",
                    data={
                        "request_id": request_id,
                        "tool": actual_tool_name,
                        "required_permission": required_permission,
                    },
                )

            if quota_required:
                try:
                    from api.services.quota_service import quota_service
                    await quota_service.check_query_quota(user_id, user_plan)
                except ValueError as e:
                    raise MCPError(
                        code=MCPErrorCode.RATE_LIMIT_EXCEEDED,
                        message=f"Quota exceeded: {sanitize_error_message(e)}",
                        data={"request_id": request_id, "tool": actual_tool_name, "user_id": user_id},
                    ) from e

            rate_limiter_instance = await get_rate_limiter(
                max_calls=MAX_RATE_LIMIT_CALLS,
                window_seconds=RATE_LIMIT_WINDOW_SECONDS,
            )
            rate_limit_id = f"{user_id}:{hashlib.sha256(f'{user_id}:{name}'.encode()).hexdigest()}"

            if not await rate_limiter_instance.check_rate_limit(rate_limit_id):
                await security_logger.log_security_event(
                    event_type=SecurityEventType.RATE_LIMIT_EXCEEDED,
                    user_id=user_id,
                    tool_name=actual_tool_name,
                    severity="MEDIUM",
                    details={"rate_limit_id": rate_limit_id[:20]},
                    alert=True,
                )

                # Phase 3: Log rate limit event
                audit_logger.log_rate_limit(
                    user_id=user_id or "unknown",
                    tool_name=actual_tool_name,
                    retry_after=RATE_LIMIT_WINDOW_SECONDS,
                )

                raise MCPError(
                    code=MCPErrorCode.RATE_LIMIT_EXCEEDED,
                    message=f"Rate limit exceeded. Limit: {MAX_RATE_LIMIT_CALLS} calls per {RATE_LIMIT_WINDOW_SECONDS} seconds. Please try again after {RATE_LIMIT_WINDOW_SECONDS} seconds.",
                    data={
                        "request_id": request_id,
                        "tool": name,
                        "user_id": user_id,
                        "limit": MAX_RATE_LIMIT_CALLS,
                        "window_seconds": RATE_LIMIT_WINDOW_SECONDS,
                        "retry_after": RATE_LIMIT_WINDOW_SECONDS,
                        "suggestion": f"Wait {RATE_LIMIT_WINDOW_SECONDS} seconds before retrying, or use a different tool if available.",
                    },
                )

            concurrent_limiter_instance = await get_concurrent_limiter(max_concurrent=MAX_CONCURRENT_TOOLS)
            acquired = False
            try:
                await concurrent_limiter_instance.acquire(user_id)
                acquired = True

                async def execute_tool() -> list[TextContent]:
                    return await _execute_tool_internal(actual_tool_name, sanitized_args, auth_ctx)

                tool_timeout = TOOL_TIMEOUTS.get(name, GLOBAL_TOOL_TIMEOUT_SECONDS)
                result = await asyncio.wait_for(execute_tool(), timeout=tool_timeout)
                success = True

                from wistx_mcp.tools.lib.protocol_handler import ensure_protocol_compliance

                protocol_version = get_request_context().get("protocol_version")
                compliant_result = ensure_protocol_compliance(result, protocol_version=protocol_version)

                if is_tool_deprecated(actual_tool_name):
                    deprecation_warning = get_deprecation_warning(actual_tool_name)
                    if deprecation_warning:
                        warning_content = TextContent(
                            type="text",
                            text=f"⚠️ DEPRECATION WARNING: {deprecation_warning}"
                        )
                        compliant_result = [warning_content] + compliant_result

                await deduplicator_instance.store_result(request_id, name, sanitized_args, compliant_result)
                return compliant_result
            finally:
                if acquired:
                    await concurrent_limiter_instance.release(user_id)

        except MCPError:
            raise
        except asyncio.TimeoutError:
            tool_timeout = TOOL_TIMEOUTS.get(name, GLOBAL_TOOL_TIMEOUT_SECONDS)
            error = TimeoutError(f"Tool {name} execution timed out after {tool_timeout} seconds")
            logger.error("Tool execution timeout: %s [request_id=%s]", name, request_id)
            raise MCPError(
                code=MCPErrorCode.TIMEOUT,
                message=f"Tool execution timed out after {tool_timeout} seconds",
                data={"request_id": request_id, "tool": name, "timeout_seconds": tool_timeout},
            ) from error
        except ValueError as e:
            error = e
            logger.error("Validation error calling tool %s: %s [request_id=%s]", name, e, request_id, exc_info=True)
            from wistx_mcp.tools.lib.error_handler import ErrorHandler
            remediation_info = ErrorHandler.format_error_with_remediation(e, {"tool_name": name})
            raise MCPError(
                code=MCPErrorCode.INVALID_PARAMS,
                message=ErrorHandler.get_user_friendly_error_message(e, name),
                data={
                    "request_id": request_id,
                    "tool": name,
                    "retryable": False,
                    "remediation_steps": remediation_info.get("remediation_steps", []),
                },
            ) from e
        except httpx.HTTPStatusError as e:
            error = e
            status_code = e.response.status_code
            logger.error("HTTP error calling tool %s: %s (status: %s) [request_id=%s]", name, e, status_code, request_id)
            
            code = MCPErrorCode.INTERNAL_ERROR
            message = f"External service error: {status_code}"
            
            if status_code == 401:
                code = MCPErrorCode.INVALID_REQUEST
                message = "Authentication failed with external service"
            elif status_code == 403:
                code = MCPErrorCode.INVALID_REQUEST
                message = "Permission denied by external service"
            elif status_code == 429:
                code = MCPErrorCode.RATE_LIMIT_EXCEEDED
                message = "Rate limit exceeded by external service"
            elif status_code >= 500:
                message = "External service error"

            raise MCPError(
                code=code,
                message=message,
                data={"request_id": request_id, "tool": name, "status_code": status_code},
            ) from e
        except (RuntimeError, ConnectionError) as e:
            error = e
            logger.error("Error calling tool %s: %s [request_id=%s]", name, e, request_id, exc_info=True)
            from wistx_mcp.tools.lib.error_handler import ErrorHandler
            remediation_info = ErrorHandler.format_error_with_remediation(e, {"tool_name": name})
            is_retryable = isinstance(e, ConnectionError)
            raise MCPError(
                code=MCPErrorCode.INTERNAL_ERROR,
                message=ErrorHandler.get_user_friendly_error_message(e, name),
                data={
                    "request_id": request_id,
                    "tool": name,
                    "retryable": is_retryable,
                    "remediation_steps": remediation_info.get("remediation_steps", []),
                },
            ) from e
        except Exception as e:
            error = e
            logger.error("Unexpected error calling tool %s [request_id=%s]", name, request_id, exc_info=True)
            from wistx_mcp.tools.lib.error_handler import ErrorHandler
            remediation_info = ErrorHandler.format_error_with_remediation(e, {"tool_name": name})
            raise MCPError(
                code=MCPErrorCode.INTERNAL_ERROR,
                message=ErrorHandler.get_user_friendly_error_message(e, name),
                data={
                    "request_id": request_id,
                    "tool": name,
                    "retryable": False,
                    "remediation_steps": remediation_info.get("remediation_steps", []),
                },
            ) from e
        finally:
            async with _inflight_lock:
                _inflight_requests.discard(request_id)
            
            duration = time.time() - start_time
            metrics = get_tool_metrics(name)
            metrics.record_call(duration, success=success, error=error)
            logger.debug(
                "Tool %s completed in %.3f seconds (success=%s) [request_id=%s]",
                name,
                duration,
                success,
                request_id,
            )
            
            if auth_ctx and hasattr(auth_ctx, '_context_token') and auth_ctx._context_token:
                from wistx_mcp.tools.lib.auth_context import auth_context as auth_context_var
                try:
                    auth_context_var.reset(auth_ctx._context_token)
                except Exception:
                    pass

    async def _execute_tool_internal(name: str, arguments: dict, auth_ctx: AuthContext) -> list[TextContent]:
        """Internal tool execution logic.

        Args:
            name: Tool name
            arguments: Tool arguments (without api_key)
            auth_ctx: Authentication context

        Returns:
            List of TextContent results
        """
        from wistx_mcp.tools.lib.context_builder import ContextBuilder
        from wistx_mcp.tools.lib.auth_context import set_api_key_context

        tools_requiring_api_key = {
            "wistx_get_compliance_requirements",
            "wistx_research_knowledge_base",
            "wistx_calculate_infrastructure_cost",
            "wistx_index_repository",
            "wistx_index_resource",
            "wistx_list_resources",
            "wistx_check_resource_status",
            "wistx_delete_resource",
            "wistx_web_search",
            "wistx_search_codebase",
            "wistx_regex_search",
            "wistx_design_architecture",
            "wistx_troubleshoot_issue",
            "wistx_generate_documentation",
            "wistx_manage_integration",
            "wistx_manage_infrastructure",
            "wistx_get_existing_infrastructure",
            "wistx_get_devops_infra_code_examples",
            "wistx_get_recommended_tools",
            "wistx_list_tools_by_category",
            "wistx_get_tool_documentation",
        }

        if name in tools_requiring_api_key:
            set_api_key_context(auth_ctx.api_key)
        else:
            set_api_key_context(None)

        try:
            tool_args = arguments.copy()

            if name == "wistx_get_compliance_requirements":
                logger.info("Calling wistx_get_compliance_requirements tool...")
                result = await mcp_tools.get_compliance_requirements(**tool_args)
                logger.info("Tool wistx_get_compliance_requirements completed successfully")
                
                resource_type = arguments.get("resource_types", [""])[0] if arguments.get("resource_types") else None
                
                controls = []
                if "controls" in result:
                    controls = result["controls"]
                elif "data" in result and isinstance(result["data"], dict):
                    controls = result["data"].get("controls", [])
                
                if not controls:
                    logger.warning("No controls found in response: %s", list(result.keys()))
                    markdown_result = "No compliance controls found for the specified criteria."
                else:
                    markdown_result = ContextBuilder.format_compliance_as_toon(controls, resource_type)
                
                if not isinstance(markdown_result, str):
                    logger.error("Markdown result is not a string: %s", type(markdown_result))
                    markdown_result = "Error formatting compliance requirements."
                
                if not markdown_result.strip():
                    logger.warning("Markdown result is empty")
                    markdown_result = "No compliance controls found for the specified criteria."
                
                if "report_id" in result:
                    report_id = result["report_id"]
                    markdown_result += "\n\n---\n\n**Report Generated:**\n"
                    markdown_result += f"- Report ID: `{report_id}`\n"
                    if result.get("report_view_url"):
                        markdown_result += f"- View Report: {result['report_view_url']}\n"
                    if result.get("report_download_url"):
                        markdown_result += f"- Download Report: {result['report_download_url']}\n"
                    markdown_result += "\nThe compliance report has been automatically generated and stored. You can view or download it using the links above."
                
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_research_knowledge_base":
                logger.info("Calling wistx_research_knowledge_base tool...")
                result = await mcp_tools.research_knowledge_base(**tool_args)
                logger.info("Tool wistx_research_knowledge_base completed successfully")
                
                markdown_result = ContextBuilder.format_knowledge_research_results(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_calculate_infrastructure_cost":
                logger.info("Calling wistx_calculate_infrastructure_cost tool...")
                result = await mcp_tools.calculate_infrastructure_cost(**tool_args)
                logger.info("Tool wistx_calculate_infrastructure_cost completed successfully")
                
                markdown_result = ContextBuilder.format_pricing_as_toon(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_index_repository":
                logger.info("Calling wistx_index_repository tool...")
                result = await user_indexing.index_repository(**tool_args)
                logger.info("Tool wistx_index_repository completed successfully")
                markdown_result = ContextBuilder.format_indexing_results(result, operation="index")
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_index_resource":
                logger.info("Calling wistx_index_resource tool...")
                result = await user_indexing.index_content(**tool_args)
                logger.info("Tool wistx_index_resource completed successfully")
                markdown_result = ContextBuilder.format_indexing_results(result, operation="index")
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_list_resources":
                logger.info("Calling wistx_list_resources tool...")
                result = await user_indexing.list_resources(**tool_args)
                logger.info("Tool wistx_list_resources completed successfully")
                markdown_result = ContextBuilder.format_indexing_results(result, operation="list")
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_check_resource_status":
                logger.info("Calling wistx_check_resource_status tool...")
                result = await user_indexing.check_resource_status(**tool_args)
                logger.info("Tool wistx_check_resource_status completed successfully")
                markdown_result = ContextBuilder.format_indexing_results(result, operation="check")
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_delete_resource":
                logger.info("Calling wistx_delete_resource tool...")
                result = await user_indexing.delete_resource(**tool_args)
                logger.info("Tool wistx_delete_resource completed successfully")
                markdown_result = ContextBuilder.format_indexing_results(result, operation="delete")
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_web_search":
                logger.info("Calling wistx_web_search tool...")
                result = await web_search.web_search(**tool_args)
                logger.info("Tool wistx_web_search completed successfully")
                
                markdown_result = ContextBuilder.format_web_search_results(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_search_codebase":
                logger.info("Calling wistx_search_codebase tool...")
                result = await search_codebase.search_codebase(**tool_args)
                logger.info("Tool wistx_search_codebase completed successfully")
                
                markdown_result = ContextBuilder.format_codebase_search_results(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_regex_search":
                logger.info("Calling wistx_regex_search tool...")
                result = await regex_search.regex_search_codebase(**tool_args)
                logger.info("Tool wistx_regex_search completed successfully")
                
                markdown_result = ContextBuilder.format_regex_search_results(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_search_packages":
                logger.info("Calling wistx_search_packages tool...")
                result = await package_search.package_search(**tool_args)
                logger.info("Tool wistx_search_packages completed successfully")
                
                markdown_result = ContextBuilder.format_package_search_results(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_search_packages_read_file":
                logger.info("Calling wistx_search_packages_read_file tool...")
                result = await package_search.package_search_read_file(**tool_args)
                logger.info("Tool wistx_search_packages_read_file completed successfully")
                
                markdown_result = ContextBuilder.format_package_read_file_results(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_design_architecture":
                logger.info("Calling wistx_design_architecture tool...")
                result = await design_architecture.design_architecture(**tool_args)
                logger.info("Tool wistx_design_architecture completed successfully")
                
                markdown_result = ContextBuilder.format_architecture_results(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_troubleshoot_issue":
                logger.info("Calling wistx_troubleshoot_issue tool...")
                result = await troubleshoot_issue.troubleshoot_issue(**tool_args)
                logger.info("Tool wistx_troubleshoot_issue completed successfully")
                
                markdown_result = ContextBuilder.format_troubleshooting_results(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_generate_documentation":
                logger.info("Calling wistx_generate_documentation tool...")
                result = await generate_documentation.generate_documentation(**tool_args)
                logger.info("Tool wistx_generate_documentation completed successfully")
                
                markdown_result = ContextBuilder.format_documentation_results(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_manage_integration":
                logger.info("Calling wistx_manage_integration tool...")
                result = await manage_integration.manage_integration(**tool_args)
                logger.info("Tool wistx_manage_integration completed successfully")
                
                markdown_result = ContextBuilder.format_integration_results(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_manage_infrastructure":
                logger.info("Calling wistx_manage_infrastructure tool...")
                result = await manage_infrastructure.manage_infrastructure(**tool_args)
                logger.info("Tool wistx_manage_infrastructure completed successfully")
                
                markdown_result = ContextBuilder.format_infrastructure_results(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_get_existing_infrastructure":
                logger.info("Calling wistx_get_existing_infrastructure tool...")
                result = await infrastructure_context.get_existing_infrastructure(**tool_args)
                logger.info("Tool wistx_get_existing_infrastructure completed successfully")
                
                markdown_result = _format_existing_infrastructure_result(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_get_devops_infra_code_examples":
                logger.info("Calling wistx_get_devops_infra_code_examples tool...")
                from wistx_mcp.tools import code_examples
                result = await code_examples.get_code_examples(**tool_args)
                logger.info("Tool wistx_get_devops_infra_code_examples completed successfully")
                
                markdown_result = ContextBuilder.format_code_examples_as_toon(result.get("examples", []))
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_get_recommended_tools":
                logger.info("Calling wistx_get_recommended_tools tool...")
                from wistx_mcp.tools import tool_discovery
                result = await tool_discovery.get_recommended_tools(**tool_args)
                logger.info("Tool wistx_get_recommended_tools completed successfully")
                
                markdown_result = ContextBuilder.format_tool_recommendations(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_list_tools_by_category":
                logger.info("Calling wistx_list_tools_by_category tool...")
                from wistx_mcp.tools import tool_discovery
                result = await tool_discovery.list_tools_by_category(**tool_args)
                logger.info("Tool wistx_list_tools_by_category completed successfully")
                
                markdown_result = ContextBuilder.format_tools_by_category(result)
                return [TextContent(type="text", text=markdown_result)]

            elif name == "wistx_get_tool_documentation":
                logger.info("Calling wistx_get_tool_documentation tool...")
                from wistx_mcp.tools import tool_discovery
                result = await tool_discovery.get_tool_documentation(**tool_args)
                logger.info("Tool wistx_get_tool_documentation completed successfully")
                
                markdown_result = ContextBuilder.format_tool_documentation(result)
                return [TextContent(type="text", text=markdown_result)]

            else:
                logger.warning("Unknown tool requested: %s", name)
                raise ValueError(f"Unknown tool: {name}")
        finally:
            set_api_key_context(None)

    async def _wait_for_inflight_requests() -> None:
        """Wait for in-flight requests to complete.

        This checks if there are any active tool executions and waits for them.
        """
        GRACEFUL_SHUTDOWN_TIMEOUT = 30.0
        check_interval = 0.5
        waited = 0.0

        while waited < GRACEFUL_SHUTDOWN_TIMEOUT:
            async with _inflight_lock:
                if not _inflight_requests:
                    logger.info("All in-flight requests completed")
                    return

            await asyncio.sleep(check_interval)
            waited += check_interval

        async with _inflight_lock:
            if _inflight_requests:
                logger.warning(
                    "Shutdown timeout: %d requests still in flight",
                    len(_inflight_requests),
                )

    try:
        logger.info("Starting stdio server...")
        async with stdio_server() as (read_stream, write_stream):
            logger.info("stdio server started, running MCP server...")
            init_options = app.create_initialization_options()
            logger.info("Initialization options created: %s", init_options)

            server_task = asyncio.create_task(
                app.run(read_stream, write_stream, init_options)
            )

            shutdown_task = asyncio.create_task(shutdown_event.wait())

            _done, pending = await asyncio.wait(
                [server_task, shutdown_task],
                return_when=asyncio.FIRST_COMPLETED,
            )

            if shutdown_event.is_set():
                logger.info("Shutdown signal received, initiating graceful shutdown...")

                try:
                    await _wait_for_inflight_requests()
                except Exception as e:
                    logger.warning("Error waiting for in-flight requests: %s", e)

                if not server_task.done():
                    server_task.cancel()
                    try:
                        await asyncio.wait_for(server_task, timeout=5.0)
                    except (asyncio.TimeoutError, asyncio.CancelledError):
                        logger.warning("Server task did not cancel cleanly")

            for task in pending:
                task.cancel()
                try:
                    await task
                except asyncio.CancelledError:
                    pass

            logger.info("MCP server stopped")
    except (RuntimeError, ConnectionError, ValueError) as e:
        logger.error("Error starting MCP server: %s", e, exc_info=True)
        raise
    except Exception as e:
        logger.error("Unexpected error starting MCP server: %s", e, exc_info=True)
        raise RuntimeError(f"Failed to start MCP server: {e}") from e
    finally:
        logger.info("Cleaning up MCP server resources...")
        try:
            if rate_limiter:
                await rate_limiter.stop()
            if deduplicator:
                await deduplicator.stop()
            if concurrent_limiter:
                await concurrent_limiter.stop()
            logger.info("Background tasks stopped")
        except Exception as e:
            logger.warning("Error stopping background tasks: %s", e)

        try:
            await resource_manager.cleanup_all()
        except Exception as e:
            logger.warning("Error during resource cleanup: %s", e)


def cli() -> None:
    """CLI entry point for package installation."""
    try:
        asyncio.run(main())
    except KeyboardInterrupt:
        logger.info("Shutting down MCP server")
        sys.exit(0)
    except (RuntimeError, ConnectionError, ValueError, AttributeError) as e:
        logger.error("Failed to start MCP server: %s", e, exc_info=True)
        sys.exit(1)


if __name__ == "__main__":
    cli()

