"""Generate documentation tool - create documentation and reports."""

import logging
import re
from datetime import datetime
from typing import Any

from wistx_mcp.tools.lib.mongodb_client import MongoDBClient
from wistx_mcp.tools.lib.document_generator import DocumentGenerator
from wistx_mcp.tools.lib.report_template_manager import ReportTemplateManager
from wistx_mcp.tools.lib.format_converter import FormatConverter
from wistx_mcp.tools.lib.template_library import TemplateLibrary
from wistx_mcp.models.report_template import OutputFormat
from wistx_mcp.config import settings

logger = logging.getLogger(__name__)


async def generate_documentation(
    document_type: str,
    subject: str,
    infrastructure_code: str | None = None,
    configuration: dict[str, Any] | None = None,
    include_compliance: bool = True,
    include_security: bool = True,
    include_cost: bool = True,
    include_best_practices: bool = True,
    resource_types: list[str] | None = None,
    resource_ids: list[str] | None = None,
    compliance_standards: list[str] | None = None,
    resources: list[dict[str, Any]] | None = None,
    api_spec: dict[str, Any] | None = None,
    format: str = "markdown",
    template_id: str | None = None,
    custom_template: dict[str, Any] | None = None,
    branding: dict[str, Any] | None = None,
    api_key: str = "",
) -> dict[str, Any]:
    """Generate documentation and reports.

    Args:
        document_type: Type of document (architecture_diagram, runbook, compliance_report,
                      cost_report, security_report, api_documentation, deployment_guide)
        subject: Subject of the document (project name, resource, topic)
        infrastructure_code: Infrastructure code to document
        configuration: Configuration to document
        include_compliance: Include compliance information
        include_security: Include security information
        include_cost: Include cost information
        include_best_practices: Include best practices
        resource_types: List of resource types (for compliance/security reports)
        resource_ids: List of specific indexed resource IDs to filter knowledge articles
        compliance_standards: List of compliance standards (for compliance report)
        resources: List of resource specifications (for cost report)
        api_spec: API specification (for api_documentation)
        format: Output format (markdown, pdf, html, json)
        template_id: Custom template ID (for compliance_report)
        custom_template: Custom template dictionary (alternative to template_id)
        branding: Branding configuration (logo, colors, etc.)

    Returns:
        Dictionary with documentation:
        - content: Generated documentation content
        - format: Output format
        - document_type: Type of document
        - sections: Document sections
        - metadata: Document metadata

    Raises:
        ValueError: If invalid document_type or parameters
        Exception: If generation fails
    """
    from wistx_mcp.tools.lib.auth_context import get_api_key_from_context

    api_key = get_api_key_from_context() or api_key

    valid_types = [
        "architecture_diagram",
        "runbook",
        "compliance_report",
        "cost_report",
        "security_report",
        "api_documentation",
        "deployment_guide",
    ]

    if document_type not in valid_types:
        raise ValueError(f"Invalid document_type: {document_type}. Must be one of {valid_types}")

    if format not in ["markdown", "pdf", "html", "json"]:
        raise ValueError(f"Invalid format: {format}. Must be one of markdown, pdf, html, json")

    from wistx_mcp.tools.lib.input_sanitizer import (
        validate_input_size,
        validate_infrastructure_code_input,
    )
    from wistx_mcp.tools.lib.constants import MAX_SUBJECT_LENGTH

    validate_input_size(subject, "subject", MAX_SUBJECT_LENGTH)

    if infrastructure_code:
        validate_infrastructure_code_input(infrastructure_code)

    from wistx_mcp.tools.lib.auth_context import validate_api_key_and_get_user_id, get_auth_context

    try:
        await validate_api_key_and_get_user_id(api_key)
    except (ValueError, RuntimeError) as e:
        raise

    logger.info(
        "Generating documentation: type=%s, subject=%s, format=%s",
        document_type,
        subject,
        format,
    )

    auth_ctx = get_auth_context()
    if auth_ctx:
        user_id = auth_ctx.get_user_id()
        if user_id:
            try:
                from api.services.quota_service import quota_service, QuotaExceededError

                plan = "scout"
                if auth_ctx.user_info:
                    plan = auth_ctx.user_info.get("plan", "scout")
                await quota_service.check_query_quota(user_id, plan)
            except QuotaExceededError as e:
                logger.warning("Quota exceeded for user %s: %s", user_id, e)
                raise RuntimeError(f"Quota exceeded: {e}") from e
            except Exception as e:
                logger.warning("Failed to check quota (continuing): %s", e)

    try:
        async with MongoDBClient() as mongodb_client:

            generator = DocumentGenerator(mongodb_client)

            selected_resources_info = []
            if resource_ids and mongodb_client.database:
                try:
                    resources_collection = mongodb_client.database.indexed_resources
                    resource_docs = []
                    async for doc in resources_collection.find({"resource_id": {"$in": resource_ids}}):
                        resource_docs.append(doc)
                    
                    selected_resources_info = [
                        {
                            "resource_id": doc.get("resource_id"),
                            "name": doc.get("name"),
                            "resource_type": doc.get("resource_type"),
                            "repo_url": doc.get("repo_url"),
                            "documentation_url": doc.get("documentation_url"),
                            "document_url": doc.get("document_url"),
                        }
                        for doc in resource_docs
                    ]
                except Exception as e:
                    logger.warning("Failed to fetch resource details: %s", e)

            content = ""

            if document_type == "architecture_diagram":
                content = await generator.generate_architecture_doc(
                    subject=subject,
                    infrastructure_code=infrastructure_code,
                    configuration=configuration,
                    include_compliance=include_compliance,
                    include_security=include_security,
                )

                if include_security and infrastructure_code:
                    try:
                        from wistx_mcp.tools import regex_search

                        if api_key:
                            security_templates = ["api_key", "password", "secret_key"]
                            security_issues = []
                            for template in security_templates:
                                try:
                                    regex_results = await regex_search.regex_search_codebase(
                                        template=template,
                                        api_key=api_key,
                                        include_context=True,
                                        limit=5,
                                    )
                                    matches = regex_results.get("matches", [])
                                    if matches:
                                        security_issues.append({
                                            "type": template,
                                            "count": len(matches),
                                            "matches": matches[:3],
                                        })
                                except Exception as e:
                                    logger.warning("Regex search failed for template %s: %s", template, e)

                            if security_issues:
                                security_section = "\n\n## Security Issues Found\n\n"
                                for issue in security_issues:
                                    security_section += f"- **{issue['type'].replace('_', ' ').title()}**: Found {issue['count']} instances\n"
                                content += security_section
                    except ImportError:
                        logger.warning("regex_search module not available, skipping security checks")
                    except Exception as e:
                        logger.warning("Failed to add regex security checks to documentation: %s", e)

            elif document_type == "runbook":
                operations = configuration.get("operations") if configuration else None
                troubleshooting = configuration.get("troubleshooting") if configuration else None
                content = await generator.generate_runbook(
                    subject=subject,
                    operations=operations,
                    troubleshooting=troubleshooting,
                )

            elif document_type == "compliance_report":
                if not resource_types and not resource_ids:
                    raise ValueError("resource_types or resource_ids required for compliance_report")

                template_manager = ReportTemplateManager(mongodb_client)
                format_converter = FormatConverter()

                if template_id or custom_template:
                    template_data = None
                    if template_id:
                        template = await template_manager.get_template(template_id)
                        if not template:
                            raise ValueError(f"Template not found: {template_id}")
                    else:
                        template_dict = custom_template or {}
                        template = await template_manager.register_template(
                            name=template_dict.get("name", "Custom Template"),
                            template_content=template_dict.get("template_content", ""),
                            document_type="compliance_report",
                            template_engine=template_dict.get("template_engine", "jinja2"),
                            compliance_standards=compliance_standards or [],
                            resource_types=resource_types or [],
                            variables=template_dict.get("variables", {}),
                            visibility="private",
                            user_id=None,
                        )

                    compliance_data = await generator.generate_compliance_report(
                        subject=subject,
                        resource_types=resource_types or [],
                        standards=compliance_standards,
                        resource_ids=resource_ids,
                    )

                    from datetime import datetime
                    template_data = {
                        "subject": subject,
                        "generated_at": datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
                        "compliance_status": "Assessed",
                        "total_requirements": len(compliance_data.get("controls", [])),
                        "compliant_requirements": 0,
                        "non_compliant_requirements": 0,
                        "compliance_score": 0.0,
                        "resources": (
                            [{"resource_id": r.get("resource_id"), "name": r.get("name"), "type": r.get("resource_type")} for r in selected_resources_info]
                            if selected_resources_info
                            else [{"type": rt, "name": rt} for rt in resource_types or []]
                        ),
                        "assessment_start": datetime.now().strftime("%Y-%m-%d"),
                        "assessment_end": datetime.now().strftime("%Y-%m-%d"),
                        "requirements": compliance_data.get("controls", []),
                        "recommendations": [],
                    }

                    rendered_content = await template_manager.render_template(
                        template_id=template.template_id,
                        data=template_data,
                        output_format=OutputFormat(format.upper()),
                    )

                    if format == "pdf":
                        pdf_bytes = format_converter.markdown_to_pdf(
                            rendered_content,
                            styles=template.styles,
                            branding=branding or template.branding,
                        )
                        return {
                            "content": pdf_bytes,
                            "format": "pdf",
                            "document_type": document_type,
                            "subject": subject,
                            "sections": _extract_sections(rendered_content),
                            "metadata": {
                                "generated_at": datetime.now().isoformat(),
                                "template_id": template.template_id,
                                "template_name": template.name,
                            },
                        }
                    elif format == "html":
                        html_content = format_converter.markdown_to_html(
                            rendered_content,
                            styles=template.styles,
                        )
                        return {
                            "content": html_content,
                            "format": "html",
                            "document_type": document_type,
                            "subject": subject,
                            "sections": _extract_sections(rendered_content),
                            "metadata": {
                                "generated_at": datetime.now().isoformat(),
                                "template_id": template.template_id,
                                "template_name": template.name,
                            },
                        }
                    elif format == "docx":
                        docx_bytes = format_converter.markdown_to_docx(
                            rendered_content,
                            branding=branding or template.branding,
                        )
                        return {
                            "content": docx_bytes,
                            "format": "docx",
                            "document_type": document_type,
                            "subject": subject,
                            "sections": _extract_sections(rendered_content),
                            "metadata": {
                                "generated_at": datetime.now().isoformat(),
                                "template_id": template.template_id,
                                "template_name": template.name,
                            },
                        }
                    else:
                        content = rendered_content
                else:
                    content = await generator.generate_compliance_report(
                        subject=subject,
                        resource_types=resource_types or [],
                        standards=compliance_standards,
                        resource_ids=resource_ids,
                    )

            elif document_type == "security_report":
                cloud_provider = configuration.get("cloud_provider") if configuration else None
                content = await generator.generate_security_report(
                    subject=subject,
                    resource_types=resource_types or [],
                    cloud_provider=cloud_provider,
                    resource_ids=resource_ids,
                )

                try:
                    from wistx_mcp.tools import regex_search

                    if api_key:
                        security_templates = ["api_key", "password", "secret_key", "token", "credential"]
                        security_issues = []
                        for template in security_templates:
                            try:
                                regex_results = await regex_search.regex_search_codebase(
                                    template=template,
                                    api_key=api_key,
                                    include_context=True,
                                    limit=10,
                                )
                                matches = regex_results.get("matches", [])
                                if matches:
                                    security_issues.append({
                                        "type": template,
                                        "count": len(matches),
                                        "matches": matches[:5],
                                    })
                            except Exception as e:
                                logger.warning("Regex search failed for template %s: %s", template, e)

                        if security_issues:
                            security_section = "\n\n## Security Audit Results\n\n"
                            security_section += f"**Total Issues Found**: {sum(issue['count'] for issue in security_issues)}\n\n"
                            for issue in security_issues:
                                security_section += f"### {issue['type'].replace('_', ' ').title()}\n\n"
                                security_section += f"Found {issue['count']} instances:\n\n"
                                for match in issue["matches"]:
                                    security_section += f"- File: `{match.get('file_path', 'unknown')}`\n"
                                    security_section += f"  Line {match.get('line_number', 'unknown')}: {match.get('match_text', '')[:100]}\n\n"
                            content += security_section
                except ImportError:
                    logger.warning("regex_search module not available, skipping security audit")
                except Exception as e:
                    logger.warning("Failed to add regex security checks to security report: %s", e)

            elif document_type == "cost_report":
                content = await generator.generate_cost_report(
                    subject=subject,
                    resources=resources,
                )

            elif document_type == "api_documentation":
                content = await generator.generate_api_documentation(
                    subject=subject,
                    api_spec=api_spec,
                )

            elif document_type == "deployment_guide":
                infrastructure_type = configuration.get("infrastructure_type") if configuration else None
                cloud_provider = configuration.get("cloud_provider") if configuration else None
                content = await generator.generate_deployment_guide(
                    subject=subject,
                    infrastructure_type=infrastructure_type,
                    cloud_provider=cloud_provider,
                )

            sections = _extract_sections(content)

            metadata = {
                "generated_at": datetime.now().isoformat(),
                "includes": {
                    "compliance": include_compliance,
                    "security": include_security,
                    "cost": include_cost,
                    "best_practices": include_best_practices,
                },
            }

            if resource_ids:
                metadata["resource_ids"] = resource_ids
            if selected_resources_info:
                metadata["selected_resources"] = selected_resources_info

            return {
                "content": content,
                "format": format,
                "document_type": document_type,
                "subject": subject,
                "sections": sections,
                "metadata": metadata,
            }
    except Exception as e:
        logger.error("Error in generate_documentation: %s", e, exc_info=True)
        raise


def _extract_sections(content: str) -> list[str]:
    """Extract section headers from markdown content.

    Args:
        content: Markdown content

    Returns:
        List of section headers
    """
    sections = re.findall(r"^#+\s+(.+)$", content, re.MULTILINE)
    return sections

