"""MCP tool for getting existing infrastructure context."""

import logging
from typing import Any

from wistx_mcp.tools.lib.api_client import WISTXAPIClient
from wistx_mcp.tools.lib.url_validator import validate_github_url
from wistx_mcp.tools.lib.plan_enforcement import require_query_quota

logger = logging.getLogger(__name__)

api_client = WISTXAPIClient()


@require_query_quota
async def get_existing_infrastructure(
    repository_url: str,
    environment_name: str | None = None,
    include_compliance: bool = True,
    include_costs: bool = True,
    api_key: str = "",
) -> dict[str, Any]:
    """Get existing infrastructure context for coding agents.

    This tool provides context about existing infrastructure so coding agents
    can make informed decisions when adding new resources.

    Args:
        repository_url: GitHub repository URL
        environment_name: Environment name (dev, stage, prod, etc.)
        include_compliance: Include compliance status
        include_costs: Include cost information
        api_key: WISTX API key (required for authentication)

    Returns:
        Dictionary with existing infrastructure context:
        - resources: List of existing resources
        - total_monthly_cost: Total monthly cost
        - compliance_summary: Compliance assessment
        - recommendations: Recommendations for new resources

    Raises:
        ValueError: If api_key is missing or invalid
        RuntimeError: If API call fails
    """
    from wistx_mcp.tools.lib.auth_context import validate_api_key_and_get_user_id
    from wistx_mcp.tools.lib.input_sanitizer import validate_repository_url_input

    validate_repository_url_input(repository_url)

    try:
        validated_repo_url = await validate_github_url(repository_url)
    except ValueError as e:
        raise ValueError(f"Invalid GitHub repository URL: {e}") from e

    try:
        user_id = await validate_api_key_and_get_user_id(api_key)
    except (ValueError, RuntimeError) as e:
        raise

    try:
        api_response = await api_client.get_infrastructure_inventory(
            repository_url=validated_repo_url,
            environment_name=environment_name,
            api_key=api_key,
        )

        if api_response.get("data"):
            return api_response["data"]
        return api_response

    except Exception as e:
        logger.error("Error getting existing infrastructure: %s", e, exc_info=True)
        raise RuntimeError(f"Failed to get existing infrastructure context: {e}") from e


def _group_by_service(resources: list[dict[str, Any]]) -> dict[str, float]:
    """Group resources by service and sum costs.

    Args:
        resources: List of resources

    Returns:
        Dictionary mapping service to total monthly cost
    """
    grouped = {}
    for resource in resources:
        service = resource.get("service", "unknown")
        cost = resource.get("monthly_cost", 0)
        grouped[service] = grouped.get(service, 0) + cost
    return grouped


def _group_by_resource_type(resources: list[dict[str, Any]]) -> dict[str, float]:
    """Group resources by resource type and sum costs.

    Args:
        resources: List of resources

    Returns:
        Dictionary mapping resource type to total monthly cost
    """
    grouped = {}
    for resource in resources:
        resource_type = resource.get("resource_type", "unknown")
        cost = resource.get("monthly_cost", 0)
        grouped[resource_type] = grouped.get(resource_type, 0) + cost
    return grouped


def _calculate_overall_compliance(compliance_summary: dict[str, Any]) -> str:
    """Calculate overall compliance status.

    Args:
        compliance_summary: Compliance summary dictionary

    Returns:
        Overall compliance status (compliant, partial, non_compliant)
    """
    if not compliance_summary:
        return "unknown"

    total_compliant = 0
    total_non_compliant = 0

    for standard_data in compliance_summary.values():
        total_compliant += standard_data.get("compliant_count", 0)
        total_non_compliant += standard_data.get("non_compliant_count", 0)

    total = total_compliant + total_non_compliant
    if total == 0:
        return "unknown"

    compliance_rate = (total_compliant / total) * 100

    if compliance_rate >= 90:
        return "compliant"
    elif compliance_rate >= 50:
        return "partial"
    else:
        return "non_compliant"


def _generate_recommendations(result: dict[str, Any]) -> list[str]:
    """Generate recommendations based on analysis results.

    Args:
        result: Analysis result dictionary

    Returns:
        List of recommendation strings
    """
    recommendations = []
    
    total_cost = result.get("total_monthly_cost", 0)
    if total_cost > 500:
        recommendations.append(f"Consider cost optimization - current spending: ${total_cost:.2f}/month")
    
    compliance_status = result.get("compliance_status", "unknown")
    if compliance_status == "non_compliant":
        recommendations.append("Address compliance issues before adding new resources")
    elif compliance_status == "partial":
        recommendations.append("Review partial compliance issues")
    
    return recommendations


def _generate_agent_context(result: dict[str, Any]) -> str:
    """Generate context message for coding agents.

    Args:
        inventory: Infrastructure inventory

    Returns:
        Context message string
    """
    resources_count = result.get("resources_count", 0)
    total_cost = result.get("total_monthly_cost", 0)
    compliance_summary = result.get("compliance_summary", {})

    context = f"Existing Infrastructure Context:\n\n"
    context += f"- Total Resources: {resources_count}\n"
    context += f"- Monthly Cost: ${total_cost:.2f}\n"

    if compliance_summary:
        context += f"\nCompliance Status:\n"
        for standard, data in compliance_summary.items():
            rate = data.get("compliance_rate", 0)
            context += f"- {standard.upper()}: {rate:.1f}% compliant "
            context += f"({data.get('compliant_count', 0)}/{data.get('total_components', 0)})\n"

    recommendations = result.get("recommendations", [])
    if recommendations:
        context += f"\nRecommendations:\n"
        for rec in recommendations[:5]:
            context += f"- {rec}\n"

    context += f"\nWhen adding new resources, consider:\n"
    context += f"- Existing resources to avoid duplicates\n"
    context += f"- Compliance requirements for new resources\n"
    context += f"- Budget constraints (current: ${total_cost:.2f}/month)\n"
    context += f"- Integration with existing infrastructure\n"

    return context

