"""Template library for common compliance standards."""

import logging
from typing import Any

from wistx_mcp.models.report_template import ReportTemplate, TemplateEngine, OutputFormat

logger = logging.getLogger(__name__)


class TemplateLibrary:
    """Library of built-in report templates for common compliance standards."""

    @staticmethod
    def get_pci_dss_template() -> dict[str, Any]:
        """Get PCI-DSS compliance report template.

        Returns:
            Template dictionary
        """
        template_content = """# PCI-DSS Compliance Report: {{ subject }}

**Generated**: {{ generated_at }}
**Assessment Period**: {{ assessment_period }}

## Executive Summary

This report assesses compliance with the Payment Card Industry Data Security Standard (PCI-DSS) for {{ subject }}.

**Overall Compliance Status**: {{ compliance_status }}

### Key Findings

- **Total Requirements**: {{ total_requirements }}
- **Compliant Requirements**: {{ compliant_requirements }}
- **Non-Compliant Requirements**: {{ non_compliant_requirements }}
- **Compliance Score**: {{ compliance_score }}%

## Scope

### Resources Assessed

{% for resource in resources %}
- **{{ resource.type }}**: {{ resource.name }}
{% endfor %}

### Assessment Period

- **Start Date**: {{ assessment_start }}
- **End Date**: {{ assessment_end }}

## Compliance Requirements

{% for requirement in requirements %}
### Requirement {{ requirement.number }}: {{ requirement.title }}

**Status**: {{ requirement.status }}
**Severity**: {{ requirement.severity }}

**Description**: {{ requirement.description }}

{% if requirement.findings %}
**Findings**:
{% for finding in requirement.findings %}
- {{ finding }}
{% endfor %}
{% endif %}

{% if requirement.remediation %}
**Remediation**: {{ requirement.remediation }}
{% endif %}

---
{% endfor %}

## Recommendations

{% for recommendation in recommendations %}
- {{ recommendation }}
{% endfor %}

## Next Steps

1. Address non-compliant requirements
2. Implement remediation measures
3. Schedule follow-up assessment
4. Update security policies and procedures

---

**Report Generated By**: {{ generator_name }}
**Report Version**: {{ report_version }}
"""

        return {
            "name": "PCI-DSS Compliance Report Template",
            "description": "Standard template for PCI-DSS compliance reports",
            "template_content": template_content,
            "document_type": "compliance_report",
            "compliance_standards": ["PCI-DSS"],
            "resource_types": ["RDS", "S3", "EC2", "Lambda", "VPC"],
            "template_engine": TemplateEngine.JINJA2,
            "output_formats": [OutputFormat.MARKDOWN, OutputFormat.HTML, OutputFormat.PDF],
            "variables": {
                "subject": {"type": "string", "required": True},
                "generated_at": {"type": "string", "required": True},
                "assessment_period": {"type": "string", "required": True},
                "compliance_status": {"type": "string", "required": True},
                "total_requirements": {"type": "integer", "required": True},
                "compliant_requirements": {"type": "integer", "required": True},
                "non_compliant_requirements": {"type": "integer", "required": True},
                "compliance_score": {"type": "float", "required": True},
                "resources": {"type": "array", "required": True},
                "assessment_start": {"type": "string", "required": True},
                "assessment_end": {"type": "string", "required": True},
                "requirements": {"type": "array", "required": True},
                "recommendations": {"type": "array", "required": True},
                "generator_name": {"type": "string", "required": False},
                "report_version": {"type": "string", "required": False},
            },
            "sections": [
                "executive_summary",
                "scope",
                "compliance_requirements",
                "recommendations",
                "next_steps",
            ],
        }

    @staticmethod
    def get_hipaa_template() -> dict[str, Any]:
        """Get HIPAA compliance report template.

        Returns:
            Template dictionary
        """
        template_content = """# HIPAA Compliance Report: {{ subject }}

**Generated**: {{ generated_at }}

## Executive Summary

This report assesses compliance with the Health Insurance Portability and Accountability Act (HIPAA) for {{ subject }}.

**Overall Compliance Status**: {{ compliance_status }}

### Key Findings

- **Total Safeguards**: {{ total_safeguards }}
- **Compliant Safeguards**: {{ compliant_safeguards }}
- **Non-Compliant Safeguards**: {{ non_compliant_safeguards }}

## Administrative Safeguards

{% for safeguard in administrative_safeguards %}
### {{ safeguard.title }}

**Status**: {{ safeguard.status }}
**Description**: {{ safeguard.description }}

{% if safeguard.findings %}
**Findings**: {{ safeguard.findings }}
{% endif %}

---
{% endfor %}

## Physical Safeguards

{% for safeguard in physical_safeguards %}
### {{ safeguard.title }}

**Status**: {{ safeguard.status }}
**Description**: {{ safeguard.description }}

---
{% endfor %}

## Technical Safeguards

{% for safeguard in technical_safeguards %}
### {{ safeguard.title }}

**Status**: {{ safeguard.status }}
**Description**: {{ safeguard.description }}

---
{% endfor %}

## Recommendations

{% for recommendation in recommendations %}
- {{ recommendation }}
{% endfor %}
"""

        return {
            "name": "HIPAA Compliance Report Template",
            "description": "Standard template for HIPAA compliance reports",
            "template_content": template_content,
            "document_type": "compliance_report",
            "compliance_standards": ["HIPAA"],
            "resource_types": ["RDS", "S3", "EC2", "Lambda"],
            "template_engine": TemplateEngine.JINJA2,
            "output_formats": [OutputFormat.MARKDOWN, OutputFormat.HTML, OutputFormat.PDF],
            "variables": {
                "subject": {"type": "string", "required": True},
                "generated_at": {"type": "string", "required": True},
                "compliance_status": {"type": "string", "required": True},
                "total_safeguards": {"type": "integer", "required": True},
                "compliant_safeguards": {"type": "integer", "required": True},
                "non_compliant_safeguards": {"type": "integer", "required": True},
                "administrative_safeguards": {"type": "array", "required": True},
                "physical_safeguards": {"type": "array", "required": True},
                "technical_safeguards": {"type": "array", "required": True},
                "recommendations": {"type": "array", "required": True},
            },
            "sections": [
                "executive_summary",
                "administrative_safeguards",
                "physical_safeguards",
                "technical_safeguards",
                "recommendations",
            ],
        }

    @staticmethod
    def get_soc2_template() -> dict[str, Any]:
        """Get SOC 2 compliance report template.

        Returns:
            Template dictionary
        """
        template_content = """# SOC 2 Compliance Report: {{ subject }}

**Generated**: {{ generated_at }}
**Trust Services Criteria**: {{ trust_services_criteria }}

## Executive Summary

This report assesses compliance with SOC 2 Trust Services Criteria for {{ subject }}.

**Overall Compliance Status**: {{ compliance_status }}

## Trust Services Criteria Assessment

{% for criterion in trust_criteria %}
### {{ criterion.name }}

**Status**: {{ criterion.status }}
**Description**: {{ criterion.description }}

{% if criterion.controls %}
**Controls**:
{% for control in criterion.controls %}
- **{{ control.name }}**: {{ control.status }}
  - {{ control.description }}
{% endfor %}
{% endif %}

---
{% endfor %}

## Recommendations

{% for recommendation in recommendations %}
- {{ recommendation }}
{% endfor %}
"""

        return {
            "name": "SOC 2 Compliance Report Template",
            "description": "Standard template for SOC 2 compliance reports",
            "template_content": template_content,
            "document_type": "compliance_report",
            "compliance_standards": ["SOC2"],
            "resource_types": ["RDS", "S3", "EC2", "Lambda", "EKS"],
            "template_engine": TemplateEngine.JINJA2,
            "output_formats": [OutputFormat.MARKDOWN, OutputFormat.HTML, OutputFormat.PDF],
            "variables": {
                "subject": {"type": "string", "required": True},
                "generated_at": {"type": "string", "required": True},
                "trust_services_criteria": {"type": "string", "required": True},
                "compliance_status": {"type": "string", "required": True},
                "trust_criteria": {"type": "array", "required": True},
                "recommendations": {"type": "array", "required": True},
            },
            "sections": [
                "executive_summary",
                "trust_services_criteria_assessment",
                "recommendations",
            ],
        }

    @staticmethod
    def get_all_templates() -> dict[str, dict[str, Any]]:
        """Get all built-in templates.

        Returns:
            Dictionary mapping template names to template dictionaries
        """
        return {
            "pci_dss": TemplateLibrary.get_pci_dss_template(),
            "hipaa": TemplateLibrary.get_hipaa_template(),
            "soc2": TemplateLibrary.get_soc2_template(),
        }

