"""This module provides a CLI tool to generate a settings file from a .env file."""
import os

import click
from dotenv import dotenv_values


def _clean_var_name(name):
    """Converts a variable name like 'DATABASE_URL' to 'database_url'.

    Args:
        name: The variable name to clean.

    Returns:
        The cleaned variable name.
    """
    return name.lower()


def _guess_type_hint(value):
    """
    Guesses the type hint for a value based on its format.

    Args:
        value: The value to guess the type hint for.

    Returns:
        The guessed type hint as a string.
    """
    if isinstance(value, str):
        if value.lower() in ("true", "false"):
            return "bool"
        if value.replace('.', '', 1).isdigit():
            return "float" if '.' in value else "int"
        if "," in value:
            return "List[str]"
        if value.startswith("[") and value.endswith("]"):
            return "list"
        return "str"
    return "Any"


def _group_variables(env_vars, separator="__"):
    """
    Groups environment variables into nested dictionaries based on a separator.

    e.g., {"DATABASE__URL": "...", "DATABASE__PORT": "..."} -> {"DATABASE": {"URL": "...", "PORT": "..."}}

    Args:
        env_vars: A dictionary of environment variables.
        separator: The separator to use for grouping.

    Returns:
        A nested dictionary of grouped variables.
    """
    grouped = {}
    for key, value in env_vars.items():
        parts = key.split(separator)
        current_level = grouped
        for part in parts[:-1]:
            current_level = current_level.setdefault(part, {})
        current_level[parts[-1]] = value
    return grouped


def _generate_nested_model_code(grouped_vars, indent_level=1):
    """
    Recursively generates Pydantic models for grouped variables.

    Args:
        grouped_vars: A nested dictionary of grouped variables.
        indent_level: The current indentation level.

    Returns:
        A string containing the generated Pydantic models.
    """
    code = ""
    for group_name, group_data in grouped_vars.items():
        if isinstance(group_data, dict):
            model_name = f"{group_name.capitalize()}Settings"
            code += f"{'    ' * (indent_level - 1)}class {model_name}(BaseSettings):\n"
            code += _generate_nested_model_code(group_data, indent_level + 1)
            code += "\n"
        else:
            var_name = _clean_var_name(group_name)
            type_hint = _guess_type_hint(group_data)
            code += f"{'    ' * indent_level}{var_name.upper()}: {type_hint}\n"
    return code


def _generate_settings_file(env_file, output_path, separator):
    """
    Reads the .env file and generates the settings.py file.

    Args:
        env_file: The path to the .env file.
        output_path: The path to the generated settings.py file.
        separator: The separator used for nesting environment variables.
    """
    env_vars = dotenv_values(env_file)
    grouped_vars = _group_variables(env_vars, separator)

    # Generate nested models

    # Generate the main Settings model and final output
    main_model_code = "class Settings(BaseSettings):\n"
    main_model_code += _generate_nested_model_code(grouped_vars)
    main_model_code += "\n"
    main_model_code += "    model_config = SettingsConfigDict(\n"
    main_model_code += f"        env_file='{os.path.basename(env_file)}',\n"
    main_model_code += "        env_file_encoding='utf-8',\n"
    main_model_code += "        extra='ignore',\n"
    main_model_code += "    )"

    # Assemble the final Python file content
    content = f"""# THIS FILE IS AUTO-GENERATED BY THE 'generate_settings.py' CLI TOOL.
# DO NOT EDIT THIS FILE MANUALLY.
#
# You can modify the generated file, but be aware that re-running the tool
# will overwrite your changes.

from pydantic_settings import BaseSettings, SettingsConfigDict
from typing import Any, List


{main_model_code}


settings = Settings()
"""

    with open(output_path, "w") as f:
        f.write(content)


@click.command()
@click.option('--env-file', default=".env", help="Path to the .env file.")
@click.option('--output-path', default="app/core/settings.py", help="Path for the generated settings.py file.")
@click.option('--separator', default="__", help="Separator for nesting environment variables (e.g., `DATABASE__URL`).")
def generate_settings_cli(env_file, output_path, separator):
    """
    CLI tool to generate a settings.py file with nested Pydantic models
    based on an existing .env file.
    """
    if not os.path.exists(env_file):
        click.echo(f"Error: .env file not found at '{env_file}'", err=True)
        return

    _generate_settings_file(env_file, output_path, separator)
    click.echo(f"Successfully generated settings file at '{output_path}'")


if __name__ == '__main__':
    generate_settings_cli()
