#![allow(
    clippy::multiple_crate_versions,
    reason = "Dependency graph pulls distinct versions (e.g., yaml-rust2)."
)]
use std::fs::File;
use std::io::IsTerminal as _;
use std::io::{self, Read};
use std::path::{Path, PathBuf};

use anyhow::{Context, Result};
use clap::{ArgAction, Parser, ValueEnum};
use content_inspector::{ContentType, inspect};

type InputEntry = (String, Vec<u8>);
type InputEntries = Vec<InputEntry>;
type IgnoreNotices = Vec<String>;

#[derive(Parser, Debug)]
#[command(
    name = "headson",
    version,
    about = "Get a small but useful preview of JSON or YAML"
)]
struct Cli {
    #[arg(short = 'n', long = "budget")]
    budget: Option<usize>,
    #[arg(
        short = 'f',
        long = "format",
        value_enum,
        default_value_t = OutputFormat::Auto,
        help = "Output format: auto|json|yaml (filesets: auto is per-file)."
    )]
    format: OutputFormat,
    #[arg(
        short = 't',
        long = "template",
        value_enum,
        default_value_t = StyleArg::Default,
        help = "Output style: strict|default|detailed."
    )]
    style: StyleArg,
    #[arg(long = "indent", default_value = "  ")]
    indent: String,
    #[arg(long = "no-space", default_value_t = false)]
    no_space: bool,
    #[arg(
        long = "no-newline",
        default_value_t = false,
        help = "Do not add newlines in the output"
    )]
    no_newline: bool,
    #[arg(
        short = 'm',
        long = "compact",
        default_value_t = false,
        conflicts_with_all = ["no_space", "no_newline", "indent"],
        help = "Compact output with no added whitespace. Not very human-readable."
    )]
    compact: bool,
    #[arg(
        long = "string-cap",
        default_value_t = 500,
        help = "Maximum string length to display"
    )]
    string_cap: usize,
    #[arg(
        short = 'N',
        long = "global-budget",
        value_name = "BYTES",
        help = "Total output budget across all inputs. When combined with --budget, the effective global limit is the smaller of the two."
    )]
    global_budget: Option<usize>,
    #[arg(
        long = "tail",
        default_value_t = false,
        help = "Prefer the end of arrays when truncating. Strings unaffected; JSON stays strict."
    )]
    tail: bool,
    #[arg(
        long = "head",
        default_value_t = false,
        conflicts_with = "tail",
        help = "Prefer the beginning of arrays when truncating (keep first N)."
    )]
    head: bool,
    #[arg(
        long = "color",
        action = ArgAction::SetTrue,
        conflicts_with = "no_color",
        help = "Force enable ANSI colors in output"
    )]
    color: bool,
    #[arg(
        long = "no-color",
        action = ArgAction::SetTrue,
        conflicts_with = "color",
        help = "Disable ANSI colors in output"
    )]
    no_color: bool,
    #[arg(
        value_name = "INPUT",
        value_hint = clap::ValueHint::FilePath,
        num_args = 0..,
        help = "Optional file paths. If omitted, reads input from stdin. Multiple input files are supported. Directories and binary files are ignored with a notice on stderr."
    )]
    inputs: Vec<PathBuf>,
    #[arg(
        short = 'i',
        long = "input-format",
        value_enum,
        default_value_t = InputFormat::Json,
        help = "Input ingestion format: json or yaml."
    )]
    input_format: InputFormat,
}

#[derive(Copy, Clone, Debug, ValueEnum)]
enum OutputFormat {
    Auto,
    Json,
    Yaml,
}

#[derive(Copy, Clone, Debug, ValueEnum)]
enum StyleArg {
    Strict,
    Default,
    Detailed,
}

#[derive(Copy, Clone, Debug, ValueEnum)]
enum InputFormat {
    Json,
    Yaml,
}

fn main() -> Result<()> {
    let cli = Cli::parse();

    let render_cfg = get_render_config_from(&cli);
    // Resolve color auto-detection now (stdout is the surface for user output).
    let _color_enabled =
        render_cfg.color_mode.effective(io::stdout().is_terminal());
    let (output, ignore_notices) = if cli.inputs.is_empty() {
        (run_from_stdin(&cli, &render_cfg)?, Vec::new())
    } else {
        run_from_paths(&cli, &render_cfg)?
    };
    println!("{output}");

    for notice in ignore_notices {
        eprintln!("{notice}");
    }

    Ok(())
}

fn compute_effective_budget(cli: &Cli, input_count: usize) -> usize {
    match (cli.global_budget, cli.budget) {
        (Some(g), Some(n)) => g.min(n.saturating_mul(input_count)),
        (Some(g), None) => g,
        (None, Some(n)) => n.saturating_mul(input_count),
        (None, None) => 500usize.saturating_mul(input_count),
    }
}

fn compute_priority(
    cli: &Cli,
    effective_budget: usize,
    input_count: usize,
) -> headson::PriorityConfig {
    let per_file_for_priority =
        if cli.global_budget.is_some() && cli.budget.is_some() {
            // When both limits are provided, base per-file heuristics on the per-file
            // budget but also respect the effective per-file slice of the final global.
            let eff_per_file = (effective_budget / input_count.max(1)).max(1);
            cli.budget.unwrap().min(eff_per_file).max(1)
        } else {
            (effective_budget / input_count.max(1)).max(1)
        };
    get_priority_config(per_file_for_priority, cli)
}

fn run_from_stdin(
    cli: &Cli,
    render_cfg: &headson::RenderConfig,
) -> Result<String> {
    let input_bytes = read_stdin()?;
    let input_count = 1usize;
    let eff = compute_effective_budget(cli, input_count);
    let prio = compute_priority(cli, eff, input_count);
    let mut cfg = render_cfg.clone();
    // Resolve effective output template for stdin:
    cfg.template = resolve_effective_template_for_stdin(cli.format, cfg.style);
    match cli.input_format {
        InputFormat::Json => headson::headson(input_bytes, &cfg, &prio, eff),
        InputFormat::Yaml => {
            headson::headson_yaml(input_bytes, &cfg, &prio, eff)
        }
    }
}

#[allow(
    clippy::cognitive_complexity,
    reason = "Keeps ingest selection and rendering logic co-located for clarity."
)]
fn run_from_paths(
    cli: &Cli,
    render_cfg: &headson::RenderConfig,
) -> Result<(String, IgnoreNotices)> {
    let (entries, ignored) = ingest_paths(&cli.inputs)?;
    let included = entries.len();
    let input_count = included.max(1);
    let eff = compute_effective_budget(cli, input_count);
    let prio = compute_priority(cli, eff, input_count);
    // In Auto template mode, choose ingestion strategy based on extensions for filesets:
    // if any included input has a YAML extension, prefer YAML ingest (can parse JSON too).
    fn any_yaml_ext(entries: &InputEntries) -> bool {
        entries.iter().any(|(name, _)| {
            let lower = name.to_ascii_lowercase();
            lower.ends_with(".yaml") || lower.ends_with(".yml")
        })
    }
    if cli.inputs.len() > 1 {
        let chosen_input = if matches!(cli.format, OutputFormat::Auto)
            && any_yaml_ext(&entries)
        {
            InputFormat::Yaml
        } else {
            cli.input_format
        };
        let mut cfg = render_cfg.clone();
        // For filesets: if format=auto, enable per-file template selection.
        cfg.template = match cli.format {
            OutputFormat::Auto => headson::OutputTemplate::Auto,
            OutputFormat::Json => map_json_template_for_style(cfg.style),
            OutputFormat::Yaml => headson::OutputTemplate::Yaml,
        };
        let out = match chosen_input {
            InputFormat::Json => {
                headson::headson_many(entries, &cfg, &prio, eff)?
            }
            InputFormat::Yaml => {
                headson::headson_many_yaml(entries, &cfg, &prio, eff)?
            }
        };
        Ok((out, ignored))
    } else if included == 0 {
        Ok((String::new(), ignored))
    } else {
        let (name, bytes) = entries.into_iter().next().unwrap();
        // Single file: pick ingest and output template per CLI format+style.
        let lower = name.to_ascii_lowercase();
        let is_yaml_ext = lower.ends_with(".yaml") || lower.ends_with(".yml");
        let chosen_input = match cli.format {
            OutputFormat::Auto => {
                if is_yaml_ext {
                    InputFormat::Yaml
                } else {
                    InputFormat::Json
                }
            }
            _ => cli.input_format,
        };
        let mut cfg = render_cfg.clone();
        cfg.template = resolve_effective_template_for_single(
            cli.format, cfg.style, &lower,
        );
        let out = match chosen_input {
            InputFormat::Json => headson::headson(bytes, &cfg, &prio, eff)?,
            InputFormat::Yaml => {
                headson::headson_yaml(bytes, &cfg, &prio, eff)?
            }
        };
        Ok((out, ignored))
    }
}

fn read_stdin() -> Result<Vec<u8>> {
    let mut buf = Vec::new();
    io::stdin()
        .read_to_end(&mut buf)
        .context("failed to read from stdin")?;
    Ok(buf)
}

fn sniff_then_read_text(path: &Path) -> Result<Option<Vec<u8>>> {
    // Inspect the first chunk with content_inspector; if it looks binary, skip.
    // Otherwise, read the remainder without further inspection for speed.
    const CHUNK: usize = 64 * 1024;
    let file = File::open(path).with_context(|| {
        format!("failed to open input file: {}", path.display())
    })?;
    let meta_len = file.metadata().ok().map(|m| m.len());
    let mut reader = io::BufReader::with_capacity(CHUNK, file);

    let mut first = [0u8; CHUNK];
    let n = reader.read(&mut first).with_context(|| {
        format!("failed to read input file: {}", path.display())
    })?;
    if n == 0 {
        return Ok(Some(Vec::new()));
    }
    if matches!(inspect(&first[..n]), ContentType::BINARY) {
        return Ok(None);
    }

    // Preallocate buffer: first chunk + estimated remainder (capped)
    let mut buf = Vec::with_capacity(
        n + meta_len
            .map(|m| m.saturating_sub(n as u64) as usize)
            .unwrap_or(0)
            .min(8 * 1024 * 1024),
    );
    buf.extend_from_slice(&first[..n]);
    reader.read_to_end(&mut buf).with_context(|| {
        format!("failed to read input file: {}", path.display())
    })?;
    Ok(Some(buf))
}

fn ingest_paths(paths: &[PathBuf]) -> Result<(InputEntries, IgnoreNotices)> {
    let mut out: InputEntries = Vec::with_capacity(paths.len());
    let mut ignored: IgnoreNotices = Vec::new();
    for path in paths.iter() {
        let display = path.display().to_string();
        if let Ok(meta) = std::fs::metadata(path) {
            if meta.is_dir() {
                ignored.push(format!("Ignored directory: {display}"));
                continue;
            }
        }
        if let Some(bytes) = sniff_then_read_text(path)? {
            out.push((display, bytes))
        } else {
            ignored.push(format!("Ignored binary file: {display}"));
            continue;
        }
    }
    Ok((out, ignored))
}

fn get_render_config_from(cli: &Cli) -> headson::RenderConfig {
    fn color_mode_from_flags(cli: &Cli) -> headson::ColorMode {
        if cli.color {
            headson::ColorMode::On
        } else if cli.no_color {
            headson::ColorMode::Off
        } else {
            headson::ColorMode::Auto
        }
    }

    // Select a baseline template; may be overridden per-input later.
    let template = match cli.format {
        OutputFormat::Auto => headson::OutputTemplate::Auto,
        OutputFormat::Json => {
            map_json_template_for_style(map_style(cli.style))
        }
        OutputFormat::Yaml => headson::OutputTemplate::Yaml,
    };
    let space = if cli.compact || cli.no_space { "" } else { " " }.to_string();
    let newline = if cli.compact || cli.no_newline {
        ""
    } else {
        "\n"
    }
    .to_string();
    let indent_unit = if cli.compact {
        String::new()
    } else {
        cli.indent.clone()
    };
    let color_mode = color_mode_from_flags(cli);
    let color_enabled = headson::resolve_color_enabled(color_mode);

    headson::RenderConfig {
        template,
        indent_unit,
        space,
        newline,
        prefer_tail_arrays: cli.tail,
        color_mode,
        color_enabled,
        style: map_style(cli.style),
    }
}

fn get_priority_config(
    per_file_budget: usize,
    cli: &Cli,
) -> headson::PriorityConfig {
    headson::PriorityConfig {
        max_string_graphemes: cli.string_cap,
        array_max_items: (per_file_budget / 2).max(1),
        prefer_tail_arrays: cli.tail,
        array_bias: headson::ArrayBias::HeadMidTail,
        array_sampler: if cli.tail {
            headson::ArraySamplerStrategy::Tail
        } else if cli.head {
            headson::ArraySamplerStrategy::Head
        } else {
            headson::ArraySamplerStrategy::Default
        },
    }
}

fn map_style(s: StyleArg) -> headson::Style {
    match s {
        StyleArg::Strict => headson::Style::Strict,
        StyleArg::Default => headson::Style::Default,
        StyleArg::Detailed => headson::Style::Detailed,
    }
}

fn map_json_template_for_style(
    style: headson::Style,
) -> headson::OutputTemplate {
    match style {
        headson::Style::Strict => headson::OutputTemplate::Json,
        headson::Style::Default => headson::OutputTemplate::Pseudo,
        headson::Style::Detailed => headson::OutputTemplate::Js,
    }
}

fn resolve_effective_template_for_stdin(
    fmt: OutputFormat,
    style: headson::Style,
) -> headson::OutputTemplate {
    match fmt {
        OutputFormat::Auto | OutputFormat::Json => {
            map_json_template_for_style(style)
        }
        OutputFormat::Yaml => headson::OutputTemplate::Yaml,
    }
}

fn resolve_effective_template_for_single(
    fmt: OutputFormat,
    style: headson::Style,
    lower_name: &str,
) -> headson::OutputTemplate {
    match fmt {
        OutputFormat::Json => map_json_template_for_style(style),
        OutputFormat::Yaml => headson::OutputTemplate::Yaml,
        OutputFormat::Auto => {
            if lower_name.ends_with(".yaml") || lower_name.ends_with(".yml") {
                headson::OutputTemplate::Yaml
            } else if lower_name.ends_with(".json") {
                map_json_template_for_style(style)
            } else {
                // Unknown: pick based on style for JSON family.
                map_json_template_for_style(style)
            }
        }
    }
}
