#!/usr/bin/env python
"""GUI tool for viewing and exploring EIS hdf5 and fit files

A PyQt GUI for loading the exploring EIS rasters. Can load both level-1 HDF5
files and .fit.h5 files generated by EISPAC. Clicking on the raster image will
display the spectrum at that location.

(2024-June-05) First beta version for testing
(2024-June-11) Various bug fixes and small improvements
(2024-July-10) Added CHIANTI lines and integrated with EISPAC

"""
__all__ = ['eis_explore_raster']

import sys
import os
import glob
import copy
import numpy as np
import shutil
from PyQt5 import QtCore, QtWidgets, QtGui

# from PyQt5 import Qt, QtCore, QtGui, QtWidgets
# from PyQt5.QtGui import QIcon, QPixmap, QImage
# from PyQt5.QtWidgets import *
# from PyQt5.QtCore import *

try:
    # Newer version of matplotlib (has support for Qt6)
    from matplotlib.backends.backend_qtagg import FigureCanvas
    from matplotlib.backends.backend_qtagg import NavigationToolbar2QT as NavigationToolbar
except:
    # Fallback imports (should still work in newer version of matplotlib)
    from matplotlib.backends.backend_qt5agg import FigureCanvas
    from matplotlib.backends.backend_qt5agg import NavigationToolbar2QT as NavigationToolbar

from matplotlib.backend_bases import MouseButton
from matplotlib.backends.qt_compat import QtWidgets
from matplotlib.figure import Figure
from matplotlib.ticker import FormatStrFormatter

import astropy.units as u
from astropy.coordinates import SkyCoord
from astropy.wcs.utils import wcs_to_celestial_frame
from astropy.visualization import ImageNormalize, AsinhStretch, LinearStretch, SqrtStretch, LogStretch

import eispac

#@window########################################################################
# MAIN GUI WINDOW ##############################################################
################################################################################
class MainWindow(QtWidgets.QWidget):
    def __init__(self, input_filename=None):
        super().__init__()
        self.font_default = QtGui.QFont("Arial", 12)
        self.font_default_italic = QtGui.QFont("Arial", 12, italic=True)
        self.font_default_bold = QtGui.QFont("Arial", 12, weight=QtGui.QFont.Bold)
        self.font_small = QtGui.QFont("Arial", 10)
        self.font_info = QtGui.QFont("Courier New", 9)
        self.base_width = 85 #75
        # Note, most screens can only handle a max of four images
        # Set higher self.max_num_rasters at your own risk!
        self.max_num_rasters = 6 # MUST be between 2 to 9! (<= 4 is best)

        ### INIT EMPTY LISTS USED BY THE APPLICATION
        self.all_line_ids = eispac.data.load_chianti_lines(sort='wave')

        # PyQT grids, layouts, and key reference indices and parameters
        self.main_grid = None
        self.sub_frame = [None]*self.max_num_rasters
        self.sub_grid = [None]*self.max_num_rasters
        self.curr_layout_ind = None
        self.lim_dialog = None
        self.dialog_r_ind = None
        self.focused_r_ind = None
        self.focused_type = None
        self.focused_fig = None
        
        # EIS data objects and ref values
        self.display_mode = [None]*self.max_num_rasters
        self.selected_dir = [None]*self.max_num_rasters
        self.selected_file = [None]*self.max_num_rasters
        self.file_date_obs = [None]*self.max_num_rasters
        self.fit_filepath = [None]*self.max_num_rasters
        self.fit_res = [None]*self.max_num_rasters
        self.cube_filepath = [None]*self.max_num_rasters
        self.eis_cube = [None]*self.max_num_rasters
        self.obs_type = [None]*self.max_num_rasters #only used for obs data
        self.var_label = [None]*self.max_num_rasters
        self.inten_units = [None]*self.max_num_rasters
        self.cube_win_ind = [None]*self.max_num_rasters
        self.wave_bin_width = [None]*self.max_num_rasters #only used for obs data
        self.wave_sum_range = [None]*self.max_num_rasters #only used for obs data
        self.rast_xcoords = [None]*self.max_num_rasters
        self.rast_ycoords = [None]*self.max_num_rasters
        self.rast_wcoords = [None]*self.max_num_rasters #only used for obs data
        self.rast_slice_ind = [None]*self.max_num_rasters #only used for CCD
        self.rast_slice_coord = [None]*self.max_num_rasters #only used for CCD
        self.rast_xaxis_type = [None]*self.max_num_rasters
        self.rast_auto_aspect = [None]*self.max_num_rasters
        self.rast_xydelta = [None]*self.max_num_rasters
        self.spec_xydelta = [None]*self.max_num_rasters

        # Controls and plots for each image
        self.button_select = [None]*self.max_num_rasters
        self.box_file_list = [None]*self.max_num_rasters
        self.box_cmap = [None]*self.max_num_rasters
        self.box_cnorm = [None]*self.max_num_rasters
        self.box_line_id = [None]*self.max_num_rasters
        self.box_var = [None]*self.max_num_rasters
        self.rast_nav = [None]*self.max_num_rasters
        self.rast_fig = [None]*self.max_num_rasters
        self.rast_canvas = [None]*self.max_num_rasters
        self.rast_ax = [None]*self.max_num_rasters
        self.rast_img = [None]*self.max_num_rasters
        self.rast_lims = [None]*self.max_num_rasters
        self.button_rast_lims = [None]*self.max_num_rasters
        self.button_reset_rast = [None]*self.max_num_rasters
        self.spec_nav = [None]*self.max_num_rasters
        self.spec_fig = [None]*self.max_num_rasters
        self.spec_canvas = [None]*self.max_num_rasters
        self.spec_ax = [None]*self.max_num_rasters
        self.spec_lims = [None]*self.max_num_rasters
        self.button_reset_spec = [None]*self.max_num_rasters

        # Matplotlib click events and selected values
        self.rast_click = [None]*self.max_num_rasters
        self.rast_keyboard = [None]*self.max_num_rasters
        self.rast_crosshair = [None]*self.max_num_rasters
        self.rast_axhline = [None]*self.max_num_rasters
        self.rast_axvline = [None]*self.max_num_rasters
        self.crosshair_new_coords = [None]*self.max_num_rasters
        self.crosshair_old_coords = [None]*self.max_num_rasters
        self.crosshair_ind = [None]*self.max_num_rasters
        self.crosshair_val = [None]*self.max_num_rasters
        self.spec_click = [None]*self.max_num_rasters
        self.spec_keyboard = [None]*self.max_num_rasters
        self.spec_vline_val = [None]*self.max_num_rasters
        self.spec_vline_ind = [None]*self.max_num_rasters

        self.initUI()

        # Loading an input file
        if input_filename is not None:
            self.load_file(input_filename, load_in_all=True)

    def initUI(self):
        """initialize all of the Qt widgets"""
        count_list = ['Single image', 'Two images', 'Three images', 
                      'Four images', 'Five images', 'Six images',
                      'Seven images', 'Eight images', 'Nine images']
        layout_list = [count_list[r] for r in range(self.max_num_rasters)]

        self.inten_cmap_list = ['Blues_r', 'gray', 'gist_heat',
                                'inferno', 'plasma', 'viridis', 
                                'turbo', 'cubehelix', 
                                'sdoaia171', 'sdoaia193', 'sdoaia211',
                                'sohoeit171', 'sohoeit195', 'sohoeit284']
        self.vel_cmap_list = ['RdBu_r', 'coolwarm', 'seismic', 
                              'Spectral_r', 'RdYlBu_r', 'PuOr_r', 'hmimag']
        
        self.default_crosshair_color = 'violet'
        self.vel_crosshair_color = 'blueviolet'
        self.alt_crosshair_color = 'deepskyblue'
        self.alt_crosshair_cmaps = ['gist_heat', 'inferno', 'plasma', 
                                    'cubehelix', 'sdoaia211']
        
        # Basic buttons and info on far left
        self.button_quit = QtWidgets.QPushButton('Quit')
        self.button_quit.setFont(self.font_default)
        self.button_quit.clicked.connect(self.event_quit)
        self.button_quit.setFixedWidth(2*self.base_width)
        # self.button_quit.setFixedHeight(40)

        self.button_open_all = QtWidgets.QPushButton('Open in ALL')
        self.button_open_all.setFont(self.font_default)
        self.button_open_all.clicked.connect(self.event_open_all)
        self.button_open_all.setFixedWidth(2*self.base_width)

        self.box_layout = QtWidgets.QComboBox()
        self.box_layout.addItems(layout_list)
        self.box_layout.setFixedWidth(2*self.base_width)
        self.box_layout.setFont(self.font_default)
        self.box_layout.setCurrentIndex(1)
        self.box_layout.currentIndexChanged.connect(self.event_change_layout)
        
        self.radio_placeholder = QtWidgets.QCheckBox("[not implemented]")
        self.radio_placeholder.setFixedWidth(2*self.base_width)
        self.radio_placeholder.setFont(self.font_default)
        self.radio_placeholder.setChecked(True)

        self.radio_sync_cross = QtWidgets.QCheckBox("Sync crosshairs")
        self.radio_sync_cross.setFixedWidth(2*self.base_width)
        self.radio_sync_cross.setFont(self.font_default)
        self.radio_sync_cross.setChecked(True)

        self.label_tips = QtWidgets.QLabel()
        self.label_tips.setFont(self.font_default_italic)
        self.label_tips.setText("Middle click to toggle\npan/zoom mode"
                                +"\n   Pan (left click)"
                                +"\n   Zoom (right click)"
                                +"\n                   \u2191 y-in"
                                +"\n    x-out \u2190 + \u2192 x-in"
                                +"\n                   \u2193 y-out")
        self.label_tips.setWordWrap(True)

        self.radio_autoscale_ylim = QtWidgets.QCheckBox("Autoscale Y-limits")
        self.radio_autoscale_ylim.setFixedWidth(2*self.base_width)
        self.radio_autoscale_ylim.setFont(self.font_default)
        self.radio_autoscale_ylim.setChecked(True)
        self.radio_autoscale_ylim.stateChanged.connect(self.event_replot_all_spec)

        self.radio_sync_vline = QtWidgets.QCheckBox("Sync spec vlines")
        self.radio_sync_vline.setFixedWidth(2*self.base_width)
        self.radio_sync_vline.setFont(self.font_default)
        self.radio_sync_vline.setChecked(True)

        self.radio_show_ids = QtWidgets.QCheckBox("Show line IDs")
        self.radio_show_ids.setFixedWidth(2*self.base_width)
        self.radio_show_ids.setFont(self.font_default)
        self.radio_show_ids.setChecked(True)
        self.radio_show_ids.stateChanged.connect(self.event_replot_all_spec)

        self.radio_show_details = QtWidgets.QCheckBox("Show details")
        self.radio_show_details.setFixedWidth(2*self.base_width)
        self.radio_show_details.setFont(self.font_default)
        self.radio_show_details.setChecked(True)
        self.radio_show_details.stateChanged.connect(self.event_replot_all_spec)

        # Sets of controls and figures for each raster
        for r in range(self.max_num_rasters):
            # Buttons and drop-down lsits
            self.button_select[r] = QtWidgets.QPushButton('Open')
            self.button_select[r].setObjectName(f'r{r}_select')
            self.button_select[r].setFont(self.font_default)
            self.button_select[r].clicked.connect(self.event_select_file)
            self.button_select[r].setFixedWidth(self.base_width)

            self.box_file_list[r] = QtWidgets.QComboBox()
            self.box_file_list[r].setObjectName(f'r{r}_file_list')
            self.box_file_list[r].addItems(['No file selected'])
            self.box_file_list[r].setFont(self.font_default)
            self.box_file_list[r].currentIndexChanged.connect(self.event_quick_select)

            self.box_cmap[r] = QtWidgets.QComboBox()
            self.box_cmap[r].setObjectName(f'r{r}_cmap')
            self.box_cmap[r].addItems(['colormap'])
            self.box_cmap[r].setFixedWidth(self.base_width)
            self.box_cmap[r].setFont(self.font_small)
            self.box_cmap[r].setCurrentIndex(0)
            self.box_cmap[r].currentIndexChanged.connect(self.event_plot_raster)

            self.box_line_id[r] = QtWidgets.QComboBox()
            self.box_line_id[r].setObjectName(f'r{r}_line_id')
            self.box_line_id[r].addItems(['Window/Line'])
            self.box_line_id[r].setFont(self.font_default)
            self.box_line_id[r].currentIndexChanged.connect(self.event_plot_raster)

            self.box_var[r] = QtWidgets.QComboBox()
            self.box_var[r].setObjectName(f'r{r}_var')
            self.box_var[r].addItems(['Range/Variable'])
            self.box_var[r].setFont(self.font_default)
            self.box_var[r].currentIndexChanged.connect(self.event_plot_raster)

            self.box_cnorm[r] = QtWidgets.QComboBox()
            self.box_cnorm[r].setObjectName(f'r{r}_cnorm')
            self.box_cnorm[r].addItems(['asinh norm', 'linear norm', 'sqrt norm', 'log norm'])
            self.box_cnorm[r].setFixedWidth(self.base_width)
            self.box_cnorm[r].setFont(self.font_small)
            self.box_cnorm[r].setCurrentIndex(1)
            self.box_cnorm[r].currentIndexChanged.connect(self.event_plot_raster)

            self.button_rast_lims[r] = QtWidgets.QPushButton('Set Limits / Crosshair')
            self.button_rast_lims[r].setObjectName(f'r{r}_rast_limits')
            self.button_rast_lims[r].setFont(self.font_default)
            self.button_rast_lims[r].clicked.connect(self.event_lim_dialog)

            self.button_reset_rast[r] = QtWidgets.QPushButton('Reset')
            self.button_reset_rast[r].setObjectName(f'r{r}_rast_reset')
            self.button_reset_rast[r].setFont(self.font_default)
            self.button_reset_rast[r].clicked.connect(self.event_reset_plot)
            self.button_reset_rast[r].setFixedWidth(self.base_width)

            self.button_reset_spec[r] = QtWidgets.QPushButton('Reset')
            self.button_reset_spec[r].setObjectName(f'r{r}_spec_reset')
            self.button_reset_spec[r].setFont(self.font_default)
            self.button_reset_spec[r].clicked.connect(self.event_reset_plot)
            self.button_reset_spec[r].setFixedWidth(self.base_width)

            # Setup the matplotlib figures
            self.rast_fig[r] = Figure(figsize=(3, 5), constrained_layout=False)
            self.rast_canvas[r] = FigureCanvas(self.rast_fig[r])
            self.rast_canvas[r].setFocusPolicy(QtCore.Qt.ClickFocus)
            self.rast_canvas[r].AltObjectName = f'r{r}_rast_canvas'
            self.rast_nav[r] = NavigationToolbar(self.rast_canvas[r], self)
                        
            self.spec_fig[r] = Figure(figsize=(2, 2), constrained_layout=True)
            self.spec_canvas[r] = FigureCanvas(self.spec_fig[r])
            self.spec_canvas[r].setFocusPolicy(QtCore.Qt.ClickFocus)
            self.spec_canvas[r].AltObjectName = f'r{r}_spec_canvas'
            self.spec_nav[r] = NavigationToolbar(self.spec_canvas[r], self)

        self.arange_layout(layout_index=1)
        self.setGeometry(50, 100, 1600, 900)
        self.setWindowTitle('EIS Explore Raster')
        self.show()

    def arange_layout(self, layout_index=0):
        """Place the widgets in the grid and show/hide frames as needed"""
        if self.main_grid is None:
            self.main_grid = QtWidgets.QGridLayout(self)
            # self.main_grid.setVerticalSpacing(0)
            self.main_grid.addWidget(self.button_quit, 0, 0)
            self.main_grid.addWidget(self.button_open_all, 1, 0)
            self.main_grid.addWidget(self.box_layout, 2, 0)
            self.main_grid.addWidget(self.radio_placeholder, 3, 0) # <-- NEW!
            self.main_grid.addWidget(self.radio_sync_cross, 4, 0)
            self.main_grid.addWidget(self.label_tips, 5, 0) 
            self.main_grid.addWidget(self.radio_autoscale_ylim, 7, 0)
            self.main_grid.addWidget(self.radio_sync_vline, 8, 0)
            self.main_grid.addWidget(self.radio_show_ids, 9, 0)
            self.main_grid.addWidget(self.radio_show_details, 10, 0)

            # Setup frames and grids for each figure (better control of layout)
            for r in range(0, self.max_num_rasters):
                self.sub_frame[r] = QtWidgets.QFrame()
                self.sub_grid[r] = QtWidgets.QGridLayout(self.sub_frame[r])
                self.sub_grid[r].setContentsMargins(0, 0, 0, 0)
                self.sub_grid[r].addWidget(self.button_select[r], 0, 0)
                self.sub_grid[r].addWidget(self.box_file_list[r], 0, 1, 1, 2)
                self.sub_grid[r].addWidget(self.box_cmap[r], 1, 0)
                self.sub_grid[r].addWidget(self.box_line_id[r], 1, 1)
                self.sub_grid[r].addWidget(self.box_var[r], 1, 2)
                self.sub_grid[r].addWidget(self.box_cnorm[r], 2, 0)
                self.sub_grid[r].addWidget(self.button_rast_lims[r], 2, 2)
                self.sub_grid[r].addWidget(self.button_reset_rast[r], 3, 0)
                self.sub_grid[r].addWidget(self.rast_nav[r], 3, 1, 1, 2)
                self.sub_grid[r].addWidget(self.rast_canvas[r], 4, 0, 4, 3)
                self.sub_grid[r].addWidget(self.spec_canvas[r], 8, 0, 2, 3)
                self.sub_grid[r].addWidget(self.button_reset_spec[r], 10, 0)
                self.sub_grid[r].addWidget(self.spec_nav[r], 10, 1, 1, 2)

                self.sub_frame[r].setLayout(self.sub_grid[r])
                self.sub_frame[r].sizePolicy().setHorizontalStretch(1)

        if layout_index == 0:
            ## Single image: Two col layout of raster (left) & spectrum (right)
            
            # Rearange first panel of plots into the two col format (full window width)
            self.sub_grid[0].addWidget(self.rast_canvas[0], 4, 0, 6, 3)
            self.sub_grid[0].addWidget(self.spec_canvas[0], 4, 3, 3, 3)
            self.sub_grid[0].addWidget(self.button_reset_spec[0], 3, 3)
            self.sub_grid[0].addWidget(self.spec_nav[0], 3, 4, 1, 2)
            self.main_grid.addWidget(self.sub_frame[0], 0, 1, 11, 3*self.max_num_rasters)

            # Hide all unused figure frames
            for r in range(1, self.max_num_rasters):
                self.main_grid.addWidget(self.sub_frame[r], 5+r, 4)
                self.sub_frame[r].hide()

            # Setting stretch on grid for better resizing
            self.main_grid.setColumnStretch(0, 0)
            self.main_grid.setColumnStretch(1, 0)
            self.main_grid.setColumnStretch(2, 3)
            self.main_grid.setColumnStretch(3, 3)
            self.main_grid.setColumnStretch(1, 1)
            self.main_grid.setColumnStretch(2, 1)
            self.main_grid.setColumnStretch(3, 1)
            self.main_grid.setRowStretch(4, 3) 
            self.main_grid.setRowStretch(8, 1) 
            if self.max_num_rasters >= 3:
                for r in range(2, self.max_num_rasters):
                    self.main_grid.setColumnStretch(1+3*r, 0)
                    self.main_grid.setColumnStretch(2+3*r, 0)
                    self.main_grid.setColumnStretch(3+3*r, 0)

        elif layout_index >= 1:
            # Multiple images: Raster (top) & spectrum (bottom) in SAME col
            
            # Ensure first panel of plots is the single col format
            self.sub_grid[0].addWidget(self.rast_canvas[0], 4, 0, 4, 3)
            self.sub_grid[0].addWidget(self.spec_canvas[0], 8, 0, 2, 3)
            self.sub_grid[0].addWidget(self.button_reset_spec[0], 10, 0)
            self.sub_grid[0].addWidget(self.spec_nav[0], 10, 1, 1, 2)
            self.main_grid.addWidget(self.sub_frame[0], 0, 1, 11, 3)

            # Show only the selected number of figures
            for r in range(1, self.max_num_rasters):
                if r <= layout_index:
                    self.main_grid.addWidget(self.sub_frame[r], 0, 1+3*r, 11, 3) 
                    self.sub_frame[r].show()
                else:
                    self.main_grid.addWidget(self.sub_frame[r], 0, 0+r)
                    self.sub_frame[r].hide()
            
            # Setting stretch on grid for better resizing
            self.main_grid.setColumnStretch(0, 0)
            self.main_grid.setRowStretch(4, 3) 
            self.main_grid.setRowStretch(8, 1)
            for r in range(0, self.max_num_rasters):
                if r <= layout_index:
                    self.main_grid.setColumnStretch(1+3*r, 0)
                    self.main_grid.setColumnStretch(2+3*r, 1)
                    self.main_grid.setColumnStretch(3+3*r, 1)
                else:
                    self.main_grid.setColumnMinimumWidth(1+3*r, 0)
                    self.main_grid.setColumnMinimumWidth(2+3*r, 0)
                    self.main_grid.setColumnMinimumWidth(3+3*r, 0)
                    self.main_grid.setColumnStretch(1+3*r, 0)
                    self.main_grid.setColumnStretch(2+3*r, 0)
                    self.main_grid.setColumnStretch(3+3*r, 0)

            # # Resize spectra to match - does not really work right...
            # for r in range(0, layout_index):
            #     sp_wid, sp_hei = self.spec_canvas[r].get_width_height()
            #     self.spec_canvas[r].resize(sp_wid, 200)

        # Display the widget grid
        self.curr_layout_ind = layout_index
        self.setLayout(self.main_grid)

        # Replot all images now that shown/hidden state has changed
        for r in range(self.max_num_rasters):
            self.plot_raster(r_ind=r)

    def load_file(self, filepath, r_ind=0, load_in_all=False):
        """Load an EIS level-1 or fit result HDF5 file"""
        if load_in_all:
            r_ind = 0

        if os.path.isfile(filepath) and filepath.endswith('.fit.h5'):
            # [FIT MODE] Load fit results
            self.display_mode[r_ind] = 'fit'
            self.fit_filepath[r_ind] = filepath
            self.fit_res[r_ind] = eispac.read_fit(filepath)
            self.inten_units[r_ind] = self.fit_res[r_ind].data_units
            file_dir = os.path.dirname(filepath)
            file_nraster = self.fit_res[r_ind].meta['mod_index']['nraster']
            try:
                # Look for level-1 data in original source location
                this_cube_filepath = self.fit_res[r_ind].meta['filename_data']
                if not os.path.isfile(this_cube_filepath):
                    this_cube_filepath = None
            except:
                this_cube_filepath = None
            if this_cube_filepath is None:
                # Look for level-1 data in local directory
                base_eis_filename = os.path.basename(filepath)
                base_eis_filename = base_eis_filename.split('.')[0]
                this_cube_filepath = os.path.join(file_dir, base_eis_filename+'.data.h5')
            if os.path.isfile(this_cube_filepath):
                # Load EISCube object (window SET to match fit data)
                mean_wave = np.mean(self.fit_res[r_ind].fit['wavelength'][0,0,:])
                apply_radcal = self.inten_units[r_ind].lower() != 'photon'
                self.cube_filepath[r_ind] = this_cube_filepath
                self.eis_cube[r_ind] = eispac.read_cube(this_cube_filepath, mean_wave, 
                                                        apply_radcal=apply_radcal)
                self.cube_win_ind[r_ind] = self.eis_cube[r_ind].meta['iwin']
                self.obs_type[r_ind] = self.eis_cube[r_ind].meta['mod_index']['obs_type']
                self.check_fit_and_cube_dims(r_ind=r_ind)
            else:
                # Clear old object
                self.cube_filepath[r_ind] = None
                self.eis_cube[r_ind] = None
                self.cube_win_ind[r_ind] = None
                self.obs_type[r_ind] = None

        elif os.path.isfile(filepath) and filepath.endswith(('.data.h5', '.head.h5')):
            # [OBS MODE] Load EISCube object (window CAN be selected later)
            filepath = filepath.replace('.head.', '.data.')
            self.display_mode[r_ind] = 'obs'
            self.fit_filepath[r_ind] = None
            self.fit_res[r_ind] = None
            self.cube_filepath[r_ind] = filepath
            self.eis_cube[r_ind] = eispac.read_cube(filepath, window=0, 
                                                    apply_radcal=True)
            self.inten_units[r_ind] = self.eis_cube[r_ind].meta['mod_index']['bunit']
            self.obs_type[r_ind] = self.eis_cube[r_ind].meta['mod_index']['obs_type']
            self.rast_wcoords[r_ind] = None
            file_nraster = self.eis_cube[r_ind].meta['mod_index']['nraster']
            # self.spec_vline_ind[r_ind] = None
            # self.spec_vline_val[r_ind] = None
        else:
            return # If invalid file, do nothing
        
        # Append raster type to display mode
        if file_nraster == 1:
            # Sit-and-stare data
            self.display_mode[r_ind] = self.display_mode[r_ind] + 'sns'
        else:
            # Scan data
            self.display_mode[r_ind] = self.display_mode[r_ind] + 'scan'
            
        # Update text and plots
        self.update_filepath(r_ind=r_ind)
        self.update_line_id_box(r_ind=r_ind)
        self.update_var_box(r_ind=r_ind)
        self.plot_raster(r_ind=r_ind)

        if load_in_all:
            for r in range(1, self.max_num_rasters):
                # Copy over objects (no need to reload)
                self.display_mode[r] = copy.deepcopy(self.display_mode[0])
                self.fit_filepath[r] = copy.deepcopy(self.fit_filepath[0])
                self.fit_res[r] = copy.deepcopy(self.fit_res[0])
                self.cube_filepath[r] = copy.deepcopy(self.cube_filepath[0])
                self.eis_cube[r] = copy.deepcopy(self.eis_cube[0])
                self.inten_units[r] = copy.deepcopy(self.inten_units[0])
                self.obs_type[r] = copy.deepcopy(self.obs_type[0])

                # Update text and plots
                self.update_filepath(r_ind=r)
                self.update_line_id_box(r_ind=r)
                self.update_var_box(r_ind=r)
                self.plot_raster(r_ind=r)

    def check_fit_and_cube_dims(self, r_ind=0):
        """Ensure that EISFitResult & EISCube objects have matching coords"""
        fit_meta = self.fit_res[r_ind].meta['mod_index']
        cube_meta = self.eis_cube[r_ind].meta['mod_index']
        
        fit_naxis = [fit_meta['naxis1'], fit_meta['naxis2']]
        cube_naxis = [cube_meta['naxis1'], cube_meta['naxis2']]
    
        if fit_naxis == cube_naxis:
            # Shapes match, no need to crop
            return
        else:
            print('Cropping EISCube to match EISFitResult...')

        # If shapes do NOT match, will need to crop the cube to match the fit
        # NB: x/y coords give pixel CENTERS and FITS headers are 1 indexed
        xdelta, ydelta = fit_meta['cdelt1'], fit_meta['cdelt2']
        bot_left = [fit_meta['crval1'] - xdelta*(fit_meta['crpix1']-1), 
                         fit_meta['crval2'] - ydelta*(fit_meta['crpix2']-1)]
        top_right = [bot_left[0] + xdelta*(fit_naxis[0]-1), 
                     bot_left[1] + ydelta*(fit_naxis[1]-1)]
        
        # Make the SkyCords objects and apply the crop
        cube_frame = wcs_to_celestial_frame(self.eis_cube[r_ind].wcs)
        bl_coord = [None, SkyCoord(Tx=bot_left[0], Ty=bot_left[1], 
                                   unit=u.arcsec, frame=cube_frame)]
        tr_coord = [None, SkyCoord(Tx=top_right[0], Ty=top_right[1], 
                                   unit=u.arcsec, frame=cube_frame)]
        self.eis_cube[r_ind] = self.eis_cube[r_ind].crop(bl_coord, tr_coord)

    def update_filepath(self, r_ind=0):
        """Scan current dir and set dropdown box of .h5 files for quick selection"""
        this_filepath = None
        if self.display_mode[r_ind].lower().startswith('fit'):
            this_filepath = self.fit_filepath[r_ind]
        elif self.display_mode[r_ind].lower().startswith('obs'):
            this_filepath = self.cube_filepath[r_ind]
        
        self.box_file_list[r_ind].currentIndexChanged.disconnect(self.event_quick_select)
        if this_filepath is not None:       
            f_name = os.path.basename(this_filepath)
            dir_name = os.path.dirname(this_filepath)
            self.selected_file[r_ind] = f_name
            self.selected_dir[r_ind] = dir_name

            # Get list all .data.h5 & .fit.h5 files in current dir
            all_dir_file_list = []
            all_dir_file_list.extend(glob.glob(os.path.join(dir_name, 'eis_*.data.h5')))
            all_dir_file_list.extend(glob.glob(os.path.join(dir_name, 'eis_*.fit.h5')))
            all_dir_file_list = sorted(all_dir_file_list)
            all_dir_file_list = [os.path.basename(PATH) for PATH in all_dir_file_list]
            this_file_ind = all_dir_file_list.index(f_name)

            self.box_file_list[r_ind].clear()
            self.box_file_list[r_ind].addItems(all_dir_file_list)
            self.box_file_list[r_ind].setCurrentIndex(this_file_ind)
            self.box_file_list[r_ind].setToolTip(f"Current dir: {dir_name}")

        self.box_file_list[r_ind].currentIndexChanged.connect(self.event_quick_select)

    def update_line_id_box(self, r_ind=0):
        """Get available windows / fit line_ids and set dropdown box options"""
        self.box_line_id[r_ind].currentIndexChanged.disconnect(self.event_plot_raster)
        curr_line_ind = self.box_line_id[r_ind].currentIndex()
        
        if (self.display_mode[r_ind].lower().startswith('fit') 
        and self.fit_res[r_ind] is not None):
            line_list = [line for line in self.fit_res[r_ind].fit['line_ids']]
            main_ind = self.fit_res[r_ind].fit['main_component']
            self.box_line_id[r_ind].clear()
            self.box_line_id[r_ind].addItems(line_list)
            self.box_line_id[r_ind].setCurrentIndex(np.min([len(line_list)-1, main_ind]))
        elif (self.display_mode[r_ind].lower().startswith('obs') 
        and self.eis_cube[r_ind] is not None):
            win_ids = self.eis_cube[r_ind].meta['wininfo']['line_id']
            if self.obs_type[r_ind].lower() != 'multi_scan':
                # Normal scan or sit-and-stare obs
                line_list = [f"{i}: {line}" for i, line in enumerate(win_ids)]
            else:
                # "multi_scan" with multiple exposures per position
                line_list = [f"{i}-sum: {line}" for i, line in enumerate(win_ids)]
                for exp_num in range(self.eis_cube[r_ind].meta['index']['nexp_prp']):
                    exp_list = [f"{i}-{exp_num}: {line}" for i, line in enumerate(win_ids)]
                    line_list.extend(exp_list)
            self.box_line_id[r_ind].clear()
            self.box_line_id[r_ind].addItems(line_list)
            if curr_line_ind < len(win_ids):
                # Use last selected window number
                self.box_line_id[r_ind].setCurrentIndex(curr_line_ind)
            else:
                # Default to first data window
                self.box_line_id[r_ind].setCurrentIndex(0)
        self.box_line_id[r_ind].currentIndexChanged.connect(self.event_plot_raster)

    def update_var_box(self, r_ind=0):
        """Get available vars/sum bin widths and set dropdown box options"""
        self.box_var[r_ind].currentIndexChanged.disconnect(self.event_plot_raster)
        curr_var_ind = self.box_var[r_ind].currentIndex()
        self.box_var[r_ind].clear()

        if self.display_mode[r_ind].lower().startswith('fit'):
            var_list = ['Intensity ', 'Velocity', 'Width', 'Centroid', 'Reduced Chi-sq']
            default_var_ind = 0
        elif self.display_mode[r_ind].lower().startswith('obs'):
            var_list = ['3-bin sum', '5-bin sum', 
                        '7-bin sum', '9-bin sum', 
                        '11-bin sum', 'Total']
            default_var_ind = 2

        # If a full EISCube with the level-1 data is loaded, add CCD option
        if self.eis_cube[r_ind] is not None:
            var_list.append('CCD view')

        self.box_var[r_ind].addItems(var_list)
        if self.rast_lims[r_ind] is not None and curr_var_ind < len(var_list):
            # Use last selected var or sum width (if valid)
            self.box_var[r_ind].setCurrentIndex(curr_var_ind)
        else:
            # Use the default var or sum width
            self.box_var[r_ind].setCurrentIndex(default_var_ind)
        self.box_var[r_ind].currentIndexChanged.connect(self.event_plot_raster)

    def plot_raster(self, r_ind=0):
        """Plot the main raster image"""
        if r_ind > self.curr_layout_ind:
            # Skip if raster is currently hidden
            return 
         
        if self.fit_res[r_ind] is not None or self.eis_cube[r_ind] is not None:
            
            # Store copies of old plot limits and crosshair coords
            if self.rast_ax[r_ind] is not None:
                # Update current RAST plot limits [[x1, x2], [y1, y2]]
                last_rast_xlim = self.rast_ax[r_ind].get_xlim()
                last_rast_ylim = self.rast_ax[r_ind].get_ylim()
                self.rast_lims[r_ind] = [last_rast_xlim, last_rast_ylim]
            else:
                last_rast_xlim = [999, -999]
                last_rast_ylim = [999, -999]
                self.rast_lims[r_ind] = None

            if self.rast_crosshair[r_ind] is not None:
                # Save old crosshair coords, then remove the object from the ax
                old_cross_x = self.rast_crosshair[r_ind].get_offsets()[0,0]
                old_cross_y = self.rast_crosshair[r_ind].get_offsets()[0,1]
                self.crosshair_old_coords[r_ind] = [old_cross_x, old_cross_y]
                self.rast_crosshair[r_ind].remove()
                self.rast_crosshair[r_ind] = None

            # If there are any lines plotted on the ax, remove them
            if self.rast_axhline[r_ind] is not None:
                self.rast_axhline[r_ind].remove()
                self.rast_axhline[r_ind] = None
            if self.rast_axvline[r_ind] is not None:
                self.rast_axvline[r_ind].remove()
                self.rast_axvline[r_ind] = None

            if self.rast_ax[r_ind] is not None:
                self.rast_ax[r_ind].cla()
            self.rast_fig[r_ind].clf() # Clear the entire figure

            if self.spec_ax[r_ind] is not None:
                # Update current SPEC plot limits [[x1, x2], [y1, y2]]
                last_spec_xlim = self.spec_ax[r_ind].get_xlim()
                last_spec_ylim = self.spec_ax[r_ind].get_ylim()
                self.spec_lims[r_ind] = [last_spec_xlim, last_spec_ylim]

                # Clear the old spectrum (will stay empty if loading new data)
                self.spec_ax[r_ind] = None
                self.spec_fig[r_ind].clf()
                self.spec_fig[r_ind].canvas.draw_idle()

            # Get selected line/window index and variable/width name
            component_index = self.box_line_id[r_ind].currentIndex()
            var_str = self.box_var[r_ind].currentText()
            last_var = self.var_label[r_ind] # Used for persistence of settings

            if self.display_mode[r_ind].lower().startswith('obs'):
                # split out window index and exposure set string
                window_text = self.box_line_id[r_ind].currentText()
                component_index = int(window_text.split(':')[0].split('-')[0])
                exp_set_str = str(window_text.split(':')[0].split('-')[-1])

            # Toggle CCD display mode
            if var_str.lower().startswith('ccd'):
                # Add ' ccd' to end of display mode, if needed
                if not self.display_mode[r_ind].lower().endswith('ccd'):
                    self.display_mode[r_ind] = self.display_mode[r_ind] + ' ccd'
            elif self.display_mode[r_ind].lower().endswith('ccd'):
                # Remove ' ccd' from end of display mode, if needed
                self.display_mode[r_ind] = self.display_mode[r_ind][0:-4]
                self.rast_slice_ind[r_ind] = None
                self.rast_slice_coord[r_ind] = None

            # Get current colormap and requested nomalization
            current_cmap = self.box_cmap[r_ind].currentText()
            current_cnorm = self.box_cnorm[r_ind].currentText()
            self.box_cnorm[r_ind].setEnabled(True)
            if current_cnorm.lower().startswith('asinh'):
                inten_scale = AsinhStretch()
            elif current_cnorm.lower().startswith('linear'):
                inten_scale = LinearStretch()
            elif current_cnorm.lower().startswith('sqrt'):
                inten_scale = SqrtStretch()
            elif current_cnorm.lower().startswith('log'):
                inten_scale = LogStretch()
            else:
                inten_scale = AsinhStretch()

            # [FIT MODE] Plotting fit results
            if self.display_mode[r_ind].lower().startswith('fit'):
                self.rast_wcoords[r_ind] = None
                self.spec_xydelta[r_ind] = None
                self.spec_vline_ind[r_ind] = None
                self.spec_vline_val[r_ind] = None
                self.wave_bin_width[r_ind] = None
                self.wave_sum_range[r_ind] = None

                # Extract selected line and variable and set plot options
                rast_index = self.fit_res[r_ind].meta['mod_index']
                if var_str.lower().startswith('int'):
                    this_var = 'intensity_fit'
                    rast_data = self.fit_res[r_ind].fit['int'][:,:,component_index]
                    rast_units = self.inten_units[r_ind]
                    inten_vmin = 0.0 #np.nanmin(rast_data)
                    # inten_vmin = np.nanpercentile(rast_data[np.where(rast_data > 0)], 0.5)
                    inten_vmax = np.nanpercentile(rast_data, 99.9)
                    valid_cmap_list = self.inten_cmap_list
                    default_cmap = 'Blues_r'
                    rast_norm = ImageNormalize(vmin=inten_vmin, vmax=inten_vmax, 
                                               stretch=inten_scale)
                elif var_str.lower().startswith('vel'):
                    this_var = 'velocity'
                    rast_data = self.fit_res[r_ind].fit['vel'][:,:,component_index]
                    rast_units = 'km/s'
                    # Autoscale color range to 3*std
                    # (rounded to nearest multiple of 5 and capped at 30 km/s)
                    vel_vlim = np.min([30, 5*round(3*np.nanstd(rast_data/5))])
                    valid_cmap_list = self.vel_cmap_list
                    default_cmap = 'RdBu_r'
                    rast_norm = ImageNormalize(vmin=-vel_vlim, vmax=vel_vlim)
                    self.box_cnorm[r_ind].setDisabled(True)
                elif var_str.lower().startswith('wid'):
                    this_var = 'width'
                    rast_data = self.fit_res[r_ind].fit['params'][:,:,2+3*component_index] #compat with old files
                    # rast_data = self.fit_res[r_ind].fit['width'][:,:,component_index]
                    rast_units = 'Angstrom'
                    width_vmax = np.nanpercentile(rast_data, 99.9)
                    width_vmin = np.min(rast_data[np.where(rast_data > 0)])
                    valid_cmap_list = self.inten_cmap_list
                    default_cmap = 'viridis'
                    rast_norm = ImageNormalize(vmin=width_vmin, vmax=width_vmax,
                                               stretch=inten_scale)
                elif var_str.lower().startswith('cen'):
                    this_var = 'centroid'
                    rast_data = self.fit_res[r_ind].fit['params'][:,:,1+3*component_index]
                    rast_units = 'Angstrom'
                    cen_vmax = np.nanpercentile(rast_data, 99.5)
                    cen_vmin = np.nanpercentile(rast_data, 0.5)
                    valid_cmap_list = self.inten_cmap_list
                    default_cmap = 'cubehelix'
                    rast_norm = ImageNormalize(vmin=cen_vmin, vmax=cen_vmax,
                                               stretch=inten_scale)
                elif var_str.lower().endswith('chi-sq'):
                    this_var = 'chisq'
                    rast_data = self.fit_res[r_ind].fit['chi2']
                    rast_units = 'unitless'
                    chisq_vmax = np.nanpercentile(rast_data, 99)
                    chisq_vmin = 0.0
                    valid_cmap_list = self.inten_cmap_list
                    default_cmap = 'turbo'
                    rast_norm = ImageNormalize(vmin=chisq_vmin, vmax=chisq_vmax,
                                               stretch=inten_scale)

            # [OBS MODE] Plotting EIS level-1 data cubes
            elif self.display_mode[r_ind].lower().startswith('obs'):
                if component_index != self.eis_cube[r_ind].meta['iwin']:
                    # Load in recently selected data cube
                    self.eis_cube[r_ind] = eispac.read_cube(self.cube_filepath[r_ind], 
                                                            window=component_index,
                                                            exp_set=exp_set_str,    
                                                            apply_radcal=True)
                    self.cube_win_ind[r_ind] = component_index
                    self.inten_units[r_ind] = str(self.eis_cube[r_ind].unit)
                    self.rast_wcoords[r_ind] = None
                    # self.spec_vline_ind[r_ind] = None
                    # self.spec_vline_val[r_ind] = None
                    self.wave_bin_width[r_ind] = None
                    self.wave_sum_range[r_ind] = None
                
                # Compute wavelength coord array
                rast_index = self.eis_cube[r_ind].meta['mod_index']
                self.spec_xydelta[r_ind] = [rast_index['cdelt3'], 0.0]
                if self.rast_wcoords[r_ind] is None:
                    n_waves = int(self.eis_cube[r_ind].data.shape[2])
                    base_wave = rast_index['crval3']
                    max_wave = base_wave + rast_index['cdelt3']*(n_waves-1)
                    self.rast_wcoords[r_ind] = np.linspace(base_wave, max_wave, 
                                                           num=n_waves)

                # Reset spec vline if outside raster wavelength range
                if self.spec_vline_val[r_ind] is not None:
                    rast_min_wave = self.rast_wcoords[r_ind][0]
                    rast_max_wave = self.rast_wcoords[r_ind][-1]
                    if ((self.spec_vline_val[r_ind] < rast_min_wave) 
                    or (self.spec_vline_val[r_ind] > rast_max_wave)):
                        self.spec_vline_ind[r_ind] = None
                        self.spec_vline_val[r_ind] = None

                # By default, select bin nearest line_id wavelength (if able)
                if self.spec_vline_ind[r_ind] is None:
                    try:
                        line_id_wave = float(rast_index['line_id'].split(' ')[2])
                        self.spec_vline_val[r_ind] = line_id_wave
                    except:
                        # skip invalid line IDs (e.g. Atlas full CCD windows)
                        pass

                # Find nearest wavelenth index from selection on spec plot
                if self.spec_vline_val[r_ind] is not None:
                    iw = np.argmin(np.abs(self.rast_wcoords[r_ind] - self.spec_vline_val[r_ind]))
                    self.spec_vline_ind[r_ind] = iw
                    self.spec_vline_val[r_ind] = self.rast_wcoords[r_ind][iw]

                # Get indices of wavelength axis to sum over
                if var_str.lower().startswith(('total', 'ccd')):
                    iwave_cen = None
                    iwave_min, iwave_max = None, None
                    self.wave_bin_width[r_ind] = None
                else:
                    num_bins = int(var_str.split('-')[0])
                    self.wave_bin_width[r_ind] = num_bins
                    iwave_cen = self.spec_vline_ind[r_ind]
                    if iwave_cen is None:
                        # Default to center bin of the window
                        iwave_cen = int(self.eis_cube[r_ind].data.shape[2]/2)
                        self.spec_vline_ind[r_ind] = iwave_cen
                        self.spec_vline_val[r_ind] = self.rast_wcoords[r_ind][iwave_cen]
                    iwave_min = iwave_cen - int((num_bins-1)/2)
                    iwave_max = iwave_cen + int((num_bins-1)/2)+1 # adj. for Python indexing [min,max)
                    if iwave_min < 0:
                        iwave_min = None
                    if iwave_max >= self.eis_cube[r_ind].data.shape[2]:
                        iwave_max = None
                wave_min_val = self.rast_wcoords[r_ind][iwave_min:iwave_max][0] - rast_index['cdelt3']/2.0
                wave_max_val = self.rast_wcoords[r_ind][iwave_min:iwave_max][-1] + rast_index['cdelt3']/2.0
                self.wave_sum_range[r_ind] = [wave_min_val, wave_max_val]

                # Sum the data along the wavelength axis
                rast_data = np.sum(self.eis_cube[r_ind].data[:,:,iwave_min:iwave_max], axis=2)
                
                # Set plot options
                this_var = 'intensity_obs'
                rast_units = self.inten_units[r_ind]
                inten_vmin = 0.0 #np.nanmin(rast_data)
                # inten_vmin = np.nanpercentile(rast_data[np.where(rast_data > 0)], 0.5)
                inten_vmax = np.nanpercentile(rast_data, 99.9)
                valid_cmap_list = self.inten_cmap_list
                default_cmap = 'gist_heat' #'Blues_r'
                rast_norm = ImageNormalize(vmin=inten_vmin, vmax=inten_vmax, 
                                           stretch=inten_scale)

            # Get or estimate exposure cadence
            if self.eis_cube[r_ind] is not None:
                rast_cad = self.eis_cube[r_ind].meta['mod_index']['cadence']
            else:
                # estimate for old fit results (before summer 2024)
                total_time = np.datetime64(rast_index['date_end']) - np.datetime64(rast_index['date_obs'])
                total_time = total_time / np.timedelta64(1, 's') # ensure [s] units
                rast_cad = total_time / rast_index['naxis1']

            # [CCD MODE] Plotting a single exposure of EIS level-1 intensities
            # NB: this will override some params from FIT and OBS mode above
            if self.display_mode[r_ind].lower().endswith('ccd'):
                # Select correct exposure index to load (i.e. X-axis index)
                if self.crosshair_ind[r_ind] is None:
                    ccd_exp_ind = 0
                else:
                    ccd_exp_ind = self.crosshair_ind[r_ind][0]
                    if ccd_exp_ind >= rast_index['naxis1']:
                        ccd_exp_ind = rast_index['naxis1'] - 1
                
                # Load the CCD exposure and set plot options
                this_var = 'intensity_ccd'
                rast_data = self.eis_cube[r_ind].data[:,ccd_exp_ind,:]
                rast_units = self.inten_units[r_ind]
                inten_vmin = 0.0 #np.nanmin(rast_data)
                # inten_vmin = np.nanpercentile(rast_data[np.where(rast_data > 0)], 0.5)
                inten_vmax = np.nanpercentile(rast_data, 99.9)
                valid_cmap_list = self.inten_cmap_list
                default_cmap = 'gist_heat' #'Blues_r'
                rast_norm = ImageNormalize(vmin=inten_vmin, vmax=inten_vmax, 
                                           stretch=inten_scale)
                
                # Save true X-axis index and coord of this slice
                self.rast_slice_ind[r_ind] = ccd_exp_ind
                if rast_index['nraster'] == 1:
                    # Sit-and-stare raster (space-time)
                    self.rast_slice_coord[r_ind] = 0.0 + rast_cad*(ccd_exp_ind-1)
                else:
                    # Scan raster (space-space)
                    self.rast_slice_coord[r_ind] = rast_index['crval1'] + rast_index['cdelt1']*(ccd_exp_ind)

                # Set base coordinate variables for space-wavelength img (CCD)
                self.rast_xaxis_type[r_ind] = 'Wavelength'
                slice_label = f"slice at X = {self.rast_slice_coord[r_ind]:.2f}″ [{ccd_exp_ind}]"
                xlabel_text = f"Wavelength [$\AA$] ({slice_label})"
                ylabel_text = "Solar-Y [arcsec]"
                n_xaxis, n_yaxis = rast_index['naxis3'], rast_index['naxis2']
                xdelta, ydelta = rast_index['cdelt3'], rast_index['cdelt2']
                rast_bot_left = [rast_index['crval3'], 
                                 rast_index['crval2'] - ydelta*(rast_index['crpix2']-1)]
                rast_aspect = 'auto'
            elif rast_index['nraster'] == 1:
                # Base coordinate variables for space-time img (sit-and-stare)
                self.rast_xaxis_type[r_ind] = 'Time'
                xlabel_text = "Time elapsed [s]"
                ylabel_text = "Solar-Y [arcsec]"
                n_xaxis, n_yaxis = rast_index['naxis1'], rast_index['naxis2']
                xdelta, ydelta = rast_cad, rast_index['cdelt2']
                rast_bot_left = [0.0 - xdelta*(rast_index['crpix1']-1), 
                                 rast_index['crval2'] - ydelta*(rast_index['crpix2']-1)]
                rast_aspect = 'auto'
            else:
                # Base coordinate variables for space-space img (scan)
                self.rast_xaxis_type[r_ind] = 'Solar-X'
                xlabel_text = "Solar-X [arcsec]"
                ylabel_text = "Solar-Y [arcsec]"
                n_xaxis, n_yaxis = rast_index['naxis1'], rast_index['naxis2']
                xdelta, ydelta = rast_index['cdelt1'], rast_index['cdelt2']
                rast_bot_left = [rast_index['crval1'] - xdelta*(rast_index['crpix1']-1), 
                                 rast_index['crval2'] - ydelta*(rast_index['crpix2']-1)]
                if self.rast_auto_aspect[r_ind] is True and not last_var.endswith('ccd'):
                    # Only used after user has changed the limits via dialog box
                    rast_aspect = 'auto'
                else:
                    # Default, force pixels to be square
                    rast_aspect = 'equal'
                    self.rast_auto_aspect[r_ind] = False

            # Update aspect flag (checked when adjusting limits via dialog box)
            if rast_aspect == 'auto':
                self.rast_auto_aspect[r_ind] = True
                
            # Compute coordinate arrays
            # NB: x/y coords give pixel CENTERS while extend gives min/max edges
            rast_top_right = [rast_bot_left[0] + xdelta*(n_xaxis-1), 
                              rast_bot_left[1] + ydelta*(n_yaxis-1)]
            subplot_extent = [rast_bot_left[0]-xdelta/2, rast_top_right[0]+xdelta/2, 
                              rast_bot_left[1]-ydelta/2, rast_top_right[1]+ydelta/2]
            self.rast_xcoords[r_ind] = np.linspace(rast_bot_left[0], rast_top_right[0], 
                                                   num=n_xaxis)
            self.rast_ycoords[r_ind] = np.linspace(rast_bot_left[1], rast_top_right[1], 
                                                   num=n_yaxis)
            self.rast_xydelta[r_ind] = [xdelta, ydelta]

            # Select cmap and and Update cmap box
            if this_var == last_var and current_cmap in valid_cmap_list:
                rast_cmap = current_cmap
            else:
                rast_cmap = default_cmap
            self.var_label[r_ind] = this_var
            self.box_cmap[r_ind].currentIndexChanged.disconnect(self.event_plot_raster)
            self.box_cmap[r_ind].clear()
            self.box_cmap[r_ind].addItems(valid_cmap_list)
            self.box_cmap[r_ind].setCurrentIndex(valid_cmap_list.index(rast_cmap))
            self.box_cmap[r_ind].currentIndexChanged.connect(self.event_plot_raster)

            # Plot the raster
            self.file_date_obs[r_ind] = rast_index['date_obs']
            self.rast_ax[r_ind] = self.rast_fig[r_ind].subplots()
            self.rast_img[r_ind] = self.rast_ax[r_ind].imshow(rast_data, extent=subplot_extent, 
                                origin='lower', cmap=rast_cmap, norm=rast_norm, 
                                aspect=rast_aspect, interpolation='nearest')
            self.rast_ax[r_ind].set_xlabel(xlabel_text, fontsize=10)
            self.rast_ax[r_ind].set_ylabel(ylabel_text, fontsize=10)

            rast_cbar = self.rast_fig[r_ind].colorbar(self.rast_img[r_ind], ax=self.rast_ax[r_ind], pad=0.01)
            rast_cbar.set_label(f"[{rast_units}]", fontsize=10)

            # Restore last selected plot limts (if valid)
            # NB: indexing in this way should ensure a subview in the current limits
            curr_xlim = self.rast_ax[r_ind].get_xlim()
            curr_ylim = self.rast_ax[r_ind].get_ylim() # not needed?
            if ((last_rast_ylim[0] - curr_ylim[0]) - (last_rast_ylim[1] - curr_ylim[1]) <= 1):
                # Check for switching between short- and longwave CCDs (yields equal Y-offsets)
                # If switching, do NOT keep old limits
                pass
            elif (last_rast_xlim[0] <= curr_xlim[1] and last_rast_xlim[1] >= curr_xlim[0]
            and last_rast_ylim[0] <= curr_ylim[1] and last_rast_ylim[1] >= curr_ylim[0]):
                self.rast_ax[r_ind].set_ylim(last_rast_ylim)

                if (not this_var.lower().endswith('ccd')) or (this_var == last_var):
                    self.rast_ax[r_ind].set_xlim(last_rast_xlim)

            self.rast_fig[r_ind].subplots_adjust(left=0.12, right=0.90, bottom=0.10, top=0.98)
            self.rast_img[r_ind].figure.canvas.draw_idle() # Update the plot!
            self.rast_ax[r_ind].autoscale(False) # Fixes issue with zooming

            # Check if dialog box is open and update as needed
            if self.lim_dialog is not None and self.lim_dialog.isVisible():
                if self.dialog_r_ind == r_ind:
                    self.lim_dialog.read_rast_limits(self) # update img limits

            self.rast_click[r_ind] = self.rast_img[r_ind].figure.canvas.mpl_connect('button_press_event', self.event_rast_click)
            # self.rast_keyboard[r_ind] = self.rast_img[r_ind].figure.canvas.mpl_connect('key_press_event', self.event_keyboard_nav)
            self.plot_crosshair(r_ind=r_ind) # this will also call plot_spectrum()

    def plot_crosshair(self, r_ind=0):
        """Compute pixel coords and plot crosshairs on a raster image"""
        # By default, skip plotting if either no coords or raster is not defined
        crosshair_in_axes = False

        if self.crosshair_new_coords[r_ind] is not None:
            # Load last selected crosshair coords (if any)
            new_x_val = self.crosshair_new_coords[r_ind][0]
            new_y_val = self.crosshair_new_coords[r_ind][1]
            if self.rast_xcoords[r_ind] is not None:
                # Check if coords are inside axes; if not, skip plotting 
                rast_xmin, rast_xmax = self.rast_xcoords[r_ind][[0,-1]]
                rast_ymin, rast_ymax = self.rast_ycoords[r_ind][[0,-1]]

                if self.crosshair_old_coords[r_ind] is not None:
                    old_x_val, old_y_val = self.crosshair_old_coords[r_ind]
                else:
                    old_x_val, old_y_val = rast_xmin-999, rast_ymin-999

                # If CCD mode is ON, ignore new_x_val for now (corrected before plotting)
                if self.display_mode[r_ind].lower().endswith('ccd'):
                    new_x_val = (rast_xmin + rast_xmax)/2.0 # always valid

                if ((new_x_val >= rast_xmin and new_x_val <= rast_xmax)
                and (new_y_val >= rast_ymin and new_y_val <= rast_ymax)):
                    # Compute the nearest pixel coords and indices
                    crosshair_in_axes = True
                    ix = np.argmin(np.abs(self.rast_xcoords[r_ind] - new_x_val))
                    iy = np.argmin(np.abs(self.rast_ycoords[r_ind] - new_y_val))
                    x_val = self.rast_xcoords[r_ind][ix]
                    y_val = self.rast_ycoords[r_ind][iy]
                    self.crosshair_new_coords[r_ind] = [x_val, y_val]
                    self.crosshair_old_coords[r_ind] = [x_val, y_val]
                    self.crosshair_ind[r_ind] = [ix, iy]
                elif ((old_x_val >= rast_xmin and old_x_val <= rast_xmax)
                and (old_y_val >= rast_ymin and old_y_val <= rast_ymax)):
                    # Use old crosshair coords
                    # NB: Should only be needed in the edge case of comparing
                    # the SAME wave window between dates with DIFFERENT FoV
                    crosshair_in_axes = True
                    x_val, y_val = old_x_val, old_y_val
                else:
                    # Skip plotting if crosshair is not in axes
                    crosshair_in_axes = False
                    self.crosshair_ind[r_ind] = None

        # Remove old crosshair
        if self.rast_crosshair[r_ind] is not None:
            self.rast_crosshair[r_ind].remove()
            self.rast_crosshair[r_ind] = None
            if self.rast_ax[r_ind].get_legend() is not None:
                self.rast_ax[r_ind].get_legend().remove()   

        # Remove horizontal and vertical lines
        if self.rast_axhline[r_ind] is not None:
            self.rast_axhline[r_ind].remove()
            self.rast_axhline[r_ind] = None
        if self.rast_axvline[r_ind] is not None:
            self.rast_axvline[r_ind].remove()
            self.rast_axvline[r_ind] = None

        # If CCD mode is on, place the crosshair in the correct location   
        if crosshair_in_axes and self.display_mode[r_ind].lower().endswith('ccd'):
            x_val = self.spec_vline_val[r_ind]
            # Ensure the x-coord slice index remains fixed
            if self.rast_slice_ind[r_ind] is not None:
                self.crosshair_ind[r_ind][0] = self.rast_slice_ind[r_ind]

        # Lookup the crosshair color
        current_cmap = self.box_cmap[r_ind].currentText()
        if current_cmap in self.vel_cmap_list:
            c_color = self.vel_crosshair_color
        elif current_cmap in self.alt_crosshair_cmaps:
            c_color = self.alt_crosshair_color
        else:
            c_color = self.default_crosshair_color

        # Plot new crosshair (if raster is currently visible)
        if crosshair_in_axes and r_ind <= self.curr_layout_ind and x_val is not None:
            # Extract the image value at the corsshair coordinates
            c_ix = np.argmin(np.abs(self.rast_xcoords[r_ind] - x_val))
            c_iy = self.crosshair_ind[r_ind][1]
            self.crosshair_val[r_ind] = self.rast_img[r_ind].get_array()[c_iy, c_ix]
            cross_label = str(round(self.crosshair_val[r_ind], 2))

            # Plot crosshair and display legend
            self.rast_crosshair[r_ind] = self.rast_ax[r_ind].scatter(x_val, y_val, s=30, 
                                                         c=c_color, marker='x',
                                                         label=cross_label)

            self.rast_ax[r_ind].legend(loc='lower right', frameon=False, handlelength=1.0, 
                                       bbox_to_anchor=(1.2, -0.11), borderaxespad=0.0, 
                                       alignment='right', handletextpad=0.6)

            # Plot horizontal and vertical lines 
            if self.display_mode[r_ind].lower().endswith('ccd'):
                # If CCD mode is ON, add a horizontal line
                self.rast_axhline[r_ind] = self.rast_ax[r_ind].axhline(y_val, color=c_color, ls='--')
            else:
                # Check if OTHER rasters of the same timestamp have CCD mode on
                for other_r in range(self.max_num_rasters):
                    if other_r == r_ind:
                        pass # ignore THIS raster
                    elif (str(self.display_mode[other_r]).lower().endswith('ccd')
                    and str(self.file_date_obs[other_r]) == str(self.file_date_obs[r_ind])):
                        # Add vertical line to show CCD mode of OTHER rast
                        self.rast_axvline[r_ind] = self.rast_ax[r_ind].axvline(x_val, color=c_color, ls='--')

            self.rast_img[r_ind].figure.canvas.draw_idle() # Update the plot!

            self.plot_spectrum(r_ind=r_ind)

        # Check if dialog box is open and update as needed
        if self.lim_dialog is not None and self.lim_dialog.isVisible():
            if self.dialog_r_ind == r_ind:
                self.lim_dialog.read_rast_crosshair(self) # Update crosshair coords
        else:
            self.plot_spectrum(r_ind=r_ind) # just update or show empty

    def plot_spectrum(self, r_ind=0):
        """Plot the spectrum at the selected crosshair location"""
        if r_ind > self.curr_layout_ind:
            # Skip if raster is currently hidden
            return 

        if ((self.fit_res[r_ind] is not None or self.eis_cube[r_ind] is not None)
        and (self.crosshair_ind[r_ind] is not None)):    
            if self.spec_ax[r_ind] is not None:
                # Get current plot limits
                last_spec_xlim = self.spec_ax[r_ind].get_xlim()
                last_spec_ylim = self.spec_ax[r_ind].get_ylim()
            elif self.spec_lims[r_ind] is not None:
                # Get last known plot limits [[x1, x2], [y1, y2]]
                last_spec_xlim = self.spec_lims[r_ind][0]
                last_spec_ylim = self.spec_lims[r_ind][1]
            else:
                # Set dummy values that will NOT get set
                last_spec_xlim = [999, -999]
                last_spec_ylim = [999, -999]


            if self.spec_click[r_ind] is not None:
                # Disconnect spec click event
                self.spec_fig[r_ind].canvas.mpl_disconnect(self.spec_click[r_ind])
                self.spec_click[r_ind] = None
            
            self.spec_fig[r_ind].clf()
            self.spec_ax[r_ind] = self.spec_fig[r_ind].subplots()

            color_list = ['tab:blue', 'tab:red', 'tab:purple', 
                          'tab:green', 'tab:orange', 'tab:brown',
                          'tab:blue', 'tab:red', 'tab:purple', 
                          'tab:green', 'tab:orange', 'tab:brown']
            ls_list = ['-', '-', '-', '-', '-', '-', 
                       '--', '--', '--', '--', '--', '--']

            # Load Y-axis data array index and coordinate value
            y_ind = self.crosshair_ind[r_ind][1]
            y_val = self.rast_ycoords[r_ind][y_ind]

            # Load X-axis data array index and coordinate value
            if self.display_mode[r_ind].lower().endswith('ccd'):
                # IF CCD mode is ON, use the saved slice X-axis index and coord
                x_ind = self.rast_slice_ind[r_ind]
                x_val = self.rast_slice_coord[r_ind]
            else:
                # Otherwise, use load values as expected
                x_ind = self.crosshair_ind[r_ind][0]
                x_val = self.rast_xcoords[r_ind][x_ind]

            # Plot the actual data values (if loaded)
            if self.eis_cube[r_ind] is not None:
                rast_index = self.eis_cube[r_ind].meta['mod_index']
                date_obs = self.eis_cube[r_ind].meta['date_obs'][x_ind]
                data_x = self.eis_cube[r_ind].wavelength[y_ind, x_ind, :]
                data_y = self.eis_cube[r_ind].data[y_ind, x_ind, :]
                data_err = self.eis_cube[r_ind].uncertainty.array[y_ind, x_ind, :]
                data_mask = self.eis_cube[r_ind].mask[y_ind, x_ind, :]
                if not np.all(data_mask):
                    data_y[data_mask] = np.nan
                    self.spec_ax[r_ind].errorbar(data_x, data_y, yerr=np.abs(data_err), ls='', marker='o', color='k')
                if np.any(data_mask):
                    data_bad_y = np.zeros_like(data_y)
                    data_bad_x = data_x.copy()
                    data_bad_x[~data_mask] = np.nan
                    self.spec_ax[r_ind].plot(data_bad_x, data_bad_y, ls='', marker='o', 
                                                markerfacecolor='none', markeredgecolor='grey')
            
            # [FIT MODE] Plot fit profiles
            if self.fit_res[r_ind] is not None:
                rast_index = self.fit_res[r_ind].meta['mod_index']
                date_obs = self.fit_res[r_ind].meta['date_obs'][x_ind]
                red_chi2 = self.fit_res[r_ind].fit['chi2'][y_ind, x_ind]
                stat_text = "${\chi^2}_{red}$ = "+ f"{red_chi2:.2f}"
            
                fit_x, fit_y = self.fit_res[r_ind].get_fit_profile(coords=[y_ind,x_ind], 
                                                                   num_wavelengths=100)
                bkg_x, bkg_y = self.fit_res[r_ind].get_fit_profile(-1, coords=[y_ind,x_ind], 
                                                                   num_wavelengths=100)
                # self.spec_ax[r_ind].plot(fit_x, fit_y, color='grey', label='Fit Profile')
                self.spec_ax[r_ind].plot(fit_x, fit_y, color='grey', label=f"Fit ({stat_text})")
                for g in range(self.fit_res[r_ind].n_gauss):
                    comp_x, comp_y = self.fit_res[r_ind].get_fit_profile(g, coords=[y_ind,x_ind], 
                                                                         num_wavelengths=100)
                    self.spec_ax[r_ind].plot(comp_x, comp_y, color=color_list[g], ls=ls_list[g],
                                    label=self.fit_res[r_ind].fit['line_ids'][g])

                self.spec_ax[r_ind].plot(bkg_x, bkg_y, color='grey', ls=':', label='Background')
                # self.spec_ax.set_title(f'Cutout indices: iy = {iy}, ix = {ix}')
                if self.radio_show_details.isChecked():
                    self.spec_ax[r_ind].legend(loc='upper left', frameon=False, handlelength=1.2, handletextpad=0.5)
                    # self.spec_ax[r_ind].legend(loc='lower left', frameon=False, handlelength=1.0, ncol=3, 
                    #                            bbox_to_anchor=(0.0, 1.01, 1.0, .12), mode="expand", borderaxespad=0.0)

            
            # [OBS MODE] Plot vertical lines for current sum range
            # NB: the bin locations are based on the NOMINAL wavelength coords
            #     while the data is plotted with the CORRECTED wavelengths.
            #     Therefore, there can be small offsets between the two
            if self.spec_vline_val[r_ind] is not None and self.wave_bin_width[r_ind] is not None:
                # First, shade regions outside the current sum
                base_spec_xlim = self.spec_ax[r_ind].get_xlim()
                self.spec_ax[r_ind].axvspan(base_spec_xlim[0]-0.1, self.wave_sum_range[r_ind][0], 
                                     color='lightgrey', zorder=1)
                self.spec_ax[r_ind].axvspan(self.wave_sum_range[r_ind][1], base_spec_xlim[1]+0.1,
                                     color='lightgrey', zorder=1)

                # Mark the selected middle of the wave range
                self.spec_ax[r_ind].axvline(self.spec_vline_val[r_ind], color='grey', ls='--', zorder=1)
                self.spec_ax[r_ind].set_xlim(base_spec_xlim) # Keep original limits

                # Display information about selected wavelength range
                if self.radio_show_details.isChecked():
                    vline_ind = self.spec_vline_ind[r_ind]
                    sum_half_width = int((self.wave_bin_width[r_ind]-1)/2)
                    sum_ind_start = vline_ind - sum_half_width
                    sum_ind_end = vline_ind + sum_half_width
                    info_text = [f"{self.wave_bin_width[r_ind]} bin range:"
                                +f"\nmin: {self.wave_sum_range[r_ind][0]:.3f} [{sum_ind_start}]"
                                +f"\ncenter: {self.spec_vline_val[r_ind]:.3f} [{vline_ind}]"
                                +f"\nmax: {self.wave_sum_range[r_ind][1]:.3f} [{sum_ind_end}]"]
                    self.spec_ax[r_ind].text(0.02, 0.98, info_text[0], 
                                            horizontalalignment='left',
                                            verticalalignment='top',
                                            transform=self.spec_ax[r_ind].transAxes)
            elif self.spec_vline_val[r_ind] is not None and self.fit_res[r_ind] is None:
                # [CCD mode] Plotting reference line matching crosshair coords
                base_spec_xlim = self.spec_ax[r_ind].get_xlim()
                self.spec_ax[r_ind].axvline(self.spec_vline_val[r_ind], color='grey', ls=':', zorder=1)
                self.spec_ax[r_ind].set_xlim(base_spec_xlim) # Keep original limits
                
                # Display information about selected wavelength range
                if self.radio_show_details.isChecked():
                    vline_ind = self.spec_vline_ind[r_ind]
                    if vline_ind is None:
                        vline_ind = 'N/A'
                    info_text = [f"Selected wavelength:"
                                +f"\nvalue: {self.spec_vline_val[r_ind]:.3f} [{vline_ind}]"]
                    self.spec_ax[r_ind].text(0.02, 0.98, info_text[0], 
                                            horizontalalignment='left',
                                            verticalalignment='top',
                                            transform=self.spec_ax[r_ind].transAxes)

            # Display location info
            # Note: we use the double prime symbols below (LaTeX looked odd)
            if self.radio_show_details.isChecked():
                if rast_index['nraster'] == 1:
                    # Sit-and-stare raster 
                    x_val_str = f'{x_val:.2f} s'
                else:
                    # Scan raster
                    x_val_str = f'{x_val:.2f}″'
                y_val_str = f'{y_val:.2f}″'
                loc_text = [f"{date_obs.replace('T', ' ')}"
                        +f'\n{x_val_str} [{x_ind}] X'
                        +f'\n{y_val_str} [{y_ind}] Y']

                self.spec_ax[r_ind].text(0.98, 0.98, loc_text[0], 
                                        horizontalalignment='right',
                                        verticalalignment='top',
                                        transform=self.spec_ax[r_ind].transAxes)
            
            curr_xlim = self.spec_ax[r_ind].get_xlim()
            curr_ylim = self.spec_ax[r_ind].get_ylim() # not needed?
            
            # Annotate all CHIANTI lines in current window
            if self.radio_show_ids.isChecked():
                mixed_trans = self.spec_ax[r_ind].get_xaxis_transform()  # x in data, y in axes
                loc_ids = np.where((self.all_line_ids['wave'] >= curr_xlim[0])
                                   & (self.all_line_ids['wave'] <= curr_xlim[1]))
                nearby_ids = self.all_line_ids[loc_ids]

                if len(nearby_ids) > 0:
                    id_wave_diff = np.diff(nearby_ids['wave'], prepend=1)
                    for i in range(len(nearby_ids)):
                        if id_wave_diff[i] <= 0.065:
                            last_id_yscale = last_id_yscale + 0.08
                            if last_id_yscale > 0.8:
                                last_id_yscale = 0.40 # reset if gets too high
                        else:
                            last_id_yscale = 0.40
                        id_x = nearby_ids['wave'][i] + 0.01
                        # id_y = last_id_yscale*curr_ylim[1]
                        # marker_y = (last_id_yscale+0.04)*curr_ylim[1]
                        id_y = last_id_yscale
                        marker_y = last_id_yscale+0.04

                        self.spec_ax[r_ind].plot([nearby_ids['wave'][i]], [marker_y], 'r+', 
                                                 zorder=3, transform=mixed_trans)
                        self.spec_ax[r_ind].annotate(nearby_ids['id'][i], xy=(id_x, id_y), color='navy', 
                                                     zorder=3, xycoords=mixed_trans,
                                                     annotation_clip=True)

            # Restore last selected plot limts (if valid)
            # NB: indexing in this way should ensure a subview in the current limits
            if last_spec_xlim[0] <= curr_xlim[1] and last_spec_xlim[1] >= curr_xlim[0]:
                self.spec_ax[r_ind].set_xlim(last_spec_xlim)
            if not self.radio_autoscale_ylim.isChecked() and last_spec_ylim[1] > 1:
                self.spec_ax[r_ind].set_ylim(last_spec_ylim)

            # Label axes and update the figure
            self.spec_ax[r_ind].set_xlabel('Wavelength [$\AA$]')
            self.spec_ax[r_ind].set_ylabel(f"Intensity\n[{self.inten_units[r_ind]}]")
            # self.spec_ax[r_ind].xaxis.set_major_formatter(FormatStrFormatter('%.2f'))
            self.spec_ax[r_ind].figure.canvas.draw_idle() # Update the plot!

            self.spec_click[r_ind] = self.spec_fig[r_ind].canvas.mpl_connect('button_press_event', self.event_spec_click)
            # self.spec_keyboard[r_ind] = self.spec_fig[r_ind].canvas.mpl_connect('key_press_event', self.event_keyboard_nav)

            # Check if dialog box is open and update as needed
            if self.lim_dialog is not None and self.lim_dialog.isVisible():
                if self.dialog_r_ind == r_ind:
                    self.lim_dialog.read_spec_limits(self) # Update spec limits
        else:
            # Plot an empty ax with a tip about selecting a point
            if self.spec_ax[r_ind] is None:
                self.spec_fig[r_ind].clf()
                spec_select_text = 'Click the image\nto display the spectrum'
                empty_ax = self.spec_fig[r_ind].subplots()
                empty_ax.axis('off')
                empty_ax.text(0.5, 0.5, spec_select_text, 
                            horizontalalignment='center',
                            verticalalignment='center',
                            transform=empty_ax.transAxes)
                self.spec_fig[r_ind].canvas.draw_idle()

    def update_canvas_focus(self, r_ind=None):
        """Update the focus of the limits dialog and active plot canvas"""
        # Check if dialog box is open and update as needed
        if self.lim_dialog is not None and self.lim_dialog.isVisible():
            if r_ind is None or r_ind > self.curr_layout_ind:
                # Skip if invalid or hidden index
                pass
            elif self.dialog_r_ind != r_ind:
                self.button_rast_lims[r_ind].click() # Reopen with new focus

        # Check if a figure is focused and update the frame as needed
        focused_name = self.focusWidget().objectName()
        if len(focused_name) <= 0: 
            focused_name = getattr(self.focusWidget(), 'AltObjectName', 'none')

        if focused_name.lower() != 'none' and focused_name.count('_') == 2:
            rv, plot_type, obj_class = focused_name.split('_')
        else:
            rv, plot_type, obj_class = 'none', 'none', 'none'

        if (self.focused_fig is not None 
        and self.focused_fig.canvas != self.focusWidget()):
            # Turn OFF border for a figure that has lost the keyboard focus
            self.focused_fig.patch.set_linewidth(0)
            self.focused_fig.patch.set_edgecolor(None)
            self.focused_fig.canvas.draw_idle() # Update the plot!
            if plot_type == 'rast':
                self.focused_fig.canvas.mpl_disconnect(self.rast_keyboard[self.focused_r_ind])
                self.rast_keyboard[self.focused_r_ind] = None
            elif plot_type == 'spec':
                self.focused_fig.canvas.mpl_disconnect(self.spec_keyboard[self.focused_r_ind])
                self.spec_keyboard[self.focused_r_ind] = None
            self.focused_r_ind = None
            self.focused_type = None
            self.focused_fig = None
        
        if obj_class.lower().startswith('canvas'):
            # Turn ON border for a figure that has gained the keyboard focus
            self.focused_r_ind = int(rv[1])
            self.focused_type = plot_type
            if plot_type == 'rast':
                self.focused_fig = self.rast_fig[self.focused_r_ind]
                self.rast_keyboard[self.focused_r_ind] = self.focused_fig.canvas.mpl_connect('key_press_event', self.event_keyboard_nav)
            elif plot_type == 'spec':
                self.focused_fig = self.spec_fig[self.focused_r_ind]
                self.spec_keyboard[self.focused_r_ind] = self.focused_fig.canvas.mpl_connect('key_press_event', self.event_keyboard_nav)
            self.focused_fig.patch.set_linewidth(5)
            self.focused_fig.patch.set_edgecolor('black')
            self.focused_fig.canvas.draw_idle() # Update the plot!

    def event_change_layout(self):
        """Event for chaning the number of displayed image panels"""
        layout_index = self.box_layout.currentIndex()
        self.arange_layout(layout_index=layout_index)

    def event_select_file(self):
        """Open file dialog for loading an image into a single selected panel"""
        r_ind = int(self.sender().objectName()[1]) # Get r_ind of selected panel
        # QtWidgets.QApplication.processEvents() ## probably not needed....
        options = QtWidgets.QFileDialog.Options()
        # options |= QtWidgets.QFileDialog.DontUseNativeDialog ## can cause long delays

        selected_file, _ = QtWidgets.QFileDialog.getOpenFileName(self, 
                                f"Select a file to open in Panel #{r_ind+1}",
                                # filter='*.data.h5', options=options)
                                filter='eis_*.h5', options=options)

        self.load_file(selected_file, r_ind=r_ind)

    def event_open_all(self):
        """Open file dialog for loading a single image into all panels"""
        # QtWidgets.QApplication.processEvents() ## probably not needed....
        options = QtWidgets.QFileDialog.Options()
        # options |= QtWidgets.QFileDialog.DontUseNativeDialog ## can cause long delays
        selected_file, _ = QtWidgets.QFileDialog.getOpenFileName(self, 
                                'Select a file to open in ALL panels',
                                filter='eis_*.h5', options=options)

        self.load_file(selected_file, r_ind=0, load_in_all=True)

    def event_quick_select(self):
        """Event for using dropdown list to quickly switch files in the same dir"""
        r_ind = int(self.sender().objectName()[1]) # Get r_ind of selected panel

        curr_dir = self.selected_dir[r_ind]
        new_filename = self.box_file_list[r_ind].currentText()
        selected_file = os.path.join(curr_dir, new_filename)

        self.load_file(selected_file, r_ind=r_ind)
        self.update_canvas_focus(r_ind=r_ind)

    def event_plot_raster(self, index):
        """Event for plotting the main image of a selected panel"""
        r_ind = int(self.sender().objectName()[1]) # Get r_ind of selected panel
        self.plot_raster(r_ind=r_ind)
        self.update_canvas_focus(r_ind=r_ind)
    
    def event_replot_all_spec(self):
        """Event for replotting all of the spectra"""
        self.update_canvas_focus(r_ind=None)
        for r in range(self.max_num_rasters):
            self.plot_spectrum(r_ind=r)

    def event_reset_plot(self):
        """Clear and replot a selected raster or spectrum"""
        r_ind = int(self.sender().objectName()[1])
        plot_type = self.sender().objectName().split('_')[1]

        if plot_type.lower().startswith('rast'):
            self.rast_ax[r_ind] = None
            self.rast_fig[r_ind].clf()
            self.rast_lims[r_ind] = None
            self.rast_auto_aspect[r_ind] = None
            self.plot_raster(r_ind=r_ind)
        elif plot_type.lower().startswith('spec'):
            self.spec_ax[r_ind] = None
            self.spec_fig[r_ind].clf()
            self.spec_lims[r_ind] = None
            self.plot_spectrum(r_ind=r_ind)

        self.update_canvas_focus(r_ind=r_ind)

    def event_rast_click(self, event):
        """Event for clicks on a raster image"""
        # First, figure out which raster canvas was clicked on
        for r in range(self.max_num_rasters):
            if self.rast_canvas[r] == event.canvas:
                r_ind = r
                break

        self.update_canvas_focus(r_ind=r_ind)
        if event.button is MouseButton.MIDDLE:
            # Toggle pan/zoom on middle click
            self.rast_nav[r_ind].pan()
        elif self.rast_fig[r_ind].canvas.widgetlock.locked():
            # If in pan/zoom mode, ignore the mouse clicks 
            return
        elif event.xdata is None or event.ydata is None:
            # Ignore clicks outside the data axes
            return
        else:
            # Get data coords and call plotting functions
            xdata, ydata = event.xdata, event.ydata # X, Y in data coords [arcsec]

            if self.radio_sync_cross.isChecked():
                # If CCD mode of CLICKED rast is ON, 
                # AND sync vline is checked, fake a click on the spectrum
                if (str(self.display_mode[r_ind]).lower().endswith('ccd')
                and self.radio_sync_vline.isChecked()
                and self.spec_ax[r_ind] is not None):
                    class FakeClickEvent():
                        pass

                    # Setup the fake event
                    fake_click = FakeClickEvent()
                    fake_click.canvas = self.spec_canvas[r_ind]
                    fake_click.button = None
                    fake_click.xdata = copy.copy(xdata)
                    fake_click.ydata = copy.copy(ydata)
                    self.event_spec_click(fake_click)

                # Update ALL crosshairs and spectra
                for r in range(self.max_num_rasters):
                    copy_xdata = xdata
                    # If CCD mode of CLICKED rast is ON, use THIS rast OLD x_coord
                    if str(self.display_mode[r_ind]).lower().endswith('ccd'):
                        if self.crosshair_old_coords[r] is not None:
                            copy_xdata = self.crosshair_old_coords[r][0]

                    self.crosshair_new_coords[r] = [copy_xdata, ydata]

                    # If CCD mode of THIS rast is ON, update rast image too
                    if str(self.display_mode[r]).lower().endswith('ccd'):
                        # Check timestamp and CCD mode of CLICKED rast
                        if str(self.display_mode[r_ind]).lower().endswith('ccd'):
                            # Only need to update spec_vline (not raster img)
                            self.spec_vline_val[r] = xdata
                        elif (self.file_date_obs[r] is not None 
                        and self.file_date_obs[r] == str(self.file_date_obs[r_ind])):
                            # If CLICKED rast is NOT in CCD mode,
                            # get index of new CCD exposure (i.e. x-axis step)
                            ix = np.argmin(np.abs(self.rast_xcoords[r_ind] - xdata))
                            iy = np.argmin(np.abs(self.rast_ycoords[r_ind] - ydata))
                            self.crosshair_ind[r] = [ix, iy]
                            self.plot_raster(r_ind=r)
                    
                    self.plot_crosshair(r_ind=r)
            else:
                # Only update the crosshair and spectrum for the canvas clicked
                self.crosshair_new_coords[r_ind] = [xdata, ydata]
                # If CCD mode is ON, make sure to still plot the spec vline
                if str(self.display_mode[r_ind]).lower().endswith('ccd'):
                    self.spec_vline_val[r_ind] = xdata
                self.plot_crosshair(r_ind=r_ind)

    def event_spec_click(self, event):
        """Event for clicks on a spectrum"""
        # First, figure out which spec canvas was clicked on
        for r in range(self.max_num_rasters):
            if self.spec_canvas[r] == event.canvas:
                r_ind = r
                break
        
        self.update_canvas_focus(r_ind=r_ind)
        if event.button is MouseButton.MIDDLE:
            # Toggle pan/zoom on middle click
            self.spec_nav[r_ind].pan()
        elif self.spec_fig[r_ind].canvas.widgetlock.locked():
            # If in pan/zoom mode, ignore the mouse clicks 
            return
        elif event.xdata is None or event.ydata is None:
            # Ignore clicks outside the data axes
            return
        else:
            # Get data coords and call plotting functions
            xdata = event.xdata # X in data coords [Angstrom]

            if self.radio_sync_vline.isChecked():
                # Update ALL spectra and rasters (if covering same wavelength)
                for r in range(self.max_num_rasters):
                    if self.rast_wcoords[r] is None:
                        continue
                    elif (xdata > self.rast_wcoords[r][0] 
                    and xdata < self.rast_wcoords[r][-1]): 
                        self.spec_vline_val[r] = xdata
                        if self.display_mode[r].lower().endswith('ccd'):
                            # If CCD mode is ON, only update crosshair and spec
                            self.plot_crosshair(r_ind=r)
                        else:
                            # Update raster too
                            self.plot_raster(r_ind=r)
            else:
                # Only update the spectrum and/or raster for the canvas clicked
                self.spec_vline_val[r_ind] = xdata
                if self.display_mode[r_ind].lower().endswith('ccd'):
                    # If CCD mode is ON, only update crosshair and spec
                    self.plot_crosshair(r_ind=r_ind)
                else:
                    # Update raster too
                    self.plot_raster(r_ind=r_ind)

    def event_keyboard_nav(self, event):
        """Event for moving crosshair or vline with the keyboard"""
        r_ind = self.focused_r_ind
        plot_type = self.focused_type

        if r_ind > self.curr_layout_ind:
            # Just ignore if active canvas is hidden
            return
        elif event.key.split('+')[-1] not in ['left', 'right', 'up', 'down']:
            # Ignore non-arrow keys
            return

        # Setup a fake click object to send the the focused canvas
        class FakeClickEvent():
            pass

        fake_click = FakeClickEvent()
        fake_click.canvas = self.focused_fig.canvas
        fake_click.button = None
        
        arrow_dir = event.key.split('+')[-1]
        mod_key = event.key.split('+')[0] # e.g. "shift" or "ctrl" (if any)
        num_steps = 1
        if mod_key == 'shift':
            num_steps = 5
        elif mod_key == 'ctrl':
            num_steps = 10

        if plot_type == 'rast':
            if self.rast_crosshair[r_ind] is None:
                return
            # Get current crosshair coords
            old_xval, old_yval = self.rast_crosshair[r_ind].get_offsets().data[0]
            shift_x, shift_y = 0.0, 0.0
            if arrow_dir == 'left':
                shift_x = -num_steps*self.rast_xydelta[r_ind][0]
            elif arrow_dir == 'right':
                shift_x = num_steps*self.rast_xydelta[r_ind][0]
            elif arrow_dir == 'up':
                shift_y = num_steps*self.rast_xydelta[r_ind][1]
            elif arrow_dir == 'down':
                shift_y = -num_steps*self.rast_xydelta[r_ind][1]
            fake_click.xdata = old_xval + shift_x
            fake_click.ydata = old_yval + shift_y
            self.event_rast_click(fake_click)
        elif plot_type == 'spec':
            if self.spec_vline_val[r_ind] is None:
                return
            # Get current crosshair coords
            old_xval, old_yval = self.spec_vline_val[r_ind], 0.0
            shift_x, shift_y = 0.0, 0.0
            if arrow_dir == 'left':
                shift_x = -num_steps*self.spec_xydelta[r_ind][0]
            elif arrow_dir == 'right':
                shift_x = num_steps*self.spec_xydelta[r_ind][0]
            fake_click.xdata = old_xval + shift_x
            fake_click.ydata = old_yval
            self.event_spec_click(fake_click)

    def event_lim_dialog(self, event):
        """Event for opening the dialog box for viewing / setting raster limits"""
        r_ind = int(self.sender().objectName()[1]) # Get r_ind of selected panel
        self.dialog_r_ind = r_ind

        old_geometry = None
        if self.lim_dialog is not None:
            dialog_is_vis = self.lim_dialog.isVisible()
            if dialog_is_vis:
                old_geometry = self.lim_dialog.geometry()
            self.lim_dialog.close()
        self.lim_dialog = LimitsDialog(self, old_geometry=old_geometry)
        self.lim_dialog.show()
        self.update_canvas_focus(r_ind=None)

    def event_quit(self):
        """Close all figures, clean-up GUI objects, and quit the app"""
        for r in range(self.max_num_rasters):
            if self.rast_img[r] is not None:
                self.rast_img[r].figure.canvas.mpl_disconnect(self.rast_click[r])
                self.rast_img[r].figure.canvas.mpl_disconnect(self.rast_keyboard[r])
            if self.spec_fig[r] is not None:
                self.spec_fig[r].canvas.mpl_disconnect(self.spec_click[r])
                self.spec_fig[r].canvas.mpl_disconnect(self.spec_keyboard[r])
            self.button_select[r].close()
            self.box_line_id[r].close()
            self.box_var[r].close()
            self.button_reset_rast[r].close()
            self.rast_nav[r].close()
            self.rast_canvas[r].close()
            self.button_reset_spec[r].close()
            self.spec_nav[r].close()
            self.spec_canvas[r].close()
        if self.lim_dialog is not None:
            self.lim_dialog.close()
        QtWidgets.QApplication.instance().quit()
        self.close()

    def closeEvent(self, event):
        """Override the close event when the "X" button on the window is clicked"""
        self.event_quit()

#@dialog########################################################################
# DIALOG BOX FOR SETTING LIMITS AND CROSSHAIR COORDINATES ######################
################################################################################
class LimitsDialog(QtWidgets.QDialog):
    """Make a cutom dialog box for viewing and setting plot limits"""
    def __init__(self, parent=None, old_geometry=None):
        super().__init__(parent)

        curr_r_ind = parent.dialog_r_ind
        curr_filename = parent.selected_file[curr_r_ind]
        curr_xaxis_type = str(parent.rast_xaxis_type[curr_r_ind])
        self.setWindowTitle(f"Set Limits / Crosshair")
        self.grid = QtWidgets.QGridLayout(self)

        self.curr_xlim = [None, None]
        self.curr_ylim = [None, None]
        self.curr_cross = [None, None]
        
        self.edit_xlim = [None, None] # QLineEdit objects
        self.edit_ylim = [None, None]
        self.edit_cross = [None, None]
        self.edit_spec_xlim = [None, None]
        self.edit_spec_ylim = [None, None]
        self.edit_vline = None

        label_panel = QtWidgets.QLabel(self)
        label_panel.setText(f"Panel {curr_r_ind+1}   :   {curr_filename}")
        label_panel.setFont(parent.font_default_bold)

        label_limits = QtWidgets.QLabel(self)
        label_limits.setText('Image')
        label_limits.setFont(parent.font_default_bold)

        label_min = QtWidgets.QLabel(self)
        label_min.setText('     Min')
        label_min.setFont(parent.font_default)

        label_max = QtWidgets.QLabel(self)
        label_max.setText('     Max')
        label_max.setFont(parent.font_default)

        self.label_xaxis = QtWidgets.QLabel(self)
        self.label_xaxis.setText(curr_xaxis_type)
        self.label_xaxis.setFont(parent.font_default)

        label_yaxis = QtWidgets.QLabel(self)
        label_yaxis.setText('Solar-Y')
        label_yaxis.setFont(parent.font_default)

        grid_spacer_1 = QtWidgets.QLabel(self)
        grid_spacer_1.setText('')
        grid_spacer_1.setFont(parent.font_default)

        label_cross = QtWidgets.QLabel(self)
        label_cross.setText('Crosshair')
        label_cross.setFont(parent.font_default_bold)

        label_coords = QtWidgets.QLabel(self)
        label_coords.setText('     Values')
        label_coords.setFont(parent.font_default)

        label_xcc = QtWidgets.QLabel(self)
        label_xcc.setText(curr_xaxis_type)
        label_xcc.setFont(parent.font_default)

        label_ycc = QtWidgets.QLabel(self)
        label_ycc.setText('Solar-Y')
        label_ycc.setFont(parent.font_default)

        self.label_c_var = QtWidgets.QLabel(self)
        self.label_c_var.setText('Img Var')
        self.label_c_var.setFont(parent.font_default)

        self.cross_val = QtWidgets.QLabel(self)
        self.cross_val.setText('')
        self.cross_val.setFont(parent.font_small)

        grid_spacer_2 = QtWidgets.QLabel(self)
        grid_spacer_2.setText('')
        grid_spacer_2.setFont(parent.font_default)

        label_spec = QtWidgets.QLabel(self)
        label_spec.setText('Spectrum')
        label_spec.setFont(parent.font_default_bold)

        label_spec_min = QtWidgets.QLabel(self)
        label_spec_min.setText('     Min')
        label_spec_min.setFont(parent.font_default)

        label_spec_max = QtWidgets.QLabel(self)
        label_spec_max.setText('     Max')
        label_spec_max.setFont(parent.font_default)

        label_spec_xax = QtWidgets.QLabel(self)
        label_spec_xax.setText('Wavelength')
        label_spec_xax.setFont(parent.font_default)

        label_spec_yax = QtWidgets.QLabel(self)
        label_spec_yax.setText('Intensity')
        label_spec_yax.setFont(parent.font_default)

        label_vline = QtWidgets.QLabel(self)
        label_vline.setText('vline')
        label_vline.setFont(parent.font_default)

        grid_spacer_bot = QtWidgets.QLabel(self)
        grid_spacer_bot.setText('')
        grid_spacer_bot.setFont(parent.font_default)
        grid_spacer_bot.setMinimumWidth(int(0.5*parent.base_width))

        # Create all of the buttons
        self.button_ok = QtWidgets.QPushButton(f'OK')
        self.button_ok.setFont(parent.font_default)
        self.button_ok.setFixedWidth(parent.base_width)
        self.button_ok.clicked.connect(lambda: self.event_apply_everything(parent, close=True))

        self.button_cancel = QtWidgets.QPushButton(f'Cancel')
        self.button_cancel.setFont(parent.font_default)
        self.button_cancel.setFixedWidth(parent.base_width)
        self.button_cancel.clicked.connect(lambda: self.event_close(parent))

        self.button_apply = QtWidgets.QPushButton(f'Apply')
        self.button_apply.setFont(parent.font_default)
        self.button_apply.setFixedWidth(parent.base_width)
        self.button_apply.clicked.connect(lambda: self.event_apply_everything(parent))

        self.button_apply_all = QtWidgets.QPushButton(f'Apply ALL')
        self.button_apply_all.setFont(parent.font_default)
        self.button_apply_all.setFixedWidth(parent.base_width)
        self.button_apply_all.clicked.connect(lambda: self.event_apply_everything(parent, apply_all=True))

        self.button_set_lims = QtWidgets.QPushButton(f'Set')
        self.button_set_lims.setFont(parent.font_small)
        self.button_set_lims.setFixedWidth(parent.base_width)
        self.button_set_lims.clicked.connect(lambda: self.event_set_rast_lims(parent))

        self.button_set_all_lims = QtWidgets.QPushButton(f'Set ALL')
        self.button_set_all_lims.setFont(parent.font_small)
        self.button_set_all_lims.setFixedWidth(parent.base_width)
        self.button_set_all_lims.clicked.connect(lambda: self.event_set_rast_lims(parent, set_all=True))

        self.button_set_cross = QtWidgets.QPushButton(f'Set')
        self.button_set_cross.setFont(parent.font_small)
        self.button_set_cross.setFixedWidth(parent.base_width)
        self.button_set_cross.clicked.connect(lambda: self.event_set_crosshair(parent))

        self.button_set_all_cross = QtWidgets.QPushButton(f'Set ALL')
        self.button_set_all_cross.setFont(parent.font_small)
        self.button_set_all_cross.setFixedWidth(parent.base_width)
        self.button_set_all_cross.clicked.connect(lambda: self.event_set_crosshair(parent, set_all=True))

        self.button_set_spec = QtWidgets.QPushButton(f'Set')
        self.button_set_spec.setFont(parent.font_small)
        self.button_set_spec.setFixedWidth(parent.base_width)
        self.button_set_spec.clicked.connect(lambda: self.event_set_spec(parent))

        self.button_set_all_spec = QtWidgets.QPushButton(f'Set ALL')
        self.button_set_all_spec.setFont(parent.font_small)
        self.button_set_all_spec.setFixedWidth(parent.base_width)
        self.button_set_all_spec.clicked.connect(lambda: self.event_set_spec(parent, set_all=True))

        # Create the textbox widgets (Note: creation order sets TAB order)
        for L in range(2):
            self.edit_xlim[L] = QtWidgets.QLineEdit(self)
            self.edit_xlim[L].setFixedWidth(parent.base_width)
            self.edit_xlim[L].setText("")
            self.edit_xlim[L].setFont(parent.font_small)
            num_validator = QtGui.QDoubleValidator(-20000.0, 20000.0, 4, notation=1, 
                                                   parent=self.edit_xlim[L])
            self.edit_xlim[L].setValidator(num_validator)
        
        for L in range(2):
            self.edit_ylim[L] = QtWidgets.QLineEdit(self)
            self.edit_ylim[L].setFixedWidth(parent.base_width)
            self.edit_ylim[L].setText("")
            self.edit_ylim[L].setFont(parent.font_small)
            num_validator = QtGui.QDoubleValidator(-20000.0, 20000.0, 4, notation=1, 
                                                   parent=self.edit_ylim[L])
            self.edit_ylim[L].setValidator(num_validator)

        for L in range(2):
            self.edit_cross[L] = QtWidgets.QLineEdit(self)
            # self.edit_cross[L].setFixedWidth(2*parent.base_width+5)
            self.edit_cross[L].setText("")
            self.edit_cross[L].setFont(parent.font_small)
            num_validator = QtGui.QDoubleValidator(-20000.0, 20000.0, 4, notation=1, 
                                                   parent=self.edit_cross[L])
            self.edit_cross[L].setValidator(num_validator)

        for L in range(2):
            self.edit_spec_xlim[L] = QtWidgets.QLineEdit(self)
            self.edit_spec_xlim[L].setFixedWidth(parent.base_width)
            self.edit_spec_xlim[L].setText("")
            self.edit_spec_xlim[L].setFont(parent.font_small)
            num_validator = QtGui.QDoubleValidator(-20000.0, 20000.0, 4, notation=1, 
                                                   parent=self.edit_spec_xlim[L])
            self.edit_spec_xlim[L].setValidator(num_validator)
        
        for L in range(2):
            self.edit_spec_ylim[L] = QtWidgets.QLineEdit(self)
            self.edit_spec_ylim[L].setFixedWidth(parent.base_width)
            self.edit_spec_ylim[L].setText("")
            self.edit_spec_ylim[L].setFont(parent.font_small)
            num_validator = QtGui.QDoubleValidator(-20000.0, 20000.0, 4, notation=1, 
                                                   parent=self.edit_spec_ylim[L])
            self.edit_spec_ylim[L].setValidator(num_validator)

        self.edit_vline = QtWidgets.QLineEdit(self)
        self.edit_vline.setText("")
        self.edit_vline.setFont(parent.font_small)
        num_validator = QtGui.QDoubleValidator(-20000.0, 20000.0, 4, notation=1, 
                                                parent=self.edit_vline)
        self.edit_vline.setValidator(num_validator)

        # Arange the widgets
        self.grid.addWidget(label_panel, 0, 0, 1, 5)
        
        self.grid.addWidget(label_limits, 1, 0)
        self.grid.addWidget(label_min, 1, 1)
        self.grid.addWidget(label_max, 1, 2)
        self.grid.addWidget(self.label_xaxis, 2, 0)
        self.grid.addWidget(self.edit_xlim[0], 2, 1)
        self.grid.addWidget(self.edit_xlim[1], 2, 2)
        self.grid.addWidget(label_yaxis, 3, 0)
        self.grid.addWidget(self.edit_ylim[0], 3, 1)
        self.grid.addWidget(self.edit_ylim[1], 3, 2)
        
        self.grid.addWidget(grid_spacer_1, 4, 0)
        self.grid.addWidget(label_cross, 5, 0)
        self.grid.addWidget(label_coords, 5, 1)
        self.grid.addWidget(label_xcc, 6, 0)
        self.grid.addWidget(self.edit_cross[0], 6, 1, 1, 2)
        self.grid.addWidget(label_ycc, 7, 0)
        self.grid.addWidget(self.edit_cross[1], 7, 1, 1, 2)
        self.grid.addWidget(self.label_c_var, 8, 0)
        self.grid.addWidget(self.cross_val, 8, 1, 1, 2)

        self.grid.addWidget(grid_spacer_2, 9, 0)
        self.grid.addWidget(label_spec, 10, 0)
        self.grid.addWidget(label_spec_min, 10, 1)
        self.grid.addWidget(label_spec_max, 10, 2)
        self.grid.addWidget(label_spec_xax, 11, 0)
        self.grid.addWidget(self.edit_spec_xlim[0], 11, 1)
        self.grid.addWidget(self.edit_spec_xlim[1], 11, 2)
        self.grid.addWidget(label_spec_yax, 12, 0)
        self.grid.addWidget(self.edit_spec_ylim[0], 12, 1)
        self.grid.addWidget(self.edit_spec_ylim[1], 12, 2)
        self.grid.addWidget(label_vline, 13, 0)
        self.grid.addWidget(self.edit_vline, 13, 1, 1, 2)

        # Note: The first button ADDED (not created) is the default for "enter"
        self.grid.addWidget(grid_spacer_bot, 14, 3)
        self.grid.addWidget(self.button_ok, 15, 0)
        self.grid.addWidget(self.button_cancel, 15, 1)
        self.grid.addWidget(self.button_apply, 15, 2)
        self.grid.addWidget(self.button_apply_all, 15, 4)

        self.grid.addWidget(self.button_set_lims, 2, 4)
        self.grid.addWidget(self.button_set_all_lims, 3, 4)

        self.grid.addWidget(self.button_set_cross, 6, 4)
        self.grid.addWidget(self.button_set_all_cross, 7, 4)

        self.grid.addWidget(self.button_set_spec, 11, 4)
        self.grid.addWidget(self.button_set_all_spec, 12, 4)

        self.setLayout(self.grid)
        self.read_rast_limits(parent)
        self.read_rast_crosshair(parent)
        self.read_spec_limits(parent)
        if old_geometry is not None:
            self.setGeometry(old_geometry) # Keep previous on-screen location

    def read_rast_limits(self, parent):
        """Get the raster limits at the time this dialog is opened"""
        r_ind = parent.dialog_r_ind
        if parent.rast_ax[r_ind] is not None:
            # Set the X-axis label
            curr_xaxis_type = str(parent.rast_xaxis_type[r_ind])
            self.label_xaxis.setText(curr_xaxis_type)

            # Get current ax limits              
            self.curr_xlim = parent.rast_ax[r_ind].get_xlim()
            self.curr_ylim = parent.rast_ax[r_ind].get_ylim()

            # Set text boxes with the values (with rounding)
            for L in range(2):
                self.edit_xlim[L].setText(str(round(self.curr_xlim[L], 4)))
                self.edit_ylim[L].setText(str(round(self.curr_ylim[L], 4)))
        
    def read_rast_crosshair(self, parent):
        """Get the crosshair coords at the time this dialog is opened"""
        r_ind = parent.dialog_r_ind
        curr_img_var = str(parent.var_label[r_ind])
        self.label_c_var.setText(curr_img_var.split('_')[0])
        var_val = ''
        if parent.rast_crosshair[r_ind] is not None:
            # Get current crosshair coords
            self.curr_cross = parent.rast_crosshair[r_ind].get_offsets().data[0]
            var_val = str(round(parent.crosshair_val[r_ind], 2))

            # Set text boxes with the values (with rounding)
            self.edit_cross[0].setText(str(round(self.curr_cross[0], 4)))
            self.edit_cross[1].setText(str(round(self.curr_cross[1], 4)))
        self.cross_val.setText(var_val)

    def read_spec_limits(self, parent):
        """Get the spec lims and vline at the time this dialog is opened"""
        r_ind = parent.dialog_r_ind
        if parent.spec_ax[r_ind] is not None:

            # Get current ax limits              
            curr_spec_xlim = parent.spec_ax[r_ind].get_xlim()
            curr_spec_ylim = parent.spec_ax[r_ind].get_ylim()

            # Set text boxes with the values (with rounding)
            for L in range(2):
                self.edit_spec_xlim[L].setText(str(round(curr_spec_xlim[L], 4)))
                self.edit_spec_ylim[L].setText(str(round(curr_spec_ylim[L], 4)))

            # Get current vline value and update the text box
            if parent.spec_vline_val[r_ind] is not None:
                curr_vline = parent.spec_vline_val[r_ind]
                self.edit_vline.setText(str(round(curr_vline, 4)))

    def event_set_rast_lims(self, parent, set_all=False):
        """Apply the input limits to the selected plots"""
        # Reading values from the text boxes and convert to floats (if able)
        new_xlim = [None, None]
        new_ylim = [None, None]
        for L in range(2):
            try:
                new_xlim[L] = float(self.edit_xlim[L].text())
            except:
                new_xlim[L] = None
            try:
                new_ylim[L] = float(self.edit_ylim[L].text())
            except:
                new_ylim[L] = None
        
        # Set list of plot indices to apply the limits to
        if set_all:
            r_ind_list = [ri for ri in range(parent.max_num_rasters)]
        else:
            r_ind_list = [parent.dialog_r_ind]
        
        # Validate and set the limits
        coord_buffer = 5
        aspect_buffer = 0.008
        source_xaxis_type = parent.rast_xaxis_type[parent.dialog_r_ind]
        for RI in r_ind_list:
            # Skip empty or hidden images
            if parent.rast_ax[RI] is not None and RI <= parent.curr_layout_ind:
                this_xaxis_type = parent.rast_xaxis_type[RI]
                this_new_xlim = new_xlim.copy()
                this_new_ylim = new_ylim.copy()
                this_curr_xlim = parent.rast_ax[RI].get_xlim()
                this_curr_ylim = parent.rast_ax[RI].get_ylim()
                this_ax_xrange = parent.rast_xcoords[RI][[0,-1]]
                this_ax_yrange = parent.rast_ycoords[RI][[0,-1]]

                # Replace "None" with min or max data values
                for V in range(2):
                    if this_new_xlim[V] is None:
                        this_new_xlim[V] = this_ax_xrange[V]
                    if this_new_ylim[V] is None:
                        this_new_ylim[V] = this_ax_yrange[V]
                
                # Check to see if these are "reasonable" limits for this plot
                # Note: will skip shifting if both limits are nearly unchanged
                go_change_xlim = False
                if ((this_new_xlim[0] < this_ax_xrange[1]) 
                and (this_new_xlim[1] > this_ax_xrange[0])
                and ((np.abs(this_new_xlim[0] - this_curr_xlim[0]) >= coord_buffer)
                    or (np.abs(this_new_xlim[1] - this_curr_xlim[1]) >= coord_buffer))
                and this_xaxis_type == source_xaxis_type):
                    # Also only change plots with the same x-axis type
                    go_change_xlim = True
                
                go_change_ylim = False
                if ((this_new_ylim[0] < this_ax_yrange[1]) 
                and (this_new_ylim[1] > this_ax_yrange[0])
                and ((np.abs(this_new_ylim[0] - this_curr_ylim[0]) >= coord_buffer)
                    or (np.abs(this_new_ylim[1] - this_curr_ylim[1]) >= coord_buffer))):
                    go_change_ylim = True

                # Actually apply the changes
                if go_change_xlim or go_change_ylim:
                    # Check aspect ratio
                    base_aspect_ratio = np.diff(this_ax_xrange)/np.diff(this_ax_yrange)
                    new_aspect_ratio = np.diff(this_new_xlim)/np.diff(this_new_ylim)

                    use_auto_aspect = True
                    if np.abs(base_aspect_ratio - new_aspect_ratio) < aspect_buffer:
                        use_auto_aspect = False

                    if use_auto_aspect:
                        # Enables non-square pixels
                        parent.rast_ax[RI].set_aspect('auto') 
                        parent.rast_auto_aspect[RI] = True

                    if go_change_xlim:
                        parent.rast_ax[RI].set_xlim(this_new_xlim)
                    if go_change_ylim:
                        parent.rast_ax[RI].set_ylim(this_new_ylim)
                    parent.rast_img[RI].figure.canvas.draw_idle() # Update the plot!
                    self.read_rast_limits(parent) # update the display of limits

    def event_set_crosshair(self, parent, set_all=False):
        """Move the crosshair location to the input coordinates"""
        new_cross = [None, None]
        for L in range(2):
            try:
                new_cross[L] = float(self.edit_cross[L].text())
            except:
                new_cross[L] = None

        r_ind = parent.dialog_r_ind
        if parent.rast_ax[r_ind] is None or r_ind > parent.curr_layout_ind:
            # Skip if the raster is empty or hidden
            pass
        else: 
            # "Easy" way, fake a click on the selected raster
            class FakeClickEvent():
                pass

            # Setup the fake event
            fake_click = FakeClickEvent()
            fake_click.canvas = parent.rast_canvas[r_ind]
            fake_click.button = None
            fake_click.xdata = new_cross[0]
            fake_click.ydata = new_cross[1]

            # Send the fake click (while preserving state of crosshair sync)
            prev_sync_cross_check = parent.radio_sync_cross.isChecked()
            if set_all:
                # Change ALL crosshairs (if valid)
                parent.radio_sync_cross.setChecked(True)
            else:
                # Only change selected panel
                parent.radio_sync_cross.setChecked(False)
            parent.event_rast_click(fake_click)
            parent.radio_sync_cross.setChecked(prev_sync_cross_check)

    def event_set_spec(self, parent, set_all=False):
        """Set the limits and vline in the selcted spectrum"""
        # Reading values from the text boxes and convert to floats (if able)
        new_spec_xlim = [None, None]
        new_spec_ylim = [None, None]
        for L in range(2):
            try:
                new_spec_xlim[L] = float(self.edit_spec_xlim[L].text())
            except:
                new_spec_xlim[L] = None
            try:
                new_spec_ylim[L] = float(self.edit_spec_ylim[L].text())
            except:
                new_spec_ylim[L] = None
        
        # Set list of plot indices to apply the limits to
        if set_all:
            r_ind_list = [ri for ri in range(parent.max_num_rasters)]
        else:
            r_ind_list = [parent.dialog_r_ind]
        
        # Validate and set the limits
        coord_buffer = 0.05
        for RI in r_ind_list:
            # Skip empty or hidden images
            if parent.spec_ax[RI] is not None and RI <= parent.curr_layout_ind:
                this_new_xlim = new_spec_xlim.copy()
                this_new_ylim = new_spec_ylim.copy()
                this_curr_xlim = parent.spec_ax[RI].get_xlim()
                this_curr_ylim = parent.spec_ax[RI].get_ylim()

                # Get max and min data values current in plot
                line_data = parent.spec_ax[RI].lines[0].get_data()
                this_ax_xrange = [np.amin(line_data[0])-0.03, np.amax(line_data[0])+0.03]
                this_ax_yrange = [np.amin(line_data[1])-10, np.amax(line_data[1])+10]

                # Replace "None" with min or max data values
                for V in range(2):
                    if this_new_xlim[V] is None:
                        this_new_xlim[V] = this_ax_xrange[V]
                    if this_new_ylim[V] is None:
                        this_new_ylim[V] = this_ax_yrange[V]
                
                # Check to see if these are "reasonable" limits for this plot
                # Note: will skip shifting if both limits are nearly unchanged
                go_change_xlim = False
                if ((this_new_xlim[0] < this_ax_xrange[1]) 
                and (this_new_xlim[1] > this_ax_xrange[0])
                and ((np.abs(this_new_xlim[0] - this_curr_xlim[0]) >= coord_buffer)
                    or (np.abs(this_new_xlim[1] - this_curr_xlim[1]) >= coord_buffer))):
                    go_change_xlim = True
                
                go_change_ylim = False
                if ((this_new_ylim[0] < this_ax_yrange[1]) 
                and (this_new_ylim[1] > this_ax_yrange[0])
                and ((np.abs(this_new_ylim[0] - this_curr_ylim[0]) >= coord_buffer)
                    or (np.abs(this_new_ylim[1] - this_curr_ylim[1]) >= coord_buffer))):
                    go_change_ylim = True

                # Actually apply the changes
                if go_change_xlim or go_change_ylim:
                    if go_change_xlim:
                        parent.spec_ax[RI].set_xlim(this_new_xlim)
                    if go_change_ylim:
                        parent.spec_ax[RI].set_ylim(this_new_ylim)
                    parent.spec_ax[RI].figure.canvas.draw_idle() # Update the plot! # Update the plot!
                    self.read_spec_limits(parent) # update the display of limits

        # Read the vline value
        try:
            new_vline = float(self.edit_vline.text())
        except:
            new_vline = None

        # Set the new vline by faking a click on the spectrum
        r_ind = parent.dialog_r_ind
        if parent.spec_ax[r_ind] is None or r_ind > parent.curr_layout_ind:
            # Skip if the spectrum is empty or hidden
            pass
        else: 
            # "Easy" way, fake a click on the selected raster
            class FakeClickEvent():
                pass

            # Setup the fake event
            fake_click = FakeClickEvent()
            fake_click.canvas = parent.spec_canvas[r_ind]
            fake_click.button = None
            fake_click.xdata = new_vline
            fake_click.ydata = 111.111

            # Send the fake click (while preserving state of crosshair sync)
            prev_sync_vline_check = parent.radio_sync_vline.isChecked()
            if set_all:
                # Change ALL crosshairs (if valid)
                parent.radio_sync_vline.setChecked(True)
            else:
                # Only change selected panel
                parent.radio_sync_vline.setChecked(False)
            parent.event_spec_click(fake_click)
            parent.radio_sync_vline.setChecked(prev_sync_vline_check)

    def event_apply_everything(self, parent, close=False, apply_all=False):
        """Event for buttons at the bottom that apply ALL changes"""
        
        self.event_set_rast_lims(parent, set_all=apply_all)
        self.event_set_crosshair(parent, set_all=apply_all)
        self.event_set_spec(parent, set_all=apply_all)

        if close:
            # Close the dialog after setting the limits (use by "OK" button)
            self.event_close(parent)

    def event_close(self, parent):
        """Just close the dialog without changing anything"""
        if parent is not None and 'dialog_r_ind' in parent.__dict__:
            parent.dialog_r_ind = None
        self.close()

    def closeEvent(self, event):
        """Override the close event when the "X" button on the window is clicked"""
        self.event_close(None)

#@main##########################################################################
# MAIN FUNCTION CALLED WHEN INVOKED ############################################
################################################################################
def eis_explore_raster():
    # check the input
    if len(sys.argv) > 1:
        fit_filepath = sys.argv[1]
    else:
        fit_filepath = None

    app = QtWidgets.QApplication.instance()
    if not app:
        app = QtWidgets.QApplication(sys.argv)

    # app = QtWidgets.QApplication(sys.argv)
    win = MainWindow(fit_filepath)
    sys.exit(app.exec_())

if __name__ == '__main__':
    eis_explore_raster()