"""Module defining element data structures."""

from dataclasses import dataclass
from typing import Final


@dataclass(frozen=True)
class Element:
    """Represents a chemical element with essential properties.

    Attributes:
        atomic_number: The atomic number (number of protons).
        symbol: The chemical symbol (e.g., "H", "C").
    """

    atomic_number: int
    symbol: str


# A registry mapping uppercase element symbols to Element instances.
# this dict was generated with https://github.com/coltonbh/qcconst/blob/master/src/qcconst/data/pubchem.csv as context
# so thanks to coltonbh for the data!
ELEMENT_DATA: Final[dict[str, Element]] = {
    "H": Element(atomic_number=1, symbol="H"),
    "HE": Element(atomic_number=2, symbol="He"),
    "LI": Element(atomic_number=3, symbol="Li"),
    "BE": Element(atomic_number=4, symbol="Be"),
    "B": Element(atomic_number=5, symbol="B"),
    "C": Element(atomic_number=6, symbol="C"),
    "N": Element(atomic_number=7, symbol="N"),
    "O": Element(atomic_number=8, symbol="O"),
    "F": Element(atomic_number=9, symbol="F"),
    "NE": Element(atomic_number=10, symbol="Ne"),
    "NA": Element(atomic_number=11, symbol="Na"),
    "MG": Element(atomic_number=12, symbol="Mg"),
    "AL": Element(atomic_number=13, symbol="Al"),
    "SI": Element(atomic_number=14, symbol="Si"),
    "P": Element(atomic_number=15, symbol="P"),
    "S": Element(atomic_number=16, symbol="S"),
    "CL": Element(atomic_number=17, symbol="Cl"),
    "AR": Element(atomic_number=18, symbol="Ar"),
    "K": Element(atomic_number=19, symbol="K"),
    "CA": Element(atomic_number=20, symbol="Ca"),
    "SC": Element(atomic_number=21, symbol="Sc"),
    "TI": Element(atomic_number=22, symbol="Ti"),
    "V": Element(atomic_number=23, symbol="V"),
    "CR": Element(atomic_number=24, symbol="Cr"),
    "MN": Element(atomic_number=25, symbol="Mn"),
    "FE": Element(atomic_number=26, symbol="Fe"),
    "CO": Element(atomic_number=27, symbol="Co"),
    "NI": Element(atomic_number=28, symbol="Ni"),
    "CU": Element(atomic_number=29, symbol="Cu"),
    "ZN": Element(atomic_number=30, symbol="Zn"),
    "GA": Element(atomic_number=31, symbol="Ga"),
    "GE": Element(atomic_number=32, symbol="Ge"),
    "AS": Element(atomic_number=33, symbol="As"),
    "SE": Element(atomic_number=34, symbol="Se"),
    "BR": Element(atomic_number=35, symbol="Br"),
    "KR": Element(atomic_number=36, symbol="Kr"),
    "RB": Element(atomic_number=37, symbol="Rb"),
    "SR": Element(atomic_number=38, symbol="Sr"),
    "Y": Element(atomic_number=39, symbol="Y"),
    "ZR": Element(atomic_number=40, symbol="Zr"),
    "NB": Element(atomic_number=41, symbol="Nb"),
    "MO": Element(atomic_number=42, symbol="Mo"),
    "TC": Element(atomic_number=43, symbol="Tc"),
    "RU": Element(atomic_number=44, symbol="Ru"),
    "RH": Element(atomic_number=45, symbol="Rh"),
    "PD": Element(atomic_number=46, symbol="Pd"),
    "AG": Element(atomic_number=47, symbol="Ag"),
    "CD": Element(atomic_number=48, symbol="Cd"),
    "IN": Element(atomic_number=49, symbol="In"),
    "SN": Element(atomic_number=50, symbol="Sn"),
    "SB": Element(atomic_number=51, symbol="Sb"),
    "TE": Element(atomic_number=52, symbol="Te"),
    "I": Element(atomic_number=53, symbol="I"),
    "XE": Element(atomic_number=54, symbol="Xe"),
    "CS": Element(atomic_number=55, symbol="Cs"),
    "BA": Element(atomic_number=56, symbol="Ba"),
    "LA": Element(atomic_number=57, symbol="La"),
    "CE": Element(atomic_number=58, symbol="Ce"),
    "PR": Element(atomic_number=59, symbol="Pr"),
    "ND": Element(atomic_number=60, symbol="Nd"),
    "PM": Element(atomic_number=61, symbol="Pm"),
    "SM": Element(atomic_number=62, symbol="Sm"),
    "EU": Element(atomic_number=63, symbol="Eu"),
    "GD": Element(atomic_number=64, symbol="Gd"),
    "TB": Element(atomic_number=65, symbol="Tb"),
    "DY": Element(atomic_number=66, symbol="Dy"),
    "HO": Element(atomic_number=67, symbol="Ho"),
    "ER": Element(atomic_number=68, symbol="Er"),
    "TM": Element(atomic_number=69, symbol="Tm"),
    "YB": Element(atomic_number=70, symbol="Yb"),
    "LU": Element(atomic_number=71, symbol="Lu"),
    "HF": Element(atomic_number=72, symbol="Hf"),
    "TA": Element(atomic_number=73, symbol="Ta"),
    "W": Element(atomic_number=74, symbol="W"),
    "RE": Element(atomic_number=75, symbol="Re"),
    "OS": Element(atomic_number=76, symbol="Os"),
    "IR": Element(atomic_number=77, symbol="Ir"),
    "PT": Element(atomic_number=78, symbol="Pt"),
    "AU": Element(atomic_number=79, symbol="Au"),
    "HG": Element(atomic_number=80, symbol="Hg"),
    "TL": Element(atomic_number=81, symbol="Tl"),
    "PB": Element(atomic_number=82, symbol="Pb"),
    "BI": Element(atomic_number=83, symbol="Bi"),
    "PO": Element(atomic_number=84, symbol="Po"),
    "AT": Element(atomic_number=85, symbol="At"),
    "RN": Element(atomic_number=86, symbol="Rn"),
    "FR": Element(atomic_number=87, symbol="Fr"),
    "RA": Element(atomic_number=88, symbol="Ra"),
    "AC": Element(atomic_number=89, symbol="Ac"),
    "TH": Element(atomic_number=90, symbol="Th"),
    "PA": Element(atomic_number=91, symbol="Pa"),
    "U": Element(atomic_number=92, symbol="U"),
    "NP": Element(atomic_number=93, symbol="Np"),
    "PU": Element(atomic_number=94, symbol="Pu"),
    "AM": Element(atomic_number=95, symbol="Am"),
    "CM": Element(atomic_number=96, symbol="Cm"),
    "BK": Element(atomic_number=97, symbol="Bk"),
    "CF": Element(atomic_number=98, symbol="Cf"),
    "ES": Element(atomic_number=99, symbol="Es"),
    "FM": Element(atomic_number=100, symbol="Fm"),
    "MD": Element(atomic_number=101, symbol="Md"),
    "NO": Element(atomic_number=102, symbol="No"),
    "LR": Element(atomic_number=103, symbol="Lr"),
    "RF": Element(atomic_number=104, symbol="Rf"),
    "DB": Element(atomic_number=105, symbol="Db"),
    "SG": Element(atomic_number=106, symbol="Sg"),
    "BH": Element(atomic_number=107, symbol="Bh"),
    "HS": Element(atomic_number=108, symbol="Hs"),
    "MT": Element(atomic_number=109, symbol="Mt"),
    "DS": Element(atomic_number=110, symbol="Ds"),
    "RG": Element(atomic_number=111, symbol="Rg"),
    "CN": Element(atomic_number=112, symbol="Cn"),
    "NH": Element(atomic_number=113, symbol="Nh"),
    "FL": Element(atomic_number=114, symbol="Fl"),
    "MC": Element(atomic_number=115, symbol="Mc"),
    "LV": Element(atomic_number=116, symbol="Lv"),
    "TS": Element(atomic_number=117, symbol="Ts"),
    "OG": Element(atomic_number=118, symbol="Og"),
}
