# Copyright 2023 Efabless Corporation
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
from enum import Enum
from typing import Dict, Optional
from dataclasses import dataclass, replace


@dataclass
class DesignFormatObject:
    """
    Metadata about the various possible text or binary representations (views)
    of any design.

    For example, ``DesignFormat.NETLIST.value`` has the metadata for Netlist
    views.

    :param id: A lowercase alphanumeric identifier for the design format.
        Some IDs in LibreLane 2.X use dashes. This is an inconsistency that will
        be addressed in the next major version of LibreLane as it would be a
        breaking change.
    :param extension: The file extension for designs saved in this format.
    :param name: A human-readable name for this design format.
    :param folder_override: The subdirectory when
        :meth:`librelane.state.State.save_snapshot` is called on a state. If
        unset, the value for ``id`` will be used.
    :param multiple: Whether this view may have multiple files (typically, files
        that are different across multiple corners or similar.)
    """

    id: str
    extension: str
    name: str
    folder_override: Optional[str] = None
    multiple: bool = False

    _instance_optional: bool = False

    @property
    def folder(self) -> str:
        return self.folder_override or self.id

    @property
    def optional(self) -> bool:
        return self._instance_optional


class DesignFormat(Enum):
    """
    An `enumeration <https://docs.python.org/3/library/enum.html>`_ of a number
    of :class:`librelane.state.DesignFormatObject`\\s representing the various
    possible text or binary representations (views) supported by LibreLane
    states.

    Members of this enumeration are used as the keys of
    :class:`librelane.state.State` objects.
    """

    NETLIST: DesignFormatObject = DesignFormatObject(
        "nl",
        "nl.v",
        "Verilog Netlist",
    )
    POWERED_NETLIST: DesignFormatObject = DesignFormatObject(
        "pnl",
        "pnl.v",
        "Powered Verilog Netlist",
    )
    POWERED_NETLIST_SDF_FRIENDLY: DesignFormatObject = DesignFormatObject(
        "pnl-sdf-friendly",
        "pnl-sdf.v",
        "Powered Verilog Netlist For SDF Simulation (Without Fill Cells)",
        folder_override="pnl",
    )
    POWERED_NETLIST_NO_PHYSICAL_CELLS: DesignFormatObject = DesignFormatObject(
        "pnl-npc",
        "pnl-npc.v",
        "Powered Verilog Netlist Without Physical Cells (Fill Cells and Diode Cells)",
        folder_override="pnl",
    )

    DEF: DesignFormatObject = DesignFormatObject(
        "def",
        "def",
        "Design Exchange Format",
    )
    LEF: DesignFormatObject = DesignFormatObject(
        "lef",
        "lef",
        "Library Exchange Format",
    )
    OPENROAD_LEF: DesignFormatObject = DesignFormatObject(
        "openroad-lef",
        "openroad.lef",
        "Library Exchange Format Generated by OpenROAD",
        folder_override="lef",
    )
    ODB: DesignFormatObject = DesignFormatObject(
        "odb",
        "odb",
        "OpenDB Database",
    )

    SDC: DesignFormatObject = DesignFormatObject(
        "sdc",
        "sdc",
        "Design Constraints",
    )
    SDF: DesignFormatObject = DesignFormatObject(
        "sdf",
        "sdf",
        "Standard Delay Format",
        multiple=True,
    )
    SPEF: DesignFormatObject = DesignFormatObject(
        "spef",
        "spef",
        "Standard Parasitics Extraction Format",
        multiple=True,  # nom, min, max, ...
    )
    LIB: DesignFormatObject = DesignFormatObject(
        "lib",
        "lib",
        "LIB Timing Library Format",
        multiple=True,
    )
    SPICE: DesignFormatObject = DesignFormatObject(
        "spice",
        "spice",
        "Simulation Program with Integrated Circuit Emphasis",
    )

    MAG: DesignFormatObject = DesignFormatObject(
        "mag",
        "mag",
        "Magic VLSI View",
    )

    GDS: DesignFormatObject = DesignFormatObject(
        "gds",
        "gds",
        "GDSII Stream",
    )
    MAG_GDS: DesignFormatObject = DesignFormatObject(
        "mag_gds",
        "magic.gds",
        "GDSII Stream (Magic)",
    )
    KLAYOUT_GDS: DesignFormatObject = DesignFormatObject(
        "klayout_gds",
        "klayout.gds",
        "GDSII Stream (KLayout)",
    )

    JSON_HEADER: DesignFormatObject = DesignFormatObject(
        "json_h",
        "h.json",
        "Design JSON Header File",
    )
    VERILOG_HEADER: DesignFormatObject = DesignFormatObject(
        "vh",
        "vh",
        "Verilog Header",
    )

    def __str__(self) -> str:
        return self.value.id

    @staticmethod
    def by_id(id: str) -> Optional["DesignFormat"]:
        return _designformat_by_id.get(id)

    def mkOptional(self) -> "DesignFormat":
        # HACK: Create ephemeral DesignFormat copy until 3.0.0 lets us do this
        # a bit more appropriately.
        clone = object.__new__(DesignFormat)
        clone._name_ = self._name_
        clone._value_ = replace(self._value_)
        clone._value_._instance_optional = True
        return clone


_designformat_by_id: Dict[str, "DesignFormat"] = {
    format.value.id: format for format in DesignFormat
}
