from pydantic import BaseModel, ConfigDict, ValidationError
from redis_om import JsonModel, EmbeddedJsonModel, Field
from enum import Enum
from datetime import date, datetime, time, timedelta

from typing import Dict, List
from enum import Enum

class Split(str, Enum):

    dev = 'dev'
    test = 'test'
    val = 'val'
    train = 'train'
    
# Questions as from the Digi dataset
class Question(BaseModel, use_enum_values=True):
    question_number: str
    question: str
    correct_answer: str
    has_media: bool = False
    dataset: str | None = None
    qualification: str
    level: str
    section: str | None = None
    session: str | None = None
    session_name: str | None = None
    split: Split
    answer_choices: Dict[str, str]
    marks: int = 1

    def prompt(self):
        return f"Q: {self.question}\n"+("\n" +"\n".join(o+': '+v for o,v in self.answer_choices.items()) if self.answer_choices else "")

# Questions generated by the model
class GeneratedQuestion(BaseModel):
    question_number: str
    question: str
    correct_answer: str
    options: List[str]
    
    def __str__(self):
        return f"N: {self.question_number}\nQ: {self.question}\n"+ ("\n".join([o for i,o in enumerate(self.options)]) if self.options else "")
        


class GeneratedQuestionnaire(BaseModel):
    questionnaire: list[GeneratedQuestion]
    
    def __str__(self):
        return "\n".join(str(q) for q in self.questionnaire)
    

class ChatRole(str, Enum):
     system = 'system'
     user = 'user'
     assistant = 'assistant'

class ChatMessage(EmbeddedJsonModel, use_enum_values=True):
    role: ChatRole
    content : str

class Chat(JsonModel):
    user: str = Field(index=True)
    agent: str
    ts: datetime
    msgs: List[ChatMessage]
    summary: str | None = None




