## Documentation Requirements

When contributing to Synqed, proper documentation is essential to ensure users can leverage your improvements.

### Primary Documentation Location

**All contributions MUST include updates to the full API documentation:**

**Location:** [https://github.com/SynqLabs/synqed/tree/main/api/python](https://github.com/SynqLabs/synqed/tree/main/api/python)

This is our **official API documentation** that users reference. Any changes to functionality, new features, or improvements must be documented here.

**What to update:**
- API method signatures and examples
- Usage patterns and code samples
- Configuration options and parameters
- Best practices and troubleshooting guides
- Complete working examples

### Secondary Documentation (Important Changes Only)

For **critical or breaking changes**, also update:

**Location:** [https://github.com/SynqLabs/synqed-python](https://github.com/SynqLabs/synqed-python)

**When to update:**
- Breaking API changes
- Major new features
- Significant architectural changes
- Security updates
- Deprecation notices

---

## Contribution Workflow

### 1. Before You Start

- Read the [Developer Onboarding Guide](DEVELOPER_ONBOARDING.md)
- Check existing issues and pull requests
- Discuss major changes in an issue first
- Ensure you understand the documentation requirements

### 2. Making Changes

```bash
# 1. Fork and clone the repository
git clone <your-fork-url>
cd Synq

# 2. Create a feature branch
git checkout -b your-name/feature-description

# 3. Make your code changes
# ... edit files ...

# 4. Write/update tests
# ... add tests in tests/ directory ...

# 5. Update documentation
# REQUIRED: Update API documentation at:
# https://github.com/SynqLabs/synqed/tree/main/api/python

# 6. Run tests locally
pytest --cov=src/synqed --cov-report=term

# 7. Format and lint
ruff format src/ tests/
ruff check src/ tests/

# 8. Commit your changes
git add .
git commit -m "feat: your feature description"

# 9. Push and create PR
git push origin your-name/feature-description
```

### 3. Pull Request Requirements

Your PR **must include**:

- ✅ **Code changes** with proper implementation
- ✅ **Tests** with >90% coverage
- ✅ **Documentation updates** in the API docs (required)
- ✅ **Example usage** demonstrating the feature
- ✅ **Clear PR description** explaining the change
- ✅ **No linting errors** (run `ruff check`)
- ✅ **Passing tests** (run `pytest`)

---

## Documentation Guidelines

### API Documentation Format

When updating the API documentation, follow these guidelines:

#### For New Features

```markdown
## Feature Name

Brief description of what this feature does and why it's useful.

### Usage

\```python
# Clear, working code example
from synqed import NewFeature

feature = NewFeature(
    param1="value1",
    param2="value2"
)

result = await feature.execute()
print(result)
\```

### Parameters

- `param1` (str): Description of parameter 1
- `param2` (int, optional): Description of parameter 2. Default: 10

### Returns

- `ResultType`: Description of what is returned

### Example

\```python
# Complete, runnable example
import asyncio
from synqed import NewFeature

async def main():
    feature = NewFeature(param1="test")
    result = await feature.execute()
    print(f"Result: {result}")

asyncio.run(main())
\```

### Notes

- Important considerations
- Edge cases
- Performance implications
```

#### For API Changes

```markdown
## Modified Feature

⚠️ **Breaking Change** (if applicable)

### What Changed

Clear description of the change and migration path.

### Before

\```python
# Old usage
old_way = OldAPI(param="value")
\```

### After

\```python
# New usage
new_way = NewAPI(new_param="value")
\```

### Migration Guide

Step-by-step instructions for updating existing code.
```

#### For Bug Fixes

```markdown
## Bug Fix: Issue Title

### Problem

Description of the bug that was fixed.

### Solution

Explanation of how it was fixed.

### Example

\```python
# This now works correctly
from synqed import FixedFeature

# Previously failed, now succeeds
result = await FixedFeature().execute()
\```
```

---

## 🔍 What Requires Documentation Updates

| Change Type | API Docs Required | synqed-python Required |
|------------|------------------|----------------------|
| New feature | ✅ Yes | ⚠️ If major |
| Bug fix | ✅ Yes | ❌ Usually no |
| API change | ✅ Yes | ✅ Yes (breaking) |
| New method | ✅ Yes | ⚠️ If important |
| Performance improvement | ✅ Yes | ❌ No |
| Internal refactor | ❌ No | ❌ No |
| New example | ✅ Yes | ⚠️ If showcasing major feature |
| Deprecation | ✅ Yes | ✅ Yes |
| Security fix | ✅ Yes | ✅ Yes |

---

## ✍️ Commit Message Format

Follow [Conventional Commits](https://www.conventionalcommits.org/):

```
<type>(<scope>): <description>

[optional body]

[optional footer]
```

**Types:**
- `feat`: New feature
- `fix`: Bug fix
- `docs`: Documentation only changes
- `test`: Adding or updating tests
- `refactor`: Code restructuring
- `perf`: Performance improvement
- `chore`: Maintenance tasks

**Examples:**

```bash
# New feature
git commit -m "feat(orchestrator): add support for Gemini 2.0 models"

# Bug fix
git commit -m "fix(client): handle connection timeout gracefully"

# Documentation
git commit -m "docs(api): add streaming examples for client"

# Breaking change
git commit -m "feat(agent)!: redesign skill registration API

BREAKING CHANGE: Skills now use new format. See migration guide."
```

---

## 🧪 Testing Requirements

All contributions must include tests:

### Unit Tests

```python
# tests/test_your_feature.py

import pytest
from synqed import YourFeature

class TestYourFeature:
    """Unit tests for YourFeature."""
    
    def test_feature_creation(self):
        """Test creating a feature instance."""
        feature = YourFeature(param="test")
        assert feature.param == "test"
    
    @pytest.mark.asyncio
    async def test_feature_execution(self):
        """Test feature execution."""
        feature = YourFeature(param="test")
        result = await feature.execute()
        assert result == "expected"
```

### Integration Tests

```python
# tests/test_integration.py

@pytest.mark.asyncio
async def test_feature_integration(running_server):
    """Test feature works with other components."""
    # Test implementation
    ...
```

### Coverage Requirements

- Minimum **90%** overall coverage
- All new code must be tested
- Check coverage: `pytest --cov=src/synqed --cov-report=term-missing`

---

## 📋 Pull Request Checklist

Before submitting your PR, ensure:

- [ ] Code follows style guide (PEP 8, formatted with Ruff)
- [ ] All tests pass (`pytest`)
- [ ] Coverage maintained at >90% (`pytest --cov`)
- [ ] No linting errors (`ruff check src/ tests/`)
- [ ] Code is properly formatted (`ruff format src/ tests/`)
- [ ] Type hints added for all public APIs
- [ ] Docstrings updated (Google style)
- [ ] **API documentation updated** (https://github.com/SynqLabs/synqed/tree/main/api/python)
- [ ] Examples added/updated in `examples/` directory
- [ ] CHANGELOG.md updated (if applicable)
- [ ] Breaking changes clearly documented
- [ ] PR description is clear and complete
- [ ] Commits follow conventional commits format

---

## 🤝 Code Review Process

1. **Submit PR** with clear description
2. **Automated checks** run (tests, linting, coverage)
3. **Documentation review** - Maintainers verify docs are updated
4. **Code review** - At least one maintainer reviews
5. **Address feedback** - Make requested changes
6. **Approval** - PR approved after all checks pass
7. **Merge** - Maintainer merges your contribution

**Review timeline:** Most PRs are reviewed within 2-3 business days.

---

## 🚫 What We Don't Accept

- PRs without documentation updates
- Code without tests
- Breaking changes without migration guide
- Hard-coded credentials or secrets
- Unclosed resources (memory leaks)
- Code that doesn't follow style guide
- Changes without clear purpose/benefit

---

## 💡 Good Contribution Examples

### Example 1: Adding a New Method

**Code:**
```python
# src/synqed/client.py

async def get_agent_status(self) -> dict:
    """
    Get the current status of the connected agent.
    
    Returns:
        dict: Agent status including health, uptime, and stats
        
    Example:
        >>> async with Client("http://localhost:8000") as client:
        ...     status = await client.get_agent_status()
        ...     print(status["health"])
    """
    response = await self._session.get(f"{self.agent_url}/status")
    return response.json()
```

**Test:**
```python
# tests/test_client.py

@pytest.mark.asyncio
async def test_get_agent_status(running_server):
    """Test getting agent status."""
    async with Client(running_server.url) as client:
        status = await client.get_agent_status()
        assert "health" in status
        assert status["health"] == "ok"
```

**Documentation:**
```markdown
# In API docs

## Getting Agent Status

Check the health and status of a connected agent:

\```python
from synqed import Client

async with Client("http://localhost:8000") as client:
    status = await client.get_agent_status()
    
    print(f"Health: {status['health']}")
    print(f"Uptime: {status['uptime']} seconds")
    print(f"Tasks processed: {status['tasks_processed']}")
\```
```

### Example 2: Bug Fix with Documentation

**Code Fix:**
```python
# src/synqed/server.py

async def stop(self) -> None:
    """Gracefully stop the server."""
    if self._server_task and not self._server_task.done():
        self._server_task.cancel()
        try:
            await self._server_task
        except asyncio.CancelledError:
            pass  # Expected during shutdown
        finally:
            await self._cleanup_resources()  # FIX: Ensure cleanup
```

**Documentation:**
```markdown
# In API docs - Troubleshooting section

### Server Shutdown Issues

**Fixed in v1.0.1:** Server resources are now properly cleaned up during shutdown.

If you're upgrading from v1.0.0:
- No code changes required
- Restart your agents to apply the fix
- Resource leaks during shutdown are now prevented
```

---

## 🌟 Recognition

Contributors who provide high-quality contributions with proper documentation will be:

- Listed in the CHANGELOG.md
- Credited in release notes
- Mentioned in project documentation
- Given contributor recognition in the repository

---

## 📞 Getting Help

- **Questions:** Open a GitHub Discussion or Issue
- **Documentation:** See [README.md](README.md) and [DEVELOPER_ONBOARDING.md](DEVELOPER_ONBOARDING.md)
- **Examples:** Check the `examples/` directory
- **Style Guide:** Run `ruff check` and `ruff format`

---

## 📄 License

By contributing to Synqed, you agree that your contributions will be licensed under the same license as the project.

---

**Thank you for contributing to Synqed! Your efforts help make multi-agent AI systems accessible to everyone.** 🚀

---

**Document Version:** 1.0  
**Last Updated:** November 2025  
**Maintainers:** Synq Team

