# -*- coding: UTF-8 -*-
# 缓存的数据
import json
import os
from functools import lru_cache
from typing import Dict, Any

import pandas as pd
from dateutil import parser
from pandas import DataFrame

import importlib
import sys
import inflection

# Try package-relative imports for normal package usage. If that fails
# (e.g., when running this file directly as a script), leave exchange
# and config as None for now and perform a local fallback only when
# executing the file as __main__ (see bottom of this file). This
# avoids mutating sys.path at import time.
try:
    from . import exchange, config
except Exception:
    exchange = None
    config = None


@lru_cache(maxsize=None)
def securities() -> pd.DataFrame:
    """
    证券列表
    """
    full_path = os.path.join(config.quant1x_config.meta_path, 'securities.csv')
    if not os.path.isfile(full_path):
        return pd.DataFrame(columns=['code', 'name'])
    # with open(full_path, 'rb') as f:
    #     f.seek(87287 - 13)
    #     print('Bytes around 85710:', f.read(19).hex(' '))
    df = pd.read_csv(full_path)
    # 转换为小写
    df.columns = df.columns.str.lower()
    # 兼容多种列名：优先匹配 'code' 和 'name'，否则尝试常见替代列名，最终回退到前两列
    cols = list(df.columns)
    code_candidates = ['code', 'symbol', 'securitycode', 'security_code', 'sec_code', 'sid']
    name_candidates = ['name', 'sec_name', 'security_name', 'secname', 'stock_name']

    def find_first(candidates, cols):
        for c in candidates:
            if c in cols:
                return c
        return None

    code_col = find_first(code_candidates, cols) or (cols[0] if cols else None)
    name_col = find_first(name_candidates, cols) or (cols[1] if len(cols) > 1 else (cols[0] if cols else None))

    if code_col is None or name_col is None:
        return pd.DataFrame(columns=['code', 'name'])

    # 返回时保持列名为标准名称 'code' 和 'name'，以便后续代码不变
    out = df[[code_col, name_col]].copy()
    out.columns = ['code', 'name']
    return out


@lru_cache(maxsize=None)
def block_list():
    """
    板块列表
    """
    df = securities()
    if df.empty or 'code' not in df.columns:
        return pd.DataFrame(columns=df.columns if not df.empty else ['code', 'name'])
    return df[df['code'].astype(str).str.startswith(('sh880', 'sh881'))]


def stock_name(code: str) -> str:
    corrected_symbol = exchange.correct_security_code(code)
    df = securities()
    if df.empty:
        return ""
    # ensure types align
    try:
        tmp = df[df['code'].astype(str) == str(corrected_symbol)]
    except Exception:
        tmp = df[df['code'] == corrected_symbol]
    if tmp.empty:
        return ""
    name = tmp['name'].iloc[0]
    return name


def klines(code: str, freq:str = 'D') -> pd.DataFrame | None:
    """
    获取缓存的日线数据
    """
    corrected_symbol = exchange.correct_security_code(code)
    suffix_length = 3  # 修正拼写并明确表示后缀长度
    freq_path = 'day' if freq == 'D' else freq
    symbol_directory = os.path.join(config.quant1x_config.data_path, freq_path, corrected_symbol[:-suffix_length])  # 更清晰表达目录用途
    file_extension = '.csv'
    filename = f"{corrected_symbol}{file_extension}"  # 使用f-string格式化
    full_path = os.path.join(symbol_directory, filename)

    if os.path.isfile(full_path):
        df = pd.read_csv(full_path)
        # 转换所有列名为 snake_case
        df.columns = [inflection.underscore(col) for col in df.columns]
        return df
    return None


def get_period_name(period: str = 'D') -> str:
    """
    根据周期标识返回中文名称

    Parameters:
    period (str): 周期标识 'W', 'M', 'Q', 'Y'

    Returns:
    str: 中文周期名称
    """
    period_names = {
        'W': '周',
        'M': '月',
        'Q': '季',
        'Y': '年',
        'D': '日'
    }
    period = period.upper()
    return period_names.get(period, period)


def convert_klines_trading(klines, period='D'):
    """
    基于实际交易日的K线转换函数

    Parameters:
    klines (pd.DataFrame): 日线数据
    period (str): 目标周期
        'W' - 周线
        'M' - 月线
        'Q' - 季度线
        'Y' - 年线

    Returns:
    pd.DataFrame: 转换后的K线数据，date字段表示实际交易日
    """
    if klines.empty:
        return klines.copy()

    df = klines.copy()
    df['date'] = pd.to_datetime(df['date'])
    df = df.sort_values('date').reset_index(drop=True)

    # 直接使用简化的周期标识
    period = period.upper()
    if period not in ['W', 'M', 'Q', 'Y']:
        return df

    # 根据周期分组
    groups = df['date'].dt.to_period(period)

    # 聚合数据，date字段保留实际的交易日
    result = df.groupby(groups).agg({
        'date': 'last',  # 实际最后一个交易日
        'open': 'first',
        'high': 'max',
        'low': 'min',
        'close': 'last',
        'volume': 'sum',
        'amount': 'sum'
    }).reset_index(drop=True)

    return result


# SectorFilename 板块缓存文件名
def sector_filename(date: str = '') -> str:
    """
    板块缓存文件名
    """
    name = 'blocks'
    cache_date = date.strip()
    if len(cache_date) == 0:
        cache_date = exchange.last_trade_date()
    filename = os.path.join(config.quant1x_config.meta_path, f'{name}.{cache_date}')
    if not os.path.isfile(filename):
        # fallback to blocks.csv
        filename = os.path.join(config.quant1x_config.meta_path, 'blocks.csv')
    return filename


@lru_cache(maxsize=None)
def get_sector_list() -> pd.DataFrame:
    """
    获取板块列表
    """
    sfn = sector_filename()
    try:
        df = pd.read_csv(sfn)
    except FileNotFoundError:
        # fallback: find latest blocks.* file under meta_path
        meta_dir = os.path.dirname(sfn)
        prefix = os.path.join(meta_dir, 'blocks.')
        candidates = [f for f in os.listdir(meta_dir) if f.startswith('blocks.')]
        if not candidates:
            raise
        # pick the latest by modification time
        candidates_full = [os.path.join(meta_dir, f) for f in candidates]
        candidates_full.sort(key=lambda p: os.path.getmtime(p), reverse=True)
        sfn2 = candidates_full[0]
        df = pd.read_csv(sfn2)
    # 补全sh前缀
    s = df['code'].astype(str)
    df['code'] = s.where(s.str.startswith('sh'), 'sh' + s)
    return df


def get_sector_constituents(code: str) -> list[str]:
    """
    获取板块成分股列表
    """
    code = code.strip()
    security_code = exchange.correct_security_code(code)
    df = get_sector_list().copy()
    cs = df[df['code'] == security_code]['ConstituentStocks']
    list = []
    if cs.empty:
        return list
    cs1 = cs.iloc[0]
    ConstituentStocks = json.loads(cs1)
    list = []
    for sc in ConstituentStocks:
        sc = sc.strip()
        sc = exchange.correct_security_code(sc)
        list.append(sc)
    return list


def date_format(date: str, layout: str = '%Y-%m-%d') -> str:
    dt = parser.parse(date)  # 自动识别各种常见日期格式
    return dt.strftime(layout)


@lru_cache(maxsize=None)
def get_minutes_data(code: str, date: str) -> DataFrame | None:
    """获取分时数据"""
    code = code.strip()
    corrected_symbol = exchange.correct_security_code(code)
    file_extension = '.csv'
    filename = f"{corrected_symbol}{file_extension}"  # 使用f-string格式化
    cache_date = date.strip()
    if len(cache_date) == 0:
        cache_date = exchange.last_trade_date()
    # 获取年份
    cache_date = date_format(cache_date, layout='%Y%m%d')
    year = cache_date[:4]
    base_path = os.path.join(config.quant1x_config.data_path, 'minutes')
    full_path = os.path.join(base_path, year, cache_date, filename)

    if os.path.isfile(full_path):
        return pd.read_csv(full_path)
    return None


@lru_cache(maxsize=None)
def cache_f10(date: str = None) -> DataFrame:
    factor_name = 'f10'
    trade_date = date or exchange.last_trade_date()
    file_extension = exchange.fix_trade_date(trade_date)
    filename = f"{factor_name}.{file_extension}"
    year = trade_date[:4]
    base_path = os.path.join(config.quant1x_config.data_path, 'flash')
    return pd.read_csv(os.path.join(base_path, year, filename))


def get_f10(code: str, date: str = None) -> Dict[str, Any]:
    """
    获取f10数据
    Args:
        code: 证券代码
        date: 日期

    Returns:
        返回f10数据
    """
    security_code = exchange.correct_security_code(code)
    df = cache_f10(date)
    result_df = df[df['Code'] == security_code]
    if result_df.empty:
        return {}

    # 取第一行，转为字典
    return result_df.iloc[0].to_dict()


@lru_cache(maxsize=None)
def get_tick_transaction(code: str, date: str) -> DataFrame | None:
    """获取分时"""
    code = code.strip()
    corrected_symbol = exchange.correct_security_code(code)
    file_extension = '.csv'
    filename = f"{corrected_symbol}{file_extension}"  # 使用f-string格式化
    cache_date = date.strip()
    if len(cache_date) == 0:
        cache_date = exchange.last_trade_date()
    # 获取年份
    cache_date = date_format(cache_date, layout='%Y%m%d')
    year = cache_date[:4]
    base_path = os.path.join(config.quant1x_config.data_path, 'trans')
    full_path = os.path.join(base_path, year, cache_date, filename)

    if os.path.isfile(full_path):
        df = pd.read_csv(full_path)
        df.columns = [col.lower() for col in df.columns]
        return df
    return None


if __name__ == '__main__':
    # If package-relative imports failed at module import time (e.g. when
    # running this file directly), perform a one-time fallback here so the
    # interactive script continues to work without mutating sys.path at
    # import time.
    if exchange is None or config is None:
        this_dir = os.path.dirname(__file__)
        pkg_parent = os.path.abspath(os.path.join(this_dir, '..'))
        if pkg_parent not in sys.path:
            sys.path.insert(0, pkg_parent)
        from quant1x import exchange as _exchange, config as _config
        globals()['exchange'] = _exchange
        globals()['config'] = _config

    print(config.get_quant1x_config_filename())
    print('data_path', config.quant1x_config.data_path)
    print('kline_path', config.quant1x_config.kline_path)
    code = '600600'
    df = klines(code)
    print(df)
    stock_name = stock_name(code)
    print(stock_name)
    security_list = securities()
    print(security_list)
    index_list = block_list()
    print(index_list)
    sfn = sector_filename()
    df = pd.read_csv(sfn)
    print(df)
    print(df['code'].dtype)
    df['code'] = 'sh' + df['code'].astype(str)
    s1 = df[df['code'] == 'sh881478']
    print(s1)

    l1 = get_sector_constituents('880675')
    print(l1)
    print(type(l1))

    df2 = get_minutes_data(code, date='2025-06-20')
    print(df2)
    df3 = get_tick_transaction(code, date='2025-06-20')
    print(df3)
    df4 = get_f10(code)
    print(df4)
