"""Workflow Object module."""

from typing import Dict, AnyStr, List
import msgpack
from omegaconf import OmegaConf, DictConfig

from flowcept.version import __version__
from flowcept.commons.utils import get_utc_now, get_git_info
from flowcept.configs import (
    settings,
    FLOWCEPT_USER,
    SYS_NAME,
    EXTRA_METADATA,
    ENVIRONMENT_ID,
    SETTINGS_PATH,
)


# Not a dataclass because a dataclass stores keys even when there's no value,
# adding unnecessary overhead.
class WorkflowObject:
    """Workflow object class.

    Represents metadata and provenance details for a workflow execution.
    """

    workflow_id: AnyStr = None
    """Unique identifier for the workflow."""

    parent_workflow_id: AnyStr = None
    """Identifier of the parent workflow, if this workflow is nested or derived."""

    machine_info: Dict = None
    """System or hardware information where the workflow is executed."""

    conf: Dict = None
    """Workflow configuration parameters, such as hyperparameters or runtime options."""

    flowcept_settings: Dict = None
    """Snapshot of Flowcept’s active settings used for this workflow."""

    flowcept_version: AnyStr = None
    """Version of Flowcept used during execution."""

    utc_timestamp: float = None
    """Timestamp (UTC, in seconds) when the workflow object was created."""

    user: AnyStr = None
    """User who launched or owns the workflow run."""

    campaign_id: AnyStr = None
    """Identifier for grouping workflows into a campaign or experiment."""

    adapter_id: AnyStr = None
    """Identifier of the adapter (e.g., Dask, MLflow) that triggered workflow capture."""

    interceptor_ids: List[AnyStr] = None
    """List of interceptors applied to this workflow (e.g., instrumentation, telemetry)."""

    name: AnyStr = None
    """Descriptive name for the workflow."""

    custom_metadata: Dict = None
    """User-defined metadata dictionary with additional annotations."""

    environment_id: str = None
    """Identifier for the runtime environment (e.g., conda env, container)."""

    sys_name: str = None
    """Logical system or facility name (e.g., HPC system name, cluster identifier)."""

    extra_metadata: str = None
    """Optional free-form metadata for extensions not covered by other fields."""

    used: Dict = None
    """Inputs consumed by the workflow (datasets, arguments, or configuration values)."""

    code_repository: Dict = None
    """Details of the code repository (URL, commit hash, branch) used to run the workflow."""

    generated: Dict = None
    """Outputs generated by the workflow (artifacts, models, or results)."""

    def __init__(self, workflow_id=None, name=None, used=None, generated=None):
        self.workflow_id = workflow_id
        self.name = name
        self.used = used
        self.generated = generated

    @staticmethod
    def workflow_id_field():
        """Get workflow id."""
        return "workflow_id"

    @staticmethod
    def from_dict(dict_obj: Dict) -> "WorkflowObject":
        """Convert from dictionary."""
        wf_obj = WorkflowObject()
        for k, v in dict_obj.items():
            setattr(wf_obj, k, v)
        return wf_obj

    def to_dict(self):
        """Convert to dictionary."""
        result_dict = {}
        for attr, value in self.__dict__.items():
            if value is not None:
                result_dict[attr] = value
        result_dict["type"] = "workflow"
        return result_dict

    def enrich(self, adapter_key=None):
        """Enrich it."""
        self.utc_timestamp = get_utc_now()
        self.flowcept_settings = OmegaConf.to_container(settings) if isinstance(settings, DictConfig) else settings
        self.conf = {"settings_path": SETTINGS_PATH}
        if adapter_key is not None:
            # TODO :base-interceptor-refactor: :code-reorg: :usability:
            # revisit all times we assume settings is not none
            self.adapter_id = adapter_key

        if self.user is None:
            self.user = FLOWCEPT_USER

        if self.environment_id is None and ENVIRONMENT_ID is not None:
            self.environment_id = ENVIRONMENT_ID

        if self.sys_name is None and SYS_NAME is not None:
            self.sys_name = SYS_NAME

        if self.extra_metadata is None and EXTRA_METADATA is not None:
            _extra_metadata = (
                OmegaConf.to_container(EXTRA_METADATA) if isinstance(EXTRA_METADATA, DictConfig) else EXTRA_METADATA
            )
            self.extra_metadata = _extra_metadata

        if self.code_repository is None:
            try:
                self.code_repository = get_git_info()
            except Exception as e:
                print(e)
                pass

        if self.flowcept_version is None:
            self.flowcept_version = __version__

    def serialize(self):
        """Serialize it."""
        return msgpack.dumps(self.to_dict())

    @staticmethod
    def deserialize(serialized_data) -> "WorkflowObject":
        """Deserialize it."""
        dict_obj = msgpack.loads(serialized_data)
        obj = WorkflowObject()
        for k, v in dict_obj.items():
            setattr(obj, k, v)
        return obj

    def __repr__(self):
        """Set the repr."""
        return (
            f"WorkflowObject("
            f"workflow_id={repr(self.workflow_id)}, "
            f"parent_workflow_id={repr(self.parent_workflow_id)}, "
            f"machine_info={repr(self.machine_info)}, "
            f"flowcept_settings={repr(self.flowcept_settings)}, "
            f"flowcept_version={repr(self.flowcept_version)}, "
            f"utc_timestamp={repr(self.utc_timestamp)}, "
            f"user={repr(self.user)}, "
            f"campaign_id={repr(self.campaign_id)}, "
            f"adapter_id={repr(self.adapter_id)}, "
            f"interceptor_ids={repr(self.interceptor_ids)}, "
            f"name={repr(self.name)}, "
            f"used={repr(self.used)}, "
            f"generated={repr(self.generated)}, "
            f"custom_metadata={repr(self.custom_metadata)})"
        )

    def __str__(self):
        """Set the string."""
        return self.__repr__()
