from enum import Enum
from typing import Optional, List, Dict, Any, Mapping
from deepmerge import always_merger
from pydantic import BaseModel, Field, StrictInt, StrictBool, ConfigDict, model_validator
from ruamel.yaml import YAML
from gdcmd.helm.values_parser import values_to_yaml

TOP_LEVEL_COMMENT: str = """###########################################################
This file is auto-generated by gdcmd. 
You should not edit it manually!
Go to platform/python/gdcmd/helm/values.py to change it.
###########################################################
TODO: Dont use secrets directly in values.yaml put them into secrets resource

"""


class Deploy(BaseModel):
    model_config = ConfigDict(extra="forbid")

    db: StrictBool = Field(default=True, description="Should Postgres be deployed inside the cluster alongside each app")
    sync: StrictBool = Field(default=True, description="Should SyncGrid app be deployed")
    link: StrictBool = Field(default=True, description="Should LinkGrid app be deployed")
    systemd: StrictBool = Field(default=False, description="Should Systemd service be deployed, only for local single-node deployments with podman kube play")
    keycloak: StrictBool = Field(default=True, description="Should one Keycloak instance for all apps be deployed, together with a default realm")
    prometheus: StrictBool = Field(default=True, description="Should Prometheus monitoring services be deployed alongside each app")
    loadBalancerServices: StrictBool = Field(default=True, description="Should LoadBalancer type services be created for apps, viable only for kubernetes deployments")


class Identity(BaseModel):
    model_config = ConfigDict(extra="forbid")

    authority: str = Field(default="http://keycloak-pod:8080/realms/platform", description="Url to identity provider, defaults to locally deployed Keycloak instance")


class RemoteWrite(BaseModel):
    model_config = ConfigDict(extra="forbid")

    enabled: StrictBool = Field(default=True, description="")
    url: str = Field(default="https://prometheus-prod-01-eu-west-0.grafana.net/api/prom/push", description="")
    username: str = Field(default="772948", description="")
    password: str = Field(default="eyJr...", description="")
    write_relabel_configs: Optional[List[Dict[str, Any]]] = []
    basic_auth: Optional[Dict[str, str]] = {}


class Prometheus(BaseModel):
    model_config = ConfigDict(extra="forbid")

    nodeExporter: StrictBool = Field(default=True, description="")
    databaseExporter: StrictBool = Field(default=True, description="")
    scrapeInterval: str = Field(default="30s", description="")
    evaluationInterval: str = Field(default="30m", description="")
    defaultGrafanaRemoteWrite: RemoteWrite = Field(default=RemoteWrite(), description="")
    externalLabels: Dict[str, str] = Field(default={"cluster": "cluster-name"}, description="")
    additionalRemoteWrites: List[RemoteWrite] = Field(default=[], description="""
For example:
- url: https://url.to.remote.write/api/v1/push
 write_relabel_configs:
   - source_labels: [ __name__ ]
     action: keep
     regex: ".*"
 basic_auth:
   username: username
   password: pass""")


class Systemd(BaseModel):
    model_config = ConfigDict(extra="forbid")

    deployFolder: str = Field(default="suite", description="This is folder on host from user home dir: e.g., \"suite\" for /home/<current-user>/suite")
    execCmd: str = Field(default="helm template -f values.yaml griddot/suite | podman kube play --replace --network suite -", description="")


class Postgres(BaseModel):
    model_config = ConfigDict(extra="forbid")

    username: str = Field(default="user", description="Username of internally or externally deployed Postgres instance")
    password: str = Field(default="pass", description="Password of internally or externally deployed Postgres instance")
    host: str | None = Field(default="db", description="Host of externally deployed Postgres instance, defaults to internal deployment instance")
    sharedBuffer: str | None = Field(default="10GB", description="")


class AppHost(str, Enum):
    sync = "sync-pod"
    link = "link-pod"
    keycloak = "keycloak-pod"


class Keycloak(BaseModel):
    model_config = ConfigDict(extra="forbid", use_enum_values=True)

    adminUser: str = Field(default="admin", description="")
    adminPass: str = Field(default="admin", description="")
    adminEmail: str = Field(default="admin@mail.com", description="")
    postgres: Postgres = Field(default=Postgres(sharedBuffer=None, host=None), description="")
    host: AppHost = Field(default=AppHost.keycloak, description="Keycloak hostname, where it can be reached; NOTE: This cannot be changed! For info only")
    port: StrictInt = Field(default=8080, ge=1, le=65535, description="Change Keycloak host port if needed")


class Image(BaseModel):
    model_config = ConfigDict(extra="forbid")

    registry: str = Field(default="registry.gitlab.com/griddot/syncgrid", description="Point to your private registry where you pull the images")
    version: str = Field(default="v2025.1.745", description="")
    pullUser: str = Field(default="user", description="Username for private registry")
    pullPassword: str = Field(default="glpat-12345", description="Access token or password for private registry")


class App(BaseModel):
    model_config = ConfigDict(extra="forbid", use_enum_values=True)

    host: AppHost = Field(default=AppHost(AppHost.sync), description="App hostname, where it can be reached; NOTE: This cannot be changed! For info only")
    hostPortHttp: StrictInt = Field(default=1000, ge=1, le=65535, description="Change app host port for HTTP if needed")
    hostPortHttps: StrictInt = Field(default=0, ge=0, le=65535, description="Change app host port for HTTPS if needed. NOTE: AppCommon.requireHttps must be set to true to enable this port")
    licenceBase64: Optional[str] = Field(default="", description="Base64 encoded licence string, for development purposes only")


class Sync(BaseModel):
    model_config = ConfigDict(extra="forbid")

    image: Image = Field(default=Image(registry="registry.gitlab.com/griddot/syncgrid", version="v2025.1.745"), description="")
    postgres: Postgres = Field(default=Postgres(), description="")
    app: App = Field(default=App(hostPortHttp=7080, host=AppHost.sync), description="")


class Import(BaseModel):
    model_config = ConfigDict(extra="forbid")

    enabled: StrictBool = Field(default=False, description="")
    username: str = Field(default="admin", description="")
    password: str = Field(default="admin", description="")
    datasetPath: str = Field(default="path", description="")


class KeyManagement(BaseModel):
    model_config = ConfigDict(extra="forbid")

    enabled: StrictBool = Field(default=True, description="")
    encryptionKey: str = Field(default="f5be9eb6d049882cba5ef743f98fbce6a96b2a30f063a516bb2f09ab02ec441c", description="")


class Link(BaseModel):
    model_config = ConfigDict(extra="forbid")

    image: Image = Field(default=Image(registry="registry.gitlab.com/griddot/linkgrid", version="v2024.1.10"), description="")
    postgres: Postgres = Field(default=Postgres(), description="")
    app: App = Field(default=App(hostPortHttp=4080, host=AppHost.link), description="")
    keyManagement: KeyManagement = Field(default=KeyManagement(), description="")
    import_: Import = Field(default=Import(), alias="import", description="")


class AppCommon(BaseModel):
    model_config = ConfigDict(extra="forbid")

    requireHttps: StrictBool = Field(default=False, description="Set to true if you want to enforce HTTPS with self-signed certificates")
    identity: Identity = Field(default=Identity(), description="")
    certificatePassword: str = Field(default="z98M9gFsF7aJBn3s", description="Certificate password for self-signed certificates, used if AppCommon.requireHttps is true")


class ValuesYaml(BaseModel):
    model_config = ConfigDict(extra="forbid", populate_by_name=True, use_enum_values=True)

    deploy: Deploy = Field(default=Deploy(), description="Deployment configuration")
    appCommon: AppCommon = Field(default=AppCommon(), description="\nCommon configuration for all apps")
    sync: Sync = Field(default=Sync(), description="\nSyncGrid configuration")
    link: Link = Field(default=Link(), description="\nLinkGrid configuration")
    prometheus: Prometheus = Field(default=Prometheus(), description="\nPrometheus configuration")
    systemd: Systemd = Field(default=Systemd(), description="\nSystemd configuration")
    keycloak: Keycloak = Field(default=Keycloak(), description="\nKeycloak configuration")

    @model_validator(mode="after")
    def validate_deployment_consistency(self) -> "ValuesYaml":
        """Validate that service configurations exist when deployment is enabled"""
        if self.deploy.prometheus and not self.prometheus:
            raise ValueError("Prometheus configuration required when deploy.prometheus is true")
        if self.deploy.keycloak and not self.keycloak:
            raise ValueError("Keycloak configuration required when deploy.keycloak is true")
        if self.deploy.sync and not self.sync:
            raise ValueError("SyncGrid configuration required when deploy.sync is true")
        if self.deploy.link and not self.link:
            raise ValueError("LinkGrid configuration required when deploy.link is true")
        if self.deploy.systemd and not self.systemd:
            raise ValueError("Systemd configuration required when deploy.systemd is true")

        # Prometheus
        if self.prometheus.databaseExporter and not self.deploy.db:
            raise ValueError("deploy.db must be true to enable Prometheus databaseExporter")

        # App https port
        if self.sync.app.hostPortHttps > 0 and not self.appCommon.requireHttps:
            raise ValueError("appCommon.requireHttps must be true to enable SyncGrid HTTPS port")
        if self.link.app.hostPortHttps > 0 and not self.appCommon.requireHttps:
            raise ValueError("appCommon.requireHttps must be true to enable LinkGrid HTTPS port")
        return self

    @classmethod
    def from_yaml(cls, yaml_str: str) -> "ValuesYaml":
        data = YAML(typ='safe').load(yaml_str)
        return ValuesYaml.model_validate(data)

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> "ValuesYaml":
        return cls.model_validate(data)

    def to_str(self, exclude_none=True, exclude_defaults: bool = False, exclude_unset: bool = False, include: Mapping[str, Any] | set[str] | None = None, exclude: Mapping[str, Any] | set[str] | None = None) -> str:
        yml = YAML()
        yml.representer.add_representer(AppHost, lambda dump, d: dump.represent_scalar("tag:yaml.org,2002:str", str(d)))
        return values_to_yaml(self, TOP_LEVEL_COMMENT, yml, exclude_none, exclude_defaults, exclude_unset, include, exclude)

    def to_dict(self) -> Dict[str, Any]:
        return self.model_dump(by_alias=True)

    def merge(self, other: "ValuesYaml") -> "ValuesYaml":
        return always_merger.merge(self, other)

    def merge_dict(self, other: Dict[str, Any]) -> "ValuesYaml":
        return ValuesYaml.from_dict(always_merger.merge(self.to_dict(), other))


if __name__ == "__main__":
    # values.yaml
    values = ValuesYaml()
    with open("../../../charts/suite/values.yaml", "w") as f:
        f.write(values.to_str())

    # values.landis-gyr.yaml
    TOP_LEVEL_COMMENT = ""
    landis = ValuesYaml(sync=Sync(image=Image(registry="registry.gitlab.com/griddot/resellers/landis-gyr/syncgrid")))
    with open("../../../charts/suite/values.landis-gyr.yaml", "w") as f:
        f.write(landis.to_str(exclude={
            "link": ...,
            "prometheus": ...,
            "keycloak": ...,
            "deploy": {
                "link"
            },
        }))
