"""LIFX product definitions and capability detection.

DO NOT EDIT THIS FILE MANUALLY.
Generated from https://github.com/LIFX/products/blob/master/products.json
by products/generator.py

This module provides pre-generated product information for efficient runtime lookups.
"""

from __future__ import annotations

from dataclasses import dataclass
from enum import IntEnum


class ProductCapability(IntEnum):
    """Product capability flags."""

    COLOR = 1
    INFRARED = 2
    MULTIZONE = 4
    CHAIN = 8
    MATRIX = 16
    RELAYS = 32
    BUTTONS = 64
    HEV = 128
    EXTENDED_MULTIZONE = 256


@dataclass
class TemperatureRange:
    """Color temperature range in Kelvin."""

    min: int
    max: int


@dataclass
class ProductInfo:
    """Information about a LIFX product.

    Attributes:
        pid: Product ID
        name: Product name
        vendor: Vendor ID (always 1 for LIFX)
        capabilities: Bitfield of ProductCapability flags
        temperature_range: Min/max color temperature in Kelvin
        min_ext_mz_firmware: Minimum firmware version for extended multizone
    """

    pid: int
    name: str
    vendor: int
    capabilities: int
    temperature_range: TemperatureRange | None
    min_ext_mz_firmware: int | None

    def has_capability(self, capability: ProductCapability) -> bool:
        """Check if product has a specific capability.

        Args:
            capability: Capability to check

        Returns:
            True if product has the capability
        """
        return bool(self.capabilities & capability)

    @property
    def has_color(self) -> bool:
        """Check if product supports color."""
        return self.has_capability(ProductCapability.COLOR)

    @property
    def has_infrared(self) -> bool:
        """Check if product supports infrared."""
        return self.has_capability(ProductCapability.INFRARED)

    @property
    def has_multizone(self) -> bool:
        """Check if product supports multizone."""
        return self.has_capability(ProductCapability.MULTIZONE)

    @property
    def has_chain(self) -> bool:
        """Check if product supports chaining."""
        return self.has_capability(ProductCapability.CHAIN)

    @property
    def has_matrix(self) -> bool:
        """Check if product supports matrix (2D grid)."""
        return self.has_capability(ProductCapability.MATRIX)

    @property
    def has_relays(self) -> bool:
        """Check if product has relays."""
        return self.has_capability(ProductCapability.RELAYS)

    @property
    def has_buttons(self) -> bool:
        """Check if product has buttons."""
        return self.has_capability(ProductCapability.BUTTONS)

    @property
    def has_hev(self) -> bool:
        """Check if product supports HEV."""
        return self.has_capability(ProductCapability.HEV)

    @property
    def has_extended_multizone(self) -> bool:
        """Check if product supports extended multizone."""
        return self.has_capability(ProductCapability.EXTENDED_MULTIZONE)

    def supports_extended_multizone(self, firmware_version: int | None = None) -> bool:
        """Check if extended multizone is supported for given firmware version.

        Args:
            firmware_version: Firmware version to check (optional)

        Returns:
            True if extended multizone is supported
        """
        if not self.has_extended_multizone:
            return False
        if self.min_ext_mz_firmware is None:
            return True
        if firmware_version is None:
            return True
        return firmware_version >= self.min_ext_mz_firmware


# Pre-generated product definitions
PRODUCTS: dict[int, ProductInfo] = {
    1: ProductInfo(
        pid=1,
        name="LIFX Original 1000",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    3: ProductInfo(
        pid=3,
        name="LIFX Color 650",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    10: ProductInfo(
        pid=10,
        name="LIFX White 800 (Low Voltage)",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2700, max=6500),
        min_ext_mz_firmware=None,
    ),
    11: ProductInfo(
        pid=11,
        name="LIFX White 800 (High Voltage)",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2700, max=6500),
        min_ext_mz_firmware=None,
    ),
    15: ProductInfo(
        pid=15,
        name="LIFX Color 1000",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    18: ProductInfo(
        pid=18,
        name="LIFX White 900 BR30 (Low Voltage)",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    19: ProductInfo(
        pid=19,
        name="LIFX White 900 BR30 (High Voltage)",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    20: ProductInfo(
        pid=20,
        name="LIFX Color 1000 BR30",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    22: ProductInfo(
        pid=22,
        name="LIFX Color 1000",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    27: ProductInfo(
        pid=27,
        name="LIFX A19",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    28: ProductInfo(
        pid=28,
        name="LIFX BR30",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    29: ProductInfo(
        pid=29,
        name="LIFX A19 Night Vision",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.INFRARED,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    30: ProductInfo(
        pid=30,
        name="LIFX BR30 Night Vision",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.INFRARED,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    31: ProductInfo(
        pid=31,
        name="LIFX Z",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MULTIZONE,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    32: ProductInfo(
        pid=32,
        name="LIFX Z",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=131149,
    ),
    36: ProductInfo(
        pid=36,
        name="LIFX Downlight",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    37: ProductInfo(
        pid=37,
        name="LIFX Downlight",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    38: ProductInfo(
        pid=38,
        name="LIFX Beam",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=131149,
    ),
    39: ProductInfo(
        pid=39,
        name="LIFX Downlight White to Warm",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    40: ProductInfo(
        pid=40,
        name="LIFX Downlight",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    43: ProductInfo(
        pid=43,
        name="LIFX A19",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    44: ProductInfo(
        pid=44,
        name="LIFX BR30",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    45: ProductInfo(
        pid=45,
        name="LIFX A19 Night Vision",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.INFRARED,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    46: ProductInfo(
        pid=46,
        name="LIFX BR30 Night Vision",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.INFRARED,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    49: ProductInfo(
        pid=49,
        name="LIFX Mini Color",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    50: ProductInfo(
        pid=50,
        name="LIFX Mini White to Warm",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=1500, max=6500),
        min_ext_mz_firmware=None,
    ),
    51: ProductInfo(
        pid=51,
        name="LIFX Mini White",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2700, max=2700),
        min_ext_mz_firmware=None,
    ),
    52: ProductInfo(
        pid=52,
        name="LIFX GU10",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    53: ProductInfo(
        pid=53,
        name="LIFX GU10",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    55: ProductInfo(
        pid=55,
        name="LIFX Tile",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.CHAIN
        | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=2500, max=9000),
        min_ext_mz_firmware=None,
    ),
    57: ProductInfo(
        pid=57,
        name="LIFX Candle",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    59: ProductInfo(
        pid=59,
        name="LIFX Mini Color",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    60: ProductInfo(
        pid=60,
        name="LIFX Mini White to Warm",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=1500, max=6500),
        min_ext_mz_firmware=None,
    ),
    61: ProductInfo(
        pid=61,
        name="LIFX Mini White",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2700, max=2700),
        min_ext_mz_firmware=None,
    ),
    62: ProductInfo(
        pid=62,
        name="LIFX A19",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    63: ProductInfo(
        pid=63,
        name="LIFX BR30",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    64: ProductInfo(
        pid=64,
        name="LIFX A19 Night Vision",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.INFRARED,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    65: ProductInfo(
        pid=65,
        name="LIFX BR30 Night Vision",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.INFRARED,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    66: ProductInfo(
        pid=66,
        name="LIFX Mini White",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2700, max=2700),
        min_ext_mz_firmware=None,
    ),
    68: ProductInfo(
        pid=68,
        name="LIFX Candle",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    70: ProductInfo(
        pid=70,
        name="LIFX Switch",
        vendor=1,
        capabilities=ProductCapability.RELAYS | ProductCapability.BUTTONS,
        temperature_range=None,
        min_ext_mz_firmware=None,
    ),
    71: ProductInfo(
        pid=71,
        name="LIFX Switch",
        vendor=1,
        capabilities=ProductCapability.RELAYS | ProductCapability.BUTTONS,
        temperature_range=None,
        min_ext_mz_firmware=None,
    ),
    81: ProductInfo(
        pid=81,
        name="LIFX Candle White to Warm",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2200, max=6500),
        min_ext_mz_firmware=None,
    ),
    82: ProductInfo(
        pid=82,
        name="LIFX Filament Clear",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2100, max=2100),
        min_ext_mz_firmware=None,
    ),
    85: ProductInfo(
        pid=85,
        name="LIFX Filament Amber",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2000, max=2000),
        min_ext_mz_firmware=None,
    ),
    87: ProductInfo(
        pid=87,
        name="LIFX Mini White",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2700, max=2700),
        min_ext_mz_firmware=None,
    ),
    88: ProductInfo(
        pid=88,
        name="LIFX Mini White",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2700, max=2700),
        min_ext_mz_firmware=None,
    ),
    89: ProductInfo(
        pid=89,
        name="LIFX Switch",
        vendor=1,
        capabilities=ProductCapability.RELAYS | ProductCapability.BUTTONS,
        temperature_range=None,
        min_ext_mz_firmware=None,
    ),
    90: ProductInfo(
        pid=90,
        name="LIFX Clean",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.HEV,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    91: ProductInfo(
        pid=91,
        name="LIFX Color",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    92: ProductInfo(
        pid=92,
        name="LIFX Color",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    93: ProductInfo(
        pid=93,
        name="LIFX A19 US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    94: ProductInfo(
        pid=94,
        name="LIFX BR30",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    96: ProductInfo(
        pid=96,
        name="LIFX Candle White to Warm",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2200, max=6500),
        min_ext_mz_firmware=None,
    ),
    97: ProductInfo(
        pid=97,
        name="LIFX A19",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    98: ProductInfo(
        pid=98,
        name="LIFX BR30",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    99: ProductInfo(
        pid=99,
        name="LIFX Clean",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.HEV,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    100: ProductInfo(
        pid=100,
        name="LIFX Filament Clear",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2100, max=2100),
        min_ext_mz_firmware=None,
    ),
    101: ProductInfo(
        pid=101,
        name="LIFX Filament Amber",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2000, max=2000),
        min_ext_mz_firmware=None,
    ),
    109: ProductInfo(
        pid=109,
        name="LIFX A19 Night Vision",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.INFRARED,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    110: ProductInfo(
        pid=110,
        name="LIFX BR30 Night Vision",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.INFRARED,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    111: ProductInfo(
        pid=111,
        name="LIFX A19 Night Vision",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.INFRARED,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    112: ProductInfo(
        pid=112,
        name="LIFX BR30 Night Vision Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.INFRARED,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    113: ProductInfo(
        pid=113,
        name="LIFX Mini WW US",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    114: ProductInfo(
        pid=114,
        name="LIFX Mini WW Intl",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    115: ProductInfo(
        pid=115,
        name="LIFX Switch",
        vendor=1,
        capabilities=ProductCapability.RELAYS | ProductCapability.BUTTONS,
        temperature_range=None,
        min_ext_mz_firmware=None,
    ),
    116: ProductInfo(
        pid=116,
        name="LIFX Switch",
        vendor=1,
        capabilities=ProductCapability.RELAYS | ProductCapability.BUTTONS,
        temperature_range=None,
        min_ext_mz_firmware=None,
    ),
    117: ProductInfo(
        pid=117,
        name="LIFX Z US",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    118: ProductInfo(
        pid=118,
        name="LIFX Z Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    119: ProductInfo(
        pid=119,
        name="LIFX Beam US",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    120: ProductInfo(
        pid=120,
        name="LIFX Beam Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    121: ProductInfo(
        pid=121,
        name="LIFX Downlight Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    122: ProductInfo(
        pid=122,
        name="LIFX Downlight US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    123: ProductInfo(
        pid=123,
        name="LIFX Color US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    124: ProductInfo(
        pid=124,
        name="LIFX Colour Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    125: ProductInfo(
        pid=125,
        name="LIFX White to Warm US",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    126: ProductInfo(
        pid=126,
        name="LIFX White to Warm Intl",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    127: ProductInfo(
        pid=127,
        name="LIFX White US",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2700, max=2700),
        min_ext_mz_firmware=None,
    ),
    128: ProductInfo(
        pid=128,
        name="LIFX White Intl",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2700, max=2700),
        min_ext_mz_firmware=None,
    ),
    129: ProductInfo(
        pid=129,
        name="LIFX Color US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    130: ProductInfo(
        pid=130,
        name="LIFX Colour Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    131: ProductInfo(
        pid=131,
        name="LIFX White To Warm US",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    132: ProductInfo(
        pid=132,
        name="LIFX White To Warm Intl",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    133: ProductInfo(
        pid=133,
        name="LIFX White US",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2700, max=2700),
        min_ext_mz_firmware=None,
    ),
    134: ProductInfo(
        pid=134,
        name="LIFX White Intl",
        vendor=1,
        capabilities=0,
        temperature_range=TemperatureRange(min=2700, max=2700),
        min_ext_mz_firmware=None,
    ),
    135: ProductInfo(
        pid=135,
        name="LIFX GU10 Color US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    136: ProductInfo(
        pid=136,
        name="LIFX GU10 Colour Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    137: ProductInfo(
        pid=137,
        name="LIFX Candle Color US",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    138: ProductInfo(
        pid=138,
        name="LIFX Candle Colour Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    141: ProductInfo(
        pid=141,
        name="LIFX Neon US",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    142: ProductInfo(
        pid=142,
        name="LIFX Neon Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    143: ProductInfo(
        pid=143,
        name="LIFX String US",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    144: ProductInfo(
        pid=144,
        name="LIFX String Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    161: ProductInfo(
        pid=161,
        name="LIFX Outdoor Neon US",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    162: ProductInfo(
        pid=162,
        name="LIFX Outdoor Neon Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    163: ProductInfo(
        pid=163,
        name="LIFX A19 US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    164: ProductInfo(
        pid=164,
        name="LIFX BR30 US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    165: ProductInfo(
        pid=165,
        name="LIFX A19 Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    166: ProductInfo(
        pid=166,
        name="LIFX BR30 Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    167: ProductInfo(
        pid=167,
        name="LIFX Downlight",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    168: ProductInfo(
        pid=168,
        name="LIFX Downlight",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    169: ProductInfo(
        pid=169,
        name="LIFX A21 1600lm US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    170: ProductInfo(
        pid=170,
        name="LIFX A21 1600lm Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    171: ProductInfo(
        pid=171,
        name="LIFX Round Spot US",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    173: ProductInfo(
        pid=173,
        name="LIFX Round Path US",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    174: ProductInfo(
        pid=174,
        name="LIFX Square Path US",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    175: ProductInfo(
        pid=175,
        name="LIFX PAR38 US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    176: ProductInfo(
        pid=176,
        name="LIFX Ceiling US",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    177: ProductInfo(
        pid=177,
        name="LIFX Ceiling Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    178: ProductInfo(
        pid=178,
        name="LIFX Downlight US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    179: ProductInfo(
        pid=179,
        name="LIFX Downlight US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    180: ProductInfo(
        pid=180,
        name="LIFX Downlight US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    181: ProductInfo(
        pid=181,
        name="LIFX Color US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    182: ProductInfo(
        pid=182,
        name="LIFX Colour Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    185: ProductInfo(
        pid=185,
        name="LIFX Candle Color US",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    186: ProductInfo(
        pid=186,
        name="LIFX Candle Colour Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    187: ProductInfo(
        pid=187,
        name="LIFX Candle Color US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    188: ProductInfo(
        pid=188,
        name="LIFX Candle Colour Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    201: ProductInfo(
        pid=201,
        name='LIFX Ceiling 13x26" US',
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    202: ProductInfo(
        pid=202,
        name='LIFX Ceiling 13x26" Intl',
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    203: ProductInfo(
        pid=203,
        name="LIFX String US",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    204: ProductInfo(
        pid=204,
        name="LIFX String Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    205: ProductInfo(
        pid=205,
        name="LIFX Indoor Neon US",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    206: ProductInfo(
        pid=206,
        name="LIFX Indoor Neon Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    213: ProductInfo(
        pid=213,
        name="LIFX Permanent Outdoor US",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    214: ProductInfo(
        pid=214,
        name="LIFX Permanent Outdoor Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MULTIZONE
        | ProductCapability.EXTENDED_MULTIZONE,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    215: ProductInfo(
        pid=215,
        name="LIFX Candle Color US",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    216: ProductInfo(
        pid=216,
        name="LIFX Candle Colour Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    217: ProductInfo(
        pid=217,
        name="LIFX Tube US",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    218: ProductInfo(
        pid=218,
        name="LIFX Tube Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    219: ProductInfo(
        pid=219,
        name="LIFX Luna US",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MATRIX
        | ProductCapability.BUTTONS,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    220: ProductInfo(
        pid=220,
        name="LIFX Luna Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR
        | ProductCapability.MATRIX
        | ProductCapability.BUTTONS,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    221: ProductInfo(
        pid=221,
        name="LIFX Round Spot Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    222: ProductInfo(
        pid=222,
        name="LIFX Round Path Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR | ProductCapability.MATRIX,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    223: ProductInfo(
        pid=223,
        name="LIFX Downlight US",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    224: ProductInfo(
        pid=224,
        name="LIFX Downlight Intl",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
    225: ProductInfo(
        pid=225,
        name="LIFX PAR38 INTL",
        vendor=1,
        capabilities=ProductCapability.COLOR,
        temperature_range=TemperatureRange(min=1500, max=9000),
        min_ext_mz_firmware=None,
    ),
}


class ProductRegistry:
    """Registry of LIFX products and their capabilities."""

    def __init__(self) -> None:
        """Initialize product registry with pre-generated data."""
        self._products = PRODUCTS.copy()  # Copy to allow test overrides
        self._loaded = True  # Always loaded in generated registry

    def load_from_dict(self, data: dict | list) -> None:
        """Load products from parsed JSON data (for testing).

        Args:
            data: Parsed products.json dictionary or array
        """
        from typing import Any

        # Clear existing products
        self._products.clear()

        # Handle both array and object formats
        all_vendors = []
        if isinstance(data, list):
            all_vendors = data
        else:
            all_vendors = [data]

        # Process each vendor
        for vendor_data in all_vendors:
            vendor_id = vendor_data.get("vid", 1)
            defaults = vendor_data.get("defaults", {})
            default_features = defaults.get("features", {})

            # Parse each product
            for product in vendor_data.get("products", []):
                pid = product["pid"]
                name = product["name"]

                # Merge features with defaults
                features: dict[str, Any] = {
                    **default_features,
                    **product.get("features", {}),
                }

                # Build capabilities bitfield
                capabilities = 0
                if features.get("color"):
                    capabilities |= ProductCapability.COLOR
                if features.get("infrared"):
                    capabilities |= ProductCapability.INFRARED
                if features.get("multizone"):
                    capabilities |= ProductCapability.MULTIZONE
                if features.get("chain"):
                    capabilities |= ProductCapability.CHAIN
                if features.get("matrix"):
                    capabilities |= ProductCapability.MATRIX
                if features.get("relays"):
                    capabilities |= ProductCapability.RELAYS
                if features.get("buttons"):
                    capabilities |= ProductCapability.BUTTONS
                if features.get("hev"):
                    capabilities |= ProductCapability.HEV

                # Check for extended multizone in upgrades
                min_ext_mz_firmware = None
                for upgrade in product.get("upgrades", []):
                    if upgrade.get("features", {}).get("extended_multizone"):
                        capabilities |= ProductCapability.EXTENDED_MULTIZONE
                        # Parse firmware version (major.minor format)
                        major = upgrade.get("major", 0)
                        minor = upgrade.get("minor", 0)
                        min_ext_mz_firmware = (major << 16) | minor
                        break

                # Parse temperature range
                temp_range = None
                if "temperature_range" in features:
                    temp_list = features["temperature_range"]
                    if len(temp_list) >= 2:
                        temp_range = TemperatureRange(
                            min=temp_list[0], max=temp_list[1]
                        )

                product_info = ProductInfo(
                    pid=pid,
                    name=name,
                    vendor=vendor_id,
                    capabilities=capabilities,
                    temperature_range=temp_range,
                    min_ext_mz_firmware=min_ext_mz_firmware,
                )

                self._products[pid] = product_info

        self._loaded = True

    @property
    def is_loaded(self) -> bool:
        """Check if registry has been loaded."""
        return self._loaded

    def get_product(self, pid: int) -> ProductInfo | None:
        """Get product info by product ID.

        Args:
            pid: Product ID

        Returns:
            ProductInfo if found, None otherwise
        """
        return self._products.get(pid)

    def get_device_class_name(
        self, pid: int, firmware_version: int | None = None
    ) -> str:
        """Get appropriate device class name for a product.

        Args:
            pid: Product ID
            firmware_version: Firmware version (optional)

        Returns:
            Device class name: "TileDevice", "MultiZoneLight", "HevLight", "InfraredLight", "Light", or "Device"
        """
        product = self.get_product(pid)
        if product is None:
            # Unknown product - default to Light if we don't know
            return "Light"

        # Matrix devices (Tiles, Candles) → TileDevice
        if product.has_matrix:
            return "TileDevice"

        # MultiZone devices (Strips, Beams) → MultiZoneLight
        if product.has_multizone:
            return "MultiZoneLight"

        # HEV lights → HevLight
        if product.has_hev:
            return "HevLight"

        # Infrared lights → InfraredLight
        if product.has_infrared:
            return "InfraredLight"

        # Color lights → Light
        if product.has_color:
            return "Light"

        # Devices with relays (switches/relays) → Device
        if product.has_relays:
            return "Device"

        # Devices with buttons but no color (switches) → Device
        if product.has_buttons:
            return "Device"

        # Everything else (basic lights, white-to-warm lights) → Light
        # These have no special capabilities but still support Light protocol
        return "Light"

    def __len__(self) -> int:
        """Get number of products in registry."""
        return len(self._products)

    def __contains__(self, pid: int) -> bool:
        """Check if product ID exists in registry."""
        return pid in self._products


# Global registry instance
_registry = ProductRegistry()


def get_registry() -> ProductRegistry:
    """Get the global product registry.

    Returns:
        Global ProductRegistry instance
    """
    return _registry


def get_product(pid: int) -> ProductInfo | None:
    """Get product info by product ID.

    Args:
        pid: Product ID

    Returns:
        ProductInfo if found, None otherwise
    """
    return _registry.get_product(pid)


def get_device_class_name(pid: int, firmware_version: int | None = None) -> str:
    """Get appropriate device class name for a product.

    Args:
        pid: Product ID
        firmware_version: Firmware version (optional)

    Returns:
        Device class name: "TileDevice", "MultiZoneLight", "Light", or "Device"
    """
    return _registry.get_device_class_name(pid, firmware_version)
