# -*- coding: utf-8 -*-
"""
World Trade Analysis module for Fincept Terminal
Updated to use centralized logging system
"""


import dearpygui.dearpygui as dpg
from fincept_terminal.Utils.base_tab import BaseTab
import random
import datetime

from fincept_terminal.Utils.Logging.logger import logger, operation

class WorldTradeAnalysisTab(BaseTab):
    """Bloomberg Terminal World Trade Analysis & Research Tab"""

    def __init__(self, main_app=None):
        super().__init__(main_app)
        self.main_app = main_app

        # Bloomberg color scheme
        self.BLOOMBERG_ORANGE = [255, 165, 0]
        self.BLOOMBERG_WHITE = [255, 255, 255]
        self.BLOOMBERG_RED = [255, 0, 0]
        self.BLOOMBERG_GREEN = [0, 200, 0]
        self.BLOOMBERG_YELLOW = [255, 255, 0]
        self.BLOOMBERG_GRAY = [120, 120, 120]
        self.BLOOMBERG_BLUE = [100, 150, 255]

        # Initialize comprehensive trade data
        logger.info("Initializing World Trade Analysis Tab")
        self.initialize_trade_data()

    def get_label(self):
        return " World Trade Analysis"

    def initialize_trade_data(self):
        """Initialize comprehensive world trade data"""
        with operation("initialize_trade_data"):
            # Major trading nations
            self.major_economies = {
                "United States": {"exports": 1645.5, "imports": 2407.5, "balance": -762.0, "gdp_share": 10.8,
                                  "growth": 2.1},
                "China": {"exports": 3363.8, "imports": 2601.2, "balance": 762.6, "gdp_share": 13.4, "growth": 5.2},
                "Germany": {"exports": 1753.9, "imports": 1571.9, "balance": 182.0, "gdp_share": 46.1, "growth": 0.3},
                "Japan": {"exports": 705.8, "imports": 720.9, "balance": -15.1, "gdp_share": 16.0, "growth": 1.8},
                "United Kingdom": {"exports": 468.2, "imports": 521.4, "balance": -53.2, "gdp_share": 17.5, "growth": 2.0},
                "France": {"exports": 625.1, "imports": 591.7, "balance": 33.4, "gdp_share": 24.8, "growth": 0.7},
                "Netherlands": {"exports": 720.1, "imports": 649.9, "balance": 70.2, "gdp_share": 83.2, "growth": 2.6},
                "South Korea": {"exports": 644.4, "imports": 615.0, "balance": 29.4, "gdp_share": 37.8, "growth": 3.1},
                "Italy": {"exports": 542.3, "imports": 473.1, "balance": 69.2, "gdp_share": 28.9, "growth": 0.9},
                "Canada": {"exports": 504.8, "imports": 505.4, "balance": -0.6, "gdp_share": 30.7, "growth": 1.5}
            }

            # Trade commodities
            self.commodities = {
                "Crude Oil": {"volume": 1.89, "value": 846.2, "price": 78.35, "change": -1.25, "unit": "trillion USD"},
                "Natural Gas": {"volume": 1.32, "value": 421.8, "price": 3.21, "change": 2.14, "unit": "trillion USD"},
                "Gold": {"volume": 0.34, "value": 348.7, "price": 2312.80, "change": 15.60, "unit": "billion USD"},
                "Iron Ore": {"volume": 1.68, "value": 189.4, "price": 112.50, "change": -0.85, "unit": "billion USD"},
                "Wheat": {"volume": 0.19, "value": 78.9, "price": 425.75, "change": 3.22, "unit": "billion USD"},
                "Soybeans": {"volume": 0.15, "value": 65.4, "price": 1456.25, "change": -2.18, "unit": "billion USD"},
                "Copper": {"volume": 0.025, "value": 156.8, "price": 8750.40, "change": 1.92, "unit": "billion USD"},
                "Cotton": {"volume": 0.009, "value": 28.4, "price": 0.78, "change": -1.05, "unit": "billion USD"}
            }

            # Trade routes
            self.trade_routes = {
                "Trans-Pacific": {"volume": 4.2, "value": 2845.6, "vessels": 1247, "avg_time": 14.5},
                "Trans-Atlantic": {"volume": 2.8, "value": 1923.4, "vessels": 892, "avg_time": 8.2},
                "Asia-Europe": {"volume": 3.7, "value": 2156.9, "vessels": 1089, "avg_time": 22.1},
                "Middle East-Asia": {"volume": 2.1, "value": 987.3, "vessels": 534, "avg_time": 12.8},
                "Africa-Europe": {"volume": 1.4, "value": 456.7, "vessels": 298, "avg_time": 6.4}
            }

            # Trade agreements
            self.trade_agreements = {
                "USMCA": {"members": 3, "trade_volume": 1.78, "tariff_avg": 2.1, "status": "Active"},
                "CPTPP": {"members": 11, "trade_volume": 2.45, "tariff_avg": 1.8, "status": "Active"},
                "ASEAN": {"members": 10, "trade_volume": 2.83, "tariff_avg": 3.2, "status": "Active"},
                "EU Single Market": {"members": 27, "trade_volume": 3.97, "tariff_avg": 0.8, "status": "Active"},
                "AfCFTA": {"members": 55, "trade_volume": 0.67, "tariff_avg": 6.4, "status": "Partial"},
                "RCEP": {"members": 15, "trade_volume": 3.21, "tariff_avg": 2.9, "status": "Active"}
            }

            # Economic indicators
            self.global_indicators = {
                "Global GDP Growth": 3.2,
                "World Trade Growth": 2.8,
                "Container Throughput": 856.4,
                "Shipping Rates Index": 142.7,
                "Trade Finance Cost": 4.35,
                "Logistics Performance": 3.48
            }

            # Current trade tensions
            self.trade_tensions = [
                {"parties": "US-China", "sector": "Technology", "impact": "High", "tariff": "25%", "status": "Ongoing"},
                {"parties": "EU-US", "sector": "Steel/Aluminum", "impact": "Medium", "tariff": "10%",
                 "status": "Negotiating"},
                {"parties": "UK-EU", "sector": "Agriculture", "impact": "Medium", "tariff": "8%", "status": "Resolved"},
                {"parties": "India-Pakistan", "sector": "Textiles", "impact": "Low", "tariff": "15%", "status": "Ongoing"},
                {"parties": "Russia-West", "sector": "Energy", "impact": "High", "tariff": "Sanctions", "status": "Ongoing"}
            ]

            logger.debug("Trade data initialization completed", context={
                "economies_count": len(self.major_economies),
                "commodities_count": len(self.commodities),
                "routes_count": len(self.trade_routes)
            })

    def create_content(self):
        """Create Bloomberg Terminal World Trade Analysis layout"""
        try:
            with operation("create_content"):
                # Top header with search and filters
                self.create_trade_header()

                # Function keys
                self.create_trade_function_keys()

                # Main analysis layout
                with dpg.group(horizontal=True):
                    # Left panel - Global Overview
                    self.create_global_overview_panel()

                    # Center panel - Detailed Analysis
                    self.create_detailed_analysis_panel()

                    # Right panel - Research & Alerts
                    self.create_research_panel()

                # Bottom workflow section
                self.create_workflow_section()

            logger.info("World Trade Analysis content created successfully")

        except Exception as e:
            logger.error("Error in create_content", context={"error": str(e)}, exc_info=True)
            dpg.add_text("World Trade Analysis Terminal", color=self.BLOOMBERG_ORANGE)
            dpg.add_text("Loading comprehensive trade data...")

    def create_trade_header(self):
        """Create trade analysis header"""
        with operation("create_trade_header"):
            with dpg.group(horizontal=True):
                dpg.add_text("FINCEPT", color=self.BLOOMBERG_ORANGE)
                dpg.add_text("WORLD TRADE ANALYSIS", color=self.BLOOMBERG_WHITE)
                dpg.add_text(" | ", color=self.BLOOMBERG_GRAY)
                dpg.add_combo(["All Regions", "North America", "Europe", "Asia-Pacific", "Middle East", "Africa"],
                              default_value="All Regions", width=120)
                dpg.add_combo(["All Commodities", "Energy", "Metals", "Agriculture", "Manufacturing"],
                              default_value="All Commodities", width=140)
                dpg.add_input_text(label="", default_value="Search Trade Data", width=200)
                dpg.add_button(label="Analyze", width=80)
                dpg.add_text(" | ", color=self.BLOOMBERG_GRAY)
                dpg.add_text(f"Last Update: {datetime.datetime.now().strftime('%H:%M:%S')}")

            dpg.add_separator()

    def create_trade_function_keys(self):
        """Create function keys for trade analysis"""
        with dpg.group(horizontal=True):
            function_keys = [
                "F1:OVERVIEW", "F2:ROUTES", "F3:COMMODITIES",
                "F4:AGREEMENTS", "F5:TENSIONS", "F6:FORECAST"
            ]
            for key in function_keys:
                dpg.add_button(label=key, width=110, height=25)

        dpg.add_separator()

    def create_global_overview_panel(self):
        """Create global trade overview panel"""
        with operation("create_global_overview_panel"):
            with dpg.child_window(width=380, height=700, border=True):
                dpg.add_text("GLOBAL TRADE OVERVIEW", color=self.BLOOMBERG_ORANGE)
                dpg.add_separator()

                # Key indicators
                dpg.add_text("GLOBAL INDICATORS", color=self.BLOOMBERG_YELLOW)
                with dpg.table(header_row=True, borders_innerH=True, borders_outerH=True):
                    dpg.add_table_column(label="Indicator")
                    dpg.add_table_column(label="Value")
                    dpg.add_table_column(label="Unit")

                    indicators = [
                        ("GDP Growth", "3.2", "%"),
                        ("Trade Growth", "2.8", "%"),
                        ("Container Volume", "856.4", "M TEU"),
                        ("Shipping Index", "142.7", "Points"),
                        ("Finance Cost", "4.35", "%"),
                        ("LPI Score", "3.48", "/5.0")
                    ]

                    for indicator, value, unit in indicators:
                        with dpg.table_row():
                            dpg.add_text(indicator)
                            dpg.add_text(value, color=self.BLOOMBERG_WHITE)
                            dpg.add_text(unit, color=self.BLOOMBERG_GRAY)

                dpg.add_separator()

                # Top trading nations
                dpg.add_text("TOP TRADING NATIONS", color=self.BLOOMBERG_YELLOW)
                with dpg.table(header_row=True, borders_innerH=True, borders_outerH=True, scrollY=True, height=200):
                    dpg.add_table_column(label="Country")
                    dpg.add_table_column(label="Exports")
                    dpg.add_table_column(label="Balance")
                    dpg.add_table_column(label="Growth")

                    for country, data in list(self.major_economies.items())[:8]:
                        with dpg.table_row():
                            dpg.add_text(country)
                            dpg.add_text(f"{data['exports']:.1f}B")

                            balance_color = self.BLOOMBERG_GREEN if data['balance'] > 0 else self.BLOOMBERG_RED
                            dpg.add_text(f"{data['balance']:+.1f}B", color=balance_color)

                            growth_color = self.BLOOMBERG_GREEN if data['growth'] > 1.5 else self.BLOOMBERG_RED
                            dpg.add_text(f"{data['growth']:+.1f}%", color=growth_color)

                dpg.add_separator()

                # Trade routes status
                dpg.add_text("MAJOR TRADE ROUTES", color=self.BLOOMBERG_YELLOW)
                for route, data in self.trade_routes.items():
                    with dpg.group(horizontal=True):
                        dpg.add_text(f"{route}:")
                        dpg.add_text(f"{data['volume']:.1f}T USD", color=self.BLOOMBERG_WHITE)
                        dpg.add_text(f"({data['vessels']} vessels)", color=self.BLOOMBERG_GRAY)

    def create_detailed_analysis_panel(self):
        """Create detailed analysis panel with tabs"""
        with operation("create_detailed_analysis_panel"):
            with dpg.child_window(width=750, height=700, border=True):
                with dpg.tab_bar():

                    # Commodity Analysis Tab
                    with dpg.tab(label="Commodity Analysis"):
                        dpg.add_text("GLOBAL COMMODITY TRADE", color=self.BLOOMBERG_ORANGE)

                        with dpg.table(header_row=True, borders_innerH=True, borders_outerH=True, scrollY=True, height=250):
                            dpg.add_table_column(label="Commodity")
                            dpg.add_table_column(label="Volume")
                            dpg.add_table_column(label="Value")
                            dpg.add_table_column(label="Price")
                            dpg.add_table_column(label="Change")
                            dpg.add_table_column(label="Status")

                            for commodity, data in self.commodities.items():
                                with dpg.table_row():
                                    dpg.add_text(commodity)
                                    dpg.add_text(f"{data['volume']:.2f}")
                                    dpg.add_text(f"{data['value']:.1f}B")
                                    dpg.add_text(f"{data['price']:.2f}")

                                    change_color = self.BLOOMBERG_GREEN if data['change'] > 0 else self.BLOOMBERG_RED
                                    dpg.add_text(f"{data['change']:+.2f}%", color=change_color)

                                    status = "Rising" if data['change'] > 0 else "Falling"
                                    dpg.add_text(status, color=change_color)

                        dpg.add_separator()

                        # Commodity details
                        dpg.add_text("COMMODITY SPOTLIGHT: CRUDE OIL", color=self.BLOOMBERG_ORANGE)
                        with dpg.group(horizontal=True):
                            with dpg.group():
                                dpg.add_text("Current Price: $78.35/barrel", color=self.BLOOMBERG_WHITE)
                                dpg.add_text("Daily Change: -1.25%", color=self.BLOOMBERG_RED)
                                dpg.add_text("Volume: 1.89T USD", color=self.BLOOMBERG_WHITE)
                                dpg.add_text("Top Exporters: Saudi Arabia, Russia, USA")

                            with dpg.group():
                                dpg.add_text("52W High: $95.20", color=self.BLOOMBERG_GRAY)
                                dpg.add_text("52W Low: $65.80", color=self.BLOOMBERG_GRAY)
                                dpg.add_text("Volatility: 28.4%", color=self.BLOOMBERG_YELLOW)
                                dpg.add_text("Next Release: OPEC+ Meeting (Mar 15)")

                    # Trade Routes Tab
                    with dpg.tab(label="Trade Routes"):
                        dpg.add_text("GLOBAL SHIPPING ROUTES", color=self.BLOOMBERG_ORANGE)

                        with dpg.table(header_row=True, borders_innerH=True, borders_outerH=True):
                            dpg.add_table_column(label="Route")
                            dpg.add_table_column(label="Volume (T USD)")
                            dpg.add_table_column(label="Vessels")
                            dpg.add_table_column(label="Avg Transit")
                            dpg.add_table_column(label="Status")

                            for route, data in self.trade_routes.items():
                                with dpg.table_row():
                                    dpg.add_text(route)
                                    dpg.add_text(f"{data['volume']:.1f}")
                                    dpg.add_text(f"{data['vessels']:,}")
                                    dpg.add_text(f"{data['avg_time']:.1f} days")

                                    if data['avg_time'] < 10:
                                        dpg.add_text("Optimal", color=self.BLOOMBERG_GREEN)
                                    elif data['avg_time'] < 20:
                                        dpg.add_text("Normal", color=self.BLOOMBERG_YELLOW)
                                    else:
                                        dpg.add_text("Delayed", color=self.BLOOMBERG_RED)

                        dpg.add_separator()

                        # Route analysis
                        dpg.add_text("ROUTE PERFORMANCE ANALYSIS", color=self.BLOOMBERG_ORANGE)
                        with dpg.group(horizontal=True):
                            with dpg.group():
                                dpg.add_text("Most Efficient: Trans-Atlantic", color=self.BLOOMBERG_GREEN)
                                dpg.add_text("Average Transit: 8.2 days")
                                dpg.add_text("Congestion Level: Low")
                                dpg.add_text("Cost Index: 98.4")

                            with dpg.group():
                                dpg.add_text("Highest Volume: Trans-Pacific", color=self.BLOOMBERG_BLUE)
                                dpg.add_text("Daily Vessels: 86")
                                dpg.add_text("Capacity Utilization: 94.2%")
                                dpg.add_text("Weather Impact: Moderate")

                    # Trade Agreements Tab
                    with dpg.tab(label="Trade Agreements"):
                        dpg.add_text("ACTIVE TRADE AGREEMENTS", color=self.BLOOMBERG_ORANGE)

                        with dpg.table(header_row=True, borders_innerH=True, borders_outerH=True):
                            dpg.add_table_column(label="Agreement")
                            dpg.add_table_column(label="Members")
                            dpg.add_table_column(label="Trade Volume")
                            dpg.add_table_column(label="Avg Tariff")
                            dpg.add_table_column(label="Status")

                            for agreement, data in self.trade_agreements.items():
                                with dpg.table_row():
                                    dpg.add_text(agreement)
                                    dpg.add_text(f"{data['members']}")
                                    dpg.add_text(f"{data['trade_volume']:.2f}T")
                                    dpg.add_text(f"{data['tariff_avg']:.1f}%")

                                    status_color = self.BLOOMBERG_GREEN if data[
                                                                               'status'] == "Active" else self.BLOOMBERG_YELLOW
                                    dpg.add_text(data['status'], color=status_color)

                        dpg.add_separator()

                        # Recent developments
                        dpg.add_text("RECENT DEVELOPMENTS", color=self.BLOOMBERG_ORANGE)
                        developments = [
                            "RCEP: New digital trade provisions implemented",
                            "USMCA: Review process initiated for 2024",
                            "EU-Mercosur: Negotiations resumed after 3-year pause",
                            "AfCFTA: Phase 2 protocols under discussion"
                        ]

                        for dev in developments:
                            dpg.add_text(f"• {dev}")

                    # Tensions & Risks Tab
                    with dpg.tab(label="Tensions & Risks"):
                        dpg.add_text("CURRENT TRADE TENSIONS", color=self.BLOOMBERG_ORANGE)

                        with dpg.table(header_row=True, borders_innerH=True, borders_outerH=True):
                            dpg.add_table_column(label="Parties")
                            dpg.add_table_column(label="Sector")
                            dpg.add_table_column(label="Impact")
                            dpg.add_table_column(label="Tariff")
                            dpg.add_table_column(label="Status")

                            for tension in self.trade_tensions:
                                with dpg.table_row():
                                    dpg.add_text(tension['parties'])
                                    dpg.add_text(tension['sector'])

                                    impact_color = (self.BLOOMBERG_RED if tension['impact'] == "High"
                                                    else self.BLOOMBERG_YELLOW if tension['impact'] == "Medium"
                                    else self.BLOOMBERG_GREEN)
                                    dpg.add_text(tension['impact'], color=impact_color)

                                    dpg.add_text(tension['tariff'])

                                    status_color = self.BLOOMBERG_RED if tension[
                                                                             'status'] == "Ongoing" else self.BLOOMBERG_GREEN
                                    dpg.add_text(tension['status'], color=status_color)

                        dpg.add_separator()

                        # Risk assessment
                        dpg.add_text("RISK ASSESSMENT", color=self.BLOOMBERG_ORANGE)
                        with dpg.group(horizontal=True):
                            with dpg.group():
                                dpg.add_text("Trade War Risk: MEDIUM", color=self.BLOOMBERG_YELLOW)
                                dpg.add_text("Supply Chain Disruption: HIGH", color=self.BLOOMBERG_RED)
                                dpg.add_text("Currency Volatility: MEDIUM", color=self.BLOOMBERG_YELLOW)

                            with dpg.group():
                                dpg.add_text("Geopolitical Risk: HIGH", color=self.BLOOMBERG_RED)
                                dpg.add_text("Energy Security: MEDIUM", color=self.BLOOMBERG_YELLOW)
                                dpg.add_text("Food Security: LOW", color=self.BLOOMBERG_GREEN)

    def create_research_panel(self):
        """Create research and alerts panel"""
        with operation("create_research_panel"):
            with dpg.child_window(width=380, height=700, border=True):
                dpg.add_text("TRADE RESEARCH CENTER", color=self.BLOOMBERG_ORANGE)
                dpg.add_separator()

                # Quick research tools
                dpg.add_text("RESEARCH TOOLS", color=self.BLOOMBERG_YELLOW)
                research_tools = [
                    "Trade Flow Analysis",
                    "Tariff Impact Calculator",
                    "Route Optimization",
                    "Cost-Benefit Analysis",
                    "Risk Scenario Modeling",
                    "Compliance Checker"
                ]

                for tool in research_tools:
                    dpg.add_button(label=tool, width=-1, height=25)

                dpg.add_separator()

                # Market alerts
                dpg.add_text("TRADE ALERTS", color=self.BLOOMBERG_YELLOW)
                alerts = [
                    ("HIGH", "Suez Canal: Congestion increasing", self.BLOOMBERG_RED),
                    ("MED", "US-China: New tech restrictions", self.BLOOMBERG_YELLOW),
                    ("LOW", "EU carbon border tax: Q2 implementation", self.BLOOMBERG_GREEN),
                    ("HIGH", "Red Sea: Security concerns affecting shipping", self.BLOOMBERG_RED),
                    ("MED", "WTO: New dispute resolution timeline", self.BLOOMBERG_YELLOW)
                ]

                for level, alert, color in alerts:
                    with dpg.group(horizontal=True):
                        dpg.add_text(f"[{level}]", color=color)
                        dpg.add_text(alert[:35] + "..." if len(alert) > 35 else alert)

                dpg.add_separator()

                # Economic calendar
                dpg.add_text("TRADE CALENDAR", color=self.BLOOMBERG_YELLOW)
                calendar_events = [
                    "Mar 15: OPEC+ Meeting",
                    "Mar 18: Fed Trade Policy Review",
                    "Mar 22: EU-China Summit",
                    "Mar 25: WTO Dispute Panel",
                    "Apr 02: G20 Trade Ministers",
                    "Apr 08: ASEAN Economic Summit"
                ]

                for event in calendar_events:
                    dpg.add_text(f"• {event}", color=self.BLOOMBERG_WHITE)

                dpg.add_separator()

                # Research reports
                dpg.add_text("LATEST REPORTS", color=self.BLOOMBERG_YELLOW)
                reports = [
                    "Q1 Global Trade Outlook",
                    "Supply Chain Resilience Study",
                    "Digital Trade Transformation",
                    "Sustainable Trade Practices",
                    "Regional Trade Agreement Impact"
                ]

                for report in reports:
                    with dpg.group(horizontal=True):
                        dpg.add_button(label="", width=25, height=20)
                        dpg.add_text(report)

    def create_workflow_section(self):
        """Create workflow analysis section"""
        with operation("create_workflow_section"):
            dpg.add_separator()
            dpg.add_text("TRADE WORKFLOW ANALYSIS", color=self.BLOOMBERG_ORANGE)

            with dpg.child_window(height=150, border=True):
                with dpg.group(horizontal=True):
                    # Workflow stage 1
                    with dpg.group():
                        dpg.add_text("1. MARKET ANALYSIS", color=self.BLOOMBERG_YELLOW)
                        dpg.add_text("• Identify opportunities")
                        dpg.add_text("• Risk assessment")
                        dpg.add_text("• Compliance check")
                        dpg.add_text("Status:  Complete", color=self.BLOOMBERG_GREEN)

                    dpg.add_separator()

                    # Workflow stage 2
                    with dpg.group():
                        dpg.add_text("2. ROUTE PLANNING", color=self.BLOOMBERG_YELLOW)
                        dpg.add_text("• Optimize logistics")
                        dpg.add_text("• Cost calculation")
                        dpg.add_text("• Timeline planning")
                        dpg.add_text("Status:  In Progress", color=self.BLOOMBERG_YELLOW)

                    dpg.add_separator()

                    # Workflow stage 3
                    with dpg.group():
                        dpg.add_text("3. EXECUTION", color=self.BLOOMBERG_YELLOW)
                        dpg.add_text("• Documentation")
                        dpg.add_text("• Shipment tracking")
                        dpg.add_text("• Quality control")
                        dpg.add_text("Status: ⏳ Pending", color=self.BLOOMBERG_GRAY)

                    dpg.add_separator()

                    # Workflow stage 4
                    with dpg.group():
                        dpg.add_text("4. SETTLEMENT", color=self.BLOOMBERG_YELLOW)
                        dpg.add_text("• Payment processing")
                        dpg.add_text("• Final documentation")
                        dpg.add_text("• Performance review")
                        dpg.add_text("Status: ⏳ Pending", color=self.BLOOMBERG_GRAY)

            # Status bar
            dpg.add_separator()
            with dpg.group(horizontal=True):
                dpg.add_text("", color=self.BLOOMBERG_GREEN)
                dpg.add_text("TRADE DATA LIVE", color=self.BLOOMBERG_GREEN)
                dpg.add_text(" | ", color=self.BLOOMBERG_GRAY)
                dpg.add_text("GLOBAL COVERAGE", color=self.BLOOMBERG_ORANGE)
                dpg.add_text(" | ", color=self.BLOOMBERG_GRAY)
                dpg.add_text("195 COUNTRIES", color=self.BLOOMBERG_WHITE)
                dpg.add_text(" | ", color=self.BLOOMBERG_GRAY)
                dpg.add_text("50+ COMMODITIES", color=self.BLOOMBERG_WHITE)
                dpg.add_text(" | ", color=self.BLOOMBERG_GRAY)
                dpg.add_text(f"UPDATED: {datetime.datetime.now().strftime('%H:%M:%S')}", color=self.BLOOMBERG_WHITE)
                dpg.add_text(" | ", color=self.BLOOMBERG_GRAY)
                dpg.add_text("LATENCY: 15ms", color=self.BLOOMBERG_GREEN)

    def resize_components(self, left_width, center_width, right_width, top_height, bottom_height, cell_height):
        """Resize components"""
        logger.debug("Resizing components", context={
            "left_width": left_width,
            "center_width": center_width,
            "right_width": right_width
        })

    def cleanup(self):
        """Clean up resources"""
        logger.info("World Trade Analysis tab cleanup completed")