"""
🧠 Core
========

🎯 ELI5 Summary:
This is the brain of our operation! Just like how your brain processes information 
and makes decisions, this file contains the main algorithm that does the mathematical 
thinking. It takes in data, processes it according to research principles, and produces 
intelligent results.

🧪 Technical Details:
===================
Implementation details and technical specifications for this component.
Designed to work seamlessly within the research framework while
maintaining high performance and accuracy standards.

"""
"""
🧠 Sparse Coding - Core Implementation Suite
==========================================

Author: Benedict Chen (benedict@benedictchen.com)

💰 Donations: Help support this research!
   PayPal: https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=WXQKYYKPHWXHS
   💖 Please consider recurring donations to support continued sparse coding research

Research-accurate implementation of Olshausen & Field's sparse coding algorithm
for dictionary learning and sparse representation discovery in natural images.

🔬 Research Foundation:
======================
Based on Olshausen & Field (1996) "Emergence of Simple-Cell Receptive Field Properties":
- Sparse coding explains how V1 simple cells develop edge-detecting receptive fields
- Overcomplete dictionary learning from natural image statistics
- Sparsity constraint produces localized, oriented basis functions
- Mathematical formulation: minimize_D,α ||x - Dα||² + λ||α||₁

Extended with modern optimization techniques:
- Beck & Teboulle (2009): FISTA for accelerated sparse inference
- Wright et al. (2010): Coordinate descent for large-scale optimization
- Aharon et al. (2006): K-SVD dictionary learning methods

ELI5 Explanation:
================
Think of sparse coding like learning to draw with a smart set of pencils! 🎨

🖼️ **The Goal**: You want to draw any picture (natural images) using the fewest pencil strokes possible.

📝 **The Learning Process**:
- Start with a random set of pencil patterns (dictionary atoms)
- Look at lots of pictures and figure out which pencil strokes capture the most important features
- Gradually improve your pencil patterns so they become really good at drawing edges, textures, and shapes
- Each picture should only need a few pencil strokes (sparse representation) to capture its essence

🧠 **Why This Matters**:
Your brain does exactly this! Visual cortex neurons learn to detect specific patterns (like vertical edges, horizontal lines, corners) that efficiently encode what you see. This algorithm discovers those same efficient representations automatically.

ASCII Sparse Coding Architecture:
=================================
    INPUT IMAGE       DICTIONARY         SPARSE CODES
    ┌─────────────┐   ┌─────────────┐   ┌─────────────┐
    │   x (8x8)   │   │ D (64x100)  │   │ α (100x1)   │
    │ ████████    │ = │ ████ ░░░░   │ × │    0.8      │
    │ ████░░░░    │   │ ░░░░ ████   │   │    0.0      │
    │ ░░░░████    │   │ ████ ████   │   │    -0.5     │
    │ ░░░░████    │   │   ...       │   │    0.0      │
    └─────────────┘   └─────────────┘   └─────────────┘
          │                 │                 │
          │ Reconstruction: │                 │
          │    x ≈ D·α      │                 │
          └─────────────────┼─────────────────┘
                           │
    ┌──────────────────────▼──────────────────────┐
    │ Learning Process: minimize ||x - Dα||² + λ||α||₁ │
    │                                                │
    │ 1. Sparse Inference: Fix D, optimize α        │
    │    (FISTA/Coordinate Descent)                  │
    │ 2. Dictionary Update: Fix α, optimize D        │
    │    (Gradient descent + normalization)          │
    └────────────────────────────────────────────────┘

📊 Mathematical Details:
=======================
**Objective Function:**
J(D,α) = ½||x - Dα||²₂ + λ||α||₁

**Alternating Optimization:**
1. **Sparse Inference** (α-step):
   α* = argmin_α ½||x - Dα||²₂ + λ||α||₁
   
2. **Dictionary Update** (D-step):
   D* = argmin_D ½||x - Dα||²₂ s.t. ||d_j||₂ ≤ 1

**FISTA Algorithm** (Fast Iterative Shrinkage-Thresholding):
- t₁ = 1, α₁ = 0
- For k = 1,2,...:
  - y_k = α_k + ((t_{k-1}-1)/t_k)(α_k - α_{k-1})
  - α_{k+1} = S_λ/L(y_k - (1/L)D^T(Dy_k - x))
  - t_{k+1} = (1 + √(1 + 4t_k²))/2

Where S_λ(z) = sign(z)max(|z| - λ, 0) is the soft-thresholding operator.
"""

from typing import Dict, List, Tuple, Union, Optional, Any, Callable
import warnings
import numpy as np

# Import core components
from .core_modules.core_algorithms import CoreAlgorithmsMixin
from .core_modules.optimization_algorithms import OptimizationAlgorithmsMixin
from .core_modules.dictionary_updates import DictionaryUpdatesMixin
from .core_modules.utilities_validation import UtilitiesValidationMixin

# Backward compatibility - export all components at module level
from .core_modules import get_complete_sparse_coder_class

def _get_refactoring_guide():
    """Returns refactoring documentation content"""
    import os
    
    # Try file-based reference first
    current_dir = os.path.dirname(__file__)
    doc_paths = [
        os.path.join(current_dir, "..", "docs", "REFACTORING_GUIDE.md"),
        os.path.join(current_dir, "..", "REFACTORING_GUIDE.md")
    ]
    
    for path in doc_paths:
        if os.path.exists(path):
            return f"📄 Complete refactoring guide: {path}"
    
    # Fallback content
    return """
📚 SPARSE CODING REFACTORING GUIDE

Package Structure:
- core.py: Main SparseCoder class
- sparse_coding_modules/: Algorithm components  
- config.py: Configuration management
- utils.py: Utility functions
- viz.py: Visualization tools

Algorithm Implementation:
- Olshausen & Field (1996) sparse coding
- FISTA acceleration (Beck & Teboulle 2009)
- K-SVD dictionary learning (Aharon et al. 2006)

Usage:
```python
from sparse_coding import SparseCoder
sc = SparseCoder(n_components=100, sparsity_penalty=0.1)
sc.fit(image_patches)
codes = sc.transform(new_patches)
```

Documentation: README.md
Support: benedict@benedictchen.com
"""

# Create the complete SparseCoder class for backward compatibility
SparseCoder = get_complete_sparse_coder_class()

# Export all components for easy access
__all__ = [
    'SparseCoder',
    'CoreAlgorithmsMixin',
    'OptimizationAlgorithmsMixin', 
    'DictionaryUpdatesMixin',
    'UtilitiesValidationMixin',
    'get_complete_sparse_coder_class'
]

# Usage examples and advanced features available in component modules

if __name__ == "__main__":
    # print("🏗️ Sparse Coding - Core Suite")
    print("=" * 50)
    print("Core sparse coding algorithms based on Olshausen & Field (1996)")
    print("Implements natural image statistics and V1 sparse coding model")
    print(f"  • Dictionary updates & learning: 380 lines") 
    print(f"  • Utilities & validation functions: 384 lines")
    print("")
    # # # # Removed print spam: "...
    # print("🏗️ Sparse coding implementation ready!")
    # # Removed print spam: "...
    print("")
    
    # Demo sparse coding workflow
    print("🔬 EXAMPLE SPARSE CODING WORKFLOW:")
    print("```python")
    print("# 1. Initialize SparseCoder with research-accurate parameters")
    print("sc = SparseCoder(n_components=64, alpha=0.1, algorithm='fista',")
    print("               sparsity_func='l1', dict_init='random')")
    print("")
    print("# 2. Fit dictionary on training data")
    print("sc.fit(X_train)  # Learn overcomplete dictionary")
    print("")
    print("# 3. Transform test data to sparse codes") 
    print("codes = sc.transform(X_test)  # Sparse coefficient inference")
    print("")
    print("# 4. Reconstruct data from sparse representation")
    print("reconstructed = sc.reconstruct(X_test)")
    print("")
    print("# 5. Comprehensive analysis")
    print("analysis = sc.comprehensive_analysis(X_test)")
    print("print(f'Sparsity level: {analysis[\"sparsity_metrics\"][\"hoyer_sparsity\"]:.3f}')")
    print("print(f'Dictionary quality: {analysis[\"dictionary_quality\"][\"max_coherence\"]:.3f}')")
    print("```")
    print("")
    print(_get_refactoring_guide())