"""
🔧 Utils Refactored
====================

🎯 ELI5 Summary:
This is like a toolbox full of helpful utilities! Just like how a carpenter has 
different tools for different jobs (hammer, screwdriver, saw), this file contains helpful 
functions that other parts of our code use to get their work done.

🧪 Technical Details:
===================
Implementation details and technical specifications for this component.
Designed to work seamlessly within the research framework while
maintaining high performance and accuracy standards.

📋 Component Integration:
========================
    ┌──────────┐
    │   This   │
    │Component │ ←→ Other Components
    └──────────┘
         ↑↓
    System Integration

"""
"""
🏗️ Sparse Coding - Refactored Utils Suite
=========================================

Utilities for sparse coding algorithms.

Author: Benedict Chen (benedict@benedictchen.com)
Based on: Olshausen & Field (1996) "Emergence of Simple-Cell Receptive Field Properties"

Modules:
- data_processing.py - Patch extraction, normalization, image reconstruction
- optimization.py - Thresholding operators, Lipschitz computation, line search
- validation_metrics.py - Data validation, dictionary coherence, convergence
- advanced_specialized.py - Gabor/DCT dictionaries, lateral inhibition, orthogonalization
"""

from typing import Dict, List, Tuple, Union, Optional, Any, Callable
import warnings
import numpy as np

# Import all modular utility components
from .utils_modules.data_processing import *
from .utils_modules.optimization import *
from .utils_modules.validation_metrics import *
from .utils_modules.advanced_specialized import *

# Export all components for easy access
__all__ = [
    # Data processing functions
    'extract_patches_2d',
    'extract_patches_from_images', 
    'normalize_patch_batch',
    'whiten_patches',
    'reconstruct_image_from_patches',
    
    # Optimization functions
    'soft_threshold',
    'hard_threshold', 
    'shrinkage_threshold',
    'compute_lipschitz_constant',
    'line_search_backtrack',
    
    # Validation and metrics functions
    'validate_sparse_coding_data',
    'compute_dictionary_coherence',
    'compute_spark',
    'validate_convergence',
    
    # Advanced and specialized functions
    'create_gabor_dictionary',
    'create_dct_dictionary',
    'lateral_inhibition_network',
    'estimate_noise_variance',
    'compute_mutual_coherence_matrix',
    'orthogonalize_dictionary'
]

# Legacy compatibility note
REFACTORING_GUIDE = """
🔄 MIGRATION GUIDE: From Monolithic to Modular Utils
====================================================

OLD (994-line monolith):
```python
from patch_processing_utilities import extract_patches_2d, soft_threshold, validate_sparse_coding_data, create_gabor_dictionary
# All functionality in one massive file
```

NEW (4 modular files):
```python
from utils_refactored import extract_patches_2d, soft_threshold, validate_sparse_coding_data, create_gabor_dictionary
# Clean imports from modular components
# data_processing, optimization, validation_metrics, advanced_specialized
```

✅ BENEFITS:
- Organized utility functions
- All modules under 800-line limit (compliance achieved)  
- Logical organization by functional domain
- Enhanced research accuracy with comprehensive FIXME annotations
- Better performance with selective imports
- Easier testing and debugging
- Clean separation of data processing, optimization, validation, and advanced tools

🎯 USAGE REMAINS IDENTICAL:
All utility functions work exactly the same!
Only internal organization changed.

🏗️ ENHANCED CAPABILITIES:
- More comprehensive data processing (ZCA whitening, multi-image patch extraction)
- Advanced thresholding operators (SCAD, non-negative garrote)
- Comprehensive validation metrics with extensive error checking
- Research-grade dictionary creation (Gabor filters, DCT basis)
- Lateral inhibition networks for biological modeling
- Dictionary orthogonalization and analysis tools

SELECTIVE IMPORTS (New Feature):
```python
# Import only what you need for better performance
from utils_modules.data_processing import extract_patches_2d, normalize_patch_batch
from utils_modules.optimization import soft_threshold, compute_lipschitz_constant
from utils_modules.validation_metrics import compute_dictionary_coherence

# Minimal footprint with just essential functionality
```

COMPLETE INTERFACE (Same as Original):
```python
# Full backward compatibility
from utils_refactored import *

# All original functions available
patches = extract_patches_2d(image, (8, 8))
normalized = normalize_patch_batch(patches)
thresh_codes = soft_threshold(codes, 0.1)
metrics = validate_sparse_coding_data(X, dictionary, codes)
```

ADVANCED FEATURES (New Capabilities):
```python
# Comprehensive data processing
patches = extract_patches_from_images(image_list, (8, 8), patches_per_image=100)
whitened = whiten_patches(patches)  # ZCA whitening

# Advanced thresholding
scad_thresh = shrinkage_threshold(codes, 0.1, shrinkage_type='scad')
garrote_thresh = shrinkage_threshold(codes, 0.1, shrinkage_type='garrote')

# Dictionary analysis
coherence_matrix = compute_mutual_coherence_matrix(dictionary)
spark_estimate = compute_spark(dictionary)
ortho_dict = orthogonalize_dictionary(dictionary, method='svd')

# Research-grade dictionary creation
gabor_dict = create_gabor_dictionary((8, 8), n_orientations=8, n_scales=3)
dct_dict = create_dct_dictionary((8, 8))

# Biological modeling
inhibited_codes = lateral_inhibition_network(codes, inhibition_strength=0.1)
noise_var = estimate_noise_variance(X, codes, dictionary)
```

RESEARCH ACCURACY (Preserved and Enhanced):
```python
# All FIXME comments preserved for research accuracy
# Extensive documentation referencing sparse coding literature
# Comprehensive validation with detailed error analysis
# Research-grade tools with proper mathematical foundations
```
"""

if __name__ == "__main__":
    # print("🏗️ Sparse Coding - Utils Suite")
    print("=" * 50)
    # Removed print spam: "...
    print(f"  Utility functions available")
    print(f"  Organized into 4 modules for clarity")
    print(f"  All modules loaded successfully")
    print("")
    # Removed print spam: "...
    print(f"  • Data processing utilities: 256 lines")
    print(f"  • Optimization utilities: 213 lines")
    print(f"  • Validation and metrics: 244 lines") 
    print(f"  • Advanced specialized tools: 284 lines")
    print("")
    # # # # Removed print spam: "...
    # # # print("🏗️ Enhanced modular architecture...")
    # # # # Removed print spam: "...
    print("")
    
    # Demo utility workflow
    print("🔬 EXAMPLE UTILITY WORKFLOW:")
    print("```python")
    print("# 1. Extract and preprocess patches")
    print("patches = extract_patches_2d(image, (8, 8), max_patches=1000)")
    print("normalized = normalize_patch_batch(patches, method='whiten')")
    print("")
    print("# 2. Apply sparse coding with utilities")
    print("thresh_codes = soft_threshold(codes, 0.1)")
    print("lipschitz_L = compute_lipschitz_constant(dictionary)")
    print("")
    print("# 3. Validate results comprehensively") 
    print("metrics = validate_sparse_coding_data(X, dictionary, codes)")
    print("coherence = compute_dictionary_coherence(dictionary)")
    print("")
    print("# 4. Advanced analysis and modeling")
    print("gabor_dict = create_gabor_dictionary((8, 8), n_orientations=8)")
    print("inhibited = lateral_inhibition_network(codes)")
    print("```")
    print("")
    print(REFACTORING_GUIDE)