"""
Detects "drift" in compiled files by comparing them against their .hash files.

This module provides functionality to verify the integrity of compiled YAML files
generated by the main compiler. The compiler creates a `.hash` file for each
YAML file it writes, containing a base64 encoded snapshot of the file's exact
content at the time of creation.

This checker iterates through all `.hash` files, decodes their contents, and
compares them with the current contents of the corresponding compiled files.
If any discrepancies are found, it indicates that a file has been manually
edited after compilation. The module will then print a user-friendly diff
report for each modified file and return a non-zero exit code, which is
useful for integration into CI/CD pipelines to prevent unintended changes.
"""

from __future__ import annotations

import base64
import difflib
import logging
from collections.abc import Generator
from pathlib import Path

__all__ = ["run_detect_drift"]

from bash2gitlab.utils.terminal_colors import Colors
from bash2gitlab.utils.utils import short_path

# Setting up a logger for this module. The calling application can configure the handler.
logger = logging.getLogger(__name__)


def decode_hash_content(hash_file: Path) -> str | None:
    """
    Reads and decodes the base64 content of a .hash file.

    Args:
        hash_file: The path to the .hash file.

    Returns:
        The decoded string content, or None if an error occurs.
    """
    try:
        last_known_base64 = hash_file.read_text(encoding="utf-8").strip()
        if not last_known_base64:
            logger.warning(f"Hash file is empty: {hash_file}")
            return None
        last_known_content_bytes = base64.b64decode(last_known_base64)
        return last_known_content_bytes.decode("utf-8")
    except (ValueError, TypeError) as e:
        logger.error(f"Could not decode the .hash file '{hash_file}'. It may be corrupted. Error: {e}")
        return None
    except FileNotFoundError:
        logger.error(f"Hash file not found: {hash_file}")
        return None


def get_source_file_from_hash(hash_file: Path) -> Path:
    """
    Derives the original source file path from a hash file path.
    Example: /path/to/file.yml.hash -> /path/to/file.yml

    Args:
        hash_file: The path to the .hash file.

    Returns:
        The corresponding Path object for the original file.
    """
    s = str(hash_file)
    if hasattr(s, "removesuffix"):  # Python 3.9+
        return Path(s.removesuffix(".hash"))
    # Python < 3.9
    if s.endswith(".hash"):
        return Path(s[: -len(".hash")])
    return Path(s)


def generate_pretty_diff(source_content: str, decoded_content: str, source_file_path: Path) -> str:
    """
    Generates a colorized (if enabled), unified diff string between two content strings.

    Args:
        source_content: The current content of the file.
        decoded_content: The original content from the hash.
        source_file_path: The path to the source file (for labeling the diff).

    Returns:
        A formatted and colorized diff string.
    """
    diff_lines = difflib.unified_diff(
        decoded_content.splitlines(),
        source_content.splitlines(),
        fromfile=f"{source_file_path} (from hash)",
        tofile=f"{source_file_path} (current, with manual edits)",
        lineterm="",
    )

    colored_diff = []
    for line in diff_lines:
        if line.startswith("+"):
            colored_diff.append(f"{Colors.OKGREEN}{line}{Colors.ENDC}")
        elif line.startswith("-"):
            colored_diff.append(f"{Colors.FAIL}{line}{Colors.ENDC}")
        elif line.startswith("@@"):
            colored_diff.append(f"{Colors.OKCYAN}{line}{Colors.ENDC}")
        else:
            colored_diff.append(line)
    return "\n".join(colored_diff)


def find_hash_files(search_paths: list[Path]) -> Generator[Path, None, None]:
    """
    Finds all .hash files recursively in the given list of directories.

    Args:
        search_paths: A list of directories to search in.

    Yields:
        Path objects for each .hash file found.
    """
    for search_path in search_paths:
        if not search_path.is_dir():
            logger.warning(f"Search path is not a directory, skipping: {search_path}")
            continue
        logger.info(f"Searching for .hash files in: {short_path(search_path)}")
        yield from search_path.rglob("*.hash")


def run_detect_drift(
    output_path: Path,
) -> int:
    """
    Checks for manual edits (drift) in compiled files by comparing them against their .hash files.

    This function iterates through all `.hash` files in the specified output directories,
    decodes their contents, and compares them with the current contents of the
    corresponding compiled files. It prints a diff for any files that have changed.

    Args:
        output_path: The main output directory containing compiled files (e.g., .gitlab-ci.yml).

    Returns:
        int: Returns 0 if no drift is detected.
             Returns 1 if drift is found or if errors occurred during the check.
    """
    drift_detected_count = 0
    error_count = 0
    search_paths = [output_path]

    hash_files = list(find_hash_files(search_paths))

    if not hash_files:
        logger.warning("No .hash files found to check for drift.")
        return 0  # No hashes means no drift to detect.

    print(f"Found {len(hash_files)} hash file(s). Checking for drift...")

    for hash_file in hash_files:
        source_file = get_source_file_from_hash(hash_file)

        if not source_file.exists():
            logger.error(f"Drift check failed: Source file '{source_file}' is missing for hash file '{hash_file}'.")
            error_count += 1
            continue

        decoded_content = decode_hash_content(hash_file)
        if decoded_content is None:
            # Error already logged in the helper function
            error_count += 1
            continue

        try:
            current_content = source_file.read_text(encoding="utf-8")
        except OSError as e:
            logger.error(f"Drift check failed: Could not read source file '{source_file}'. Error: {e}")
            error_count += 1
            continue

        if current_content != decoded_content:
            drift_detected_count += 1
            diff_text = generate_pretty_diff(current_content, decoded_content, source_file)

            # Print a clear, formatted report for the user, adapting to color support.
            if Colors.ENDC:  # Check if colors are enabled
                print("\n" + f"{Colors.RED_BG}{Colors.BOLD} DRIFT DETECTED IN: {source_file} {Colors.ENDC}")
            else:
                print(f"\n--- DRIFT DETECTED IN: {source_file} ---")

            print(diff_text)

            if Colors.ENDC:
                print(f"{Colors.RED_BG}{' ' * 80}{Colors.ENDC}")

    if drift_detected_count > 0 or error_count > 0:
        # Print summary, adapting to color support
        if Colors.ENDC:
            print("\n" + f"{Colors.HEADER}{Colors.BOLD}{'-' * 25} DRIFT DETECTION SUMMARY {'-' * 25}{Colors.ENDC}")
            if drift_detected_count > 0:
                print(f"{Colors.FAIL}  - Found {drift_detected_count} file(s) with manual edits.{Colors.ENDC}")
            if error_count > 0:
                print(
                    f"{Colors.WARNING}  - Encountered {error_count} error(s) during the check (see logs for details).{Colors.ENDC}"
                )
        else:
            print("\n" + "--- DRIFT DETECTION SUMMARY ---")
            if drift_detected_count > 0:
                print(f"  - Found {drift_detected_count} file(s) with manual edits.")
            if error_count > 0:
                print(f"  - Encountered {error_count} error(s) during the check (see logs for details).")

        print("\n  To resolve, you can either:")
        print("    1. Revert the manual changes in the files listed above.")
        print("    2. Update the input files to match and recompile.")
        print("    3. Recompile and lose any changes to the above files.")

        if Colors.ENDC:
            print(f"{Colors.HEADER}{Colors.BOLD}{'-' * 79}{Colors.ENDC}")
        else:
            print(f"{'-' * 79}")

        return 1

    # Else, print success message, adapting to color support
    if Colors.ENDC:
        print(f"\n{Colors.OKGREEN}Drift detection complete. No drift detected.{Colors.ENDC}")
    else:
        print("\nDrift detection complete. No drift detected.")

    print("All compiled files match their hashes.")
    return 0
