# -*- coding: utf-8 -*-
# Created on Sat Jan 22 09:18:20 2022
# @author: toj
"""
Tools for extracting isosurfaces from images or implicit functions


.. currentmodule:: mymesh.contour

Contouring
==========
.. autosummary::
    :toctree: submodules/

    MarchingSquares
    MarchingSquaresImage
    MarchingTriangles
    MarchingCubes
    MarchingCubesImage
    MarchingTetrahedra
    MarchingElements
    Adaptive


"""
#%%
import time
import sys
from . import utils, tree, converter
import numpy as np
import scipy
import copy, warnings


#      7_______6              *__10___*
#     /|      /|             /|      /|
#    / |     / |           11 7     9 6
#   /  |    /  |           /  |    /  |
#  4___|___5   |          *___|_8_*   |
#  |   3___|___2          |   *__2|___*
#  |  /    |  /           4  /    5  /  
#  | /     | /            | 3     | 1  
#  |/      |/             |/      |/   
#  0_______1              *___0___*    
#

# 3___6___2
# |       |
# 7       5
# |       |
# 0___4___1


MSTriangle_Lookup = np.array([
            [[]],               # 0-0000
            [[7,6,3]],          # 1-0001
            [[5,2,6]],          # 2-0010
            [[7,5,2],[2,3,7]],  # 3-0011
            [[4,1,5]],          # 4-0100
            [[4,1,5],[5,6,4],[4,6,7],[7,6,3]], # 5-0101
            [[4,1,2],[2,6,4]],  # 6-0110
            [[1,2,3],[3,4,1],[4,3,7]], # 7-0111
            [[0,4,7]],          # 8-1000
            [[0,4,6],[6,3,0]],  # 9-1001
            [[0,4,7],[4,5,7],[7,5,6],[6,5,2]], # 10-1010
            [[0,2,3],[0,4,5],[5,2,0]], # 11-1011
            [[0,1,5],[5,7,0]], # 12-1100
            [[0,1,3],[1,5,6],[6,3,1]], # 13-1101
            [[0,1,2],[0,2,6],[6,7,0]], # 14-1110
            [[0,1,2],[2,3,0]]   # 15-1111
        ],dtype=object)
MSMixed_Lookup = np.array([
            [[]],               # 0-0000
            [[7,6,3]],          # 1-0001
            [[5,2,6]],          # 2-0010
            [[7,5,2,3]],  # 3-0011
            [[4,1,5]],          # 4-0100
            [[4,1,5,6],[4,6,3,7]], # 5-0101
            [[1,2,6,4]],  # 6-0110
            [[1,2,3,4],[4,3,7]], # 7-0111
            [[0,4,7]],          # 8-1000
            [[0,4,6,3]],  # 9-1001
            [[0,4,5,7],[7,5,2,6]], # 10-1010
            [[0,5,2,3],[0,4,5]], # 11-1011
            [[0,1,5,7]], # 12-1100
            [[0,1,6,3],[1,5,6]], # 13-1101
            [[0,1,2,6],[6,7,0]], # 14-1110
            [[0,1,2,3]]   # 15-1111
        ],dtype=object)
MSMixedSplit_Lookup = np.array([
            [[0,1,2,3]],                                # 0-0000
            [[7,6,3],[0,1,2,6],[6,7,0]],                # 1-0001
            [[5,2,6],[0,1,6,3],[1,5,6]],                # 2-0010
            [[7,5,2,3],[0,1,5,7]],                      # 3-0011
            [[4,1,5],[0,5,2,3],[0,4,5]],                # 4-0100
            [[4,1,5,6],[4,6,3,7],[0,4,5,7],[7,5,2,6]],  # 5-0101
            [[1,2,6,4],[0,4,6,3]],                      # 6-0110
            [[1,2,3,4],[4,3,7],[0,4,7]],                # 7-0111
            [[1,2,3,4],[4,3,7],[0,4,7]],                # 8-1000
            [[1,2,6,4],[0,4,6,3]],                      # 9-1001
            [[4,1,5,6],[4,6,3,7],[0,4,5,7],[7,5,2,6]],  # 10-1010
            [[4,1,5],[0,5,2,3],[0,4,5]],                # 11-1011
            [[7,5,2,3],[0,1,5,7]],                      # 12-1100
            [[5,2,6],[0,1,6,3],[1,5,6]],                # 13-1101
            [[7,6,3],[0,1,2,6],[6,7,0]],                # 14-1110
            [[0,1,2,3]]   # 15-1111
        ],dtype=object)
MSEdge_Lookup = np.array([
            [[]],           # 0-0000
            [[6,7]],        # 1-0001
            [[5,6]],        # 2-0010
            [[5,7]],        # 3-0011
            [[4,5]],        # 4-0100
            [[7,4],[5,6]],  # 5-0101
            [[4,6]],        # 6-0110
            [[4,7]],        # 7-0111
            [[7,4]],        # 8-1000
            [[6,4]],        # 9-1001
            [[4,5],[6,7]],  # 10-1010
            [[5,4]],        # 11-1011
            [[7,5]],        # 12-1100
            [[6,5]],        # 13-1101
            [[7,6]], # 14-1110
            [[]]   # 15-1111
        ],dtype=object)
MTriEdge_Lookup = np.array([
            [],                     # 0-000
            [[2,1]],                # 1-001
            [[1,0]],                # 2-010
            [[2,0]],                # 3-011
            [[0,2]],                # 4-100
            [[0,1]],                # 5-101
            [[1,2]],                # 6-110
            [],                     # 7-111
        ],dtype=object)
MTriMixed_Lookup = np.array([
        [],                     # 0-000
        [[2,1,5]],              # 1-001
        [[1,0,4]],              # 2-010
        [[2,0,4,5]],            # 3-011
        [[0,2,3]],              # 4-100
        [[0,1,5,3]],            # 5-101
        [[1,2,3,4]],            # 6-110
        [[3,4,5]],              # 7-111
    ],dtype=object)
MTri_Lookup = np.array([
        [],                     # 0-000
        [[2,1,5]],              # 1-001
        [[1,0,4]],              # 2-010
        [[2,0,4],[2,4,5]],            # 3-011
        [[0,2,3]],              # 4-100
        [[0,1,5],[0,5,3]],            # 5-101
        [[1,2,3],[1,3,4]],            # 6-110
        [[3,4,5]],              # 7-111
    ],dtype=object)
MC_Lookup = np.array([
            [[]],
            [[7, 10, 11]],
            [[6, 9, 10]],
            [[6, 9, 7], [7, 9, 11]],
            [[5, 8, 9]],
            [[5, 8, 9], [7, 10, 11]],
            [[5, 8, 6], [6, 8, 10]],
            [[5, 7, 6], [5, 8, 11], [5, 11, 7]],
            [[11, 8, 4]],
            [[8, 7, 10], [4, 7, 8]],
            [[4, 11, 8], [6, 9, 10]],
            [[7, 6, 4], [9, 8, 4], [6, 9, 4]],
            [[4, 9, 5], [11, 9, 4]],
            [[7, 5, 4], [7, 10, 9], [7, 9, 5]],
            [[4, 6, 5], [4, 11, 10], [4, 10, 6]],
            [[4, 6, 5], [4, 7, 6]],
            [[7, 3, 2]],
            [[3, 2, 11], [11, 2, 10]],
            [[3, 2, 7], [9, 10, 6]],
            [[11, 3, 9], [2, 6, 9], [3, 2, 9]],
            [[2, 7, 3], [5, 8, 9]],
            [[5, 8, 9], [10, 3, 2], [11, 3, 10]],
            [[2, 7, 3], [5, 8, 6], [6, 8, 10]],
            [[3, 2, 6], [8, 3, 6], [8, 6, 5], [11, 3, 8]],
            [[2, 7, 3], [8, 4, 11]],
            [[10, 8, 2], [4, 3, 2], [8, 4, 2]],
            [[2, 7, 3], [4, 11, 8], [6, 9, 10]],
            [[2, 4, 3], [6, 4, 2], [6, 8, 4], [9, 8, 6]],
            [[7, 3, 2], [9, 5, 11], [11, 5, 4]],
            [[4, 3, 2], [4, 2, 9], [5, 4, 9], [9, 2, 10]],
            [[10, 4, 11], [6, 4, 10], [6, 5, 4], [3, 2, 7]],
            [[5, 4, 6], [3, 2, 6], [4, 3, 6]],
            [[6, 2, 1]],
            [[6, 2, 1], [7, 10, 11]],
            [[2, 9, 10], [1, 9, 2]],
            [[9, 11, 1], [7, 2, 1], [11, 7, 1]],
            [[2, 1, 6], [8, 9, 5]],
            [[8, 9, 5], [1, 6, 2], [11, 7, 10]],
            [[2, 8, 10], [2, 1, 5], [2, 5, 8]],
            [[5, 8, 1], [1, 8, 2], [8, 11, 2], [2, 11, 7]],
            [[1, 6, 2], [11, 8, 4]],
            [[6, 2, 1], [8, 4, 10], [10, 4, 7]],
            [[4, 11, 8], [9, 2, 1], [10, 2, 9]],
            [[9, 8, 4], [9, 4, 2], [1, 9, 2], [4, 7, 2]],
            [[1, 6, 2], [4, 11, 5], [5, 11, 9]],
            [[1, 7, 2], [5, 7, 1], [5, 4, 7], [10, 9, 6]],
            [[2, 1, 5], [11, 2, 5], [11, 5, 4], [10, 2, 11]],
            [[4, 7, 5], [2, 1, 5], [7, 2, 5]],
            [[7, 1, 6], [3, 1, 7]],
            [[1, 11, 3], [1, 6, 10], [1, 10, 11]],
            [[1, 9, 3], [10, 7, 3], [9, 10, 3]],
            [[11, 1, 9], [11, 3, 1]],
            [[9, 5, 8], [7, 3, 6], [6, 3, 1]],
            [[8, 1, 5], [11, 1, 8], [11, 3, 1], [6, 10, 9]],
            [[1, 5, 8], [1, 8, 7], [3, 1, 7], [7, 8, 10]],
            [[3, 1, 11], [5, 8, 11], [1, 5, 11]],
            [[8, 4, 11], [7, 1, 6], [3, 1, 7]],
            [[1, 6, 10], [1, 10, 4], [3, 1, 4], [4, 10, 8]],
            [[7, 9, 10], [3, 9, 7], [3, 1, 9], [8, 4, 11]],
            [[9, 3, 1], [9, 8, 4], [9, 4, 3]],
            [[5, 11, 9], [5, 4, 11], [6, 7, 3], [6, 3, 1]],
            [[9, 6, 10], [4, 3, 5], [5, 3, 1]],
            [[10, 7, 11], [4, 1, 5], [3, 1, 4]],
            [[1, 5, 3], [3, 5, 4]],
            [[0, 5, 1]],
            [[11, 7, 10], [5, 1, 0]],
            [[5, 1, 0], [6, 9, 10]],
            [[0, 5, 1], [6, 11, 7], [9, 11, 6]],
            [[1, 8, 9], [0, 8, 1]],
            [[10, 11, 7], [1, 0, 9], [9, 0, 8]],
            [[8, 10, 0], [6, 1, 0], [10, 6, 0]],
            [[6, 1, 0], [11, 6, 0], [11, 0, 8], [7, 6, 11]],
            [[1, 0, 5], [11, 8, 4]],
            [[0, 5, 1], [7, 10, 4], [4, 10, 8]],
            [[0, 5, 1], [6, 9, 10], [4, 11, 8]],
            [[0, 4, 1], [1, 4, 6], [4, 7, 6], [5, 9, 8]],
            [[1, 11, 9], [1, 0, 4], [1, 4, 11]],
            [[1, 0, 4], [10, 1, 4], [10, 4, 7], [9, 1, 10]],
            [[6, 1, 10], [10, 1, 11], [1, 0, 11], [11, 0, 4]],
            [[6, 4, 7], [6, 1, 0], [6, 0, 4]],
            [[5, 1, 0], [7, 3, 2]],
            [[5, 1, 0], [3, 10, 11], [2, 10, 3]],
            [[3, 2, 7], [10, 6, 9], [0, 5, 1]],
            [[6, 9, 2], [2, 9, 3], [9, 11, 3], [1, 0, 5]],
            [[3, 2, 7], [8, 9, 0], [0, 9, 1]],
            [[0, 9, 1], [0, 8, 9], [2, 10, 11], [2, 11, 3]],
            [[1, 0, 6], [6, 0, 10], [0, 8, 10], [2, 7, 3]],
            [[1, 2, 6], [8, 11, 0], [0, 11, 3]],
            [[1, 0, 5], [8, 4, 11], [2, 7, 3]],
            [[1, 8, 5], [2, 8, 1], [2, 10, 8], [4, 3, 0]],
            [[0, 5, 1], [5, 10, 6], [4, 11, 8], [2, 7, 3]],
            [[3, 0, 4], [6, 1, 2], [5, 9, 8]],
            [[4, 1, 0], [11, 1, 4], [11, 9, 1], [2, 7, 3]],
            [[4, 3, 0], [1, 10, 9], [2, 10, 1]],
            [[1, 2, 6], [10, 7, 11], [0, 4, 3]],
            [[4, 3, 0], [6, 1, 2]],
            [[0, 5, 2], [2, 5, 6]],
            [[11, 7, 10], [6, 0, 5], [2, 0, 6]],
            [[0, 10, 2], [0, 5, 9], [0, 9, 10]],
            [[11, 7, 2], [5, 11, 2], [5, 2, 0], [5, 9, 11]],
            [[0, 8, 2], [9, 6, 2], [8, 9, 2]],
            [[7, 8, 11], [2, 8, 7], [2, 0, 8], [9, 6, 10]],
            [[0, 10, 2], [0, 8, 10]],
            [[0, 8, 2], [11, 7, 2], [8, 11, 2]],
            [[8, 4, 11], [6, 2, 5], [5, 2, 0]],
            [[5, 6, 0], [6, 2, 0], [7, 10, 8], [4, 7, 8]],
            [[11, 0, 4], [10, 0, 11], [10, 2, 0], [5, 9, 8]],
            [[8, 5, 9], [7, 2, 4], [4, 2, 0]],
            [[9, 6, 2], [4, 9, 2], [4, 2, 0], [4, 11, 9]],
            [[9, 6, 10], [7, 0, 4], [2, 0, 7]],
            [[0, 10, 2], [0, 4, 11], [0, 11, 10]],
            [[0, 4, 2], [2, 4, 7]],
            [[6, 7, 5], [3, 0, 5], [7, 3, 5]],
            [[6, 10, 11], [0, 6, 11], [0, 11, 3], [0, 5, 6]],
            [[0, 5, 3], [3, 5, 7], [5, 9, 7], [7, 9, 10]],
            [[3, 9, 11], [3, 0, 5], [3, 5, 9]],
            [[8, 9, 6], [3, 8, 6], [3, 6, 7], [0, 8, 3]],
            [[10, 9, 6], [3, 0, 11], [11, 0, 8]],
            [[10, 0, 8], [10, 7, 3], [10, 3, 0]],
            [[8, 11, 0], [0, 11, 3]],
            [[0, 5, 3], [3, 5, 7], [5, 6, 7], [4, 11, 8]],
            [[0, 4, 3], [6, 10, 5], [5, 10, 8]],
            [[7, 11, 10], [0, 4, 3], [8, 5, 9]],
            [[3, 0, 4], [9, 8, 5]],
            [[4, 3, 0], [9, 6, 11], [11, 6, 7]],
            [[10, 9, 6], [3, 0, 4]],
            [[4, 3, 0], [7, 11, 10]],
            [[4, 3, 0]]],dtype=object)
MC33_Lookup = np.array([[[[]]],
         [[[7, 10, 11]]],
         [[[9, 10, 6]]],
         [[[6, 9, 7], [7, 9, 11]]],
         [[[8, 9, 5]]],
         [[[5, 8, 9], [7, 10, 11]], [[5, 7, 9], [5, 8, 7], [8, 11, 7], [7, 10, 9]]],
         [[[5, 8, 6], [6, 8, 10]]],
         [[[5, 7, 6], [5, 8, 11], [5, 11, 7]]],
         [[[11, 8, 4]]],
         [[[8, 4, 10], [10, 4, 7]]],
         [[[4, 11, 8], [6, 9, 10]], [[4, 6, 8], [4, 11, 6], [11, 10, 6], [6, 9, 8]]],
         [[[6, 4, 7], [6, 9, 8], [6, 8, 4]]],
         [[[4, 11, 5], [5, 11, 9]]],
         [[[7, 5, 4], [7, 10, 9], [7, 9, 5]]],
         [[[4, 6, 5], [4, 11, 10], [4, 10, 6]]],
         [[[4, 6, 5], [4, 7, 6]]],
         [[[7, 3, 2]]],
         [[[10, 11, 2], [2, 11, 3]]],
         [[[3, 2, 7], [9, 10, 6]], [[3, 9, 7], [3, 2, 9], [2, 6, 9], [9, 10, 7]]],
         [[[3, 9, 11], [3, 2, 6], [3, 6, 9]]],
         [[[2, 7, 3], [5, 8, 9]],
          [[7, 8, 9], [8, 7, 3], [3, 5, 8], [5, 3, 2], [2, 9, 5], [9, 7, 2]]],
         [[[8, 9, 5], [3, 2, 11], [11, 2, 10]],
          [[11, 8, 9],
           [11, 9, 10],
           [10, 9, 2],
           [2, 9, 5],
           [2, 5, 3],
           [8, 3, 5],
           [8, 11, 3]],
          [[5, 3, 2], [5, 2, 9], [9, 2, 10], [8, 3, 5], [3, 8, 11]]],
         [[[2, 7, 3], [5, 8, 6], [6, 8, 10]],
          [[6, 2, 7],
           [6, 7, 10],
           [10, 7, 8],
           [8, 7, 3],
           [8, 3, 5],
           [2, 5, 3],
           [2, 6, 5]],
          [[3, 5, 8], [3, 8, 7], [7, 8, 10], [2, 5, 3], [5, 2, 6]]],
         [[[3, 2, 6], [8, 3, 6], [8, 6, 5], [11, 3, 8]]],
         [[[2, 7, 3], [8, 4, 11]], [[2, 8, 3], [2, 7, 8], [7, 11, 8], [8, 4, 3]]],
         [[[8, 2, 10], [8, 4, 3], [8, 3, 2]]],
         [[[2, 7, 3], [4, 11, 8], [6, 9, 10]],
          [[3, 2, 6], [3, 6, 9], [3, 9, 10], [3, 10, 7], [4, 11, 8]],
          [[2, 4, 3], [2, 8, 4], [2, 11, 8], [2, 7, 11], [6, 9, 10]],
          [[6, 9, 8], [6, 8, 4], [6, 4, 10], [4, 11, 10], [2, 7, 3]],
          [[6, 12, 10],
           [10, 12, 11],
           [11, 12, 7],
           [7, 12, 2],
           [2, 12, 3],
           [3, 12, 4],
           [4, 12, 8],
           [8, 12, 9],
           [9, 12, 6]],
          [[10, 12, 11],
           [11, 12, 4],
           [4, 12, 8],
           [8, 12, 9],
           [9, 12, 6],
           [6, 12, 2],
           [2, 12, 3],
           [3, 12, 7],
           [7, 12, 10]],
          [[8, 12, 11],
           [11, 12, 7],
           [7, 12, 10],
           [10, 12, 9],
           [9, 12, 6],
           [6, 12, 2],
           [2, 12, 3],
           [3, 12, 4],
           [4, 12, 8]],
          [[7, 11, 10], [2, 6, 9], [2, 9, 4], [2, 4, 3], [4, 9, 8]],
          [[10, 8, 11],
           [10, 9, 8],
           [6, 9, 10],
           [6, 10, 7],
           [6, 7, 2],
           [2, 7, 3],
           [7, 4, 3],
           [7, 11, 4],
           [11, 8, 4]]],
         [[[3, 2, 4], [4, 2, 8], [2, 6, 8], [8, 6, 9]]],
         [[[7, 3, 2], [9, 5, 11], [11, 5, 4]],
          [[11, 7, 3],
           [11, 3, 4],
           [4, 3, 5],
           [5, 3, 2],
           [5, 2, 9],
           [7, 9, 2],
           [7, 11, 9]],
          [[2, 9, 5], [2, 5, 3], [3, 5, 4], [7, 9, 2], [9, 7, 11]]],
         [[[4, 3, 2], [9, 4, 2], [9, 2, 10], [9, 5, 4]]],
         [[[7, 3, 2], [11, 10, 4], [6, 4, 10], [4, 6, 5]],
          [[7, 11, 10], [3, 2, 4], [6, 4, 2], [4, 6, 5]],
          [[7, 10, 2],
           [6, 2, 10],
           [2, 6, 5],
           [2, 5, 3],
           [4, 3, 5],
           [3, 4, 11],
           [3, 11, 7],
           [10, 7, 11]],
          [[12, 10, 7],
           [12, 7, 3],
           [12, 3, 2],
           [12, 2, 6],
           [12, 6, 5],
           [12, 5, 4],
           [12, 4, 11],
           [12, 11, 10]],
          [[12, 6, 5],
           [12, 5, 4],
           [12, 4, 3],
           [12, 3, 2],
           [12, 2, 7],
           [12, 7, 11],
           [12, 11, 10],
           [12, 10, 6]]],
         [[[5, 4, 6], [3, 2, 6], [4, 3, 6]]],
         [[[2, 1, 6]]],
         [[[1, 6, 2], [11, 7, 10]], [[1, 11, 2], [1, 6, 11], [6, 10, 11], [11, 7, 2]]],
         [[[9, 10, 1], [1, 10, 2]]],
         [[[11, 1, 9], [11, 7, 2], [11, 2, 1]]],
         [[[8, 9, 5], [2, 1, 6]], [[8, 2, 5], [8, 9, 2], [9, 6, 2], [2, 1, 5]]],
         [[[7, 10, 11], [8, 9, 5], [2, 1, 6]],
          [[11, 7, 2], [11, 2, 1], [11, 1, 6], [11, 6, 10], [8, 9, 5]],
          [[7, 8, 11], [7, 5, 8], [7, 9, 5], [7, 10, 9], [2, 1, 6]],
          [[2, 1, 5], [2, 5, 8], [2, 8, 6], [8, 9, 6], [7, 10, 11]],
          [[2, 12, 6],
           [6, 12, 9],
           [9, 12, 10],
           [10, 12, 7],
           [7, 12, 11],
           [11, 12, 8],
           [8, 12, 5],
           [5, 12, 1],
           [1, 12, 2]],
          [[6, 12, 9],
           [9, 12, 8],
           [8, 12, 5],
           [5, 12, 1],
           [1, 12, 2],
           [2, 12, 7],
           [7, 12, 11],
           [11, 12, 10],
           [10, 12, 6]],
          [[5, 12, 9],
           [9, 12, 10],
           [10, 12, 6],
           [6, 12, 1],
           [1, 12, 2],
           [2, 12, 7],
           [7, 12, 11],
           [11, 12, 8],
           [8, 12, 5]],
          [[10, 9, 6], [7, 2, 1], [7, 1, 8], [7, 8, 11], [8, 1, 5]],
          [[6, 5, 9],
           [6, 1, 5],
           [2, 1, 6],
           [2, 6, 10],
           [2, 10, 7],
           [7, 10, 11],
           [10, 8, 11],
           [10, 9, 8],
           [9, 5, 8]]],
         [[[2, 8, 10], [2, 1, 5], [2, 5, 8]]],
         [[[2, 1, 7], [7, 1, 11], [1, 5, 11], [11, 5, 8]]],
         [[[2, 1, 6], [11, 8, 4]],
          [[1, 8, 4], [8, 1, 6], [6, 11, 8], [11, 6, 2], [2, 4, 11], [4, 1, 2]]],
         [[[6, 2, 1], [8, 4, 10], [10, 4, 7]],
          [[10, 6, 2],
           [10, 2, 7],
           [7, 2, 4],
           [4, 2, 1],
           [4, 1, 8],
           [6, 8, 1],
           [6, 10, 8]],
          [[1, 8, 4], [1, 4, 2], [2, 4, 7], [6, 8, 1], [8, 6, 10]]],
         [[[11, 8, 4], [2, 1, 10], [10, 1, 9]],
          [[10, 11, 8],
           [10, 8, 9],
           [9, 8, 1],
           [1, 8, 4],
           [1, 4, 2],
           [11, 2, 4],
           [11, 10, 2]],
          [[4, 2, 1], [4, 1, 8], [8, 1, 9], [11, 2, 4], [2, 11, 10]]],
         [[[7, 2, 1], [8, 7, 1], [8, 1, 9], [8, 4, 7]]],
         [[[1, 6, 2], [4, 11, 5], [5, 11, 9]],
          [[5, 1, 6],
           [5, 6, 9],
           [9, 6, 11],
           [11, 6, 2],
           [11, 2, 4],
           [1, 4, 2],
           [1, 5, 4]],
          [[2, 4, 11], [2, 11, 6], [6, 11, 9], [1, 4, 2], [4, 1, 5]]],
         [[[6, 2, 1], [10, 9, 7], [5, 7, 9], [7, 5, 4]],
          [[6, 10, 9], [2, 1, 7], [5, 7, 1], [7, 5, 4]],
          [[6, 9, 1],
           [5, 1, 9],
           [1, 5, 4],
           [1, 4, 2],
           [7, 2, 4],
           [2, 7, 10],
           [2, 10, 6],
           [9, 6, 10]],
          [[12, 9, 6],
           [12, 6, 2],
           [12, 2, 1],
           [12, 1, 5],
           [12, 5, 4],
           [12, 4, 7],
           [12, 7, 10],
           [12, 10, 9]],
          [[12, 5, 4],
           [12, 4, 7],
           [12, 7, 2],
           [12, 2, 1],
           [12, 1, 6],
           [12, 6, 10],
           [12, 10, 9],
           [12, 9, 5]]],
         [[[2, 1, 5], [11, 2, 5], [11, 5, 4], [10, 2, 11]]],
         [[[4, 7, 5], [2, 1, 5], [7, 2, 5]]],
         [[[1, 6, 3], [3, 6, 7]]],
         [[[1, 11, 3], [1, 6, 10], [1, 10, 11]]],
         [[[9, 3, 1], [9, 10, 7], [9, 7, 3]]],
         [[[11, 1, 9], [11, 3, 1]]],
         [[[9, 5, 8], [7, 3, 6], [6, 3, 1]],
          [[6, 9, 5],
           [6, 5, 1],
           [1, 5, 3],
           [3, 5, 8],
           [3, 8, 7],
           [9, 7, 8],
           [9, 6, 7]],
          [[8, 7, 3], [8, 3, 5], [5, 3, 1], [9, 7, 8], [7, 9, 6]]],
         [[[9, 5, 8], [6, 10, 1], [11, 1, 10], [1, 11, 3]],
          [[9, 6, 10], [5, 8, 1], [11, 1, 8], [1, 11, 3]],
          [[9, 10, 8],
           [11, 8, 10],
           [8, 11, 3],
           [8, 3, 5],
           [1, 5, 3],
           [5, 1, 6],
           [5, 6, 9],
           [10, 9, 6]],
          [[12, 10, 9],
           [12, 9, 5],
           [12, 5, 8],
           [12, 8, 11],
           [12, 11, 3],
           [12, 3, 1],
           [12, 1, 6],
           [12, 6, 10]],
          [[12, 11, 3],
           [12, 3, 1],
           [12, 1, 5],
           [12, 5, 8],
           [12, 8, 9],
           [12, 9, 6],
           [12, 6, 10],
           [12, 10, 11]]],
         [[[1, 5, 8], [7, 1, 8], [7, 8, 10], [7, 3, 1]]],
         [[[3, 1, 11], [5, 8, 11], [1, 5, 11]]],
         [[[4, 11, 8], [1, 6, 3], [3, 6, 7]],
          [[3, 4, 11],
           [3, 11, 7],
           [7, 11, 6],
           [6, 11, 8],
           [6, 8, 1],
           [4, 1, 8],
           [4, 3, 1]],
          [[8, 1, 6], [8, 6, 11], [11, 6, 7], [4, 1, 8], [1, 4, 3]]],
         [[[8, 4, 3], [6, 8, 3], [6, 3, 1], [10, 8, 6]]],
         [[[11, 8, 4], [10, 7, 9], [3, 9, 7], [9, 3, 1]],
          [[11, 10, 7], [8, 4, 9], [3, 9, 4], [9, 3, 1]],
          [[11, 7, 4],
           [3, 4, 7],
           [4, 3, 1],
           [4, 1, 8],
           [9, 8, 1],
           [8, 9, 10],
           [8, 10, 11],
           [7, 11, 10]],
          [[12, 7, 11],
           [12, 11, 8],
           [12, 8, 4],
           [12, 4, 3],
           [12, 3, 1],
           [12, 1, 9],
           [12, 9, 10],
           [12, 10, 7]],
          [[12, 3, 1],
           [12, 1, 9],
           [12, 9, 8],
           [12, 8, 4],
           [12, 4, 11],
           [12, 11, 10],
           [12, 10, 7],
           [12, 7, 3]]],
         [[[1, 9, 3], [8, 4, 3], [9, 8, 3]]],
         [[[3, 6, 7], [3, 1, 6], [11, 9, 5], [11, 5, 4]],
          [[3, 5, 4], [3, 1, 5], [7, 9, 6], [7, 11, 9]],
          [[3, 1, 4],
           [4, 1, 5],
           [3, 4, 7],
           [7, 4, 11],
           [7, 11, 6],
           [11, 9, 6],
           [1, 6, 5],
           [6, 9, 5]],
          [[5, 12, 1],
           [1, 12, 3],
           [3, 12, 7],
           [7, 12, 6],
           [6, 12, 9],
           [9, 12, 11],
           [11, 12, 4],
           [4, 12, 5]],
          [[6, 12, 1],
           [7, 12, 6],
           [11, 12, 7],
           [9, 12, 11],
           [5, 12, 9],
           [4, 12, 5],
           [3, 12, 4],
           [1, 12, 3]]],
         [[[10, 9, 6], [1, 4, 3], [5, 4, 1]],
          [[9, 6, 1],
           [5, 9, 1],
           [4, 9, 5],
           [10, 9, 4],
           [3, 10, 4],
           [10, 3, 6],
           [3, 1, 6]],
          [[4, 3, 10], [9, 4, 10], [5, 4, 9], [10, 3, 6], [1, 6, 3]]],
         [[[10, 7, 11], [4, 1, 5], [3, 1, 4]],
          [[7, 11, 4],
           [3, 7, 4],
           [1, 7, 3],
           [10, 7, 1],
           [5, 10, 1],
           [10, 5, 11],
           [5, 4, 11]],
          [[1, 5, 10], [7, 1, 10], [3, 1, 7], [10, 5, 11], [4, 11, 5]]],
         [[[1, 4, 3], [5, 4, 1]]],
         [[[0, 5, 1]]],
         [[[7, 10, 11], [0, 5, 1]],
          [[10, 5, 1], [5, 10, 11], [11, 0, 5], [0, 11, 7], [7, 1, 0], [1, 10, 7]]],
         [[[10, 6, 9], [0, 5, 1]], [[10, 0, 9], [10, 6, 0], [6, 1, 0], [0, 5, 9]]],
         [[[5, 1, 0], [11, 7, 9], [9, 7, 6]],
          [[9, 5, 1],
           [9, 1, 6],
           [6, 1, 7],
           [7, 1, 0],
           [7, 0, 11],
           [5, 11, 0],
           [5, 9, 11]],
          [[0, 11, 7], [0, 7, 1], [1, 7, 6], [5, 11, 0], [11, 5, 9]]],
         [[[1, 0, 9], [9, 0, 8]]],
         [[[10, 11, 7], [1, 0, 9], [9, 0, 8]],
          [[9, 10, 11],
           [9, 11, 8],
           [8, 11, 0],
           [0, 11, 7],
           [0, 7, 1],
           [10, 1, 7],
           [10, 9, 1]],
          [[7, 1, 0], [7, 0, 11], [11, 0, 8], [10, 1, 7], [1, 10, 9]]],
         [[[10, 0, 8], [10, 6, 1], [10, 1, 0]]],
         [[[8, 11, 7], [1, 8, 7], [1, 7, 6], [1, 0, 8]]],
         [[[11, 8, 4], [1, 0, 5]], [[11, 1, 4], [11, 8, 1], [8, 5, 1], [1, 0, 4]]],
         [[[0, 5, 1], [7, 10, 4], [4, 10, 8]],
          [[4, 0, 5],
           [4, 5, 8],
           [8, 5, 10],
           [10, 5, 1],
           [10, 1, 7],
           [0, 7, 1],
           [0, 4, 7]],
          [[1, 7, 10], [1, 10, 5], [5, 10, 8], [0, 7, 1], [7, 0, 4]]],
         [[[0, 5, 1], [6, 9, 10], [4, 11, 8]],
          [[1, 0, 4], [1, 4, 11], [1, 11, 8], [1, 8, 5], [6, 9, 10]],
          [[0, 6, 1], [0, 10, 6], [0, 9, 10], [0, 5, 9], [4, 11, 8]],
          [[4, 11, 10], [4, 10, 6], [4, 6, 8], [6, 9, 8], [0, 5, 1]],
          [[4, 12, 8],
           [8, 12, 9],
           [9, 12, 5],
           [5, 12, 0],
           [0, 12, 1],
           [1, 12, 6],
           [6, 12, 10],
           [10, 12, 11],
           [11, 12, 4]],
          [[8, 12, 9],
           [9, 12, 6],
           [6, 12, 10],
           [10, 12, 11],
           [11, 12, 4],
           [4, 12, 0],
           [0, 12, 1],
           [1, 12, 5],
           [5, 12, 8]],
          [[10, 12, 9],
           [9, 12, 5],
           [5, 12, 8],
           [8, 12, 11],
           [11, 12, 4],
           [4, 12, 0],
           [0, 12, 1],
           [1, 12, 6],
           [6, 12, 10]],
          [[5, 9, 8], [0, 4, 11], [0, 11, 6], [0, 6, 1], [6, 11, 10]],
          [[8, 10, 9],
           [8, 11, 10],
           [4, 11, 8],
           [4, 8, 5],
           [4, 5, 0],
           [0, 5, 1],
           [5, 6, 1],
           [5, 9, 6],
           [9, 10, 6]]],
         [[[5, 1, 0], [9, 8, 6], [4, 6, 8], [6, 4, 7]],
          [[5, 9, 8], [1, 0, 6], [4, 6, 0], [6, 4, 7]],
          [[5, 8, 0],
           [4, 0, 8],
           [0, 4, 7],
           [0, 7, 1],
           [6, 1, 7],
           [1, 6, 9],
           [1, 9, 5],
           [8, 5, 9]],
          [[12, 8, 5],
           [12, 5, 1],
           [12, 1, 0],
           [12, 0, 4],
           [12, 4, 7],
           [12, 7, 6],
           [12, 6, 9],
           [12, 9, 8]],
          [[12, 4, 7],
           [12, 7, 6],
           [12, 6, 1],
           [12, 1, 0],
           [12, 0, 5],
           [12, 5, 9],
           [12, 9, 8],
           [12, 8, 4]]],
         [[[1, 11, 9], [1, 0, 4], [1, 4, 11]]],
         [[[7, 10, 9], [0, 7, 9], [0, 9, 1], [4, 7, 0]]],
         [[[4, 11, 0], [0, 11, 1], [11, 10, 1], [1, 10, 6]]],
         [[[7, 6, 4], [1, 0, 4], [6, 1, 4]]],
         [[[7, 3, 2], [5, 1, 0]], [[7, 5, 2], [7, 3, 5], [3, 0, 5], [5, 1, 2]]],
         [[[1, 0, 5], [10, 11, 2], [2, 11, 3]],
          [[2, 1, 0],
           [2, 0, 3],
           [3, 0, 11],
           [11, 0, 5],
           [11, 5, 10],
           [1, 10, 5],
           [1, 2, 10]],
          [[5, 10, 11], [5, 11, 0], [0, 11, 3], [1, 10, 5], [10, 1, 2]]],
         [[[10, 6, 9], [5, 1, 0], [7, 3, 2]],
          [[9, 10, 7], [9, 7, 3], [9, 3, 2], [9, 2, 6], [5, 1, 0]],
          [[10, 5, 9], [10, 0, 5], [10, 1, 0], [10, 6, 1], [7, 3, 2]],
          [[7, 3, 0], [7, 0, 5], [7, 5, 2], [5, 1, 2], [10, 6, 9]],
          [[7, 12, 2],
           [2, 12, 1],
           [1, 12, 6],
           [6, 12, 10],
           [10, 12, 9],
           [9, 12, 5],
           [5, 12, 0],
           [0, 12, 3],
           [3, 12, 7]],
          [[2, 12, 1],
           [1, 12, 5],
           [5, 12, 0],
           [0, 12, 3],
           [3, 12, 7],
           [7, 12, 10],
           [10, 12, 9],
           [9, 12, 6],
           [6, 12, 2]],
          [[0, 12, 1],
           [1, 12, 6],
           [6, 12, 2],
           [2, 12, 3],
           [3, 12, 7],
           [7, 12, 10],
           [10, 12, 9],
           [9, 12, 5],
           [5, 12, 0]],
          [[6, 1, 2], [10, 7, 3], [10, 3, 5], [10, 5, 9], [5, 3, 0]],
          [[2, 0, 1],
           [2, 3, 0],
           [7, 3, 2],
           [7, 2, 6],
           [7, 6, 10],
           [10, 6, 9],
           [6, 5, 9],
           [6, 1, 5],
           [1, 0, 5]]],
         [[[1, 0, 5], [2, 6, 3], [9, 3, 6], [3, 9, 11]],
          [[1, 2, 6], [0, 5, 3], [9, 3, 5], [3, 9, 11]],
          [[1, 6, 5],
           [9, 5, 6],
           [5, 9, 11],
           [5, 11, 0],
           [3, 0, 11],
           [0, 3, 2],
           [0, 2, 1],
           [6, 1, 2]],
          [[12, 6, 1],
           [12, 1, 0],
           [12, 0, 5],
           [12, 5, 9],
           [12, 9, 11],
           [12, 11, 3],
           [12, 3, 2],
           [12, 2, 6]],
          [[12, 9, 11],
           [12, 11, 3],
           [12, 3, 0],
           [12, 0, 5],
           [12, 5, 1],
           [12, 1, 2],
           [12, 2, 6],
           [12, 6, 9]]],
         [[[3, 2, 7], [8, 9, 0], [0, 9, 1]],
          [[0, 3, 2],
           [0, 2, 1],
           [1, 2, 9],
           [9, 2, 7],
           [9, 7, 8],
           [3, 8, 7],
           [3, 0, 8]],
          [[7, 8, 9], [7, 9, 2], [2, 9, 1], [3, 8, 7], [8, 3, 0]]],
         [[[0, 9, 1], [0, 8, 9], [2, 10, 11], [2, 11, 3]],
          [[0, 11, 3], [0, 8, 11], [1, 10, 9], [1, 2, 10]],
          [[0, 8, 3],
           [3, 8, 11],
           [0, 3, 1],
           [1, 3, 2],
           [1, 2, 9],
           [2, 10, 9],
           [8, 9, 11],
           [9, 10, 11]],
          [[11, 12, 8],
           [8, 12, 0],
           [0, 12, 1],
           [1, 12, 9],
           [9, 12, 10],
           [10, 12, 2],
           [2, 12, 3],
           [3, 12, 11]],
          [[9, 12, 8],
           [1, 12, 9],
           [2, 12, 1],
           [10, 12, 2],
           [11, 12, 10],
           [3, 12, 11],
           [0, 12, 3],
           [8, 12, 0]]],
         [[[2, 7, 3], [6, 1, 10], [0, 10, 1], [10, 0, 8]],
          [[2, 6, 1], [7, 3, 10], [0, 10, 3], [10, 0, 8]],
          [[2, 1, 3],
           [0, 3, 1],
           [3, 0, 8],
           [3, 8, 7],
           [10, 7, 8],
           [7, 10, 6],
           [7, 6, 2],
           [1, 2, 6]],
          [[12, 1, 2],
           [12, 2, 7],
           [12, 7, 3],
           [12, 3, 0],
           [12, 0, 8],
           [12, 8, 10],
           [12, 10, 6],
           [12, 6, 1]],
          [[12, 0, 8],
           [12, 8, 10],
           [12, 10, 7],
           [12, 7, 3],
           [12, 3, 2],
           [12, 2, 6],
           [12, 6, 1],
           [12, 1, 0]]],
         [[[6, 1, 2], [3, 8, 11], [0, 8, 3]],
          [[1, 2, 3],
           [0, 1, 3],
           [8, 1, 0],
           [6, 1, 8],
           [11, 6, 8],
           [6, 11, 2],
           [11, 3, 2]],
          [[8, 11, 6], [1, 8, 6], [0, 8, 1], [6, 11, 2], [3, 2, 11]]],
         [[[8, 4, 11], [7, 3, 2], [5, 1, 0]],
          [[11, 8, 5], [11, 5, 1], [11, 1, 0], [11, 0, 4], [7, 3, 2]],
          [[8, 7, 11], [8, 2, 7], [8, 3, 2], [8, 4, 3], [5, 1, 0]],
          [[5, 1, 2], [5, 2, 7], [5, 7, 0], [7, 3, 0], [8, 4, 11]],
          [[5, 12, 0],
           [0, 12, 3],
           [3, 12, 4],
           [4, 12, 8],
           [8, 12, 11],
           [11, 12, 7],
           [7, 12, 2],
           [2, 12, 1],
           [1, 12, 5]],
          [[0, 12, 3],
           [3, 12, 7],
           [7, 12, 2],
           [2, 12, 1],
           [1, 12, 5],
           [5, 12, 8],
           [8, 12, 11],
           [11, 12, 4],
           [4, 12, 0]],
          [[2, 12, 3],
           [3, 12, 4],
           [4, 12, 0],
           [0, 12, 1],
           [1, 12, 5],
           [5, 12, 8],
           [8, 12, 11],
           [11, 12, 7],
           [7, 12, 2]],
          [[4, 3, 0], [8, 5, 1], [8, 1, 7], [8, 7, 11], [7, 1, 2]],
          [[0, 2, 3],
           [0, 1, 2],
           [5, 1, 0],
           [5, 0, 4],
           [5, 4, 8],
           [8, 4, 11],
           [4, 7, 11],
           [4, 3, 7],
           [3, 2, 7]]],
         [[[0, 5, 1], [4, 3, 8], [2, 8, 3], [8, 2, 10]],
          [[0, 4, 3], [5, 1, 8], [2, 8, 1], [8, 2, 10]],
          [[0, 3, 1],
           [2, 1, 3],
           [1, 2, 10],
           [1, 10, 5],
           [8, 5, 10],
           [5, 8, 4],
           [5, 4, 0],
           [3, 0, 4]],
          [[12, 3, 0],
           [12, 0, 5],
           [12, 5, 1],
           [12, 1, 2],
           [12, 2, 10],
           [12, 10, 8],
           [12, 8, 4],
           [12, 4, 3]],
          [[12, 2, 10],
           [12, 10, 8],
           [12, 8, 5],
           [12, 5, 1],
           [12, 1, 0],
           [12, 0, 4],
           [12, 4, 3],
           [12, 3, 2]]],
         [[[0, 5, 1], [9, 10, 6], [4, 11, 8], [2, 7, 3]],
          [[0, 5, 1], [2, 7, 3], [4, 11, 6], [6, 11, 10], [6, 8, 4], [6, 9, 8]],
          [[4, 11, 8], [2, 7, 3], [0, 5, 9], [0, 9, 10], [0, 6, 1], [0, 10, 6]],
          [[4, 11, 8], [9, 10, 6], [3, 5, 7], [0, 5, 3], [1, 2, 5], [2, 7, 5]],
          [[2, 7, 3], [9, 10, 6], [5, 11, 8], [1, 11, 5], [0, 4, 11], [0, 11, 1]],
          [[6, 9, 10], [0, 5, 1], [2, 11, 8], [2, 7, 11], [2, 8, 4], [2, 4, 3]],
          [[0, 5, 1], [4, 11, 8], [9, 7, 3], [9, 10, 7], [9, 3, 2], [9, 2, 6]],
          [[11, 8, 4],
           [7, 3, 12],
           [10, 7, 12],
           [9, 10, 12],
           [5, 9, 12],
           [0, 5, 12],
           [1, 0, 12],
           [6, 1, 12],
           [2, 6, 12],
           [3, 2, 12]],
          [[7, 3, 2],
           [12, 10, 6],
           [12, 6, 1],
           [12, 1, 0],
           [12, 0, 4],
           [12, 4, 11],
           [12, 11, 8],
           [12, 8, 5],
           [12, 5, 9],
           [12, 9, 10]],
          [[11, 8, 4],
           [9, 10, 12],
           [5, 9, 12],
           [0, 5, 12],
           [3, 0, 12],
           [7, 3, 12],
           [2, 7, 12],
           [1, 2, 12],
           [6, 1, 12],
           [10, 6, 12]],
          [[7, 3, 2],
           [1, 0, 12],
           [6, 1, 12],
           [10, 6, 12],
           [11, 10, 12],
           [4, 11, 12],
           [8, 4, 12],
           [9, 8, 12],
           [5, 9, 12],
           [0, 5, 12]],
          [[5, 1, 0],
           [12, 8, 4],
           [12, 4, 3],
           [12, 3, 2],
           [12, 2, 6],
           [12, 6, 9],
           [12, 9, 10],
           [12, 10, 7],
           [12, 7, 11],
           [12, 11, 8]],
          [[11, 8, 4],
           [12, 9, 10],
           [12, 10, 7],
           [12, 7, 3],
           [12, 3, 0],
           [12, 0, 5],
           [12, 5, 1],
           [12, 1, 2],
           [12, 2, 6],
           [12, 6, 9]],
          [[1, 0, 5],
           [12, 3, 2],
           [12, 2, 6],
           [12, 6, 9],
           [12, 9, 8],
           [12, 8, 4],
           [12, 4, 11],
           [12, 11, 10],
           [12, 10, 7],
           [12, 7, 3]],
          [[6, 9, 10],
           [5, 1, 12],
           [8, 5, 12],
           [11, 8, 12],
           [7, 11, 12],
           [2, 7, 12],
           [3, 2, 12],
           [4, 3, 12],
           [0, 4, 12],
           [1, 0, 12]],
          [[6, 9, 10],
           [11, 8, 12],
           [7, 11, 12],
           [2, 7, 12],
           [1, 2, 12],
           [5, 1, 12],
           [0, 5, 12],
           [3, 0, 12],
           [4, 3, 12],
           [8, 4, 12]],
          [[1, 0, 5],
           [3, 2, 12],
           [4, 3, 12],
           [8, 4, 12],
           [9, 8, 12],
           [6, 9, 12],
           [10, 6, 12],
           [11, 10, 12],
           [7, 11, 12],
           [2, 7, 12]],
          [[6, 9, 10],
           [12, 11, 8],
           [12, 8, 5],
           [12, 5, 1],
           [12, 1, 2],
           [12, 2, 7],
           [12, 7, 3],
           [12, 3, 0],
           [12, 0, 4],
           [12, 4, 11]],
          [[7, 3, 2],
           [12, 1, 0],
           [12, 0, 4],
           [12, 4, 11],
           [12, 11, 10],
           [12, 10, 6],
           [12, 6, 9],
           [12, 9, 8],
           [12, 8, 5],
           [12, 5, 1]],
          [[7, 3, 12],
           [10, 7, 12],
           [11, 10, 12],
           [4, 11, 12],
           [8, 4, 12],
           [9, 8, 12],
           [5, 9, 12],
           [0, 5, 12],
           [1, 0, 12],
           [6, 1, 12],
           [2, 6, 12],
           [3, 2, 12]],
          [[12, 0, 4],
           [12, 4, 11],
           [12, 11, 8],
           [12, 8, 5],
           [12, 5, 9],
           [12, 9, 10],
           [12, 10, 6],
           [12, 6, 1],
           [12, 1, 2],
           [12, 2, 7],
           [12, 7, 3],
           [12, 3, 0]],
          [[12, 2, 6],
           [12, 6, 9],
           [12, 9, 10],
           [12, 10, 7],
           [12, 7, 11],
           [12, 11, 8],
           [12, 8, 4],
           [12, 4, 3],
           [12, 3, 0],
           [12, 0, 5],
           [12, 5, 1],
           [12, 1, 2]],
          [[5, 1, 12],
           [8, 5, 12],
           [9, 8, 12],
           [6, 9, 12],
           [10, 6, 12],
           [11, 10, 12],
           [7, 11, 12],
           [2, 7, 12],
           [3, 2, 12],
           [4, 3, 12],
           [0, 4, 12],
           [1, 0, 12]],
          [[4, 11, 8], [6, 1, 2], [3, 0, 7], [10, 7, 0], [5, 10, 0], [10, 5, 9]],
          [[7, 3, 2], [5, 9, 8], [4, 1, 0], [1, 4, 11], [1, 11, 6], [10, 6, 11]],
          [[5, 1, 0], [7, 11, 10], [6, 3, 2], [3, 6, 9], [3, 9, 4], [8, 4, 9]],
          [[6, 9, 10], [4, 3, 0], [1, 2, 5], [8, 5, 2], [7, 8, 2], [8, 7, 11]],
          [[6, 1, 2],
           [5, 9, 8],
           [0, 5, 8],
           [4, 0, 8],
           [0, 4, 3],
           [7, 3, 4],
           [11, 7, 4],
           [7, 11, 10],
           [8, 10, 11],
           [10, 8, 9]],
          [[5, 9, 8],
           [3, 0, 4],
           [3, 4, 11],
           [3, 11, 7],
           [10, 7, 11],
           [7, 10, 6],
           [7, 6, 2],
           [1, 2, 6],
           [2, 1, 3],
           [0, 3, 1]],
          [[7, 11, 10],
           [1, 2, 6],
           [1, 6, 9],
           [1, 9, 5],
           [8, 5, 9],
           [5, 8, 4],
           [5, 4, 0],
           [3, 0, 4],
           [0, 3, 1],
           [2, 1, 3]],
          [[4, 3, 0],
           [7, 11, 10],
           [2, 7, 10],
           [6, 2, 10],
           [2, 6, 1],
           [5, 1, 6],
           [9, 5, 6],
           [8, 5, 9],
           [10, 8, 9],
           [8, 10, 11]]],
         [[[8, 5, 9], [3, 0, 4], [1, 2, 6]],
          [[6, 9, 8], [2, 6, 8], [1, 2, 8], [5, 1, 8], [3, 0, 4]],
          [[8, 4, 9], [4, 3, 9], [3, 0, 9], [0, 5, 9], [1, 2, 6]],
          [[3, 2, 6], [4, 3, 6], [1, 4, 6], [1, 0, 4], [8, 5, 9]],
          [[1, 12, 6],
           [0, 12, 1],
           [5, 12, 0],
           [9, 12, 5],
           [8, 12, 9],
           [4, 12, 8],
           [3, 12, 4],
           [2, 12, 3],
           [6, 12, 2]],
          [[0, 12, 1],
           [4, 12, 0],
           [3, 12, 4],
           [2, 12, 3],
           [6, 12, 2],
           [9, 12, 6],
           [8, 12, 9],
           [5, 12, 8],
           [1, 12, 5]],
          [[0, 12, 3],
           [5, 12, 0],
           [1, 12, 5],
           [2, 12, 1],
           [6, 12, 2],
           [9, 12, 6],
           [8, 12, 9],
           [4, 12, 8],
           [3, 12, 4]],
          [[1, 0, 5], [2, 6, 9], [4, 2, 9], [8, 4, 9], [3, 2, 4]],
          [[0, 3, 1],
           [3, 2, 1],
           [1, 2, 6],
           [5, 1, 6],
           [9, 5, 6],
           [8, 5, 9],
           [8, 4, 5],
           [4, 0, 5],
           [4, 3, 0]]],
         [[[3, 2, 7], [0, 4, 1], [11, 1, 4], [1, 11, 9]],
          [[3, 0, 4], [2, 7, 1], [11, 1, 7], [1, 11, 9]],
          [[3, 4, 7],
           [11, 7, 4],
           [7, 11, 9],
           [7, 9, 2],
           [1, 2, 9],
           [2, 1, 0],
           [2, 0, 3],
           [4, 3, 0]],
          [[12, 4, 3],
           [12, 3, 2],
           [12, 2, 7],
           [12, 7, 11],
           [12, 11, 9],
           [12, 9, 1],
           [12, 1, 0],
           [12, 0, 4]],
          [[12, 11, 9],
           [12, 9, 1],
           [12, 1, 2],
           [12, 2, 7],
           [12, 7, 3],
           [12, 3, 0],
           [12, 0, 4],
           [12, 4, 11]]],
         [[[4, 3, 0], [1, 10, 9], [2, 10, 1]],
          [[3, 0, 1],
           [2, 3, 1],
           [10, 3, 2],
           [4, 3, 10],
           [9, 4, 10],
           [4, 9, 0],
           [9, 1, 0]],
          [[10, 9, 4], [3, 10, 4], [2, 10, 3], [4, 9, 0], [1, 0, 9]]],
         [[[4, 3, 0], [10, 7, 11], [2, 6, 1]],
          [[1, 0, 4], [6, 1, 4], [2, 6, 4], [3, 2, 4], [10, 7, 11]],
          [[4, 11, 0], [11, 10, 0], [10, 7, 0], [7, 3, 0], [2, 6, 1]],
          [[10, 6, 1], [11, 10, 1], [2, 11, 1], [2, 7, 11], [4, 3, 0]],
          [[2, 12, 1],
           [7, 12, 2],
           [3, 12, 7],
           [0, 12, 3],
           [4, 12, 0],
           [11, 12, 4],
           [10, 12, 11],
           [6, 12, 10],
           [1, 12, 6]],
          [[7, 12, 2],
           [11, 12, 7],
           [10, 12, 11],
           [6, 12, 10],
           [1, 12, 6],
           [0, 12, 1],
           [4, 12, 0],
           [3, 12, 4],
           [2, 12, 3]],
          [[7, 12, 10],
           [3, 12, 7],
           [2, 12, 3],
           [6, 12, 2],
           [1, 12, 6],
           [0, 12, 1],
           [4, 12, 0],
           [11, 12, 4],
           [10, 12, 11]],
          [[2, 7, 3], [6, 1, 0], [11, 6, 0], [4, 11, 0], [10, 6, 11]],
          [[7, 10, 2],
           [10, 6, 2],
           [2, 6, 1],
           [3, 2, 1],
           [0, 3, 1],
           [4, 3, 0],
           [4, 11, 3],
           [11, 7, 3],
           [11, 10, 7]]],
         [[[3, 0, 4], [1, 2, 6]], [[3, 6, 4], [6, 0, 4], [6, 1, 0], [3, 2, 6]]],
         [[[6, 2, 5], [5, 2, 0]]],
         [[[7, 10, 11], [0, 5, 2], [2, 5, 6]],
          [[2, 7, 10],
           [2, 10, 6],
           [6, 10, 5],
           [5, 10, 11],
           [5, 11, 0],
           [7, 0, 11],
           [7, 2, 0]],
          [[11, 0, 5], [11, 5, 10], [10, 5, 6], [7, 0, 11], [0, 7, 2]]],
         [[[0, 10, 2], [0, 5, 9], [0, 9, 10]]],
         [[[11, 7, 2], [5, 11, 2], [5, 2, 0], [9, 11, 5]]],
         [[[8, 2, 0], [8, 9, 6], [8, 6, 2]]],
         [[[10, 11, 7], [9, 6, 8], [2, 8, 6], [8, 2, 0]],
          [[10, 9, 6], [11, 7, 8], [2, 8, 7], [8, 2, 0]],
          [[10, 6, 7],
           [2, 7, 6],
           [7, 2, 0],
           [7, 0, 11],
           [8, 11, 0],
           [11, 8, 9],
           [11, 9, 10],
           [6, 10, 9]],
          [[12, 6, 10],
           [12, 10, 11],
           [12, 11, 7],
           [12, 7, 2],
           [12, 2, 0],
           [12, 0, 8],
           [12, 8, 9],
           [12, 9, 6]],
          [[12, 2, 0],
           [12, 0, 8],
           [12, 8, 11],
           [12, 11, 7],
           [12, 7, 10],
           [12, 10, 9],
           [12, 9, 6],
           [12, 6, 2]]],
         [[[8, 2, 0], [8, 10, 2]]],
         [[[0, 8, 2], [11, 7, 2], [8, 11, 2]]],
         [[[8, 4, 11], [6, 2, 5], [5, 2, 0]],
          [[5, 8, 4],
           [5, 4, 0],
           [0, 4, 2],
           [2, 4, 11],
           [2, 11, 6],
           [8, 6, 11],
           [8, 5, 6]],
          [[11, 6, 2], [11, 2, 4], [4, 2, 0], [8, 6, 11], [6, 8, 5]]],
         [[[10, 4, 7], [10, 8, 4], [2, 0, 5], [2, 5, 6]],
          [[10, 5, 6], [10, 8, 5], [7, 0, 4], [7, 2, 0]],
          [[10, 8, 6],
           [6, 8, 5],
           [10, 6, 7],
           [7, 6, 2],
           [7, 2, 4],
           [2, 0, 4],
           [8, 4, 5],
           [4, 0, 5]],
          [[5, 12, 8],
           [8, 12, 10],
           [10, 12, 7],
           [7, 12, 4],
           [4, 12, 0],
           [0, 12, 2],
           [2, 12, 6],
           [6, 12, 5]],
          [[4, 12, 8],
           [7, 12, 4],
           [2, 12, 7],
           [0, 12, 2],
           [5, 12, 0],
           [6, 12, 5],
           [10, 12, 6],
           [8, 12, 10]]],
         [[[8, 4, 11], [5, 9, 0], [10, 0, 9], [0, 10, 2]],
          [[8, 5, 9], [4, 11, 0], [10, 0, 11], [0, 10, 2]],
          [[8, 9, 11],
           [10, 11, 9],
           [11, 10, 2],
           [11, 2, 4],
           [0, 4, 2],
           [4, 0, 5],
           [4, 5, 8],
           [9, 8, 5]],
          [[12, 9, 8],
           [12, 8, 4],
           [12, 4, 11],
           [12, 11, 10],
           [12, 10, 2],
           [12, 2, 0],
           [12, 0, 5],
           [12, 5, 9]],
          [[12, 10, 2],
           [12, 2, 0],
           [12, 0, 4],
           [12, 4, 11],
           [12, 11, 8],
           [12, 8, 5],
           [12, 5, 9],
           [12, 9, 10]]],
         [[[9, 8, 5], [0, 7, 2], [4, 7, 0]],
          [[8, 5, 0],
           [4, 8, 0],
           [7, 8, 4],
           [9, 8, 7],
           [2, 9, 7],
           [9, 2, 5],
           [2, 0, 5]],
          [[7, 2, 9], [8, 7, 9], [4, 7, 8], [9, 2, 5], [0, 5, 2]]],
         [[[9, 6, 2], [4, 9, 2], [4, 2, 0], [4, 11, 9]]],
         [[[9, 6, 10], [7, 0, 4], [2, 0, 7]],
          [[6, 10, 7],
           [2, 6, 7],
           [0, 6, 2],
           [9, 6, 0],
           [4, 9, 0],
           [9, 4, 10],
           [4, 7, 10]],
          [[0, 4, 9], [6, 0, 9], [2, 0, 6], [9, 4, 10], [7, 10, 4]]],
         [[[0, 10, 2], [0, 4, 11], [0, 11, 10]]],
         [[[7, 2, 4], [4, 2, 0]]],
         [[[7, 5, 6], [7, 3, 0], [7, 0, 5]]],
         [[[6, 10, 11], [0, 6, 11], [0, 11, 3], [0, 5, 6]]],
         [[[0, 5, 3], [3, 5, 7], [5, 9, 7], [7, 9, 10]]],
         [[[11, 3, 9], [0, 5, 9], [3, 0, 9]]],
         [[[8, 9, 6], [3, 8, 6], [3, 6, 7], [0, 8, 3]]],
         [[[6, 10, 9], [8, 3, 0], [11, 3, 8]],
          [[10, 9, 8],
           [11, 10, 8],
           [3, 10, 11],
           [6, 10, 3],
           [0, 6, 3],
           [6, 0, 9],
           [0, 8, 9]],
          [[3, 0, 6], [10, 3, 6], [11, 3, 10], [6, 0, 9], [8, 9, 0]]],
         [[[8, 10, 0], [7, 3, 0], [10, 7, 0]]],
         [[[3, 0, 11], [11, 0, 8]]],
         [[[4, 11, 8], [3, 0, 7], [5, 7, 0], [7, 5, 6]],
          [[4, 3, 0], [11, 8, 7], [5, 7, 8], [7, 5, 6]],
          [[4, 0, 8],
           [5, 8, 0],
           [8, 5, 6],
           [8, 6, 11],
           [7, 11, 6],
           [11, 7, 3],
           [11, 3, 4],
           [0, 4, 3]],
          [[12, 0, 4],
           [12, 4, 11],
           [12, 11, 8],
           [12, 8, 5],
           [12, 5, 6],
           [12, 6, 7],
           [12, 7, 3],
           [12, 3, 0]],
          [[12, 5, 6],
           [12, 6, 7],
           [12, 7, 11],
           [12, 11, 8],
           [12, 8, 4],
           [12, 4, 3],
           [12, 3, 0],
           [12, 0, 5]]],
         [[[3, 0, 4], [8, 6, 10], [5, 6, 8]],
          [[0, 4, 8],
           [5, 0, 8],
           [6, 0, 5],
           [3, 0, 6],
           [10, 3, 6],
           [3, 10, 4],
           [10, 8, 4]],
          [[6, 10, 3], [0, 6, 3], [5, 6, 0], [3, 10, 4], [8, 4, 10]]],
         [[[0, 4, 3], [9, 8, 5], [11, 10, 7]],
          [[7, 3, 0], [10, 7, 0], [11, 10, 0], [4, 11, 0], [9, 8, 5]],
          [[0, 5, 3], [5, 9, 3], [9, 8, 3], [8, 4, 3], [11, 10, 7]],
          [[9, 10, 7], [5, 9, 7], [11, 5, 7], [11, 8, 5], [0, 4, 3]],
          [[11, 12, 7],
           [8, 12, 11],
           [4, 12, 8],
           [3, 12, 4],
           [0, 12, 3],
           [5, 12, 0],
           [9, 12, 5],
           [10, 12, 9],
           [7, 12, 10]],
          [[8, 12, 11],
           [5, 12, 8],
           [9, 12, 5],
           [10, 12, 9],
           [7, 12, 10],
           [3, 12, 7],
           [0, 12, 3],
           [4, 12, 0],
           [11, 12, 4]],
          [[8, 12, 9],
           [4, 12, 8],
           [11, 12, 4],
           [10, 12, 11],
           [7, 12, 10],
           [3, 12, 7],
           [0, 12, 3],
           [5, 12, 0],
           [9, 12, 5]],
          [[11, 8, 4], [10, 7, 3], [5, 10, 3], [0, 5, 3], [9, 10, 5]],
          [[8, 9, 11],
           [9, 10, 11],
           [11, 10, 7],
           [4, 11, 7],
           [3, 4, 7],
           [0, 4, 3],
           [0, 5, 4],
           [5, 8, 4],
           [5, 9, 8]]],
         [[[8, 5, 9], [0, 4, 3]], [[8, 3, 9], [3, 5, 9], [3, 0, 5], [8, 4, 3]]],
         [[[0, 4, 3], [7, 9, 6], [11, 9, 7]],
          [[4, 3, 7],
           [11, 4, 7],
           [9, 4, 11],
           [0, 4, 9],
           [6, 0, 9],
           [0, 6, 3],
           [6, 7, 3]],
          [[9, 6, 0], [4, 9, 0], [11, 9, 4], [0, 6, 3], [7, 3, 6]]],
         [[[6, 10, 9], [4, 3, 0]],
          [[4, 3, 10], [6, 10, 3], [3, 0, 6], [9, 6, 0], [0, 4, 9], [9, 10, 4]]],
         [[[0, 4, 3], [10, 7, 11]], [[0, 10, 3], [0, 4, 10], [4, 11, 10], [10, 7, 3]]],
         [[[3, 0, 4]]],
         [[[4, 0, 3]]],
         [[[0, 3, 4], [10, 11, 7]], [[0, 10, 4], [0, 3, 10], [3, 7, 10], [10, 11, 4]]],
         [[[4, 0, 3], [10, 6, 9]],
          [[0, 6, 9], [6, 0, 3], [3, 10, 6], [10, 3, 4], [4, 9, 10], [9, 0, 4]]],
         [[[3, 4, 0], [6, 9, 7], [7, 9, 11]],
          [[7, 3, 4],
           [7, 4, 11],
           [11, 4, 9],
           [9, 4, 0],
           [9, 0, 6],
           [3, 6, 0],
           [3, 7, 6]],
          [[0, 6, 9], [0, 9, 4], [4, 9, 11], [3, 6, 0], [6, 3, 7]]],
         [[[3, 4, 0], [9, 5, 8]], [[3, 9, 0], [3, 4, 9], [4, 8, 9], [9, 5, 0]]],
         [[[3, 4, 0], [5, 8, 9], [7, 10, 11]],
          [[0, 3, 7], [0, 7, 10], [0, 10, 11], [0, 11, 4], [5, 8, 9]],
          [[3, 5, 0], [3, 9, 5], [3, 8, 9], [3, 4, 8], [7, 10, 11]],
          [[7, 10, 9], [7, 9, 5], [7, 5, 11], [5, 8, 11], [3, 4, 0]],
          [[7, 12, 11],
           [11, 12, 8],
           [8, 12, 4],
           [4, 12, 3],
           [3, 12, 0],
           [0, 12, 5],
           [5, 12, 9],
           [9, 12, 10],
           [10, 12, 7]],
          [[11, 12, 8],
           [8, 12, 5],
           [5, 12, 9],
           [9, 12, 10],
           [10, 12, 7],
           [7, 12, 3],
           [3, 12, 0],
           [0, 12, 4],
           [4, 12, 11]],
          [[9, 12, 8],
           [8, 12, 4],
           [4, 12, 11],
           [11, 12, 10],
           [10, 12, 7],
           [7, 12, 3],
           [3, 12, 0],
           [0, 12, 5],
           [5, 12, 9]],
          [[4, 8, 11], [3, 7, 10], [3, 10, 5], [3, 5, 0], [5, 10, 9]],
          [[11, 9, 8],
           [11, 10, 9],
           [7, 10, 11],
           [7, 11, 4],
           [7, 4, 3],
           [3, 4, 0],
           [4, 5, 0],
           [4, 8, 5],
           [8, 9, 5]]],
         [[[4, 0, 3], [10, 6, 8], [8, 6, 5]],
          [[8, 4, 0],
           [8, 0, 5],
           [5, 0, 6],
           [6, 0, 3],
           [6, 3, 10],
           [4, 10, 3],
           [4, 8, 10]],
          [[3, 10, 6], [3, 6, 0], [0, 6, 5], [4, 10, 3], [10, 4, 8]]],
         [[[4, 0, 3], [8, 11, 5], [7, 5, 11], [5, 7, 6]],
          [[4, 8, 11], [0, 3, 5], [7, 5, 3], [5, 7, 6]],
          [[4, 11, 3],
           [7, 3, 11],
           [3, 7, 6],
           [3, 6, 0],
           [5, 0, 6],
           [0, 5, 8],
           [0, 8, 4],
           [11, 4, 8]],
          [[12, 11, 4],
           [12, 4, 0],
           [12, 0, 3],
           [12, 3, 7],
           [12, 7, 6],
           [12, 6, 5],
           [12, 5, 8],
           [12, 8, 11]],
          [[12, 7, 6],
           [12, 6, 5],
           [12, 5, 0],
           [12, 0, 3],
           [12, 3, 4],
           [12, 4, 8],
           [12, 8, 11],
           [12, 11, 7]]],
         [[[0, 3, 8], [8, 3, 11]]],
         [[[0, 10, 8], [0, 3, 7], [0, 7, 10]]],
         [[[9, 10, 6], [0, 3, 8], [8, 3, 11]],
          [[8, 9, 10],
           [8, 10, 11],
           [11, 10, 3],
           [3, 10, 6],
           [3, 6, 0],
           [9, 0, 6],
           [9, 8, 0]],
          [[6, 0, 3], [6, 3, 10], [10, 3, 11], [9, 0, 6], [0, 9, 8]]],
         [[[6, 9, 8], [3, 6, 8], [3, 8, 0], [7, 6, 3]]],
         [[[9, 3, 11], [9, 5, 0], [9, 0, 3]]],
         [[[0, 3, 5], [5, 3, 9], [3, 7, 9], [9, 7, 10]]],
         [[[11, 10, 6], [0, 11, 6], [0, 6, 5], [0, 3, 11]]],
         [[[5, 7, 6], [5, 0, 3], [5, 3, 7]]],
         [[[2, 7, 0], [0, 7, 4]]],
         [[[10, 0, 2], [10, 11, 4], [10, 4, 0]]],
         [[[10, 6, 9], [4, 0, 7], [7, 0, 2]],
          [[7, 10, 6],
           [7, 6, 2],
           [2, 6, 0],
           [0, 6, 9],
           [0, 9, 4],
           [10, 4, 9],
           [10, 7, 4]],
          [[9, 4, 0], [9, 0, 6], [6, 0, 2], [10, 4, 9], [4, 10, 7]]],
         [[[11, 4, 0], [6, 11, 0], [6, 0, 2], [6, 9, 11]]],
         [[[5, 8, 9], [2, 7, 0], [0, 7, 4]],
          [[0, 5, 8],
           [0, 8, 4],
           [4, 8, 7],
           [7, 8, 9],
           [7, 9, 2],
           [5, 2, 9],
           [5, 0, 2]],
          [[9, 2, 7], [9, 7, 8], [8, 7, 4], [5, 2, 9], [2, 5, 0]]],
         [[[8, 9, 5], [11, 4, 10], [0, 10, 4], [10, 0, 2]],
          [[8, 11, 4], [9, 5, 10], [0, 10, 5], [10, 0, 2]],
          [[8, 4, 5],
           [0, 5, 4],
           [5, 0, 2],
           [5, 2, 9],
           [10, 9, 2],
           [9, 10, 11],
           [9, 11, 8],
           [4, 8, 11]],
          [[12, 4, 8],
           [12, 8, 9],
           [12, 9, 5],
           [12, 5, 0],
           [12, 0, 2],
           [12, 2, 10],
           [12, 10, 11],
           [12, 11, 4]],
          [[12, 0, 2],
           [12, 2, 10],
           [12, 10, 9],
           [12, 9, 5],
           [12, 5, 8],
           [12, 8, 11],
           [12, 11, 4],
           [12, 4, 0]]],
         [[[6, 8, 10], [6, 5, 8], [7, 4, 0], [7, 0, 2]],
          [[6, 0, 2], [6, 5, 0], [10, 4, 8], [10, 7, 4]],
          [[6, 5, 2],
           [2, 5, 0],
           [6, 2, 10],
           [10, 2, 7],
           [10, 7, 8],
           [7, 4, 8],
           [5, 8, 0],
           [8, 4, 0]],
          [[0, 12, 5],
           [5, 12, 6],
           [6, 12, 10],
           [10, 12, 8],
           [8, 12, 4],
           [4, 12, 7],
           [7, 12, 2],
           [2, 12, 0]],
          [[8, 12, 5],
           [10, 12, 8],
           [7, 12, 10],
           [4, 12, 7],
           [0, 12, 4],
           [2, 12, 0],
           [6, 12, 2],
           [5, 12, 6]]],
         [[[11, 4, 8], [5, 2, 6], [0, 2, 5]],
          [[4, 8, 5],
           [0, 4, 5],
           [2, 4, 0],
           [11, 4, 2],
           [6, 11, 2],
           [11, 6, 8],
           [6, 5, 8]],
          [[2, 6, 11], [4, 2, 11], [0, 2, 4], [11, 6, 8], [5, 8, 6]]],
         [[[2, 8, 0], [2, 7, 11], [2, 11, 8]]],
         [[[10, 0, 2], [10, 8, 0]]],
         [[[10, 6, 9], [7, 11, 2], [8, 2, 11], [2, 8, 0]],
          [[10, 7, 11], [6, 9, 2], [8, 2, 9], [2, 8, 0]],
          [[10, 11, 9],
           [8, 9, 11],
           [9, 8, 0],
           [9, 0, 6],
           [2, 6, 0],
           [6, 2, 7],
           [6, 7, 10],
           [11, 10, 7]],
          [[12, 11, 10],
           [12, 10, 6],
           [12, 6, 9],
           [12, 9, 8],
           [12, 8, 0],
           [12, 0, 2],
           [12, 2, 7],
           [12, 7, 11]],
          [[12, 8, 0],
           [12, 0, 2],
           [12, 2, 6],
           [12, 6, 9],
           [12, 9, 10],
           [12, 10, 7],
           [12, 7, 11],
           [12, 11, 8]]],
         [[[2, 8, 0], [2, 6, 9], [2, 9, 8]]],
         [[[9, 5, 0], [7, 9, 0], [7, 0, 2], [11, 9, 7]]],
         [[[2, 10, 0], [9, 5, 0], [10, 9, 0]]],
         [[[10, 7, 11], [5, 0, 6], [6, 0, 2]],
          [[6, 10, 7],
           [6, 7, 2],
           [2, 7, 0],
           [0, 7, 11],
           [0, 11, 5],
           [10, 5, 11],
           [10, 6, 5]],
          [[11, 5, 0], [11, 0, 7], [7, 0, 2], [10, 5, 11], [5, 10, 6]]],
         [[[2, 5, 0], [6, 5, 2]]],
         [[[6, 2, 1], [4, 0, 3]], [[6, 4, 1], [6, 2, 4], [2, 3, 4], [4, 0, 1]]],
         [[[11, 7, 10], [6, 2, 1], [4, 0, 3]],
          [[10, 11, 4], [10, 4, 0], [10, 0, 3], [10, 3, 7], [6, 2, 1]],
          [[11, 6, 10], [11, 1, 6], [11, 2, 1], [11, 7, 2], [4, 0, 3]],
          [[4, 0, 1], [4, 1, 6], [4, 6, 3], [6, 2, 3], [11, 7, 10]],
          [[4, 12, 3],
           [3, 12, 2],
           [2, 12, 7],
           [7, 12, 11],
           [11, 12, 10],
           [10, 12, 6],
           [6, 12, 1],
           [1, 12, 0],
           [0, 12, 4]],
          [[3, 12, 2],
           [2, 12, 6],
           [6, 12, 1],
           [1, 12, 0],
           [0, 12, 4],
           [4, 12, 11],
           [11, 12, 10],
           [10, 12, 7],
           [7, 12, 3]],
          [[1, 12, 2],
           [2, 12, 7],
           [7, 12, 3],
           [3, 12, 0],
           [0, 12, 4],
           [4, 12, 11],
           [11, 12, 10],
           [10, 12, 6],
           [6, 12, 1]],
          [[7, 2, 3], [11, 4, 0], [11, 0, 6], [11, 6, 10], [6, 0, 1]],
          [[3, 1, 2],
           [3, 0, 1],
           [4, 0, 3],
           [4, 3, 7],
           [4, 7, 11],
           [11, 7, 10],
           [7, 6, 10],
           [7, 2, 6],
           [2, 1, 6]]],
         [[[0, 3, 4], [9, 10, 1], [1, 10, 2]],
          [[1, 0, 3],
           [1, 3, 2],
           [2, 3, 10],
           [10, 3, 4],
           [10, 4, 9],
           [0, 9, 4],
           [0, 1, 9]],
          [[4, 9, 10], [4, 10, 3], [3, 10, 2], [0, 9, 4], [9, 0, 1]]],
         [[[3, 4, 0], [7, 2, 11], [1, 11, 2], [11, 1, 9]],
          [[3, 7, 2], [4, 0, 11], [1, 11, 0], [11, 1, 9]],
          [[3, 2, 0],
           [1, 0, 2],
           [0, 1, 9],
           [0, 9, 4],
           [11, 4, 9],
           [4, 11, 7],
           [4, 7, 3],
           [2, 3, 7]],
          [[12, 2, 3],
           [12, 3, 4],
           [12, 4, 0],
           [12, 0, 1],
           [12, 1, 9],
           [12, 9, 11],
           [12, 11, 7],
           [12, 7, 2]],
          [[12, 1, 9],
           [12, 9, 11],
           [12, 11, 4],
           [12, 4, 0],
           [12, 0, 3],
           [12, 3, 7],
           [12, 7, 2],
           [12, 2, 1]]],
         [[[9, 5, 8], [4, 0, 3], [6, 2, 1]],
          [[8, 9, 6], [8, 6, 2], [8, 2, 1], [8, 1, 5], [4, 0, 3]],
          [[9, 4, 8], [9, 3, 4], [9, 0, 3], [9, 5, 0], [6, 2, 1]],
          [[6, 2, 3], [6, 3, 4], [6, 4, 1], [4, 0, 1], [9, 5, 8]],
          [[6, 12, 1],
           [1, 12, 0],
           [0, 12, 5],
           [5, 12, 9],
           [9, 12, 8],
           [8, 12, 4],
           [4, 12, 3],
           [3, 12, 2],
           [2, 12, 6]],
          [[1, 12, 0],
           [0, 12, 4],
           [4, 12, 3],
           [3, 12, 2],
           [2, 12, 6],
           [6, 12, 9],
           [9, 12, 8],
           [8, 12, 5],
           [5, 12, 1]],
          [[3, 12, 0],
           [0, 12, 5],
           [5, 12, 1],
           [1, 12, 2],
           [2, 12, 6],
           [6, 12, 9],
           [9, 12, 8],
           [8, 12, 4],
           [4, 12, 3]],
          [[5, 0, 1], [9, 6, 2], [9, 2, 4], [9, 4, 8], [4, 2, 3]],
          [[1, 3, 0],
           [1, 2, 3],
           [6, 2, 1],
           [6, 1, 5],
           [6, 5, 9],
           [9, 5, 8],
           [5, 4, 8],
           [5, 0, 4],
           [0, 3, 4]]],
         [[[3, 4, 0], [8, 9, 5], [7, 10, 11], [1, 6, 2]],
          [[3, 4, 0], [1, 6, 2], [7, 10, 5], [5, 10, 9], [5, 11, 7], [5, 8, 11]],
          [[7, 10, 11], [1, 6, 2], [3, 4, 8], [3, 8, 9], [3, 5, 0], [3, 9, 5]],
          [[7, 10, 11], [8, 9, 5], [2, 4, 6], [3, 4, 2], [0, 1, 4], [1, 6, 4]],
          [[1, 6, 2], [8, 9, 5], [4, 10, 11], [0, 10, 4], [3, 7, 10], [3, 10, 0]],
          [[5, 8, 9], [3, 4, 0], [1, 10, 11], [1, 6, 10], [1, 11, 7], [1, 7, 2]],
          [[3, 4, 0], [7, 10, 11], [8, 6, 2], [8, 9, 6], [8, 2, 1], [8, 1, 5]],
          [[10, 11, 7],
           [6, 2, 12],
           [9, 6, 12],
           [8, 9, 12],
           [4, 8, 12],
           [3, 4, 12],
           [0, 3, 12],
           [5, 0, 12],
           [1, 5, 12],
           [2, 1, 12]],
          [[6, 2, 1],
           [12, 9, 5],
           [12, 5, 0],
           [12, 0, 3],
           [12, 3, 7],
           [12, 7, 10],
           [12, 10, 11],
           [12, 11, 4],
           [12, 4, 8],
           [12, 8, 9]],
          [[10, 11, 7],
           [8, 9, 12],
           [4, 8, 12],
           [3, 4, 12],
           [2, 3, 12],
           [6, 2, 12],
           [1, 6, 12],
           [0, 1, 12],
           [5, 0, 12],
           [9, 5, 12]],
          [[6, 2, 1],
           [0, 3, 12],
           [5, 0, 12],
           [9, 5, 12],
           [10, 9, 12],
           [7, 10, 12],
           [11, 7, 12],
           [8, 11, 12],
           [4, 8, 12],
           [3, 4, 12]],
          [[4, 0, 3],
           [12, 11, 7],
           [12, 7, 2],
           [12, 2, 1],
           [12, 1, 5],
           [12, 5, 8],
           [12, 8, 9],
           [12, 9, 6],
           [12, 6, 10],
           [12, 10, 11]],
          [[10, 11, 7],
           [12, 8, 9],
           [12, 9, 6],
           [12, 6, 2],
           [12, 2, 3],
           [12, 3, 4],
           [12, 4, 0],
           [12, 0, 1],
           [12, 1, 5],
           [12, 5, 8]],
          [[0, 3, 4],
           [12, 2, 1],
           [12, 1, 5],
           [12, 5, 8],
           [12, 8, 11],
           [12, 11, 7],
           [12, 7, 10],
           [12, 10, 9],
           [12, 9, 6],
           [12, 6, 2]],
          [[5, 8, 9],
           [4, 0, 12],
           [11, 4, 12],
           [10, 11, 12],
           [6, 10, 12],
           [1, 6, 12],
           [2, 1, 12],
           [7, 2, 12],
           [3, 7, 12],
           [0, 3, 12]],
          [[5, 8, 9],
           [10, 11, 12],
           [6, 10, 12],
           [1, 6, 12],
           [0, 1, 12],
           [4, 0, 12],
           [3, 4, 12],
           [2, 3, 12],
           [7, 2, 12],
           [11, 7, 12]],
          [[0, 3, 4],
           [2, 1, 12],
           [7, 2, 12],
           [11, 7, 12],
           [8, 11, 12],
           [5, 8, 12],
           [9, 5, 12],
           [10, 9, 12],
           [6, 10, 12],
           [1, 6, 12]],
          [[5, 8, 9],
           [12, 10, 11],
           [12, 11, 4],
           [12, 4, 0],
           [12, 0, 1],
           [12, 1, 6],
           [12, 6, 2],
           [12, 2, 3],
           [12, 3, 7],
           [12, 7, 10]],
          [[6, 2, 1],
           [12, 0, 3],
           [12, 3, 7],
           [12, 7, 10],
           [12, 10, 9],
           [12, 9, 5],
           [12, 5, 8],
           [12, 8, 11],
           [12, 11, 4],
           [12, 4, 0]],
          [[6, 2, 12],
           [9, 6, 12],
           [10, 9, 12],
           [7, 10, 12],
           [11, 7, 12],
           [8, 11, 12],
           [4, 8, 12],
           [3, 4, 12],
           [0, 3, 12],
           [5, 0, 12],
           [1, 5, 12],
           [2, 1, 12]],
          [[12, 3, 7],
           [12, 7, 10],
           [12, 10, 11],
           [12, 11, 4],
           [12, 4, 8],
           [12, 8, 9],
           [12, 9, 5],
           [12, 5, 0],
           [12, 0, 1],
           [12, 1, 6],
           [12, 6, 2],
           [12, 2, 3]],
          [[12, 1, 5],
           [12, 5, 8],
           [12, 8, 9],
           [12, 9, 6],
           [12, 6, 10],
           [12, 10, 11],
           [12, 11, 7],
           [12, 7, 2],
           [12, 2, 3],
           [12, 3, 4],
           [12, 4, 0],
           [12, 0, 1]],
          [[4, 0, 12],
           [11, 4, 12],
           [8, 11, 12],
           [5, 8, 12],
           [9, 5, 12],
           [10, 9, 12],
           [6, 10, 12],
           [1, 6, 12],
           [2, 1, 12],
           [7, 2, 12],
           [3, 7, 12],
           [0, 3, 12]],
          [[7, 10, 11], [5, 0, 1], [2, 3, 6], [9, 6, 3], [4, 9, 3], [9, 4, 8]],
          [[6, 2, 1], [4, 8, 11], [7, 0, 3], [0, 7, 10], [0, 10, 5], [9, 5, 10]],
          [[4, 0, 3], [6, 10, 9], [5, 2, 1], [2, 5, 8], [2, 8, 7], [11, 7, 8]],
          [[5, 8, 9], [7, 2, 3], [0, 1, 4], [11, 4, 1], [6, 11, 1], [11, 6, 10]],
          [[5, 0, 1],
           [4, 8, 11],
           [3, 4, 11],
           [7, 3, 11],
           [3, 7, 2],
           [6, 2, 7],
           [10, 6, 7],
           [6, 10, 9],
           [11, 9, 10],
           [9, 11, 8]],
          [[4, 8, 11],
           [2, 3, 7],
           [2, 7, 10],
           [2, 10, 6],
           [9, 6, 10],
           [6, 9, 5],
           [6, 5, 1],
           [0, 1, 5],
           [1, 0, 2],
           [3, 2, 0]],
          [[6, 10, 9],
           [0, 1, 5],
           [0, 5, 8],
           [0, 8, 4],
           [11, 4, 8],
           [4, 11, 7],
           [4, 7, 3],
           [2, 3, 7],
           [3, 2, 0],
           [1, 0, 2]],
          [[7, 2, 3],
           [6, 10, 9],
           [1, 6, 9],
           [5, 1, 9],
           [1, 5, 0],
           [4, 0, 5],
           [8, 4, 5],
           [11, 4, 8],
           [9, 11, 8],
           [11, 9, 10]]],
         [[[0, 3, 4], [1, 5, 2], [8, 2, 5], [2, 8, 10]],
          [[0, 1, 5], [3, 4, 2], [8, 2, 4], [2, 8, 10]],
          [[0, 5, 4],
           [8, 4, 5],
           [4, 8, 10],
           [4, 10, 3],
           [2, 3, 10],
           [3, 2, 1],
           [3, 1, 0],
           [5, 0, 1]],
          [[12, 5, 0],
           [12, 0, 3],
           [12, 3, 4],
           [12, 4, 8],
           [12, 8, 10],
           [12, 10, 2],
           [12, 2, 1],
           [12, 1, 5]],
          [[12, 8, 10],
           [12, 10, 2],
           [12, 2, 3],
           [12, 3, 4],
           [12, 4, 0],
           [12, 0, 1],
           [12, 1, 5],
           [12, 5, 8]]],
         [[[2, 3, 7], [11, 4, 8], [1, 5, 0]],
          [[7, 2, 1], [7, 1, 5], [7, 5, 0], [7, 0, 3], [11, 4, 8]],
          [[2, 11, 7], [2, 8, 11], [2, 4, 8], [2, 3, 4], [1, 5, 0]],
          [[1, 5, 8], [1, 8, 11], [1, 11, 0], [11, 4, 0], [2, 3, 7]],
          [[1, 12, 0],
           [0, 12, 4],
           [4, 12, 3],
           [3, 12, 2],
           [2, 12, 7],
           [7, 12, 11],
           [11, 12, 8],
           [8, 12, 5],
           [5, 12, 1]],
          [[0, 12, 4],
           [4, 12, 11],
           [11, 12, 8],
           [8, 12, 5],
           [5, 12, 1],
           [1, 12, 2],
           [2, 12, 7],
           [7, 12, 3],
           [3, 12, 0]],
          [[8, 12, 4],
           [4, 12, 3],
           [3, 12, 0],
           [0, 12, 5],
           [5, 12, 1],
           [1, 12, 2],
           [2, 12, 7],
           [7, 12, 11],
           [11, 12, 8]],
          [[3, 4, 0], [2, 1, 5], [2, 5, 11], [2, 11, 7], [11, 5, 8]],
          [[0, 8, 4],
           [0, 5, 8],
           [1, 5, 0],
           [1, 0, 3],
           [1, 3, 2],
           [2, 3, 7],
           [3, 11, 7],
           [3, 4, 11],
           [4, 8, 11]]],
         [[[2, 1, 6], [11, 8, 3], [3, 8, 0]],
          [[3, 2, 1],
           [3, 1, 0],
           [0, 1, 8],
           [8, 1, 6],
           [8, 6, 11],
           [2, 11, 6],
           [2, 3, 11]],
          [[6, 11, 8], [6, 8, 1], [1, 8, 0], [2, 11, 6], [11, 2, 3]]],
         [[[2, 1, 6], [3, 7, 0], [10, 0, 7], [0, 10, 8]],
          [[2, 3, 7], [1, 6, 0], [10, 0, 6], [0, 10, 8]],
          [[2, 7, 6],
           [10, 6, 7],
           [6, 10, 8],
           [6, 8, 1],
           [0, 1, 8],
           [1, 0, 3],
           [1, 3, 2],
           [7, 2, 3]],
          [[12, 7, 2],
           [12, 2, 1],
           [12, 1, 6],
           [12, 6, 10],
           [12, 10, 8],
           [12, 8, 0],
           [12, 0, 3],
           [12, 3, 7]],
          [[12, 10, 8],
           [12, 8, 0],
           [12, 0, 1],
           [12, 1, 6],
           [12, 6, 2],
           [12, 2, 3],
           [12, 3, 7],
           [12, 7, 10]]],
         [[[8, 3, 11], [8, 0, 3], [10, 2, 1], [10, 1, 9]],
          [[8, 1, 9], [8, 0, 1], [11, 2, 3], [11, 10, 2]],
          [[8, 0, 9],
           [9, 0, 1],
           [8, 9, 11],
           [11, 9, 10],
           [11, 10, 3],
           [10, 2, 3],
           [0, 3, 1],
           [3, 2, 1]],
          [[1, 12, 0],
           [0, 12, 8],
           [8, 12, 11],
           [11, 12, 3],
           [3, 12, 2],
           [2, 12, 10],
           [10, 12, 9],
           [9, 12, 1]],
          [[3, 12, 0],
           [11, 12, 3],
           [10, 12, 11],
           [2, 12, 10],
           [1, 12, 2],
           [9, 12, 1],
           [8, 12, 9],
           [0, 12, 8]]],
         [[[7, 2, 3], [0, 9, 8], [1, 9, 0]],
          [[2, 3, 0],
           [1, 2, 0],
           [9, 2, 1],
           [7, 2, 9],
           [8, 7, 9],
           [7, 8, 3],
           [8, 0, 3]],
          [[9, 8, 7], [2, 9, 7], [1, 9, 2], [7, 8, 3], [0, 3, 8]]],
         [[[1, 6, 2], [5, 0, 9], [3, 9, 0], [9, 3, 11]],
          [[1, 5, 0], [6, 2, 9], [3, 9, 2], [9, 3, 11]],
          [[1, 0, 2],
           [3, 2, 0],
           [2, 3, 11],
           [2, 11, 6],
           [9, 6, 11],
           [6, 9, 5],
           [6, 5, 1],
           [0, 1, 5]],
          [[12, 0, 1],
           [12, 1, 6],
           [12, 6, 2],
           [12, 2, 3],
           [12, 3, 11],
           [12, 11, 9],
           [12, 9, 5],
           [12, 5, 0]],
          [[12, 3, 11],
           [12, 11, 9],
           [12, 9, 6],
           [12, 6, 2],
           [12, 2, 1],
           [12, 1, 5],
           [12, 5, 0],
           [12, 0, 3]]],
         [[[7, 2, 3], [0, 1, 5], [10, 9, 6]],
          [[3, 7, 10], [3, 10, 9], [3, 9, 6], [3, 6, 2], [0, 1, 5]],
          [[7, 0, 3], [7, 5, 0], [7, 1, 5], [7, 2, 1], [10, 9, 6]],
          [[10, 9, 5], [10, 5, 0], [10, 0, 6], [0, 1, 6], [7, 2, 3]],
          [[10, 12, 6],
           [6, 12, 1],
           [1, 12, 2],
           [2, 12, 7],
           [7, 12, 3],
           [3, 12, 0],
           [0, 12, 5],
           [5, 12, 9],
           [9, 12, 10]],
          [[6, 12, 1],
           [1, 12, 0],
           [0, 12, 5],
           [5, 12, 9],
           [9, 12, 10],
           [10, 12, 7],
           [7, 12, 3],
           [3, 12, 2],
           [2, 12, 6]],
          [[5, 12, 1],
           [1, 12, 2],
           [2, 12, 6],
           [6, 12, 9],
           [9, 12, 10],
           [10, 12, 7],
           [7, 12, 3],
           [3, 12, 0],
           [0, 12, 5]],
          [[2, 1, 6], [7, 10, 9], [7, 9, 0], [7, 0, 3], [0, 9, 5]],
          [[6, 5, 1],
           [6, 9, 5],
           [10, 9, 6],
           [10, 6, 2],
           [10, 2, 7],
           [7, 2, 3],
           [2, 0, 3],
           [2, 1, 0],
           [1, 5, 0]]],
         [[[0, 1, 5], [11, 10, 3], [3, 10, 2]],
          [[3, 0, 1],
           [3, 1, 2],
           [2, 1, 10],
           [10, 1, 5],
           [10, 5, 11],
           [0, 11, 5],
           [0, 3, 11]],
          [[5, 11, 10], [5, 10, 1], [1, 10, 2], [0, 11, 5], [11, 0, 3]]],
         [[[0, 1, 5], [2, 3, 7]], [[0, 7, 5], [7, 1, 5], [7, 2, 1], [0, 3, 7]]],
         [[[4, 6, 7], [4, 0, 1], [4, 1, 6]]],
         [[[10, 11, 6], [6, 11, 1], [11, 4, 1], [1, 4, 0]]],
         [[[4, 0, 1], [10, 4, 1], [10, 1, 9], [7, 4, 10]]],
         [[[9, 11, 1], [4, 0, 1], [11, 4, 1]]],
         [[[5, 8, 9], [0, 1, 4], [6, 4, 1], [4, 6, 7]],
          [[5, 0, 1], [8, 9, 4], [6, 4, 9], [4, 6, 7]],
          [[5, 1, 9],
           [6, 9, 1],
           [9, 6, 7],
           [9, 7, 8],
           [4, 8, 7],
           [8, 4, 0],
           [8, 0, 5],
           [1, 5, 0]],
          [[12, 1, 5],
           [12, 5, 8],
           [12, 8, 9],
           [12, 9, 6],
           [12, 6, 7],
           [12, 7, 4],
           [12, 4, 0],
           [12, 0, 1]],
          [[12, 6, 7],
           [12, 7, 4],
           [12, 4, 8],
           [12, 8, 9],
           [12, 9, 5],
           [12, 5, 0],
           [12, 0, 1],
           [12, 1, 6]]],
         [[[1, 5, 0], [4, 8, 11], [6, 10, 9]],
          [[0, 1, 6], [0, 6, 10], [0, 10, 9], [0, 9, 5], [4, 8, 11]],
          [[1, 4, 0], [1, 11, 4], [1, 8, 11], [1, 5, 8], [6, 10, 9]],
          [[6, 10, 11], [6, 11, 4], [6, 4, 9], [4, 8, 9], [1, 5, 0]],
          [[6, 12, 9],
           [9, 12, 8],
           [8, 12, 5],
           [5, 12, 1],
           [1, 12, 0],
           [0, 12, 4],
           [4, 12, 11],
           [11, 12, 10],
           [10, 12, 6]],
          [[9, 12, 8],
           [8, 12, 4],
           [4, 12, 11],
           [11, 12, 10],
           [10, 12, 6],
           [6, 12, 1],
           [1, 12, 0],
           [0, 12, 5],
           [5, 12, 9]],
          [[11, 12, 8],
           [8, 12, 5],
           [5, 12, 9],
           [9, 12, 10],
           [10, 12, 6],
           [6, 12, 1],
           [1, 12, 0],
           [0, 12, 4],
           [4, 12, 11]],
          [[5, 8, 9], [1, 6, 10], [1, 10, 4], [1, 4, 0], [4, 10, 11]],
          [[9, 11, 8],
           [9, 10, 11],
           [6, 10, 9],
           [6, 9, 5],
           [6, 5, 1],
           [1, 5, 0],
           [5, 4, 0],
           [5, 8, 4],
           [8, 11, 4]]],
         [[[5, 0, 1], [10, 7, 8], [8, 7, 4]],
          [[8, 5, 0],
           [8, 0, 4],
           [4, 0, 7],
           [7, 0, 1],
           [7, 1, 10],
           [5, 10, 1],
           [5, 8, 10]],
          [[1, 10, 7], [1, 7, 0], [0, 7, 4], [5, 10, 1], [10, 5, 8]]],
         [[[11, 4, 8], [1, 5, 0]], [[11, 1, 8], [11, 4, 1], [4, 0, 1], [1, 5, 8]]],
         [[[7, 11, 8], [1, 7, 8], [1, 8, 0], [1, 6, 7]]],
         [[[8, 0, 10], [1, 6, 10], [0, 1, 10]]],
         [[[11, 10, 7], [0, 1, 8], [8, 1, 9]],
          [[8, 11, 10],
           [8, 10, 9],
           [9, 10, 1],
           [1, 10, 7],
           [1, 7, 0],
           [11, 0, 7],
           [11, 8, 0]],
          [[7, 0, 1], [7, 1, 10], [10, 1, 9], [11, 0, 7], [0, 11, 8]]],
         [[[0, 1, 8], [8, 1, 9]]],
         [[[0, 1, 5], [9, 7, 11], [6, 7, 9]],
          [[1, 5, 9],
           [6, 1, 9],
           [7, 1, 6],
           [0, 1, 7],
           [11, 0, 7],
           [0, 11, 5],
           [11, 9, 5]],
          [[7, 11, 0], [1, 7, 0], [6, 7, 1], [0, 11, 5], [9, 5, 11]]],
         [[[10, 9, 6], [0, 1, 5]], [[10, 0, 6], [10, 9, 0], [9, 5, 0], [0, 1, 6]]],
         [[[5, 0, 1], [10, 7, 11]],
          [[0, 7, 11], [7, 0, 1], [1, 10, 7], [10, 1, 5], [5, 11, 10], [11, 0, 5]]],
         [[[5, 0, 1]]],
         [[[3, 4, 1], [1, 4, 5]]],
         [[[11, 7, 10], [5, 1, 4], [4, 1, 3]],
          [[4, 11, 7],
           [4, 7, 3],
           [3, 7, 1],
           [1, 7, 10],
           [1, 10, 5],
           [11, 5, 10],
           [11, 4, 5]],
          [[10, 5, 1], [10, 1, 7], [7, 1, 3], [11, 5, 10], [5, 11, 4]]],
         [[[6, 9, 10], [3, 4, 1], [1, 4, 5]],
          [[1, 6, 9],
           [1, 9, 5],
           [5, 9, 4],
           [4, 9, 10],
           [4, 10, 3],
           [6, 3, 10],
           [6, 1, 3]],
          [[10, 3, 4], [10, 4, 9], [9, 4, 5], [6, 3, 10], [3, 6, 1]]],
         [[[7, 9, 11], [7, 6, 9], [4, 5, 1], [4, 1, 3]],
          [[7, 1, 3], [7, 6, 1], [11, 5, 9], [11, 4, 5]],
          [[7, 6, 3],
           [3, 6, 1],
           [7, 3, 11],
           [11, 3, 4],
           [11, 4, 9],
           [4, 5, 9],
           [6, 9, 1],
           [9, 5, 1]],
          [[1, 12, 6],
           [6, 12, 7],
           [7, 12, 11],
           [11, 12, 9],
           [9, 12, 5],
           [5, 12, 4],
           [4, 12, 3],
           [3, 12, 1]],
          [[9, 12, 6],
           [11, 12, 9],
           [4, 12, 11],
           [5, 12, 4],
           [1, 12, 5],
           [3, 12, 1],
           [7, 12, 3],
           [6, 12, 7]]],
         [[[3, 9, 1], [3, 4, 8], [3, 8, 9]]],
         [[[11, 7, 10], [4, 8, 3], [9, 3, 8], [3, 9, 1]],
          [[11, 4, 8], [7, 10, 3], [9, 3, 10], [3, 9, 1]],
          [[11, 8, 10],
           [9, 10, 8],
           [10, 9, 1],
           [10, 1, 7],
           [3, 7, 1],
           [7, 3, 4],
           [7, 4, 11],
           [8, 11, 4]],
          [[12, 8, 11],
           [12, 11, 7],
           [12, 7, 10],
           [12, 10, 9],
           [12, 9, 1],
           [12, 1, 3],
           [12, 3, 4],
           [12, 4, 8]],
          [[12, 9, 1],
           [12, 1, 3],
           [12, 3, 7],
           [12, 7, 10],
           [12, 10, 11],
           [12, 11, 4],
           [12, 4, 8],
           [12, 8, 9]]],
         [[[3, 4, 8], [6, 3, 8], [6, 8, 10], [1, 3, 6]]],
         [[[8, 11, 4], [3, 6, 1], [7, 6, 3]],
          [[11, 4, 3],
           [7, 11, 3],
           [6, 11, 7],
           [8, 11, 6],
           [1, 8, 6],
           [8, 1, 4],
           [1, 3, 4]],
          [[6, 1, 8], [11, 6, 8], [7, 6, 11], [8, 1, 4], [3, 4, 1]]],
         [[[11, 1, 3], [11, 8, 5], [11, 5, 1]]],
         [[[8, 5, 1], [7, 8, 1], [7, 1, 3], [7, 10, 8]]],
         [[[9, 10, 6], [8, 5, 11], [1, 11, 5], [11, 1, 3]],
          [[9, 8, 5], [10, 6, 11], [1, 11, 6], [11, 1, 3]],
          [[9, 5, 6],
           [1, 6, 5],
           [6, 1, 3],
           [6, 3, 10],
           [11, 10, 3],
           [10, 11, 8],
           [10, 8, 9],
           [5, 9, 8]],
          [[12, 5, 9],
           [12, 9, 10],
           [12, 10, 6],
           [12, 6, 1],
           [12, 1, 3],
           [12, 3, 11],
           [12, 11, 8],
           [12, 8, 5]],
          [[12, 1, 3],
           [12, 3, 11],
           [12, 11, 10],
           [12, 10, 6],
           [12, 6, 9],
           [12, 9, 8],
           [12, 8, 5],
           [12, 5, 1]]],
         [[[8, 5, 9], [6, 3, 7], [1, 3, 6]],
          [[5, 9, 6],
           [1, 5, 6],
           [3, 5, 1],
           [8, 5, 3],
           [7, 8, 3],
           [8, 7, 9],
           [7, 6, 9]],
          [[3, 7, 8], [5, 3, 8], [1, 3, 5], [8, 7, 9], [6, 9, 7]]],
         [[[1, 11, 9], [1, 3, 11]]],
         [[[3, 9, 1], [3, 7, 10], [3, 10, 9]]],
         [[[3, 11, 1], [10, 6, 1], [11, 10, 1]]],
         [[[6, 3, 7], [1, 3, 6]]],
         [[[5, 7, 4], [5, 1, 2], [5, 2, 7]]],
         [[[10, 11, 4], [1, 10, 4], [1, 4, 5], [2, 10, 1]]],
         [[[6, 9, 10], [1, 2, 5], [7, 5, 2], [5, 7, 4]],
          [[6, 1, 2], [9, 10, 5], [7, 5, 10], [5, 7, 4]],
          [[6, 2, 10],
           [7, 10, 2],
           [10, 7, 4],
           [10, 4, 9],
           [5, 9, 4],
           [9, 5, 1],
           [9, 1, 6],
           [2, 6, 1]],
          [[12, 2, 6],
           [12, 6, 9],
           [12, 9, 10],
           [12, 10, 7],
           [12, 7, 4],
           [12, 4, 5],
           [12, 5, 1],
           [12, 1, 2]],
          [[12, 7, 4],
           [12, 4, 5],
           [12, 5, 9],
           [12, 9, 10],
           [12, 10, 6],
           [12, 6, 1],
           [12, 1, 2],
           [12, 2, 7]]],
         [[[6, 1, 2], [11, 4, 9], [9, 4, 5]],
          [[9, 6, 1],
           [9, 1, 5],
           [5, 1, 4],
           [4, 1, 2],
           [4, 2, 11],
           [6, 11, 2],
           [6, 9, 11]],
          [[2, 11, 4], [2, 4, 1], [1, 4, 5], [6, 11, 2], [11, 6, 9]]],
         [[[1, 2, 7], [8, 1, 7], [8, 7, 4], [8, 9, 1]]],
         [[[8, 11, 4], [1, 2, 9], [9, 2, 10]],
          [[9, 8, 11],
           [9, 11, 10],
           [10, 11, 2],
           [2, 11, 4],
           [2, 4, 1],
           [8, 1, 4],
           [8, 9, 1]],
          [[4, 1, 2], [4, 2, 11], [11, 2, 10], [8, 1, 4], [1, 8, 9]]],
         [[[2, 6, 1], [4, 8, 7], [7, 8, 10]],
          [[7, 2, 6],
           [7, 6, 10],
           [10, 6, 8],
           [8, 6, 1],
           [8, 1, 4],
           [2, 4, 1],
           [2, 7, 4]],
          [[1, 4, 8], [1, 8, 6], [6, 8, 10], [2, 4, 1], [4, 2, 7]]],
         [[[6, 1, 2], [11, 4, 8]],
          [[1, 4, 8], [4, 1, 2], [2, 11, 4], [11, 2, 6], [6, 8, 11], [8, 1, 6]]],
         [[[11, 8, 7], [7, 8, 2], [8, 5, 2], [2, 5, 1]]],
         [[[10, 8, 2], [5, 1, 2], [8, 5, 2]]],
         [[[5, 9, 8], [2, 6, 1], [10, 7, 11]],
          [[11, 8, 5], [7, 11, 5], [10, 7, 5], [9, 10, 5], [2, 6, 1]],
          [[5, 1, 8], [1, 2, 8], [2, 6, 8], [6, 9, 8], [10, 7, 11]],
          [[2, 7, 11], [1, 2, 11], [10, 1, 11], [10, 6, 1], [5, 9, 8]],
          [[10, 12, 11],
           [6, 12, 10],
           [9, 12, 6],
           [8, 12, 9],
           [5, 12, 8],
           [1, 12, 5],
           [2, 12, 1],
           [7, 12, 2],
           [11, 12, 7]],
          [[6, 12, 10],
           [1, 12, 6],
           [2, 12, 1],
           [7, 12, 2],
           [11, 12, 7],
           [8, 12, 11],
           [5, 12, 8],
           [9, 12, 5],
           [10, 12, 9]],
          [[6, 12, 2],
           [9, 12, 6],
           [10, 12, 9],
           [7, 12, 10],
           [11, 12, 7],
           [8, 12, 11],
           [5, 12, 8],
           [1, 12, 5],
           [2, 12, 1]],
          [[10, 6, 9], [7, 11, 8], [1, 7, 8], [5, 1, 8], [2, 7, 1]],
          [[6, 2, 10],
           [2, 7, 10],
           [10, 7, 11],
           [9, 10, 11],
           [8, 9, 11],
           [5, 9, 8],
           [5, 1, 9],
           [1, 6, 9],
           [1, 2, 6]]],
         [[[8, 5, 9], [2, 6, 1]], [[8, 2, 9], [8, 5, 2], [5, 1, 2], [2, 6, 9]]],
         [[[1, 11, 9], [1, 2, 7], [1, 7, 11]]],
         [[[1, 10, 9], [2, 10, 1]]],
         [[[2, 6, 1], [10, 7, 11]], [[2, 11, 1], [11, 6, 1], [11, 10, 6], [2, 7, 11]]],
         [[[1, 2, 6]]],
         [[[6, 4, 5], [6, 2, 3], [6, 3, 4]]],
         [[[7, 10, 11], [2, 3, 6], [4, 6, 3], [6, 4, 5]],
          [[7, 2, 3], [10, 11, 6], [4, 6, 11], [6, 4, 5]],
          [[7, 3, 11],
           [4, 11, 3],
           [11, 4, 5],
           [11, 5, 10],
           [6, 10, 5],
           [10, 6, 2],
           [10, 2, 7],
           [3, 7, 2]],
          [[12, 3, 7],
           [12, 7, 10],
           [12, 10, 11],
           [12, 11, 4],
           [12, 4, 5],
           [12, 5, 6],
           [12, 6, 2],
           [12, 2, 3]],
          [[12, 4, 5],
           [12, 5, 6],
           [12, 6, 10],
           [12, 10, 11],
           [12, 11, 7],
           [12, 7, 2],
           [12, 2, 3],
           [12, 3, 4]]],
         [[[5, 9, 10], [3, 5, 10], [3, 10, 2], [3, 4, 5]]],
         [[[2, 3, 7], [11, 5, 9], [4, 5, 11]],
          [[3, 7, 11],
           [4, 3, 11],
           [5, 3, 4],
           [2, 3, 5],
           [9, 2, 5],
           [2, 9, 7],
           [9, 11, 7]],
          [[5, 9, 2], [3, 5, 2], [4, 5, 3], [2, 9, 7], [11, 7, 9]]],
         [[[3, 4, 2], [2, 4, 6], [4, 8, 6], [6, 8, 9]]],
         [[[8, 11, 4], [6, 10, 9], [7, 2, 3]],
          [[3, 4, 8], [2, 3, 8], [7, 2, 8], [11, 7, 8], [6, 10, 9]],
          [[8, 9, 4], [9, 6, 4], [6, 10, 4], [10, 11, 4], [7, 2, 3]],
          [[6, 2, 3], [9, 6, 3], [7, 9, 3], [7, 10, 9], [8, 11, 4]],
          [[7, 12, 3],
           [10, 12, 7],
           [11, 12, 10],
           [4, 12, 11],
           [8, 12, 4],
           [9, 12, 8],
           [6, 12, 9],
           [2, 12, 6],
           [3, 12, 2]],
          [[10, 12, 7],
           [9, 12, 10],
           [6, 12, 9],
           [2, 12, 6],
           [3, 12, 2],
           [4, 12, 3],
           [8, 12, 4],
           [11, 12, 8],
           [7, 12, 11]],
          [[10, 12, 6],
           [11, 12, 10],
           [7, 12, 11],
           [2, 12, 7],
           [3, 12, 2],
           [4, 12, 3],
           [8, 12, 4],
           [9, 12, 8],
           [6, 12, 9]],
          [[7, 10, 11], [2, 3, 4], [9, 2, 4], [8, 9, 4], [6, 2, 9]],
          [[10, 6, 7],
           [6, 2, 7],
           [7, 2, 3],
           [11, 7, 3],
           [4, 11, 3],
           [8, 11, 4],
           [8, 9, 11],
           [9, 10, 11],
           [9, 6, 10]]],
         [[[10, 2, 8], [3, 4, 8], [2, 3, 8]]],
         [[[11, 4, 8], [3, 7, 2]], [[11, 2, 8], [2, 4, 8], [2, 3, 4], [11, 7, 2]]],
         [[[6, 2, 3], [8, 6, 3], [8, 3, 11], [5, 6, 8]]],
         [[[3, 7, 2], [6, 8, 5], [10, 8, 6]],
          [[7, 2, 6],
           [10, 7, 6],
           [8, 7, 10],
           [3, 7, 8],
           [5, 3, 8],
           [3, 5, 2],
           [5, 6, 2]],
          [[8, 5, 3], [7, 8, 3], [10, 8, 7], [3, 5, 2], [6, 2, 5]]],
         [[[5, 9, 8], [11, 2, 3], [10, 2, 11]],
          [[9, 8, 11],
           [10, 9, 11],
           [2, 9, 10],
           [5, 9, 2],
           [3, 5, 2],
           [5, 3, 8],
           [3, 11, 8]],
          [[2, 3, 5], [9, 2, 5], [10, 2, 9], [5, 3, 8], [11, 8, 3]]],
         [[[2, 3, 7], [8, 5, 9]],
          [[8, 5, 3], [2, 3, 5], [5, 9, 2], [7, 2, 9], [9, 8, 7], [7, 3, 8]]],
         [[[9, 3, 11], [9, 6, 2], [9, 2, 3]]],
         [[[6, 10, 9], [7, 2, 3]], [[6, 3, 9], [3, 10, 9], [3, 7, 10], [6, 2, 3]]],
         [[[2, 3, 10], [10, 3, 11]]],
         [[[7, 2, 3]]],
         [[[4, 6, 7], [4, 5, 6]]],
         [[[5, 6, 4], [10, 11, 4], [6, 10, 4]]],
         [[[4, 5, 7], [9, 10, 7], [5, 9, 7]]],
         [[[5, 11, 4], [9, 11, 5]]],
         [[[7, 4, 6], [8, 9, 6], [4, 8, 6]]],
         [[[8, 11, 4], [10, 9, 6]], [[8, 6, 4], [6, 11, 4], [6, 10, 11], [8, 9, 6]]],
         [[[4, 10, 7], [8, 10, 4]]],
         [[[11, 4, 8]]],
         [[[7, 5, 6], [7, 11, 8], [7, 8, 5]]],
         [[[6, 10, 5], [5, 10, 8]]],
         [[[7, 11, 10], [5, 9, 8]], [[7, 5, 10], [7, 11, 5], [11, 8, 5], [5, 9, 10]]],
         [[[9, 8, 5]]],
         [[[7, 9, 6], [11, 9, 7]]],
         [[[10, 9, 6]]],
         [[[10, 7, 11]]],
         [[[]]]], dtype=object)
MC33_FaceTest = np.array([[[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[4, 5, 6, 7]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[7, 4, 5, 6]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[2, 3, 7, 6]],
         [[]],
         [[]],
         [[4, 5, 6, 7]],
         [[2, 3, 7, 6]],
         [[]],
         [[7, 3, 0, 4]],
         [[]],
         [[2, 3, 7, 6], [3, 0, 4, 7], [6, 7, 4, 5]],
         [[]],
         [[7, 3, 0, 4]],
         [[]],
         [[7, 6, 2, 3], [4, 7, 3, 0]],
         [[]],
         [[]],
         [[6, 2, 3, 7]],
         [[]],
         [[]],
         [[6, 5, 1, 2]],
         [[3, 7, 6, 2], [7, 4, 5, 6], [2, 6, 5, 1]],
         [[]],
         [[]],
         [[]],
         [[6, 2, 3, 7]],
         [[7, 4, 5, 6]],
         [[]],
         [[1, 2, 6, 5]],
         [[6, 5, 1, 2], [7, 6, 2, 3]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[6, 5, 1, 2]],
         [[6, 7, 4, 5], [2, 6, 5, 1]],
         [[]],
         [[]],
         [[0, 4, 7, 3]],
         [[]],
         [[7, 3, 0, 4], [6, 7, 4, 5]],
         [[]],
         [[0, 3, 7, 4], [1, 2, 6, 5]],
         [[1, 5, 6, 2]],
         [[0, 3, 7, 4]],
         [[]],
         [[]],
         [[]],
         [[2, 6, 5, 1]],
         [[5, 1, 2, 6]],
         [[]],
         [[6, 7, 4, 5]],
         [[]],
         [[]],
         [[5, 4, 0, 1]],
         [[0, 1, 5, 4]],
         [[0, 1, 5, 4], [1, 2, 6, 5], [4, 5, 6, 7]],
         [[5, 4, 0, 1], [6, 5, 1, 2]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[0, 3, 2, 1]],
         [[2, 1, 0, 3]],
         [[7, 6, 2, 3], [6, 5, 1, 2], [3, 2, 1, 0]],
         [[2, 6, 5, 1], [3, 2, 1, 0]],
         [[0, 3, 2, 1]],
         [[0, 1, 2, 3], [4, 5, 6, 7]],
         [[2, 1, 0, 3], [6, 2, 3, 7]],
         [[0, 1, 2, 3]],
         [[5, 4, 0, 1], [4, 7, 3, 0], [1, 0, 3, 2]],
         [[0, 3, 2, 1], [4, 0, 1, 5]],
         [[0, 1, 2, 3],
          [0, 1, 5, 4],
          [1, 2, 6, 5],
          [2, 3, 7, 6],
          [3, 0, 4, 7],
          [4, 5, 6, 7]],
         [[2, 1, 5, 6], [1, 0, 4, 5], [3, 0, 1, 2]],
         [[0, 4, 7, 3], [1, 0, 3, 2]],
         [[2, 3, 0, 1]],
         [[2, 3, 0, 1], [3, 7, 4, 0], [6, 7, 3, 2]],
         [[2, 3, 0, 1]],
         [[]],
         [[3, 7, 6, 2]],
         [[]],
         [[]],
         [[]],
         [[6, 2, 3, 7], [5, 6, 7, 4]],
         [[]],
         [[]],
         [[5, 4, 0, 1]],
         [[6, 7, 3, 2], [5, 4, 0, 1]],
         [[5, 6, 7, 4], [1, 5, 4, 0]],
         [[0, 4, 5, 1]],
         [[]],
         [[3, 2, 6, 7]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[4, 7, 6, 5]],
         [[]],
         [[]],
         [[0, 1, 5, 4], [3, 0, 4, 7]],
         [[5, 1, 0, 4]],
         [[7, 4, 0, 3], [4, 5, 1, 0], [6, 5, 4, 7]],
         [[0, 4, 5, 1]],
         [[7, 4, 0, 3]],
         [[]],
         [[4, 0, 3, 7]],
         [[]],
         [[]],
         [[3, 0, 4, 7]],
         [[]],
         [[3, 0, 4, 7]],
         [[4, 0, 1, 5]],
         [[3, 0, 4, 7], [0, 1, 5, 4], [7, 4, 5, 6]],
         [[4, 0, 1, 5]],
         [[4, 7, 3, 0], [5, 4, 0, 1]],
         [[]],
         [[]],
         [[5, 6, 7, 4]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[7, 6, 2, 3]],
         [[]],
         [[1, 5, 4, 0]],
         [[4, 0, 1, 5], [7, 4, 5, 6]],
         [[2, 6, 7, 3], [1, 5, 4, 0]],
         [[1, 0, 4, 5]],
         [[]],
         [[]],
         [[7, 4, 5, 6], [3, 7, 6, 2]],
         [[]],
         [[]],
         [[]],
         [[6, 7, 3, 2]],
         [[]],
         [[3, 2, 1, 0]],
         [[4, 7, 3, 0], [7, 6, 2, 3], [0, 3, 2, 1]],
         [[1, 0, 3, 2]],
         [[3, 2, 1, 0], [7, 3, 0, 4]],
         [[6, 5, 1, 2], [5, 4, 0, 1], [2, 1, 0, 3]],
         [[3, 0, 1, 2],
          [3, 0, 4, 7],
          [0, 1, 5, 4],
          [1, 2, 6, 5],
          [2, 3, 7, 6],
          [7, 4, 5, 6]],
         [[1, 5, 4, 0], [2, 1, 0, 3]],
         [[2, 3, 0, 1], [3, 7, 4, 0], [1, 0, 4, 5]],
         [[3, 2, 1, 0]],
         [[3, 7, 6, 2], [0, 3, 2, 1]],
         [[5, 4, 7, 6], [1, 0, 3, 2]],
         [[1, 2, 3, 0]],
         [[1, 0, 3, 2], [5, 1, 2, 6]],
         [[7, 3, 2, 6], [3, 0, 1, 2], [6, 2, 1, 5]],
         [[0, 1, 2, 3]],
         [[3, 0, 1, 2]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[1, 2, 6, 5], [0, 1, 5, 4]],
         [[2, 1, 5, 6], [1, 0, 4, 5], [6, 5, 4, 7]],
         [[5, 1, 0, 4]],
         [[0, 4, 5, 1]],
         [[]],
         [[]],
         [[4, 7, 6, 5]],
         [[]],
         [[6, 2, 1, 5]],
         [[5, 6, 2, 1]],
         [[]],
         [[]],
         [[]],
         [[4, 7, 3, 0]],
         [[2, 6, 5, 1]],
         [[3, 7, 4, 0], [2, 6, 5, 1]],
         [[]],
         [[4, 5, 6, 7], [0, 4, 7, 3]],
         [[]],
         [[3, 7, 4, 0]],
         [[]],
         [[]],
         [[5, 1, 2, 6], [4, 5, 6, 7]],
         [[2, 1, 5, 6]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[2, 3, 7, 6], [1, 2, 6, 5]],
         [[6, 2, 1, 5]],
         [[]],
         [[5, 4, 7, 6]],
         [[3, 2, 6, 7]],
         [[]],
         [[]],
         [[]],
         [[7, 6, 5, 4], [6, 2, 1, 5], [3, 2, 6, 7]],
         [[1, 5, 6, 2]],
         [[]],
         [[]],
         [[7, 3, 2, 6]],
         [[]],
         [[]],
         [[3, 0, 4, 7], [2, 3, 7, 6]],
         [[]],
         [[4, 0, 3, 7]],
         [[]],
         [[3, 7, 4, 0], [7, 6, 5, 4], [2, 6, 7, 3]],
         [[]],
         [[3, 7, 4, 0]],
         [[]],
         [[6, 7, 3, 2]],
         [[7, 6, 5, 4]],
         [[]],
         [[]],
         [[3, 2, 6, 7]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[6, 5, 4, 7]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[4, 7, 6, 5]],
         [[]],
         [[]],
         [[]],
         [[]],
         [[]]],
         dtype=object)
MC33_Cases = np.array([0,
         1,
         1,
         2,
         1,
         3,
         2,
         5,
         1,
         2,
         3,
         5,
         2,
         5,
         5,
         8,
         1,
         2,
         3,
         5,
         4,
         6,
         6,
         11,
         3,
         5,
         7,
         9,
         6,
         14,
         12,
         5,
         1,
         3,
         2,
         5,
         3,
         7,
         5,
         9,
         4,
         6,
         6,
         14,
         6,
         12,
         11,
         5,
         2,
         5,
         5,
         8,
         6,
         12,
         14,
         5,
         6,
         11,
         12,
         5,
         10,
         6,
         6,
         2,
         1,
         4,
         3,
         6,
         2,
         6,
         5,
         14,
         3,
         6,
         7,
         12,
         5,
         11,
         9,
         5,
         3,
         6,
         7,
         12,
         6,
         10,
         12,
         6,
         7,
         12,
         13,
         7,
         12,
         6,
         7,
         3,
         2,
         6,
         5,
         11,
         5,
         12,
         8,
         5,
         6,
         10,
         12,
         6,
         14,
         6,
         5,
         2,
         5,
         14,
         9,
         5,
         11,
         6,
         5,
         2,
         12,
         6,
         7,
         3,
         6,
         4,
         3,
         1,
         1,
         3,
         4,
         6,
         3,
         7,
         6,
         12,
         2,
         5,
         6,
         11,
         5,
         9,
         14,
         5,
         2,
         5,
         6,
         14,
         6,
         12,
         10,
         6,
         5,
         8,
         12,
         5,
         11,
         5,
         6,
         2,
         3,
         7,
         6,
         12,
         7,
         13,
         12,
         7,
         6,
         12,
         10,
         6,
         12,
         7,
         6,
         3,
         5,
         9,
         11,
         5,
         12,
         7,
         6,
         3,
         14,
         5,
         6,
         2,
         6,
         3,
         4,
         1,
         2,
         6,
         6,
         10,
         5,
         12,
         11,
         6,
         5,
         14,
         12,
         6,
         8,
         5,
         5,
         2,
         5,
         11,
         12,
         6,
         14,
         6,
         6,
         4,
         9,
         5,
         7,
         3,
         5,
         2,
         3,
         1,
         5,
         12,
         14,
         6,
         9,
         7,
         5,
         3,
         11,
         6,
         6,
         4,
         5,
         3,
         2,
         1,
         8,
         5,
         5,
         2,
         5,
         3,
         2,
         1,
         5,
         2,
         3,
         1,
         2,
         1,
         1,
         0])
MC33_Signs = np.array([-1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         1,
         -1,
         -1,
         -1,
         1,
         -1,
         1,
         1,
         1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         1,
         -1,
         -1,
         -1,
         1,
         -1,
         1,
         1,
         1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         1,
         -1,
         -1,
         -1,
         1,
         -1,
         1,
         1,
         1,
         -1,
         -1,
         -1,
         1,
         -1,
         1,
         1,
         1,
         -1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         1,
         -1,
         -1,
         -1,
         1,
         -1,
         1,
         1,
         1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         1,
         -1,
         -1,
         -1,
         1,
         -1,
         1,
         1,
         1,
         -1,
         -1,
         -1,
         1,
         -1,
         1,
         1,
         1,
         -1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         -1,
         1,
         -1,
         -1,
         -1,
         1,
         -1,
         1,
         1,
         1,
         -1,
         -1,
         -1,
         1,
         -1,
         1,
         1,
         1,
         -1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         -1,
         -1,
         -1,
         1,
         -1,
         1,
         1,
         1,
         -1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         -1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         -1])

def MarchingSquaresImage(I, h=1, threshold=0, z=0, interpolation='linear', Type='surf', VertexValues=False, flip=False, mixed_elements=False, edgemode='constant', cleanup=True):
    """
    Marching squares algorithm applied to 2D image data.

    Image data is assumed to be pixel data and will be interpolated to vertices to get the corners of each square. 

    Parameters
    ----------
    I : np.ndarray
        Image array (2D) of grayscale data
    h : float or tuple, optional
        Element (pixel) size. Can be specified as a single value for isotropic pixel sizes or a tuple of two values, by default 1
    threshold : int, optional
        Isosurface level that defines the boundary, by default 0
    z : float, optional
        z coordinate that will be used for all nodes, by default 0.
    interpolation : str, optional
        Interpolation to interpolate the position of the new nodes along the edges of the input mesh, by default 'linear'. 
        
        'midpoint' - No interpolation is performed, new nodes are placed at the midpoint of edges 
        
        'linear' - Bilinear interpolation is performed between adjacent nodes on the input mesh

        'cubic' - Bicubic interpolation is performed between adjacent nodes on the input mesh
    Type : str, optional
        Determines whether to generate a surface mesh ('surf') or volume mesh ('vol'), 
        by default 'surf'
    VertexValues : bool, optional
        If True, the values in the image array are treated as the vertices of 
        the cubes, otherwise, they are treated as voxel values and vertices
        are obtained through interpolation, by default, False.
    flip : bool, optional
        Flip the interior/exterior of the mesh, by default False. By default, values less than the threshold are assumed to be the “inside” of the mesh. If the inside is denoted by values greater than the threshold, set flip=True.
    edgemode : str, optional
        For interpolation='cubic', edgemode specifies how to handle boundary nodes. The image matrix will be padded using ``np.pad(I, 1, mode=edgemode)``, by default 'constant'
    mixed_elements : bool, optional
        If True and Type='surf', the generated mesh will have mixed element types 
        (triangles/quadrilateral otherwise a single element type (triangles), by 
        default False.
    cleanup : bool, optional
        Determines whether or not to perform mesh cleanup, removing degenerate elements and duplicate nodes, by default True

    Returns
    -------
    NewCoords : np.ndarray
        Numpy array of node coordinates for the contour mesh
    NewConn : list
        List of node connectivities 

    """    
    
    assert len(I.shape) == 2, 'I must be a 2D numpy array of image data. For 3D, use MarchingCubesImage.'
    I = I - threshold  
    if Type.lower() == 'surf':
        if mixed_elements:
            LookupTable = MSMixed_Lookup
        else:
            LookupTable = MSTriangle_Lookup
    elif Type.lower() == 'line':
        LookupTable = MSEdge_Lookup
    else:
        raise Exception('Invalid Type. Must be "surf" or "line".')
    if flip: 
        I = -I
    if isinstance(h, (int, float, np.number)):
        h = (h,h)
    
    Padding = 0
    if interpolation == "cubic":
        # I = np.pad(I,1,mode=edgemode)
        Padding = 1

    if not VertexValues:
        x1 = np.arange(0,I.shape[1],1)*h[0] + h[0]/2
        y1 = np.arange(0,I.shape[0],1)*h[1] + h[1]/2
        
        X = np.repeat(np.atleast_2d(x1),I.shape[0],axis=0) - Padding*h[0]
        Y = np.repeat(np.atleast_2d(y1).T,I.shape[1],axis=1) - Padding*h[1]
        Z = np.zeros(X.shape)
        
        
        Xv = np.repeat(np.atleast_2d(np.arange(-Padding,I.shape[1]+(1+Padding),1)),  I.shape[0]+1+2*Padding,axis=0) * h[0]
        Yv = np.repeat(np.atleast_2d(np.arange(-Padding,I.shape[0]+(1+Padding),1)).T,I.shape[1]+1+2*Padding,axis=1) * h[1]
        
        # TODO: This is major bottleneck for cubic
        interp = scipy.interpolate.RegularGridInterpolator((y1,x1),I,fill_value=None,method='linear',bounds_error=False)
        Iv = interp((Yv.flatten(),Xv.flatten())).reshape(Xv.shape)
        
        X = Xv; Y = Yv; I = Iv; #Z = np.zeros(Xv.shape)
    else:
        X, Y = np.meshgrid(np.arange(-Padding,I.shape[0]+(Padding)),
                                 np.arange(-Padding,I.shape[1]+(Padding)),
                                 indexing='ij'
                                 )*np.asarray(h)[:,None,None]
        I = np.pad(I,Padding,mode=edgemode)

    edgeLookup = np.array([
        [0, 0],  # Corner 0
        [1, 1],  # Corner 1
        [2, 2],  # Corner 2
        [3, 3],  # Corner 3
        [0, 1],  # Edge 0
        [1, 2],  # Edge 1
        [2, 3],  # Edge 2
        [3, 0],  # Edge 3
        ])
    iidx = np.repeat(np.arange(X.shape[0]-1-2*Padding),X.shape[1]-1-2*Padding)+Padding
    jidx = np.tile(np.arange(X.shape[1]-1-2*Padding),X.shape[0]-1-2*Padding)+Padding
    
    isquares = np.vstack([iidx,iidx+1,iidx+1,iidx]).T
    jsquares = np.vstack([jidx,jidx,jidx+1,jidx+1]).T
    
    vals = I[isquares,jsquares]
    inside = (vals <= 0).astype(int)
    tableIdx = np.dot(inside, 2**np.arange(inside.shape[1] - 1, -1, -1))
    
    edgeList =  LookupTable[tableIdx]
    edgeConnections = np.array([x for y in edgeList for x in y if len(x) != 0])
    numbering = np.array([j for j,y in enumerate(edgeList) for i,x in enumerate(y) if len(x) != 0])
    
    
    i_indices = isquares[numbering][np.arange(len(numbering))[:, np.newaxis,np.newaxis], edgeLookup[edgeConnections]]
    j_indices = jsquares[numbering][np.arange(len(numbering))[:, np.newaxis,np.newaxis], edgeLookup[edgeConnections]]
    
    
    if interpolation == "cubic":
        NewCoords = []
        ishiftdir = i_indices[:,:,1] - i_indices[:,:,0]
        jshiftdir = j_indices[:,:,1] - j_indices[:,:,0]
        
        i_interp = np.stack([i_indices[:,:,0]-1*ishiftdir, i_indices[:,:,0], i_indices[:,:,1], i_indices[:,:,1]+1*ishiftdir],axis=2)
        j_interp = np.stack([j_indices[:,:,0]-1*jshiftdir, j_indices[:,:,0], j_indices[:,:,1], j_indices[:,:,1]+1*jshiftdir],axis=2)
        
        X_interp = X[i_interp,j_interp]
        Y_interp = Y[i_interp,j_interp]
        I_interp = I[i_interp,j_interp]
        
        
        x = X_interp.reshape(I_interp.shape[0]*I_interp.shape[1],I_interp.shape[2],order='F')
        y = Y_interp.reshape(I_interp.shape[0]*I_interp.shape[1],I_interp.shape[2],order='F')
        v = I_interp.reshape(I_interp.shape[0]*I_interp.shape[1],I_interp.shape[2],order='F')
        
        # Build coefficients for cubic polynomials
        xbool = x[:,1]!=x[:,2] # Only interpolate in x if on an x edge
        ybool = y[:,1]!=y[:,2] # Only interpolate in y if on an y edge
        
        xsort = np.argsort(x[xbool],axis=1)
        ysort = np.argsort(y[ybool],axis=1)
        x[xbool,:] = np.take_along_axis(x[xbool], xsort, axis=1)
        y[ybool,:] = np.take_along_axis(y[ybool], ysort, axis=1)
        v[xbool,:] = np.take_along_axis(v[xbool], xsort, axis=1)
        v[ybool,:] = np.take_along_axis(v [ybool], ysort, axis=1)
        
        xAs = np.stack([x[xbool]**3, x[xbool]**2, x[xbool]**1, x[xbool]**0],axis=2)
        yAs = np.stack([y[ybool]**3, y[ybool]**2, y[ybool]**1, y[ybool]**0],axis=2)
        xCoeff = np.linalg.solve(xAs,v[xbool,:,None])[:,:,0]
        yCoeff = np.linalg.solve(yAs,v[ybool,:,None])[:,:,0]
        
        # cf. https://en.wikipedia.org/wiki/Cubic_equation#General_cubic_formula
        Roots = []
        for coeff in [xCoeff, yCoeff]:
            a = coeff[:,0]
            b = coeff[:,1]
            c = coeff[:,2]
            d = coeff[:,3]
            
            delta0 = b**2 - 3*a*c
            delta1 = 2*b**3 - 9*a*b*c + 27*a**2*d
            
            C = ((delta1 + np.sqrt(delta1**2+0J - 4*delta0**3))/2)**(1/3)
            C[C==0] = ((delta1[C==0] - np.sqrt(delta1[C==0]**2 - 4*delta0[C==0]**3))/2)**(1/3)
            
            zeta = (-1 + np.sqrt(-3+0J))/2
            roots = np.vstack([-1/(3*a) * ( b + zeta**k*C + delta0/(zeta**k*C) ) for k in [0, 1, 2]]).T
            realcheck = np.isclose(np.imag(roots), 0)
            roots[realcheck]  = np.real(roots[realcheck])
            
            # overwrite with quadratic for a ~ 0
            near0 = np.isclose(a,0)
            roots[near0] = np.vstack([(-c[near0] + np.sqrt(c[near0]**2 - 4*b[near0]*d[near0]))/(2*b[near0]), 
                                       (-c[near0] - np.sqrt(c[near0]**2 - 4*b[near0]*d[near0]))/(2*b[near0]),
                                        np.repeat(np.nan,len(b[near0]))]).T
            Roots.append(roots)
        xRoots = Roots[0]
        yRoots = Roots[1]
        
        NewCoords = np.column_stack([x[:,1], y[:,1], np.repeat(z,len(x))]).astype(np.float64)
        
        xRootsReal = np.real(xRoots)
        NewCoords[xbool,0] = xRootsReal[(x[xbool,1,None] <= xRootsReal) & (x[xbool,2,None] >= xRootsReal) & np.isreal(xRoots)]
        # NewCoords[xbool,1] = y[xbool,1]
        
        # NewCoords[ybool,0] = x[ybool,1]
        yRootsReal = np.real(yRoots)
        NewCoords[ybool,1] = yRootsReal[(y[ybool,1,None] <= yRootsReal) & (y[ybool,2,None] >= yRootsReal) & np.isreal(yRoots)]
        
        NewCoords[:,2] = z
        
    elif interpolation == "linear" or interpolation == "midpoint":
        ishiftdir = i_indices[:,:,1] - i_indices[:,:,0]
        jshiftdir = j_indices[:,:,1] - j_indices[:,:,0]
        
        i_interp = i_indices
        j_interp = j_indices
        
        X_interp = X[i_interp,j_interp]
        Y_interp = Y[i_interp,j_interp]
        I_interp = I[i_interp,j_interp]
        
        x = X_interp.reshape(I_interp.shape[0]*I_interp.shape[1],I_interp.shape[2],order='F')
        y = Y_interp.reshape(I_interp.shape[0]*I_interp.shape[1],I_interp.shape[2],order='F')
        v = I_interp.reshape(I_interp.shape[0]*I_interp.shape[1],I_interp.shape[2],order='F')
        
        if interpolation == "linear":
            with np.errstate(divide='ignore', invalid='ignore'):
                NewCoords = np.vstack([
                     x[:,0] + np.nan_to_num((0-v[:,0])*(x[:,1]-x[:,0])/(v[:,1]-v[:,0])),
                     y[:,0]  + np.nan_to_num((0-v[:,0])*(y[:,1]-y[:,0])/(v[:,1]-v[:,0])),
                     np.repeat(z,len(x))
                    ]).T
        else:
            NewCoords = np.vstack([
                 (x[:,0] + x[:,1])/2,
                 (y[:,0] + y[:,1])/2,
                 np.repeat(z,len(x))
                ]).T
    else:
        raise Exception('Invalid input "{:s}" for interpolation. Must be one of "midpoint", "linear", or "cubic".'.format(interpolation))
    NewCoords[:,[0,1]] = NewCoords[:,[1,0]]  
    NewConn = np.arange(len(NewCoords)).reshape(edgeConnections.shape,order='F')
    if cleanup:
        NewCoords,NewConn,Idx = utils.DeleteDuplicateNodes(NewCoords,NewConn,return_idx=True)
        if (interpolation=='linear' or interpolation=='cubic') and Type.lower()=='surf':
            NewCoords,NewConn = utils.DeleteDegenerateElements(NewCoords,NewConn,strict=True)
            
    return NewCoords, NewConn

def MarchingSquares(NodeCoords, NodeConn, NodeValues, threshold=0, interpolation='linear', Type='surf', flip=False, mixed_elements=False, return_anchors=False, cleanup=True):
    """
    Marching squares algorithm for extracting an isocontour from a quadrilateral mesh.

    For a two dimensional grid (NodeCoords, NodeConn) with values (NodeValues),
    Marching Squares will extract the isovalue contour defined by threshold. 
    This can either be used to obtain line segments along the isoline 
    (Type='line) or triangles filling the region inside the isoline
    (Type='surf'). 

    Parameters
    ----------
    NodeCoords : array_like
        Node coordinates of a two dimensional quadrilateral grid (e.g. from 
        primitives.Grid2D)
    NodeConn : array_like
        Node connectivities of a two dimensional grid of rectilinear grid (e.g. 
        from primitives.Grid2D) of quadrilaterals. 
    NodeValues : array_like
        Array of node values corresponding to each node in the mesh.
    threshold : int, optional
        Isosurface level that defines the boundary, by default 0
    interpolation : str, optional
        Interpolation to interpolate the position of the new nodes along the 
        edges of the input mesh, by default 'linear'. 
        
        'midpoint' - No interpolation is performed, new nodes are placed at the 
        midpoint of edges 
        
        'linear' - Bilinear interpolation is performed between adjacent nodes on 
        the input mesh

    Type : str, optional
        Determines whether to generate a surface mesh ('surf') or volume mesh ('vol'), 
        by default 'surf'
    flip : bool, optional
        Flip the interior/exterior of the mesh, by default False. By default, 
        values less than the threshold are assumed to be the “inside” of the 
        mesh. If the inside is denoted by values greater than the threshold, set 
        flip=True.
    mixed_elements : bool, optional
        If True and Type='surf', the generated mesh will have mixed element types 
        (triangles/quadrilateral otherwise a single element type (triangles), by 
        default False.
    cleanup : bool, optional
        Determines whether or not to perform mesh cleanup, removing degenerate elements and duplicate nodes, by default True

    Returns
    -------
    NewCoords : np.ndarray
        Numpy array of node coordinates for the contour mesh
    NewConn : list
        List of node connectivities 

    """   

    NewCoords = []
    NewConn = []
    Anchors = []
    AnchorAxis = []
    AnchorDir = []
    NodeValues = np.array([v-threshold for v in NodeValues]).astype('float64')
    if flip:
        NodeValues = -1*NodeValues
    edgeLookup = [
        [0, 0],  # Corner 0
        [1, 1],  # Corner 1
        [2, 2],  # Corner 2
        [3, 3],  # Corner 3
        [0, 1],  # Edge 0
        [1, 2],  # Edge 1
        [2, 3],  # Edge 2
        [3, 0],  # Edge 3
        ]
    
    for e in range(len(NodeConn)):
        vals = np.array([NodeValues[node] for node in NodeConn[e]])
        inside = [1 if v < 0 else 0 for v in vals]
        i = int("".join(str(j) for j in inside), 2)
        if Type.lower() == 'surf':
            if mixed_elements:
                NewElems = MSMixed_Lookup[i]
            else:
                NewElems = MSTriangle_Lookup[i]
        elif Type.lower() == 'line':
            NewElems = MSEdge_Lookup[i]
        else:
            raise Exception('Invalid method. Must be "surf" or "line".')
    
        if len(NewElems) > 0:
            for t in NewElems:
                elem = []
                for n in t:
                    node1 = NodeConn[e][edgeLookup[n][0]]
                    node2 = NodeConn[e][edgeLookup[n][1]]
                    coords1 = NodeCoords[node1]
                    coords2 = NodeCoords[node2]
                    v1 = NodeValues[node1]
                    v2 = NodeValues[node2]
                    if interpolation == 'midpoint' or v1 == v2:
                        newNode = [
                            (coords1[0] + coords2[0])/2,
                            (coords1[1] + coords2[1])/2,
                            (coords1[2] + coords2[2])/2
                            ]
                        elem.append(len(NewCoords))
                        NewCoords.append(newNode)
                    elif interpolation == 'linear':
                        newNode = [
                            coords1[0] + (0-v1)*(coords2[0]-coords1[0])/(v2-v1),
                            coords1[1] + (0-v1)*(coords2[1]-coords1[1])/(v2-v1),
                            coords1[2] + (0-v1)*(coords2[2]-coords1[2])/(v2-v1)
                            ]
                        elem.append(len(NewCoords))
                        NewCoords.append(newNode)                            
                    else:
                        raise Exception('Invalid interpolation method')
                    if return_anchors:
                        anchor = [node1,node2][np.argmin([v1,v2])] # Pick the point in negative domain 
                        Anchors.append(anchor)

                        if n == 4 or n == 6:
                            AnchorAxis.append(0)
                        elif n == 5 or n == 7: 
                            AnchorAxis.append(1)
                        else:
                            AnchorAxis.append(-1)

                        mid = np.array([
                            (coords1[0] + coords2[0])/2,
                            (coords1[1] + coords2[1])/2,
                            (coords1[2] + coords2[2])/2
                            ])
                        if np.all(mid-NodeCoords[anchor] >= 0):
                            AnchorDir.append(1)
                        else:
                            AnchorDir.append(-1)

                    
                if len(elem) > 0:
                    NewConn.append(elem)  
    NewCoords = np.asarray(NewCoords)
    if cleanup:                  
        NewCoords,NewConn,Idx = utils.DeleteDuplicateNodes(NewCoords,NewConn,return_idx=True)
        if interpolation=='linear' and Type.lower()=='surf':
            NewCoords,NewConn = utils.DeleteDegenerateElements(NewCoords,NewConn,strict=True)
    else:
        Idx = np.arange(len(NewCoords),dtype=int)
    if return_anchors:
        Anchors = np.array(Anchors)
        AnchorAxis = np.array(AnchorAxis)
        AnchorDir = np.array(AnchorDir)
        return NewCoords, NewConn, Anchors[Idx], AnchorAxis[Idx], AnchorDir[Idx]
    return NewCoords, NewConn

def MarchingTriangles(TriNodeCoords, TriNodeConn, NodeValues, threshold=0, interpolation='linear', Type='surf', mixed_elements=False, flip=False, return_NodeValues=False, return_ParentIds=False, cleanup_tol=1e-10, cleanup=True):
    """
    Marching tetrahedra algorithm for extracting an isoline from a triangular
    mesh. This can be used to generate either a surface mesh of the region 
    above or below the threshold or a line mesh of the isoline. Note that this
    method is different than the Marching Triangles algorithm by :cite:`Hilton1996`
    and is instead analogous to the other "marching" contouring algorithms.

    Parameters
    ----------
    TriNodeCoords : array_like
        Node coordinates for the input triangular mesh
    TriNodeConn : array_like
        Node connectivity for the input triangular mesh. All elements must be triangles.
    NodeValues : array_like
        Values at each node in the mesh that are used to extract the isoline
    threshold : int, optional
        Isoline level, by default 0
    interpolation : str, optional
        Interpolation to interpolate the position of the new nodes along the edges
        of the input mesh, by default 'linear'.

        'midpoint' - No interpolation is performed, new nodes are placed at the midpoint of edges

        'linear' - Linear interpolation is performed between adjacent nodes on the input mesh.

    Type : str, optional
        Determines whether to generate a surface mesh ('surf') or volume mesh ('vol'), 
        by default 'surf'
    mixed_elements : bool, optional
        If True and Type='surf', the generated mesh will have mixed element types 
        (triangles/quadrilateral otherwise a single element type (triangles), by 
        default False.
    flip : bool, optional
        Flip the interior/exterior of the mesh, by default False.
        By default, values less than the threshold are assumed to be the "inside" 
        of the mesh. If the inside is denoted by values greater than the threshold,
        set flip=True.
    return_NodeValues : bool, optional
        Return the node values for each node in the triangular mesh. If Type='line',
        these will all be `threshold`, if Type='surf', these will be `threshold` 
        for the boundary nodes and the original grid values for the interior nodes
    return_ParentIds : bool, optional
        Return Ids that refer to the original triangles that each of the new elements is connected to, by default False.
    cleanup_tol : float, optional
        Tolerance value used to classify whether two nodes are sufficiently close to be considered a single node (see :func:`mymesh.utils.DeleteDuplicateNodes`), by default 1e-12.
    Returns
    -------
    NodeCoords : np.ndarray
        Node coordinates of the generated mesh
    NodeConn : list
        Node connetivity for the generated mesh
    NewValues : np.ndarray, optional
        Node values transfered to the generated mesh, returned if return_NodeValues=True 
    ParentIds : np.ndarray, optional
        Element ids that refer to the original triangles that each of the new elements is connected to,
        returned if return_ParentIds=True


    Examples
    --------


    """    
    TriNodeCoords = np.asarray(TriNodeCoords)
    TriNodeConn = np.asarray(TriNodeConn)
    NodeValues = np.asarray(NodeValues).flatten()
    assert len(NodeValues) == len(TriNodeCoords), 'Number of nodes must match number of node values.'

    #      5
    #     / \
    #    /   \
    #   2     1
    #  /       \
    # 3----0----4
    
    
    if interpolation.lower() == 'midpoint' or interpolation.lower() == 'linear':
        edgeLookup = np.array([
            [0, 1],  # (0) Edge 0 - Between nodes 0 and 1
            [1, 2],  # (1) Edge 1
            [2, 0],  # (2) Edge 2
            [0, 0],  # (3) Node 0
            [1, 1],  # (4) Node 1
            [2, 2],  # (5) Node 2
            ])
        PadEdgeLookup = np.vstack([edgeLookup, [-1,-1]])
        ninterppts = 2
    else:
        raise ValueError(f'Interpolation must be one of "midpoint", "linear", or "quadratic", not "{interpolation:s}"')
    
    NodeValues = np.asarray(NodeValues) - threshold
    if flip:
        NodeValues = -1*NodeValues

    # Determine configuration of nodes
    TriVals = NodeValues[TriNodeConn[:,:3]]
    inside = TriVals <= 0
    if not np.any(inside):
        NodeCoords = np.empty((0,3))
        NodeConn = np.empty((0,3),dtype=int)
        NewValues = np.empty((0))
        ParentIds = np.empty((0), dtype=int)
        if return_NodeValues:
            if return_ParentIds:
                return NodeCoords, NodeConn, NewValues, ParentIds
            return NodeCoords, NodeConn, NewValues
        if return_ParentIds:
            return NodeCoords, NodeConn, ParentIds
        return NodeCoords, NodeConn

    ints = np.sum(inside[:,:3] * 2**np.arange(0,3)[::-1], axis=1)

    # Query lookup tables
    if Type.lower() == 'surf':
        if mixed_elements:
            element_lists = MTriMixed_Lookup[ints]
        else:
            element_lists = MTri_Lookup[ints]
    elif Type.lower() == 'line':
        element_lists = MTriEdge_Lookup[ints]
    else:
        raise ValueError('Invalid Type, Type must be "surf" or "line".')

    # Process lookup results
    trinum, elem = zip(*[(i,e) for i,lst in enumerate(element_lists) for e in lst if len(lst) > 0])
    trinum = np.array(trinum)
    nelem = len(trinum)

    relevant_tris = TriNodeConn[trinum]
    
    PadElem = utils.PadRagged(elem)
    pad_relevant_tris = np.hstack([relevant_tris, -1*np.ones((len(elem),1))])
    
    lookup_indices = pad_relevant_tris[:, PadEdgeLookup]
    interpolation_pairs = (lookup_indices[np.arange(len(elem))[:, None], PadElem]).reshape((np.prod(PadElem.shape),ninterppts))
    interpolation_pairs = (interpolation_pairs[np.any(interpolation_pairs!=-1,axis=1)]).astype(int)
    
    uinterpolation_pairs,inv = np.unique(np.sort(interpolation_pairs,axis=1),axis=0,return_inverse=True)

    # Interpolation
    if interpolation.lower() == 'midpoint':
        NodeCoords = np.mean(TriNodeCoords[uinterpolation_pairs],axis=1)
        if return_NodeValues: 
            NewValues = np.zeros(len(NodeCoords))
            NewValues = np.mean(NodeValues[uinterpolation_pairs],axis=1)

    elif interpolation.lower() == 'linear':
        check = uinterpolation_pairs[:,0] != uinterpolation_pairs[:,1]
        coords1 = TriNodeCoords[uinterpolation_pairs[check,0]]
        coords2 = TriNodeCoords[uinterpolation_pairs[check,1]]
        vals1 = NodeValues[uinterpolation_pairs[check,0]][:,None]
        vals2 = NodeValues[uinterpolation_pairs[check,1]][:,None]
        NodeCoords = np.copy(TriNodeCoords[uinterpolation_pairs[:,0]])
        coefficient = (0 - vals1)/(vals2-vals1)
        position = coords1 + coefficient*(coords2 - coords1)
        position[coefficient.flatten() >= 1] = coords2[coefficient.flatten() >= 1]  # This is to prevent floating pt errors inverting elements
        NodeCoords[check] =  position

        if return_NodeValues: 
            NewValues = np.zeros(len(NodeCoords))
            NewValues[~check] = NodeValues[uinterpolation_pairs[~check,0]]

    else:
        raise ValueError('interpolation must be either "midpoint", "linear", or "quadratic"')

    # Format points into NodeCoords, NodeConn
    if Type.lower() == 'line':
        NodeConn = inv[np.reshape(np.arange(nelem*2), (nelem, 2))]
    elif Type.lower() == 'surf' and not mixed_elements:
        NodeConn = inv[np.reshape(np.arange(nelem*3), (nelem, 3))]
    else:
        lengths = [len(e) for e in elem]
        sums = np.append([0],np.cumsum(lengths))
        NodeConn = [[inv[n+sums[i]] for n in range(lengths[i])] for i in range(len(lengths))]
    ParentIds = trinum

    if cleanup: 
        NodeCoords,NodeConn,Idx = utils.DeleteDuplicateNodes(NodeCoords,NodeConn,return_idx=True, tol=cleanup_tol)
        if return_NodeValues:
            NewValues = NewValues[Idx]
        # NodeCoords,NodeConn,EIdx = utils.CleanupDegenerateElements(NodeCoords,NodeConn,Type=Type, return_idx=True)
        if return_ParentIds:
            ParentIds = ParentIds[EIdx]

    if return_NodeValues:
        if flip:
            NewValues = -1*NewValues
        NewValues += threshold
        if return_ParentIds:
            return NodeCoords, NodeConn, NewValues, ParentIds
        return NodeCoords, NodeConn, NewValues
    if return_ParentIds:
        return NodeCoords, NodeConn, ParentIds
    return NodeCoords, NodeConn

def MarchingCubesImage(I, h=1, threshold=0, interpolation='linear', method='original', VertexValues=False, edgemode='constant', flip=False, cleanup=True):
    """
    Marching cubes algorithm :cite:p:`Lorensen1987` applied to 3D image data.

    Parameters
    ----------
    I : np.ndarray
        Image array (3D) of grayscale data. The (0,1,2) axes of the image should
        correspond to the (z,y,x) axes, respectively.
    h : float or tuple, optional
        Element (pixel) size. Can be specified as a single value for isotropic pixel sizes or a tuple of three values, by default 1.
        If a tuple, entries should correspond to (hx, hy, hz).
    threshold : int, optional
        Isosurface level that defines the boundary, by default 0
    interpolation : str, optional
        Interpolation to interpolate the position of the new nodes along the edges of the input mesh, by default 'linear'. 
        
        'midpoint' - No interpolation is performed, new nodes are placed at the midpoint of edges 
        
        'linear' - Bilinear interpolation is performed between adjacent nodes on the input mesh

        'cubic' - Bicubic interpolation is performed between adjacent nodes on the input mesh
    method : str, optional
        Mesh generation method, either 'original' for the original marching cubes
        algorithm or '33' for the marching cubes 33 algorithm :cite:p:`Chernyaev1995`.
    VertexValues : bool, optional
        If True, the values in the image array are treated as the vertices of 
        the cubes, otherwise, they are treated as voxel values and vertices
        are obtained through interpolation, by default, False.
    flip : bool, optional
        Flip the interior/exterior of the mesh, by default False. By default, values less than the threshold are assumed to be the “inside” of the mesh. If the inside is denoted by values greater than the threshold, set flip=True.
    edgemode : str, optional
        For interpolation='cubic', edgemode specifies how to handle boundary nodes. The image matrix will be padded using ``np.pad(I, 1, mode=edgemode)``, by default 'constant'
    cleanup : bool, optional
        Determines whether or not to perform mesh cleanup, removing degenerate elements and duplicate nodes, by default True

    Returns
    -------
    NewCoords : np.ndarray
        Numpy array of node coordinates for the contour mesh
    NewConn : list
        List of node connectivities 

    """  
    
    if np.issubdtype(I.dtype, np.unsignedinteger):
        I = I.astype(float)
        
    assert len(I.shape) == 3, 'I must be a 3D numpy array of image data. For 2D, use MarchingSquaresImage.'
    I = I - threshold  
    if method == 'original':
        LookupTable = MC_Lookup
    elif method == '33':
        LookupTable = MC33_Lookup
    else:
        raise Exception('Invalid method. Must be "original" or "33".')
    if flip: 
        I = -I
    if (not np.any(I <= 0)) or np.all(I > 0):
        return np.empty((0,3)), []
    if isinstance(h, (int, float, np.number)):
        h = (h,h,h)
    elif isinstance(h, tuple):
        # Flipping so that element sizes match image axes
        h = h[::-1]
    
    Padding = 0
    if interpolation == "cubic":
        Padding = 1
    
    if not VertexValues:
        x1 = np.arange(0,I.shape[0])*h[0] + h[0]/2
        y1 = np.arange(0,I.shape[1])*h[1] + h[1]/2
        z1 = np.arange(0,I.shape[2])*h[2] + h[2]/2

        Xv, Yv, Zv = np.meshgrid(np.arange(-Padding,I.shape[0]+(1+Padding)),
                                 np.arange(-Padding,I.shape[1]+(1+Padding)),
                                 np.arange(-Padding,I.shape[2]+(1+Padding)),
                                 indexing='ij'
                                 )*np.asarray(h)[:,None,None,None]
        
        # TODO: This is major bottleneck for cubic
        interp = scipy.interpolate.RegularGridInterpolator((x1,y1,z1),I,fill_value=None,method='linear',bounds_error=False)
        Iv = interp((Xv.flatten(),Yv.flatten(),Zv.flatten())).reshape(Xv.shape)
        X = Xv; Y = Yv; Z = Zv; I = Iv; 
    else:
        X, Y, Z = np.meshgrid(np.arange(-Padding,I.shape[0]+(Padding)),
                                 np.arange(-Padding,I.shape[1]+(Padding)),
                                 np.arange(-Padding,I.shape[2]+(Padding)),
                                 indexing='ij'
                                 )*np.asarray(h)[:,None,None,None]
        I = np.pad(I,Padding,mode=edgemode)
    edgeLookup = np.array([
        [0, 1],  # Edge 0 - Between nodes 0 and 1
        [1, 2],  # Edge 1
        [2, 3],  # Edge 2
        [3, 0],  # Edge 3
        [0, 4],  # Edge 4
        [1, 5],  # Edge 5
        [2, 6],  # Edge 6
        [3, 7],  # Edge 7
        [4, 5],  # Edge 8
        [5, 6],  # Edge 9
        [6, 7],  # Edge 10
        [7, 4],  # Edge 11
        # []       # Center
        ])
    iidx = np.repeat(np.arange(X.shape[0]-1-2*Padding),(X.shape[1]-1-2*Padding)*(X.shape[2]-1-2*Padding))+Padding
    jidx = np.repeat(np.tile(np.arange(X.shape[1]-1-2*Padding),(X.shape[0]-1-2*Padding)),(X.shape[2]-1-2*Padding))+Padding
    kidx = np.tile(np.arange(X.shape[2]-1-2*Padding),(X.shape[1]-1-2*Padding)*(X.shape[0]-1-2*Padding))+Padding
    
    icubes = np.vstack([iidx,iidx+1,iidx+1,iidx,iidx,iidx+1,iidx+1,iidx]).T
    jcubes = np.vstack([jidx,jidx,jidx+1,jidx+1,jidx,jidx,jidx+1,jidx+1]).T
    kcubes = np.vstack([kidx,kidx,kidx,kidx,kidx+1,kidx+1,kidx+1,kidx+1]).T
    
    vals = I[icubes,jcubes,kcubes]
    inside = (vals <= 0).astype(int)
    if not np.any(inside):
        return np.empty((0,3)), np.empty((0,3))
    tableIdx = np.dot(inside, 2**np.arange(inside.shape[1] - 1, -1, -1))
    
    if method == 'original':
        flip = tableIdx > 127
        tableIdx[flip] = 128 - (tableIdx[flip]-127)
        edgeList =  LookupTable[tableIdx]
        edgeConnections = np.array([x[::-1] if flip[i] else x for i,y in enumerate(edgeList) for x in y if len(x) != 0])
        numbering = np.array([j for j,y in enumerate(edgeList) for i,x in enumerate(y) if len(x) != 0])
    
    elif method == '33':
        ##### TODO ####
        pass
    if len(numbering) == 0:
        return np.empty((0,3)), np.empty((0,3))
    i_indices = icubes[numbering][np.arange(len(numbering))[:, np.newaxis, np.newaxis], edgeLookup[edgeConnections]]
    j_indices = jcubes[numbering][np.arange(len(numbering))[:, np.newaxis, np.newaxis], edgeLookup[edgeConnections]]
    k_indices = kcubes[numbering][np.arange(len(numbering))[:, np.newaxis, np.newaxis], edgeLookup[edgeConnections]]
    
    ishiftdir = i_indices[:,:,1] - i_indices[:,:,0]
    jshiftdir = j_indices[:,:,1] - j_indices[:,:,0]
    kshiftdir = k_indices[:,:,1] - k_indices[:,:,0]
    if interpolation == "cubic":
        
        i_interp = np.stack([i_indices[:,:,0]-1*ishiftdir, i_indices[:,:,0], i_indices[:,:,1], i_indices[:,:,1]+1*ishiftdir],axis=2)
        j_interp = np.stack([j_indices[:,:,0]-1*jshiftdir, j_indices[:,:,0], j_indices[:,:,1], j_indices[:,:,1]+1*jshiftdir],axis=2)
        k_interp = np.stack([k_indices[:,:,0]-1*kshiftdir, k_indices[:,:,0], k_indices[:,:,1], k_indices[:,:,1]+1*kshiftdir],axis=2)
        
        X_interp = X[i_interp,j_interp,k_interp]
        Y_interp = Y[i_interp,j_interp,k_interp]
        Z_interp = Z[i_interp,j_interp,k_interp]
        I_interp = I[i_interp,j_interp,k_interp]
        
        
        x = X_interp.reshape(I_interp.shape[0]*I_interp.shape[1],I_interp.shape[2],order='F')
        y = Y_interp.reshape(I_interp.shape[0]*I_interp.shape[1],I_interp.shape[2],order='F')
        z = Z_interp.reshape(I_interp.shape[0]*I_interp.shape[1],I_interp.shape[2],order='F')
        v = I_interp.reshape(I_interp.shape[0]*I_interp.shape[1],I_interp.shape[2],order='F')
        
        # Build coefficients for cubic polynomials
        xbool = x[:,1]!=x[:,2] # Only interpolate in x if on an x edge
        ybool = y[:,1]!=y[:,2] # Only interpolate in y if on an y edge
        zbool = z[:,1]!=z[:,2] # Only interpolate in y if on an y edge

        
        xsort = np.argsort(x[xbool],axis=1)
        ysort = np.argsort(y[ybool],axis=1)
        zsort = np.argsort(z[zbool],axis=1)
        x[xbool,:] = np.take_along_axis(x[xbool], xsort, axis=1)
        y[ybool,:] = np.take_along_axis(y[ybool], ysort, axis=1)
        z[zbool,:] = np.take_along_axis(z[zbool], zsort, axis=1)
        v[xbool,:] = np.take_along_axis(v[xbool], xsort, axis=1)
        v[ybool,:] = np.take_along_axis(v[ybool], ysort, axis=1)
        v[zbool,:] = np.take_along_axis(v[zbool], zsort, axis=1)
        
        xAs = np.stack([x[xbool]**3, x[xbool]**2, x[xbool]**1, x[xbool]**0],axis=2)
        yAs = np.stack([y[ybool]**3, y[ybool]**2, y[ybool]**1, y[ybool]**0],axis=2)
        zAs = np.stack([z[zbool]**3, z[zbool]**2, z[zbool]**1, z[zbool]**0],axis=2)
        xCoeff = np.linalg.solve(xAs,v[xbool,:,None])
        yCoeff = np.linalg.solve(yAs,v[ybool,:,None])
        zCoeff = np.linalg.solve(zAs,v[zbool,:,None])
        
        # cf. https://en.wikipedia.org/wiki/Cubic_equation#General_cubic_formula
        Roots = []
        for coeff in [xCoeff, yCoeff, zCoeff]:
            a = coeff[:,0]
            b = coeff[:,1]
            c = coeff[:,2]
            d = coeff[:,3]
            
            delta0 = b**2 - 3*a*c
            delta1 = 2*b**3 - 9*a*b*c + 27*a**2*d
            
            C = ((delta1 + np.sqrt(delta1**2+0J - 4*delta0**3))/2)**(1/3)
            C[C==0] = ((delta1[C==0] - np.sqrt(delta1[C==0]**2 - 4*delta0[C==0]**3))/2)**(1/3)
            
            zeta = (-1 + np.sqrt(-3+0J))/2
            roots = np.column_stack([-1/(3*a) * ( b + zeta**k*C + delta0/(zeta**k*C) ) for k in [0, 1, 2]])
            
            # overwrite with quadratic for a ~ 0
            near0 = np.isclose(a,0).flatten()
            with np.errstate(divide='ignore', invalid='ignore'):
                roots[near0] = np.column_stack([(-c[near0] + np.sqrt(c[near0]**2 - 4*b[near0]*d[near0]))/(2*b[near0]), 
                                        (-c[near0] - np.sqrt(c[near0]**2 - 4*b[near0]*d[near0]))/(2*b[near0]),
                                            np.repeat(np.nan,len(b[near0]))])
            realcheck = np.isclose(np.imag(roots), 0)
            roots[realcheck]  = np.real(roots[realcheck])
            Roots.append(roots)
        xRoots = Roots[0]
        yRoots = Roots[1]
        zRoots = Roots[2]
        
        NewCoords = np.nan*np.ones((v.shape[0],3))
        eps = 1e-10
        warnings.filterwarnings('ignore', category=np.exceptions.ComplexWarning)
        with np.errstate(divide='ignore', invalid='ignore'):
            xCheck = (x[xbool,1,None] <= xRoots) & (x[xbool,2,None] >= xRoots) & np.isreal(xRoots)
            xcubic = np.repeat(False,len(xbool)); xlinear = np.repeat(False,len(xbool))
            xcubic[xbool] = np.sum(xCheck,axis=1)==1
            xlinear[xbool] = np.sum(xCheck,axis=1)!=1
            xlin = x[xbool][np.sum(xCheck,axis=1)!=1]; vxlin = v[xbool][np.sum(xCheck,axis=1)!=1]; 
            NewCoords[xcubic,0] = xRoots[np.sum(xCheck,axis=1)==1][xCheck[np.sum(xCheck,axis=1)==1]]
            # Fall back to linear for any failed interpolations
            NewCoords[xlinear,0] = xlin[:,1] + np.nan_to_num((0-vxlin[:,1])*(xlin[:,2]-xlin[:,1])/(vxlin[:,2]-vxlin[:,1]))
            NewCoords[xbool,1] = y[xbool,1]
            NewCoords[xbool,2] = z[xbool,1]
            
            yCheck = (y[ybool,1,None] <= yRoots) & (y[ybool,2,None] >= yRoots) & np.isreal(yRoots)
            ycubic = np.repeat(False,len(ybool)); ylinear = np.repeat(False,len(ybool))
            ycubic[ybool] = np.sum(yCheck,axis=1)==1
            ylinear[ybool] = np.sum(yCheck,axis=1)!=1
            ylin = y[ybool][np.sum(yCheck,axis=1)!=1]; vylin = v[ybool][np.sum(yCheck,axis=1)!=1]; 
            NewCoords[ybool,0] = x[ybool,1]
            NewCoords[ycubic,1] = yRoots[np.sum(yCheck,axis=1)==1][yCheck[np.sum(yCheck,axis=1)==1]]
            # Fall back to linear for any failed interpolations
            NewCoords[ylinear,1] = ylin[:,1] + np.nan_to_num((0-vylin[:,1])*(ylin[:,2]-ylin[:,1])/(vylin[:,2]-vylin[:,1]))
            NewCoords[ybool,2] = z[ybool,1]
            
            zCheck = (z[zbool,1,None] <= zRoots) & (z[zbool,2,None] >= zRoots) & np.isreal(zRoots)
            zcubic = np.repeat(False,len(zbool)); zlinear = np.repeat(False,len(zbool))
            zcubic[zbool] = np.sum(zCheck,axis=1)==1
            zlinear[zbool] = np.sum(zCheck,axis=1)!=1
            zlin = z[zbool][np.sum(zCheck,axis=1)!=1]; vzlin = v[zbool][np.sum(zCheck,axis=1)!=1]; 
            NewCoords[zbool,0] = x[zbool,1]
            NewCoords[zbool,1] = y[zbool,1]
            NewCoords[zcubic,2] = zRoots[np.sum(zCheck,axis=1)==1][zCheck[np.sum(zCheck,axis=1)==1]]
            # Fall back to linear for any failed interpolations
            NewCoords[zlinear,2] = zlin[:,1] + np.nan_to_num((0-vzlin[:,1])*(zlin[:,2]-zlin[:,1])/(vzlin[:,2]-vzlin[:,1]))
        
    elif interpolation == "linear" or interpolation == "midpoint":
        
        i_interp = i_indices
        j_interp = j_indices
        k_interp = k_indices
        
        X_interp = X[i_interp,j_interp,k_interp]
        Y_interp = Y[i_interp,j_interp,k_interp]
        Z_interp = Z[i_interp,j_interp,k_interp]
        I_interp = I[i_interp,j_interp,k_interp]
        
        x = X_interp.reshape(I_interp.shape[0]*I_interp.shape[1],I_interp.shape[2],order='F')
        y = Y_interp.reshape(I_interp.shape[0]*I_interp.shape[1],I_interp.shape[2],order='F')
        z = Z_interp.reshape(I_interp.shape[0]*I_interp.shape[1],I_interp.shape[2],order='F')
        v = I_interp.reshape(I_interp.shape[0]*I_interp.shape[1],I_interp.shape[2],order='F')
        
        if interpolation == "linear":
            with np.errstate(divide='ignore', invalid='ignore'):
                NewCoords = np.vstack([
                     x[:,0] + np.nan_to_num((0-v[:,0])*(x[:,1]-x[:,0])/(v[:,1]-v[:,0])),
                     y[:,0] + np.nan_to_num((0-v[:,0])*(y[:,1]-y[:,0])/(v[:,1]-v[:,0])),
                     z[:,0] + np.nan_to_num((0-v[:,0])*(z[:,1]-z[:,0])/(v[:,1]-v[:,0]))
                    ]).T
        else:
            NewCoords = np.vstack([
                 (x[:,0] + x[:,1])/2,
                 (y[:,0] + y[:,1])/2,
                 (z[:,0] + z[:,1])/2,
                ]).T
    else:
        raise Exception('Invalid input "{:s}" for interpolation. Must be one of "midpoint", "linear", or "cubic".'.format(interpolation))
    
    NewCoords = np.fliplr(NewCoords) # Flipping so that index 0 for the image -> z axis
    NewConn = np.fliplr(np.arange(len(NewCoords)).reshape(edgeConnections.shape,order='F'))
    if cleanup:
        NewCoords,NewConn,Idx = utils.DeleteDuplicateNodes(NewCoords,NewConn,return_idx=True)
        if (interpolation=='linear' or interpolation=='cubic') and method=='triangle':
            NewCoords,NewConn = utils.DeleteDegenerateElements(NewCoords,NewConn,strict=True)
            
    return NewCoords, NewConn

def MarchingCubes(VoxelNodeCoords,VoxelNodeConn,NodeValues,threshold=0,interpolation='linear',method='33',flip=False, return_anchors=False):
    """
    Marching cubes algorithm :cite:p:`Lorensen1987` for extracting an isosurface from a hexahedral mesh.

    Parameters
    ----------
    
    NodeCoords : array_like
        Node coordinates of a two dimensional hexahedral grid (e.g. from 
        primitives.Grid)
    NodeConn : array_like
        Node connectivities of a two dimensional grid of rectilinear grid (e.g. 
        from primitives.Grid) of hexahedra. 
    NodeValues : array_like
        Array of node values corresponding to each node in the mesh.
    threshold : int, optional
        Isosurface level that defines the boundary, by default 0
    interpolation : str, optional
        Interpolation to interpolate the position of the new nodes along the edges of the input mesh, by default 'linear'. 
        
        'midpoint' - No interpolation is performed, new nodes are placed at the midpoint of edges 
        
        'linear' - Bilinear interpolation is performed between adjacent nodes on the input mesh
    method : str, optional
        Mesh generation method, either 'original' for the original marching cubes
        algorithm or '33' for the marching cubes 33 algorithm :cite:p:`Chernyaev1995`.
    flip : bool, optional
        Flip the interior/exterior of the mesh, by default False. By default, values less than the threshold are assumed to be the “inside” of the mesh. If the inside is denoted by values greater than the threshold, set flip=True.

    Returns
    -------
    NewCoords : np.ndarray
        Numpy array of node coordinates for the contour mesh
    NewConn : list
        List of node connectivities 

    """  
    
    TriNodeCoords = []
    TriNodeConn = []
    Anchors = []
    AnchorAxis = []
    AnchorDir = []
    NodeValues = np.asarray(NodeValues) - threshold
    if flip:
        NodeValues = -1*NodeValues
    if method == '33':
        _MarchingCubes33Lookup.LookupTable = MC33_Lookup
        _MarchingCubes33Lookup.FaceTests = MC33_FaceTest
        _MarchingCubes33Lookup.Cases = MC33_Cases
        _MarchingCubes33Lookup.Signs = MC33_Signs
    elif method == 'original':
        _MarchingCubesLookup.LookupTable = MC_Lookup
    else:
        raise Exception('Unknown method "{:s}"'.format(method))
    edgeLookup = [
        [0, 1],  # Edge 0 - Between nodes 0 and 1
        [1, 2],  # Edge 1
        [2, 3],  # Edge 2
        [3, 0],  # Edge 3
        [0, 4],  # Edge 4
        [1, 5],  # Edge 5
        [2, 6],  # Edge 6
        [3, 7],  # Edge 7
        [4, 5],  # Edge 8
        [5, 6],  # Edge 9
        [6, 7],  # Edge 10
        [7, 4],  # Edge 11
        []       # Center
        ]
    arrayCoords = np.array(VoxelNodeCoords)
    for e in range(len(VoxelNodeConn)):
        vals = np.array([NodeValues[node] for node in VoxelNodeConn[e]])
        inside = [1 if v <= 0 else 0 for v in vals]
        i = int("".join(str(j) for j in inside), 2)
        if method == 'original':
            TriElems = _MarchingCubesLookup(i)
        elif method == '33':
            TriElems = _MarchingCubes33Lookup(i,vals)
        else:
            raise Exception('Invalid method')
    
        if len(TriElems) > 0:
            for t in TriElems:
                elem = []
                for n in t:
                    if n == 12:
                        newNode = np.mean([VoxelNodeCoords[node] for node in VoxelNodeConn[e]], axis=0)
                        elem.append(len(TriNodeCoords))
                        TriNodeCoords.append(newNode)
                    else:
                        node1 = VoxelNodeConn[e][edgeLookup[n][0]]
                        node2 = VoxelNodeConn[e][edgeLookup[n][1]]
                        coords1 = VoxelNodeCoords[node1]
                        coords2 = VoxelNodeCoords[node2]
                        v1 = NodeValues[node1]
                        v2 = NodeValues[node2]
                        if interpolation == 'midpoint' or v1 == v2:
                            newNode = [
                                (coords1[0] + coords2[0])/2,
                                (coords1[1] + coords2[1])/2,
                                (coords1[2] + coords2[2])/2
                                ]
                            elem.append(len(TriNodeCoords))
                            TriNodeCoords.append(newNode)
                        elif interpolation == 'linear':
                            newNode = [
                                coords1[0] + (0-v1)*(coords2[0]-coords1[0])/(v2-v1),
                                coords1[1] + (0-v1)*(coords2[1]-coords1[1])/(v2-v1),
                                coords1[2] + (0-v1)*(coords2[2]-coords1[2])/(v2-v1)
                                ]
                            if np.sign(v2) == np.sign(v1):
                                print('Marching cubes fuckup')
                                print(str(e) + str(np.sign(vals)) + str(edgeLookup[n]))
                            elem.append(len(TriNodeCoords))
                            TriNodeCoords.append(newNode)                            
                        else:
                            raise Exception('Invalid interpolation method')
                        if return_anchors:
                            if flip:
                                anchor = [node1,node2][np.argmax([v1,v2])] # Pick the point in positive domain
                            else:
                                anchor = [node1,node2][np.argmin([v1,v2])] # Pick the point in negative domain
                            Anchors.append(anchor)

                            if n == 0 or n == 2 or n == 8 or n == 10:
                                AnchorAxis.append(0)
                            elif n == 1 or n == 3 or n == 9 or n == 11:
                                AnchorAxis.append(1)    
                            elif n == 4 or n == 5 or n == 6 or n == 7:
                                AnchorAxis.append(2)
                            else:
                                AnchorAxis.append(-1)

                            mid = np.array([
                                (coords1[0] + coords2[0])/2,
                                (coords1[1] + coords2[1])/2,
                                (coords1[2] + coords2[2])/2
                                ])
                            if np.all(mid-VoxelNodeCoords[anchor] >= 0):
                                AnchorDir.append(1)
                            else:
                                AnchorDir.append(-1)

                if len(elem) > 0:
                    TriNodeConn.append(elem)  
                      
    TriNodeCoords,TriNodeConn,Idx = utils.DeleteDuplicateNodes(TriNodeCoords,TriNodeConn,return_idx=True)
    if interpolation=='linear':
        TriNodeCoords,TriNodeConn = utils.DeleteDegenerateElements(TriNodeCoords,TriNodeConn,strict=True)
    TriNodeCoords = np.asarray(TriNodeCoords)
    if return_anchors: 
        Anchors = np.array(Anchors)
        AnchorAxis = np.array(AnchorAxis)
        AnchorDir = np.array(AnchorDir)
        return TriNodeCoords, TriNodeConn, Anchors[Idx], AnchorAxis[Idx], AnchorDir[Idx]
    return np.asarray(TriNodeCoords), TriNodeConn

def MarchingTetrahedra(TetNodeCoords, TetNodeConn, NodeValues, threshold=0, interpolation='linear', Type='surf', mixed_elements=False, flip=False, return_NodeValues=False, return_ParentIds=False, cleanup_tol=1e-10, cleanup=True):
    """
    Marching tetrahedra algorithm :cite:p:`Bloomenthal1994` for extracting an isosurface from a tetrahedral mesh. This can be used to generate either a surface mesh or a volume mesh, with either simplex elements (triangles, tetrahedra) or mixed elements (triangles/quadrilaterals, tetrahedra/wedges).

    Parameters
    ----------
    TetNodeCoords : array_like
        Node coordinates for the input tetrahedral mesh
    TetNodeConn : array_like
        Node connectivity for the input tetrahedral mesh. All elements must be tetrahedra.
    NodeValues : array_like
        Values at each node in the mesh that are used to extract the isosurface
    threshold : int, optional
        Isosurface level, by default 0
    interpolation : str, optional
        Interpolation to interpolate the position of the new nodes along the edges
        of the input mesh, by default 'linear'.

        'midpoint' - No interpolation is performed, new nodes are placed at the midpoint of edges

        'linear' - Linear interpolation is performed between adjacent nodes on the input mesh.

        'quadratic' - Quadratic interpolation is performed. This requires that the provided
        tetrahedral mesh has quadratic elements (10 node tetrahedra).
    Type : str, optional
        Determines whether to generate a surface mesh ('surf') or volume mesh ('vol'), 
        by default 'surf'
    mixed_elements : bool, optional
        If True, the generated mesh will have mixed element types 
        (triangles/quadrilaterals, tetrahedra/wedges), otherwise a single element 
        type (triangles, tetrahedra), by default False. For surface meshes, either 
        version is generated directly. For a volume mesh, a mixed mesh is generated 
        initially and then converted to tetrahedra using converter.solid2tets()
        which results in additional computational cost.
    flip : bool, optional
        Flip the interior/exterior of the mesh, by default False.
        By default, values less than the threshold are assumed to be the "inside" 
        of the mesh. If the inside is denoted by values greater than the threshold,
        set flip=True.
    return_NodeValues : bool, optional
        Return the node values for each node in the tetrahedral mesh. If 
        Type='surf', these will all be `threshold`, if Type='vol', these will be 
        `threshold` for the surface nodes and the original grid values for the 
        interior nodes
    return_ParentIds : bool, optional
        Return Ids that refer to the original tetrahedra that each of the new 
        elements is connected to, by default False.
    cleanup_tol : float, optional
        Tolerance value used to classify whether two nodes are sufficiently close 
        to be considered a single node (see :func:`mymesh.utils.DeleteDuplicateNodes`), 
        by default 1e-12.
    Returns
    -------
    NodeCoords : np.ndarray
        Node coordinates of the generated mesh
    NodeConn : list
        Node connetivity for the generated mesh
    NewValues : np.ndarray, optional
        Node values transfered to the generated mesh, returned if return_NodeValues=True 
    ParentIds : np.ndarray, optional
        Element ids that refer to the original tetrahedra that each of the new elements is connected to,
        returned if return_ParentIds=True


    Examples
    --------
    >>> # Create a uniform tetrahedral grid
    >>> Grid = primitives.Grid([0,2,0,2,0,2],0.1,ElemType='tet')
    >>> # Define an implicit function for two spheres and evaluate on the grid
    >>> func = lambda x,y,z : implicit.union(implicit.sphere(x,y,z,0.3,[0.7,0.7,0.7]),implicit.sphere(x,y,z,0.3,[1.2,1.2,1.2]))
    >>> NodeValues = func(Grid.NodeCoords[:,0],Grid.NodeCoords[:,1],Grid.NodeCoords[:,2])
    >>> # Perform marching tetrahedra
    >>> NodeCoords, NodeConn = contour.MarchingTetrahedra(Grid.NodeCoords,Grid.NodeConn,NodeValues, method='surface')


    """    
    TetNodeCoords = np.asarray(TetNodeCoords)
    TetNodeConn = np.asarray(TetNodeConn)
    NodeValues = np.asarray(NodeValues).flatten()
    assert len(NodeValues) == len(TetNodeCoords), 'Number of nodes must match number of node values.'

    MT_Lookup = np.array([
            [],                     # 0-0000
            [[3, 5, 4]],            # 1-0001
            [[4, 2, 1]],            # 2-0010
            [[1, 3, 5], [1, 5, 2]], # 3-0011
            [[0, 3, 1]],            # 4-0100
            [[1, 0, 5], [1, 5, 4]], # 5-0101
            [[0, 3, 4], [0, 4, 2]], # 6-0110
            [[0, 5, 2]],            # 7-0111
            [[0, 2, 5]],            # 8-1000
            [[0, 2, 4], [0, 4, 3]], # 9-1001
            [[0, 1, 4], [0, 4, 5]], # 10-1010
            [[0, 1, 3]],            # 11-1011
            [[1, 2, 5], [1, 5, 3]], # 12-1100
            [[1, 2, 4]],            # 13-1101
            [[3, 4, 5]],            # 14-1110
            []                      # 15-1111
        ],dtype=object)

    MTMixed_Lookup = np.array([
            [],             # 0-0000
            [[3, 5, 4]],    # 1-0001
            [[4, 2, 1]],    # 2-0010
            [[1, 3, 5, 2]], # 3-0011
            [[0, 3, 1]],    # 4-0100
            [[1, 0, 5, 4]], # 5-0101
            [[0, 3, 4, 2]], # 6-0110
            [[0, 5, 2]],    # 7-0111
            [[0, 2, 5]],    # 8-1000
            [[0, 2, 4, 3]], # 9-1001
            [[0, 1, 4, 5]], # 10-1010
            [[0, 1, 3]],    # 11-1011
            [[1, 2, 5, 3]], # 12-1100
            [[1, 2, 4]],    # 13-1101
            [[3, 4, 5]],    # 14-1110
            []              # 15-1111
        ],dtype=object)

    MTVMixed_Lookup = np.array([
            [],                     # 0-0000
            [[9, 3, 5, 4]],         # 1-0001
            [[8, 4, 2, 1]],         # 2-0010
            [[3, 5, 9, 1, 2, 8]],   # 3-0011
            [[7, 0, 3, 1]],         # 4-0100
            [[1, 0, 7, 4, 5, 9]],   # 5-0101
            [[0, 3, 7, 2, 4, 8]],   # 6-0110
            [[7, 9, 8, 0, 5, 2]],   # 7-0111
            [[6, 0, 2, 5]],         # 8-1000
            [[0, 2, 6, 3, 4, 9]],   # 9-1001
            [[1, 4, 8, 0, 5, 6]],   # 10-1010
            [[6, 8, 9, 0, 1, 3]],   # 11-1011
            [[3, 1, 7, 5, 2, 6]],   # 12-1100
            [[7, 6, 9, 1, 2, 4]],   # 13-1101
            [[7, 8, 6, 3, 4, 5]],   # 14-1110
            [[6, 7, 8, 9]]          # 15-1111
        ],dtype=object)
    
    if interpolation.lower() == 'midpoint' or interpolation.lower() == 'linear':
        edgeLookup = np.array([
            [0, 1],  # (0) Edge 0 - Between nodes 0 and 1
            [1, 2],  # (1) Edge 1
            [2, 0],  # (2) Edge 2
            [1, 3],  # (3) Edge 3
            [2, 3],  # (4) Edge 4
            [0, 3],  # (5) Edge 5
            [0, 0],  # (6) Node 0
            [1, 1],  # (7) Node 1
            [2, 2],  # (8) Node 2
            [3, 3],  # (9) Node 3
            ])
        PadEdgeLookup = np.vstack([edgeLookup, [-1,-1]])
        ninterppts = 2
    elif interpolation.lower() == 'quadratic':
        assert np.shape(TetNodeConn)[1] == 10, '10 Node tetrahedra are needed for quadratic interpolation.'
        edgeLookup = np.array([
            [0, 4, 1],  # (0) Edge 0 - Between nodes 0 and 1
            [1, 5, 2],  # (1) Edge 1
            [2, 6, 0],  # (2) Edge 2
            [1, 8, 3],  # (3) Edge 3
            [2, 9, 3],  # (4) Edge 4
            [0, 7, 3],  # (5) Edge 5
            [0, 0, 0],  # (6) Node 0
            [1, 1, 1],  # (7) Node 1
            [2, 2, 2],  # (8) Node 2
            [3, 3, 3],  # (9) Node 3
            ])
        PadEdgeLookup = np.vstack([edgeLookup, [-1,-1, -1]])
        ninterppts = 3
    else:
        raise ValueError(f'Interpolation must be one of "midpoint", "linear", or "quadratic", not "{interpolation:s}"')
    
    NodeValues = np.asarray(NodeValues) - threshold
    if flip:
        NodeValues = -1*NodeValues

    # Determine configuration of nodes
    TetVals = NodeValues[TetNodeConn[:,:4]]
    inside = TetVals <= 0
    if not np.any(inside) or (np.all(inside) and Type=='surf'):
        NodeCoords = np.empty((0,3))
        NodeConn = np.empty((0,4),dtype=int)
        NewValues = np.empty((0))
        ParentIds = np.empty((0), dtype=int)
        if return_NodeValues:
            if return_ParentIds:
                return NodeCoords, NodeConn, NewValues, ParentIds
            return NodeCoords, NodeConn, NewValues
        if return_ParentIds:
            return NodeCoords, NodeConn, ParentIds
        return NodeCoords, NodeConn

    ints = np.sum(inside[:,:4] * 2**np.arange(0,4)[::-1], axis=1)

    # Query lookup tables
    if Type.lower() == 'surf':
        if mixed_elements:
            element_lists = MTMixed_Lookup[ints]
        else:
            element_lists = MT_Lookup[ints]
    elif Type.lower() == 'vol':
        element_lists = MTVMixed_Lookup[ints]
    else:
        raise ValueError('Invalid Type, Type must be "surf" or "vol".')

    # Process lookup results
    tetnum, elem = zip(*[(i,e) for i,lst in enumerate(element_lists) for e in lst if len(lst) > 0])
    tetnum = np.array(tetnum)
    nelem = len(tetnum)

    relevant_tets = TetNodeConn[tetnum]
    
    PadElem = utils.PadRagged(elem)
    pad_relevant_tets = np.hstack([relevant_tets, -1*np.ones((len(elem),1))])
    
    lookup_indices = pad_relevant_tets[:, PadEdgeLookup]
    interpolation_pairs = (lookup_indices[np.arange(len(elem))[:, None], PadElem]).reshape((np.prod(PadElem.shape),ninterppts))
    interpolation_pairs = (interpolation_pairs[np.any(interpolation_pairs!=-1,axis=1)]).astype(int)
    
    uinterpolation_pairs,inv = np.unique(np.sort(interpolation_pairs,axis=1),axis=0,return_inverse=True)

    # Interpolation
    if interpolation.lower() == 'midpoint':
        NodeCoords = np.mean(TetNodeCoords[uinterpolation_pairs],axis=1)
        if return_NodeValues: 
            NewValues = np.zeros(len(NodeCoords))
            NewValues = np.mean(NodeValues[uinterpolation_pairs],axis=1)

    elif interpolation.lower() == 'linear':
        check = uinterpolation_pairs[:,0] != uinterpolation_pairs[:,1]
        coords1 = TetNodeCoords[uinterpolation_pairs[check,0]]
        coords2 = TetNodeCoords[uinterpolation_pairs[check,1]]
        vals1 = NodeValues[uinterpolation_pairs[check,0]][:,None]
        vals2 = NodeValues[uinterpolation_pairs[check,1]][:,None]
        NodeCoords = np.copy(TetNodeCoords[uinterpolation_pairs[:,0]])
        coefficient = (0 - vals1)/(vals2-vals1)
        position = coords1 + coefficient*(coords2 - coords1)
        position[coefficient.flatten() >= 1] = coords2[coefficient.flatten() >= 1]  # This is to prevent floating pt errors inverting elements
        NodeCoords[check] =  position

        if return_NodeValues: 
            NewValues = np.zeros(len(NodeCoords))
            NewValues[~check] = NodeValues[uinterpolation_pairs[~check,0]]
    
    elif interpolation.lower() == 'quadratic':
        
        check = uinterpolation_pairs[:,0] != uinterpolation_pairs[:,1]
        coords1 = TetNodeCoords[uinterpolation_pairs[check,0]]
        coords2 = TetNodeCoords[uinterpolation_pairs[check,1]]
        coords3 = TetNodeCoords[uinterpolation_pairs[check,2]]

        vals1 = NodeValues[uinterpolation_pairs[check,0]][:,None]
        vals2 = NodeValues[uinterpolation_pairs[check,1]][:,None]
        vals3 = NodeValues[uinterpolation_pairs[check,2]][:,None]


        NodeCoords = np.copy(TetNodeCoords[uinterpolation_pairs[:,0]])
        
        for i in [0,1,2]:
            # Loop over axes
            notconstant = ~np.isclose(coords1[:,i], coords3[:,i]) # identifies points where there is variation in position along this axis

            # Build coefficients for quadratic polynomials
            mat = np.empty((np.sum(notconstant), 3, 3))
            mat[:,0,0] = coords1[notconstant,i]**2
            mat[:,0,1] = coords1[notconstant,i]

            mat[:,1,0] = coords2[notconstant,i]**2
            mat[:,1,1] = coords2[notconstant,i]

            mat[:,2,0] = coords3[notconstant,i]**2
            mat[:,2,1] = coords3[notconstant,i]

            mat[:,:,2] = 1

            a,b,c = (np.linalg.solve(mat, np.column_stack([vals1[notconstant], vals2[notconstant], vals3[notconstant]])[...,None])[:,:,0]).T

            # Solve for roots
            root1 = (-b + np.sqrt(b**2 - 4*a*c))/(2*a)
            root2 = (-b - np.sqrt(b**2 - 4*a*c))/(2*a)

            position = np.copy(coords1[:,i])
            roots = np.copy(root1)
            # Select the appropriate root that falls between
            root1idx = (((root1 <= coords1[notconstant,i]) & (root1 >= coords3[notconstant,i])) | 
                ((root1 >= coords1[notconstant,i]) & (root1 <= coords3[notconstant,i])))
            root2idx = (((root2 <= coords1[notconstant,i]) & (root2 >= coords3[notconstant,i])) | 
                ((root2 >= coords1[notconstant,i]) & (root2 <= coords3[notconstant,i])))

            roots[root2idx] = root2[root2idx]

            # For instances when neither root is in between the bounds of edge being interpolated,
            # fall back on linear interpolation between two edge nodes with a sign change
            linear = ~(root1idx|root2idx)
            v1 = vals1[notconstant][linear][:,0]
            c1 = coords1[notconstant,i][linear]
            sgnchk1 = np.sign(v1) == np.sign(vals2[notconstant][linear])[:,0]
            v1[sgnchk1] = vals2[notconstant][linear][sgnchk1,0]
            c1[sgnchk1] = coords2[notconstant,i][linear][sgnchk1]
                                        
            v2 = vals3[notconstant][linear][:,0]
            c2 = coords3[notconstant,i][linear]
            sgnchk2 = np.sign(v1) == np.sign(v2)
            v2[sgnchk2] = vals2[notconstant][linear][sgnchk2,0]
            c2[sgnchk2] = coords2[notconstant,i][linear][sgnchk2]
            
            coefficient = (0 - v1)/(v2-v1)
            roots[linear] = c1 + coefficient*(c2 - c1)

            position[notconstant] = roots

            NodeCoords[check,i] = position

        if return_NodeValues: 
            NewValues = np.zeros(len(NodeCoords))
            NewValues[~check] = NodeValues[uinterpolation_pairs[~check,0]]

    else:
        raise ValueError('interpolation must be either "midpoint", "linear", or "quadratic"')

    # Format points into NodeCoords, NodeConn
    if Type.lower() == 'surf' and not mixed_elements:
        NodeConn = inv[np.reshape(np.arange(nelem*3), (nelem, 3))]
    else:
        lengths = [len(e) for e in elem]
        sums = np.append([0],np.cumsum(lengths))
        NodeConn = [[inv[n+sums[i]] for n in range(lengths[i])] for i in range(len(lengths))]

    if Type.lower() == 'vol' and not mixed_elements:
        NodeCoords, NodeConn, ids = converter.solid2tets(NodeCoords, NodeConn, return_ids=True)
        if return_ParentIds:
            ParentIds = np.zeros(len(NodeConn),dtype=int)
            for i,Id in enumerate(ids):
                ParentIds[Id] = tetnum[i]
    else:
        ParentIds = tetnum

    if cleanup: 
        NodeCoords,NodeConn,Idx = utils.DeleteDuplicateNodes(NodeCoords,NodeConn,return_idx=True, tol=cleanup_tol)
        if return_NodeValues:
            NewValues = NewValues[Idx]
        NodeCoords,NodeConn,EIdx = utils.CleanupDegenerateElements(NodeCoords,NodeConn,Type=Type, return_idx=True)
        if return_ParentIds:
            ParentIds = ParentIds[EIdx]

    if return_NodeValues:
        if flip:
            NewValues = -1*NewValues
        NewValues += threshold
        if return_ParentIds:
            return NodeCoords, NodeConn, NewValues, ParentIds
        return NodeCoords, NodeConn, NewValues
    if return_ParentIds:
        return NodeCoords, NodeConn, ParentIds
    return NodeCoords, NodeConn

def MarchingElements(NodeCoords, NodeConn, NodeValues, threshold=0, interpolation='linear', Type=None, mixed_elements=True, flip=False):
    """
    Generic wrapper class for :func:`MarchingSquares`, :func:'MarchingTriangles',
    :func:`MarchingCubes`, and :func:`MarchingTetrahedra`. Mixed element meshes
    will be split up and processed by the appropriate method. Only options that
    are available for all other methods are available, more advanced options
    (like higher order interpolation) must be accessed directly through the
    more specific functions. 

    Parameters
    ----------
    NodeCoords : array_like
        Node coordinates for the input mesh
    NodeConn : array_like
        Node connectivity for the input mesh. All elements must be tetrahedra.
    NodeValues : array_like
        Values at each node in the mesh that are used to extract the isosurface
    threshold : int, optional
        Isosurface level, by default 0
    interpolation : str, optional
        Interpolation to interpolate the position of the new nodes along the edges
        of the input mesh, by default 'linear'.

        'midpoint' - No interpolation is performed, new nodes are placed at the midpoint of edges

        'linear' - Linear interpolation is performed between adjacent nodes on the input mesh.

    Type : str, optional
        Determines whether to generate a surface mesh ('surf') or volume mesh ('vol'), 
        by default None. If None is provided, the output type will be the same as the
        input type
    mixed_elements : bool, optional
        If True, the generated mesh will have mixed element types 
        (triangles/quadrilaterals, tetrahedra/wedges), otherwise a single element 
        type (triangles, tetrahedra), by default False. 
    flip : bool, optional
        Flip the interior/exterior of the mesh, by default False.
        By default, values less than the threshold are assumed to be the "inside" 
        of the mesh. If the inside is denoted by values greater than the threshold,
        set flip=True.

    Returns
    -------
    NodeCoords : np.ndarray
        Node coordinates of the generated mesh
    NodeConn : list
        Node connetivity for the generated mesh


    """    

    InputType = utils.identify_type(NodeCoords, NodeConn)
    if Type is None:
        Type = InputType
    SplitConn = utils.SplitRaggedByLength(NodeConn)

    NewCoords = np.empty((0,3))
    NewConn = np.empty((0,3))

    for conn in SplitConn:
        if InputType == 'surf':
            if np.shape(conn)[1] == 3:
                # Marching Triangles
                ccoords, cconn = MarchingTriangles(NodeCoords, conn, NodeValues, threshold=threshold, interpolation=interpolation, Type=Type, mixed_elements=mixed_elements, flip=flip)
            elif np.shape(conn)[1] == 4:
                # Marching squares
                ccoords, cconn = MarchingSquares(NodeCoords, conn, NodeValues, threshold=threshold, interpolation=interpolation, Type=Type, mixed_elements=mixed_elements, flip=flip)
            else:
                raise ValueError('Unsupported element type for input surface mesh.')
            
        elif InputType == 'vol':
            if Type == 'vol':
                # Currently no volumentric option for marching cubes, must convert all to tetrahedra
                ccoords, cconn = MarchingTetrahedra(*converter.solid2tets(NodeCoords, conn), NodeValues, threshold=threshold, interpolation=interpolation, Type=Type, mixed_elements=mixed_elements, flip=flip)
            elif Type == 'surf':
                if np.shape(conn)[1] == 4:
                    ccoords, cconn = MarchingTetrahedra(NodeCoords, conn, NodeValues, threshold=threshold, interpolation=interpolation, Type=Type, mixed_elements=mixed_elements, flip=flip)
                elif np.shape(conn)[1] == 8:
                    ccoords, cconn = MarchingCubes(NodeCoords, conn, NodeValues, threshold=threshold, interpolation=interpolation, method='33', flip=flip)
                else:
                    ccoords, cconn = MarchingTetrahedra(*converter.solid2tets(NodeCoords, conn), NodeValues, threshold=threshold, interpolation=interpolation, Type=Type, mixed_elements=mixed_elements, flip=flip)

        NewCoords, NewConn = utils.MergeMesh(NewCoords, NewConn, ccoords, cconn)

    return NewCoords, NewConn

def Adaptive(func, bounds, threshold=0, method=None, grad=None, mindepth=2, maxdepth=5, octree_strategy='EDerror', octree_eps=0.1, dualgrid_method='centroid', Type='surf'):
    """
    Adaptively contour an implicit function. Uses an octree 
    (:func:`~mymesh.tree.Function2Octree`) to adaptively sample the function,
    then uses the chosen method to contour the dual grid 
    (:func:`~mymesh.tree.Octree2Dual`) of the octree.
    
    Based on, but not an exact implementation of, Dual Marching Cubes
    :cite:p:`Schaefer2005`.

    Parameters
    ----------
    func : function
        Implicit function that describes the geometry of the object to be meshed. 
        The function should be of the form v = f(x,y,z) and capable of handling
        vectorized inputs.
    bounds : array_like
        6 element array, list, or tuple with the minimum and maximum bounds in each direction that the function will be evaluated. This should be formatted as: [xmin, xmax, ymin, ymax, zmin, zmax]
    threshold : scalar
        Isovalue threshold to use for keeping/removing elements, by default 0.
    method : str, optional
        Contouring method to be used

            - "mc": Marching Cubes
            - "mc33": Marching Cubes 33
            - "mt": Marching Tetrahedra
        
        By default, "mc33" will be chosen if Type='surf' and "mt" if Type='vol'.

    grad : function, optional
        Gradient of the function. By default, the gradient will be obtained 
        through either symbolic differentiation or finite difference 
        approximation.
    mindepth : int, optional
        Minimum octree depth, by default 2
    maxdepth : int, optional
        Maximum octree depth, by default 5
    octree_strategy : str, optional
        Octree subdivision strategy (see :func:`~mymesh.tree.Function2Octree`), 
        by default 'EDerror'.
    octree_eps : float, optional
        Error tolerance for octree subdivision 
        (see :func:`~mymesh.tree.Function2Octree`), by default 0.1
    dualgrid_method : str, optional
        Method for placing dual grid vertices (see 
        :func:`~mymesh.tree.Octree2Dual`), by default 'centroid'.
    Type : str, optional
        Specifies whether to generate a surface mesh ("surf") or a volume
        mesh ("vol"), by default 'surf'

    Returns
    -------
    NodeCoords : np.ndarray
        Node Coordinates of the contour mesh.
    NodeConn : list
        Node connectivity list.

    Examples
    --------
    .. plot::

        func = implicit.unionf(implicit.box(-1.5,0,-1.5,0,-1.5,1.5),implicit.sphere([0,0,0], 1))
        bounds = [-2,2,-2,2,-2,2]

        S = mesh(*contour.Adaptive(func, bounds, mindepth=2, maxdepth=7))
        S.plot(bgcolor='w', show_edges=True)
    """    


    if Type == 'vol':
        if method is None:
            method = 'mt'
        elif method != 'mt':
            warnings.warn('Invalid method for Type=vol, switching to marching tetrahedra ("mt").')
    elif Type == 'surf' and method is None:
        method = 'mc33'

    root = tree.Function2Octree(func, bounds, threshold=threshold, grad=grad, mindepth=mindepth, maxdepth=maxdepth, strategy=octree_strategy, eps=octree_eps)
    DualCoords, DualConn = tree.Octree2Dual(root,method=dualgrid_method)
    NodeValues = func(DualCoords[:,0],DualCoords[:,1],DualCoords[:,2])
    
    if method == 'mc':
        NodeCoords, NodeConn = MarchingCubes(DualCoords, DualConn, NodeValues, threshold=threshold)
    elif method == 'mc33':
        NodeCoords, NodeConn = MarchingCubes(DualCoords, DualConn, NodeValues, threshold=threshold, method='33')
    elif method == 'mt':
        TetCoords, TetConn = converter.hex2tet(DualCoords, DualConn)
        NodeCoords, NodeConn = MarchingTetrahedra(TetCoords, TetConn, NodeValues, threshold=threshold, Type=Type)
    
            
    return NodeCoords, NodeConn

def SnapGrid2Surf(NodeCoords,NodeConn,NodeValues,threshold=0,snap=0.1, edges=None,FixedNodes=set()):
    """
    Deform a mesh before contouring to move near-surface nodes onto the surface. 
    Based on ideas from cite:t:`Schaefer2005`, cite:t:`Labelle2007`, cite:t:`Raman2008`.

    Parameters
    ----------
    NodeCoords : array_like
        Node coordinates
    NodeConn : list, array_like
        Node connectivity
    NodeValues : array_like
        Values at each node to be used for contouring
    threshold : int, optional
        Isosurface value that defines the surface, by default 0
    snap : float, optional
        Snapping parameter - relative distance along edge within which
        nodes will be snapped to the surface. Along an edge, the nearest
        of the two nodes will be snapped to the surface, so the snapping
        parameter ranges from [0, 0.5], by default 0.1. 
    edges : array_like, optional
        Node connectivity of the edges of the mesh (e.g. mesh.Edges). These
        should be the unique edges (not half-edges), by default None. If None 
        are provided, these will be calculated.
    FixedNodes : set, optional
        Set of nodes to be held fixed, by default set()

    Returns
    -------
    NodeCoords : np.ndarray
        Modified node coordinates
    NodeConn : list, array_like
        Node connectivity
    NodeValues : np.ndarray
        Modified node values where snapped values have been set to <threshold>.

    """    
    
    assert (snap <= 0.5) and (snap >= 0), 'The snap parameter must be in the range [0, 0.5].'
    NodeCoords = np.copy(NodeCoords)
    NodeValues = np.copy(NodeValues)
    if edges is None:
        e = converter.solid2edges(NodeCoords,NodeConn)
        edges = converter.edges2unique(e)
    if type(FixedNodes) is set:
        FixedNodes = list(FixedNodes)
    
    edgeshape = np.shape(edges)
    if len(edgeshape) == 1:
        raise ValueError('Input mesh appears to contain both linear and quadratic elements.')
    elif edgeshape[1] == 2:
        interpolation = 'linear'
    elif edgeshape[1] == 3:
        # interpolation = 'quadratic'
        interpolation = 'linear'
        edges = edges[:,[0,2]]
    else:
        raise Exception('Unexpected shape of edge list.')
        
    EdgeVals = NodeValues[edges]
    if interpolation == 'linear':
        SignChangeEdges = np.where(np.sign(EdgeVals[:,0]-threshold) != np.sign(EdgeVals[:,1]-threshold))[0]
        
        EdgeIds = SignChangeEdges
        closest = np.argmin(np.abs(EdgeVals-threshold),axis=1)[EdgeIds]
        furthest = ~(closest.astype(bool)).astype(int)
        
        coords1 = NodeCoords[edges[EdgeIds,closest]]
        coords2 = NodeCoords[edges[EdgeIds,furthest]]
        vals1 = NodeValues[edges[EdgeIds,closest]]
        vals2 = NodeValues[edges[EdgeIds,furthest]]
        
        coefficient = (threshold - vals1)/(vals2-vals1)
        
        position = coords1 + coefficient[:,None]*(coords2 - coords1)
        
        nodes = edges[EdgeIds,closest][coefficient <= snap]
        free = ~np.isin(nodes, FixedNodes)
        NodeCoords[nodes[free]] = position[coefficient <= snap][free]
        NodeValues[nodes[free]] = threshold
    elif interpolation == 'quadratic':
        raise Exception('Quadratic elements not supported yet')
        SignChangeEdges = np.where((np.sign(EdgeVals[:,0]-threshold) != np.sign(EdgeVals[:,1]-threshold))|(np.sign(EdgeVals[:,0]-threshold) != np.sign(EdgeVals[:,2]-threshold)))[0]
        
        coords1 = NodeCoords[edges[EdgeIds,closest]]
        coords2 = NodeCoords[edges[EdgeIds,furthest]]
        coords3 = NodeCoords[edges[EdgeIds,furthest]]
        
        vals1 = NodeValues[edges[EdgeIds,closest]]
        vals2 = NodeValues[edges[EdgeIds,furthest]]
        vals2 = NodeValues[edges[EdgeIds,furthest]]
        
        for i in [0,1,2]:
            # Loop over axes
            
            # Build coefficients for quadratic polynomials
            mat = np.empty((np.sum(SignChangeEdges), 3, 3))
            mat[:,0,0] = coords1[SignChangeEdges,i]**2
            mat[:,0,1] = coords1[SignChangeEdges,i]

            mat[:,1,0] = coords2[SignChangeEdges,i]**2
            mat[:,1,1] = coords2[SignChangeEdges,i]

            mat[:,2,0] = coords3[SignChangeEdges,i]**2
            mat[:,2,1] = coords3[SignChangeEdges,i]

            mat[:,:,2] = 1

            a, b, c = np.linalg.solve(mat, np.hstack([vals1[notconstant], vals2[notconstant], vals3[notconstant]])).T

            # Solve for roots
            root1 = (-b + np.sqrt(b**2 - 4*a*c))/(2*a)
            root2 = (-b - np.sqrt(b**2 - 4*a*c))/(2*a)

            position = np.copy(coords1[:,i])
            roots = np.copy(root1)
            # Select the appropriate root that falls between
            root1idx = (((root1 <= coords1[notconstant,i]) & (root1 >= coords3[notconstant,i])) | 
                ((root1 >= coords1[notconstant,i]) & (root1 <= coords3[notconstant,i])))
            root2idx = (((root2 <= coords1[notconstant,i]) & (root2 >= coords3[notconstant,i])) | 
                ((root2 >= coords1[notconstant,i]) & (root2 <= coords3[notconstant,i])))

            roots[root2idx] = root2[root2idx]

            # For instances when neither root is in between the bounds of edge being interpolated,
            # fall back on linear interpolation between two edge nodes with a sign change
            linear = ~(root1idx|root2idx)
            v1 = vals1[notconstant][linear][:,0]
            c1 = coords1[notconstant,i][linear]
            sgnchk1 = np.sign(v1) == np.sign(vals2[notconstant][linear])[:,0]
            v1[sgnchk1] = vals2[notconstant][linear][sgnchk1,0]
            c1[sgnchk1] = coords2[notconstant,i][linear][sgnchk1]
                                        
            v2 = vals3[notconstant][linear][:,0]
            c2 = coords3[notconstant,i][linear]
            sgnchk2 = np.sign(v1) == np.sign(v2)
            v2[sgnchk2] = vals2[notconstant][linear][sgnchk2,0]
            c2[sgnchk2] = coords2[notconstant,i][linear][sgnchk2]
            
            coefficient = (0 - v1)/(v2-v1)
            roots[linear] = c1 + coefficient*(c2 - c1)

            position[notconstant] = roots
    else:
        raise ValueError('Currently only interpolation="linear" is implemented.')
    
    return NodeCoords, NodeConn, NodeValues

def _MarchingCubesLookup(i):
    assert i < 256, 'There are only 256 possible states of the voxel, i must be less than 256'
    if i > 127:
        i = 128-(i-127)
        flip = True
    else:
        flip = False
    _MarchingCubesLookup.LookupTable[i]
    # Flip
    if flip:
        TriElems = [[n for n in reversed(tri)] for tri in _MarchingCubesLookup.LookupTable[i]]
    else:
        TriElems = _MarchingCubesLookup.LookupTable[i]
    return TriElems

def _MarchingCubes33Lookup(i,vals):
    
    assert i < 256, 'There are only 256 possible states of the voxel, i must be less than 256'
    
    # nodes :  [0 1 2 3 4 5 6 7]
    
    configs = _MarchingCubes33Lookup.LookupTable[i]
    case = _MarchingCubes33Lookup.Cases[i]
    FaceTest = _MarchingCubes33Lookup.FaceTests[i]
    sign = _MarchingCubes33Lookup.Signs[i]
    # Resolve Ambiguities
    vs = vals-1e-16
    if case == 0:
        config = configs[0]
        
    elif case == 1:
        config = configs[0]
        
    elif case == 2:
        config = configs[0]
    
    elif case == 3:
        # Case 3 Ambiguity
        if _isConnected(FaceTest[0], sign*vs):
            # Nodes are not separated - Case 3.2
            config = configs[1]
        else:
            # Nodes are separated - Case 3.1
            config = configs[0]
            
    elif case == 4:
        # Case 4 Ambiguity
        if _InternalConnection(sign*vs):
            # Nodes are not separated - Case 4.2
            config = configs[1]
        else: 
            # Nodes are separated - Case 4.1
            config = configs[0]
                    
    elif case == 5:
        config = configs[0]
        
    elif case == 6:
        # Case 6 Ambiguity
        if _isConnected(FaceTest[0], sign*vs):
            # Nodes are connected - Case 6.2
            config = configs[2]  
        else:
            if _InternalConnection(sign*vs):
                # Nodes of ambiguous face are joined inside the cube - Case 6.1.2
                config = configs[1]
            else:
                # Nodes of ambiguous face are completely separated - Case 6.1.1
                config = configs[0]
    
    elif case == 7:
        results = [_isConnected(face,sign*vs) for face in FaceTest]
        passes = np.where(results)[0]
        if len(passes) == 0:
            # Case 7.1
            config = configs[0]
        elif len(passes) == 1:
            # Case 7.2
            if 0 in passes:
                # Case 7.2.1
                config = configs[1]
            elif 1 in passes:
                # Case 7.2.2
                config = configs[2]
            elif 2 in passes:
                # Case 7.2.3
                config = configs[3]
            else:
                print('problem 7.2')
        elif len(passes) == 2:
            # Case 7.3
            if 1 in passes and 2 in passes:
                # Case 7.3.1
                config = configs[4]
            elif 0 in passes and 2 in passes:
                # Case 7.3.2
                config = configs[5]
            elif 0 in passes and 1 in passes:
                # Case 7.3.3
                config = configs[6]
            else:
                print('problem 7.3')
        elif len(passes == 3):
            # Case 7.4
            if _InternalConnection(-sign*vs):
                # Case 7.4.2
                config = configs[8]
            else:
                # Case 7.4.1
                config = configs[7]
        else:
            print('problem 7')
    
    elif case == 8:
        config = configs[0]
        
    elif case == 9:
        config = configs[0]
        
    elif case == 10:
        results = [_isConnected(face,sign*vs) for face in FaceTest]
        passes = np.where(results)[0]
        
        if len(passes) == 0:
            
            if _InternalConnection(-sign*vs):
                # Case 10.1.2
                config = configs[2]
            else:
                # Case 10.1.1 (No Internal Connection)
                config = configs[0]
        elif len(passes) == 1:
            # Case 10.2 
            # config = configs[3]
            if 1 in passes:
                # Case 10.2.1 Top Face Connected
                config = configs[3]
            else:
                # Case 10.2.2 Bottom Face Connected
                config = configs[4]
        elif len(passes) == 2:
            # Case 10.1.1 (Internal Connection)
            config = configs[1]
        else:
            print('problem 10')
    
    elif case == 11:
        config = configs[0]
    
    elif case == 12:
        results = [_isConnected(face,sign*vs) for face in FaceTest]
        passes = np.where(results)[0]
        
        if len(passes) == 0:
            if _InternalConnection(-sign*vs):
                # Case 12.1.2
                config = configs[2]
            else:
                # Case 12.1.1
                config = configs[0]
        elif len(passes) == 1:
            # Case 12.2 (Top Face Connected)
            if 0 in passes:
                # Case 12.2
                config = configs[3]
            elif 1 in passes:
                # Case 12.3
                config = configs[4]
            else:
                # Case 12.3
                warnings.warn('MC error - Case 12.3')
        elif len(passes) == 2:
            # Case 12.1.1
            config = configs[1]
        else:
            warnings.warn('MC error - Case 12')
 
    elif case == 13:
        results = [_isConnected(face,sign*vs) for face in FaceTest]
        passes = np.where(results)[0]
        if len(passes) == 0:
            # Case 13.1
            config = configs[0]
        elif len(passes) == 1:
            if 5 in passes:
                # Case 13.2.1
                config = configs[1]
            elif 2 in passes:
                # Case 13.2.2
                config = configs[2]
            elif 0 in passes:
                # Case 13.2.3
                config = configs[3]
            elif 1 in passes:
                # Case 13.2.4
                config = configs[4]
            elif 4 in passes:
                # Case 13.2.5
                config = configs[5]
            elif 3 in passes:
                # Case 13.2.6
                config = configs[6]
            else:
                warnings.warn('MC error - Case 13.2')
        elif len(passes) == 2:
            if all([1,2] == passes):
                # Case 13.3.1
                config = configs[7]
            elif all([1,4] == passes):
                # Case 13.3.2
                config = configs[8]
            elif all([0,1] == passes):
                # Case 13.3.3
                config = configs[9]
            elif all([1,5] == passes):
                # Case 13.3.4
                config = configs[10]
            elif all([2,3] == passes):
                # Case 13.3.5
                config = configs[11]
            elif all([0,2] == passes):
                # Case 13.3.6
                config = configs[12]
            elif all([2,5] == passes):
                # Case 13.3.7
                config = configs[13]
            elif all([3,4] == passes):
                # Case 13.3.8
                config = configs[14]
            elif all([0,3] == passes):
                # Case 13.3.9
                config = configs[15]
            elif all([3,5] == passes):
                # Case 13.3.10
                config = configs[16]
            elif all([0,4] == passes):
                # Case 13.3.11
                config = configs[17]
            elif all([4,5] == passes):
                # Case 13.3.12
                config = configs[18]
            else:
                warnings.warn('MC error - Case 13.3')
        elif len(passes) == 3:
            if all([1,2,5] == passes):
                # Case 13.4.1
                config = configs[19]
            elif all([0,1,4] == passes):
                # Case 13.4.2
                config = configs[20]
            elif all([0,2,3] == passes):
                # Case 13.4.3
                config = configs[21]
            elif all([3,4,5] == passes):
                # Case 13.4.4
                config = configs[22]
            elif not _InternalConnection(-vs,case13=True):
                if all([0,1,2] == passes):
                    # Case 13.5.1.1
                    config = configs[23]
                elif all([1,4,5] == passes):
                    # Case 13.5.1.2
                    config = configs[24]
                elif all([2,3,5] == passes):
                    # Case 13.5.1.3
                    config = configs[25]
                elif all([0,3,4] == passes):
                    # Case 13.5.1.4
                    config = configs[26]
                else:
                    warnings.warn('MC error - Case 13.5.1')
            else:
                if all([0,1,2] == passes):
                    # Case 13.5.2.1
                    config = configs[27]
                elif all([1,4,5] == passes):
                    # Case 13.5.2.2
                    config = configs[28]
                elif all([2,3,5] == passes):
                    # Case 13.5.2.3
                    config = configs[29]
                elif all([0,3,4] == passes):
                    # Case 13.5.2.4
                    config = configs[30]
                else:
                    warnings.warn('MC error - Case 13.5.2')
        else:
            warnings.warn('MC error - Case 13')
            config = configs[0]
        
    elif case == 14:
        config = configs[0]
    else:
        print('problem 15')
        config = configs[0]
                
    TriElems = config
            
    return TriElems

def _isConnected(face, vals):
    facevals = vals[face].tolist()
    while facevals[0] < max(facevals):
        # Cycle the list until it starts with a positive (inside) value
        facevals += [facevals.pop(0)]
    [A,B,C,D] = facevals
    if A*C <= B*D:
        # Nodes are Separated
        return False
    else:
        return True

def _InternalConnection(vals,case13=False):
    faces = [[0,1,2,3],[0,1,5,4],[1,2,6,5],[2,3,7,6],[3,0,4,7],[4,5,6,7]]
    face0vals = vals[faces[0]]    # Bottom Face
    face1vals = vals[faces[5]]    # Top Face      
    
    face0vals = vals[faces[0]].tolist()
    face1vals = vals[faces[5]].tolist()
    while face0vals[0] < 0:
        # Cycle the list until it starts with a positive (inside) value
        face0vals += [face0vals.pop(0)]
        face1vals += [face1vals.pop(0)]
    
    [A0, B0, C0, D0] = face0vals
    [A1, B1, C1, D1] = face1vals
    a = (A1 - A0)*(C1 - C0) - (B1 - B0)*(D1 - D0)
    if a >= 0:
        # Nodes are separated 
        connected = False
    else:
        b = C0*(A1 - A0) + A0*(C1 - C0) - D0*(B1 - B0) - B0*(D1 - D0)
        tmax = -b/(2*a)
        if (tmax <= 0) or (1 <= tmax):
            # Nodes are separated 
            connected = False
        else:            
            if case13:
                # At = A0 + (A1-A0)*tmax
                Bt = B0 + (B1-B0)*tmax
                Ct = C0 + (C1-C0)*tmax
                Dt = D0 + (D1-D0)*tmax
                c = A0*Ct - Bt*Dt
                if a*tmax**2 + b*tmax + c <= 0:
                    # Nodes are separated
                    connected = False
                else:
                    # Nodes are not separated
                    connected = True

            else:
                c = A0*C0 - B0*D0
                if a*tmax**2 + b*tmax + c <= 0:
                    # Nodes are separated
                    connected = False
                else:
                    # Nodes are not separated
                    connected = True
                
    return connected

def _generateLookup33():
    # LookupTable, Cases, FaceTests, Signs = _generateLookup33()
    import random
    def R1x(bits, k, sign):
        # 90 deg x-axis rotation (ccw)
        R1xnodes = np.array([4, 5, 1, 0, 7, 6, 2, 3])
        R1xedges = np.array([8, 5, 0, 4, 11, 9, 1, 3, 10, 6, 2, 7, 12])
        R1xbits = bits[R1xnodes]
        TriElems, Case, FaceTest, sign = lookup(R1xbits, sign, k=k, primary=False)
        R1xtris = [[[R1xedges[n] for n in tri] for tri in config] for config in TriElems]
        R1xface = [[R1xnodes[n] for n in face] for face in FaceTest]
        return R1xtris, Case, R1xface, sign

    def R1z(bits, k, sign):
        # 90 deg z-axis rotation (ccw)
        R1znodes = np.array([3, 0, 1, 2, 7, 4, 5, 6])
        R1zedges = np.array([3, 0, 1, 2, 7, 4, 5, 6, 11, 8, 9, 10, 12])
        R1zbits = bits[R1znodes]
        TriElems, Case, FaceTest, sign = lookup(R1zbits, sign, k=k, primary=False)
        R1ztris = [[[R1zedges[n] for n in tri] for tri in config] for config in TriElems]
        R1zface = [[R1znodes[n] for n in face] for face in FaceTest]
        return R1ztris, Case, R1zface, sign

    def Mxy(bits, k, sign):
        # Mirror across XY plane
        Mxynodes = np.array([3,2,1,0,7,6,5,4])
        Mxyedges = np.array([2, 1, 0, 3, 7, 6, 5, 4, 10, 9, 8, 11, 12])
        Mxybits = bits[Mxynodes]
        TriElems, Case, FaceTest,sign = lookup(Mxybits, sign, k=k, primary=False)
        Mxytris = [[list(reversed([Mxyedges[n] for n in tri])) for tri in config] for config in TriElems]
        Mxyface = [list(reversed([Mxynodes[n] for n in face])) for face in FaceTest]
        return Mxytris, Case, Mxyface, sign
    
    def Re(bits, k, sign):
        # Bit reflection (switching 0s and 1s)
        Rebits = np.array([1 if bit == 0 else 0 for bit in bits])
        TriElems, Case, FaceTest, sign, = lookup(Rebits, sign, k=k, primary=False)
        Retris = [[list(reversed(tri)) for tri in config] for config in TriElems]
        Reface = [list(reversed(face)) for face in FaceTest]
        return Retris, Case, Reface, -sign
        
        
    def lookup(bits, sign, k=0, primary=True):
        i = int(''.join([str(bit) for bit in bits]),2)
        if i == 0 or i == 255:
            # Case 0
            Case = 0
            TriElems  = [[[]]]
            FaceTest = [[]]
        elif i == 1:
            # Case 1
            Case = 1
            TriElems = [
                [[7, 10, 11]]
                ]
            FaceTest = [[]]
        elif i == 3:
            # Case 2
            Case = 2
            TriElems = [
                [[6, 9, 7], [7, 9, 11]]
                ]
            FaceTest = [[]]
        elif i == 5:
            # Case 3 - 5 : 0 0 0 0 0 1 0 1
            Case = 3
            TriElems = [
                [[5, 8, 9], [7, 10, 11]],                       # Case 3.1
                [[5, 7, 9], [5, 8, 7], [8, 11, 7], [7, 10 ,9]]  # Case 3.2
                ]        
            FaceTest = [
                [4, 5, 6, 7]
                ]
        elif i == 20:
            # Case 4 - 20 : 0 0 0 1 0 1 0 0
            Case = 4
            TriElems = [
                [[2, 7, 3], [5, 8, 9]],
                [[7, 8, 9], [8, 7, 3], [3, 5, 8], [5, 3, 2], [2, 9, 5], [9, 7, 2]]
                ]
            FaceTest = [[]]
        elif i == 7:
            # Case 5 - 7 : 0 0 0 0 0 1 1 1
            Case = 5
            TriElems = [
                [[5, 7, 6], [5, 8, 11], [5, 11, 7]]
                ]
            FaceTest = [[]]
        elif i == 22:
            # Case 6 - 22 : 0 0 0 1 0 1 1 0
            Case = 6
            TriElems = [
                [[2, 7, 3], [5, 8, 6], [6, 8, 10]],                                                 # Case 6.1.1
                [[6, 2, 7], [6, 7, 10], [10, 7, 8], [8, 7, 3], [8, 3, 5], [2, 5, 3], [2, 6, 5]],    # Case 6.1.2
                [[3, 5, 8], [3, 8, 7], [7, 8, 10], [2, 5, 3], [5, 2, 6]]                            # Case 6.2   
                ]
            FaceTest = [
                [2, 3, 7, 6]
                ]
        elif i == 74:
            # Case 7 - 74 : 0 1 0 0 1 0 1 0
            Case = 7
            TriElems = [
                [[0, 5, 1], [6, 9, 10], [4, 11, 8]],                            # Case 7.1   (No Nodes Connected)
                [[1, 0, 4], [1, 4, 11], [1, 11, 8], [1, 8, 5], [6, 9, 10]],     # Case 7.2.1 (Nodes 1 and 4 Connected)
                [[0, 6, 1], [0, 10, 6], [0, 9, 10], [0, 5, 9], [4, 11, 8]],     # Case 7.2.2 (Nodes 1 and 6 Connected)
                [[4, 11, 10], [4, 10, 6], [4, 6, 8], [6, 9, 8], [0, 5, 1]],      # Case 7.2.3 (Nodes 4 and 6 Connected)
                [[4, 12, 8], [8, 12, 9], [9, 12, 5], [5, 12, 0], [0, 12, 1], [1, 12, 6], [6, 12, 10], [10, 12, 11], [11, 12, 4]],    # Case 7.3.1 (Nodes 1, 6 and 6, 4 Connected)
                [[8, 12, 9], [9, 12, 6], [6, 12, 10], [10, 12, 11], [11, 12, 4], [4, 12, 0], [0, 12, 1], [1, 12, 5], [5, 12, 8]],    # Case 7.3.2 (Nodes 1, 4 and 4, 6 Connected)
                [[10, 12, 9], [9, 12, 5], [5, 12, 8], [8, 12, 11], [11, 12, 4], [4, 12, 0], [0, 12, 1], [1, 12, 6], [6, 12, 10]],       # Case 7.3.3 (Nodes 4, 1 and 1, 6 Connected)
                [[5, 9, 8], [0, 4, 11], [0, 11, 6], [0, 6, 1], [6, 11, 10]], # Case 7.4.1
                [[8, 10, 9], [8, 11, 10], [4, 11, 8], [4, 8, 5], [4, 5, 0], [0, 5, 1], [5, 6, 1], [5, 9, 6], [9, 10, 6]]    # Case 7.4.2
                
                ]
            FaceTest = [
                [0, 1, 5, 4],
                [1, 2, 6, 5],
                [4, 5, 6, 7]
                ]
        elif i == 15:
            # Case 8 - 15 : 0 0 0 0 1 1 1 1
            Case = 8
            TriElems = [
                [[4, 6, 5], [4, 7, 6]]
                ]
            FaceTest = [[]]
        elif i == 114:
            # Case 9 - 114 : 0 1 1 1 0 0 1 0
            Case = 9
            TriElems = [
                [[0, 5, 3], [3, 5, 7], [5, 9, 7], [7, 9, 10]]
                ]
            FaceTest = [[]]
        elif i == 85:
            # Case 10 - 85 : 0 1 0 1 0 1 0 1
            Case = 10
            TriElems = [
                [[0, 9, 1], [0, 8, 9], [2, 10, 11], [2, 11, 3]],    # Case 10.1.1 (No internal connection)
                [[0, 11, 3], [0, 8, 11], [1, 10, 9], [1, 2, 10]], # Case 10.1.1 (Internal connection)
                [[0, 8, 3], [3, 8, 11], [0, 3, 1], [1, 3, 2], [1, 2, 9], [2, 10, 9], [8, 9, 11], [9, 10, 11]],         # Case 10.1.2
                [[11, 12, 8], [8, 12, 0], [0, 12, 1], [1, 12, 9], [9, 12, 10], [10, 12, 2], [2, 12, 3], [3, 12, 11]],  # Case 10.2.1 (Top Face Connected)
                [[9, 12, 8], [1, 12, 9], [2, 12, 1], [10, 12, 2], [11, 12, 10], [3, 12, 11], [0, 12, 3], [8, 12, 0]]   # Case 10.2.2 (Bottom Face Connected)
                ]
            FaceTest = [
                [0, 1, 2, 3],
                [4, 5, 6, 7]
                ]
        elif i == 116:
            # Case 11
            Case = 11
            TriElems = [
                [[8, 9, 6],[3, 8, 6],[3, 6, 7],[0, 8, 3]]
                ]
            FaceTest = [[]]
        elif i == 120:
            # Case 12 - 120 : 0 1 1 1 1 0 0 0
            Case = 12
            TriElems = [
                # [[0, 5, 3], [3, 5, 7], [5, 6, 7], [4, 11, 8]],  # Case 12.1.1
                # [[6, 11, 8], [5, 8, 6], [0, 8, 5], [0, 4, 8], [0, 3, 4], [3, 11, 4], [3, 7, 11], [6, 11, 7]],       # Case 12.1.2
                # [[0, 4, 12], [4, 11, 8], [8, 12, 11], [5, 12, 8], [5, 6, 12], [6, 7, 12], [3, 12, 7], [0, 12, 3]],  # Case 12.2
                # [[0, 12, 3], [3, 12, 4], [4, 12, 8], [8, 12, 11], [11, 12, 7], [7, 12, 6], [6, 12, 5], [5, 12, 0]]  # Case 12.3
                [[4, 11, 8], [3, 0, 7], [5, 7, 0], [7, 5, 6]],                                                      # Case 12.1.1
                [[4, 3, 0], [11, 8, 7], [5, 7, 8], [7, 5, 6]],                                                      # Case 12.1.1r (reversed)
                # [[4, 5, 8], [5, 4, 0], [3, 0, 4], [11, 3, 4], [3, 11, 10], [9, 10, 11], [8, 9, 11], [9, 8, 5]],     # Case 12.1.2
                [[4, 0, 8], [5, 8, 0], [8, 5, 6], [8, 6, 11], [7, 11, 6], [11, 7, 3], [11, 3, 4], [0, 4, 3]],       # Case 12.1.2 
                [[12, 0, 4], [12, 4, 11], [12, 11, 8], [12, 8, 5], [12, 5, 6], [12, 6, 7], [12, 7, 3], [12, 3, 0]], # Case 12.2 
                [[12, 5, 6], [12, 6, 7], [12, 7, 11], [12, 11, 8], [12, 8, 4], [12, 4, 3], [12, 3, 0], [12, 0, 5]]  # Case 12.3       
                ]
            FaceTest = [
                [0, 1, 5, 4],
                [3, 0, 4, 7]
                ]
        elif i == 90:
            # Case 13 - 90 : 0 1 0 1 1 0 1 0
            Case = 13
            TriElems = [                
                [[0, 5, 1],[9, 10, 6],[4, 11, 8],[2, 7, 3]],            # Case 13.1                
                [[0, 5, 1], [2, 7, 3], [4, 11, 6], [6, 11, 10], [6, 8, 4], [6, 9, 8]],  # Case 13.2.1 Face 5 (Nodes 4 and 6 Connected)
                [[4, 11, 8], [2, 7, 3], [0, 5, 9], [0, 9, 10], [0, 6, 1], [0, 10, 6]],  # Case 13.2.2 Face 2 (Nodes 1 and 6 Connected)
                [[4, 11, 8], [9, 10, 6], [3, 5, 7], [0, 5, 3], [1, 2, 5], [2, 7, 5]],   # Case 13.2.3 Face 0 (Nodes 1 and 3 Connected)
                [[2, 7, 3], [9, 10, 6], [5, 11, 8], [1, 11, 5], [0, 4, 11], [0, 11, 1]],# Case 13.2.4 Face 1 (Nodes 1 and 4 Connected)
                [[6, 9, 10], [0, 5, 1], [2, 11, 8], [2, 7, 11], [2, 8, 4], [2, 4, 3]],  # Case 13.2.5 Face 4 (Nodes 3 and 4 Connected)
                [[0, 5, 1], [4, 11, 8], [9, 7, 3], [9, 10, 7], [9, 3, 2], [9, 2, 6]],   # Case 13.2.6 Face 3 (Nodes 3 and 6 Connected)
                [[11, 8, 4], [7, 3, 12], [10, 7, 12], [9, 10, 12], [5, 9, 12], [0, 5, 12], [1, 0, 12], [6, 1, 12], [2, 6, 12], [3, 2, 12]], # Case 13.3.1 Faces 1, 2 
                [[7, 3, 2], [12, 10, 6], [12, 6, 1], [12, 1, 0], [12, 0, 4], [12, 4, 11], [12, 11, 8], [12, 8, 5], [12, 5, 9], [12, 9, 10]], # Case 13.3.2 Faces 1, 4 
                [[11, 8, 4], [9, 10, 12], [5, 9, 12], [0, 5, 12], [3, 0, 12], [7, 3, 12], [2, 7, 12], [1, 2, 12], [6, 1, 12], [10, 6, 12]], # Case 13.3.3 Faces 1, 0 
                [[7, 3, 2], [1, 0, 12], [6, 1, 12], [10, 6, 12], [11, 10, 12], [4, 11, 12], [8, 4, 12], [9, 8, 12], [5, 9, 12], [0, 5, 12]], # Case 13.3.4 Faces 1, 5
                [[5, 1, 0], [12, 8, 4], [12, 4, 3], [12, 3, 2], [12, 2, 6], [12, 6, 9], [12, 9, 10], [12, 10, 7], [12, 7, 11], [12, 11, 8]], # Case 13.3.5 Faces 2, 3
                [[11, 8, 4], [12, 9, 10], [12, 10, 7], [12, 7, 3], [12, 3, 0], [12, 0, 5], [12, 5, 1], [12, 1, 2], [12, 2, 6], [12, 6, 9]], # Case 13.3.6 Faces 2, 0
                [[1, 0, 5], [12, 3, 2], [12, 2, 6], [12, 6, 9], [12, 9, 8], [12, 8, 4], [12, 4, 11], [12, 11, 10], [12, 10, 7], [12, 7, 3]], # Case 13.3.7 Faces 2, 5
                [[6, 9, 10], [5, 1, 12], [8, 5, 12], [11, 8, 12], [7, 11, 12], [2, 7, 12], [3, 2, 12], [4, 3, 12], [0, 4, 12], [1, 0, 12]], # Case 13.3.8 Faces 3, 4
                [[6, 9, 10], [11, 8, 12], [7, 11, 12], [2, 7, 12], [1, 2, 12], [5, 1, 12], [0, 5, 12], [3, 0, 12], [4, 3, 12], [8, 4, 12]], # Case 13.3.9 Faces 3, 0
                [[1, 0, 5], [3, 2, 12], [4, 3, 12], [8, 4, 12], [9, 8, 12], [6, 9, 12], [10, 6, 12], [11, 10, 12], [7, 11, 12], [2, 7, 12]], # Case 13.3.10 Faces 3, 5
                [[6, 9, 10], [12, 11, 8], [12, 8, 5], [12, 5, 1], [12, 1, 2], [12, 2, 7], [12, 7, 3], [12, 3, 0], [12, 0, 4], [12, 4, 11]], # Case 13.3.11 Faces 4, 0
                [[7, 3, 2], [12, 1, 0], [12, 0, 4], [12, 4, 11], [12, 11, 10], [12, 10, 6], [12, 6, 9], [12, 9, 8], [12, 8, 5], [12, 5, 1]], # Case 13.3.12 Faces 4, 5
                [[7, 3, 12], [10, 7, 12], [11, 10, 12], [4, 11, 12], [8, 4, 12], [9, 8, 12], [5, 9, 12], [0, 5, 12], [1, 0, 12], [6, 1, 12], [2, 6, 12], [3, 2, 12]], # Case 13.4.1 Faces 1, 2, 5 
                [[12, 0, 4], [12, 4, 11], [12, 11, 8], [12, 8, 5], [12, 5, 9], [12, 9, 10], [12, 10, 6], [12, 6, 1], [12, 1, 2], [12, 2, 7], [12, 7, 3], [12, 3, 0]], # Case 13.4.2 Faces 1, 4, 0
                [[12, 2, 6], [12, 6, 9], [12, 9, 10], [12, 10, 7], [12, 7, 11], [12, 11, 8], [12, 8, 4], [12, 4, 3], [12, 3, 0], [12, 0, 5], [12, 5, 1], [12, 1, 2]], # Case 13.4.3 Faces 2, 3, 0
                [[5, 1, 12], [8, 5, 12], [9, 8, 12], [6, 9, 12], [10, 6, 12], [11, 10, 12], [7, 11, 12], [2, 7, 12], [3, 2, 12], [4, 3, 12], [0, 4, 12], [1, 0, 12]], # Case 13.4.4 Faces 3, 4, 5
                [[4, 11, 8], [6, 1, 2], [3, 0, 7], [10, 7, 0], [5, 10, 0], [10, 5, 9]], # Case 13.5.1.1 Faces 1, 2, 0
                [[7, 3, 2], [5, 9, 8], [4, 1, 0], [1, 4, 11], [1, 11, 6], [10, 6, 11]], # Case 13.5.1.2 Faces 1, 4, 5
                [[5, 1, 0], [7, 11, 10], [6, 3, 2], [3, 6, 9], [3, 9, 4], [8, 4, 9]], # Case 13.5.1.3 Faces 2, 3, 5
                [[6, 9, 10], [4, 3, 0], [1, 2, 5], [8, 5, 2], [7, 8, 2], [8, 7, 11]], # Case 13.5.1.4 Faces 3, 4, 0
                [[6, 1, 2], [5, 9, 8], [0, 5, 8], [4, 0, 8], [0, 4, 3], [7, 3, 4], [11, 7, 4], [7, 11, 10], [8, 10, 11], [10, 8, 9]], # Case 13.5.2.1 Faces 1, 2, 0
                [[5, 9, 8], [3, 0, 4], [3, 4, 11], [3, 11, 7], [10, 7, 11], [7, 10, 6], [7, 6, 2], [1, 2, 6], [2, 1, 3], [0, 3, 1]], # Case 13.5.2.2 Faces 1, 4, 5
                [[7, 11, 10], [1, 2, 6], [1, 6, 9], [1, 9, 5], [8, 5, 9], [5, 8, 4], [5, 4, 0], [3, 0, 4], [0, 3, 1], [2, 1, 3]], # Case 13.5.2.3 Faces 2, 3, 5
                [[4, 3, 0], [7, 11, 10], [2, 7, 10], [6, 2, 10], [2, 6, 1], [5, 1, 6], [9, 5, 6], [8, 5, 9], [10, 8, 9], [8, 10, 11]], # Case 13.5.2.4 Faces 3, 4, 0
                ]
            FaceTest = [
                [0, 1, 2, 3],
                [0, 1, 5, 4],
                [1, 2, 6, 5],
                [2, 3, 7, 6],
                [3, 0, 4, 7],
                [4, 5, 6, 7]
                ]
        elif i == 113: 
            # Case 14 - 113 : 0 1 1 1 0 0 0 1
            Case = 14
            TriElems = [
                [[6,10,11],[0,6,11],[0,11,3],[0,5,6]]
                ]
            FaceTest = [[]]
        else:
            if primary: 
                thinking = True                
                while thinking == True:
                    TriElems, Case, FaceTest, sign = random.choice([R1x, R1z])(bits, 0, sign)
                    if Case != False:
                        thinking = False
            else:
                k += 1
                if k > 1000:
                    raise Exception('oops')
                if sum(bits) > 4:
                    TriElems, Case, FaceTest, sign = random.choice([R1x, R1z, Mxy, Re])(bits, k, sign)  
                else:
                    TriElems, Case, FaceTest, sign = random.choice([R1x, R1z])(bits, k, sign)  
            
        return TriElems, Case, FaceTest, sign
        
    LookupTable = [[] for i in range(256)]
    FaceTests = [[] for i in range(256)]
    Cases = [0 for i in range(256)]
    Signs = [0 for i in range(256)]
    for i in range(256):
        bits = np.array([int(b) for b in list('{:08b}'.format(i))])
        LookupTable[i], Cases[i], FaceTests[i], Signs[i] = lookup(bits,-1)
    return LookupTable, Cases, FaceTests, Signs

def _generateLookup():
    # LookupTable, Cases = _generateLookup()
    import random
    def R1x(bits, k):
        # 90 deg x-axis rotation (ccw)
        R1xnodes = np.array([4, 5, 1, 0, 7, 6, 2, 3])
        R1xedges = np.array([8, 5, 0, 4, 11, 9, 1, 3, 10, 6, 2, 7, 12])
        R1xbits = bits[R1xnodes]
        TriElems, Case = lookup(R1xbits, k=k, primary=False)
        R1xtris = [[R1xedges[n] for n in tri] for tri in TriElems]
        return R1xtris, Case

    def R1z(bits, k):
        # 90 deg z-axis rotation (ccw)
        R1znodes = np.array([3, 0, 1, 2, 7, 4, 5, 6])
        R1zedges = np.array([3, 0, 1, 2, 7, 4, 5, 6, 11, 8, 9, 10, 12])
        R1zbits = bits[R1znodes]
        TriElems, Case = lookup(R1zbits, k=k, primary=False)
        R1ztris = [[R1zedges[n] for n in tri] for tri in TriElems]
        return R1ztris, Case

    def Mxy(bits, k):
        # Mirror across XY plane
        Mxynodes = np.array([3,2,1,0,7,6,5,4])
        Mxyedges = np.array([2, 1, 0, 3, 7, 6, 5, 4, 10, 9, 8, 11, 12])
        Mxybits = bits[Mxynodes]
        TriElems, Case = lookup(Mxybits, k=k, primary=False)
        Mxytris = [list(reversed([Mxyedges[n] for n in tri])) for tri in TriElems]
        return Mxytris, Case
    
    def Re(bits, k):
        # Bit reflection (switching 0s and 1s)
        Rebits = np.array([1 if bit == 0 else 0 for bit in bits])
        TriElems, Case = lookup(Rebits, k=k, primary=False)
        Retris = [list(reversed(tri)) for tri in TriElems]
        return Retris, Case
        
        
    def lookup(bits, k=0, primary=True):
        i = int(''.join([str(bit) for bit in bits]),2)
        # print(i)
        # if i > 127:
        #     TriElems, Case, Rebits = Re(bits)
        if i == 0:
            # Case 0
            Case = 0
            TriElems  = [[]]
        elif i == 1:
            # Case 1
            Case = 1
            TriElems = [[7, 10, 11]]
                
        elif i == 3:
            # Case 2
            Case = 2
            TriElems = [[6, 9, 7], [7, 9, 11]]
                
        elif i == 5:
            # Case 3 - 5 : 0 0 0 0 0 1 0 1
            Case = 3
            TriElems = [[5, 8, 9], [7, 10, 11]]          
                       
        elif i == 20:
            # Case 4 - 20 : 0 0 0 1 0 1 0 0
            Case = 4
            TriElems = [[2, 7, 3], [5, 8, 9]]
                
        elif i == 7:
            # Case 5 - 7 : 0 0 0 0 0 1 1 1
            Case = 5
            TriElems = [[5, 7, 6], [5, 8, 11], [5, 11, 7]]
                
        elif i == 22:
            # Case 6 - 22 : 0 0 0 1 0 1 1 0
            Case = 6
            TriElems = [[2, 7, 3], [5, 8, 6], [6, 8, 10]]
                
        elif i == 74:
            # Case 7 - 74 : 0 1 0 0 1 0 1 0
            Case = 7
            TriElems = [[0, 5, 1], [6, 9, 10], [4, 11, 8]]
                
        elif i == 15:
            # Case 8 - 15 : 0 0 0 0 1 1 1 1
            Case = 8
            TriElems = [[4, 6, 5], [4, 7, 6]]
                
        elif i == 114:
            # Case 9 - 114 : 0 1 1 1 0 0 1 0
            Case = 9
            TriElems = [[0, 5, 3], [3, 5, 7], [5, 9, 7], [7, 9, 10]]
                
        elif i == 85:
            # Case 10 - 85 : 0 1 0 1 0 1 0 1
            Case = 10
            TriElems = [[0, 9, 1], [0, 8, 9], [2, 10, 11], [2, 11, 3]]    
                
        elif i == 116:
            # Case 11
            Case = 11
            TriElems = [[8, 9, 6],[3, 8, 6],[3, 6, 7],[0, 8, 3]]
                
        elif i == 120:
            # Case 12 - 120 : 0 1 1 1 1 0 0 0
            Case = 12
            TriElems = [[0, 5, 3], [3, 5, 7], [5, 6, 7], [4, 11, 8]]
                
        elif i == 90:
            # Case 13 - 90 : 0 1 0 1 1 0 1 0
            Case = 13
            TriElems = [[0, 5, 1], [5, 10, 6], [4, 11, 8], [2, 7, 3]]
                
        elif i == 113: 
            # Case 14 - 113 : 0 1 1 1 0 0 0 1
            Case = 14
            TriElems = [[6,10,11], [0,6,11], [0,11,3], [0,5,6]]
                
        else:
            if primary: 
                thinking = True
                while thinking == True:
                    TriElems, Case = random.choice([R1x, R1z, Mxy, Re])(bits, 0)
                    if not Case is False:
                        thinking = False
            else:
                k += 1
                if k > 50:
                    return [], False
                TriElems, Case = random.choice([R1x, R1z, Mxy, Re])(bits, k)            
            
        return TriElems, Case
        
    LookupTable = [[] for i in range(256)]
    Cases = [0 for i in range(256)]
    for i in range(256):
        bits = np.array([int(b) for b in list('{:08b}'.format(i))])
        LookupTable[i], Cases[i] = lookup(bits)
    return LookupTable, Cases
 
# %%
