#!/usr/bin/env bash
# Copyright 2023 BentoML Team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

set -e

if ! [ "$GITHUB_ACTIONS" = true ]; then
    echo "This script should only be run on GitHub Actions. Aborting."
    exit 1
fi

echo "Cleaning previously built artifacts..." && hatch clean

CURRENT_VERSION=$(hatch version)

if [[ $CURRENT_VERSION =~ \.dev ]]; then
    RELEASE_VERSION="${CURRENT_VERSION%%.dev*}"
else
    echo "Current version is not properly setup as dev version. Aborting..."
    exit 1
fi

echo "Releasing version $RELEASE_VERSION..." && hatch version "${RELEASE_VERSION}"

jq --arg release_version "${RELEASE_VERSION}" '.version = $release_version' < package.json > package.json.tmp && mv package.json.tmp package.json

towncrier build --yes --version "${RELEASE_VERSION}" && git add CHANGELOG.md changelog.d
git add src/openllm/__about__.py package.json && git commit -sm "infra: prepare for release ${RELEASE_VERSION} [generated]"
git push origin main

echo "Building artifacts for releasing..." && hatch build

echo "Releasing version ${RELEASE_VERSION}" && hatch publish

echo "Releasing tag ${RELEASE_VERSION}..." && git tag -a "v${RELEASE_VERSION}" -sm "Release ${RELEASE_VERSION} [generated by tools/release]"
git push origin "v${RELEASE_VERSION}"

echo "Finish releasing version ${RELEASE_VERSION}"
