# Lytspel, a Simple Phonetic Respelling for the English Language

The idea: Spelling should be fun, not a burden. The traditional English
spelling system is afflicted with exceptions and conflicting rules, making
writing and reading texts unnecessarily hard.

Lytspel is a proposal for reforming the English spelling in order to make
it strictly follow the alphabetic principle. The alphabetic principle means
that there is a predictable relationship between written letters and spoken
sounds. When you see a written word, you know how to pronounce it (even if
you don't know the word itself), and vice versa.

Traditional English spelling (tradspell) deviates from the alphabetic
principle in many ways. The mapping between written and spoken words is
very complex and riddled with irregularities and exceptions. Lytspel (for
«lytwait speling» or ‹lightweight spelling›) is a proposal to change this,
while keeping as much visual similarity to tradspell as reasonably
possible.

Lytspel not only creates an unambiguous mapping between the sounds used in
English and their written representations, it also indicates which syllable
in each word is the stressed one.

In addition to the reform proposal, there is a comprehensive dictionary and
a [converter](https://www.lytspel.org/) that translates traditional
spelling into Lytspel.

Here are two example paragraphs written in Lytspel. The first is [The North
Wind and the
Sun](https://en.wikipedia.org/wiki/The_North_Wind_and_the_Sun), one of
Aesop's fables that's frequently used as an example in phonetics, while the
second is made up of two nonsense sentences that contain most of the sounds
of English.

> Dhe North Wind and dhe Sun wur di'spiuting wich wos dhe strongger, wen a
> traveler caim a'long rapd in a worm cloak. Dhay a'greed dhat dhe won hu
> furst su'xeeded in maiking dhe traveler taik his cloak of shood bee
> con'siderd strongger dhan dhe udher. Dhen dhe North Wind blu as hard as
> hi cood, but dhe mor hi blu dhe mor cloassli did dhe traveler foald his
> cloak e'round him; and at last dhe North Wind gaiv up dhe a'tempt. Dhen
> dhe Sun shynd out wormli, and i'meediatli dhe traveler took of his cloak.
> And so dhe North Wind wos o'blyjd tu con'fess dhat dhe Sun wos dhe
> strongger ov dhe tuu.
>
> Dhat quik baizh fox jumpd in dhe air oaver eech thin dog under a caam
> autum muun. Look out, y shout, for hi's foild iu yet a'gen, cri'aiting
> cayoss.

Here are the same paragraphs written in tradspell. Lytspel might look a
bit unusual at first, but should be easy to get used too. Just try reading
it out aloud and you'll soon get the hang of it.

> The North Wind and the Sun were disputing which was the stronger, when a
> traveler came along wrapped in a warm cloak. They agreed that the one who
> first succeeded in making the traveler take his cloak off should be
> considered stronger than the other. Then the North Wind blew as hard as
> he could, but the more he blew the more closely did the traveler fold his
> cloak around him; and at last the North Wind gave up the attempt. Then
> the Sun shined out warmly, and immediately the traveler took off his
> cloak. And so the North Wind was obliged to confess that the Sun was the
> stronger of the two.
>
> That quick beige fox jumped in the air over each thin dog under a calm
> autumn moon. Look out, I shout, for he's foiled you yet again, creating
> chaos.


## Conventions and Abbreviations

The following formatting conventions are used in this document:

* «double guillemets» for Lytspel (and spellings that haven't changed)
* ‹single guillemets› for traditional spellings
* /slashes/ for phonetics, written in the
  [SAMPA](https://en.wikipedia.org/wiki/SAMPA_chart_for_English) alphabet
  (with some slight modifications to make it more suitable for English)

Abbreviations used:

* RP: [Received
  Pronunciation](https://en.wikipedia.org/wiki/Received_Pronunciation), the
  standard form of English spoken in the United Kingdom, especially in
  England and Wales
* GA: [General American](https://en.wikipedia.org/wiki/General_American),
  English as typically spoken in the United States


## The Rules of Lytspel

The syllable that carries the primary stress in a word is preceded by an
apostrophe, e.g. «dis'cuss, in'telijent, co'miti, cri'ait» ‹discuss,
intelligent, committee, create›. No stress marker is used if a word has
just one syllable or if the first syllable carries the primary stress, e.g.
«big, week, camel, garden».

The short vowels are written as follows:

* «a» /a/ as in «cat»
* «e» /E/ as in «pen»
* «i» /I/ as in «big»
* «o» /Q/ as in «dog»
* «oo» /U/ as in «book»
* «u» /V/ as in «club»

The long vowels and diphthongs are written as follows:

* «aa» /A/ as in «paam, faadher» ‹palm, father›
* «ai» /eI/ as in «aim, saint, faiss» ‹...face›
* «au» /O:/ as in «auther, paus» ‹author, pause›
* «ee» /i:/ as in «tree, teem» ‹...team›
* «oa» /oU/ as in «boat, hoam» ‹...home›
* «oi» /OI/ as in «oil, point, a'void» ‹...avoid›
* «ou» /aU/ as in «mouth, doun» ‹...down›
* «uu» /u:/ as in «muun, ix'cluud, gruup» ‹moon, exclude, group›
* «iu» (/jU/ or /ju:/) as in «valiu, kiut, com'piuter, modiulait» ‹value,
  cute, computer, modulate›
* The spelling «ue» is used in words that are typically spoken with /jU/ or
  /ju:/ in RP, with /u:/ or /U/ in GA, e.g. «due, nuetrel, tuen»
  ‹...neutral, tune›
* «y» /aI/ as in «pryss, styl, dry» ‹price, style...›

The consonants are written as follows:

* «b» /b/ as in «bed»
* «ch» /tS/ as in «much»
* «d» /d/ as in «desk»
* «dh» /D/ as in «dhem» ‹them›
* «f» /f/ as in «fat»
* «g» /g/ as in «big, garden, gess» ‹...guess›
* «h» /h/ as in «hot»
* «j» /dZ/ as in «joy, dijit» ‹...digit›
* «l» /l/ as in «leg»
* «m» /m/ as in «mad»
* «n» /n/ as in «now»
* «ng» /N/ as in «long»
* «p» /p/ as in «pop»
* «r» /r/ as in «run»
* «sh» /S/ as in «ship»
* «t» /t/ as in «test»
* «th» /T/ as in «thin»
* «v» /v/ as in «ever»
* «w» /w/ as in «west, wen» ‹...when›. Note: In some English accepts, the
  «w» in «wen» ‹when› is spoken differently than the «w» in «west», but
  since most accents (including RP and GA) no longer make this difference,
  it is not preserved in Lytspel ([wine-whine
  merger](https://en.wikipedia.org/wiki/Pronunciation_of_English_%E2%9F%A8wh%E2%9F%A9#Wine.E2.80.93whine_merger)).
* «y» /j/ as in «yet»
* «zh» /Z/ as in «mi'raazh» ‹mirage›
* The sound /k/ is written as «c» or «k», depending on context:

  * «k» is used before «e, i, y», e.g. «keep, king, kynd, calkyu'laition»
    ‹...kind, calculation›
  * «k» is also used before «h» to prevent confusion with the digraph «ch»
    /tS/, e.g. «lunkhed» ‹lunkhead›
  * «c» is used before any other letter, e.g. «cat, cost, crisp»
  * «c» is also used at the end of words ending in unstressed /ak/ or /Ik/,
    e.g. «mainiac, majic, public» ‹maniac, magic...›
  * «k» is used at the end of other words ending in /k/, e.g. «book, drink,
    quik, speek, naimsaik» ‹...quick, speak, namesake›

* The sound /s/ is written as «c», «s», or «ss», depending on context:

  * «c» is used between any vowel and a vowel starting with «e, i, y», e.g.
    «deecent, ix'plicit, nececerri, di'cyd» ‹decent, explicit, necessary,
    decide›
  * «ss» is used between any vowel and a vowel starting with «a, o, u»,
    e.g. «as'sault, epissoad, as'sumption» ‹assault, episode, assumption›.
    Note: If a stressed syllable starts with «ss», the stress marker is
    inserted before the second «s».
  * «ss» is also used at the end of words except after any of the
    consonants /f, k, p, t, T/, e.g. «less, miss, evidenss» ‹...evidence›
  * «s» is used anywhere else, e.g. «sun, desk, chips, up'sets» ‹...upsets›
  * «s» is also used in the prefixes «dis, mis» regardless of which letter
    follows, e.g. «disa'gree, disen'chantment, misin'turprit,
    misunder'stand» ‹disagree, disenchantment, misinterpret, misunderstand›

* The sound /z/ is written as «s» or «z», depending on context:

  * «s» is used between two vowels, e.g. «visit, eesi, dusen, di'syr,
    bi'sar, ri'sult» ‹...easy, dozen, desire, bizarre, result›
  * «s» is also used at the end of words except after any of the consonants
    /f, k, p, t, T/, e.g. «his, wishes, meens, paus, quis» ‹...means,
    pause, quiz›
  * «z» is used anywhere else, e.g. «zeero, ig'zact, ob'zurv» ‹zero, exact,
    observe›

The schwa /@/:

* The schwa is an unstressed and neutral vowel that occurs frequently in
  English words. Lytspel generally writes it as «e» or «u».
* «u» is used whenever the traditional spelling contains this vowel, e.g.
  «album, nurvuss, su'spend» ‹...nervous, suspend›.
* «e» is used otherwise, e.g. «camel, problem, hiden, sentrel» ‹...hidden,
  central›.
* This means that Lytspel doesn't distinguish between unstressed /V/ as in
  «funde'mentl» ‹fundamental› or /E/ as in «index» and the schwa /@/.
  However, since the unstressed forms of these vowels sound quite similar
  to the schwa and there is often no agreement between speakers whether to
  use one or the other, this shouldn't hurt.
* A schwa immediately preceding the primarily stressed syllable can also be
  represented by one of the other three vowel letters («a, i, o», but not
  «y»), e.g. «a» in «a'bout» ‹about›, «i» in «imaji'naition» ‹imagination›,
  «o» in «com'piut» ‹compute›. The reason for this is that the vowel
  preceding the stressed syllable is often spoken so quickly that it's hard
  to decide whether a schwa or one of the short vowels is pronounced and
  not all speakers will agree on which one to use. For clarity, it's
  therefore better to stick with the original representation of the vowel
  in such cases, even if most speakers might reduce it to a schwa.
  Additionally, the reduced vowel often becomes clearly audible in related
  words, e.g. while the second «i» in «imaji'naition» ‹imagination› is
  often reduced to a schwa, in «i'majin» ‹imagine› it's clearly audible as
  /I/.
* Between some consonant pairs at the end of words, the schwa is omitted in
  writing; since these pairs never occur *without* an intervening vowel at
  the end of words, there is no risk of confusion:

  * Final /s@l/ is written «ssl», e.g. «counssl, mussl» ‹council, muscle›
  * Final /z@l/ is written «sl», e.g. «pro'poasl, pusl» ‹proposal, puzzle›
  * The schwa is also omitted between any of /b, p, d, t, f, g, k/ and
    final /l/, e.g. «flexebl, simpl, midl, hospitl, litl, bafl, singgl,
    leegl, sta'tisticl, articl» ‹flexible, simple, middle, hospital,
    little, baffle, single, legal, statistical, article›. An exception is
    the suffix «ful», which is sometimes spoken with a schwa but remains
    unchanged for clarity; e.g. «helpful, meeningful» ‹...meaningful›.
  * Final /z@m/ is written «sm», e.g. «criticism, sarcasm»
  * The schwa is also omitted between /D/ and final /m/, e.g. «algeridhm,
    ridhm» ‹algorithm, rhythm›

Before «r», the vowels are spelled as follows:

* «air» /e@/ as in «pair, shair, vairi'aition» ‹...share, variation›
* «ar» /Ar/ as in «dark, argyument» ‹...argument›
* «arr» /ar/ as in «carri, em'barress» ‹carry, embarrass›
* «eer» /i@/ as in «cheer, yeer» ‹...year›. Note: GA speakers often
  pronounce this in the same way as the «ir» /Ir/ in «mirer» ‹mirror›
  ([mirror–nearer
  merger](https://en.wikipedia.org/wiki/English-language_vowel_changes_before_historic_/r/#Mirror%E2%80%93nearer_merger)).
* «er» /@`/ as in «number, modern» (r-colored schwa). Note: In RP and other
  [non-rhotic accents](https://en.wikipedia.org/wiki/Rhoticity_in_English),
  this usually sounds like a normal schwa; likewise, «r» after other vowels
  is often inaudible or reduced to a schwa.
* «err» /Er/ as in «merri, errer» ‹merry, error›
* «ir» /Ir/ as in «mirer, iritait» ‹mirror, irritate›
* «oar» /or/ as in «poart, ig'noar, keeboard» ‹port, ignore, keyboard›
* «or» /O:r/ as in «north, order, iuniform» ‹...uniform›. Note: Most RP
  speakers pronounce this in the same way as the «oar» /or/ in «board», but
  the two sounds are distinguished in GA.
* «orr» /Qr/ as in «sorri, to'morro» ‹sorry, tomorrow›. Note: Most GA
  speakers pronounce this in the same way as the «or» /O:r/ in «north», but
  the two sounds are distinguished in RP.
* «oor» /Ur/ as in «poor, toor, in'shoorenss» ‹...tour, insurance›
* «iur» (/ju:r/ or /jUr/) as in «Iurep, piur, ob'skiur» ‹Europe, pure,
  obscure›
* The spelling «uer» is used in words that are typically spoken with /ju:r/
  or /jUr/ in RP, with /Ur/ in GA, e.g. «duering, nuerel» ‹during, neural›
* «our» /aUr/ as in «flour, our» ‹flour, our *or* hour›
* «ur» /3:/ as in «turn, furst, con'surn, obzur'vaition» ‹...first,
  concern, observation›. This sound only occurs before «r» and can be
  considered a stronger (and often stressed) variant of the schwa.
* «urr» /Vr/ as in «hurri, current» ‹hurry...›
* «yr» is spoken /aI@`/ at the end of words (e.g. «fyr, in'spyr» ‹fire,
  inspire›), /aIr/ elsewhere (e.g. «vyruss, yreni, spyrel» ‹virus, irony,
  spiral›)

Some vowels are spelled differently at the end of words or before other
vowels:

* /eI/ is spelled «ay» at the end of words and before other vowels (instead
  of «ai»), e.g. «play, holiday, layer, cayoss» ‹...chaos›.
* /aI/ is spelled «ai» before other vowels (instead of «y»), e.g.
  «dai'amiter, quaiet, vaielenss, aiern, haierarki, Ha'waii» ‹diameter,
  quiet, violence, iron, hierarchy, Hawaii›. This prevents confusion with
  the consonant /j/ (as in «yet») which is also spelled «y» but only occurs
  before vowels. There is no risk of confusion with /eI/ (as in «aim,
  layer») which is always spelled «ay» in this position.
* The same spelling is used if /aI/ occurs after a vowel at the end of a
  word, e.g. «raidiai» ‹radii›.
* /O:/ is spelled «aw» at the end of words and before other vowels (instead
  of «au»), e.g. «law, drawing».
* Final /E/ is spelled «eh» (instead of «e»), e.g. «eh, Yaaweh»
  ‹...Yahweh›.
* Unstressed /i:/ is spelled «i» at the end of words and before other
  vowels (instead of «ee»), e.g. «fansi, ne'ceciti, vidio, cri'aition»
  ‹fancy, necessity, video, creation›. This means that Lytspel cannot
  distinguish between unstressed /i:/ and unstressed /I/ (as in «big») in
  these positions, but that shouldn't hurt, as /I/ is rarely used in these
  positions and dictionaries often disagree on which of these sounds to use
  anyway.
* This spelling is also used in the personal pronouns ending in /i:/ (as
  these are often less stressed than other words in a sentence): «hi, shi,
  wi, mi» ‹he, she, we, me›. And it's used at the end of prefixes such as
  «anti, semi» (spoken with /i:/ by many speakers, with /aI/ by others) as
  well as «di, pri, ri» ‹de, pre, re› (even if the prefix is stressed),
  e.g. «anti'aircraft, anti'soashel; semi'fynel, semiaute'matic; di'graid,
  di'bug; prima'choor, pri'requisit; ri'act, ripro'duess» ‹antiaircraft,
  antisocial; semifinal, semiautomatic; degrade, debug; premature,
  prerequisite; react, reproduce›.
* /oU/ is spelled «o» at the end of words and before other vowels (instead
  of «oa»), e.g. «yelo, zeero, co'operait, po'etic, sho» ‹yellow, zero,
  cooperate, poetic, show›. This poses little risk of confusion, as /Q/ (as
  in «dog») occurs rarely or never before other vowels. (Final /Q/ could be
  written «oh», but in practice it never seems to occur.)
* This spelling is also used at the end of prefixes such as «astro, baio,
  endo, ethno, ferro, fyto, galveno, hetero, hypo, imyuno, jio, keemo,
  macro, mycro, nio, nuemo, nuero, palio, soacio, spectro, suudo, thurmo,
  ysso» ‹astro, bio, endo, ethno, ferro, phyto, galvano, hetero, hypo,
  immuno, geo, chemo, macro, micro, neo, pneumo, neuro, paleo, socio,
  spectro, pseudo, thermo, iso›, e.g. «baio'kemistri, ethno'sentric,
  hetero'secshual, imyunodi'fishenssi, jio'fisicl, keemo'therrepi,
  mycrowaiv, nio'classic, nuero'surjeri, suudo'saienss, thurmody'namics»
  ‹biochemistry, ethnocentric, heterosexual, immunodeficiency, geophysical,
  chemotherapy, microwave, neoclassic, neurosurgery, pseudoscience,
  thermodynamics›.
* /OI/ is spelled «oy» at the end of words and before other vowels (instead
  of «oi»), e.g. «boy, en'joy, royel» ‹...enjoy, royal›.
* /aU/ is spelled «ow» at the end of words and before other vowels (instead
  of «ou»), e.g. «now, power, a'low» ‹...allow›.
* Final /V/ is spelled «uh» (instead of «u»), e.g. «huh».
* /u:/ is spelled «u» at the end of words and before other vowels (instead
  of «uu»), e.g. «clu, ishu, sichu'aition» ‹clue, issue, situation›. This
  poses little risk of confusion, as /V/ (as in «bus») occurs rarely or
  never before other vowels.
* Final schwa /@/ is spelled «a» (instead of «e» or «u»), e.g. «extra,
  daita» ‹...data›. (Should unstressed /a/ (as in «cat») occur at the end
  of words, it could be written «ah», but in practice it never seems to
  occur.)
* Schwa /@/ before other vowels is always spelled «e» (never «u») to
  prevent confusion with /u:/ (spelled «u» in this position). Schwa in this
  position is very rare.

Spellings involving several sounds:

* The vowel combination /i:@/ or /I@/ is written «ia», e.g. «mi'steeriass,
  ma'teerial, meediam, obviass, vairiabl, airia, i'meediat» ‹mysterious,
  material, medium, obvious, variable, area, immediate›. In a few words,
  «ia» instead represents /i:a/, e.g. «en'thuesiasm, zoadiac» ‹enthusiasm,
  zodiac› (this distinction is not represented in writing).
* The r-colored vowel combination /i:@`/ is written «ier», e.g. «thieri»
  ‹theory›.
* The vowel combination /oU@/ is written «oe», e.g. «boe, coe'lition,
  ys'socroess» ‹boa, coalition, isochroous›. In a few words, «oe» instead
  represents /oUE/, e.g. «floem» ‹phloem› (this distinction is not
  represented in writing).
* The vowel combination /u:@/ or /U@/ is written «ua», e.g. «acchual,
  indi'vijual, influanss, soo'purfluass, crual» ‹actual, individual,
  influence, superfluous, cruel›. In a few words, «ua» instead represents
  /u:a/, e.g. «bivuac» ‹bivouac› (this distinction is not represented in
  writing).
* Likewise, /ju:@/ and /jU@/ are written «iua», e.g. «am'bigiuass, maniual»
  ‹ambiguous, manual›.
* The consonant combination /kw/ is written «qu», e.g. «quit, ri'quest»
  ‹...request›; the letter «q» only occurs in this combination.
* The consonant combination /ks/ is written «x», e.g. «mix, next,
  ix'plicit, axident» ‹...explicit, accident›.
* Since «ng» represents the single sound /N/ (as in «long»), the sound
  combination /ng/ is written «n'g» if a stress marker is needed in this
  position (e.g. «en'gaij, in'grain» ‹engage, ingrain›), «nng» otherwise
  (e.g. «cairnngorm, martenngail» ‹cairngorm, martingale›).
* Before /k/, «n» is pronounced /N/ rather then /n/ (e.g. «bank, tranquil,
  dis'tinct, bron'kytiss» ‹...distinct, bronchitis›), except if «n» and /k/
  belong to different parts of a compound (e.g. «mankynd, raincoat»
  ‹mankind...›) or if a word starts with any of «con, en, in, non, un»
  followed by a /k/ sound (e.g. «con'cluusion, en'counter, in'compitent,
  increess *or* in'creess, noncon'formist, un'cleer» ‹conclusion,
  encounter, incompetent, increase, nonconformist, unclear›).
* If /Nk/ is spoken where the preceding rule suggests /nk/, the /N/ is
  written «ng», e.g. «congker, ingk, ingcling, ungcl, ungction» ‹conker
  *or* conquer, ink, inkling, uncle, unction›. Some words starting with
  «con» or «in» are pronounced with /n/ by some speakers, with /N/ by
  others; in such cases, Lytspel uses «n» for simplicity, e.g. «conquest,
  increment, concreet, inqui'sition» ‹...concrete, inquisition›.
* /n/ is written «nn» if it occurs before /k/ in places where «n» would be
  spoken /N/, e.g. «canncan, melenncoli, noamennclaicher» ‹cancan,
  melancholy, nomenclature›.

Some sound combinations are written in a special way if they occur at the
end of words:

* Final /O:l/ is written «all», e.g. «ball, in'stall» ‹...install›. This
  spelling is preserved in derived words, e.g. «walls, calling,
  in'stallment» ‹...installment›.
* Final /S@n/ is written «tion», e.g. «recog'nition, obzur'vaition,
  section, per'mition, moation, oation, miu'sition» ‹recognition,
  observation, section, permission, motion, ocean, musician›.
* Final /Z@n/ is written «sion», e.g. «vision, con'fiusion, i'quaision»
  ‹...confusion, equation›.
* The «tion» and «sion» spellings aren't used in words that are derived by
  appending «n» or «en» to another word, e.g. «ashen, freshen, Rushan,
  Mela'neezhan» ‹ashen, freshen, Russian, Melanesian› from «ash, fresh,
  Rusha, Mela'neezha» ‹ash, fresh, Russia, Melanesia›.
* The «tion» and «sion» spellings are preserved in derived words (e.g.
  «ri'laitionship, tra'ditionel, o'caisionel» ‹relationship, traditional,
  occasional›), even if the derived form is irregular (e.g. «nationel»
  ‹national› from «naition» ‹nation›).

Disambiguating multigraphs (sequences of multiple characters representing a
single sound or an r-colored vowel):

* Spellings involving digraphs (such as «ch, sh, ss, ai, oa, oy, ur») and
  trigraphs (such as «arr, eer») are generally read from left to right: the
  first letter sequence that *can* be read as a multigraph *should* be read
  as such; e.g. «dueel» ‹dual *or* duel› is read as «d-ue-e-l» (/dj'u:@l/
  or /d'u:@l/), «layer» is read as «l-ay-er» (/l'eI@`/).
* But the trigraphs «iai, uai» are read as «i» /i:/ or «u» /u:/ followed by
  «ai» (spoken /aI/ at the end of words, /eI/ otherwise), e.g.
  «a'preeshiait, flucchuait, raidiai» ‹appreciate, fluctuate, radii›.
* Since final /OI/ is spelled «oy», final «oi» unambiguously represents the
  two vowels /oUi:/. This combination occasionally occurs in derived words,
  e.g. «shoi» ‹showy› from «sho» ‹show›.
* «oic» at the end of words is always spoken /oUIk/ (two separate vowels),
  e.g. «hi'roic, stoic» ‹heroic...›.
* Double «rr» always forms a trigraph with the vowel to its left, e.g.
  «Februerri» ‹February› is read as «F-e-b-r-u-err-i».
* In cases where the previous rules would lead to a misreading, Lytspel
  recommends inserting a hyphen to indicate the correct reading, e.g.
  «Cro-at, po-it, co-in'syd, co-ope'raition, su-er, Lu-eesi'ana,
  Aalts-hymer, sheeps-hed» ‹Croat, poet, coincide, cooperation, sewer,
  Louisiana, Alzheimer, sheepshead›.
* Those who prefer a more classical form may omit the hyphen between two
  vowels and place a diaeresis over the second one, e.g. «Croät, poït,
  coïn'syd, coöpe'raition, Luëesi'ana».
* Multigraphs ending in «r» may be interrupted by a stress marker before
  the final «r» and are still read as multigraphs, e.g. «co-ope'raition,
  hor'rific» ‹cooperation, horrific›. The multigraph «ss» is broken in the
  middle, e.g. «as'sumption» ‹assumption›.
* Other multigraphs are never interrupted by a stress marker, hence in such
  cases a marker has the same effect as a hyphen: it indicates that the
  letters to the left and to the right of it should be read in isolation,
  e.g. «co'operait» ‹cooperate›.
* In particular, the vowel multigraphs «aw, ay, ow, oy» /O:, eI, aU, OI/
  are *not* broken in the rare cases where a stressed vowel follows, to
  prevent misreadings. Contrast «draw'ee» ‹drawee› with «a'waik» ‹awake›,
  «cay'otic» ‹chaotic› with  «ga'yaal» ‹gayal›.
* Hyphens are generally not inserted between a base word and a suffix, e.g.
  «truer» from «tru» ‹true›.
* After a prefix or the first part of a compound, a hyphen is only inserted
  if the last letter of the first part and the first letter of the second
  part would otherwise form one of the combinations «oa, oi, oo, ou, iu»
  (each of which represents a single vowel), e.g. «thro-away, co-ig'zist,
  blo-out, poli-un'sacheraited» ‹throwaway, coexist, blowout,
  polyunsaturated›. In other cases, no hyphens are inserted, e.g.
  «shorthand».
* Writers may, of course, insert a hyphen (or a diaeresis) between the
  parts of an affixed or compound word whenever they consider such a visual
  separator helpful to prevent misreadings or confusion.

Rules for derived and related words:

* In front of suffixes starting with «e, i, y», final «c» is changed to
  «k», e.g. «trafiking» ‹trafficking› from «trafic» ‹traffic›.
* Final «c» remains unchanged in front of such suffixes if its
  pronunciation changes from /k/ to /s/, e.g. «publicist, publicys»
  ‹...publicize› from «public».
* In front of suffixes consisting in a single consonant («d, n, s»), final
  «i» (unstressed /i:/) becomes «ie» to mark it as long, e.g. «studied»
  from «studi» ‹study›, «vairies» ‹varies› from «vairi» ‹vary›. Otherwise
  forms such as "studid" would look like «splendid», but be spoken quite
  differently.
* To prevent misreadings, final «o» /oU/ is changed back to «oa» and final
  «u» /u:/ to «uu» in front of the same suffixes, e.g. «foload» ‹followed›
  from «folo» ‹follow›, «chuud» ‹chewed› from «chu» ‹chew›, «throan»
  ‹thrown› from «thro» ‹throw›, «struun» ‹strewn› from «stru» ‹strew›,
  «po'taitoas» ‹potatoes› from «po'taito» ‹potato›, «shuus» ‹shoes› from
  «shu» ‹shoe›.
* A schwa /@/ at the start of suffixes is written as «a» if the base word
  ends in /i:/ or /u:/, e.g. «eesiast» ‹easiest› from «eesi» ‹easy›,
  «vairiabl» ‹variable› from «vairi» ‹vary›, «duabl» ‹doable› from «du»
  ‹do›, «a'greeabl» ‹agreeable› from «a'gree» ‹agree›. This ensures
  consistency with the combined spellings «ia» and «ua» used elsewhere (see
  above) and prevents confusion with «ie» in words such as «studied». But
  note that r-colored schwa remains «er», e.g. «eesier» ‹easier›.
* As noted above, the schwa is omitted between certain consonants and final
  «l» or «m». To prevent misreadings, the omitted schwa is restored as «e»
  in front of the suffixes «ait/et, erri, ism, ist, oid, uss, ys» ‹ate,
  ary..., ous, ise/ize› as well as before stressed «ee», e.g.
  «en'capselait» ‹encapsulate› from «capssl» ‹capsule›, «piupelerri»
  ‹pupillary› from «piupl» ‹pupil›, «vandelism» ‹vandalism› from «vandl»
  ‹vandal›, «iuni'vurselism» ‹universalism› from «iuni'vurssl» ‹universal›,
  «voakelist» ‹vocalist› from «voacl» ‹vocal›, «cristelys, cristeloid»
  ‹crystallize, crystalloid› from «cristl» ‹crystal›, «lybeluss» ‹libelous›
  and «lybe'lee» ‹libelee› from «lybl» ‹libel›. The obvious exception are
  derived words where no schwa is spoken, e.g. «syclys» ‹cyclize› from
  «sycl» ‹cycle›, «simplism» from «simpl» ‹simple›.
* Otherwise, end-of-word spellings are preserved in all derived forms as
  long as the base word doesn't change (except for possible shifts of
  stress), e.g.

  * «c» representing /k/: «publicli, publi'caition» ‹publicly, publication›
    from «public»; «me'canics, me'canicl» ‹mechanics, mechanical› from
    «me'canic» ‹mechanic›
  * «k» representing /k/: «weeks, weekli» ‹...weekly› from «week»
  * «ss» representing /s/: «classic» from «class»; «con'vinssd,
    con'vinssing» ‹convinced, convincing› from «con'vinss» ‹convince›;
    «a'nounssment» ‹announcement› from «a'nounss» ‹announce›; «prinssess»
    ‹princess› from «prinss» ‹prince›
  * «a» representing /@/: «extras» from «extra», «A'merrican» ‹American›
    from «A'merrica» ‹America›
  * «aw» representing /O:/: «draws, drawn, drawing» from «draw»
  * «i» representing unstressed /i:/: «fansiful, fansiing» ‹fanciful,
    fancying› from «fansi» ‹fancy›; «hapiness» ‹happiness› from «hapi»
    ‹happy›
  * «oy» representing /OI/: «di'stroys, di'stroyd, di'stroyer» ‹destroys,
    destroyed, destroyer› from «di'stroy» ‹destroy›
  * «y» representing /aI/ (unchanged even before vowels): «hyer» ‹higher›
    from «hy» ‹high›; «su'plyer» ‹supplier› from «su'ply» ‹supply›
  * «yr» representing /aI@`/: «in'spyrd» ‹inspired› from «in'spyr»
    ‹inspire›

* They are also preserved in compounds, e.g. «bakground, extrate'restrial,
  fyrwurks›» ‹background, extraterrestrial, fireworks›.
* If the pronunciation of the base word itself changes (beyond mere shifts
  of stress), the spelling reflects this, e.g. «creecher» ‹creature› from
  «cri'ait» ‹create›; «dis'cution» ‹discussion› from «dis'cuss» ‹discuss›;
  «con'cluusion» ‹conclusion› from «con'cluud» ‹conclude›; «acchual»
  ‹actual› from «act»; «as'sumption» ‹assumption› from «as'suum» ‹assume›;
  «men'taliti» ‹mentality› from «mentl» ‹mental›.
* The past tense of regular verbs is formed by appending «ed» if /@d/ is
  spoken (e.g. «si'lected, waisted» ‹selected, wasted›), «d» if /d/ or /t/
  is spoken (e.g. «ri'veeld, con'siderd, payd, fixd, con'vinssd» ‹revealed,
  considered, paid, fixed, convinced›). The past tense of irregular verbs
  is spelled as it's spoken, e.g. «kept» from «keep».
* While in tradspell a noun and a related verb are often written the same
  even if spoken differently, in Lytspel the spelling reflects the actual
  pronunciation, e.g. «a suspect, to su'spect; a record, to ri'cord; a
  houss, to hous» ‹a suspect, to suspect; a record, to record; a house, to
  house›.
* Normally Lytspel doesn't use double consonants, except in those cases
  where a double consonant is spoken differently than a single one (e.g.
  «ss» spoken /s/ in certain contexts where «s» is spoken /z/). However,
  double consonants can also occur if they belong to different parts of a
  compound or if one belongs to a prefix or suffix and the other to the
  base word, e.g. «toatlli, suuper'reejenel, un'nececerri» ‹totally,
  superregional, unnecessary›.
* Unstressed final /Iks/ traditionally written ‹ics› is treated as if it is
  a plural (and hence written «ics») even if the corresponding singular
  form (ending in /Ik/) is unknown or rarely used, e.g. «astro'fisics,
  daia'netics» ‹astrophysics, dianetics›. This is for consistency with the
  many cases where «ics» /Iks/ indeed marks a plural or quasi-plural, e.g.
  «ilec'tronics» ‹electronics› from «ilec'tronic» ‹electronic›.
* For consistency with forms derived by appending «li» ‹ly› to words ending
  in «cl» ‹cal› (e.g. «practiclli» ‹practically› from «practicl»
  ‹practical›), all forms derived by appending ‹ally› (pronounced /@li:/ or
  /li:/) to words ending in «c« are written «clli», e.g. «baissiclli»
  ‹basically› from «baissic» ‹basic›, «spe'cificlli» ‹specifically› from
  «spe'cific» ‹specific›.

Irregular words:

* The vowels in the articles «dhe, a, an» ‹the...› retain their traditional
  spelling.
* «ey» ‹eye› is written irregularly to distinguish it from «y» ‹I› and for
  easier recognition in compounds such as «fishey» ‹fisheye›.
* «noa» ‹know› is written somewhat irregularly to distinguish it from «no».
* «oa» ‹owe› is written somewhat irregularly since a verb with just one
  letter could be confusing (and also to distinguish it from the
  interjection «o» ‹oh›).
* «tuu» ‹too *or* two› is written somewhat irregularly to distinguish it
  from «tu» ‹to›.

Uses of the apostrophe:

* Lytspel uses an apostrophe to mark the syllable that carries the main
  stress (provided it's not the first one). Additionally, as in tradspell,
  apostrophes are used to mark genitives and contractions.
* «'s» is used for the genitive (as in tradspell) and for contractions with
  «is, has», e.g. «Linda's sister, dhair's» ‹...there's›.
* «'d» is used for contractions with «had, wood, shood» ‹...would, should›,
  e.g. «it'd».
* «'l» ‹'ll› is used for contractions with «wil, shal» ‹will, shall›, e.g.
  «shi'l» ‹she'll›.
* The contraction «y'm» ‹I'm› is short for «y am» ‹I am›.
* «'r» ‹'re› is used for contractions with «ar» ‹are›, e.g. «dhay'r»
  ‹they're›.
* «'v» ‹'ve› is used for contractions with «hav» ‹have›, e.g. «iu'v»
  ‹you've›.
* «n't» is used for contractions with «not» where the /Q/ is not spoken,
  e.g. «shoodn't» ‹shouldn't›. If the first word ends in «n», the second
  «n» is omitted: «can't» from «can not». In a few cases, the first word
  may be further contracted: «ain't» for one of «am/is/ar/has/hav not»
  ‹am/is/are/has/have not›, «han't» for «has/hav not» ‹has/have not›,
  «shan't» for «shal not» ‹shall not›, «woan't» ‹won't› for «wil not». The
  pronunciations of «doan't» ‹don't› and «musn't» ‹mustn't› differ from the
  uncontracted forms, and the spelling reflects this.
* In all these cases, the genitive or contraction marker is inserted after
  the last vowel and hence cannot be confused with a stress marker.
* In other cases, a traditional contraction marker doubles as a stress
  marker in Lytspel, e.g. «o'clok» ‹o'clock› for «ov dhe clok» ‹of the
  clock›.
* This also applies to Irish names starting with «O'», e.g. «O'Coner,
  O'Neel» ‹O'Connor, O'Neil›. The capitalized «O'» is spoken /oU/ (as in
  «boat»). Of course, proper names can be spelled as their owners prefer
  it, so many names won't be regularized according to the Lytspel rules.
* In the nonstandard expression ‹y'all›, short for «iu all» ‹you all›, the
  contraction marker is placed in front of the stressed vowel, marking the
  original word boundary.
* In a few cases, words are contracted by eliminating a syllable or a sound
  for brevity or for literary effect. In these cases, Lytspel recommends
  placing a contraction marker before the final consonant in monosyllabic
  words («ai'r, nai'r, oa'r, ee'n, ma'm, bru'r» ‹e'er, ne'er, o'er, e'en,
  ma'am, br'er› from «ever, never, oaver, eeven, madem, brudher» ‹...,
  over, even, madam, brother›) and just using a regular stress marker
  otherwise («wair'air, wairso'air, nor'eester» ‹where'er, wheresoe'er,
  nor'easter› from «wair'ever, wairso'ever, north'eester» ‹wherever,
  wheresoever, northeaster›).

Inner capitalization instead of a stress marker:

* Various Scottish and Irish names start with ‹Mc› or ‹Mac› /m@k/ followed
  by a capitalized and stressed second syllable. Lytspel recommends writing
  these names with «Mc» and omitting the stress marker, e.g. «McAdem,
  McDonel» ‹McAdam, McDonnell›. If the stressed syllable starts with a /k/
  sound, that letter is capitalized (and repeated after the initial «Mc»),
  e.g. «McCarti, McKee, McQueen» ‹McCarty...›. If any other syllable is
  stressed, Lytspel recommend writing the word regularly without inner
  capitalization, e.g. «Makintosh» ‹Macintosh *or* McIntosh›.


## International Pronunciation Differences

As an international language, English is currently written and spoken
somewhat differently in various parts of the world.

On the one hand, by using a phonetic spelling, Lytspel eliminates most
differences that traditionally exist between written British (BE) and
American English (AE), e.g. «culer, senter, orgenys, anelys, lycenss,
catalog, aijing, program, chek, gray» ‹colour/color, centre/center,
organise/organize, analyse/analyze, licence/license, catalogue/catalog,
ageing/aging, programme/program, cheque/check, grey/gray›.

On the other hand, tailoring Lytspel to faithfully represent just one
regional variety would cause many deviations from the alphabetic principle
for everyone else. And creating different variants of Lytspel each fitting
just one regional variety would introduce even larger differences between
different regional varieties of written English than traditionally exist
between BE and AE.

Instead of going down either of theses routes, Lytspel had been designed as
some kind of "global compromise." It aims to keep the phonetic principle
reasonably intact for all varieties of spoken English, and particularly for
RP (since the English language originates in England) and GA (since the
United States are the country with the highest number of native English
speakers).

Various details of the Lytspel spellings have already been motivated as
representing sound distinctions that exist either in RP or in GA, and the
«ue» and «uer» spellings are a compromise used in words spoken differently
in RP and GA. But such compromises are not almost possible without making
the written language extremely complex and unwieldy. In other cases,
Lytspel therefore resolves differences between RP and GA by choosing one
spelling -- often the one that keeps the written language more similar to
tradspell or else the shorter one:

* /a/ («a») is preferred if a vowel is commonly spoken as either /a/ or
  /A/, e.g. «last, guulash» ‹...goulash›.
* /a/ is also preferred if a vowel is commonly spoken as either /a/ or
  /eI/, e.g. «patent, patriat» ‹...patriot›.
* /a/ is also preferred if a vowel is commonly spoken as either /a/ or /Q/,
  e.g. «rath» ‹wrath›.
* /A/ («aa)» is preferred if a vowel is commonly spoken as either /A/ or
  /eI/, e.g. «to'maato, swaadh» ‹tomato, swathe›.
* /aI/ («y», «ai» before vowels) is preferred if a vowel is commonly spoken
  as either /aI/ or /i:/, e.g. «nydher, filistyn» ‹neither, philistine›.
* /E/ («e)» is preferred if a vowel is commonly spoken as either /E/ or
  /eI/, e.g. «a'gen» ‹again›.
* /E/ is also preferred if a vowel is commonly spoken as either /E/ or
  /i:/, e.g. «lezher, predececer» ‹leisure, predecessor›.
* /I/ («i») is preferred if a vowel is commonly spoken as either /I/ or
  /aI/, e.g. «cristelin, di'rect, orgeni'saition» ‹crystalline, direct,
  organization›.
* /I/ is also preferred if a vowel is commonly spoken as either /I/ or /i:/
  (e.g. «afro'disiac» ‹aphrodisiac›), unless its original representation is
  ‹ee› (e.g. «been»).
* /i:/ («ee», «i» before vowels) is preferred if a vowel is commonly spoken
  as either /i:/ or /eI/, e.g. «beeta, ga'seebo, Izrial» ‹beta, gazebo,
  Israel›.
* /Q/ («o») is preferred if a vowel is commonly spoken as either /Q/ or
  /O:/, e.g. «alkehol, cloth» ‹alcohol...›.
* /Q/ is also preferred if a vowel is commonly spoken as either /Q/ or
  /oU/, e.g. «process, shon» ‹...shone›.
* /Q/ is also preferred if a vowel is commonly spoken as either /Q/ or /V/,
  e.g. «from, wot, a'complish» ‹...what, accomplish›.
* /oU/ («oa») is preferred if a vowel is commonly spoken as either /oU/ or
  /O:/, e.g. «sloath» ‹sloth›.
* /U/ («oo») is preferred if a vowel is commonly spoken as either /U/ or
  /u:/, e.g. «coocu, hoof» ‹cuckoo...›.
* /Ir/ («ir») is preferred if an r-colored vowel is commonly spoken as
  either /Ir/ or /3:/, e.g. «squirel» ‹squirrel›.
* /3:/ («ur») is preferred if an r-colored vowel is commonly spoken as
  either /3:/ or /A(r)/, e.g. «clurk, durbi» ‹clerk, derby›.
* /w/ («w») is preferred if a sound preceding a vowel is commonly spoken as
  either /w/ or one of /ju:/ or /jU/, e.g. «jagwar» ‹jaguar›.
* If a vowel is spoken as a schwa or omitted altogether in some regions, as
  /aI/ elsewhere, Lytspel generally uses «i» /I/ as a compromise, e.g.
  «ajil, vursetil», ‹agile, versatile›.
* Otherwise, if a vowel is spoken with a full vowel sound in some regions,
  while reduced to a schwa or omitted altogether elsewhere, Lytspel tends
  to write the full version, e.g. «categoari, inevaitiv, militerri,
  poartrait, prymerri, python, roazmairi, testimoani» ‹category,
  innovative, military, portrait, primary, python, rosemary, testimony›.
* /z/ («s» or «z», depending on position) is preferred if a consonant is
  commonly spoken as either /z/ or /s/, e.g. «i'rais, venisen, talizmen»
  ‹erase, venison, talisman›.
* If a consonant is commonly spoken as either /Z/ or /S/, final «sion»
  /Z@n/ is preserved if it's already the traditional ending (e.g.
  «con'vursion» ‹conversion›), otherwise /S/ (usually «sh») is preferred,
  e.g. «Aisha, i'raisher» ‹Asia, erasure›.
* /si:/ (depending on position, but often «ci») is preferred if a sequence
  is commonly spoken as either /si:/ or /S/, e.g. «hecian, om'niciant»
  ‹hessian, omniscient›.
* /zi:/ (depending on position, but often «si») is preferred if a sequence
  is commonly spoken as either /zi:/ or /Z/, e.g. «hoasieri, Ma'laisia»
  ‹hosiery, Malaysia›.
* /ti:/ («ti» before vowels) is preferred if a sequence is commonly spoken
  as either /ti:/ or as one of /S, Si:, tS, tSi:/, e.g. «sentiant,
  con'sortiam, bestial, bestierri» ‹sentient, consortium, bestial,
  bestiary›.

If a letter is commonly either spoken or silent, Lytspel tends to preserve
it, e.g. «platinum, trait» (not "platnum, tray").

If speakers in different regions tend to stress different syllables,
Lytspel usually places the stress early rather than late, e.g. «an adress,
decaid, truncait, weekend» ‹an address, decade, truncate...›. But in the
case of originally foreign words where a late stress placement corresponds
to the pronunciation in the original language, it is often preserved, e.g.
«ca'feen, expoa'say» ‹caffeine, exposé›.

Other differences are usually unique to just one or very few words. In such
cases, Lytspel generally prefers the pronunciation that is more similar to
the traditional written form, e.g. «cordial, figer, shediul» ‹...figure,
schedule› (rather than "corjel, figyer, skejuul").

In a few cases, one pronunciation is preferred because it is more similar
to that of a closely related word, e.g. «pryveci» ‹privacy› from «pryvet»
‹private› (rather than "priveci").


## Limitations of the Current Dictionary

The Lytspel dictionary, as currently published, can distinguish between
words traditionally written the same but pronounced differently
(heteronyms) as long as their grammatical role in a sentence is different
-- one is a noun, while the other is a verb, for example. This allows
distinguishing the verb «cloas» ‹close› from the adjective «cloass», the
noun «object» from the verb «ob'ject», etc. («Dhay wur tu cloass to dhe
doar to cloas it.» ‹They were too close to the door to close it.› / «Y did
not ob'ject to dhe object.» ‹I did not object to the object.›)

However, heteronyms are currently only disambiguated by their grammatical
role (commonly known as *part of speech,* or *POS*), hence if two
heteronyms share the same grammatical role, the dictionary will list only
one of them. The other one should, of course, nevertheless be used when
appropriate. Words where this is the case include (the missing spelling is
listed in parentheses):

* ‹bass›: «baiss» (or «bass» for various fish species)
* ‹drawer›: «dror» (or «drawer» for a person who draws)
* ‹logos›: «loagoas» (plural of «loago» ‹logo›; or «logoss» for the Greek
  word)
* ‹micrometer›: «my'cromiter» (a measuring device; or «mycromeeter» for the
  millionth part of a meter)
* ‹prayer›: «prair» (or «prayer» for a person who prays)
* ‹read›: «reed» (or «red» for the past tense)
* ‹routed, router, routing›: «ruuted, ruuter, ruuting» (if derived from
  «ruut» ‹route›; or «routed, router, routing» if derived from «rout»)
* ‹subsequence›: «subsiquenss» (for some subsequent occurrence; or
  «subseequenss» for a sequence derived from another sequence)
* ‹taxis›: «taxies» (multiple taxis, or taxicabs; or «taxiss», a scientific
  term)
* ‹tier›: «teer» (or «tyer» for a person who ties)


## Other Documentation

**docs/lytspel-on-one-page.md|odt|pdf**: A compact, single-page description
of the most important rules. The PDF is exported from the ODT
(OpenDocument/LibreOffice) document, while the Markdown (md) version is
maintained independently.

**LICENSE.txt** contains the license used for the Lytspel program suite (a
permissive license known as "ISC license"). The proposed spellings do not
fall under copyright (you cannot copyright a single word, let alone a way
to spell it) and may be considered a part of the public domain.

**CHANGELOG.md** contains a compact list of the changes made between the
various released versions of the program (and dictionary).

If you want to install and use the program locally (rather then using the
web frontend), see **INSTALL-USE.md**.

If you want to experiment with the program suite locally, in order to add
some functionality, fix bugs, or modify spellings, see **devnotes.md**.

**files.md** quickly describes the various files in the Lytspel repository.
This too is mostly interesting if you want to modify the dictionary or some
of the programs used to generate it.
