import time
import enum
import struct
import abc
import math
import asyncio
import socket
import threading
import serial

from typing import Callable, Union, Optional, Any, List
from ipaddress import ip_network, ip_address
from ctypes import c_ushort
from ipaddress import IPv4Network
from psutil import net_if_addrs
from datetime import datetime, timedelta, date as d_date, time as d_time
from .base import BasePoint, BaseDriver, PointTableDict

'''
    pip install pyserial==3.5
    pip install psutil==5.8.0
'''

class BACnetAbortReason(enum.IntEnum):
    other = 0
    buffer_overflow = 1
    invalid_apdu_in_this_state = 2
    preempted_by_higher_priority_task = 3
    segmentation_not_supported = 4
    security_error = 5
    insufficient_security = 6
    window_size_out_of_range = 7
    application_exceeded_reply_time = 8
    out_of_resources = 9
    tsm_timeout = 10
    apdu_too_long = 11

class BACnetRejectReason(enum.IntEnum):
    other = 0
    buffer_overflow = 1
    inconsistent_parameters = 2
    invalid_parameter_data_type = 3
    invalid_tag = 4
    missing_required_parameter = 5
    parameter_out_of_range = 6
    too_many_arguments = 7
    undefined_enumeration = 8
    unrecognized_service = 9

class BACnetDeviceStatus(enum.IntEnum):
    OPERATIONAL = 0
    OPERATIONAL_READONLY = 1
    DOWNLOAD_REQUIRED = 2
    DOWNLOAD_IN_PROGRESS = 3
    NON_OPERATIONAL = 4
    BACKUP_IN_PROGRESS = 5

class BACnetBackupState(enum.IntEnum):
    IDLE = 0
    PREPARING_FOR_BACKUP = 1
    PREPARING_FOR_RESTORE = 2
    PERFORMING_A_BACKUP = 3
    PERFORMING_A_RESTORE = 4

class BACnetRestartReason(enum.IntEnum):
    UNKNOWN = 0
    COLD_START = 1
    WARM_START = 2
    DETECTED_POWER_LOST = 3
    DETECTED_POWER_OFF = 4
    HARDWARE_WATCHDOG = 5
    SOFTWARE_WATCHDOG = 6
    SUSPENDED = 7

class BACnetProgramChange(enum.IntEnum):
    UNKNOWN = 0
    COLD_START = 1
    WARM_START = 2
    DETECTED_POWER_LOST = 3
    DETECTED_POWER_OFF = 4
    HARDWARE_WATCHDOG = 5
    SOFTWARE_WATCHDOG = 6
    SUSPENDED = 7

class BACnetProgramState(enum.IntEnum):
    IDLE = 0
    LOADING = 1
    RUNNING = 2
    WAITING = 3
    HALTED = 4
    UNLOADING = 5

class BACnetReasonForHalt(enum.IntEnum):
    NORMAL = 0
    LOAD_FAILED = 1
    INTERNAL = 2
    PROGRAM = 3
    OTHER = 4

class BACnetResultFlags(enum.IntEnum):
    NONE = 0
    FIRST_ITEM = 1
    LAST_ITEM = 2
    MORE_ITEMS = 4

class BACnetRejectReasons(enum.IntEnum):
    REJECT_REASON_OTHER = 0,
    REJECT_REASON_BUFFER_OVERFLOW = 1,
    REJECT_REASON_INCONSISTENT_PARAMETERS = 2,
    REJECT_REASON_INVALID_PARAMETER_DATA_TYPE = 3,
    REJECT_REASON_INVALID_TAG = 4,
    REJECT_REASON_MISSING_REQUIRED_PARAMETER = 5,
    REJECT_REASON_PARAMETER_OUT_OF_RANGE = 6,
    REJECT_REASON_TOO_MANY_ARGUMENTS = 7,
    REJECT_REASON_UNDEFINED_ENUMERATION = 8,
    REJECT_REASON_UNRECOGNIZED_SERVICE = 9,
    '''
        Enumerated values 0-63 are reserved for definition by ASHRAE.
        Enumerated values 64-65535 may be used by others subject to
        the procedures and constraints described in Clause 23.
    '''
    MAX_BACNET_REJECT_REASON = 10,
    '''do the MAX here instead of outside of enum so that compilers will allocate adequate sized datatype for enum'''
    REJECT_REASON_PROPRIETARY_FIRST = 64,
    REJECT_REASON_PROPRIETARY_LAST = 65535

class BACnetErrorClasses(enum.IntEnum):
    ERROR_CLASS_DEVICE = 0,
    ERROR_CLASS_OBJECT = 1,
    ERROR_CLASS_PROPERTY = 2,
    ERROR_CLASS_RESOURCES = 3,
    ERROR_CLASS_SECURITY = 4,
    ERROR_CLASS_SERVICES = 5,
    ERROR_CLASS_VT = 6,
    ERROR_CLASS_COMMUNICATION = 7,
    '''/* Enumerated values 0-63 are reserved for definition by ASHRAE. */
        /* Enumerated values 64-65535 may be used by others subject to */
        /* the procedures and constraints described in Clause 23. */'''
    MAX_BACNET_ERROR_CLASS = 8,
    '''do the MAX here instead of outside of enum so that
           compilers will allocate adequate sized datatype for enum'''
    ERROR_CLASS_PROPRIETARY_FIRST = 64,
    ERROR_CLASS_PROPRIETARY_LAST = 65535

'''These are sorted in the order given in
       Clause 18. ERROR, REJECT AND ABORT CODES
       The Class and Code pairings are required
       to be used in accordance with Clause 18.'''

class BACnetErrorCodes(enum.IntEnum):

    '''valid for all classes'''
    ERROR_CODE_OTHER = 0,

    '''Error Class - Device'''
    ERROR_CODE_DEVICE_BUSY = 3,
    ERROR_CODE_CONFIGURATION_IN_PROGRESS = 2,
    ERROR_CODE_OPERATIONAL_PROBLEM = 25,

    '''Error Class - Object'''
    ERROR_CODE_DYNAMIC_CREATION_NOT_SUPPORTED = 4,
    ERROR_CODE_NO_OBJECTS_OF_SPECIFIED_TYPE = 17,
    ERROR_CODE_OBJECT_DELETION_NOT_PERMITTED = 23,
    ERROR_CODE_OBJECT_IDENTIFIER_ALREADY_EXISTS = 24,
    ERROR_CODE_READ_ACCESS_DENIED = 27,
    ERROR_CODE_UNKNOWN_OBJECT = 31,
    ERROR_CODE_UNSUPPORTED_OBJECT_TYPE = 36,

    '''Error Class - Property'''
    ERROR_CODE_CHARACTER_SET_NOT_SUPPORTED = 41,
    ERROR_CODE_DATATYPE_NOT_SUPPORTED = 47,
    ERROR_CODE_INCONSISTENT_SELECTION_CRITERION = 8,
    ERROR_CODE_INVALID_ARRAY_INDEX = 42,
    ERROR_CODE_INVALID_DATA_TYPE = 9,
    ERROR_CODE_NOT_COV_PROPERTY = 44,
    ERROR_CODE_OPTIONAL_FUNCTIONALITY_NOT_SUPPORTED = 45,
    ERROR_CODE_PROPERTY_IS_NOT_AN_ARRAY = 50,
    #ERROR_CODE_READ_ACCESS_DENIED = 27
    ERROR_CODE_UNKNOWN_PROPERTY = 32,
    ERROR_CODE_VALUE_OUT_OF_RANGE = 37,
    ERROR_CODE_WRITE_ACCESS_DENIED = 40,

    '''Error Class - Resources'''
    ERROR_CODE_NO_SPACE_FOR_OBJECT = 18,
    ERROR_CODE_NO_SPACE_TO_ADD_LIST_ELEMENT = 19,
    ERROR_CODE_NO_SPACE_TO_WRITE_PROPERTY = 20,

    '''Error Class - Security '''
    ERROR_CODE_AUTHENTICATION_FAILED = 1,
    #ERROR_CODE_CHARACTER_SET_NOT_SUPPORTED = 41
    ERROR_CODE_INCOMPATIBLE_SECURITY_LEVELS = 6,
    ERROR_CODE_INVALID_OPERATOR_NAME = 12,
    ERROR_CODE_KEY_GENERATION_ERROR = 15,
    ERROR_CODE_PASSWORD_FAILURE = 26,
    ERROR_CODE_SECURITY_NOT_SUPPORTED = 28,
    ERROR_CODE_TIMEOUT = 30,

    '''Error Class - Services '''
    #ERROR_CODE_CHARACTER_SET_NOT_SUPPORTED = 41,
    ERROR_CODE_COV_SUBSCRIPTION_FAILED = 43,
    ERROR_CODE_DUPLICATE_NAME = 48,
    ERROR_CODE_DUPLICATE_OBJECT_ID = 49,
    ERROR_CODE_FILE_ACCESS_DENIED = 5,
    ERROR_CODE_INCONSISTENT_PARAMETERS = 7,
    ERROR_CODE_INVALID_CONFIGURATION_DATA = 46,
    ERROR_CODE_INVALID_FILE_ACCESS_METHOD = 10,
    ERROR_CODE_INVALID_FILE_START_POSITION = 11,
    ERROR_CODE_INVALID_PARAMETER_DATA_TYPE = 13,
    ERROR_CODE_INVALID_TIME_STAMP = 14,
    ERROR_CODE_MISSING_REQUIRED_PARAMETER = 16,
    #ERROR_CODE_OPTIONAL_FUNCTIONALITY_NOT_SUPPORTED = 45,
    ERROR_CODE_PROPERTY_IS_NOT_A_LIST = 22,
    ERROR_CODE_SERVICE_REQUEST_DENIED = 29,

    '''Error Class - VT'''
    ERROR_CODE_UNKNOWN_VT_CLASS = 34,
    ERROR_CODE_UNKNOWN_VT_SESSION = 35,
    ERROR_CODE_NO_VT_SESSIONS_AVAILABLE = 21,
    ERROR_CODE_VT_SESSION_ALREADY_CLOSED = 38,
    ERROR_CODE_VT_SESSION_TERMINATION_FAILURE = 39,

    '''unused'''
    ERROR_CODE_RESERVED1 = 33,
    '''new error codes from new addenda'''
    ERROR_CODE_ABORT_BUFFER_OVERFLOW = 51,
    ERROR_CODE_ABORT_INVALID_APDU_IN_THIS_STATE = 52,
    ERROR_CODE_ABORT_PREEMPTED_BY_HIGHER_PRIORITY_TASK = 53,
    ERROR_CODE_ABORT_SEGMENTATION_NOT_SUPPORTED = 54,
    ERROR_CODE_ABORT_PROPRIETARY = 55,
    ERROR_CODE_ABORT_OTHER = 56,
    ERROR_CODE_INVALID_TAG = 57,
    ERROR_CODE_NETWORK_DOWN = 58,
    ERROR_CODE_REJECT_BUFFER_OVERFLOW = 59,
    ERROR_CODE_REJECT_INCONSISTENT_PARAMETERS = 60,
    ERROR_CODE_REJECT_INVALID_PARAMETER_DATA_TYPE = 61,
    ERROR_CODE_REJECT_INVALID_TAG = 62,
    ERROR_CODE_REJECT_MISSING_REQUIRED_PARAMETER = 63,
    ERROR_CODE_REJECT_PARAMETER_OUT_OF_RANGE = 64,
    ERROR_CODE_REJECT_TOO_MANY_ARGUMENTS = 65,
    ERROR_CODE_REJECT_UNDEFINED_ENUMERATION = 66,
    ERROR_CODE_REJECT_UNRECOGNIZED_SERVICE = 67,
    ERROR_CODE_REJECT_PROPRIETARY = 68,
    ERROR_CODE_REJECT_OTHER = 69,
    ERROR_CODE_UNKNOWN_DEVICE = 70,
    ERROR_CODE_UNKNOWN_ROUTE = 71,
    ERROR_CODE_VALUE_NOT_INITIALIZED = 72,
    ERROR_CODE_INVALID_EVENT_STATE = 73,
    ERROR_CODE_NO_ALARM_CONFIGURED = 74,
    ERROR_CODE_LOG_BUFFER_FULL = 75,
    ERROR_CODE_LOGGED_VALUE_PURGED = 76,
    ERROR_CODE_NO_PROPERTY_SPECIFIED = 77,
    ERROR_CODE_NOT_CONFIGURED_FOR_TRIGGERED_LOGGING = 78,
    ERROR_CODE_UNKNOWN_SUBSCRIPTION = 79,
    ERROR_CODE_PARAMETER_OUT_OF_RANGE = 80,
    ERROR_CODE_LIST_ELEMENT_NOT_FOUND = 81,
    ERROR_CODE_BUSY = 82,
    ERROR_CODE_COMMUNICATION_DISABLED = 83,
    ERROR_CODE_SUCCESS = 84,
    ERROR_CODE_ACCESS_DENIED = 85,
    ERROR_CODE_BAD_DESTINATION_ADDRESS = 86,
    ERROR_CODE_BAD_DESTINATION_DEVICE_ID = 87,
    ERROR_CODE_BAD_SIGNATURE = 88,
    ERROR_CODE_BAD_SOURCE_ADDRESS = 89,
    ERROR_CODE_BAD_TIMESTAMP = 90,
    ERROR_CODE_CANNOT_USE_KEY = 91,
    ERROR_CODE_CANNOT_VERIFY_MESSAGE_ID = 92,
    ERROR_CODE_CORRECT_KEY_REVISION = 93,
    ERROR_CODE_DESTINATION_DEVICE_ID_REQUIRED = 94,
    ERROR_CODE_DUPLICATE_MESSAGE = 95,
    ERROR_CODE_ENCRYPTION_NOT_CONFIGURED = 96,
    ERROR_CODE_ENCRYPTION_REQUIRED = 97,
    ERROR_CODE_INCORRECT_KEY = 98,
    ERROR_CODE_INVALID_KEY_DATA = 99,
    ERROR_CODE_KEY_UPDATE_IN_PROGRESS = 100,
    ERROR_CODE_MALFORMED_MESSAGE = 101,
    ERROR_CODE_NOT_KEY_SERVER = 102,
    ERROR_CODE_SECURITY_NOT_CONFIGURED = 103,
    ERROR_CODE_SOURCE_SECURITY_REQUIRED = 104,
    ERROR_CODE_TOO_MANY_KEYS = 105,
    ERROR_CODE_UNKNOWN_AUTHENTICATION_TYPE = 106,
    ERROR_CODE_UNKNOWN_KEY = 107,
    ERROR_CODE_UNKNOWN_KEY_REVISION = 108,
    ERROR_CODE_UNKNOWN_SOURCE_MESSAGE = 109,
    ERROR_CODE_NOT_ROUTER_TO_DNET = 110,
    ERROR_CODE_ROUTER_BUSY = 111,
    ERROR_CODE_UNKNOWN_NETWORK_MESSAGE = 112,
    ERROR_CODE_MESSAGE_TOO_LONG = 113,
    ERROR_CODE_SECURITY_ERROR = 114,
    ERROR_CODE_ADDRESSING_ERROR = 115,
    ERROR_CODE_WRITE_BDT_FAILED = 116,
    ERROR_CODE_READ_BDT_FAILED = 117,
    ERROR_CODE_REGISTER_FOREIGN_DEVICE_FAILED = 118,
    ERROR_CODE_READ_FDT_FAILED = 119,
    ERROR_CODE_DELETE_FDT_ENTRY_FAILED = 120,
    ERROR_CODE_DISTRIBUTE_BROADCAST_FAILED = 121,
    ERROR_CODE_UNKNOWN_FILE_SIZE = 122,
    ERROR_CODE_ABORT_APDU_TOO_LONG = 123,
    ERROR_CODE_ABORT_APPLICATION_EXCEEDED_REPLY_TIME = 124,
    ERROR_CODE_ABORT_OUT_OF_RESOURCES = 125,
    ERROR_CODE_ABORT_TSM_TIMEOUT = 126,
    ERROR_CODE_ABORT_WINDOW_SIZE_OUT_OF_RANGE = 127,
    ERROR_CODE_FILE_FULL = 128,
    ERROR_CODE_INCONSISTENT_CONFIGURATION = 129,
    ERROR_CODE_INCONSISTENT_OBJECT_TYPE = 130,
    ERROR_CODE_INTERNAL_ERROR = 131,
    ERROR_CODE_NOT_CONFIGURED = 132,
    ERROR_CODE_OUT_OF_MEMORY = 133,
    ERROR_CODE_VALUE_TOO_LONG = 134,
    ERROR_CODE_ABORT_INSUFFICIENT_SECURITY = 135,
    ERROR_CODE_ABORT_SECURITY_ERROR = 136,

    MAX_BACNET_ERROR_CODE = 137,

    '''Enumerated values 0-255 are reserved for definition by ASHRAE.
       Enumerated values 256-65535 may be used by others subject to
       the procedures and constraints described in Clause 23.
       do the max range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding'''
    ERROR_CODE_PROPRIETARY_FIRST = 256,
    ERROR_CODE_PROPRIETARY_LAST = 65535

class BACnetStatusFlags(enum.IntEnum):
    STATUS_FLAG_IN_ALARM = 1,
    STATUS_FLAG_FAULT = 2,
    STATUS_FLAG_OVERRIDDEN = 4,
    STATUS_FLAG_OUT_OF_SERVICE = 8,

class BACnetServicesSupported(enum.IntEnum):
    '''Alarm and Event Services'''
    SERVICE_SUPPORTED_ACKNOWLEDGE_ALARM = 0,
    SERVICE_SUPPORTED_CONFIRMED_COV_NOTIFICATION = 1,
    SERVICE_SUPPORTED_CONFIRMED_EVENT_NOTIFICATION = 2,
    SERVICE_SUPPORTED_GET_ALARM_SUMMARY = 3,
    SERVICE_SUPPORTED_GET_ENROLLMENT_SUMMARY = 4,
    SERVICE_SUPPORTED_GET_EVENT_INFORMATION = 39,
    SERVICE_SUPPORTED_SUBSCRIBE_COV = 5,
    SERVICE_SUPPORTED_SUBSCRIBE_COV_PROPERTY = 38,
    SERVICE_SUPPORTED_LIFE_SAFETY_OPERATION = 37,
    '''File Access Services '''
    SERVICE_SUPPORTED_ATOMIC_READ_FILE = 6,
    SERVICE_SUPPORTED_ATOMIC_WRITE_FILE = 7,
    '''Object Access Services'''
    SERVICE_SUPPORTED_ADD_LIST_ELEMENT = 8,
    SERVICE_SUPPORTED_REMOVE_LIST_ELEMENT = 9,
    SERVICE_SUPPORTED_CREATE_OBJECT = 10,
    SERVICE_SUPPORTED_DELETE_OBJECT = 11,
    SERVICE_SUPPORTED_READ_PROPERTY = 12,
    SERVICE_SUPPORTED_READ_PROP_CONDITIONAL = 13,
    SERVICE_SUPPORTED_READ_PROP_MULTIPLE = 14,
    SERVICE_SUPPORTED_READ_RANGE = 35,
    SERVICE_SUPPORTED_WRITE_PROPERTY = 15,
    SERVICE_SUPPORTED_WRITE_PROP_MULTIPLE = 16,
    SERVICE_SUPPORTED_WRITE_GROUP = 40,
    '''Remote Device Management Services'''
    SERVICE_SUPPORTED_DEVICE_COMMUNICATION_CONTROL = 17,
    SERVICE_SUPPORTED_PRIVATE_TRANSFER = 18,
    SERVICE_SUPPORTED_TEXT_MESSAGE = 19,
    SERVICE_SUPPORTED_REINITIALIZE_DEVICE = 20,
    '''Virtual Terminal Services'''
    SERVICE_SUPPORTED_VT_OPEN = 21,
    SERVICE_SUPPORTED_VT_CLOSE = 22,
    SERVICE_SUPPORTED_VT_DATA = 23,
    '''Security Services'''
    SERVICE_SUPPORTED_AUTHENTICATE = 24,
    SERVICE_SUPPORTED_REQUEST_KEY = 25,
    SERVICE_SUPPORTED_I_AM = 26,
    SERVICE_SUPPORTED_I_HAVE = 27,
    SERVICE_SUPPORTED_UNCONFIRMED_COV_NOTIFICATION = 28,
    SERVICE_SUPPORTED_UNCONFIRMED_EVENT_NOTIFICATION = 29,
    SERVICE_SUPPORTED_UNCONFIRMED_PRIVATE_TRANSFER = 30,
    SERVICE_SUPPORTED_UNCONFIRMED_TEXT_MESSAGE = 31,
    SERVICE_SUPPORTED_TIME_SYNCHRONIZATION = 32,
    SERVICE_SUPPORTED_UTC_TIME_SYNCHRONIZATION = 36,
    SERVICE_SUPPORTED_WHO_HAS = 33,
    SERVICE_SUPPORTED_WHO_IS = 34,
    '''Other services to be added as they are defined. */
        /* All values in this production are reserved */
        /* for definition by ASHRAE.'''
    MAX_BACNET_SERVICES_SUPPORTED = 41,

class BACnetUnconfirmedServices(enum.IntEnum):
    SERVICE_UNCONFIRMED_I_AM = 0,
    SERVICE_UNCONFIRMED_I_HAVE = 1,
    SERVICE_UNCONFIRMED_COV_NOTIFICATION = 2,
    SERVICE_UNCONFIRMED_EVENT_NOTIFICATION = 3,
    SERVICE_UNCONFIRMED_PRIVATE_TRANSFER = 4,
    SERVICE_UNCONFIRMED_TEXT_MESSAGE = 5,
    SERVICE_UNCONFIRMED_TIME_SYNCHRONIZATION = 6,
    SERVICE_UNCONFIRMED_WHO_HAS = 7,
    SERVICE_UNCONFIRMED_WHO_IS = 8,
    SERVICE_UNCONFIRMED_UTC_TIME_SYNCHRONIZATION = 9,
    '''addendum 2010 - aa'''
    SERVICE_UNCONFIRMED_WRITE_GROUP = 10,
    '''Other services to be added as they are defined. */
        /* All choice values in this production are reserved */
        /* for definition by ASHRAE. */
        /* Proprietary extensions are made by using the */
        /* UnconfirmedPrivateTransfer service. See Clause 23.'''
    MAX_BACNET_UNCONFIRMED_SERVICE = 11,

class BACnetConfirmedServices(enum.IntEnum):
    '''Alarm and Event Services '''

    SERVICE_CONFIRMED_ACKNOWLEDGE_ALARM = 0,
    SERVICE_CONFIRMED_COV_NOTIFICATION = 1,
    SERVICE_CONFIRMED_EVENT_NOTIFICATION = 2,
    SERVICE_CONFIRMED_GET_ALARM_SUMMARY = 3,
    SERVICE_CONFIRMED_GET_ENROLLMENT_SUMMARY = 4,
    SERVICE_CONFIRMED_GET_EVENT_INFORMATION = 29,
    SERVICE_CONFIRMED_SUBSCRIBE_COV = 5,
    SERVICE_CONFIRMED_SUBSCRIBE_COV_PROPERTY = 28,
    SERVICE_CONFIRMED_LIFE_SAFETY_OPERATION = 27,
    '''File Access Services '''
    SERVICE_CONFIRMED_ATOMIC_READ_FILE = 6,
    SERVICE_CONFIRMED_ATOMIC_WRITE_FILE = 7,
    '''Object Access Services '''
    SERVICE_CONFIRMED_ADD_LIST_ELEMENT = 8,
    SERVICE_CONFIRMED_REMOVE_LIST_ELEMENT = 9,
    SERVICE_CONFIRMED_CREATE_OBJECT = 10,
    SERVICE_CONFIRMED_DELETE_OBJECT = 11,
    SERVICE_CONFIRMED_READ_PROPERTY = 12,
    SERVICE_CONFIRMED_READ_PROP_CONDITIONAL = 13,
    SERVICE_CONFIRMED_READ_PROP_MULTIPLE = 14,
    SERVICE_CONFIRMED_READ_RANGE = 26,
    SERVICE_CONFIRMED_WRITE_PROPERTY = 15,
    SERVICE_CONFIRMED_WRITE_PROP_MULTIPLE = 16,
    '''Remote Device Management Services '''
    SERVICE_CONFIRMED_DEVICE_COMMUNICATION_CONTROL = 17,
    SERVICE_CONFIRMED_PRIVATE_TRANSFER = 18,
    SERVICE_CONFIRMED_TEXT_MESSAGE = 19,
    SERVICE_CONFIRMED_REINITIALIZE_DEVICE = 20,
    '''Virtual Terminal Services'''
    SERVICE_CONFIRMED_VT_OPEN = 21,
    SERVICE_CONFIRMED_VT_CLOSE = 22,
    SERVICE_CONFIRMED_VT_DATA = 23,
    '''Security Services '''
    SERVICE_CONFIRMED_AUTHENTICATE = 24,
    SERVICE_CONFIRMED_REQUEST_KEY = 25,
    '''Services added after 1995 */
        /* readRange (26) see Object Access Services */
        /* lifeSafetyOperation (27) see Alarm and Event Services */
        /* subscribeCOVProperty (28) see Alarm and Event Services */
        /* getEventInformation (29) see Alarm and Event Services'''
    MAX_BACNET_CONFIRMED_SERVICE = 30

class BACnetUnitsId(enum.IntEnum):
    UNITS_METERS_PER_SECOND_PER_SECOND = 166,
    UNITS_SQUARE_METERS = 0,
    UNITS_SQUARE_CENTIMETERS = 116,
    UNITS_SQUARE_FEET = 1,
    UNITS_SQUARE_INCHES = 115,
    UNITS_CURRENCY1 = 105,
    UNITS_CURRENCY2 = 106,
    UNITS_CURRENCY3 = 107,
    UNITS_CURRENCY4 = 108,
    UNITS_CURRENCY5 = 109,
    UNITS_CURRENCY6 = 110,
    UNITS_CURRENCY7 = 111,
    UNITS_CURRENCY8 = 112,
    UNITS_CURRENCY9 = 113,
    UNITS_CURRENCY10 = 114,
    UNITS_MILLIAMPERES = 2,
    UNITS_AMPERES = 3,
    UNITS_AMPERES_PER_METER = 167,
    UNITS_AMPERES_PER_SQUARE_METER = 168,
    UNITS_AMPERE_SQUARE_METERS = 169,
    UNITS_DECIBELS = 199,
    UNITS_DECIBELS_MILLIVOLT = 200,
    UNITS_DECIBELS_VOLT = 201,
    UNITS_FARADS = 170,
    UNITS_HENRYS = 171,
    UNITS_OHMS = 4,
    UNITS_OHM_METERS = 172,
    UNITS_MILLIOHMS = 145,
    UNITS_KILOHMS = 122,
    UNITS_MEGOHMS = 123,
    UNITS_MICROSIEMENS = 190,
    UNITS_MILLISIEMENS = 202,
    UNITS_SIEMENS = 173,
    UNITS_SIEMENS_PER_METER = 174,
    UNITS_TESLAS = 175,
    UNITS_VOLTS = 5,
    UNITS_MILLIVOLTS = 124,
    UNITS_KILOVOLTS = 6,
    UNITS_MEGAVOLTS = 7,
    UNITS_VOLT_AMPERES = 8,
    UNITS_KILOVOLT_AMPERES = 9,
    UNITS_MEGAVOLT_AMPERES = 10,
    UNITS_VOLT_AMPERES_REACTIVE = 11,
    UNITS_KILOVOLT_AMPERES_REACTIVE = 12,
    UNITS_MEGAVOLT_AMPERES_REACTIVE = 13,
    UNITS_VOLTS_PER_DEGREE_KELVIN = 176,
    UNITS_VOLTS_PER_METER = 177,
    UNITS_DEGREES_PHASE = 14,
    UNITS_POWER_FACTOR = 15,
    UNITS_WEBERS = 178,
    UNITS_JOULES = 16,
    UNITS_KILOJOULES = 17,
    UNITS_KILOJOULES_PER_KILOGRAM = 125,
    UNITS_MEGAJOULES = 126,
    UNITS_WATT_HOURS = 18,
    UNITS_KILOWATT_HOURS = 19,
    UNITS_MEGAWATT_HOURS = 146,
    UNITS_WATT_HOURS_REACTIVE = 203,
    UNITS_KILOWATT_HOURS_REACTIVE = 204,
    UNITS_MEGAWATT_HOURS_REACTIVE = 205,
    UNITS_BTUS = 20,
    UNITS_KILO_BTUS = 147,
    UNITS_MEGA_BTUS = 148,
    UNITS_THERMS = 21,
    UNITS_TON_HOURS = 22,
    UNITS_JOULES_PER_KILOGRAM_DRY_AIR = 23,
    UNITS_KILOJOULES_PER_KILOGRAM_DRY_AIR = 149,
    UNITS_MEGAJOULES_PER_KILOGRAM_DRY_AIR = 150,
    UNITS_BTUS_PER_POUND_DRY_AIR = 24,
    UNITS_BTUS_PER_POUND = 117,
    UNITS_JOULES_PER_DEGREE_KELVIN = 127,
    UNITS_KILOJOULES_PER_DEGREE_KELVIN = 151,
    UNITS_MEGAJOULES_PER_DEGREE_KELVIN = 152,
    UNITS_JOULES_PER_KILOGRAM_DEGREE_KELVIN = 128,
    UNITS_NEWTON = 153,
    UNITS_CYCLES_PER_HOUR = 25,
    UNITS_CYCLES_PER_MINUTE = 26,
    UNITS_HERTZ = 27,
    UNITS_KILOHERTZ = 129,
    UNITS_MEGAHERTZ = 130,
    UNITS_PER_HOUR = 131,
    UNITS_GRAMS_OF_WATER_PER_KILOGRAM_DRY_AIR = 28,
    UNITS_PERCENT_RELATIVE_HUMIDITY = 29,
    UNITS_MICROMETERS = 194,
    UNITS_MILLIMETERS = 30,
    UNITS_CENTIMETERS = 118,
    UNITS_KILOMETERS = 193,
    UNITS_METERS = 31,
    UNITS_INCHES = 32,
    UNITS_FEET = 33,
    UNITS_CANDELAS = 179,
    UNITS_CANDELAS_PER_SQUARE_METER = 180,
    UNITS_WATTS_PER_SQUARE_FOOT = 34,
    UNITS_WATTS_PER_SQUARE_METER = 35,
    UNITS_LUMENS = 36,
    UNITS_LUXES = 37,
    UNITS_FOOT_CANDLES = 38,
    UNITS_MILLIGRAMS = 196,
    UNITS_GRAMS = 195,
    UNITS_KILOGRAMS = 39,
    UNITS_POUNDS_MASS = 40,
    UNITS_TONS = 41,
    UNITS_GRAMS_PER_SECOND = 154,
    UNITS_GRAMS_PER_MINUTE = 155,
    UNITS_KILOGRAMS_PER_SECOND = 42,
    UNITS_KILOGRAMS_PER_MINUTE = 43,
    UNITS_KILOGRAMS_PER_HOUR = 44,
    UNITS_POUNDS_MASS_PER_SECOND = 119,
    UNITS_POUNDS_MASS_PER_MINUTE = 45,
    UNITS_POUNDS_MASS_PER_HOUR = 46,
    UNITS_TONS_PER_HOUR = 156,
    UNITS_MILLIWATTS = 132,
    UNITS_WATTS = 47,
    UNITS_KILOWATTS = 48,
    UNITS_MEGAWATTS = 49,
    UNITS_BTUS_PER_HOUR = 50,
    UNITS_KILO_BTUS_PER_HOUR = 157,
    UNITS_HORSEPOWER = 51,
    UNITS_TONS_REFRIGERATION = 52,
    UNITS_PASCALS = 53,
    UNITS_HECTOPASCALS = 133,
    UNITS_KILOPASCALS = 54,
    UNITS_MILLIBARS = 134,
    UNITS_BARS = 55,
    UNITS_POUNDS_FORCE_PER_SQUARE_INCH = 56,
    UNITS_MILLIMETERS_OF_WATER = 206,
    UNITS_CENTIMETERS_OF_WATER = 57,
    UNITS_INCHES_OF_WATER = 58,
    UNITS_MILLIMETERS_OF_MERCURY = 59,
    UNITS_CENTIMETERS_OF_MERCURY = 60,
    UNITS_INCHES_OF_MERCURY = 61,
    UNITS_DEGREES_CELSIUS = 62,
    UNITS_DEGREES_KELVIN = 63,
    UNITS_DEGREES_KELVIN_PER_HOUR = 181,
    UNITS_DEGREES_KELVIN_PER_MINUTE = 182,
    UNITS_DEGREES_FAHRENHEIT = 64,
    UNITS_DEGREE_DAYS_CELSIUS = 65,
    UNITS_DEGREE_DAYS_FAHRENHEIT = 66,
    UNITS_DELTA_DEGREES_FAHRENHEIT = 120,
    UNITS_DELTA_DEGREES_KELVIN = 121,
    UNITS_YEARS = 67,
    UNITS_MONTHS = 68,
    UNITS_WEEKS = 69,
    UNITS_DAYS = 70,
    UNITS_HOURS = 71,
    UNITS_MINUTES = 72,
    UNITS_SECONDS = 73,
    UNITS_HUNDREDTHS_SECONDS = 158,
    UNITS_MILLISECONDS = 159,
    UNITS_NEWTON_METERS = 160,
    UNITS_MILLIMETERS_PER_SECOND = 161,
    UNITS_MILLIMETERS_PER_MINUTE = 162,
    UNITS_METERS_PER_SECOND = 74,
    UNITS_METERS_PER_MINUTE = 163,
    UNITS_METERS_PER_HOUR = 164,
    UNITS_KILOMETERS_PER_HOUR = 75,
    UNITS_FEET_PER_SECOND = 76,
    UNITS_FEET_PER_MINUTE = 77,
    UNITS_MILES_PER_HOUR = 78,
    UNITS_CUBIC_FEET = 79,
    UNITS_CUBIC_METERS = 80,
    UNITS_IMPERIAL_GALLONS = 81,
    UNITS_MILLILITERS = 197,
    UNITS_LITERS = 82,
    UNITS_US_GALLONS = 83,
    UNITS_CUBIC_FEET_PER_SECOND = 142,
    UNITS_CUBIC_FEET_PER_MINUTE = 84,
    UNITS_MILLION_CUBIC_FEET_PER_MINUTE = 254,
    UNITS_CUBIC_FEET_PER_HOUR = 191,
    UNITS_STANDARD_CUBIC_FEET_PER_DAY = 47808,
    UNITS_MILLION_STANDARD_CUBIC_FEET_PER_DAY = 47809,
    UNITS_THOUSAND_CUBIC_FEET_PER_DAY = 47810,
    UNITS_THOUSAND_STANDARD_CUBIC_FEET_PER_DAY = 47811,
    UINITS_POUNDS_MASS_PER_DAY = 47812,
    UNITS_CUBIC_METERS_PER_SECOND = 85,
    UNITS_CUBIC_METERS_PER_MINUTE = 165,
    UNITS_CUBIC_METERS_PER_HOUR = 135,
    UNITS_IMPERIAL_GALLONS_PER_MINUTE = 86,
    UNITS_MILLILITERS_PER_SECOND = 198,
    UNITS_LITERS_PER_SECOND = 87,
    UNITS_LITERS_PER_MINUTE = 88,
    UNITS_LITERS_PER_HOUR = 136,
    UNITS_US_GALLONS_PER_MINUTE = 89,
    UNITS_US_GALLONS_PER_HOUR = 192,
    UNITS_DEGREES_ANGULAR = 90,
    UNITS_DEGREES_CELSIUS_PER_HOUR = 91,
    UNITS_DEGREES_CELSIUS_PER_MINUTE = 92,
    UNITS_DEGREES_FAHRENHEIT_PER_HOUR = 93,
    UNITS_DEGREES_FAHRENHEIT_PER_MINUTE = 94,
    UNITS_JOULE_SECONDS = 183,
    UNITS_KILOGRAMS_PER_CUBIC_METER = 186,
    UNITS_KW_HOURS_PER_SQUARE_METER = 137,
    UNITS_KW_HOURS_PER_SQUARE_FOOT = 138,
    UNITS_MEGAJOULES_PER_SQUARE_METER = 139,
    UNITS_MEGAJOULES_PER_SQUARE_FOOT = 140,
    UNITS_NO_UNITS = 95,
    UNITS_NEWTON_SECONDS = 187,
    UNITS_NEWTONS_PER_METER = 188,
    UNITS_PARTS_PER_MILLION = 96,
    UNITS_PARTS_PER_BILLION = 97,
    UNITS_PERCENT = 98,
    UNITS_PERCENT_OBSCURATION_PER_FOOT = 143,
    UNITS_PERCENT_OBSCURATION_PER_METER = 144,
    UNITS_PERCENT_PER_SECOND = 99,
    UNITS_PER_MINUTE = 100,
    UNITS_PER_SECOND = 101,
    UNITS_PSI_PER_DEGREE_FAHRENHEIT = 102,
    UNITS_RADIANS = 103,
    UNITS_RADIANS_PER_SECOND = 184,
    UNITS_REVOLUTIONS_PER_MINUTE = 104,
    UNITS_SQUARE_METERS_PER_NEWTON = 185,
    UNITS_WATTS_PER_METER_PER_DEGREE_KELVIN = 189,
    UNITS_WATTS_PER_SQUARE_METER_DEGREE_KELVIN = 141,
    UNITS_PER_MILLE = 207,
    UNITS_GRAMS_PER_GRAM = 208,
    UNITS_KILOGRAMS_PER_KILOGRAM = 209,
    UNITS_GRAMS_PER_KILOGRAM = 210,
    UNITS_MILLIGRAMS_PER_GRAM = 211,
    UNITS_MILLIGRAMS_PER_KILOGRAM = 212,
    UNITS_GRAMS_PER_MILLILITER = 213,
    UNITS_GRAMS_PER_LITER = 214,
    UNITS_MILLIGRAMS_PER_LITER = 215,
    UNITS_MICROGRAMS_PER_LITER = 216,
    UNITS_GRAMS_PER_CUBIC_METER = 217,
    UNITS_MILLIGRAMS_PER_CUBIC_METER = 218,
    UNITS_MICROGRAMS_PER_CUBIC_METER = 219,
    UNITS_NANOGRAMS_PER_CUBIC_METER = 220,
    UNITS_GRAMS_PER_CUBIC_CENTIMETER = 221,
    UNITS_BECQUERELS = 222,
    UNITS_KILOBECQUERELS = 223,
    UNITS_MEGABECQUERELS = 224,
    UNITS_GRAY = 225,
    UNITS_MILLIGRAY = 226,
    UNITS_MICROGRAY = 227,
    UNITS_SIEVERTS = 228,
    UNITS_MILLISIEVERTS = 229,
    UNITS_MICROSIEVERTS = 230,
    UNITS_MICROSIEVERTS_PER_HOUR = 231,
    UNITS_MILLIREMS = 47814,
    UNITS_MILLIREMS_PER_HOUR = 47815,
    UNITS_DECIBELS_A = 232,
    UNITS_NEPHELOMETRIC_TURBIDITY_UNIT = 233,
    UNITS_PH = 234,
    UNITS_GRAMS_PER_SQUARE_METER = 235,
    UNITS_MINUTES_PER_DEGREE_KELVIN = 236,
    UNITS_METER_SQUARED_PER_METER = 237,
    UNITS_AMPERE_SECONDS = 238,
    UNITS_VOLT_AMPERE_HOURS = 239,
    UNITS_KILOVOLT_AMPERE_HOURS = 240,
    UNITS_MEGAVOLT_AMPERE_HOURS = 241,
    UNITS_VOLT_AMPERE_HOURS_REACTIVE = 242,
    UNITS_KILOVOLT_AMPERE_HOURS_REACTIVE = 243,
    UNITS_MEGAVOLT_AMPERE_HOURS_REACTIVE = 244,
    UNITS_VOLT_SQUARE_HOURS = 245,
    UNITS_AMPERE_SQUARE_HOURS = 246,
    UNITS_JOULE_PER_HOURS = 247,
    UNITS_CUBIC_FEET_PER_DAY = 248,
    UNITS_CUBIC_METERS_PER_DAY = 249,
    UNITS_WATT_HOURS_PER_CUBIC_METER = 250,
    UNITS_JOULES_PER_CUBIC_METER = 251,
    UNITS_MOLE_PERCENT = 252,
    UNITS_PASCAL_SECONDS = 253,
    UNITS_MILLION_STANDARD_CUBIC_FEET_PER_MINUTE = 254

class BACnetPolarity(enum.IntEnum):
    POLARITY_NORMAL = 0,
    POLARITY_REVERSE = 1

class BACnetReliability(enum.IntEnum):
    RELIABILITY_NO_FAULT_DETECTED = 0,
    RELIABILITY_NO_SENSOR = 1,
    RELIABILITY_OVER_RANGE = 2,
    RELIABILITY_UNDER_RANGE = 3,
    RELIABILITY_OPEN_LOOP = 4,
    RELIABILITY_SHORTED_LOOP = 5,
    RELIABILITY_NO_OUTPUT = 6,
    RELIABILITY_UNRELIABLE_OTHER = 7,
    RELIABILITY_PROCESS_ERROR = 8,
    RELIABILITY_MULTI_STATE_FAULT = 9,
    RELIABILITY_CONFIGURATION_ERROR = 10,
    RELIABILITY_MEMBER_FAULT = 11,
    RELIABILITY_COMMUNICATION_FAILURE = 12,
    RELIABILITY_TRIPPED = 13,

class BACnetMaxSegments(enum.IntEnum):
    MAX_SEG0 = 0,
    MAX_SEG2 = 0x10,
    MAX_SEG4 = 0x20,
    MAX_SEG8 = 0x30,
    MAX_SEG16 = 0x40,
    MAX_SEG32 = 0x50,
    MAX_SEG64 = 0x60,
    MAX_SEG65 = 0x70,

class BACnetMaxAdpu(enum.IntEnum):
    MAX_APDU50 = 0,
    MAX_APDU128 = 1,
    MAX_APDU206 = 2,
    MAX_APDU480 = 3,
    MAX_APDU1024 = 4,
    MAX_APDU1476 = 5,

class BACnetObjectTypes(enum.IntEnum):
    OBJECT_ANALOG_INPUT = 0,
    OBJECT_ANALOG_OUTPUT = 1,
    OBJECT_ANALOG_VALUE = 2,
    OBJECT_BINARY_INPUT = 3,
    OBJECT_BINARY_OUTPUT = 4,
    OBJECT_BINARY_VALUE = 5,
    OBJECT_CALENDAR = 6,
    OBJECT_COMMAND = 7,
    OBJECT_DEVICE = 8,
    OBJECT_EVENT_ENROLLMENT = 9,
    OBJECT_FILE = 10,
    OBJECT_GROUP = 11,
    OBJECT_LOOP = 12,
    OBJECT_MULTI_STATE_INPUT = 13,
    OBJECT_MULTI_STATE_OUTPUT = 14,
    OBJECT_NOTIFICATION_CLASS = 15,
    OBJECT_PROGRAM = 16,
    OBJECT_SCHEDULE = 17,
    OBJECT_AVERAGING = 18,
    OBJECT_MULTI_STATE_VALUE = 19,
    OBJECT_TRENDLOG = 20,
    OBJECT_LIFE_SAFETY_POINT = 21,
    OBJECT_LIFE_SAFETY_ZONE = 22,
    OBJECT_ACCUMULATOR = 23,
    OBJECT_PULSE_CONVERTER = 24,
    OBJECT_EVENT_LOG = 25,
    OBJECT_GLOBAL_GROUP = 26,
    OBJECT_TREND_LOG_MULTIPLE = 27,
    OBJECT_LOAD_CONTROL = 28,
    OBJECT_STRUCTURED_VIEW = 29,
    OBJECT_ACCESS_DOOR = 30,
    OBJECT_TIMER = 31,
    OBJECT_ACCESS_CREDENTIAL = 32,
    OBJECT_ACCESS_POINT = 33,
    OBJECT_ACCESS_RIGHTS = 34,
    OBJECT_ACCESS_USER = 35,
    OBJECT_ACCESS_ZONE = 36,
    OBJECT_CREDENTIAL_DATA_INPUT = 37,
    OBJECT_NETWORK_SECURITY = 38,
    OBJECT_BITSTRING_VALUE = 39,
    OBJECT_CHARACTERSTRING_VALUE = 40,
    OBJECT_DATE_PATTERN_VALUE = 41,
    OBJECT_DATE_VALUE = 42,
    OBJECT_DATETIME_PATTERN_VALUE = 43,
    OBJECT_DATETIME_VALUE = 44,
    OBJECT_INTEGER_VALUE = 45,
    OBJECT_LARGE_ANALOG_VALUE = 46,
    OBJECT_OCTETSTRING_VALUE = 47,
    OBJECT_POSITIVE_INTEGER_VALUE = 48,
    OBJECT_TIME_PATTERN_VALUE = 49,
    OBJECT_TIME_VALUE = 50,
    OBJECT_NOTIFICATION_FORWARDER = 51,
    OBJECT_ALERT_ENROLLMENT = 52,
    OBJECT_CHANNEL = 53,
    OBJECT_LIGHTING_OUTPUT = 54,
    OBJECT_BINARY_LIGHTING_OUTPUT = 55,
    OBJECT_PROPRIETARY_MIN = 128,
    OBJECT_PROPRIETARY_MAX = 1023,
    MAX_BACNET_OBJECT_TYPE = 1024,
    MAX_ASHRAE_OBJECT_TYPE = 56,

class BACnetApplicationTags(enum.IntEnum):
    BACNET_APPLICATION_TAG_NULL = 0,
    BACNET_APPLICATION_TAG_BOOLEAN = 1,
    BACNET_APPLICATION_TAG_UNSIGNED_INT = 2,
    BACNET_APPLICATION_TAG_SIGNED_INT = 3,
    BACNET_APPLICATION_TAG_REAL = 4,
    BACNET_APPLICATION_TAG_DOUBLE = 5,
    BACNET_APPLICATION_TAG_OCTET_STRING = 6,
    BACNET_APPLICATION_TAG_CHARACTER_STRING = 7,
    BACNET_APPLICATION_TAG_BIT_STRING = 8,
    BACNET_APPLICATION_TAG_ENUMERATED = 9,
    BACNET_APPLICATION_TAG_DATE = 10,
    BACNET_APPLICATION_TAG_TIME = 11,
    BACNET_APPLICATION_TAG_OBJECT_ID = 12,
    BACNET_APPLICATION_TAG_RESERVE1 = 13,
    BACNET_APPLICATION_TAG_RESERVE2 = 14,
    BACNET_APPLICATION_TAG_RESERVE3 = 15,
    MAX_BACNET_APPLICATION_TAG = 16,
    BACNET_APPLICATION_TAG_EMPTYLIST = 17,
    BACNET_APPLICATION_TAG_WEEKNDAY = 18,
    BACNET_APPLICATION_TAG_DATERANGE = 19,
    BACNET_APPLICATION_TAG_DATETIME = 20,
    BACNET_APPLICATION_TAG_TIMESTAMP = 21,
    BACNET_APPLICATION_TAG_ERROR = 22,
    BACNET_APPLICATION_TAG_DEVICE_OBJECT_PROPERTY_REFERENCE = 23,
    BACNET_APPLICATION_TAG_DEVICE_OBJECT_REFERENCE = 24,
    BACNET_APPLICATION_TAG_OBJECT_PROPERTY_REFERENCE = 25,
    BACNET_APPLICATION_TAG_DESTINATION = 26,
    BACNET_APPLICATION_TAG_RECIPIENT = 27,
    BACNET_APPLICATION_TAG_COV_SUBSCRIPTION = 28,
    BACNET_APPLICATION_TAG_CALENDAR_ENTRY = 29,
    BACNET_APPLICATION_TAG_WEEKLY_SCHEDULE = 30,
    BACNET_APPLICATION_TAG_SPECIAL_EVENT = 31,
    BACNET_APPLICATION_TAG_READ_ACCESS_SPECIFICATION = 32,
    BACNET_APPLICATION_TAG_READ_ACCESS_RESULT = 33,
    BACNET_APPLICATION_TAG_LIGHTING_COMMAND = 34,
    BACNET_APPLICATION_TAG_CONTEXT_SPECIFIC_DECODED = 35,
    BACNET_APPLICATION_TAG_CONTEXT_SPECIFIC_ENCODED = 36,
    BACNET_APPLICATION_TAG_LOG_RECORD = 37,

class BACnetWritePriority(enum.IntEnum):
    NO_PRIORITY = 0,
    MANUAL_LIFE_SAFETY = 1,
    AUTOMATIC_LIFE_SAFETY = 2,
    UNSPECIFIED_LEVEL_3 = 3,
    UNSPECIFIED_LEVEL_4 = 4,
    CRITICAL_EQUIPMENT_CONTROL = 5,
    MINIMUM_ON_OFF = 6,
    UNSPECIFIED_LEVEL_7 = 7,
    MANUAL_OPERATOR = 8,
    UNSPECIFIED_LEVEL_9 = 9,
    UNSPECIFIED_LEVEL_10 = 10,
    UNSPECIFIED_LEVEL_11 = 11,
    UNSPECIFIED_LEVEL_12 = 12,
    UNSPECIFIED_LEVEL_13 = 13,
    UNSPECIFIED_LEVEL_14 = 14,
    UNSPECIFIED_LEVEL_15 = 15,
    LOWEST_AND_DEFAULT = 16

class BACnetFileAccessMethod(enum.IntEnum):
    RECORD_ACCESS = 0,
    STREAM_ACCESS = 1

class BACnetCharacterStringEncodings(enum.IntEnum):
    CHARACTER_ANSI_X34 = 0,
    CHARACTER_UTF8 = 0,
    CHARACTER_MS_DBCS = 1,
    CHARACTER_JISC_6226 = 2,
    CHARACTER_JISX_0208 = 2,
    CHARACTER_UCS4 = 3,
    CHARACTER_UCS2 = 4,
    CHARACTER_ISO8859 = 5,

class BACnetPropertyIds(enum.IntEnum):
    PROP_ACKED_TRANSITIONS = 0,
    PROP_ACK_REQUIRED = 1,
    PROP_ACTION = 2,
    PROP_ACTION_TEXT = 3,
    PROP_ACTIVE_TEXT = 4,
    PROP_ACTIVE_VT_SESSIONS = 5,
    PROP_ALARM_VALUE = 6,
    PROP_ALARM_VALUES = 7,
    PROP_ALL = 8,
    PROP_ALL_WRITES_SUCCESSFUL = 9,
    PROP_APDU_SEGMENT_TIMEOUT = 10,
    PROP_APDU_TIMEOUT = 11,
    PROP_APPLICATION_SOFTWARE_VERSION = 12,
    PROP_ARCHIVE = 13,
    PROP_BIAS = 14,
    PROP_CHANGE_OF_STATE_COUNT = 15,
    PROP_CHANGE_OF_STATE_TIME = 16,
    PROP_NOTIFICATION_CLASS = 17,
    PROP_BLANK_1 = 18,
    PROP_CONTROLLED_VARIABLE_REFERENCE = 19,
    PROP_CONTROLLED_VARIABLE_UNITS = 20,
    PROP_CONTROLLED_VARIABLE_VALUE = 21,
    PROP_COV_INCREMENT = 22,
    PROP_DATE_LIST = 23,
    PROP_DAYLIGHT_SAVINGS_STATUS = 24,
    PROP_DEADBAND = 25,
    PROP_DERIVATIVE_CONSTANT = 26,
    PROP_DERIVATIVE_CONSTANT_UNITS = 27,
    PROP_DESCRIPTION = 28,
    PROP_DESCRIPTION_OF_HALT = 29,
    PROP_DEVICE_ADDRESS_BINDING = 30,
    PROP_DEVICE_TYPE = 31,
    PROP_EFFECTIVE_PERIOD = 32,
    PROP_ELAPSED_ACTIVE_TIME = 33,
    PROP_ERROR_LIMIT = 34,
    PROP_EVENT_ENABLE = 35,
    PROP_EVENT_STATE = 36,
    PROP_EVENT_TYPE = 37,
    PROP_EXCEPTION_SCHEDULE = 38,
    PROP_FAULT_VALUES = 39,
    PROP_FEEDBACK_VALUE = 40,
    PROP_FILE_ACCESS_METHOD = 41,
    PROP_FILE_SIZE = 42,
    PROP_FILE_TYPE = 43,
    PROP_FIRMWARE_REVISION = 44,
    PROP_HIGH_LIMIT = 45,
    PROP_INACTIVE_TEXT = 46,
    PROP_IN_PROCESS = 47,
    PROP_INSTANCE_OF = 48,
    PROP_INTEGRAL_CONSTANT = 49,
    PROP_INTEGRAL_CONSTANT_UNITS = 50,
    PROP_ISSUE_CONFIRMED_NOTIFICATIONS = 51,
    PROP_LIMIT_ENABLE = 52,
    PROP_LIST_OF_GROUP_MEMBERS = 53,
    PROP_LIST_OF_OBJECT_PROPERTY_REFERENCES = 54,
    PROP_LIST_OF_SESSION_KEYS = 55,
    PROP_LOCAL_DATE = 56,
    PROP_LOCAL_TIME = 57,
    PROP_LOCATION = 58,
    PROP_LOW_LIMIT = 59,
    PROP_MANIPULATED_VARIABLE_REFERENCE = 60,
    PROP_MAXIMUM_OUTPUT = 61,
    PROP_MAX_APDU_LENGTH_ACCEPTED = 62,
    PROP_MAX_INFO_FRAMES = 63,
    PROP_MAX_MASTER = 64,
    PROP_MAX_PRES_VALUE = 65,
    PROP_MINIMUM_OFF_TIME = 66,
    PROP_MINIMUM_ON_TIME = 67,
    PROP_MINIMUM_OUTPUT = 68,
    PROP_MIN_PRES_VALUE = 69,
    PROP_MODEL_NAME = 70,
    PROP_MODIFICATION_DATE = 71,
    PROP_NOTIFY_TYPE = 72,
    PROP_NUMBER_OF_APDU_RETRIES = 73,
    PROP_NUMBER_OF_STATES = 74,
    PROP_OBJECT_IDENTIFIER = 75,
    PROP_OBJECT_LIST = 76,
    PROP_OBJECT_NAME = 77,
    PROP_OBJECT_PROPERTY_REFERENCE = 78,
    PROP_OBJECT_TYPE = 79,
    PROP_OPTIONAL = 80,
    PROP_OUT_OF_SERVICE = 81,
    PROP_OUTPUT_UNITS = 82,
    PROP_EVENT_PARAMETERS = 83,
    PROP_POLARITY = 84,
    PROP_PRESENT_VALUE = 85,
    PROP_PRIORITY = 86,
    PROP_PRIORITY_ARRAY = 87,
    PROP_PRIORITY_FOR_WRITING = 88,
    PROP_PROCESS_IDENTIFIER = 89,
    PROP_PROGRAM_CHANGE = 90,
    PROP_PROGRAM_LOCATION = 91,
    PROP_PROGRAM_STATE = 92,
    PROP_PROPORTIONAL_CONSTANT = 93,
    PROP_PROPORTIONAL_CONSTANT_UNITS = 94,
    PROP_PROTOCOL_CONFORMANCE_CLASS = 95,
    PROP_PROTOCOL_OBJECT_TYPES_SUPPORTED = 96,
    PROP_PROTOCOL_SERVICES_SUPPORTED = 97,
    PROP_PROTOCOL_VERSION = 98,
    PROP_READ_ONLY = 99,
    PROP_REASON_FOR_HALT = 100,
    PROP_RECIPIENT = 101,
    PROP_RECIPIENT_LIST = 102,
    PROP_RELIABILITY = 103,
    PROP_RELINQUISH_DEFAULT = 104,
    PROP_REQUIRED = 105,
    PROP_RESOLUTION = 106,
    PROP_SEGMENTATION_SUPPORTED = 107,
    PROP_SETPOINT = 108,
    PROP_SETPOINT_REFERENCE = 109,
    PROP_STATE_TEXT = 110,
    PROP_STATUS_FLAGS = 111,
    PROP_SYSTEM_STATUS = 112,
    PROP_TIME_DELAY = 113,
    PROP_TIME_OF_ACTIVE_TIME_RESET = 114,
    PROP_TIME_OF_STATE_COUNT_RESET = 115,
    PROP_TIME_SYNCHRONIZATION_RECIPIENTS = 116,
    PROP_UNITS = 117,
    PROP_UPDATE_INTERVAL = 118,
    PROP_UTC_OFFSET = 119,
    PROP_VENDOR_IDENTIFIER = 120,
    PROP_VENDOR_NAME = 121,
    PROP_VT_CLASSES_SUPPORTED = 122,
    PROP_WEEKLY_SCHEDULE = 123,
    PROP_ATTEMPTED_SAMPLES = 124,
    PROP_AVERAGE_VALUE = 125,
    PROP_BUFFER_SIZE = 126,
    PROP_CLIENT_COV_INCREMENT = 127,
    PROP_COV_RESUBSCRIPTION_INTERVAL = 128,
    PROP_CURRENT_NOTIFY_TIME = 129,
    PROP_EVENT_TIME_STAMPS = 130,
    PROP_LOG_BUFFER = 131,
    PROP_LOG_DEVICE_OBJECT_PROPERTY = 132,
    PROP_ENABLE = 133,
    PROP_LOG_INTERVAL = 134,
    PROP_MAXIMUM_VALUE = 135,
    PROP_MINIMUM_VALUE = 136,
    PROP_NOTIFICATION_THRESHOLD = 137,
    PROP_PREVIOUS_NOTIFY_TIME = 138,
    PROP_PROTOCOL_REVISION = 139,
    PROP_RECORDS_SINCE_NOTIFICATION = 140,
    PROP_RECORD_COUNT = 141,
    PROP_START_TIME = 142,
    PROP_STOP_TIME = 143,
    PROP_STOP_WHEN_FULL = 144,
    PROP_TOTAL_RECORD_COUNT = 145,
    PROP_VALID_SAMPLES = 146,
    PROP_WINDOW_INTERVAL = 147,
    PROP_WINDOW_SAMPLES = 148,
    PROP_MAXIMUM_VALUE_TIMESTAMP = 149,
    PROP_MINIMUM_VALUE_TIMESTAMP = 150,
    PROP_VARIANCE_VALUE = 151,
    PROP_ACTIVE_COV_SUBSCRIPTIONS = 152,
    PROP_BACKUP_FAILURE_TIMEOUT = 153,
    PROP_CONFIGURATION_FILES = 154,
    PROP_DATABASE_REVISION = 155,
    PROP_DIRECT_READING = 156,
    PROP_LAST_RESTORE_TIME = 157,
    PROP_MAINTENANCE_REQUIRED = 158,
    PROP_MEMBER_OF = 159,
    PROP_MODE = 160,
    PROP_OPERATION_EXPECTED = 161,
    PROP_SETTING = 162,
    PROP_SILENCED = 163,
    PROP_TRACKING_VALUE = 164,
    PROP_ZONE_MEMBERS = 165,
    PROP_LIFE_SAFETY_ALARM_VALUES = 166,
    PROP_MAX_SEGMENTS_ACCEPTED = 167,
    PROP_PROFILE_NAME = 168,
    PROP_AUTO_SLAVE_DISCOVERY = 169,
    PROP_MANUAL_SLAVE_ADDRESS_BINDING = 170,
    PROP_SLAVE_ADDRESS_BINDING = 171,
    PROP_SLAVE_PROXY_ENABLE = 172,
    PROP_LAST_NOTIFY_RECORD = 173,
    PROP_SCHEDULE_DEFAULT = 174,
    PROP_ACCEPTED_MODES = 175,
    PROP_ADJUST_VALUE = 176,
    PROP_COUNT = 177,
    PROP_COUNT_BEFORE_CHANGE = 178,
    PROP_COUNT_CHANGE_TIME = 179,
    PROP_COV_PERIOD = 180,
    PROP_INPUT_REFERENCE = 181,
    PROP_LIMIT_MONITORING_INTERVAL = 182,
    PROP_LOGGING_OBJECT = 183,
    PROP_LOGGING_RECORD = 184,
    PROP_PRESCALE = 185,
    PROP_PULSE_RATE = 186,
    PROP_SCALE = 187,
    PROP_SCALE_FACTOR = 188,
    PROP_UPDATE_TIME = 189,
    PROP_VALUE_BEFORE_CHANGE = 190,
    PROP_VALUE_SET = 191,
    PROP_VALUE_CHANGE_TIME = 192,
    PROP_ALIGN_INTERVALS = 193,
    PROP_INTERVAL_OFFSET = 195,
    PROP_LAST_RESTART_REASON = 196,
    PROP_LOGGING_TYPE = 197,
    PROP_RESTART_NOTIFICATION_RECIPIENTS = 202,
    PROP_TIME_OF_DEVICE_RESTART = 203,
    PROP_TIME_SYNCHRONIZATION_INTERVAL = 204,
    PROP_TRIGGER = 205,
    PROP_UTC_TIME_SYNCHRONIZATION_RECIPIENTS = 206,
    PROP_NODE_SUBTYPE = 207,
    PROP_NODE_TYPE = 208,
    PROP_STRUCTURED_OBJECT_LIST = 209,
    PROP_SUBORDINATE_ANNOTATIONS = 210,
    PROP_SUBORDINATE_LIST = 211,
    PROP_ACTUAL_SHED_LEVEL = 212,
    PROP_DUTY_WINDOW = 213,
    PROP_EXPECTED_SHED_LEVEL = 214,
    PROP_FULL_DUTY_BASELINE = 215,
    PROP_REQUESTED_SHED_LEVEL = 218,
    PROP_SHED_DURATION = 219,
    PROP_SHED_LEVEL_DESCRIPTIONS = 220,
    PROP_SHED_LEVELS = 221,
    PROP_STATE_DESCRIPTION = 222,
    PROP_DOOR_ALARM_STATE = 226,
    PROP_DOOR_EXTENDED_PULSE_TIME = 227,
    PROP_DOOR_MEMBERS = 228,
    PROP_DOOR_OPEN_TOO_LONG_TIME = 229,
    PROP_DOOR_PULSE_TIME = 230,
    PROP_DOOR_STATUS = 231,
    PROP_DOOR_UNLOCK_DELAY_TIME = 232,
    PROP_LOCK_STATUS = 233,
    PROP_MASKED_ALARM_VALUES = 234,
    PROP_SECURED_STATUS = 235,
    PROP_ABSENTEE_LIMIT = 244,
    PROP_ACCESS_ALARM_EVENTS = 245,
    PROP_ACCESS_DOORS = 246,
    PROP_ACCESS_EVENT = 247,
    PROP_ACCESS_EVENT_AUTHENTICATION_FACTOR = 248,
    PROP_ACCESS_EVENT_CREDENTIAL = 249,
    PROP_ACCESS_EVENT_TIME = 250,
    PROP_ACCESS_TRANSACTION_EVENTS = 251,
    PROP_ACCOMPANIMENT = 252,
    PROP_ACCOMPANIMENT_TIME = 253,
    PROP_ACTIVATION_TIME = 254,
    PROP_ACTIVE_AUTHENTICATION_POLICY = 255,
    PROP_ASSIGNED_ACCESS_RIGHTS = 256,
    PROP_AUTHENTICATION_FACTORS = 257,
    PROP_AUTHENTICATION_POLICY_LIST = 258,
    PROP_AUTHENTICATION_POLICY_NAMES = 259,
    PROP_AUTHENTICATION_STATUS = 260,
    PROP_AUTHORIZATION_MODE = 261,
    PROP_BELONGS_TO = 262,
    PROP_CREDENTIAL_DISABLE = 263,
    PROP_CREDENTIAL_STATUS = 264,
    PROP_CREDENTIALS = 265,
    PROP_CREDENTIALS_IN_ZONE = 266,
    PROP_DAYS_REMAINING = 267,
    PROP_ENTRY_POINTS = 268,
    PROP_EXIT_POINTS = 269,
    PROP_EXPIRY_TIME = 270,
    PROP_EXTENDED_TIME_ENABLE = 271,
    PROP_FAILED_ATTEMPT_EVENTS = 272,
    PROP_FAILED_ATTEMPTS = 273,
    PROP_FAILED_ATTEMPTS_TIME = 274,
    PROP_LAST_ACCESS_EVENT = 275,
    PROP_LAST_ACCESS_POINT = 276,
    PROP_LAST_CREDENTIAL_ADDED = 277,
    PROP_LAST_CREDENTIAL_ADDED_TIME = 278,
    PROP_LAST_CREDENTIAL_REMOVED = 279,
    PROP_LAST_CREDENTIAL_REMOVED_TIME = 280,
    PROP_LAST_USE_TIME = 281,
    PROP_LOCKOUT = 282,
    PROP_LOCKOUT_RELINQUISH_TIME = 283,
    PROP_MASTER_EXEMPTION = 284,
    PROP_MAX_FAILED_ATTEMPTS = 285,
    PROP_MEMBERS = 286,
    PROP_MUSTER_POINT = 287,
    PROP_NEGATIVE_ACCESS_RULES = 288,
    PROP_NUMBER_OF_AUTHENTICATION_POLICIES = 289,
    PROP_OCCUPANCY_COUNT = 290,
    PROP_OCCUPANCY_COUNT_ADJUST = 291,
    PROP_OCCUPANCY_COUNT_ENABLE = 292,
    PROP_OCCUPANCY_EXEMPTION = 293,
    PROP_OCCUPANCY_LOWER_LIMIT = 294,
    PROP_OCCUPANCY_LOWER_LIMIT_ENFORCED = 295,
    PROP_OCCUPANCY_STATE = 296,
    PROP_OCCUPANCY_UPPER_LIMIT = 297,
    PROP_OCCUPANCY_UPPER_LIMIT_ENFORCED = 298,
    PROP_PASSBACK_EXEMPTION = 299,
    PROP_PASSBACK_MODE = 300,
    PROP_PASSBACK_TIMEOUT = 301,
    PROP_POSITIVE_ACCESS_RULES = 302,
    PROP_REASON_FOR_DISABLE = 303,
    PROP_SUPPORTED_FORMATS = 304,
    PROP_SUPPORTED_FORMAT_CLASSES = 305,
    PROP_THREAT_AUTHORITY = 306,
    PROP_THREAT_LEVEL = 307,
    PROP_TRACE_FLAG = 308,
    PROP_TRANSACTION_NOTIFICATION_CLASS = 309,
    PROP_USER_EXTERNAL_IDENTIFIER = 310,
    PROP_USER_INFORMATION_REFERENCE = 311,
    PROP_USER_NAME = 317,
    PROP_USER_TYPE = 318,
    PROP_USES_REMAINING = 319,
    PROP_ZONE_FROM = 320,
    PROP_ZONE_TO = 321,
    PROP_ACCESS_EVENT_TAG = 322,
    PROP_GLOBAL_IDENTIFIER = 323,
    PROP_VERIFICATION_TIME = 326,
    PROP_BASE_DEVICE_SECURITY_POLICY = 327,
    PROP_DISTRIBUTION_KEY_REVISION = 328,
    PROP_DO_NOT_HIDE = 329,
    PROP_KEY_SETS = 330,
    PROP_LAST_KEY_SERVER = 331,
    PROP_NETWORK_ACCESS_SECURITY_POLICIES = 332,
    PROP_PACKET_REORDER_TIME = 333,
    PROP_SECURITY_PDU_TIMEOUT = 334,
    PROP_SECURITY_TIME_WINDOW = 335,
    PROP_SUPPORTED_SECURITY_ALGORITHM = 336,
    PROP_UPDATE_KEY_SET_TIMEOUT = 337,
    PROP_BACKUP_AND_RESTORE_STATE = 338,
    PROP_BACKUP_PREPARATION_TIME = 339,
    PROP_RESTORE_COMPLETION_TIME = 340,
    PROP_RESTORE_PREPARATION_TIME = 341,
    PROP_BIT_MASK = 342,
    PROP_BIT_TEXT = 343,
    PROP_IS_UTC = 344,
    PROP_GROUP_MEMBERS = 345,
    PROP_GROUP_MEMBER_NAMES = 346,
    PROP_MEMBER_STATUS_FLAGS = 347,
    PROP_REQUESTED_UPDATE_INTERVAL = 348,
    PROP_COVU_PERIOD = 349,
    PROP_COVU_RECIPIENTS = 350,
    PROP_EVENT_MESSAGE_TEXTS = 351,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG = 352,
    PROP_EVENT_DETECTION_ENABLE = 353,
    PROP_EVENT_ALGORITHM_INHIBIT = 354,
    PROP_EVENT_ALGORITHM_INHIBIT_REF = 355,
    PROP_TIME_DELAY_NORMAL = 356,
    PROP_RELIABILITY_EVALUATION_INHIBIT = 357,
    PROP_FAULT_PARAMETERS = 358,
    PROP_FAULT_TYPE = 359,
    PROP_LOCAL_FORWARDING_ONLY = 360,
    PROP_PROCESS_IDENTIFIER_FILTER = 361,
    PROP_SUBSCRIBED_RECIPIENTS = 362,
    PROP_PORT_FILTER = 363,
    PROP_AUTHORIZATION_EXEMPTIONS = 364,
    PROP_ALLOW_GROUP_DELAY_INHIBIT = 365,
    PROP_CHANNEL_NUMBER = 366,
    PROP_CONTROL_GROUPS = 367,
    PROP_EXECUTION_DELAY = 368,
    PROP_LAST_PRIORITY = 369,
    PROP_WRITE_STATUS = 370,
    PROP_PROPERTY_LIST = 371,
    PROP_SERIAL_NUMBER = 372,
    PROP_BLINK_WARN_ENABLE = 373,
    PROP_DEFAULT_FADE_TIME = 374,
    PROP_DEFAULT_RAMP_RATE = 375,
    PROP_DEFAULT_STEP_INCREMENT = 376,
    PROP_EGRESS_TIME = 377,
    PROP_IN_PROGRESS = 378,
    PROP_INSTANTANEOUS_POWER = 379,
    PROP_LIGHTING_COMMAND = 380,
    PROP_LIGHTING_COMMAND_DEFAULT_PRIORITY = 381,
    PROP_MAX_ACTUAL_VALUE = 382,
    PROP_MIN_ACTUAL_VALUE = 383,
    PROP_POWER = 384,
    PROP_TRANSITION = 385,
    PROP_EGRESS_ACTIVE = 386,

    PROP_INTERFACE_VALUE = 387,
    PROP_FAULT_HIGH_LIMIT = 388,
    PROP_FAULT_LOW_LIMIT = 389,
    PROP_LOW_DIFF_LIMIT = 390,
    PROP_STRIKE_COUNT = 391,
    PROP_TIME_OF_STRIKE_COUNT_RESET = 392,
    PROP_DEFAULT_TIMEOUT = 393,
    PROP_INITIAL_TIMEOUT = 394,
    PROP_LAST_STATE_CHANGE = 395,
    PROP_STATE_CHANGE_VALUES = 396,
    PROP_TIMER_RUNNING = 397,
    PROP_TIMER_STATE = 398,
    PROP_COMMAND_TIME_ARRAY = 430,
    PROP_CURRENT_COMMAND_PRIORITY = 431,
    PROP_LAST_COMMAND_TIME = 432,
    PROP_VALUE_SOURCE = 433,
    PROP_VALUE_SOURCE_ARRAY = 434,
    MAX_BACNET_PROPERTY_ID = 4194303,

class BACnetTimestampTags(enum.IntEnum):
    TIME_STAMP_NONE = -1,
    TIME_STAMP_TIME = 0,
    TIME_STAMP_SEQUENCE = 1,
    TIME_STAMP_DATETIME = 2

class BACnetTrendLogValueType(enum.IntEnum):
    TL_TYPE_STATUS = 0,
    TL_TYPE_BOOL = 1,
    TL_TYPE_REAL = 2,
    TL_TYPE_ENUM = 3,
    TL_TYPE_UNSIGN = 4,
    TL_TYPE_SIGN = 5,
    TL_TYPE_BITS = 6,
    TL_TYPE_NULL = 7,
    TL_TYPE_ERROR = 8,
    TL_TYPE_DELTA = 9,
    TL_TYPE_ANY = 10

class BACnetAddressTypes(enum.IntEnum):
    NONE = 0,
    IP = 1,
    MSTP = 2,
    Ethernet = 3,
    ArcNet = 4,
    LonTalk = 5,
    PTP = 6,
    IPV6 = 7

class BACnetPtpFrameTypes(enum.IntEnum):
    FRAME_TYPE_HEARTBEAT_XOFF = 0,
    FRAME_TYPE_HEARTBEAT_XON = 1,
    FRAME_TYPE_DATA0 = 2,
    FRAME_TYPE_DATA1 = 3,
    FRAME_TYPE_DATA_ACK0_XOFF = 4,
    FRAME_TYPE_DATA_ACK1_XOFF = 5,
    FRAME_TYPE_DATA_ACK0_XON = 6,
    FRAME_TYPE_DATA_ACK1_XON = 7,
    FRAME_TYPE_DATA_NAK0_XOFF = 8,
    FRAME_TYPE_DATA_NAK1_XOFF = 9,
    FRAME_TYPE_DATA_NAK0_XON = 0x0A,
    FRAME_TYPE_DATA_NAK1_XON = 0x0B,
    FRAME_TYPE_CONNECT_REQUEST = 0x0C,
    FRAME_TYPE_CONNECT_RESPONSE = 0x0D,
    FRAME_TYPE_DISCONNECT_REQUEST = 0x0E,
    FRAME_TYPE_DISCONNECT_RESPONSE = 0x0F,
    FRAME_TYPE_TEST_REQUEST = 0x14,
    FRAME_TYPE_TEST_RESPONSE = 0x15,
    FRAME_TYPE_GREETING = 0xFF,

class BACnetPtpDisconnectReasons(enum.IntEnum):
    PTP_DISCONNECT_NO_MORE_DATA = 0,
    PTP_DISCONNECT_PREEMPTED = 1,
    PTP_DISCONNECT_INVALID_PASSWORD = 2,
    PTP_DISCONNECT_OTHER = 3,

class BACnetMstpFrameTypes(enum.IntEnum):
    FRAME_TYPE_TOKEN = 0,
    FRAME_TYPE_POLL_FOR_MASTER = 1,
    FRAME_TYPE_REPLY_TO_POLL_FOR_MASTER = 2,
    FRAME_TYPE_TEST_REQUEST = 3,
    FRAME_TYPE_TEST_RESPONSE = 4,
    FRAME_TYPE_BACNET_DATA_EXPECTING_REPLY = 5,
    FRAME_TYPE_BACNET_DATA_NOT_EXPECTING_REPLY = 6,
    FRAME_TYPE_REPLY_POSTPONED = 7,
    FRAME_TYPE_PROPRIETARY_MIN = 128,
    FRAME_TYPE_PROPRIETARY_MAX = 255,

class BACnetNodeTypes(enum.IntEnum):
    NT_UNKNOWN = 0,
    NT_SYSTEM = 1,
    NT_NETWORK = 2,
    NT_DEVICE = 3,
    NT_ORGANIZATIONAL = 4,
    NT_AREA = 5,
    NT_EQUIPMENT = 6,
    NT_POINT = 7,
    NT_COLLECTION = 8,
    NT_PROPERTY = 9,
    NT_FUNCTIONAL = 10,
    NT_OTHER = 11,

class BACnetNpduControls(enum.IntEnum):
    PriorityNormalMessage = 0,
    PriorityUrgentMessage = 1,
    PriorityCriticalMessage = 2,
    PriorityLifeSafetyMessage = 3,
    ExpectingReply = 4,
    SourceSpecified = 8,
    DestinationSpecified = 32,
    NetworkLayerMessage = 128,

class BACnetNetworkMessageTypes(enum.IntEnum):
    NETWORK_MESSAGE_WHO_IS_ROUTER_TO_NETWORK = 0,
    NETWORK_MESSAGE_I_AM_ROUTER_TO_NETWORK = 1,
    NETWORK_MESSAGE_I_COULD_BE_ROUTER_TO_NETWORK = 2,
    NETWORK_MESSAGE_REJECT_MESSAGE_TO_NETWORK = 3,
    NETWORK_MESSAGE_ROUTER_BUSY_TO_NETWORK = 4,
    NETWORK_MESSAGE_ROUTER_AVAILABLE_TO_NETWORK = 5,
    NETWORK_MESSAGE_INIT_RT_TABLE = 6,
    NETWORK_MESSAGE_INIT_RT_TABLE_ACK = 7,
    NETWORK_MESSAGE_ESTABLISH_CONNECTION_TO_NETWORK = 8,
    NETWORK_MESSAGE_DISCONNECT_CONNECTION_TO_NETWORK = 9,

class BACnetReinitializedStates(enum.IntEnum):
    BACNET_REINIT_COLDSTART = 0,
    BACNET_REINIT_WARMSTART = 1,
    BACNET_REINIT_STARTBACKUP = 2,
    BACNET_REINIT_ENDBACKUP = 3,
    BACNET_REINIT_STARTRESTORE = 4,
    BACNET_REINIT_ENDRESTORE = 5,
    BACNET_REINIT_ABORTRESTORE = 6,
    BACNET_REINIT_IDLE = 255

class EncodeResult(enum.IntEnum):
    Good = 0,
    NotEnoughBuffer = 1,

class BACnetReadRangeRequestTypes(enum.IntEnum):
    RR_BY_POSITION = 1,
    RR_BY_SEQUENCE = 2,
    RR_BY_TIME = 4,
    RR_READ_ALL = 8,

class BACnetEventStates(enum.IntEnum):
    EVENT_STATE_NORMAL = 0,
    EVENT_STATE_FAULT = 1,
    EVENT_STATE_OFFNORMAL = 2,
    EVENT_STATE_HIGH_LIMIT = 3,
    EVENT_STATE_LOW_LIMIT = 4,
    EVENT_STATE_LIFE_SAFETY_ALARM = 5

class BACnetEventEnable(enum.IntEnum):
    EVENT_ENABLE_TO_OFFNORMAL = 1,
    EVENT_ENABLE_TO_FAULT = 2,
    EVENT_ENABLE_TO_NORMAL = 4

class BACnetLimitEnable(enum.IntEnum):
    EVENT_LOW_LIMIT_ENABLE = 1,
    EVENT_HIGH_LIMIT_ENABLE = 2

class BACnetNotifyTypes(enum.IntEnum):
    NOTIFY_ALARM = 0,
    NOTIFY_EVENT = 1,
    NOTIFY_ACK_NOTIFICATION = 2

class BACnetEventTypes(enum.IntEnum):
    EVENT_CHANGE_OF_BITSTRING = 0,
    EVENT_CHANGE_OF_STATE = 1,
    EVENT_CHANGE_OF_VALUE = 2,
    EVENT_COMMAND_FAILURE = 3,
    EVENT_FLOATING_LIMIT = 4,
    EVENT_OUT_OF_RANGE = 5,
    EVENT_CHANGE_OF_LIFE_SAFETY = 8,
    EVENT_EXTENDED = 9,
    EVENT_BUFFER_READY = 10,
    EVENT_UNSIGNED_RANGE = 11,

class BACnetCOVTypes(enum.IntEnum):
    CHANGE_OF_VALUE_BITS = 0,
    CHANGE_OF_VALUE_REAL = 1

class BACnetLifeSafetyStates(enum.IntEnum):
    MIN_LIFE_SAFETY_STATE = 0,
    LIFE_SAFETY_STATE_QUIET = 0,
    LIFE_SAFETY_STATE_PRE_ALARM = 1,
    LIFE_SAFETY_STATE_ALARM = 2,
    LIFE_SAFETY_STATE_FAULT = 3,
    LIFE_SAFETY_STATE_FAULT_PRE_ALARM = 4,
    LIFE_SAFETY_STATE_FAULT_ALARM = 5,
    LIFE_SAFETY_STATE_NOT_READY = 6,
    LIFE_SAFETY_STATE_ACTIVE = 7,
    LIFE_SAFETY_STATE_TAMPER = 8,
    LIFE_SAFETY_STATE_TEST_ALARM = 9,
    LIFE_SAFETY_STATE_TEST_ACTIVE = 10,
    LIFE_SAFETY_STATE_TEST_FAULT = 11,
    LIFE_SAFETY_STATE_TEST_FAULT_ALARM = 12,
    LIFE_SAFETY_STATE_HOLDUP = 13,
    LIFE_SAFETY_STATE_DURESS = 14,
    LIFE_SAFETY_STATE_TAMPER_ALARM = 15,
    LIFE_SAFETY_STATE_ABNORMAL = 16,
    LIFE_SAFETY_STATE_EMERGENCY_POWER = 17,
    LIFE_SAFETY_STATE_DELAYED = 18,
    LIFE_SAFETY_STATE_BLOCKED = 19,
    LIFE_SAFETY_STATE_LOCAL_ALARM = 20,
    LIFE_SAFETY_STATE_GENERAL_ALARM = 21,
    LIFE_SAFETY_STATE_SUPERVISORY = 22,
    LIFE_SAFETY_STATE_TEST_SUPERVISORY = 23,
    MAX_LIFE_SAFETY_STATE = 24,
    LIFE_SAFETY_STATE_PROPRIETARY_MIN = 256,
    LIFE_SAFETY_STATE_PROPRIETARY_MAX = 65535

class BACnetLifeSafetyModes(enum.IntEnum):
    MIN_LIFE_SAFETY_MODE = 0,
    LIFE_SAFETY_MODE_OFF = 0,
    LIFE_SAFETY_MODE_ON = 1,
    LIFE_SAFETY_MODE_TEST = 2,
    LIFE_SAFETY_MODE_MANNED = 3,
    LIFE_SAFETY_MODE_UNMANNED = 4,
    LIFE_SAFETY_MODE_ARMED = 5,
    LIFE_SAFETY_MODE_DISARMED = 6,
    LIFE_SAFETY_MODE_PREARMED = 7,
    LIFE_SAFETY_MODE_SLOW = 8,
    LIFE_SAFETY_MODE_FAST = 9,
    LIFE_SAFETY_MODE_DISCONNECTED = 10,
    LIFE_SAFETY_MODE_ENABLED = 11,
    LIFE_SAFETY_MODE_DISABLED = 12,
    LIFE_SAFETY_MODE_AUTOMATIC_RELEASE_DISABLED = 13,
    LIFE_SAFETY_MODE_DEFAULT = 14,
    MAX_LIFE_SAFETY_MODE = 15,
    LIFE_SAFETY_MODE_PROPRIETARY_MIN = 256,
    LIFE_SAFETY_MODE_PROPRIETARY_MAX = 65535

class BACnetLifeSafetyOperations(enum.IntEnum):
    LIFE_SAFETY_OP_NONE = 0,
    LIFE_SAFETY_OP_SILENCE = 1,
    LIFE_SAFETY_OP_SILENCE_AUDIBLE = 2,
    LIFE_SAFETY_OP_SILENCE_VISUAL = 3,
    LIFE_SAFETY_OP_RESET = 4,
    LIFE_SAFETY_OP_RESET_ALARM = 5,
    LIFE_SAFETY_OP_RESET_FAULT = 6,
    LIFE_SAFETY_OP_UNSILENCE = 7,
    LIFE_SAFETY_OP_UNSILENCE_AUDIBLE = 8,
    LIFE_SAFETY_OP_UNSILENCE_VISUAL = 9,
    LIFE_SAFETY_OP_PROPRIETARY_MIN = 64,
    LIFE_SAFETY_OP_PROPRIETARY_MAX = 65535

class BACnetPduTypes(enum.IntEnum):
    PDU_TYPE_CONFIRMED_SERVICE_REQUEST = 0
    SERVER = 1
    NEGATIVE_ACK = 2
    SEGMENTED_RESPONSE_ACCEPTED = 2
    MORE_FOLLOWS = 4
    SEGMENTED_MESSAGE = 8
    PDU_TYPE_UNCONFIRMED_SERVICE_REQUEST = 0x10
    PDU_TYPE_SIMPLE_ACK = 0x20
    PDU_TYPE_COMPLEX_ACK = 0x30
    PDU_TYPE_SEGMENT_ACK = 0x40
    PDU_TYPE_ERROR = 0x50
    PDU_TYPE_REJECT = 0x60
    PDU_TYPE_ABORT = 0x70
    PDU_TYPE_MASK = 0xF0

class BACnetSegmentations(enum.IntEnum):
    SEGMENTATION_BOTH = 0
    SEGMENTATION_TRANSMIT = 1
    SEGMENTATION_RECEIVE = 2
    SEGMENTATION_NONE = 3

class NetworkPriority(enum.IntEnum):
    Life_Safety_Message = 1
    Critical_Equipment_Message = 2
    Urgent_Message = 3
    Normal_Message = 4

class NetworkLayerMessageType(enum.IntEnum):
    WHO_IS_ROUTER_TO_NETWORK = 0
    I_AM_ROUTER_TO_NETWORK = 1
    I_COULD_BE_ROUTER_TO_NETWORK = 2
    REJECT_MESSAGE_TO_NETWORK = 3
    ROUTER_BUSY_TO_NETWORK = 4
    ROUTER_AVAILABLE_TO_NETWORK = 5
    INIT_RT_TABLE = 6
    INIT_RT_TABLE_ACK = 7
    ESTABLISH_CONNECTION_TO_NETWORK = 8
    DISCONNECT_CONNECTION_TO_NETWORK = 9
    Challenge_Request = 10
    Security_Payload = 11
    Security_Response = 12
    Request_Key_Update = 13
    Update_Key_Set = 14
    Update_Distribution_Key = 15
    Request_Master_Key = 16
    Set_Master_Key = 17
    What_Is_Network_Number = 18
    Network_Number_Is = 19

class ErrorClass(enum.IntEnum):
    device = 0
    object = 1
    property = 2
    resources = 3
    security = 4
    services = 5
    vt = 6
    communication = 7

class ErrorCode(enum.IntEnum):
    other = 0
    authentication_failed = 1  # formerly: removed version 1 revision 11
    configuration_in_progress = 2
    device_busy = 3
    dynamic_creation_not_supported = 4
    file_access_denied = 5
    incompatible_security_levels = 6  # formerly:removed in version 1 revision 11
    inconsistent_parameters = 7
    inconsistent_selection_criterion = 8
    invalid_data_type = 9
    invalid_file_access_method = 10
    invalid_file_start_position = 11
    invalid_operator_name = 12  # formerly:removed in version 1 revision 11
    invalid_parameter_data_type = 13
    invalid_timestamp = 14
    key_generation_error = 15  # formerly:removed in version 1 revision 11
    missing_required_parameter = 16
    no_objects_of_specified_type = 17
    no_space_for_object = 18
    no_space_to_add_list_element = 19
    no_space_to_write_property = 20
    no_vt_sessions_available = 21
    property_is_not_a_list = 22
    object_deletion_not_permitted = 23
    object_identifier_already_exists = 24
    operational_problem = 25
    password_failure = 26
    read_access_denied = 27
    security_not_supported = 28  # formerly:removed in version 1 revision 11
    service_request_denied = 29
    timeout = 30
    unknown_object = 31
    unknown_property = 32
    # this enumeration was removed = 33
    unknown_vt_class = 34
    unknown_vt_session = 35
    unsupported_object_type = 36
    value_out_of_range = 37
    vt_session_already_closed = 38
    vt_session_termination_failure = 39
    write_access_denied = 40
    character_set_not_supported = 41
    invalid_array_index = 42
    cov_subscription_failed = 43
    not_cov_property = 44
    optional_functionality_not_supported = 45
    invalid_configuration_data = 46
    datatype_not_supported = 47
    duplicate_name = 48
    duplicate_object_id = 49
    property_is_not_an_array = 50
    abort_buffer_overflow = 51
    abort_invalid_apdu_in_this_state = 52
    abort_preempted_by_higher_priority_task = 53
    abort_segmentation_not_supported = 54
    abort_proprietary = 55
    abort_other = 56
    invalid_tag = 57
    network_down = 58
    reject_buffer_overflow = 59
    reject_inconsistent_parameters = 60
    reject_invalid_parameter_data_type = 61
    reject_invalid_tag = 62
    reject_missing_required_parameter = 63
    reject_parameter_out_of_range = 64
    reject_too_many_arguments = 65
    reject_undefined_enumeration = 66
    reject_unrecognized_service = 67
    reject_proprietary = 68
    reject_other = 69
    unknown_device = 70
    unknown_route = 71
    value_not_initialized = 72
    invalid_event_state = 73
    no_alarm_configured = 74
    log_buffer_full = 75
    logged_value_purged = 76
    no_property_specified = 77
    not_configured_for_triggered_logging = 78
    unknown_subscription = 79
    parameter_out_of_range = 80
    list_element_not_found = 81
    busy = 82
    communication_disabled = 83
    success = 84
    access_denied = 85
    bad_destination_address = 86
    bad_destination_device_id = 87
    bad_signature = 88
    bad_source_address = 89
    bad_timestamp = 90
    cannot_use_key = 91
    cannot_verify_message_id = 92
    correct_key_revision = 93
    destination_device_id_required = 94
    duplicate_message = 95
    encryption_not_configured = 96
    encryption_required = 97
    incorrect_key = 98
    invalid_key_data = 99
    key_update_in_progress = 100
    malformed_message = 101
    not_key_server = 102
    security_not_configured = 103
    source_security_required = 104
    too_many_keys = 105
    unknown_authentication_type = 106
    unknown_key = 107
    unknown_key_revision = 108
    unknown_source_message = 109
    not_router_to_dnet = 110
    router_busy = 111
    unknown_network_message = 112
    message_too_long = 113
    security_error = 114
    addressing_error = 115
    write_bdt_failed = 116
    read_bdt_failed = 117
    register_foreign_device_failed = 118
    read_fdt_failed = 119
    delete_fdt_entry_failed = 120
    distribute_broadcast_failed = 121
    unknown_file_size = 122
    abort_apdu_too_long = 123
    abort_application_exceeded_reply_time = 124
    abort_out_of_resources = 125
    abort_tsm_timeout = 126
    abort_window_size_out_of_range = 127
    file_full = 128
    inconsistent_configuration = 129
    inconsistent_object_type = 130
    internal_error = 131
    not_configured = 132
    out_of_memory = 133
    value_too_long = 134
    abort_insufficient_security = 135
    abort_security_error = 136
    duplicate_entry = 137
    invalid_value_in_this_state = 138

class BACnetPropertyStateTypes(enum.IntEnum):
    BOOLEAN_VALUE = 0,
    BINARY_VALUE = 1,
    EVENT_TYPE = 2,
    POLARITY = 3,
    PROGRAM_CHANGE = 4,
    PROGRAM_STATE = 5,
    REASON_FOR_HALT = 6,
    RELIABILITY = 7,
    STATE = 8,
    SYSTEM_STATUS = 9,
    UNITS = 10,
    UNSIGNED_VALUE = 11,
    LIFE_SAFETY_MODE = 12,
    LIFE_SAFETY_STATE = 13

class BACnetLogRecordChoice(enum.IntEnum):
    log_status = 0
    boolean_value = 1
    real_value = 2
    enumerated_value = 3
    unsigned_value = 4
    integer_value = 5
    bitstring_value = 6
    null_value = 7
    failure = 8
    time_change = 9
    any_value = 10

class BACnetShedLevelChoice(enum.IntEnum):
    percent = 0
    level = 1
    amount = 2

class BACnetPropertyStatesChoice(enum.IntEnum):
    BOOLEAN_VALUE = 0
    BINARY_VALUE = 1
    EVENT_TYPE = 2
    POLARITY = 3
    PROGRAM_CHANGE = 4
    PROGRAM_STATE = 5
    REASON_FOR_HALT = 6
    RELIABILITY = 7
    STATE = 8
    SYSTEM_STATUS = 9
    UNITS = 10
    UNSIGNED_VALUE = 11
    LIFE_SAFETY_MODE = 12
    LIFE_SAFETY_STATE = 13
    RESTART_REASON = 14
    DOOR_ALARM_STATE = 15
    ACTION = 16
    DOOR_SECURED_STATUS = 17
    DOOR_STATUS = 18
    DOOR_VALUE = 19
    FILE_ACCESS_METHOD = 20
    LOCK_STATUS = 21
    LIFE_SAFETY_OPERATION = 22
    MAINTENANCE = 23
    NODE_TYPE = 24
    NOTIFY_TYPE = 25
    SECURITY_LEVEL = 26
    SHED_STATE = 27
    SILENCED_STATE = 28
    ACCESS_EVENT = 30
    ZONE_OCCUPANCY_STATE = 31
    ACCESS_CREDENTIAL_DISABLE_REASON = 32
    ACCESS_CREDENTIAL_DISABLE = 33
    AUTHENTICATION_STATUS = 34
    BACKUP_STATE = 36
    WRITE_STATUS = 37
    LIGHTING_IN_PROGRESS = 38
    LIGHTING_OPERATION = 39
    LIGHTING_TRANSITION = 40
    INTEGER_VALUE = 41
    BINARY_LIGHTING_VALUE = 42
    TIMER_STATE = 43
    TIMER_TRANSITION = 44
    BACNET_IP_MODE = 45
    NETWORK_PORT_COMMAND = 46
    NETWORK_TYPE = 47
    NETWORK_NUMBER_QUALITY = 48
    ESCALATOR_OPERATION_DIRECTION = 49
    ESCALATOR_FAULT = 50
    ESCALATOR_MODE = 51
    LIFT_CAR_DIRECTION = 52
    LIFT_CAR_DOOR_COMMAND = 53
    LIFT_CAR_DRIVE_STATUS = 54
    LIFT_CAR_MODE = 55
    LIFT_GROUP_MODE = 56
    LIFT_FAULT = 57
    PROTOCOL_LEVEL = 58
    EXTENDED_VALUE = 63

class BACnetAccessAuthenticationFactorDisable(enum.IntEnum):
    NONE = 0
    DISABLED = 1
    DISABLED_LOST = 2
    DISABLED_STOLEN = 3
    DISABLED_DAMAGED = 4
    DISABLED_DESTROYED = 5

class BACnetAccessCredentialDisable(enum.IntEnum):
    NONE = 0
    DISABLE = 1
    DISABLE_MANUAL = 2
    DISABLE_LOCKOUT = 3

class BACnetAccessCredentialDisableReason(enum.IntEnum):
    DISABLED = 0
    DISABLED_NEEDS_PROVISIONING = 1
    DISABLED_UNASSIGNED = 2
    DISABLED_NOT_YET_ACTIVE = 3
    DISABLED_EXPIRED = 4
    DISABLED_LOCKOUT = 5
    DISABLED_MAX_DAYS = 6
    DISABLED_MAX_USES = 7
    DISABLED_INACTIVITY = 8
    DISABLED_MANUAL = 9

class BACnetAccessEvent(enum.IntEnum):
    NONE = 0
    GRANTED = 1
    MUSTER = 2
    PASSBACK_DETECTED = 3
    DURESS = 4
    TRACE = 5
    LOCKOUT_MAX_ATTEMPTS = 6
    LOCKOUT_OTHER = 7
    LOCKOUT_RELINQUISHED = 8
    LOCKED_BY_HIGHER_PRIORITY = 9
    OUT_OF_SERVICE = 10
    OUT_OF_SERVICE_RELINQUISHED = 11
    ACCOMPANIMENT_BY = 12
    AUTHENTICATION_FACTOR_READ = 13
    AUTHORIZATION_DELAYED = 14
    VERIFICATION_REQUIRED = 15
    NO_ENTRY_AFTER_GRANT = 16
    DENIED_DENY_ALL = 128
    DENIED_UNKNOWN_CREDENTIAL = 129
    DENIED_AUTHENTICATION_UNAVAILABLE = 130
    DENIED_AUTHENTICATION_FACTOR_TIMEOUT = 131
    DENIED_INCORRECT_AUTHENTICATION_FACTOR = 132
    DENIED_ZONE_NO_ACCESS_RIGHTS = 133
    DENIED_POINT_NO_ACCESS_RIGHTS = 134
    DENIED_NO_ACCESS_RIGHTS = 135
    DENIED_OUT_OF_TIME_RANGE = 136
    DENIED_THREAT_LEVEL = 137
    DENIED_PASSBACK = 138
    DENIED_UNEXPECTED_LOCATION_USAGE = 139
    DENIED_MAX_ATTEMPTS = 140
    DENIED_LOWER_OCCUPANCY_LIMIT = 141
    DENIED_UPPER_OCCUPANCY_LIMIT = 142
    DENIED_AUTHENTICATION_FACTOR_LOST = 143
    DENIED_AUTHENTICATION_FACTOR_STOLEN = 144
    DENIED_AUTHENTICATION_FACTOR_DAMAGED = 145
    DENIED_AUTHENTICATION_FACTOR_DESTROYED = 146
    DENIED_AUTHENTICATION_FACTOR_DISABLED = 147
    DENIED_AUTHENTICATION_FACTOR_ERROR = 148
    DENIED_CREDENTIAL_UNASSIGNED = 149
    DENIED_CREDENTIAL_NOT_PROVISIONED = 150
    DENIED_CREDENTIAL_NOT_YET_ACTIVE = 151
    DENIED_CREDENTIAL_EXPIRED = 152
    DENIED_CREDENTIAL_MANUAL_DISABLE = 153
    DENIED_CREDENTIAL_LOCKOUT = 154
    DENIED_CREDENTIAL_MAX_DAYS = 155
    DENIED_CREDENTIAL_MAX_USES = 156
    DENIED_CREDENTIAL_INACTIVITY = 157
    DENIED_CREDENTIAL_DISABLED = 158
    DENIED_NO_ACCOMPANIMENT = 159
    DENIED_INCORRECT_ACCOMPANIMENT = 160
    DENIED_LOCKOUT = 161
    DENIED_VERIFICATION_FAILED = 162
    DENIED_VERIFICATION_TIMEOUT = 163
    DENIED_OTHER = 164

class BACnetAccessPassbackMode(enum.IntEnum):
    PASSBACK_OFF = 0
    HARD_PASSBACK = 1
    SOFT_PASSBACK = 2

class BACnetAccessUserType(enum.IntEnum):
    ASSET = 0
    GROUP = 1
    PERSON = 2

class BACnetAccessZoneOccupancyState(enum.IntEnum):
    NORMAL = 0
    BELOW_LOWER_LIMIT = 1
    AT_LOWER_LIMIT = 2
    AT_UPPER_LIMIT = 3
    ABOVE_UPPER_LIMIT = 4
    DISABLED = 5
    NOT_SUPPORTED = 6

class BACnetAction(enum.IntEnum):
    DIRECT = 0
    REVERSE = 1

class BACnetAuthenticationFactorType(enum.IntEnum):
    UNDEFINED = 0
    ERROR = 1
    CUSTOM = 2
    SIMPLE_NUMBER16 = 3
    SIMPLE_NUMBER32 = 4
    SIMPLE_NUMBER56 = 5
    SIMPLE_ALPHA_NUMERIC = 6
    ABA_TRACK2 = 7
    WIEGAND26 = 8
    WIEGAND37 = 9
    WIEGAND37_FACILITY = 10
    FACILITY16_CARD32 = 11
    FACILITY32_CARD32 = 12
    FASC_N = 13
    FASC_N_BCD = 14
    FASC_N_LARGE = 15
    FASC_N_LARGE_BCD = 16
    GSA75 = 17
    CHUID = 18
    CHUID_FULL = 19
    GUID = 20
    CBEFF_A = 21
    CBEFF_B = 22
    CBEFF_C = 23
    USER_PASSWORD = 24

class BACnetAuthenticationStatus(enum.IntEnum):
    NOT_READY = 0
    READY = 1
    DISABLED = 2
    WAITING_FOR_AUTHENTICATION_FACTOR = 3
    WAITING_FOR_ACCOMPANIMENT = 4
    WAITING_FOR_VERIFICATION = 5
    IN_PROGRESS = 6

class BACnetAuthorizationExemption(enum.IntEnum):
    PASSBACK = 0
    OCCUPANCY_CHECK = 1
    ACCESS_RIGHTS = 2
    LOCKOUT = 3
    DENY = 4
    VERIFICATION = 5
    AUTHORIZATION_DELAY = 6

class BACnetAuthorizationMode(enum.IntEnum):
    AUTHORIZE = 0
    GRANT_ACTIVE = 1
    DENY_ALL = 2
    VERIFICATION_REQUIRED = 3
    AUTHORIZATION_DELAYED = 4
    NONE = 5

class BACnetBinaryLightingPV(enum.IntEnum):
    OFF = 0
    ON = 1
    WARN = 2
    WARN_OFF = 3
    WARN_RELINQUISH = 4
    STOP = 5

class BACnetBinaryPV(enum.IntEnum):
    inactive = 0
    active = 1

class BACnetDoorAlarmState(enum.IntEnum):
    NORMAL = 0
    ALARM = 1
    DOOR_OPEN_TOO_LONG = 2
    FORCED_OPEN = 3
    TAMPER = 4
    DOOR_FAULT = 5
    LOCK_DOWN = 6
    FREE_ACCESS = 7
    EGRESS_OPEN = 8

class BACnetDoorSecuredStatus(enum.IntEnum):
    SECURED = 0
    UNSECURED = 1
    UNKNOWN = 2

class BACnetDoorStatus(enum.IntEnum):
    CLOSED = 0
    OPENED = 1
    UNKNOWN = 2
    DOOR_FAULT = 3
    UNUSED = 4
    NONE = 5
    CLOSING = 6
    OPENING = 7
    SAFETY_LOCKED = 8
    LIMITED_OPENED = 9

class BACnetDoorValue(enum.IntEnum):
    LOCK = 0
    UNLOCK = 1
    PULSE_UNLOCK = 2
    EXTENDED_PULSE_UNLOCK = 3

class BACnetEngineeringUnits(enum.IntEnum):
    METERS_PER_SECOND_PER_SECOND = 166
    SQUARE_METERS = 0
    SQUARE_CENTIMETERS = 116
    SQUARE_FEET = 1
    SQUARE_INCHES = 115
    CURRENCY1 = 105
    CURRENCY2 = 106
    CURRENCY3 = 107
    CURRENCY4 = 108
    CURRENCY5 = 109
    CURRENCY6 = 110
    CURRENCY7 = 111
    CURRENCY8 = 112
    CURRENCY9 = 113
    CURRENCY10 = 114
    MILLIAMPERES = 2
    AMPERES = 3
    AMPERES_PER_METER = 167
    AMPERES_PER_SQUARE_METER = 168
    AMPERE_SQUARE_METERS = 169
    DECIBELS = 199
    DECIBELS_MILLIVOLT = 200
    DECIBELS_VOLT = 201
    FARADS = 170
    HENRYS = 171
    OHMS = 4
    OHM_METERS = 172
    MILLIOHMS = 145
    KILOHMS = 122
    MEGOHMS = 123
    MICROSIEMENS = 190
    MILLISIEMENS = 202
    SIEMENS = 173
    SIEMENS_PER_METER = 174
    TESLAS = 175
    VOLTS = 5
    MILLIVOLTS = 124
    KILOVOLTS = 6
    MEGAVOLTS = 7
    VOLT_AMPERES = 8
    KILOVOLT_AMPERES = 9
    MEGAVOLT_AMPERES = 10
    VOLT_AMPERES_REACTIVE = 11
    KILOVOLT_AMPERES_REACTIVE = 12
    MEGAVOLT_AMPERES_REACTIVE = 13
    VOLTS_PER_DEGREE_KELVIN = 176
    VOLTS_PER_METER = 177
    DEGREES_PHASE = 14
    POWER_FACTOR = 15
    WEBERS = 178
    JOULES = 16
    KILOJOULES = 17
    KILOJOULES_PER_KILOGRAM = 125
    MEGAJOULES = 126
    WATT_HOURS = 18
    KILOWATT_HOURS = 19
    MEGAWATT_HOURS = 146
    WATT_HOURS_REACTIVE = 203
    KILOWATT_HOURS_REACTIVE = 204
    MEGAWATT_HOURS_REACTIVE = 205
    BTUS = 20
    KILO_BTUS = 147
    MEGA_BTUS = 148
    THERMS = 21
    TON_HOURS = 22
    JOULES_PER_KILOGRAM_DRY_AIR = 23
    KILOJOULES_PER_KILOGRAM_DRY_AIR = 149
    MEGAJOULES_PER_KILOGRAM_DRY_AIR = 150
    BTUS_PER_POUND_DRY_AIR = 24
    BTUS_PER_POUND = 117
    JOULES_PER_DEGREE_KELVIN = 127
    KILOJOULES_PER_DEGREE_KELVIN = 151
    MEGAJOULES_PER_DEGREE_KELVIN = 152
    JOULES_PER_KILOGRAM_DEGREE_KELVIN = 128
    NEWTON = 153
    CYCLES_PER_HOUR = 25
    CYCLES_PER_MINUTE = 26
    HERTZ = 27
    KILOHERTZ = 129
    MEGAHERTZ = 130
    PER_HOUR = 131
    GRAMS_OF_WATER_PER_KILOGRAM_DRY_AIR = 28
    PERCENT_RELATIVE_HUMIDITY = 29
    MICROMETERS = 194
    MILLIMETERS = 30
    CENTIMETERS = 118
    KILOMETERS = 193
    METERS = 31
    INCHES = 32
    FEET = 33
    CANDELAS = 179
    CANDELAS_PER_SQUARE_METER = 180
    WATTS_PER_SQUARE_FOOT = 34
    WATTS_PER_SQUARE_METER = 35
    LUMENS = 36
    LUXES = 37
    FOOT_CANDLES = 38
    MILLIGRAMS = 196
    GRAMS = 195
    KILOGRAMS = 39
    POUNDS_MASS = 40
    TONS = 41
    GRAMS_PER_SECOND = 154
    GRAMS_PER_MINUTE = 155
    KILOGRAMS_PER_SECOND = 42
    KILOGRAMS_PER_MINUTE = 43
    KILOGRAMS_PER_HOUR = 44
    POUNDS_MASS_PER_SECOND = 119
    POUNDS_MASS_PER_MINUTE = 45
    POUNDS_MASS_PER_HOUR = 46
    TONS_PER_HOUR = 156
    MILLIWATTS = 132
    WATTS = 47
    KILOWATTS = 48
    MEGAWATTS = 49
    BTUS_PER_HOUR = 50
    KILO_BTUS_PER_HOUR = 157
    HORSEPOWER = 51
    TONS_REFRIGERATION = 52
    PASCALS = 53
    HECTOPASCALS = 133
    KILOPASCALS = 54
    MILLIBARS = 134
    BARS = 55
    POUNDS_FORCE_PER_SQUARE_INCH = 56
    MILLIMETERS_OF_WATER = 206
    CENTIMETERS_OF_WATER = 57
    INCHES_OF_WATER = 58
    MILLIMETERS_OF_MERCURY = 59
    CENTIMETERS_OF_MERCURY = 60
    INCHES_OF_MERCURY = 61
    DEGREES_CELSIUS = 62
    DEGREES_KELVIN = 63
    DEGREES_KELVIN_PER_HOUR = 181
    DEGREES_KELVIN_PER_MINUTE = 182
    DEGREES_FAHRENHEIT = 64
    DEGREE_DAYS_CELSIUS = 65
    DEGREE_DAYS_FAHRENHEIT = 66
    DELTA_DEGREES_FAHRENHEIT = 120
    DELTA_DEGREES_KELVIN = 121
    YEARS = 67
    MONTHS = 68
    WEEKS = 69
    DAYS = 70
    HOURS = 71
    MINUTES = 72
    SECONDS = 73
    HUNDREDTHS_SECONDS = 158
    MILLISECONDS = 159
    NEWTON_METERS = 160
    MILLIMETERS_PER_SECOND = 161
    MILLIMETERS_PER_MINUTE = 162
    METERS_PER_SECOND = 74
    METERS_PER_MINUTE = 163
    METERS_PER_HOUR = 164
    KILOMETERS_PER_HOUR = 75
    FEET_PER_SECOND = 76
    FEET_PER_MINUTE = 77
    MILES_PER_HOUR = 78
    CUBIC_FEET = 79
    CUBIC_METERS = 80
    IMPERIAL_GALLONS = 81
    MILLILITERS = 197
    LITERS = 82
    US_GALLONS = 83
    CUBIC_FEET_PER_SECOND = 142
    CUBIC_FEET_PER_MINUTE = 84
    MILLION_CUBIC_FEET_PER_MINUTE = 254
    CUBIC_FEET_PER_HOUR = 191
    STANDARD_CUBIC_FEET_PER_DAY = 47808
    MILLION_STANDARD_CUBIC_FEET_PER_DAY = 47809
    THOUSAND_CUBIC_FEET_PER_DAY = 47810
    THOUSAND_STANDARD_CUBIC_FEET_PER_DAY = 47811
    POUNDS_MASS_PER_DAY = 47812
    CUBIC_METERS_PER_SECOND = 85
    CUBIC_METERS_PER_MINUTE = 165
    CUBIC_METERS_PER_HOUR = 135
    IMPERIAL_GALLONS_PER_MINUTE = 86
    MILLILITERS_PER_SECOND = 198
    LITERS_PER_SECOND = 87
    LITERS_PER_MINUTE = 88
    LITERS_PER_HOUR = 136
    US_GALLONS_PER_MINUTE = 89
    US_GALLONS_PER_HOUR = 192
    DEGREES_ANGULAR = 90
    DEGREES_CELSIUS_PER_HOUR = 91
    DEGREES_CELSIUS_PER_MINUTE = 92
    DEGREES_FAHRENHEIT_PER_HOUR = 93
    DEGREES_FAHRENHEIT_PER_MINUTE = 94
    JOULE_SECONDS = 183
    KILOGRAMS_PER_CUBIC_METER = 186
    KW_HOURS_PER_SQUARE_METER = 137
    KW_HOURS_PER_SQUARE_FOOT = 138
    MEGAJOULES_PER_SQUARE_METER = 139
    MEGAJOULES_PER_SQUARE_FOOT = 140
    NO_UNITS = 95
    NEWTON_SECONDS = 187
    NEWTONS_PER_METER = 188
    PARTS_PER_MILLION = 96
    PARTS_PER_BILLION = 97
    PERCENT = 98
    PERCENT_OBSCURATION_PER_FOOT = 143
    PERCENT_OBSCURATION_PER_METER = 144
    PERCENT_PER_SECOND = 99
    PER_MINUTE = 100
    PER_SECOND = 101
    PSI_PER_DEGREE_FAHRENHEIT = 102
    RADIANS = 103
    RADIANS_PER_SECOND = 184
    REVOLUTIONS_PER_MINUTE = 104
    SQUARE_METERS_PER_NEWTON = 185
    WATTS_PER_METER_PER_DEGREE_KELVIN = 189
    WATTS_PER_SQUARE_METER_DEGREE_KELVIN = 141
    PER_MILLE = 207
    GRAMS_PER_GRAM = 208
    KILOGRAMS_PER_KILOGRAM = 209
    GRAMS_PER_KILOGRAM = 210
    MILLIGRAMS_PER_GRAM = 211
    MILLIGRAMS_PER_KILOGRAM = 212
    GRAMS_PER_MILLILITER = 213
    GRAMS_PER_LITER = 214
    MILLIGRAMS_PER_LITER = 215
    MICROGRAMS_PER_LITER = 216
    GRAMS_PER_CUBIC_METER = 217
    MILLIGRAMS_PER_CUBIC_METER = 218
    MICROGRAMS_PER_CUBIC_METER = 219
    NANOGRAMS_PER_CUBIC_METER = 220
    GRAMS_PER_CUBIC_CENTIMETER = 221
    BECQUERELS = 222
    KILOBECQUERELS = 223
    MEGABECQUERELS = 224
    GRAY = 225
    MILLIGRAY = 226
    MICROGRAY = 227
    SIEVERTS = 228
    MILLISIEVERTS = 229
    MICROSIEVERTS = 230
    MICROSIEVERTS_PER_HOUR = 231
    MILLIREMS = 47814
    MILLIREMS_PER_HOUR = 47815
    DECIBELS_A = 232
    NEPHELOMETRIC_TURBIDITY_UNIT = 233
    PH = 234
    GRAMS_PER_SQUARE_METER = 235
    MINUTES_PER_DEGREE_KELVIN = 236
    METER_SQUARED_PER_METER = 237
    AMPERE_SECONDS = 238
    VOLT_AMPERE_HOURS = 239
    KILOVOLT_AMPERE_HOURS = 240
    MEGAVOLT_AMPERE_HOURS = 241
    VOLT_AMPERE_HOURS_REACTIVE = 242
    KILOVOLT_AMPERE_HOURS_REACTIVE = 243
    MEGAVOLT_AMPERE_HOURS_REACTIVE = 244
    VOLT_SQUARE_HOURS = 245
    AMPERE_SQUARE_HOURS = 246
    JOULE_PER_HOURS = 247
    CUBIC_FEET_PER_DAY = 248
    CUBIC_METERS_PER_DAY = 249
    WATT_HOURS_PER_CUBIC_METER = 250
    JOULES_PER_CUBIC_METER = 251
    MOLE_PERCENT = 252
    PASCAL_SECONDS = 253
    MILLION_STANDARD_CUBIC_FEET_PER_MINUTE = 254

class BACnetEscalatorFault(enum.IntEnum):
    CONTROLLER_FAULT = 0
    DRIVE_AND_MOTOR_FAULT = 1
    MECHANICAL_COMPONENT_FAULT = 2
    OVERSPEED_FAULT = 3
    POWER_SUPPLY_FAULT = 4
    SAFETY_DEVICE_FAULT = 5
    CONTROLLER_SUPPLY_FAULT = 6
    DRIVE_TEMPERATURE_EXCEEDED = 7
    COMB_PLATE_FAULT = 8

class BACnetEscalatorMode(enum.IntEnum):
    UNKNOWN = 0
    STOP = 1
    UP = 2
    DOWN = 3
    INSPECTION = 4
    OUT_OF_SERVICE = 5

class BACnetEscalatorOperationDirection(enum.IntEnum):
    UNKNOWN = 0
    STOPPED = 1
    UP_RATED_SPEED = 2
    UP_REDUCED_SPEED = 3
    DOWN_RATED_SPEED = 4
    DOWN_REDUCED_SPEED = 5

class BACnetEventState(enum.IntEnum):
    NORMAL = 0
    FAULT = 1
    OFFNORMAL = 2
    HIGH_LIMIT = 3
    LOW_LIMIT = 4
    LIFE_SAFETY_ALARM = 5

class BACnetEventType(enum.IntEnum):
    CHANGE_OF_BITSTRING = 0
    CHANGE_OF_STATE = 1
    CHANGE_OF_VALUE = 2
    COMMAND_FAILURE = 3
    FLOATING_LIMIT = 4
    OUT_OF_RANGE = 5
    COMPLEX = 6
    CHANGE_OF_LIFE_SAFETY = 8
    EXTENDED = 9
    BUFFER_READY = 10
    UNSIGNED_RANGE = 11
    ACCESS_EVENT = 13
    DOUBLE_OUT_OF_RANGE = 14
    SIGNED_OUT_OF_RANGE = 15
    UNSIGNED_OUT_OF_RANGE = 16
    CHANGE_OF_CHARACTERSTRING = 17
    CHANGE_OF_STATUS_FLAG = 18
    CHANGE_OF_RELIABILITY = 19
    NONE = 20
    CHANGE_OF_DISCRETE_VALUE = 21
    CHANGE_OF_TIMER = 22

class BACnetFaultType(enum.IntEnum):
    NONE = 0
    FAULT_CHARACTERSTRING = 1
    FAULT_EXTENDED = 2
    FAULT_LIFE_SAFETY = 3
    FAULT_STATE = 4
    FAULT_STATUS_FLAGS = 5
    FAULT_OUT_OF_RANGE = 6
    FAULT_LISTED = 7

class BACnetIPMode(enum.IntEnum):
    NORMAL = 0
    FOREIGN = 1
    BBMD = 2

class BACnetLifeSafetyMode(enum.IntEnum):
    OFF = 0
    LON = 1
    TEST = 2
    MANNED = 3
    UNMANNED = 4
    ARMED = 5
    DISARMED = 6
    PREARMED = 7
    SLOW = 8
    FAST = 9
    DISCONNECTED = 10
    ENABLED = 11
    DISABLED = 12
    AUTOMATIC_RELEASE_DISABLED = 13
    DEFAULT = 14

class BACnetLifeSafetyOperation(enum.IntEnum):
    NONE = 0,
    SILENCE = 1
    SILENCE_AUDIBLE = 2
    SILENCE_VISUAL = 3
    RESET = 4
    RESET_ALARM = 5
    RESET_FAULT = 6
    UNSILENCE = 7
    UNSILENCE_AUDIBLE = 8
    UNSILENCE_VISUAL = 9

class BACnetLifeSafetyState(enum.IntEnum):
    QUIET = 0
    PRE_ALARM = 1
    ALARM = 2
    FAULT = 3
    FAULT_PRE_ALARM = 4
    FAULT_ALARM = 5
    NOT_READY = 6
    ACTIVE = 7
    TAMPER = 8
    TEST_ALARM = 9
    TEST_ACTIVE = 10
    TEST_FAULT = 11
    TEST_FAULT_ALARM = 12
    HOLDUP = 13
    DURESS = 14
    TAMPER_ALARM = 15
    ABNORMAL = 16
    EMERGENCY_POWER = 17
    DELAYED = 18
    BLOCKED = 19
    LOCAL_ALARM = 20
    GENERAL_ALARM = 21
    SUPERVISORY = 22
    TEST_SUPERVISORY = 23

class BACnetLiftCarDirection(enum.IntEnum):
    UNKNOWN = 0
    NONE = 1
    STOPPED = 2
    UP = 3
    DOWN = 4
    UP_AND_DOWN = 5

class BACnetLiftCarDoorCommand(enum.IntEnum):
    NONE = 0
    OPEN = 1
    CLOSE = 2

class BACnetLiftCarDriveStatus(enum.IntEnum):
    UNKNOWN = 0
    STATIONARY = 1
    BRAKING = 2
    ACCELERATE = 3
    DECELERATE = 4
    RATED_SPEED = 5
    SINGLE_FLOOR_JUMP = 6
    TWO_FLOOR_JUMP = 7
    THREE_FLOOR_JUMP = 8
    MULTI_FLOOR_JUMP = 9

class BACnetLiftCarMode(enum.IntEnum):
    UNKNOWN = 0
    NORMAL = 1
    VIP = 2
    HOMING = 3
    PARKING = 4
    ATTENDANT_CONTROL = 5
    FIREFIGHTER_CONTROL = 6
    EMERGENCY_POWER = 7
    INSPECTION = 8
    CABINET_RECALL = 9
    EARTHQUAKE_OPERATION = 10
    FIRE_OPERATION = 11
    OUT_OF_SERVICE = 12
    OCCUPANT_EVACUATION = 13

class BACnetLiftFault(enum.IntEnum):
    CONTROLLER_FAULT = 0
    DRIVE_AND_MOTOR_FAULT = 1
    GOVERNOR_AND_SAFETY_GEAR_FAULT = 2
    LIFT_SHAFT_DEVICE_FAULT = 3
    POWER_SUPPLY_FAULT = 4
    SAFETY_INTERLOCK_FAULT = 5
    DOOR_CLOSING_FAULT = 6
    DOOR_OPENING_FAULT = 7
    CAR_STOPPED_OUTSIDE_LANDING_ZONE = 8
    CALL_BUTTON_STUCK = 9
    START_FAILURE = 10
    CONTROLLER_SUPPLY_FAULT = 11
    SELF_TEST_FAILURE = 12
    RUNTIME_LIMIT_EXCEEDED = 13
    POSITION_LOST = 14
    DRIVE_TEMPERATURE_EXCEEDED = 15
    LOAD_MEASUREMENT_FAULT = 16

class BACnetLiftGroupMode(enum.IntEnum):
    UNKNOWN = 0
    NORMAL = 1
    DOWN_PEAK = 2
    TWO_WAY = 3
    FOUR_WAY = 4
    EMERGENCY_POWER = 5
    UP_PEAK = 6

class BACnetLightingInProgress(enum.IntEnum):
    IDLE = 0
    FADE_ACTIVE = 1
    RAMP_ACTIVE = 2
    NOT_CONTROLLED = 3
    OTHER = 4

class BACnetLightingOperation(enum.IntEnum):
    NONE = 0
    FADE_TO = 1
    RAMP_TO = 2
    STEP_UP = 3
    STEP_DOWN = 4
    STEP_ON = 5
    STEP_OFF = 6
    WARN = 7
    WARN_OFF = 8
    WARN_RELINQUISH = 9
    STOP = 10

class BACnetLightingTransition(enum.IntEnum):
    NONE = 0
    FADE = 1
    RAMP = 2

class BACnetLockStatus(enum.IntEnum):
    LOCKED = 0
    UNLOCKED = 1
    LOCK_FAULT = 2
    UNUSED = 3
    UNKNOWN = 4

class BACnetLoggingType(enum.IntEnum):
    POLLED = 0
    COV = 1
    TRIGGERED = 2

class BACnetMaintenance(enum.IntEnum):
    NONE = 0
    PERIODIC_TEST = 1
    NEED_SERVICE_OPERATIONAL = 2
    NEED_SERVICE_INOPERATIVE = 3

class BACnetNetworkNumberQuality(enum.IntEnum):
    UNKNOWN = 0
    LEARNED = 1
    LEARNED_CONFIGURED = 2
    CONFIGURED = 3

class BACnetNetworkPortCommand(enum.IntEnum):
    IDLE = 0
    DISCARD_CHANGES = 1
    RENEW_FD_REGISTRATION = 2
    RESTART_SLAVE_DISCOVERY = 3
    RENEW_DHCP = 4
    RESTART_AUTONEGOTIATION = 5
    DISCONNECT = 6
    RESTART_PORT = 7

class BACnetNetworkType(enum.IntEnum):
    ETHERNET = 0
    ARCNET = 1
    MSTP = 2
    PTP = 3
    LONTALK = 4
    IPV4 = 5
    ZIGBEE = 6
    VIRTUAL = 7
    IPV6 = 9
    SERIAL = 10

class BACnetNodeType(enum.IntEnum):
    UNKNOWN = 0
    SYSTEM = 1
    NETWORK = 2
    DEVICE = 3
    ORGANIZATIONAL = 4
    AREA = 5
    EQUIPMENT = 6
    POINT = 7
    COLLECTION = 8
    PROPERTY = 9
    FUNCTIONAL = 10
    OTHER = 11
    SUBSYSTEM = 12
    BUILDING = 13
    FLOOR = 14
    SECTION = 15,
    MODULE = 16
    TREE = 17
    MEMBER = 18
    PROTOCOL = 19
    ROOM = 20
    ZONE = 21

class BACnetNotifyType(enum.IntEnum):
    ALARM = 0
    EVENT = 1
    ACK_NOTIFICATION = 2

class BACnetObjectType(enum.IntEnum):
    Analog_Input = 0
    Analog_Output = 1
    Analog_Value = 2
    Binary_Input = 3
    Binary_Output = 4
    Binary_Value = 5
    Calendar = 6
    Command = 7
    Device = 8
    Event_Enrollment = 9
    File = 10
    Group = 11
    Loop = 12
    Multi_State_Input = 13
    Multi_State_Output = 14
    Notification_Class = 15
    Program = 16
    Schedule = 17
    Averaging = 18
    Multi_State_Value = 19
    TrendLog = 20
    Life_Safety_Point = 21
    Life_Safety_Zone = 22
    Accumulator = 23
    Pulse_Converter = 24
    Event_Log = 25
    Global_Group = 26
    Trend_Log_Multiple = 27
    Load_Control = 28
    Structured_View = 29
    Access_Door = 30
    Timer = 31
    Access_Credential = 32
    Access_Point = 33
    Access_Rights = 34
    Access_User = 35
    Access_Zone = 36
    Credential_Data_Input = 37
    Network_Security = 38
    BitString_Value = 39
    CharacterString_Value = 40
    Date_Pattern_Value = 41
    Date_Value = 42
    DateTime_Pattern_Value = 43
    DateTime_Value = 44
    Integer_Value = 45
    Large_Analog_Value = 46
    OctetString_Value = 47
    Positive_Integer_Value = 48
    Time_Pattern_Value = 49
    Time_Value = 50
    Notification_Forwarder = 51
    Alert_Enrollment = 52
    Channel = 53
    Lighting_Output = 54
    Binary_Lighting_Output = 55
    Network_Port = 56
    Elevator_Group = 57
    Escalator = 58
    Lift = 59
    Staging = 60

class BACnetProgramError(enum.IntEnum):
    NORMAL = 0
    LOAD_FAILED = 1
    INTERNAL = 2
    PROGRAM = 3
    OTHER = 4

class BACnetProgramRequest(enum.IntEnum):
    READY = 0
    LOAD = 1
    RUN = 2
    HALT = 3
    RESTART = 4
    UNLOAD = 5

class BACnetPropertyIdentifier(enum.IntEnum):
    ACKED_TRANSITIONS = 0
    ACK_REQUIRED = 1
    ACTION = 2
    ACTION_TEXT = 3
    ACTIVE_TEXT = 4
    ACTIVE_VT_SESSIONS = 5
    ALARM_VALUE = 6
    ALARM_VALUES = 7
    ALL = 8
    ALL_WRITES_SUCCESSFUL = 9
    APDU_SEGMENT_TIMEOUT = 10
    APDU_TIMEOUT = 11
    APPLICATION_SOFTWARE_VERSION = 12
    ARCHIVE = 13
    BIAS = 14
    CHANGE_OF_STATE_COUNT = 15
    CHANGE_OF_STATE_TIME = 16
    NOTIFICATION_CLASS = 17
    BLANK_1 = 18
    CONTROLLED_VARIABLE_REFERENCE = 19
    CONTROLLED_VARIABLE_UNITS = 20
    CONTROLLED_VARIABLE_VALUE = 21
    COV_INCREMENT = 22
    DATE_LIST = 23
    DAYLIGHT_SAVINGS_STATUS = 24
    DEADBAND = 25
    DERIVATIVE_CONSTANT = 26
    DERIVATIVE_CONSTANT_UNITS = 27
    DESCRIPTION = 28
    DESCRIPTION_OF_HALT = 29
    DEVICE_ADDRESS_BINDING = 30
    DEVICE_TYPE = 31
    EFFECTIVE_PERIOD = 32
    ELAPSED_ACTIVE_TIME = 33
    ERROR_LIMIT = 34
    EVENT_ENABLE = 35
    EVENT_STATE = 36
    EVENT_TYPE = 37
    EXCEPTION_SCHEDULE = 38
    FAULT_VALUES = 39
    FEEDBACK_VALUE = 40
    FILE_ACCESS_METHOD = 41
    FILE_SIZE = 42
    FILE_TYPE = 43
    FIRMWARE_REVISION = 44
    HIGH_LIMIT = 45
    INACTIVE_TEXT = 46
    IN_PROCESS = 47
    INSTANCE_OF = 48
    INTEGRAL_CONSTANT = 49
    INTEGRAL_CONSTANT_UNITS = 50
    ISSUE_CONFIRMED_NOTIFICATIONS = 51
    LIMIT_ENABLE = 52
    LIST_OF_GROUP_MEMBERS = 53
    LIST_OF_OBJECT_PROPERTY_REFERENCES = 54
    LIST_OF_SESSION_KEYS = 55
    LOCAL_DATE = 56
    LOCAL_TIME = 57
    LOCATION = 58
    LOW_LIMIT = 59
    MANIPULATED_VARIABLE_REFERENCE = 60
    MAXIMUM_OUTPUT = 61
    MAX_APDU_LENGTH_ACCEPTED = 62
    MAX_INFO_FRAMES = 63
    MAX_MASTER = 64
    MAX_PRES_VALUE = 65
    MINIMUM_OFF_TIME = 66
    MINIMUM_ON_TIME = 67
    MINIMUM_OUTPUT = 68
    MIN_PRES_VALUE = 69
    MODEL_NAME = 70
    MODIFICATION_DATE = 71
    NOTIFY_TYPE = 72
    NUMBER_OF_APDU_RETRIES = 73
    NUMBER_OF_STATES = 74
    OBJECT_IDENTIFIER = 75
    OBJECT_LIST = 76
    OBJECT_NAME = 77
    OBJECT_PROPERTY_REFERENCE = 78
    OBJECT_TYPE = 79
    OPTIONAL = 80
    OUT_OF_SERVICE = 81
    OUTPUT_UNITS = 82
    EVENT_PARAMETERS = 83
    POLARITY = 84
    PRESENT_VALUE = 85
    PRIORITY = 86
    PRIORITY_ARRAY = 87
    PRIORITY_FOR_WRITING = 88
    PROCESS_IDENTIFIER = 89
    PROGRAM_CHANGE = 90
    PROGRAM_LOCATION = 91
    PROGRAM_STATE = 92
    PROPORTIONAL_CONSTANT = 93
    PROPORTIONAL_CONSTANT_UNITS = 94
    PROTOCOL_CONFORMANCE_CLASS = 95
    PROTOCOL_OBJECT_TYPES_SUPPORTED = 96
    PROTOCOL_SERVICES_SUPPORTED = 97
    PROTOCOL_VERSION = 98
    READ_ONLY = 99
    REASON_FOR_HALT = 100
    RECIPIENT = 101
    RECIPIENT_LIST = 102
    RELIABILITY = 103
    RELINQUISH_DEFAULT = 104
    REQUIRED = 105
    RESOLUTION = 106
    SEGMENTATION_SUPPORTED = 107
    SETPOINT = 108
    SETPOINT_REFERENCE = 109
    STATE_TEXT = 110
    STATUS_FLAGS = 111
    SYSTEM_STATUS = 112
    TIME_DELAY = 113
    TIME_OF_ACTIVE_TIME_RESET = 114
    TIME_OF_STATE_COUNT_RESET = 115
    TIME_SYNCHRONIZATION_RECIPIENTS = 116
    UNITS = 117
    UPDATE_INTERVAL = 118
    UTC_OFFSET = 119
    VENDOR_IDENTIFIER = 120
    VENDOR_NAME = 121
    VT_CLASSES_SUPPORTED = 122
    WEEKLY_SCHEDULE = 123
    ATTEMPTED_SAMPLES = 124
    AVERAGE_VALUE = 125
    BUFFER_SIZE = 126
    CLIENT_COV_INCREMENT = 127
    COV_RESUBSCRIPTION_INTERVAL = 128
    CURRENT_NOTIFY_TIME = 129
    EVENT_TIME_STAMPS = 130
    LOG_BUFFER = 131
    LOG_DEVICE_OBJECT_PROPERTY = 132
    ENABLE = 133
    LOG_INTERVAL = 134
    MAXIMUM_VALUE = 135
    MINIMUM_VALUE = 136
    NOTIFICATION_THRESHOLD = 137
    PREVIOUS_NOTIFY_TIME = 138
    PROTOCOL_REVISION = 139
    RECORDS_SINCE_NOTIFICATION = 140
    RECORD_COUNT = 141
    START_TIME = 142
    STOP_TIME = 143
    STOP_WHEN_FULL = 144
    TOTAL_RECORD_COUNT = 145
    VALID_SAMPLES = 146
    WINDOW_INTERVAL = 147
    WINDOW_SAMPLES = 148
    MAXIMUM_VALUE_TIMESTAMP = 149
    MINIMUM_VALUE_TIMESTAMP = 150
    VARIANCE_VALUE = 151
    ACTIVE_COV_SUBSCRIPTIONS = 152
    BACKUP_FAILURE_TIMEOUT = 153
    CONFIGURATION_FILES = 154
    DATABASE_REVISION = 155
    DIRECT_READING = 156
    LAST_RESTORE_TIME = 157
    MAINTENANCE_REQUIRED = 158
    MEMBER_OF = 159
    MODE = 160
    OPERATION_EXPECTED = 161
    SETTING = 162
    SILENCED = 163
    TRACKING_VALUE = 164
    ZONE_MEMBERS = 165
    LIFE_SAFETY_ALARM_VALUES = 166
    MAX_SEGMENTS_ACCEPTED = 167
    PROFILE_NAME = 168
    AUTO_SLAVE_DISCOVERY = 169
    MANUAL_SLAVE_ADDRESS_BINDING = 170
    SLAVE_ADDRESS_BINDING = 171
    SLAVE_PROXY_ENABLE = 172
    LAST_NOTIFY_RECORD = 173
    SCHEDULE_DEFAULT = 174
    ACCEPTED_MODES = 175
    ADJUST_VALUE = 176
    COUNT = 177
    COUNT_BEFORE_CHANGE = 178
    COUNT_CHANGE_TIME = 179
    COV_PERIOD = 180
    INPUT_REFERENCE = 181
    LIMIT_MONITORING_INTERVAL = 182
    LOGGING_OBJECT = 183
    LOGGING_RECORD = 184
    PRESCALE = 185
    PULSE_RATE = 186
    SCALE = 187
    SCALE_FACTOR = 188
    UPDATE_TIME = 189
    VALUE_BEFORE_CHANGE = 190
    VALUE_SET = 191
    VALUE_CHANGE_TIME = 192
    ALIGN_INTERVALS = 193
    INTERVAL_OFFSET = 195
    LAST_RESTART_REASON = 196
    LOGGING_TYPE = 197
    RESTART_NOTIFICATION_RECIPIENTS = 202
    TIME_OF_DEVICE_RESTART = 203
    TIME_SYNCHRONIZATION_INTERVAL = 204
    TRIGGER = 205
    UTC_TIME_SYNCHRONIZATION_RECIPIENTS = 206
    NODE_SUBTYPE = 207
    NODE_TYPE = 208
    STRUCTURED_OBJECT_LIST = 209
    SUBORDINATE_ANNOTATIONS = 210
    SUBORDINATE_LIST = 211
    ACTUAL_SHED_LEVEL = 212
    DUTY_WINDOW = 213
    EXPECTED_SHED_LEVEL = 214
    FULL_DUTY_BASELINE = 215
    REQUESTED_SHED_LEVEL = 218
    SHED_DURATION = 219
    SHED_LEVEL_DESCRIPTIONS = 220
    SHED_LEVELS = 221
    STATE_DESCRIPTION = 222
    DOOR_ALARM_STATE = 226
    DOOR_EXTENDED_PULSE_TIME = 227
    DOOR_MEMBERS = 228
    DOOR_OPEN_TOO_LONG_TIME = 229
    DOOR_PULSE_TIME = 230
    DOOR_STATUS = 231
    DOOR_UNLOCK_DELAY_TIME = 232
    LOCK_STATUS = 233
    MASKED_ALARM_VALUES = 234
    SECURED_STATUS = 235
    ABSENTEE_LIMIT = 244
    ACCESS_ALARM_EVENTS = 245
    ACCESS_DOORS = 246
    ACCESS_EVENT = 247
    ACCESS_EVENT_AUTHENTICATION_FACTOR = 248
    ACCESS_EVENT_CREDENTIAL = 249
    ACCESS_EVENT_TIME = 250
    ACCESS_TRANSACTION_EVENTS = 251
    ACCOMPANIMENT = 252
    ACCOMPANIMENT_TIME = 253
    ACTIVATION_TIME = 254
    ACTIVE_AUTHENTICATION_POLICY = 255
    ASSIGNED_ACCESS_RIGHTS = 256
    AUTHENTICATION_FACTORS = 257
    AUTHENTICATION_POLICY_LIST = 258
    AUTHENTICATION_POLICY_NAMES = 259
    AUTHENTICATION_STATUS = 260
    AUTHORIZATION_MODE = 261
    BELONGS_TO = 262
    CREDENTIAL_DISABLE = 263
    CREDENTIAL_STATUS = 264
    CREDENTIALS = 265
    CREDENTIALS_IN_ZONE = 266
    DAYS_REMAINING = 267
    ENTRY_POINTS = 268
    EXIT_POINTS = 269
    EXPIRY_TIME = 270
    EXTENDED_TIME_ENABLE = 271
    FAILED_ATTEMPT_EVENTS = 272
    FAILED_ATTEMPTS = 273
    FAILED_ATTEMPTS_TIME = 274
    LAST_ACCESS_EVENT = 275
    LAST_ACCESS_POINT = 276
    LAST_CREDENTIAL_ADDED = 277
    LAST_CREDENTIAL_ADDED_TIME = 278
    LAST_CREDENTIAL_REMOVED = 279
    LAST_CREDENTIAL_REMOVED_TIME = 280
    LAST_USE_TIME = 281
    LOCKOUT = 282
    LOCKOUT_RELINQUISH_TIME = 283
    MASTER_EXEMPTION = 284
    MAX_FAILED_ATTEMPTS = 285
    MEMBERS = 286
    MUSTER_POINT = 287
    NEGATIVE_ACCESS_RULES = 288
    NUMBER_OF_AUTHENTICATION_POLICIES = 289
    OCCUPANCY_COUNT = 290
    OCCUPANCY_COUNT_ADJUST = 291
    OCCUPANCY_COUNT_ENABLE = 292
    OCCUPANCY_EXEMPTION = 293
    OCCUPANCY_LOWER_LIMIT = 294
    OCCUPANCY_LOWER_LIMIT_ENFORCED = 295
    OCCUPANCY_STATE = 296
    OCCUPANCY_UPPER_LIMIT = 297
    OCCUPANCY_UPPER_LIMIT_ENFORCED = 298
    PASSBACK_EXEMPTION = 299
    PASSBACK_MODE = 300
    PASSBACK_TIMEOUT = 301
    POSITIVE_ACCESS_RULES = 302
    REASON_FOR_DISABLE = 303
    SUPPORTED_FORMATS = 304
    SUPPORTED_FORMAT_CLASSES = 305
    THREAT_AUTHORITY = 306
    THREAT_LEVEL = 307
    TRACE_FLAG = 308
    TRANSACTION_NOTIFICATION_CLASS = 309
    USER_EXTERNAL_IDENTIFIER = 310
    USER_INFORMATION_REFERENCE = 311
    USER_NAME = 317
    USER_TYPE = 318
    USES_REMAINING = 319
    ZONE_FROM = 320
    ZONE_TO = 321
    ACCESS_EVENT_TAG = 322
    GLOBAL_IDENTIFIER = 323
    VERIFICATION_TIME = 326
    BASE_DEVICE_SECURITY_POLICY = 327
    DISTRIBUTION_KEY_REVISION = 328
    DO_NOT_HIDE = 329
    KEY_SETS = 330
    LAST_KEY_SERVER = 331
    NETWORK_ACCESS_SECURITY_POLICIES = 332
    PACKET_REORDER_TIME = 333
    SECURITY_PDU_TIMEOUT = 334
    SECURITY_TIME_WINDOW = 335
    SUPPORTED_SECURITY_ALGORITHM = 336
    UPDATE_KEY_SET_TIMEOUT = 337
    BACKUP_AND_RESTORE_STATE = 338
    BACKUP_PREPARATION_TIME = 339
    RESTORE_COMPLETION_TIME = 340
    RESTORE_PREPARATION_TIME = 341
    BIT_MASK = 342
    BIT_TEXT = 343
    IS_UTC = 344
    GROUP_MEMBERS = 345
    GROUP_MEMBER_NAMES = 346
    MEMBER_STATUS_FLAGS = 347
    REQUESTED_UPDATE_INTERVAL = 348
    COVU_PERIOD = 349
    COVU_RECIPIENTS = 350
    EVENT_MESSAGE_TEXTS = 351
    EVENT_MESSAGE_TEXTS_CONFIG = 352
    EVENT_DETECTION_ENABLE = 353
    EVENT_ALGORITHM_INHIBIT = 354
    EVENT_ALGORITHM_INHIBIT_REF = 355
    TIME_DELAY_NORMAL = 356
    RELIABILITY_EVALUATION_INHIBIT = 357
    FAULT_PARAMETERS = 358
    FAULT_TYPE = 359
    LOCAL_FORWARDING_ONLY = 360
    PROCESS_IDENTIFIER_FILTER = 361
    SUBSCRIBED_RECIPIENTS = 362
    PORT_FILTER = 363
    AUTHORIZATION_EXEMPTIONS = 364
    ALLOW_GROUP_DELAY_INHIBIT = 365
    CHANNEL_NUMBER = 366
    CONTROL_GROUPS = 367
    EXECUTION_DELAY = 368
    LAST_PRIORITY = 369
    WRITE_STATUS = 370
    PROPERTY_LIST = 371
    SERIAL_NUMBER = 372
    BLINK_WARN_ENABLE = 373
    DEFAULT_FADE_TIME = 374
    DEFAULT_RAMP_RATE = 375
    DEFAULT_STEP_INCREMENT = 376
    EGRESS_TIME = 377
    IN_PROGRESS = 378
    INSTANTANEOUS_POWER = 379
    LIGHTING_COMMAND = 380
    LIGHTING_COMMAND_DEFAULT_PRIORITY = 381
    MAX_ACTUAL_VALUE = 382
    MIN_ACTUAL_VALUE = 383
    POWER = 384
    TRANSITION = 385
    EGRESS_ACTIVE = 386
    INTERFACE_VALUE = 387
    FAULT_HIGH_LIMIT = 388
    FAULT_LOW_LIMIT = 389
    LOW_DIFF_LIMIT = 390
    STRIKE_COUNT = 391
    TIME_OF_STRIKE_COUNT_RESET = 392
    DEFAULT_TIMEOUT = 393
    INITIAL_TIMEOUT = 394
    LAST_STATE_CHANGE = 395
    STATE_CHANGE_VALUES = 396
    TIMER_RUNNING = 397
    TIMER_STATE = 398
    APDU_LENGTH = 399
    IP_ADDRESS = 400
    IP_DEFAULT_GATEWAY = 401
    IP_DHCP_ENABLE = 402
    IP_DHCP_LEASE_TIME = 403
    IP_DHCP_LEASE_TIME_REMAINING = 404
    IP_DHCP_SERVER = 405
    IP_DNS_SERVER = 406
    BACNET_IP_GLOBAL_ADDRESS = 407
    BACNET_IP_MODE = 408
    BACNET_IP_MULTICAST_ADDRESS = 409
    BACNET_IP_NAT_TRAVERSAL = 410
    IP_SUBNET_MASK = 411
    BACNET_IP_UDP_PORT = 412
    BBMD_ACCEPT_FD_REGISTRATIONS = 413
    BBMD_BROADCAST_DISTRIBUTION_TABLE = 414
    BBMD_FOREIGN_DEVICE_TABLE = 415
    CHANGES_PENDING = 416
    COMMAND = 417
    FD_BBMD_ADDRESS = 418
    FD_SUBSCRIPTION_LIFETIME = 419
    LINK_SPEED = 420
    LINK_SPEEDS = 421
    LINK_SPEED_AUTONEGOTIATE = 422
    MAC_ADDRESS = 423
    NETWORK_INTERFACE_NAME = 424
    NETWORK_NUMBER = 425
    NETWORK_NUMBER_QUALITY = 426
    NETWORK_TYPE = 427
    ROUTING_TABLE = 428
    VIRTUAL_MAC_ADDRESS_TABLE = 429
    COMMAND_TIME_ARRAY = 430
    CURRENT_COMMAND_PRIORITY = 431
    LAST_COMMAND_TIME = 432
    VALUE_SOURCE = 433
    VALUE_SOURCE_ARRAY = 434
    BACNET_IPV6_MODE = 435
    IPV6_ADDRESS = 436
    IPV6_PREFIX_LENGTH = 437
    BACNET_IPV6_UDP_PORT = 438
    IPV6_DEFAULT_GATEWAY = 439
    BACNET_IPV6_MULTICAST_ADDRESS = 440
    IPV6_DNS_SERVER = 441
    IPV6_AUTO_ADDRESSING_ENABLE = 442
    IPV6_DHCP_LEASE_TIME = 443
    IPV6_DHCP_LEASE_TIME_REMAINING = 444
    IPV6_DHCP_SERVER = 445
    IPV6_ZONE_INDEX = 446
    ASSIGNED_LANDING_CALLS = 447
    CAR_ASSIGNED_DIRECTION = 448
    CAR_DOOR_COMMAND = 449
    CAR_DOOR_STATUS = 450
    CAR_DOOR_TEXT = 451
    CAR_DOOR_ZONE = 452
    CAR_DRIVE_STATUS = 453
    CAR_LOAD = 454
    CAR_LOAD_UNITS = 455
    CAR_MODE = 456
    CAR_MOVING_DIRECTION = 457
    CAR_POSITION = 458
    ELEVATOR_GROUP = 459
    ENERGY_METER = 460
    ENERGY_METER_REF = 461
    ESCALATOR_MODE = 462
    FLOOR_TEXT = 464
    GROUP_ID = 465
    GROUP_MODE = 467
    HIGHER_DECK = 468
    INSTALLATION_ID = 469
    LANDING_CALLS = 470
    LANDING_CALL_CONTROL = 471
    LANDING_DOOR_STATUS = 472
    LOWER_DECK = 473
    MACHINE_ROOM_ID = 474
    MAKING_CAR_CALL = 475
    NEXT_STOPPING_FLOOR = 476
    OPERATION_DIRECTION = 477
    PASSENGER_ALARM = 478
    POWER_MODE = 479
    REGISTERED_CAR_CALL = 480
    ACTIVE_COV_MULTIPLE_SUBSCRIPTIONS = 481
    PROTOCOL_LEVEL = 482
    REFERENCE_PORT = 483
    DEPLOYED_PROFILE_LOCATION = 484
    PROFILE_LOCATION = 485
    TAGS = 486
    SUBORDINATE_NODE_TYPES = 487
    SUBORDINATE_RELATIONSHIPS = 489
    SUBORDINATE_TAGS = 488
    DEFAULT_SUBORDINATE_RELATIONSHIP = 490
    REPRESENTS = 491
    DEFAULT_PRESENT_VALUE = 492
    PRESENT_STAGE = 493
    STAGES = 494
    STAGE_NAMES = 495
    TARGET_REFERENCES = 496
    fault_signals = 463

class BACnetProtocolLevel(enum.IntEnum):
    PHYSICAL = 0
    PROTOCOL = 1
    BACNET_APPLICATION = 2
    NON_BACNET_APPLICATION = 3

class BACnetRelationship(enum.IntEnum):
    UNKNOWN = 0
    DEFAULT = 1
    CONTAINS = 2
    CONTAINED_BY = 3
    USES = 4
    USED_BY = 5
    COMMANDS = 6
    COMMANDED_BY = 7
    ADJUSTS = 8
    ADJUSTED_BY = 9
    INGRESS = 10
    EGRESS = 11
    SUPPLIES_AIR = 12
    RECEIVES_AIR = 13
    SUPPLIES_HOT_AIR = 14
    RECEIVES_HOT_AIR = 15
    SUPPLIES_COOL_AIR = 16
    RECEIVES_COOL_AIR = 17
    SUPPLIES_POWER = 18
    RECEIVES_POWER = 19
    SUPPLIES_GAS = 20
    RECEIVES_GAS = 21
    SUPPLIES_WATER = 22
    RECEIVES_WATER = 23
    SUPPLIES_HOT_WATER = 24
    RECEIVES_HOT_WATER = 25
    SUPPLIES_COOL_WATER = 26
    RECEIVES_COOL_WATER = 27
    SUPPLIES_STEAM = 28
    RECEIVES_STEAM = 29

class BACnetSecurityLevel(enum.IntEnum):
    INCAPABLE = 0
    PLAIN = 1
    SIGNED = 2
    ENCRYPTED = 3
    SIGNED_END_TO_END = 4
    ENCRYPTED_END_TO_END = 5

class BACnetSecurityPolicy(enum.IntEnum):
    PLAIN_NON_TRUSTED = 0
    PLAIN_TRUSTED = 1
    SIGNED_TRUSTED = 2
    ENCRYPTED_TRUSTED = 3

class BACnetSegmentation(enum.IntEnum):
    SEGMENTED_BOTH = 0
    SEGMENTED_TRANSMIT = 1
    SEGMENTED_RECEIVE = 2
    NO_SEGMENTATION = 3

class BACnetShedState(enum.IntEnum):
    INACTIVE = 0
    REQUEST_PENDING = 1
    COMPLIANT = 2
    NON_COMPLIANT = 3

class BACnetSilencedState(enum.IntEnum):
    UNSILENCED = 0
    AUDIBLE_SILENCED = 1
    VISIBLE_SILENCED = 2
    ALL_SILENCED = 3

class BACnetTimerState(enum.IntEnum):
    IDLE = 0
    RUNNING = 1
    EXPIRED = 2

class BACnetTimerTransition(enum.IntEnum):
    NONE = 0
    IDLE_TO_RUNNING = 1
    RUNNING_TO_IDLE = 2
    RUNNING_TO_RUNNING = 3
    RUNNING_TO_EXPIRED = 4
    FORCED_TO_EXPIRED = 5
    EXPIRED_TO_IDLE = 6
    EXPIRED_TO_RUNNING = 7

class BACnetVTClass(enum.IntEnum):
    DEFAULT_TERMINAL = 0
    ANSI_X3_64 = 1
    DEC_VT52 = 2
    DEC_VT100 = 3
    DEC_VT220 = 4
    HP_700_94 = 5
    IBM_3130 = 6

class BACnetWriteStatus(enum.IntEnum):
    IDLE = 0
    IN_PROGRESS = 1
    SUCCESSFUL = 2
    FAILED = 3

class BACnetConfirmedServiceChoice(enum.IntEnum):
    # Alarm and Event Services
    ACKNOWLEDGE_ALARM = 0
    CONFIRMED_COV_NOTIFICATION = 1
    CONFIRMED_COV_NOTIFICATION_MULTIPLE = 31
    CONFIRMED_EVENT_NOTIFICATION = 2
    GET_ALARM_SUMMARY = 3
    GET_ENROLLMENT_SUMMARY = 4
    GET_EVENT_INFORMATION = 29
    LIFE_SAFETY_OPERATION = 27
    SUBSCRIBE_COV = 5
    SUBSCRIBE_COV_PROPERTY = 28
    SUBSCRIBE_COV_PROPERTY_MULTIPLE = 30
    # File Access Services
    ATOMIC_READ_FILE = 6
    ATOMIC_WRITE_FILE = 7
    # Object Access Services
    ADD_LIST_ELEMENT = 8
    REMOVE_LIST_ELEMENT = 9
    CREATE_OBJECT = 10
    DELETE_OBJECT = 11
    READ_PROPERTY = 12
    # SERVICE_CONFIRMED_READ_CONDITIONAL = 13 removed
    READ_PROPERTY_MULTIPLE = 14
    READ_RANGE = 26
    WRITE_PROPERTY = 15
    WRITE_PROPERTY_MULTIPLE = 16
    # Remote Device Management Services
    DEVICE_COMMUNICATION_CONTROL = 17
    CONFIRMED_PRIVATE_TRANSFER = 18
    CONFIRMED_TEXT_MESSAGE = 19
    REINITIALIZE_DEVICE = 20
    # Virtual Terminal Services
    VT_OPEN = 21
    VT_CLOSE = 22
    VT_DATA = 23
    # Security Services
    # SERVICE_CONFIRMED_AUTHENTICATE = 24 removed
    # SERVICE_CONFIRMED_REQUEST_KEY = 25 remove

class BACnetUnconfirmedServiceChoice(enum.IntEnum):
    I_AM = 0
    I_HAVE = 1
    UNCONFIRMED_COV_NOTIFICATION = 2
    UNCONFIRMED_EVENT_NOTIFICATION = 3
    UNCONFIRMED_PRIVATE_TRANSFER = 4
    UNCONFIRMED_TEXT_MESSAGE = 5
    TIME_SYNCHRONIZATION = 6
    WHO_HAS = 7
    WHO_IS = 8
    UTC_TIME_SYNCHRONIZATION = 9
    WRITE_GROUP = 10
    UNCONFIRMED_COV_NOTIFICATION_MULTIPLE = 11
    WHO_AM_I = 12 # not added to wireshark number unclear
    YOU_ARE = 13 # not added to wireshark number unclear

class BACnetBvlcFunctions(enum.IntEnum):
    BVLC_RESULT = 0
    BVLC_WRITE_BROADCAST_DISTRIBUTION_TABLE = 1
    BVLC_READ_BROADCAST_DIST_TABLE = 2
    BVLC_READ_BROADCAST_DIST_TABLE_ACK = 3
    BVLC_FORWARDED_NPDU = 4
    BVLC_REGISTER_FOREIGN_DEVICE = 5
    BVLC_READ_FOREIGN_DEVICE_TABLE = 6
    BVLC_READ_FOREIGN_DEVICE_TABLE_ACK = 7
    BVLC_DELETE_FOREIGN_DEVICE_TABLE_ENTRY = 8
    BVLC_DISTRIBUTE_BROADCAST_TO_NETWORK = 9
    BVLC_ORIGINAL_UNICAST_NPDU = 10
    BVLC_ORIGINAL_BROADCAST_NPDU = 11

class BACnetBvlcResults(enum.IntEnum):
    BVLC_RESULT_SUCCESSFUL_COMPLETION = 0x0000,
    BVLC_RESULT_WRITE_BROADCAST_DISTRIBUTION_TABLE_NAK = 0x0010,
    BVLC_RESULT_READ_BROADCAST_DISTRIBUTION_TABLE_NAK = 0x0020,
    BVLC_RESULT_REGISTER_FOREIGN_DEVICE_NAK = 0X0030,
    BVLC_RESULT_READ_FOREIGN_DEVICE_TABLE_NAK = 0x0040,
    BVLC_RESULT_DELETE_FOREIGN_DEVICE_TABLE_ENTRY_NAK = 0x0050,
    BVLC_RESULT_DISTRIBUTE_BROADCAST_TO_NETWORK_NAK = 0x0060

class StateChanges(enum.IntEnum):
    Reset = 0
    DoneInitializing = 1

    GenerateToken = 2
    ReceivedDataNeedingReply = 3
    ReceivedToken = 4

    ReceivedUnexpectedFrame = 5
    DoneWithPFM = 6
    ReceivedReplyToPFM = 7
    SoleMaster = 8
    DeclareSoleMaster = 9

    SendAndWait = 10
    NothingToSend = 11
    SendNoWait = 12

    SendToken = 13
    ResetMaintenancePFM = 14
    SendMaintenancePFM = 15
    SoleMasterRestartMaintenancePFM = 16
    SendAnotherFrame = 17
    NextStationUnknown = 18

    ReplyTimeOut = 19
    InvalidFrame = 20
    ReceivedReply = 22
    ReceivedPostpone = 22

    FindNewSuccessor = 23
    SawTokenUser = 24

    Reply = 25
    DeferredReply = 26

class GetMessageStatus(enum.IntEnum):
    Good = 0
    Timeout = 1
    SubTimeout = 2
    ConnectionClose = 3
    ConnectionError = 4
    DecodeError = 5

##BACnetBase########################
class BACnetTool:

    @staticmethod
    def int_to_byte(value: int):
        return c_ushort(value).value.to_bytes(2, 'little')[0]

    @staticmethod
    def int_to_ushort(value: int):
        return c_ushort(value).value

    @staticmethod
    def int_to_bytes(value: int):
        return c_ushort(value).value.to_bytes(1, 'little')

    @staticmethod
    def format_buffer(value: bytearray, length: int):
        return f"""[{''.join(["%02X " % value[i] for i in range(length)]).strip()}]"""

class ASN:
    BACNET_MAX_OBJECT = 0x3FF
    BACNET_INSTANCE_BITS = 22
    BACNET_MAX_INSTANCE = 0x3FFFFF
    MAX_BITSTRING_BYTES = 15
    BACNET_ARRAY_ALL = 0xFFFFFFFF
    BACNET_NO_PRIORITY = 0
    BACNET_MIN_PRIORITY = 1
    BACNET_MAX_PRIORITY = 16

class BACnetPropetyState:
    def __init__(self):
        self.tag: Optional[BACnetPropertyStateTypes] = None
        self.state = 0

class BACnetObjectDescription:
    def __init__(self):
        self.typeId: Optional[BACnetObjectTypes] = None
        self.propsId: list = []   #BACnetPropertyIds

# edit 20210922
class EncodeBuffer:

    def __init__(self, buffer: bytearray=bytearray(128), offset: int=0, expandable: bool = True):
        self.buffer = buffer
        self.offset = offset
        self.max_offset = len(buffer)
        self.expandable = expandable
        self.serialize_counter = 0
        self.min_limit = 0
        self.result: EncodeResult = EncodeResult.Good

    def Increment(self):
        if self.offset < self.max_offset:
            if self.serialize_counter >= self.min_limit:
                self.offset = self.offset + 1
            self.serialize_counter = self.serialize_counter + 1
        else:
            if self.serialize_counter >= self.min_limit:
                self.offset = self.offset + 1

    def AddByte(self, value: int):
        if self.offset < self.max_offset:
            if self.serialize_counter >= self.min_limit:
                self.buffer[self.offset] = BACnetTool.int_to_byte(value)
        else:
            if self.expandable:
                self.buffer = self.buffer + bytearray(len(self.buffer))
                self.max_offset = len(self.buffer) - 1
                if self.serialize_counter >= self.min_limit:
                    self.buffer[self.offset] = BACnetTool.int_to_byte(value)
            else:
                self.result = self.result | EncodeResult.NotEnoughBuffer
        self.Increment()

    def Add(self, buffer: bytearray, count: int):
        for i in range(count):
            self.AddByte(buffer[i])

    def ToArray(self):
        return self.buffer[0: self.offset]

    def Reset(self, offset: int):
        self.offset = offset
        self.serialize_counter = 0
        self.result = EncodeResult.Good

    def GetLength(self):
        return min(self.offset, self.max_offset)

#基类
class IASN1encode:

    @abc.abstractmethod
    def ASN1encode(self, buffer: EncodeBuffer):
        pass

class BACnetObjectId:

    def __init__(self, type: Optional[Union[BACnetObjectTypes, int]] = None, instance: int = 0):
        self.type = type
        self.instance = instance

    def __str__(self):
        return f"{self.type.__str__()}:{self.instance}"

    def __lt__(self, other):
        if self.type == other.type:
            return self.instance.__lt__(other.instance)
        else:
            if self.type == BACnetObjectTypes.OBJECT_DEVICE:
                return -1
            if other.type == BACnetObjectTypes.OBJECT_DEVICE:
                return -1
            return int(self.type).__lt__(int(other.type))

    def __hash__(self):
        return hash(self.__str__())

    def __eq__(self, other):
        return self.Equals(other)

    def Equals(self, obj: Any):
        if obj is None:
            return False
        else:
            return obj.__str__() == self.__str__()

    def Parse(self, value: str):
        ret = BACnetObjectId()
        if value is None or len(value) == 0:
            return ret
        p = value.find(':')
        if p < 0:
            return ret
        str_type = value[:p]
        str_instance = value[p+1:]
        ret.type = BACnetObjectTypes(int(str_type))
        ret.instance = int(str_instance)
        return ret

class BACnetBitString:

    def __init__(self, bits_used: int = 0, value: bytearray = None):
        self.bits_used = bits_used
        self.value = value

    def __str__(self):
        ret = ""
        for i in range(self.bits_used):
            ret = ret + "1" if (self.value[int(i / 8)] & (1 << (i % 8))) > 0 else "0"

    def SetBit(self, bit_number: int, v: bool):
        byte_number = int(bit_number / 8)
        bit_mask = 1
        if self.value is None:
            self.value = bytearray(ASN1.MAX_BITSTRING_BYTES)

        if byte_number < ASN1.MAX_BITSTRING_BYTES:
            if self.bits_used < bit_number + 1:
                self.bits_used = bit_number + 1
            bit_mask = BACnetTool.int_to_byte((bit_mask << (bit_number - (byte_number * 8))))

            if v:
                self.value[byte_number] |= bit_mask
            else:
                self.value[byte_number] &= BACnetTool.int_to_byte(~(bit_mask))

    def Parse(self, value: str):
        ret = BACnetBitString()
        ret.value = bytearray(ASN1.MAX_BITSTRING_BYTES)

        if value is not None and len(value) > 0:
            ret.bits_used = len(value)
            for i in range(ret.bits_used):
                is_set = value[i] == '1'
                if is_set:
                    ret.value[int(i / 8)] |= BACnetTool.int_to_byte(1 << (i % 8))
        return ret

    def ConvertToInt(self):
        return 0 if self.value is None else struct.unpack('I', self.value[:4])   # return value == null ? 0 : BitConverter.ToUInt32(value, 0);   struct.unpack( 'f', buffer )

    def ConvertFromInt(self, value: int):
        ret = BACnetBitString()
        ret.value = struct.pack('I', value)
        ret.bits_used = BACnetTool.int_to_byte(math.ceil(math.log(value, 2)))
        return ret

class BACnetValue:

    def __init__(self, tag: BACnetApplicationTags=None, value: Any=None):
        self.Tag = tag
        self.Value = value

        if value is not None and self.Tag is None:
            self.Tag = self.TagFromType(value)

    def TagFromType(self, value):   #TODO
        if isinstance(value, str):
            return BACnetApplicationTags.BACNET_APPLICATION_TAG_CHARACTER_STRING
        elif isinstance(value, int):
            return BACnetApplicationTags.BACNET_APPLICATION_TAG_SIGNED_INT
        elif isinstance(value, bool):
            return BACnetApplicationTags.BACNET_APPLICATION_TAG_BOOLEAN
        elif isinstance(value, float):
            return BACnetApplicationTags.BACNET_APPLICATION_TAG_REAL
        elif isinstance(value, BACnetBitString):
            return BACnetApplicationTags.BACNET_APPLICATION_TAG_BIT_STRING
        elif isinstance(value, BACnetObjectId):
            return BACnetApplicationTags.BACNET_APPLICATION_TAG_OBJECT_ID
        return BACnetApplicationTags.BACNET_APPLICATION_TAG_CONTEXT_SPECIFIC_ENCODED

    def __str__(self):
        if self.Value is None:
            return ''
        elif isinstance(self.Value, bytearray) or isinstance(self.Value, bytes):
            return ''.join(["%02X " % x for x in self.Value]).strip()
        else:
            return str(self.Value)

class BACnetDeviceObjectPropertyReference(IASN1encode):

    def __init__(self, objectIdentifier: BACnetObjectId=None, propertyIdentifier: BACnetPropertyIds=None, deviceIndentifier: BACnetObjectId=None, arrayIndex: int=ASN.BACNET_ARRAY_ALL):
        if objectIdentifier is not None:
            self.objectIdentifier = objectIdentifier
        else:
            self.objectIdentifier = BACnetObjectId(BACnetObjectTypes.MAX_BACNET_OBJECT_TYPE, 0)

        self.propertyIdentifier = propertyIdentifier
        self.arrayIndex = arrayIndex
        if deviceIndentifier is not None:
            self.deviceIndentifier = deviceIndentifier
        else:
            self.deviceIndentifier = BACnetObjectId(BACnetObjectTypes.MAX_BACNET_OBJECT_TYPE, 0)

    def ASN1encode(self, buffer: EncodeBuffer):
        ASN1.bacapp_encode_device_obj_property_ref(buffer, self)

class BACnetDeviceObjectReference(IASN1encode):

    def __init__(self, objectIdentifier: BACnetObjectId = None, deviceIndentifier: BACnetObjectId=None):
        if (objectIdentifier != None):
            self.objectIdentifier = objectIdentifier
        else:
            self.objectIdentifier = BACnetObjectId(BACnetObjectTypes.MAX_BACNET_OBJECT_TYPE, 0)

        if (deviceIndentifier != None):
            self.deviceIndentifier = deviceIndentifier
        else:
            self.deviceIndentifier = BACnetObjectId(BACnetObjectTypes.MAX_BACNET_OBJECT_TYPE, 0)

    def ASN1encode(self, buffer: EncodeBuffer):
        if self.deviceIndentifier.type == BACnetObjectTypes.OBJECT_DEVICE:
            ASN1.encode_context_object_id(buffer, 0, self.deviceIndentifier.type, self.deviceIndentifier.instance)
        ASN1.encode_context_object_id(buffer, 1, self.objectIdentifier.type, self.objectIdentifier.instance)

    def ASN1decode(self, buffer: bytearray, offset: int, len_value: int):
        length = 0
        (l, TagNum) = ASN1.decode_tag_number(buffer, offset)
        length += l

        if TagNum == 0:
            (l, self.deviceIndentifier.type, self.deviceIndentifier.instance) = ASN1.decode_object_id(buffer, offset + length)
            length += l

            (l, TagNum) = ASN1.decode_tag_number(buffer, offset)
            length += l
        else:
            deviceIndentifier = BACnetObjectId(BACnetObjectTypes.MAX_BACNET_OBJECT_TYPE, 0)

        (l, self.deviceIndentifier.type, self.deviceIndentifier.instance) = ASN1.decode_object_id(buffer, offset + length)
        length += l
        return length

    def __str__(self):
        if self.deviceIndentifier.type == BACnetObjectTypes.OBJECT_DEVICE:
            return f"{self.objectIdentifier.__str__()} on Device:{self.deviceIndentifier.instance.__str__()}"
        else:
            return f"{self.objectIdentifier.__str__()}"

class BACnetAddress(IASN1encode):

    def __init__(self, type: BACnetAddressTypes=BACnetAddressTypes.NONE, net: int=0, adr: bytearray=None):
        self.type = type
        self.net = net
        self.adr = adr
        self.VMac = bytearray(3)
        self.RoutedSource = None
        if self.adr is None:
            self.adr = bytearray(0)

    def __str__(self):
        return self.ToString(self.type)

    def __hash__(self):
        return hash(self.adr)

    def ToString(self, type: BACnetAddressTypes):
        if type == BACnetAddressTypes.IP:
            if self.adr is None or len(self.adr) < 6:
                return '0.0.0.0'
            sb1 = f"{self.adr[0]}.{self.adr[1]}.{self.adr[2]}.{self.adr[3]}:{int(self.adr[4] << 8 | self.adr[5] << 0)}"
            if isinstance(self.RoutedSource, BACnetAddress):
                sb1 = f"{sb1}:{self.RoutedSource.net}:{self.RoutedSource.adr[0]}"

            return sb1
        elif type == BACnetAddressTypes.MSTP:
            if self.adr is None or len(self.adr) < 1:
                return '-1'
            return str(self.adr[0])
        elif type == BACnetAddressTypes.PTP:
            return 'x'
        elif type == BACnetAddressTypes.Ethernet:
            sb1 = ''
            for i in range(6):
                sb1 = f"{sb1}{self.adr[i]:X2}"
                if i != 5:
                    sb1 += '-'
            return sb1
        elif type == BACnetAddressTypes.IPV6:
            if self.adr is None or len(self.adr) != 18:
                return '[::]'
            port = BACnetTool.int_to_ushort((self.adr[16] << 8) | (self.adr[17] << 0))
            Ipv6 = bytearray(16)
            Ipv6[0:16] = self.adr[:16]
            ep = (Ipv6, port)
            return str(ep)
        else:
            if self.adr is None:
                return '?'
            if len(self.adr) == 6:
                return self.ToString(BACnetAddressTypes.IP)
            if len(self.adr) == 18:
                return self.ToString(BACnetAddressTypes.IPV6)
            if len(self.adr) == 3:
                return f"IPv6 VMac : {int((self.adr[0] << 16) | (self.adr[1] << 8) | self.adr[2])}"
            sb2 = ''
            for i in range(len(self.adr)):
                sb2 = f"{sb2}{self.adr[i]} "
            return sb2

    def IP_and_port(self) -> tuple:
        return f"{self.adr[0]}.{self.adr[1]}.{self.adr[2]}.{self.adr[3]}", int(self.adr[4] << 8 | self.adr[5] << 0)

    def __eq__(self, other):
        return self.Equals(other)

    '''
    def ToSting(self, SourceOnly: bool):
        if self.RoutedSource is None:
            return self.__str__()
        if SourceOnly:
            return self.RoutedSource.__str__()
        else:
            return self.RoutedSource.__str__() + ' via ' + self.__str__()

    '''
    def Equals(self, obj: object):
        if not isinstance(obj, BACnetAddress):
            return False
        d: BACnetAddress = obj
        if self.adr is None and d.adr is None:
            return True
        elif self.adr is None or d.adr is None:
            return False
        elif len(self.adr) != len(d.adr):
            return False
        else:
            for i in range(len(self.adr)):
                if self.adr[i] != d.adr[i]:
                    return False
            if self.RoutedSource is None and d.RoutedSource is not None:
                return False
            if self.RoutedSource is None and d.RoutedSource is None:
                return True
            return self.RoutedSource.Equals(d.RoutedSource)

    def IsMyRouter(self, device):
        if device.RoutedSource is None or self.RoutedSource is not None:
            return False
        if len(self.adr) != len(device.adr):
            return False
        for i in range(len(self.adr)):
            if self.adr[i] != device.adr[i]:
                return False
        return True

    def ASN1encode(self, buffer: EncodeBuffer):
        ASN1.encode_opening_tag(buffer, 1)
        ASN1.encode_application_unsigned(buffer, self.net)
        ASN1.encode_application_octet_string(buffer, self.adr, 0, len(self.adr))
        ASN1.encode_closing_tag(buffer, 1)

    def FullHashString(self):
        s = str(self.type) + '.' + str(self.net) + '.'
        for i in range(len(self.adr)):
            s = f"{s}{self.adr[i]:X2}"
        if self.RoutedSource is not None:
            s = f"{s}:{self.RoutedSource.FullHashString()}"
        return s

class BACnetGenericTime:

    def __init__(self, Time: datetime = None, Tag: BACnetTimestampTags = None, Sequence: int = 0):
        self.Time = Time
        self.Tag = Tag
        self.Sequence = Sequence

class BACnetEventNotificationData:

    def __init__(self):
        self.processIdentifier: Optional[int] = None
        self.initiatingObjectIdentifier: BACnetObjectId = BACnetObjectId()
        self.eventObjectIdentifier: BACnetObjectId = BACnetObjectId()
        self.timeStamp: BACnetGenericTime = BACnetGenericTime()
        self.notificationClass: int = 0
        self.priority: int = 0
        self.eventType: Optional[BACnetEventTypes] = None
        self.messageText: Optional[str] = None
        self.notifyType: Optional[BACnetNotifyTypes] = None
        self.ackRequired: bool = False
        self.fromState: Optional[BACnetEventStates] = None
        self.toState: Optional[BACnetEventStates] = None
        self.changeOfBitstring_referencedBitString: Optional[BACnetBitString] = None
        self.changeOfBitstring_statusFlags: Optional[BACnetBitString] = None
        self.changeOfState_newState: BACnetPropetyState = BACnetPropetyState()
        self.changeOfState_statusFlags: Optional[BACnetBitString] = None
        self.changeOfValue_changedBits: Optional[BACnetBitString] = None
        self.changeOfValue_changeValue: float = 0
        self.changeOfValue_tag: Optional[BACnetCOVTypes] = None
        self.changeOfValue_statusFlags: Optional[BACnetBitString] = None
        self.floatingLimit_referenceValue: float = 0
        self.floatingLimit_statusFlags: Optional[BACnetBitString] = None
        self.floatingLimit_setPointValue: float = 0
        self.floatingLimit_errorLimit: float = 0

        self.outOfRange_exceedingValue: float = 0
        self.outOfRange_statusFlags: Optional[BACnetBitString] = None
        self.outOfRange_deadband: float = 0
        self.outOfRange_exceededLimit: float = 0
        self.changeOfLifeSafety_newState: Optional[BACnetLifeSafetyStates] = None
        self.changeOfLifeSafety_newMode: Optional[BACnetLifeSafetyModes] = None
        self.changeOfLifeSafety_statusFlags: Optional[BACnetBitString] = None
        self.changeOfLifeSafety_operationExpected: Optional[BACnetLifeSafetyOperations] = None

        self.bufferReady_bufferProperty: Optional[BACnetDeviceObjectPropertyReference] = None
        self.bufferReady_previousNotification: int = 0
        self.bufferReady_currentNotification: int = 0

        self.unsignedRange_exceedingValue: int = 0
        self.unsignedRange_statusFlags: Optional[BACnetBitString] = None
        self.unsignedRange_exceededLimit: int = 0

class BACnetPropertyReference:

    def __init__(self, id: int = 0, array_index: int = 0):
        self.propertyIdentifier = id
        self.propertyArrayIndex = array_index

    def __str__(self):
        return f"{self.propertyIdentifier}"

class BACnetPropertyValue:

    def __init__(self):
        self.property: BACnetPropertyReference = BACnetPropertyReference()
        self.value: list = []
        self.priority: int = 0

    def __str__(self):
        return self.property.__str__()

class BACnetDate(IASN1encode):

    def __init__(self, year: int = 0, month: int = 0, day: int = 0, wday: int = 255):
        self.year = year
        self.month = month
        self.day = day
        self.wday = wday

    def ASN1encode(self, buffer: EncodeBuffer):
        buffer.AddByte(self.year)
        buffer.AddByte(self.month)
        buffer.AddByte(self.day)
        buffer.AddByte(self.wday)

    def ASN1decode(self, buffer: bytearray, offset: int, len_value: int):
        self.year = buffer[offset]
        self.month = buffer[offset + 1]
        self.day = buffer[offset + 2]
        self.wday = buffer[offset + 3]
        return 4

    def IsPeriodic(self):
        return (self.year == 255) or (self.month > 12) or (self.day == 255)

    def IsAFittingDate(self, date: datetime):
        if date.year != (self.year + 1900) and self.year != 255:
            return False
        if (date.month != self.month) and (self.month != 255) and (self.month != 13) and (self.month != 14):
            return False
        if (self.month == 13) and ((date.month & 1) != 1):
            return False
        if (self.month == 14) and ((date.month & 1) == 1):
            return False

        if (date.day != self.day) and (self.day != 255):
            return False

        if self.wday == 255:
            return True
        if (self.wday == 7) and (date.isoweekday() == 0):
            return True
        date.weekday()
        if self.wday == date.isoweekday():
            return True
        return False

    def toDateTime(self):
        try:
            if self.IsPeriodic() is True:
                return datetime(1, 1, 1)
            else:
                return datetime(self.year+1900, self.month, self.day)
        except:
            pass
        return datetime.now()

    def GetDayName(self, day: int): #TODO
        if day == 7:
            day = 0
        DayNames = ["Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"]
        return DayNames[day]

    def __str__(self):
        ret: str = ''

        if self.wday != 255:
            ret = self.GetDayName(self.wday) + ' '
        else:
            ret = ''

        if self.day != 255:
            ret = ret + str(self.day) + "/"
        else:
            ret = ret + "**/"

        if self.month == 13:
            ret = ret + "odd/"
        elif self.month == 13:
            ret = ret + "even/"
        elif self.month == 13:
            ret = ret + "**"
        else:
            ret = ret + str(self.month) + "/"

        if self.year != 255:
            ret = ret + str(self.year + 1900)
        else:
            ret = ret + "****"

        return ret

class BACnetDateRange(IASN1encode):

    def __init__(self, start: BACnetDate = None, end: BACnetDate = None):
        self.startDate = start
        self.endDate = end

    def ASN1encode(self, buffer: EncodeBuffer):
        ASN1.encode_tag(buffer, BACnetApplicationTags.BACNET_APPLICATION_TAG_DATE, False, 4)
        self.startDate.ASN1encode(buffer)
        ASN1.encode_tag(buffer, BACnetApplicationTags.BACNET_APPLICATION_TAG_DATE, False, 4)
        self.endDate.ASN1encode(buffer)

    def ASN1decode(self, buffer: bytearray, offset: int, len_value: int):
        length = 1
        length += self.startDate.ASN1decode(buffer, offset + length, len_value)
        length += 1
        length += self.endDate.ASN1decode(buffer, offset + length, len_value)
        return length

    def IsAFittingDate(self, date: datetime):
        date = datetime(date.year, date.month, date.day)
        if (date >= self.startDate.toDateTime()) and (date <= self.endDate.toDateTime()):
            return True
        else:
            return False

    def __str__(self):
        ret = ''
        if self.startDate.day != 255:
            ret = "From " + self.startDate.__str__()
        else:
            ret = "From **/**/**"

        if self.endDate.day != 255:
            ret = " to " + self.endDate.__str__()
        else:
            ret = ret + " to **/**/**"
        return ret

class BACnetweekNDay(IASN1encode):

    def __init__(self, day: int = 0, month: int = 0, week: int = 255):
        self.month = month
        self.week = week
        self.wday = day

    def __str__(self):
        ret: str = ''

        if self.wday != 255:
            ret = self.GetDayName(self.wday) + ' '
        else:
            ret = 'Every days'

        if self.week >= 1 and self.week <= 5:
            ret = ret + " (days " + str(((self.week - 1) * 7) + 1) + "-" + str(self.week * 7) + ")"

        if self.week == 6:
            ret = ret + " (last 7 days of month)"

        if self.month <= 12:
            MonthNames = ["January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"]
            ret = ret + " on " + MonthNames[self.month - 1]
        else:
            if self.month == 255:
                ret = ret + " on every months"
            elif self.month == 13:
                ret = ret + " on odd months"
            else:
                ret = ret + " on even months"

        return ret

    def ASN1encode(self, buffer: EncodeBuffer):
        buffer.AddByte(self.month)
        buffer.AddByte(self.week)
        buffer.AddByte(self.wday)

    def ASN1decode(self, buffer: bytearray, offset: int, len_value: int):
        self.month = buffer[offset]
        offset += 1
        self.week = buffer[offset]
        offset += 1
        self.wday = buffer[offset]
        return 3

    def IsAFittingDate(self, date: datetime):
        if (date.month != self.month) and (self.month != 255) and (self.month != 13) and (self.month != 14):
            return False
        if (self.month == 13) and ((date.month & 1) != 1):
            return False
        if (self.month == 14) and ((date.month & 1) == 1):
            return False

        if self.week >= 1 and self.week <= 5:
            if date.day <= (self.week-1) * 7:
                return False
            if date.day > self.week * 7:
                return False

        if self.week == 6:
            if (date + timedelta(days=7)).month == date.month:
                return False

        if self.wday == 255:
            return True
        if (self.wday == 7) and (date.isoweekday() == 0):
            return True
        date.weekday()
        if self.wday == date.isoweekday():
            return True
        return False

    def GetDayName(self, day: int):  # TODO
        if day == 7:
            day = 0
        DayNames = ["Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"]
        return DayNames[day]

class BACnetCalendarEntry(IASN1encode):

    def __init__(self):
        self.Entries: Optional[List[object]] = None

    def ASN1encode(self, buffer: EncodeBuffer):
        if self.Entries is not None:
            for entry in self.Entries:
                if isinstance(entry, BACnetDate):
                    ASN1.encode_tag(buffer, 0, True, 4)
                    entry.ASN1encode(buffer)
                elif isinstance(entry, BACnetDateRange):
                    ASN1.encode_opening_tag(buffer, 1)
                    entry.ASN1encode(buffer)
                    ASN1.encode_closing_tag(buffer, 1)
                elif isinstance(entry, BACnetweekNDay):
                    ASN1.encode_tag(buffer, 2, True, 3)
                    entry.ASN1encode(buffer)

    def ASN1decode(self, buffer: bytearray, offset: int, len_value: int):
        length = 0
        Entries = []

        while True:
            b = buffer[offset + length]
            (l, tag_number) = ASN1.decode_tag_number(buffer, offset + length)
            length += l

            if tag_number == 0:
                bdt = BACnetDate()
                length += bdt.ASN1decode(buffer, offset + length, len_value)
                Entries.append(bdt)
            elif tag_number == 1:
                bdr = BACnetDateRange()
                length += bdr.ASN1decode(buffer, offset + length, len_value)
                Entries.append(bdr)
                length += 1
            elif tag_number == 2:
                bwd = BACnetweekNDay()
                length += bwd.ASN1decode(buffer, offset + length, len_value)
                Entries.append(bwd)
            else:
                return length-1

class BACnetGetEventInformationData:

    def __init__(self):
        self.objectIdentifier: BACnetObjectId = BACnetObjectId()
        self.eventState: Optional[BACnetEventStates] = None
        self.acknowledgedTransitions: Optional[BACnetBitString] = None
        self.eventTimeStamps: Optional[List[BACnetGenericTime]] = []
        self.notifyType: Optional[BACnetNotifyTypes] = None
        self.eventEnable: Optional[BACnetBitString] = None
        self.eventPriorities: Optional[List[int]] = []

class BACnetReadAccessSpecification:

    def __init__(self, objectIdentifier: BACnetObjectId = None, propertyReferences: list = None):
        self.objectIdentifier = objectIdentifier
        self.propertyReferences = propertyReferences

    def Parse(self, value: str):
        ret = BACnetReadAccessSpecification()
        if value is None or len(value) == 0:
            return ret
        tmp = value.split(':')
        if tmp is None or len(tmp) < 2:
            return ret
        ret.objectIdentifier.type = BACnetObjectTypes(int(tmp[0]))
        ret.objectIdentifier.instance = int(tmp[1])

        refs = []
        for i in range(2, len(tmp)):
            n = BACnetPropertyReference()
            n.propertyArrayIndex = ASN1.BACNET_ARRAY_ALL
            n.propertyIdentifier = int(tmp[i])
            refs.append(n)

        ret.propertyReferences = refs
        return ret

    def __str__(self):
        ret = self.objectIdentifier.__str__()
        for r in self.propertyReferences:
            ret += ":" + BACnetPropertyIds(r.propertyIdentifier).__str__()
        return ret

class BACnetReadAccessResult:

    def __init__(self, objectIdentifier: BACnetObjectId = None, values: list = None):
        if objectIdentifier is None:
            self.objectIdentifier = BACnetObjectId()
        else:
            self.objectIdentifier = objectIdentifier
        self.values = values

class BACnetCOVSubscription:

    def __init__(self):
        self.Recipient: BACnetAddress = BACnetAddress()
        self.subscriptionProcessIdentifier: int = 0
        self.monitoredObjectIdentifier: BACnetObjectId = BACnetObjectId()
        self.monitoredProperty: BACnetPropertyReference = BACnetPropertyReference()
        self.IssueConfirmedNotifications: bool = False
        self.TimeRemaining: int = 0
        self.COVIncrement: float = 0

    def __str__(self):
        name = self.monitoredObjectIdentifier.__str__()
        if name.startswith('OBJECT_'):
            name = name[7:]
        return name + " by " + self.Recipient.__str__() + ", remain " + str(self.TimeRemaining) + "s"

class BACnetError:

    def __init__(self, error_class: BACnetErrorClasses = None, error_code: BACnetErrorCodes = None):
        self.error_class = error_class
        self.error_code = error_code

    def __str__(self):
        return self.error_class.__str__() + ": " + self.error_code.__str__()

class BACnetLogRecord:

    class Value(object):

        def __init__(self):
           pass

        def __get__(self, instance, owner):
            if instance.type == BACnetTrendLogValueType.TL_TYPE_ANY:
                return instance.any_value
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_BITS:
                return BACnetBitString().Parse(instance.any_value.__str__())
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_BOOL:
                return bool(instance.any_value)
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_DELTA:
                return float(instance.any_value)
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_ENUM:
                return int(instance.any_value)
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_ERROR:
                if instance.any_value is not None:
                    return instance.any_value
                else:
                    return BACnetError(BACnetErrorClasses.ERROR_CLASS_DEVICE, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_NULL:
                return None
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_REAL:
                return float(instance.any_value)
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_SIGN:
                return int(instance.any_value)
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_STATUS:
                return BACnetBitString().Parse(instance.any_value.__str__())
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_UNSIGN:
                return int(instance.any_value)
            else:
                raise Exception(" NotSupported")

        def __set__(self, instance, value):
            if instance.type == BACnetTrendLogValueType.TL_TYPE_ANY:
                instance.any_value = value
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_BITS:
                if value is None:
                    value = BACnetBitString()
                if not instance(value, BACnetBitString):
                    value = BACnetBitString().ConvertFromInt(int(value))
                instance.any_value = value
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_BOOL:
                if value is None:
                    value = False
                if not instance(value, bool):
                    value = bool(value)
                instance.any_value = value
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_DELTA:
                if value is None:
                    value = 0
                if not instance(value, float):
                    value = float(value)
                instance.any_value = value
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_ENUM:
                if value is None:
                    value = 0
                if not instance(value, int):
                    value = int(value)
                instance.any_value = value
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_ERROR:
                if value is None:
                    value = BACnetError()
                if not isinstance(value, BACnetError):
                    raise Exception('Argument')
                instance.any_value = value
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_NULL:
                if value is not None:
                    raise Exception('Argument')
                instance.any_value = value
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_REAL:
                if value is None:
                    value = 0
                if not instance(value, float):
                    value = float(value)
                instance.any_value = value
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_SIGN:
                if value is None:
                    value = 0
                if not instance(value, int):
                    value = int(value)
                instance.any_value = value
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_STATUS:
                if value is None:
                    value = BACnetBitString()
                if not instance(value, BACnetBitString):
                    value = BACnetBitString().ConvertFromInt(int(value))
                instance.any_value = value
            elif instance.type == BACnetTrendLogValueType.TL_TYPE_UNSIGN:
                if value is None:
                    value = 0
                if not instance(value, int):
                    value = int(value)
                instance.any_value = value
            else:
                raise Exception(" NotSupported")

    def __init__(self, type: BACnetTrendLogValueType = 0, value: object = None, stamp: datetime = None, status: int = 0):
        self.type = type
        self.timestamp = stamp
        self.statusFlags = BACnetBitString().ConvertFromInt(status)
        self.any_value = None
        self.Value = value

    def GetValue(self, type): #TODO
        value = self.Value
        if isinstance(type, int):
            value = int(self.any_value)
        elif isinstance(type, float):
            value = float(self.any_value)
        elif isinstance(type, bool):
            value = bool(self.Value)
        elif isinstance(type, BACnetBitString):
            value = BACnetBitString().ConvertFromInt(int(self.any_value))
        return value

class DeviceReportingRecipient(IASN1encode):
    '''
    def __init__(self, v0: BACnetValue=None, v1: BACnetValue=None, v2: BACnetValue=None, v3: BACnetValue=None, v4: BACnetValue=None, v5: BACnetValue=None, v6: BACnetValue=None):
        self.Id = BACnetObjectId()
        self.adr = None
        self.WeekofDay = v0.Value
        self.fromTime = v1.Value
        self.toTime = v2.Value
        if isinstance(v3.Value, BACnetObjectId):
            self.Id = v3.Value
        else:
            netdescr = v3.Value
            s = int(netdescr[0].Value)
            b = netdescr[0].Value
            self.adr = BACnetAddress(BACnetAddressTypes.IP, s, b)

        self.processIdentifier = int(v4.Value)
        self.Ack_Required = bool(v5.Value)
        self.evenType = v6.Value
    '''

    def __init__(self, WeekofDay: BACnetBitString=None, fromTime: datetime=None, toTime: datetime=None, Id: BACnetObjectId=None, processIdentifier: int=0, Ack_Required: bool=False, evenType: BACnetBitString=None):
        self.WeekofDay = WeekofDay
        self.fromTime = fromTime
        self.toTime = toTime
        self.Id = Id
        self.processIdentifier = processIdentifier
        self.Ack_Required = Ack_Required
        self.evenType = evenType
        self.adr: BACnetAddress = BACnetAddress()

    #TODO
    '''
    def __init__(self, WeekofDay: BACnetBitString=None, fromTime: datetime=None, toTime: datetime=None, adr: BACnetAddress=None, processIdentifier: int=0, Ack_Required: bool=False, evenType: BACnetBitString=None):
        self.adr = adr
        self.WeekofDay = WeekofDay
        self.fromTime = fromTime
        self.toTime = toTime
        self.Id = BACnetObjectId()
        self.processIdentifier = processIdentifier
        self.Ack_Required = Ack_Required
        self.evenType = evenType
    '''

    def ASN1encode(self, buffer: EncodeBuffer):
        ASN1.bacapp_encode_application_data(buffer, BACnetValue(BACnetApplicationTags.BACNET_APPLICATION_TAG_BIT_STRING, self.WeekofDay))
        ASN1.bacapp_encode_application_data(buffer, BACnetValue(BACnetApplicationTags.BACNET_APPLICATION_TAG_TIME, self.fromTime))
        if self.adr is not None:
            self.adr.ASN1encode(buffer)
        else:
            ASN1.encode_context_object_id(buffer, 0, self.Id.type, self.Id.instance)
        ASN1.bacapp_encode_application_data(buffer, BACnetValue(BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT, self.processIdentifier))
        ASN1.bacapp_encode_application_data(buffer, BACnetValue(BACnetApplicationTags.BACNET_APPLICATION_TAG_BOOLEAN, self.Ack_Required))
        ASN1.bacapp_encode_application_data(buffer, BACnetValue(BACnetApplicationTags.BACNET_APPLICATION_TAG_BIT_STRING, self.evenType))

class BACnetObject:

    def __init__(self, object_id: BACnetObjectId, object_name: Optional[str]=None):
        self.object_id = object_id
        self.object_identifier = None
        self.object_name = object_name
        self.object_type = None
        self.present_value = None
        self.unit = None
        self.description = None

    def IsValid(self):
        if self.object_id.type in [BACnetObjectTypes.OBJECT_ANALOG_INPUT, BACnetObjectTypes.OBJECT_ANALOG_OUTPUT, BACnetObjectTypes.OBJECT_ANALOG_VALUE, BACnetObjectTypes.OBJECT_BINARY_INPUT, BACnetObjectTypes.OBJECT_BINARY_OUTPUT, BACnetObjectTypes.OBJECT_BINARY_VALUE, BACnetObjectTypes.OBJECT_MULTI_STATE_INPUT, BACnetObjectTypes.OBJECT_MULTI_STATE_OUTPUT, BACnetObjectTypes.OBJECT_MULTI_STATE_VALUE]:
            return True
        return False

    def UpdateProperty(self, properties: list):
        for property in properties:
            value = None
            b_values: list = []
            if property.value is not None:
                b_values = property.value.copy()
                if len(b_values) > 1:
                    arr = []
                    for j in range(len(b_values)):
                        arr.append(b_values[j].Value)
                    value = arr
                elif len(b_values) == 1:
                    value = b_values[0].Value
            else:
                b_values = []

            if property.property.propertyIdentifier == BACnetPropertyIds.PROP_OBJECT_IDENTIFIER:
                self.object_identifier = value
            elif property.property.propertyIdentifier == BACnetPropertyIds.PROP_OBJECT_NAME:
                self.object_name = value
            elif property.property.propertyIdentifier == BACnetPropertyIds.PROP_OBJECT_TYPE:
                self.object_type = value
            elif property.property.propertyIdentifier == BACnetPropertyIds.PROP_PRESENT_VALUE:
                self.present_value = value
            elif property.property.propertyIdentifier == BACnetPropertyIds.PROP_UNITS:
                self.unit = value
            elif property.property.propertyIdentifier == BACnetPropertyIds.PROP_DESCRIPTION:
                self.description = value

class BACnetDevice:

    def __init__(self, **kwargs):
        self.adr: BACnetAddress = kwargs.get('adr')
        self.device_id: int = kwargs.get('device_id')
        self.max_apdu: int = kwargs.get('max_apdu')
        self.vendor_id: int = kwargs.get('vendor_id')
        self.segmentation: BACnetSegmentations = kwargs.get('segmentation')
        self.object_list: list = []

    def Update(self, **kwargs):
        if 'adr' in kwargs.keys():
            self.adr: BACnetAddress = kwargs.get('adr')
        if 'device_id' in kwargs.keys():
            self.device_id: int = kwargs.get('device_id')
        if 'max_apdu' in kwargs.keys():
            self.max_apdu: int = kwargs.get('max_apdu')
        if 'vendor_id' in kwargs.keys():
            self.vendor_id: int = kwargs.get('vendor_id')
        if 'segmentation' in kwargs.keys():
            self.segmentation: BACnetSegmentations = kwargs.get('segmentation')

# edit 20210922-20210924
class ASN1:
    BACNET_MAX_OBJECT = 0x3FF
    BACNET_INSTANCE_BITS = 22
    BACNET_MAX_INSTANCE = 0x3FFFFF
    MAX_BITSTRING_BYTES = 15
    BACNET_ARRAY_ALL = 0xFFFFFFFF
    BACNET_NO_PRIORITY = 0
    BACNET_MIN_PRIORITY = 1
    BACNET_MAX_PRIORITY = 16

    @staticmethod
    def encode_bacnet_object_id(buffer: EncodeBuffer, object_type: BACnetObjectTypes, instance: int):
        value = ((object_type & ASN1.BACNET_MAX_OBJECT) << ASN1.BACNET_INSTANCE_BITS) | (instance & ASN1.BACNET_MAX_INSTANCE)
        ASN1.encode_unsigned32(buffer, value)

    @staticmethod
    def encode_tag(buffer: EncodeBuffer, tag_number: int, context_specific: bool, len_value_type: int):
        length = 1
        tem = bytearray(3)
        tem[0] = 0
        if context_specific:
            tem[0] = tem[0] | 0x8
        if tag_number <= 14:
            tem[0] = tem[0] | (tag_number << 4)
        else:
            tem[0] = tem[0] | 0xF0
            tem[1] = tag_number
            length = length + 1

        if len_value_type <= 4:
            tem[0] = tem[0] | len_value_type
            buffer.Add(tem, length)
        else:
            tem[0] = tem[0] | 5
            if len_value_type <= 253:
                length = length + 1
                tem[length] = len_value_type
                buffer.Add(tem, length)
            elif len_value_type <= 65535:
                length = length + 1
                tem[length] = 254
                buffer.Add(tem, length)
                ASN1.encode_unsigned16(buffer, len_value_type)

            else:
                length = length + 1
                tem[length] = 255
                buffer.Add(tem, length)
                ASN1.encode_unsigned16(buffer, len_value_type)

    @staticmethod
    def encode_bacnet_enumerated(buffer: EncodeBuffer, value: int):
        ASN1.encode_bacnet_unsigned(buffer, value)

    @staticmethod
    def encode_application_object_id(buffer: EncodeBuffer, object_type: BACnetObjectTypes, instance: int):
        tmp1 = EncodeBuffer()
        ASN1.encode_bacnet_object_id(tmp1, object_type, instance)
        ASN1.encode_tag(buffer, BACnetApplicationTags.BACNET_APPLICATION_TAG_OBJECT_ID, False, tmp1.offset)
        buffer.Add(tmp1.buffer, tmp1.offset)

    @staticmethod
    def encode_application_unsigned(buffer: EncodeBuffer, value: int):
        tmp1 = EncodeBuffer()
        ASN1.encode_bacnet_unsigned(tmp1, value)
        ASN1.encode_tag(buffer, BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT, False, tmp1.offset)
        buffer.Add(tmp1.buffer, tmp1.offset)

    @staticmethod
    def encode_application_enumerated(buffer: EncodeBuffer, value: int):
        tmp1 = EncodeBuffer()
        ASN1.encode_bacnet_enumerated(tmp1, value)
        ASN1.encode_tag(buffer, BACnetApplicationTags.BACNET_APPLICATION_TAG_ENUMERATED, False, tmp1.offset)
        buffer.Add(tmp1.buffer, tmp1.offset)

    @staticmethod
    def encode_application_signed(buffer: EncodeBuffer, value: int):
        tmp1 = EncodeBuffer()
        ASN1.encode_bacnet_signed(tmp1, value)
        ASN1.encode_tag(buffer, BACnetApplicationTags.BACNET_APPLICATION_TAG_SIGNED_INT, False, tmp1.offset)
        buffer.Add(tmp1.buffer, tmp1.offset)

    @staticmethod
    def encode_bacnet_unsigned(buffer: EncodeBuffer, value: int):
        if value < 0x100:
            buffer.AddByte(value)
        elif value < 0x10000:
            ASN1.encode_unsigned16(buffer, value)
        elif value < 0x1000000:
            ASN1.encode_unsigned24(buffer, value)
        else:
            ASN1.encode_unsigned32(buffer, value)

    @staticmethod
    def encode_context_boolean(buffer: EncodeBuffer, tag_number: int, boolean_value: bool):
        ASN1.encode_tag(buffer, tag_number, True, 1)
        if boolean_value is True:
            buffer.AddByte(1)
        else:
            buffer.AddByte(0)

    @staticmethod
    def encode_context_real(buffer: EncodeBuffer, tag_number: int, value: float):
        ASN1.encode_tag(buffer, tag_number, True, 4)
        ASN1.encode_bacnet_real(buffer, value)

    @staticmethod
    def encode_context_unsigned(buffer: EncodeBuffer, tag_number: int, value: int):
        length = 0
        if value < 0x100:
            length = 1
        elif value < 0x10000:
            length = 2
        elif value < 0x1000000:
            length = 3
        else:
            length = 4

        ASN1.encode_tag(buffer, tag_number, True, length)
        ASN1.encode_bacnet_unsigned(buffer, value)

    @staticmethod
    def encode_context_character_string(buffer: EncodeBuffer, tag_number: int, value: str):
        tmp = EncodeBuffer()
        ASN1.encode_bacnet_character_string(tmp, value)
        ASN1.encode_tag(buffer, tag_number, True, tmp.offset)
        buffer.Add(tmp.buffer, tmp.offset)

    @staticmethod
    def encode_context_enumerated(buffer: EncodeBuffer, tag_number: int, value: int):
        length = 0
        if value < 0x100:
            length = 1
        elif value < 0x10000:
            length = 2
        elif value < 0x1000000:
            length = 3
        else:
            length = 4

        ASN1.encode_tag(buffer, tag_number, True, length)
        ASN1.encode_bacnet_enumerated(buffer, value)

    @staticmethod
    def encode_bacnet_signed(buffer: EncodeBuffer, value: int):
        if value >= -128 and value < 128:
            buffer.AddByte(value)
        elif value >= -32768 and value < 32768:
            ASN1.encode_signed16(buffer, value)
        elif value > -8388607 and value < 8388608:
            ASN1.encode_signed24(buffer, value)
        else:
            ASN1.encode_signed32(buffer, value)

    @staticmethod
    def encode_octet_string(buffer: EncodeBuffer, octet_string: bytearray, octet_offset: int, octet_count: int):
        if octet_string is not None:
            for i in range(octet_offset, (octet_offset + octet_count)):
                buffer.AddByte((octet_string[i]))

    @staticmethod
    def encode_application_octet_string(buffer: EncodeBuffer, octet_string: bytearray, octet_offset: int, octet_count: int):
        ASN1.encode_tag(buffer, BACnetApplicationTags.BACNET_APPLICATION_TAG_OCTET_STRING, False, octet_count)
        ASN1.encode_octet_string(buffer, octet_string, octet_offset, octet_count)

    @staticmethod
    def encode_application_boolean(buffer: EncodeBuffer, boolean_value: bool):
        ASN1.encode_tag(buffer, BACnetApplicationTags.BACNET_APPLICATION_TAG_BOOLEAN, False, 1 if boolean_value else 0)

    @staticmethod
    def encode_bacnet_real(buffer: EncodeBuffer, value: float):
        buffer.Add(bytearray(struct.pack('!f', value)), 4)

    @staticmethod
    def encode_bacnet_double(buffer: EncodeBuffer, value: float):
        buffer.Add(bytearray(struct.pack('!d', value)), 8)
        return bytearray(struct.pack('!d', value))

    @staticmethod
    def encode_application_real(buffer: EncodeBuffer, value: float):
        ASN1.encode_tag(buffer, BACnetApplicationTags.BACNET_APPLICATION_TAG_REAL, False, 4)
        ASN1.encode_bacnet_real(buffer, value)

    @staticmethod
    def encode_application_double(buffer: EncodeBuffer, value: float):
        ASN1.encode_tag(buffer, BACnetApplicationTags.BACNET_APPLICATION_TAG_DOUBLE, False, 8)
        ASN1.encode_bacnet_double(buffer, value)

    @staticmethod
    def bitstring_bytes_used(bit_string: BACnetBitString):
        length = 0
        used_bytes = 0
        last_bit = 0
        if bit_string.bits_used > 0:
            last_bit = bit_string.bits_used - 1
            used_bytes = int(last_bit/8)
            used_bytes = used_bytes + 1
            length = used_bytes
        return length

    @staticmethod
    def byte_reverse_bits(in_byte: int):
        out_byte = 0
        if (in_byte & 1) > 0:
            out_byte = out_byte | 0x80
        elif (in_byte & 2) > 0:
            out_byte = out_byte | 0x40
        elif (in_byte & 4) > 0:
            out_byte = out_byte | 0x20
        elif (in_byte & 8) > 0:
            out_byte = out_byte | 0x10
        elif (in_byte & 16) > 0:
            out_byte = out_byte | 0x8
        elif (in_byte & 32) > 0:
            out_byte = out_byte | 0x4
        elif (in_byte & 64) > 0:
            out_byte = out_byte | 0x2
        elif (in_byte & 128) > 0:
            out_byte = out_byte | 1
        return out_byte

    #TODO
    @staticmethod
    def bitstring_octet(bit_string: BACnetBitString, octet_index: int):
        octet = 0
        if bit_string.value is not None:
            if octet_index < ASN1.MAX_BITSTRING_BYTES:
                octet = bit_string.value[octet_index]
        return octet

    @staticmethod
    def encode_bitstring(buffer: EncodeBuffer, bit_string: BACnetBitString):
        if bit_string.bits_used == 0:
            buffer.AddByte(0)
        else:
            used_bytes = ASN1.bitstring_bytes_used(bit_string)
            remaining_used_bits = BACnetTool.int_to_byte(bit_string.bits_used - (used_bytes - 1) * 8)
            buffer.AddByte(BACnetTool.int_to_byte(8 - remaining_used_bits))
            for i in range(0, used_bytes):
                buffer.AddByte(ASN1.byte_reverse_bits(ASN1.bitstring_octet(bit_string, i)))

    @staticmethod
    def encode_application_bitstring(buffer: EncodeBuffer, bit_string: BACnetBitString):
        bit_string_encoded_length = 1
        bit_string_encoded_length = bit_string_encoded_length + ASN1.bitstring_bytes_used(bit_string)
        ASN1.encode_tag(buffer, BACnetApplicationTags.BACNET_APPLICATION_TAG_BIT_STRING, False, bit_string_encoded_length)
        ASN1.encode_bitstring(buffer, bit_string)

    @staticmethod
    def bacapp_encode_application_data(buffer: EncodeBuffer, value: BACnetValue):
        if value.Value is None:
            buffer.AddByte(BACnetApplicationTags.BACNET_APPLICATION_TAG_NULL)
            return
        if value.Tag == BACnetApplicationTags.NULL:
            pass
        elif value.Tag == BACnetApplicationTags.BOOLEAN:
            ASN1.encode_application_boolean(buffer, value.Value)
            return
        elif value.Tag == BACnetApplicationTags.UNSIGNED_INT:
            ASN1.encode_application_unsigned(buffer, value.Value)
            return
        elif value.Tag == BACnetApplicationTags.SIGNED_INT:
            ASN1.encode_application_signed(buffer, value.Value)
            return
        elif value.Tag == BACnetApplicationTags.REAL:
            ASN1.encode_application_real(buffer, value.Value)
            return
        elif value.Tag == BACnetApplicationTags.DOUBLE:
            ASN1.encode_application_double(buffer, value.Value)
            return
        elif value.Tag == BACnetApplicationTags.OCTET_STRING:
            ASN1.encode_application_octet_string(buffer, value.Value, 0, len(value.Value))
            return
        elif value.Tag == BACnetApplicationTags.CHARACTER_STRING:
            ASN1.encode_application_character_string(buffer, value.Value)
            return
        elif value.Tag == BACnetApplicationTags.BIT_STRING:
            ASN1.encode_application_bitstring(buffer, value.Value)
            return
        elif value.Tag == BACnetApplicationTags.ENUMERATED:
            ASN1.encode_application_enumerated(buffer, value.Value)
            return
        elif value.Tag == BACnetApplicationTags.DATE:
            ASN1.encode_application_date(buffer, value.Value)
            return
        elif value.Tag == BACnetApplicationTags.TIME:
            ASN1.encode_application_time(buffer, value.Value)
            return
        else:
            print("bacapp_encode_application_data missing tag")

    @staticmethod
    def bacapp_encode_device_obj_property_ref(buffer: EncodeBuffer, value: BACnetDeviceObjectPropertyReference):
        ASN1.encode_context_object_id(buffer, 0, value.objectIdentifier.type, value.objectIdentifier.instance)
        ASN1.encode_context_enumerated(buffer, 1, value.propertyIdentifier)
        if value.arrayIndex != ASN1.BACNET_ARRAY_ALL:
            ASN1.encode_context_unsigned(buffer, 2, value.arrayIndex)
        if value.deviceIndentifier.type == BACnetObjectTypes.OBJECT_DEVICE:
            ASN1.encode_context_object_id(buffer, 3, value.deviceIndentifier.type, value.deviceIndentifier.instance)

    @staticmethod
    def bacapp_encode_context_device_obj_property_ref(buffer: EncodeBuffer, tag_number: int, value: BACnetDeviceObjectPropertyReference):
        ASN1.encode_opening_tag(buffer, tag_number)
        ASN1.bacapp_encode_device_obj_property_ref(buffer, value)
        ASN1.encode_closing_tag(buffer, tag_number)

    @staticmethod
    def bacapp_encode_property_state(buffer: EncodeBuffer, value: BACnetPropetyState):
        if value.tag == BACnetPropertyStateTypes.BOOLEAN_VALUE:
            if value.state == 1:
                ASN1.encode_context_boolean(buffer, 0, True)
            else:
                ASN1.encode_context_boolean(buffer, 0, False)
        elif value.tag == BACnetPropertyStateTypes.BINARY_VALUE:
            ASN1.encode_context_enumerated(buffer, 1, value.state)
        elif value.tag == BACnetPropertyStateTypes.EVENT_TYPE:
            ASN1.encode_context_enumerated(buffer, 2, value.state)
        elif value.tag == BACnetPropertyStateTypes.POLARITY:
            ASN1.encode_context_enumerated(buffer, 3, value.state)
        elif value.tag == BACnetPropertyStateTypes.PROGRAM_CHANGE:
            ASN1.encode_context_enumerated(buffer, 4, value.state)
        elif value.tag == BACnetPropertyStateTypes.PROGRAM_STATE:
            ASN1.encode_context_enumerated(buffer, 5, value.state)
        elif value.tag == BACnetPropertyStateTypes.REASON_FOR_HALT:
            ASN1.encode_context_enumerated(buffer, 6, value.state)
        elif value.tag == BACnetPropertyStateTypes.RELIABILITY:
            ASN1.encode_context_enumerated(buffer, 7, value.state)
        elif value.tag == BACnetPropertyStateTypes.STATE:
            ASN1.encode_context_enumerated(buffer, 8, value.state)
        elif value.tag == BACnetPropertyStateTypes.SYSTEM_STATUS:
            ASN1.encode_context_enumerated(buffer, 9, value.state)
        elif value.tag == BACnetPropertyStateTypes.UNITS:
            ASN1.encode_context_enumerated(buffer, 10, value.state)
        elif value.tag == BACnetPropertyStateTypes.UNSIGNED_VALUE:
            ASN1.encode_context_enumerated(buffer, 11, value.state)
        elif value.tag == BACnetPropertyStateTypes.LIFE_SAFETY_MODE:
            ASN1.encode_context_enumerated(buffer, 12, value.state)
        elif value.tag == BACnetPropertyStateTypes.LIFE_SAFETY_STATE:
            ASN1.encode_context_enumerated(buffer, 13, value.state)

    @staticmethod
    def encode_context_bitstring(buffer: EncodeBuffer, tag_number: int, bit_string: BACnetBitString):
        bit_string_encoded_length = 1
        bit_string_encoded_length = bit_string_encoded_length + ASN1.bitstring_bytes_used(bit_string)
        ASN1.encode_tag(buffer, tag_number, False, bit_string_encoded_length)
        ASN1.encode_bitstring(buffer, bit_string)

    @staticmethod
    def encode_opening_tag(buffer: EncodeBuffer, tag_number: int):
        length = 1
        tmp = bytearray(2)
        tmp[0] = 0x8
        if tag_number <= 14:
            tmp[0] = tmp[0] | (tag_number << 4)
        else:
            tmp[0] = tmp[0] | 0xF0
            tmp[1] = tag_number
            length = length + 1
        tmp[0] = tmp[0] | 6
        buffer.Add(tmp, length)

    @staticmethod
    def encode_context_signed(buffer: EncodeBuffer, tag_number: int, value: int):
        length = 0
        if (value >= -128) and (value < 128):
            length = 1
        elif (value >= -32768) and (value < 32768):
            length = 2
        elif (value >= -8388608) and (value < 8388608):
            length = 2
        else:
            length = 4

        ASN1.encode_tag(buffer, tag_number, True, length)
        ASN1.encode_bacnet_signed(buffer, value)

    @staticmethod
    def encode_context_object_id(buffer: EncodeBuffer, tag_number: int, object_type: BACnetObjectTypes, instance: int):
        ASN1.encode_tag(buffer, tag_number, True, 4)
        ASN1.encode_bacnet_object_id(buffer, object_type, instance)

    @staticmethod
    def encode_closing_tag(buffer: EncodeBuffer, tag_number: int):
        length = 1
        tmp = bytearray(2)
        tmp[0] = 0x8
        if tag_number <= 14:
            tmp[0] = tmp[0] | (tag_number << 4)
        else:
            tmp[0] = tmp[0] | 0xF0
            tmp[1] = tag_number
            length = length + 1
        tmp[0] = tmp[0] | 7
        buffer.Add(tmp, length)

    @staticmethod
    def encode_bacnet_time(buffer: EncodeBuffer, value: datetime):
        buffer.AddByte(value.hour)
        buffer.AddByte(value.minute)
        buffer.AddByte(value.second)
        buffer.AddByte((int)(value.microsecond / 10000))

    @staticmethod
    def encode_context_time(buffer: EncodeBuffer, tag_number: int, value: datetime):
        ASN1.encode_tag(buffer, tag_number, True, 4)
        ASN1.encode_bacnet_time(buffer, value)

    @staticmethod
    def encode_bacnet_date(buffer: EncodeBuffer, value: datetime):
        if value == datetime(1, 1, 1):
            buffer.AddByte(0xFF)
            buffer.AddByte(0xFF)
            buffer.AddByte(0xFF)
            buffer.AddByte(0xFF)
            return

        if value.year >= 1900:
            buffer.AddByte(value.year-1900)
        elif value.year < 0x100:
            buffer.AddByte(value.year)
        else:
            return

        buffer.AddByte(value.month)
        buffer.AddByte(value.day)
        buffer.AddByte(value.isoweekday())

    @staticmethod
    def encode_application_date(buffer: EncodeBuffer, value: datetime):
        ASN1.encode_tag(buffer, BACnetApplicationTags.BACNET_APPLICATION_TAG_DATE, False, 4)
        ASN1.encode_bacnet_date(buffer, value)

    @staticmethod
    def encode_application_time(buffer: EncodeBuffer, value: datetime):
        ASN1.encode_tag(buffer, BACnetApplicationTags.BACNET_APPLICATION_TAG_TIME, False, 4)
        ASN1.encode_bacnet_time(buffer, value)

    @staticmethod
    def bacapp_encode_datetime(buffer: EncodeBuffer, value: datetime):
        if value != datetime(1, 1, 1):
            ASN1.encode_application_date(buffer, value)
            ASN1.encode_application_time(buffer, value)

    @staticmethod
    def bacapp_encode_context_datetime(buffer: EncodeBuffer, tag_number: int, value: datetime):
        if value != datetime(1, 1, 1):
            ASN1.encode_opening_tag(buffer, tag_number)
            ASN1.bacapp_encode_datetime(buffer, value)
            ASN1.encode_closing_tag(buffer, tag_number)

    @staticmethod
    def bacapp_encode_timestamp(buffer: EncodeBuffer, value: BACnetGenericTime):
        if value.Tag == BACnetTimestampTags.TIME_STAMP_TIME:
            ASN1.encode_context_time(buffer, 0, value.Time)
        elif value.Tag == BACnetTimestampTags.TIME_STAMP_SEQUENCE:
            ASN1.encode_context_unsigned(buffer, 1, value.Sequence)
        elif value.Tag == BACnetTimestampTags.TIME_STAMP_DATETIME:
            ASN1.bacapp_encode_context_datetime(buffer, 2, value.Time)
        elif value.Tag == BACnetTimestampTags.TIME_STAMP_TIME:
            pass
        else:
            raise Exception('NotImplemented')

    @staticmethod
    def bacapp_encode_context_timestamp(buffer: EncodeBuffer, tag_number: int, value: BACnetGenericTime):
        if value != BACnetTimestampTags.TIME_STAMP_NONE:
            ASN1.encode_opening_tag(buffer, tag_number)
            ASN1.bacapp_encode_timestamp(buffer, value)
            ASN1.encode_closing_tag(buffer, tag_number)

    @staticmethod
    def encode_application_character_string(buffer: EncodeBuffer, value: str):
        tmp = EncodeBuffer()
        ASN1.encode_bacnet_character_string(tmp, value)
        ASN1.encode_tag(buffer, BACnetApplicationTags.BACNET_APPLICATION_TAG_CHARACTER_STRING, False, tmp.offset)
        buffer.Add(tmp.buffer, tmp.offset)

    @staticmethod
    def encode_bacnet_character_string(buffer: EncodeBuffer, value: str):
        buffer.AddByte(BACnetCharacterStringEncodings.CHARACTER_UTF8)
        tmp = bytes(value, "utf-8")
        buffer.Add(bytearray(tmp), len(tmp))

    @staticmethod
    def encode_unsigned16(buffer: EncodeBuffer, value: int):
        buffer.AddByte(int((value & 0xFF00) >> 8))
        buffer.AddByte(int((value & 0x00FF) >> 0))

    @staticmethod
    def encode_unsigned24(buffer: EncodeBuffer, value: int):
        buffer.AddByte(int((value & 0xff0000) >> 16))
        buffer.AddByte(int((value & 0x00ff00) >> 8))
        buffer.AddByte(int((value & 0x0000ff) >> 0))

    @staticmethod
    def encode_unsigned32(buffer: EncodeBuffer, value: int):
        buffer.AddByte(int((value & 0xff000000) >> 24))
        buffer.AddByte(int((value & 0x00ff0000) >> 16))
        buffer.AddByte(int((value & 0x0000ff00) >> 8))
        buffer.AddByte(int((value & 0x000000ff) >> 0))

    @staticmethod
    def encode_signed16(buffer: EncodeBuffer, value: int):
        buffer.AddByte(int((value & 0xff00) >> 8))
        buffer.AddByte(int((value & 0x00ff) >> 0))

    @staticmethod
    def encode_signed24(buffer: EncodeBuffer, value: int):
        buffer.AddByte(int((value & 0xff0000) >> 16))
        buffer.AddByte(int((value & 0x00ff00) >> 8))
        buffer.AddByte(int((value & 0x0000ff) >> 0))

    @staticmethod
    def encode_signed32(buffer: EncodeBuffer, value: int):
        buffer.AddByte(int((value & 0xff000000) >> 24))
        buffer.AddByte(int((value & 0x00ff0000) >> 16))
        buffer.AddByte(int((value & 0x0000ff00) >> 8))
        buffer.AddByte(int((value & 0x000000ff) >> 0))

    @staticmethod
    def encode_read_access_specification(buffer: EncodeBuffer, value: BACnetReadAccessSpecification):
        ASN1.encode_context_object_id(buffer, 0, value.objectIdentifier.type, value.objectIdentifier.instance)
        ASN1.encode_opening_tag(buffer, 1)
        for p in value.propertyReferences:
            ASN1.encode_context_enumerated(buffer, 0, p.propertyIdentifier)
            if p.propertyArrayIndex != ASN1.BACNET_ARRAY_ALL:
                ASN1.encode_context_unsigned(buffer, 1, p.propertyArrayIndex)
        ASN1.encode_closing_tag(buffer, 1)

    @staticmethod
    def encode_read_access_result(buffer: EncodeBuffer, value: BACnetReadAccessResult):
        ASN1.encode_context_object_id(buffer, 0, value.objectIdentifier.type, value.objectIdentifier.instance)
        ASN1.encode_opening_tag(buffer, 1)
        for p_value in value.values:
            ASN1.encode_context_enumerated(buffer, 2, p_value.property.propertyIdentifier)
            if p_value.property.propertyArrayIndex != ASN1.BACNET_ARRAY_ALL:
                ASN1.encode_context_unsigned(buffer, 3, p_value.property.propertyArrayIndex)

            if p_value.value != None and isinstance(p_value.value[0].Value, BACnetError) is True:
                ASN1.encode_opening_tag(buffer, 5)
                ASN1.encode_application_enumerated(buffer, (int)(p_value.value[0].Value.error_class))
                ASN1.encode_application_enumerated(buffer, (int)(p_value.value[0].Value.error_code))
                ASN1.encode_closing_tag(buffer, 5)
            else:
                ASN1.encode_opening_tag(buffer, 4)
                for v in p_value.value:
                    ASN1.bacapp_encode_application_data(buffer, v)
                ASN1.encode_closing_tag(buffer, 4)

    @staticmethod
    def decode_read_access_result(buffer: bytearray, offset: int, apdu_len: int):

        length = 0

        value = BACnetReadAccessResult()
        if not ASN1.decode_is_context_tag(buffer, offset + length, 0):
            return (-1, value)

        length = 1
        (l, value.objectIdentifier.type, value.objectIdentifier.instance) = ASN1.decode_object_id(buffer, offset + length)
        length = length + l

        if not ASN1.decode_is_opening_tag_number(buffer, offset + length, 1):
            return (-1, value)

        length = length + 1
        _value_list = []
        while (apdu_len - length) > 0:
            new_entry = BACnetPropertyValue()
            if ASN1.decode_is_closing_tag_number(buffer, offset + length, 1):
                length = length + 1
                break

            #Tag 2: propertyIdentifier
            (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)

            length = length + l
            if tag_number != 2:
                return (-1, value)

            (l, new_entry.property.propertyIdentifier) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
            length = length + l

            #Tag 3: Optional Array Index
            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            if tag_number == 3:
                length = length + tag_len
                (l, new_entry.property.propertyArrayIndex) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
                length = length + l
            else:
                new_entry.property.propertyArrayIndex = ASN1.BACNET_ARRAY_ALL

            #Tag 4: Value
            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length = length + tag_len
            if tag_number == 4:
                local_value_list = []
                while not ASN1.decode_is_closing_tag_number(buffer, offset + length, 4):
                    tag_len, v = ASN1.bacapp_decode_application_data(buffer, offset + length, apdu_len + offset - 1, value.objectIdentifier.type, new_entry.property.propertyIdentifier)
                    if tag_len < 0:
                        return (-1, value)
                    length = length + tag_len
                    local_value_list.append(v)

                #FC : two values one Date & one Time => change to one datetime
                if len(local_value_list) == 2 and local_value_list[0].Tag == BACnetApplicationTags.BACNET_APPLICATION_TAG_DATE and local_value_list[1].Tag == BACnetApplicationTags.BACNET_APPLICATION_TAG_TIME:
                    date = local_value_list[0].Value
                    time = local_value_list[1].Value
                    bdatetime = datetime(date.year, date.month, date.day, time.hour, time.minute, time.second, time.microsecond)
                    local_value_list.clear()
                    local_value_list.append(BACnetValue(BACnetApplicationTags.BACNET_APPLICATION_TAG_DATETIME, bdatetime))
                    new_entry.value = local_value_list
                else:
                    new_entry.value = local_value_list
                length = length + 1
            elif tag_number == 5:
                err = BACnetError()
                (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length = length + l
                (l, len_value_type) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
                length = length + l
                err.error_class = len_value_type
                (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length = length + l
                (l, len_value_type) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
                length = length + l
                err.error_code = len_value_type
                if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 5):
                    return (-1, value)
                length = length + 1
                new_entry.value = [BACnetValue(BACnetApplicationTags.BACNET_APPLICATION_TAG_ERROR, err)]
            _value_list.append(new_entry)
        value.values = _value_list
        return (length, value)

    @staticmethod
    def decode_read_access_specification(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        value = BACnetReadAccessSpecification()

        #Tag 0: Object ID
        if not ASN1.decode_is_context_tag(buffer, offset + length, 0):
            return (-1, value)
        length = length + 1

        (l, value.objectIdentifier.type, value.objectIdentifier.instance) = ASN1.decode_object_id(buffer, offset + length)
        length = length + l

        #Tag 1: sequence of ReadAccessSpecification
        if not ASN1.decode_is_opening_tag_number(buffer, offset + length, 1):
            return (-1, value)
        length = length + 1

        __property_id_and_array_index = []
        while (apdu_len - length) > 1 and not ASN1.decode_is_closing_tag_number(buffer, offset + length, 1):
            p_ref = BACnetPropertyReference()

            #Tag 0: propertyIdentifier
            if not ASN1.IS_CONTEXT_SPECIFIC(buffer[offset + length]):
                return (-1, value)

            (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length = length + l
            if tag_number != 0:
                return (-1, value)

            if length + len_value_type >= apdu_len:
                return (-1, value)

            (l, p_ref.propertyIdentifier) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
            length = length + l

            p_ref.propertyArrayIndex = ASN1.BACNET_ARRAY_ALL
            if ASN1.IS_CONTEXT_SPECIFIC(buffer[offset + length]) and not ASN1.IS_CLOSING_TAG(buffer[offset + length]):
                (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                if tag_number == 1:
                    length = length + l
                    if length + len_value_type >= apdu_len:
                        return -1
                    (l, p_ref.propertyArrayIndex) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
                    length = length + l
            __property_id_and_array_index.append(p_ref)

        if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 1):
            return (-1, value)
        length = length + 1

        value.propertyReferences = __property_id_and_array_index
        return (length, value)

    @staticmethod
    def decode_device_obj_property_ref(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        value = BACnetDeviceObjectPropertyReference()
        value.arrayIndex = ASN1.BACNET_ARRAY_ALL

        # Tag 0: Object ID
        if not ASN1.decode_is_context_tag(buffer, offset + length, 0):
            return (-1, value)
        length = length + 1

        (l, value.objectIdentifier.type, value.objectIdentifier.instance) = ASN1.decode_object_id(buffer, offset + length)
        length = length + l

        # Tag 1: Property identifier
        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length = length + l
        if tag_number != 1:
            return (-1, value)

        (l, propertyIdentifier) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
        length = length + l
        value.propertyIdentifier = propertyIdentifier

        #Tag 2: Optional Array Index
        (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        if tag_number == 2:
            length = length + tag_len
            (l, value.arrayIndex) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
            length = length + l

        #Tag 3 : Optional Device Identifier
        if not ASN1.decode_is_context_tag(buffer, offset + length, 3):  #length=7
            return (length, value)
        if ASN1.IS_CLOSING_TAG(buffer[offset+length]):
            return (length, value)

        length = length + 1

        (l, value.objectIdentifier.type, value.objectIdentifier.instance) = ASN1.decode_object_id(buffer, offset + length)
        length = length + l
        return (length, value)

    @staticmethod
    def decode_unsigned(buffer: bytearray, offset: int, len_value: int):
        value = 0
        if len_value == 1:
            value = buffer[offset]
        elif len_value == 2:
            (l, value) = ASN1.decode_unsigned16(buffer, offset)
        elif len_value == 3:
            (l, value) = ASN1.decode_unsigned24(buffer, offset)
        elif len_value == 4:
            (l, value) = ASN1.decode_unsigned32(buffer, offset)
        return (len_value, value)

    @staticmethod
    def decode_unsigned32(buffer: bytearray, offset: int):
        value = (int)((buffer[offset+0] << 24) & 0xff000000)
        value = value | (int)((buffer[offset+1] << 16) & 0x00ff0000)
        value = value | (int)((buffer[offset + 2] << 8) & 0x0000ff00)
        value = value | (int)((buffer[offset + 3]) & 0x000000ff)
        return (4, value)

    @staticmethod
    def decode_unsigned24(buffer: bytearray, offset: int):
        value = (int)((buffer[offset + 0] << 16) & 0x00ff0000)
        value = value | (int)((buffer[offset + 1] << 8) & 0x0000ff00)
        value = value | (int)((buffer[offset + 2]) & 0x000000ff)
        return (3, value)

    @staticmethod
    def decode_unsigned16(buffer: bytearray, offset: int):
        value = BACnetTool.int_to_ushort((buffer[offset + 0] << 8) & 0x0000ff00)
        value = value | BACnetTool.int_to_ushort((buffer[offset + 1]) & 0x000000ff)
        return (2, value)

    @staticmethod
    def decode_unsigned8(buffer: bytearray, offset: int):
        value = buffer[offset + 0]
        return (3, value)

    @staticmethod
    def decode_signed32(buffer: bytearray, offset: int):
        value = (int)((buffer[offset + 0] << 24) & 0xff000000)
        value = value | (int)((buffer[offset + 1] << 16) & 0x00ff0000)
        value = value | (int)((buffer[offset + 2] << 8) & 0x0000ff00)
        value = value | (int)((buffer[offset + 3]) & 0x000000ff)
        return (4, value)

    @staticmethod
    def decode_signed24(buffer: bytearray, offset: int):
        value = (int)((buffer[offset + 0] << 16) & 0x00ff0000)
        value = value | (int)((buffer[offset + 1] << 8) & 0x0000ff00)
        value = value | (int)((buffer[offset + 2]) & 0x000000ff)
        return (3, value)

    @staticmethod
    def decode_signed16(buffer: bytearray, offset: int):
        value = (int)((buffer[offset + 0] << 8) & 0x0000ff00)
        value = value | (int)((buffer[offset + 1]) & 0x000000ff)
        return (2, value)

    @staticmethod
    def decode_signed8(buffer: bytearray, offset: int):
        value = buffer[offset + 0]
        return (3, value)

    @staticmethod
    def IS_EXTENDED_TAG_NUMBER(x):
        return ((x & 0xF0) == 0xF0)

    @staticmethod
    def IS_EXTENDED_VALUE(x):
        return ((x & 0x07) == 5)

    @staticmethod
    def IS_CONTEXT_SPECIFIC(x):
        return ((x & 0x8) == 0x8)

    @staticmethod
    def IS_OPENING_TAG(x):
        return ((x & 0x07) == 6)

    @staticmethod
    def IS_CLOSING_TAG(x):
        return ((x & 0x07) == 7)

    @staticmethod
    def decode_tag_number(buffer: bytearray, offset: int):
        length = 1
        # decode the tag number first
        if ASN1.IS_EXTENDED_TAG_NUMBER(buffer[offset]):
            # extended tag
            tag_number = buffer[offset + length]
            length = length + 1
        else:
            tag_number = BACnetTool.int_to_byte(buffer[offset] >> 4)
        return (length, tag_number)

    @staticmethod
    def decode_signed(buffer: bytearray, offset: int, len_value: int):
        value = 0
        if len_value == 1:
            value = ASN1.decode_signed8(buffer, offset)
        elif len_value == 2:
            value = ASN1.decode_signed16(buffer, offset)
        elif len_value == 3:
            value = ASN1.decode_signed24(buffer, offset)
        elif len_value == 4:
            value = ASN1.decode_signed32(buffer, offset)
        return (len_value, value)

    @staticmethod
    def decode_real(buffer: bytearray, offset: int):
        return (4, struct.unpack("!f", buffer[offset:(offset + 4)])[0])

    @staticmethod
    def decode_real_safe(buffer: bytearray, offset: int, len_value: int):
        if len_value != 4:
            value = 0.0
            return (len_value, value)
        else:
            (leng, value) = ASN1.decode_real(buffer, offset)
        return (leng, value)

    @staticmethod
    def decode_double(buffer: bytearray, offset: int):
        return (8, struct.unpack("!d", buffer[offset:(offset + 8)])[0])

    @staticmethod
    def decode_double_safe(buffer: bytearray, offset: int, len_value: int):
        if (len_value != 8):
            value = 0.0
            return (len_value, value)
        else:
            (leng, value) = ASN1.decode_double(buffer, offset)
        return (leng, value)

    @staticmethod
    def octetstring_copy(buffer: bytearray, offset: int, max_offset: int, octet_string: bytearray, octet_string_offset: int, octet_string_length: int):
        status = False
        if octet_string_length <= max_offset + offset:
            if octet_string is not None:
                length = min(len(octet_string), len(buffer) - offset)
                octet_string[octet_string_offset:length] = buffer[offset: offset + length]
                status = True
        return status

    @staticmethod
    def decode_octet_string(buffer: bytearray, offset: int, max_length: int, octet_string: bytearray, octet_string_offset: int, octet_string_length: int):
        ASN1.octetstring_copy(buffer, offset, max_length, octet_string, octet_string_offset, octet_string_length)
        return (octet_string_length, octet_string)

    @staticmethod
    def decode_context_octet_string(buffer: bytearray, offset: int, max_length: int, tag_number: int, octet_string: bytearray, octet_string_offset: int):
        length = 0
        if ASN1.decode_is_context_tag(buffer, offset, tag_number):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length = length + l
            if ASN1.octetstring_copy(buffer, offset + length, max_length, octet_string, octet_string_offset, len_value):
                length = length + len_value
        else:
            length = -1
        return (length, buffer)

    @staticmethod
    def multi_charset_characterstring_decode(buffer: bytearray, offset: int, max_length: int, encoding: int, length: int):
        char_string = ""
        e = 'utf_8'  # default
        if encoding == BACnetCharacterStringEncodings.CHARACTER_UTF8:
            e = 'utf_8'
        elif encoding == BACnetCharacterStringEncodings.CHARACTER_UCS2:
            if (buffer[offset] == 0xFF) and (buffer[offset + 1] == 0xFE):
                e = 'utf_16'
            else:
                e = 'utf_16be'
        elif encoding == BACnetCharacterStringEncodings.CHARACTER_UCS4:
            if (buffer[offset] == 0xFF) and (buffer[offset + 1] == 0xFE) and (buffer[offset + 2] == 0) and (buffer[offset + 3] == 0):
                e = 'utf_32'  # probaby doesn't exist
            else:
                e = 'utf_32be'
        elif encoding == BACnetCharacterStringEncodings.CHARACTER_ISO8859:
            e = 'latin_1'
        elif encoding == BACnetCharacterStringEncodings.CHARACTER_MS_DBCS:
            e = 'shift_jisx0213'
        elif encoding == BACnetCharacterStringEncodings.CHARACTER_JISX_0208:
            e = 'shift_jisx0213'
        elif encoding == BACnetCharacterStringEncodings.CHARACTER_MS_DBCS:
            e = 'dbcs'

        char_string = buffer[offset: offset+length].decode(e)

        return (True, char_string)

    @staticmethod
    def decode_character_string(buffer: bytearray, offset: int, max_length: int, len_value: int):
        length = 0
        status = False

        (status, char_string) = ASN1.multi_charset_characterstring_decode(buffer, offset + 1, max_length, buffer[offset], len_value - 1)
        if (status):
            length = len_value
        return (length, char_string)

    @staticmethod
    def bitstring_set_octet(bit_string: BACnetBitString, index: int, octet: int):
        status = False
        if index < ASN1.MAX_BITSTRING_BYTES:
            bit_string.value[index] = octet
            status = True
        return status

    @staticmethod
    def bitstring_set_bits_used(bit_string: BACnetBitString, bytes_used: int, unused_bits: int):
        status = False
        bit_string.bits_used = BACnetTool.int_to_byte(bytes_used * 8)
        bit_string.bits_used = bit_string.bits_used - unused_bits
        status = True
        return status

    @staticmethod
    def decode_bitstring(buffer: bytearray, offset: int, len_value: int):
        length = 0
        bit_string = BACnetBitString()
        bit_string.value = bytearray(ASN1.MAX_BITSTRING_BYTES)
        if len_value > 0:
            bytes_used = len_value - 1
            if bytes_used <= ASN1.MAX_BITSTRING_BYTES:
                length = 1
                for i in range(0, bytes_used):
                    length = length + 1
                    ASN1.bitstring_set_octet(bit_string, i, ASN1.byte_reverse_bits(buffer[offset+length]))
                unused_bits = buffer[offset] & 0x07
                ASN1.bitstring_set_bits_used(bit_string, bytes_used, unused_bits)
        return length, bit_string

    @staticmethod
    def decode_context_character_string(buffer: bytearray, offset: int, max_length: int, tag_number: int):
        length = 0
        char_string = ""
        if ASN1.decode_is_context_tag(buffer, offset + length, tag_number):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length = length + l
            (status, char_string) = ASN1.multi_charset_characterstring_decode(buffer, offset + 1 + length, max_length, buffer[offset + length], len_value - 1)
            if status:
                length = length + len_value
        else:
            length = -1

        return (length, char_string)

    @staticmethod
    def decode_date(buffer: bytearray, offset: int):
        (year, month, day, wday) = struct.unpack('BBBB', buffer[offset:(offset + 4)])
        if (month == 0xFF and day == 0xFF and wday == 0xFF and year == 0xFF):
            bdate = d_date(1, 1, 1)
        else:
            bdate = d_date(year + 1900, month, day)
        return (4, bdate)

    @staticmethod
    def decode_date_safe(buffer: bytearray, offset: int, len_value: int):
        if (len_value != 4):
            return (len_value, d_date(1, 1, 1))
        else:
            return ASN1.decode_date(buffer, offset)

    @staticmethod
    def decode_bacnet_time(buffer: bytearray, offset: int):
        (hour, min, sec, hundredths) = struct.unpack('BBBB', buffer[offset:(offset + 4)])
        if (hour == 0xFF and min == 0xFF and sec == 0xFF and hundredths == 0xFF):
            btime = d_time(1, 1, 1, 1)
        else:
            if (hundredths > 100):
                hundredths = 0
            btime = d_time(hour, min, sec, hundredths * 10000)
        return (4, btime)

    @staticmethod
    def decode_bacnet_time_safe(buffer: bytearray, offset: int, len_value: int):
        if (len_value != 4):
            return (len_value, d_time(1, 1, 1, 1))
        else:
            return ASN1.decode_bacnet_time(buffer, offset)

    @staticmethod
    def decode_bacnet_datetime(buffer: bytearray, offset: int):
        length = 0
        (l, date) = ASN1.decode_application_date(buffer, offset + length)
        length = length + l
        (l, time) = ASN1.decode_application_time(buffer, offset + length)
        length = length + l
        bdatetime = datetime(date.year, date.month, date.day, time.hour, time.minute, time.second, time.microsecond)
        return (length, bdatetime)

    @staticmethod
    def decode_object_id(buffer: bytearray, offset: int):
        (length, value) = ASN1.decode_unsigned32(buffer, offset)
        object_instance = (value & ASN1.BACNET_MAX_INSTANCE)
        object_type = BACnetTool.int_to_ushort((value >> ASN1.BACNET_INSTANCE_BITS) & ASN1.BACNET_MAX_OBJECT)
        return (length, object_type, object_instance)

    @staticmethod
    def decode_object_id_safe(buffer: bytearray, offset: int, len_value: int):
        if len_value != 4:
            return (0, 0, 0)
        else:
            return ASN1.decode_object_id(buffer, offset)

    @staticmethod
    def decode_context_object_id(buffer: bytearray, offset: int, tag_number: int):
        length = 0
        (length, is_context) = ASN1.decode_is_context_tag_with_length(buffer, offset + length, tag_number)

        if is_context:
            (l, object_type, instance) = ASN1.decode_object_id(buffer, offset + length)
            length = length + l
        else:
            object_type = 0
            instance = 0
            length = -1

        return (length, object_type, instance)

    @staticmethod
    def decode_application_time(buffer: bytearray, offset: int):
        length = 0
        (l, tag_number) = ASN1.decode_tag_number(buffer, offset + length)
        length = length + l

        if tag_number == BACnetApplicationTags.TIME:
            (l, btime) = ASN1.decode_bacnet_time(buffer, offset + length)
            length = length + l
        else:
            btime = d_time(1, 1, 1)
            length = -1
        return (length, btime)

    @staticmethod
    def decode_application_date(buffer: bytearray, offset: int):

        length = 0
        (l, tag_number) = ASN1.decode_tag_number(buffer, offset + length)
        length = length + l
        if tag_number == BACnetApplicationTags.DATE:
            (l, bdate) = ASN1.decode_date(buffer, offset + length)
            length = length + l
        else:
            bdate = d_date(1, 1, 1)
            length = -1
        return (length, bdate)

    @staticmethod
    def decode_is_context_tag_with_length(buffer: bytearray, offset: int, tag_number: int):
        (tag_length, my_tag_number) = ASN1.decode_tag_number(buffer, offset)
        return (tag_length, ((ASN1.IS_CONTEXT_SPECIFIC(buffer[offset]) and (my_tag_number == tag_number))))

    @staticmethod
    def bacapp_decode_data(buffer: bytearray, offset: int, max_length: int, tag_data_type: BACnetApplicationTags, len_value_type: int):
        length = 0
        value = BACnetValue()
        value.Tag = tag_data_type
        if tag_data_type == BACnetApplicationTags.BACNET_APPLICATION_TAG_NULL:
            pass
        elif tag_data_type == BACnetApplicationTags.BACNET_APPLICATION_TAG_BOOLEAN:
            if len_value_type > 0:
                value.Value = True
            else:
                value.Value = False
        elif tag_data_type == BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT:
            (length, value.Value) = ASN1.decode_unsigned(buffer, offset, len_value_type)
        elif tag_data_type == BACnetApplicationTags.BACNET_APPLICATION_TAG_SIGNED_INT:
            (length, value.Value) = ASN1.decode_signed(buffer, offset, len_value_type)
        elif tag_data_type == BACnetApplicationTags.BACNET_APPLICATION_TAG_REAL:
            (length, value.Value) = ASN1.decode_real_safe(buffer, offset, len_value_type)
        elif tag_data_type == BACnetApplicationTags.BACNET_APPLICATION_TAG_DOUBLE:
            (length, value.Value) = ASN1.decode_double_safe(buffer, offset, len_value_type)
        elif tag_data_type == BACnetApplicationTags.BACNET_APPLICATION_TAG_OCTET_STRING:
            octet_string = bytearray(len_value_type)
            (length, value.Value) = ASN1.decode_octet_string(buffer, offset, max_length, octet_string, 0, len_value_type)
        elif tag_data_type == BACnetApplicationTags.BACNET_APPLICATION_TAG_CHARACTER_STRING:
            (length, value.Value) = ASN1.decode_character_string(buffer, offset, max_length, len_value_type)
        elif tag_data_type == BACnetApplicationTags.BACNET_APPLICATION_TAG_BIT_STRING:
            (length, value.Value) = ASN1.decode_bitstring(buffer, offset, len_value_type)
        elif tag_data_type == BACnetApplicationTags.BACNET_APPLICATION_TAG_ENUMERATED:
            (length, value.Value) = ASN1.decode_enumerated(buffer, offset, len_value_type)
        elif tag_data_type == BACnetApplicationTags.BACNET_APPLICATION_TAG_DATE:
            (length, value.Value) = ASN1.decode_date_safe(buffer, offset, len_value_type)
        elif tag_data_type == BACnetApplicationTags.BACNET_APPLICATION_TAG_TIME:
            (length, value.Value) = ASN1.decode_bacnet_time_safe(buffer, offset, len_value_type)
        elif tag_data_type == BACnetApplicationTags.BACNET_APPLICATION_TAG_OBJECT_ID:
            (length, object_type, instance) = ASN1.decode_object_id_safe(buffer, offset, len_value_type)
            value.Value = BACnetObjectId(object_type, instance)
        return(length, value)

    @staticmethod
    def bacapp_context_tag_type(property: BACnetPropertyIds, tag_number: int):
        tag = BACnetApplicationTags.MAX_BACNET_APPLICATION_TAG
        if property == BACnetPropertyIds.PROP_ACTUAL_SHED_LEVEL or property == BACnetPropertyIds.PROP_REQUESTED_SHED_LEVEL or property == BACnetPropertyIds.PROP_EXPECTED_SHED_LEVEL:
            if tag_number == 0 or tag_number == 1:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT
            elif tag_number == 2:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_REAL
        elif property == BACnetPropertyIds.PROP_ACTION:
            if tag_number == 0 or tag_number == 1:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_OBJECT_ID
            elif tag_number == 2:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_ENUMERATED
            elif tag_number == 3 or tag_number == 5 or tag_number == 6:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT
            elif tag_number == 7 or tag_number == 8:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_BOOLEAN
        elif property == BACnetPropertyIds.PROP_EXCEPTION_SCHEDULE:
            if tag_number == 1:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_OBJECT_ID
            elif tag_number == 3:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT
        elif property == BACnetPropertyIds.PROP_LOG_DEVICE_OBJECT_PROPERTY:
            if tag_number == 0 or tag_number == 3:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_OBJECT_ID
            elif tag_number == 1:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_ENUMERATED
            elif tag_number == 2:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT
        elif property == BACnetPropertyIds.PROP_SUBORDINATE_LIST:
            if tag_number == 0 or tag_number == 1:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_OBJECT_ID
        elif property == BACnetPropertyIds.PROP_ACTIVE_COV_SUBSCRIPTIONS:
            if tag_number == 0 or tag_number == 1:
                pass
            elif tag_number == 2:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_BOOLEAN
            elif tag_number == 3:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT
            elif tag_number == 4:
                tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_REAL
        return tag

    @staticmethod
    def bacapp_decode_context_data(buffer: bytearray, offset: int, max_apdu_len: int, property_tag: BACnetApplicationTags):
        length = 0
        apdu_len = 0
        value = BACnetValue()
        if ASN1.IS_CONTEXT_SPECIFIC(buffer[offset]):
            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            apdu_len = tag_len
            if tag_len > 0 and tag_len <= max_apdu_len and not ASN1.decode_is_closing_tag_number(buffer, offset + length, tag_number):
                if property_tag < BACnetApplicationTags.MAX_BACNET_APPLICATION_TAG:
                    (length, value) = ASN1.bacapp_decode_data(buffer, offset + apdu_len, max_apdu_len, property_tag, len_value_type)
                    apdu_len = apdu_len + length
                elif len_value_type > 0:
                    apdu_len = apdu_len + len_value_type
                else:
                    apdu_len = -1
            elif tag_len == 1:
                apdu_len = 0
        return (apdu_len, value)

    @staticmethod
    def bacapp_decode_application_data(buffer: bytearray, offset: int, max_offset: int, object_type: Union[BACnetObjectTypes, int], property_id: Union[BACnetPropertyIds, int]):
        length = 0
        value = BACnetValue()
        if not ASN1.IS_CONTEXT_SPECIFIC(buffer[offset]):
            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            if tag_len > 0:
                length = length + tag_len
                (decode_len, value) = ASN1.bacapp_decode_data(buffer, offset + length, max_offset, BACnetApplicationTags(tag_number), len_value_type)
                if decode_len < 0:
                    return (decode_len, value)
                length = length + decode_len
        else:
            return ASN1.bacapp_decode_context_application_data(buffer, offset, max_offset, object_type, property_id)
        return (length, value)

    @staticmethod
    def bacapp_decode_context_application_data(buffer: bytearray, offset: int, max_offset: int, object_type: BACnetObjectTypes, property_id: BACnetPropertyIds):
        length = 0
        value = BACnetValue()

        if ASN1.IS_CONTEXT_SPECIFIC(buffer[offset]):
            if property_id == BACnetPropertyIds.PROP_LIST_OF_GROUP_MEMBERS:
                (tag_len, v) = ASN1.decode_read_access_specification(buffer, offset, max_offset)
                if tag_len < 0:
                    return (-1, value)
                value.Tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_READ_ACCESS_SPECIFICATION
                value.Value = v
                return (tag_len, value)
            elif property_id == BACnetPropertyIds.PROP_ACTIVE_COV_SUBSCRIPTIONS:
                (tag_len, v) = ASN1.decode_cov_subscription(buffer, offset, max_offset)
                if tag_len < 0:
                    return (-1, value)
                value.Tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_COV_SUBSCRIPTION
                value.Value = v
                return (tag_len, value)
            elif object_type == BACnetObjectTypes.OBJECT_GROUP and property_id == BACnetPropertyIds.PROP_PRESENT_VALUE:
                (tag_len, v) = ASN1.decode_read_access_result(buffer, offset, max_offset)
                if tag_len < 0:
                    return (-1, value)
                value.Tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_READ_ACCESS_RESULT
                value.Value = v
                return (tag_len, value)
            elif property_id == BACnetPropertyIds.PROP_LIST_OF_OBJECT_PROPERTY_REFERENCES or property_id == BACnetPropertyIds.PROP_LOG_DEVICE_OBJECT_PROPERTY:
                (tag_len, v) = ASN1.decode_device_obj_property_ref(buffer, offset, max_offset)
                if tag_len < 0:
                    return (-1, value)
                value.Tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_OBJECT_PROPERTY_REFERENCE
                value.Value = v
                return (tag_len, value)
            elif property_id == BACnetPropertyIds.PROP_DATE_LIST:
                v = BACnetCalendarEntry()
                tag_len = v.ASN1decode(buffer, offset, max_offset)
                if tag_len < 0:
                    return (-1, value)
                value.Tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_CONTEXT_SPECIFIC_DECODED
                value.Value = v
                return (tag_len, value)
            elif property_id == BACnetPropertyIds.PROP_SUBORDINATE_LIST:
                v = BACnetDeviceObjectReference()
                tag_len = v.ASN1decode(buffer, offset, max_offset)
                if tag_len < 0:
                    return (-1, value)
                value.Tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_DEVICE_OBJECT_REFERENCE
                value.Value = v
                return (tag_len, value)
            elif property_id == BACnetPropertyIds.PROP_EVENT_TIME_STAMPS:
                (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length = length + 1
                if tag_number == 0:
                    (l, v) = ASN1.decode_bacnet_time(buffer, offset + length)
                    value.Tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_TIMESTAMP
                    value.Value = v
                elif tag_number == 1:
                    (l, v) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
                    value.Tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT
                    value.Value = v
                elif tag_number == 2:
                    (l, v) = ASN1.decode_bacnet_datetime(buffer, offset + length)
                    value.Tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_TIMESTAMP
                    length = length + 1
                    value.Value = v
                else:
                    return (-1, value)
                return (length, value)

            value.Tag = BACnetApplicationTags.BACNET_APPLICATION_TAG_CONTEXT_SPECIFIC_DECODED
            values = []

            (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            MultiplValue = ASN1.IS_OPENING_TAG(buffer[offset + length])

            while (length + offset) <= max_offset and ASN1.IS_OPENING_TAG(buffer[offset + length]):
                (tag_len, sub_tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                if tag_len < 0:
                    return (-1, value)
                if len_value_type == 0:
                    length = length + tag_len
                    (tag_len, sub_value) = ASN1.bacapp_decode_application_data(buffer, offset + length, max_offset, BACnetObjectTypes.MAX_BACNET_OBJECT_TYPE, BACnetPropertyIds.MAX_BACNET_PROPERTY_ID)
                    if tag_len < 0:
                        return (-1, value)
                    values.append(sub_value)
                    length = length + tag_len
                else:
                    sub_value = BACnetValue()
                    override_tag_number = ASN1.bacapp_context_tag_type(property_id, sub_tag_number)
                    if override_tag_number != BACnetApplicationTags.MAX_BACNET_APPLICATION_TAG:
                        sub_tag_number = override_tag_number
                    (sub_tag_len, sub_value) = ASN1.bacapp_decode_data(buffer, offset + length + tag_len, max_offset, sub_tag_number, len_value_type)
                    if sub_tag_len == len_value_type:
                        values.append(sub_value)
                        length = length + tag_len + len_value_type
                    else:
                        context_specific = bytearray(len_value_type)
                        context_specific[0:len_value_type] = buffer[offset+length+tag_len: offset+length+tag_len+len_value_type]
                        sub_value = BACnetValue(BACnetApplicationTags.BACNET_APPLICATION_TAG_CONTEXT_SPECIFIC_ENCODED, context_specific)
                        values.append(sub_value)
                        length = length + tag_len + len_value_type

                if MultiplValue is False:
                    value = values[0]
                    return (length, value)

            if ASN1.decode_is_closing_tag_number(buffer, offset + length, tag_number):
                length = length + 1

            value.Value = values
        else:
            return (-1, value)
        return (length, value)

    @staticmethod
    def decode_enumerated(buffer: bytearray, offset: int, len_value: int):
        return ASN1.decode_unsigned(buffer, offset, len_value)

    @staticmethod
    def decode_is_context_tag(buffer: bytearray, offset: int, tag_number: int):
        (leng, my_tag_number) = ASN1.decode_tag_number(buffer, offset)
        return ASN1.IS_CONTEXT_SPECIFIC(buffer[offset]) and (my_tag_number == tag_number)

    @staticmethod
    def decode_is_opening_tag_number(buffer: bytearray, offset: int, tag_number: int):
        (leng, my_tag_number) = ASN1.decode_tag_number(buffer, offset)
        return (ASN1.IS_OPENING_TAG(buffer[offset]) and (my_tag_number == tag_number))

    @staticmethod
    def decode_is_closing_tag_number(buffer: bytearray, offset: int, tag_number: int):
        (leng, my_tag_number) = ASN1.decode_tag_number(buffer, offset)
        return (ASN1.IS_CLOSING_TAG(buffer[offset]) and (my_tag_number == tag_number))

    @staticmethod
    def decode_is_closing_tag(buffer: bytearray, offset: int):
        return (buffer[offset] & 0x07) == 7

    @staticmethod
    def decode_is_opening_tag(buffer: bytearray, offset: int):
        return (buffer[offset] & 0x07) == 6

    @staticmethod
    def decode_tag_number_and_value(buffer: bytearray, offset: int):
        (length, tag_number) = ASN1.decode_tag_number(buffer, offset)
        if ASN1.IS_EXTENDED_VALUE(buffer[offset]):
            if buffer[offset + length] == 255:
                length = length + 1
                (l, value32) = ASN1.decode_unsigned(buffer, offset + length, 4)
                length = length + l
                value = value32
            elif buffer[offset + length] == 254:
                length = length + 1
                (l, value16) = ASN1.decode_unsigned(buffer, offset + length, 2)
                length = length + l
                value = value16
            else:
                value = buffer[offset + length]
                length = length + 1
        elif ASN1.IS_OPENING_TAG(buffer[offset]):
            value = 0
        elif ASN1.IS_CLOSING_TAG(buffer[offset]):
            value = 0
        else:
            value = buffer[offset] & 0x07

        return (length, tag_number, value)

    @staticmethod
    def encode_cov_subscription(buffer: EncodeBuffer, value: BACnetCOVSubscription):
        ASN1.encode_opening_tag(buffer, 0)
        ASN1.encode_opening_tag(buffer, 0)
        ASN1.encode_opening_tag(buffer, 1)
        ASN1.encode_application_unsigned(buffer, value.Recipient.net)
        if value.Recipient.net == 0xFFFF:
            ASN1.encode_application_octet_string(buffer, bytearray(0), 0, 0)
        else:
            ASN1.encode_application_octet_string(buffer, value.Recipient.adr, 0, len(value.Recipient.adr))
        ASN1.encode_closing_tag(buffer, 1)
        ASN1.encode_closing_tag(buffer, 0)
        ASN1.encode_context_unsigned(buffer, 1, value.subscriptionProcessIdentifier)
        ASN1.encode_closing_tag(buffer, 0)
        ASN1.encode_opening_tag(buffer, 1)
        ASN1.encode_context_object_id(buffer, 0, value.monitoredObjectIdentifier.type, value.monitoredObjectIdentifier.instance)
        ASN1.encode_context_enumerated(buffer, 1, value.monitoredProperty.propertyIdentifier)
        if value.monitoredProperty.propertyArrayIndex != ASN1.BACNET_ARRAY_ALL:
            ASN1.encode_context_unsigned(buffer, 2, value.monitoredProperty.propertyArrayIndex)
        ASN1.encode_closing_tag(buffer, 1)
        ASN1.encode_context_boolean(buffer, 2, value.IssueConfirmedNotifications)
        ASN1.encode_context_unsigned(buffer, 3, value.TimeRemaining)
        if value.COVIncrement > 0:
            ASN1.encode_context_real(buffer, 4, value.COVIncrement)

    @staticmethod
    def decode_cov_subscription(buffer: bytearray, offset: int, apdu_len: int):
        length = 0

        value = BACnetCOVSubscription()
        value.Recipient = BACnetAddress(type=BACnetAddressTypes.NONE, net=0, adr=None)
        if not ASN1.decode_is_opening_tag_number(buffer, offset + length, 0):
            return (-1, value)

        length = length + 1
        if not ASN1.decode_is_opening_tag_number(buffer, offset + length, 0):
            return (-1, value)

        length = length + 1
        if not ASN1.decode_is_opening_tag_number(buffer, offset + length, 1):
            return (-1, value)

        length = length + 1
        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length = length + l
        if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT:
            return (-1, value)

        (l, tmp) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
        length = length + l
        value.Recipient.net = BACnetTool.int_to_ushort(tmp)

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length = length + l
        if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_OCTET_STRING:
            return (-1, value)
        value.Recipient.adr = bytearray(len_value_type)
        (l, tmp) = ASN1.decode_octet_string(buffer, offset + length, apdu_len, value.Recipient.adr, 0, len_value_type)
        length = length + l
        if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 1):
            return (-1, value)
        length = length + 1
        if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 0):
            return (-1, value)
        length = length + 1

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length = length + l
        if tag_number != 1:
            return (-1, value)
        (l, value.subscriptionProcessIdentifier) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
        length = length + l
        if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 0):
            return (-1, value)
        length = length + 1

        if not ASN1.decode_is_opening_tag_number(buffer, offset + length, 1):
            return (-1, value)
        length = length + 1

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length = length + l
        if tag_number != 0:
            return (-1, value)
        (l, value.monitoredObjectIdentifier.type, value.monitoredObjectIdentifier.instance) = ASN1.decode_object_id(buffer, offset + length)
        length = length + l
        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length = length + l
        if tag_number != 1:
            return (-1, value)

        (l, value.monitoredProperty.propertyIdentifier) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
        length = length + l
        (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        if tag_len == 2:
            length = length + tag_len
            (l, value.monitoredProperty.propertyArrayIndex) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
            length = length + l
        else:
            value.monitoredProperty.propertyArrayIndex = ASN1.BACNET_ARRAY_ALL

        if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 1):
            return (-1, value)
        length = length + 1

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length = length + l
        if tag_number != 2:
            return (-1, value)
        if buffer[offset + length] > 0:
            value.IssueConfirmedNotifications = True
        else:
            value.IssueConfirmedNotifications = False
        length = length + 1

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length = length + l
        if tag_number != 3:
            return (-1, value)
        (l, value.TimeRemaining) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
        length = length + l

        if length < apdu_len and not ASN1.IS_CLOSING_TAG(buffer[offset + length]):
            (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            if tag_number != 4:
                return (length, value)
            length = length + 1
            (l, value.COVIncrement) = ASN1.decode_real(buffer, offset + length)
            length = length + l
        return (length, value)

# edit 20210924
class APDU:

    @staticmethod
    def GetDecodedType(buffer: bytearray, offset: int):
        return BACnetPduTypes(buffer[offset])

    @staticmethod
    def EncodeConfirmedServiceRequest(buffer: EncodeBuffer, type: BACnetPduTypes, service: BACnetConfirmedServices, max_segments: BACnetMaxSegments, max_adpu: BACnetMaxAdpu, invoke_id: int, sequence_number: int=0, proposed_window_size: int=0):
        buffer.buffer[buffer.offset] = type
        buffer.offset = buffer.offset + 1
        buffer.buffer[buffer.offset] = BACnetTool.int_to_byte(BACnetTool.int_to_byte(max_segments) | BACnetTool.int_to_byte(max_adpu))
        buffer.offset = buffer.offset + 1
        buffer.buffer[buffer.offset] = invoke_id
        buffer.offset = buffer.offset + 1

        if type & BACnetPduTypes.SEGMENTED_MESSAGE > 0:
            buffer.buffer[buffer.offset] = sequence_number
            buffer.offset = buffer.offset + 1
            buffer.buffer[buffer.offset] = proposed_window_size
            buffer.offset = buffer.offset + 1

        buffer.buffer[buffer.offset] = service
        buffer.offset = buffer.offset + 1

    @staticmethod
    def DecodeConfirmedServiceRequest(buffer: bytearray, offset: int):
        org_offset = offset
        type = BACnetPduTypes(buffer[offset])
        offset = offset + 1
        max_segments = BACnetMaxSegments(buffer[offset] & 0xF0)
        max_adpu = BACnetMaxAdpu(buffer[offset] & 0x0F)
        offset = offset + 1
        invoke_id = buffer[offset]
        offset = offset + 1

        sequence_number = 0
        proposed_window_number = 0

        if type & BACnetPduTypes.SEGMENTED_MESSAGE > 0:
            sequence_number = buffer[offset]
            offset = offset + 1
            proposed_window_number = buffer[offset]
            offset = offset + 1

        service = BACnetConfirmedServices(buffer[offset])
        offset = offset + 1

        return (offset - org_offset, type, service, max_segments, max_adpu, invoke_id, sequence_number, proposed_window_number)

    @staticmethod
    def EncodeUnconfirmedServiceRequest(buffer: EncodeBuffer, type: BACnetPduTypes, service: BACnetUnconfirmedServices):
        buffer.buffer[buffer.offset] = type
        buffer.offset = buffer.offset + 1
        buffer.buffer[buffer.offset] = service
        buffer.offset = buffer.offset + 1

    @staticmethod
    def DecodeUnconfirmedServiceRequest(buffer: bytearray, offset: int):
        org_offset = offset

        type = BACnetPduTypes(buffer[offset])
        offset = offset + 1
        service = BACnetUnconfirmedServices(buffer[offset])
        offset = offset + 1

        return (offset - org_offset, type, service)

    @staticmethod
    def EncodeSimpleAck(buffer: EncodeBuffer, type: BACnetPduTypes, service: BACnetUnconfirmedServices, invoke_id: int):
        buffer.buffer[buffer.offset] = type
        buffer.offset = buffer.offset + 1
        buffer.buffer[buffer.offset] = invoke_id
        buffer.offset = buffer.offset + 1
        buffer.buffer[buffer.offset] = service
        buffer.offset = buffer.offset + 1

    @staticmethod
    def DecodeSimpleAck(buffer: bytearray, offset: int):
        org_offset = offset

        type = BACnetPduTypes(buffer[offset])
        offset = offset + 1
        invoke_id = buffer[offset]
        offset = offset + 1
        service = BACnetConfirmedServices(buffer[offset])
        offset = offset + 1

        return (offset - org_offset, type, service, invoke_id)

    @staticmethod
    def EncodeComplexAck(buffer: EncodeBuffer, type: BACnetPduTypes, service: BACnetConfirmedServices, invoke_id: int, sequence_number: int, proposed_window_number: int):
        length = 3
        buffer.buffer[buffer.offset] = type
        buffer.offset = buffer.offset + 1
        buffer.buffer[buffer.offset] = invoke_id
        buffer.offset = buffer.offset + 1
        if type & BACnetPduTypes.SEGMENTED_MESSAGE > 0:
            buffer.buffer[buffer.offset] = sequence_number
            buffer.offset = buffer.offset + 1
            buffer.buffer[buffer.offset] = proposed_window_number
            buffer.offset = buffer.offset + 1
            length = length + 1
        buffer.buffer[buffer.offset] = service
        buffer.offset = buffer.offset + 1
        return length

    @staticmethod
    def DecodeComplexAck(buffer: bytearray, offset: int):
        org_offset = offset

        type = BACnetPduTypes(buffer[offset])
        offset = offset + 1
        invoke_id = buffer[offset]
        offset = offset + 1

        sequence_number = 0
        proposed_window_number = 0
        if type & BACnetPduTypes.SEGMENTED_MESSAGE > 0:
            sequence_number = buffer[offset]
            offset = offset + 1
            proposed_window_number = buffer[offset]
            offset = offset + 1

        service = BACnetConfirmedServices(buffer[offset])
        offset = offset + 1

        return (offset - org_offset, type, service, invoke_id, sequence_number, proposed_window_number)

    @staticmethod
    def EncodeSegmentAck(buffer: EncodeBuffer, type: BACnetPduTypes, original_invoke_id: int, sequence_number: int, actual_window_size: int):
        buffer.buffer[buffer.offset] = type
        buffer.offset = buffer.offset + 1
        buffer.buffer[buffer.offset] = original_invoke_id
        buffer.offset = buffer.offset + 1
        buffer.buffer[buffer.offset] = sequence_number
        buffer.offset = buffer.offset + 1
        buffer.buffer[buffer.offset] = actual_window_size
        buffer.offset = buffer.offset + 1

    @staticmethod
    def DecodeSegmentAck(buffer: bytearray, offset: int):
        org_offset = offset

        type = BACnetPduTypes(buffer[offset])
        offset = offset + 1
        original_invoke_id = buffer[offset]
        offset = offset + 1
        sequence_number = buffer[offset]
        offset = offset + 1
        actual_window_size = buffer[offset]
        offset = offset + 1

        return (offset - org_offset, type, original_invoke_id, sequence_number, actual_window_size)

    @staticmethod
    def EncodeError(buffer: EncodeBuffer, type: BACnetPduTypes, service: BACnetConfirmedServices, invoke_id: int):
        buffer.buffer[buffer.offset] = type
        buffer.offset = buffer.offset + 1
        buffer.buffer[buffer.offset] = invoke_id
        buffer.offset = buffer.offset + 1
        buffer.buffer[buffer.offset] = service
        buffer.offset = buffer.offset + 1

    @staticmethod
    def DecodeError(buffer: bytearray, offset: int):
        org_offset = offset

        type = BACnetPduTypes(buffer[offset])
        offset = offset + 1
        invoke_id = buffer[offset]
        offset = offset + 1
        service = BACnetConfirmedServices(buffer[offset])
        offset = offset + 1

        return (offset - org_offset, type, service, invoke_id)

    @staticmethod
    def EncodeAbort(buffer: EncodeBuffer, type: BACnetPduTypes, invoke_id: int, reason: int):
        buffer.buffer[buffer.offset] = type
        buffer.offset = buffer.offset + 1
        buffer.buffer[buffer.offset] = invoke_id
        buffer.offset = buffer.offset + 1
        buffer.buffer[buffer.offset] = reason
        buffer.offset = buffer.offset + 1

    @staticmethod
    def DecodeAbort(buffer: bytearray, offset: int):
        org_offset = offset

        type = BACnetPduTypes(buffer[offset])
        offset = offset + 1
        invoke_id = buffer[offset]
        offset = offset + 1
        reason = buffer[offset]
        offset = offset + 1

        return (offset - org_offset, type, invoke_id, reason)

# edit 20210924
class NPDU:

    BACNET_PROTOCOL_VERSION = 1

    @staticmethod
    def DecodeFunction(buffer: bytearray, offset: int):
        if buffer[offset+0] != NPDU.BACNET_PROTOCOL_VERSION:
            return 0
        return BACnetNpduControls(buffer[offset + 1])

    @staticmethod
    def Decode(buffer: bytearray, offset: int):
        org_offset = offset

        offset = offset + 1
        function = BACnetNpduControls(buffer[offset])
        offset = offset + 1

        destination = None
        if function & BACnetNpduControls.DestinationSpecified == BACnetNpduControls.DestinationSpecified:
            a1 = buffer[offset] << 8
            offset = offset + 1
            a2 = buffer[offset] << 0
            offset = offset + 1
            destination = BACnetAddress(type=BACnetAddressTypes.NONE, net=BACnetTool.int_to_ushort(a1 | a2), adr=None)
            adr_len = buffer[offset]
            offset = offset + 1
            if adr_len > 0:
                destination.adr = bytearray(adr_len)
                for i in range(0, adr_len):
                    destination.adr[i] = buffer[offset]
                    offset = offset + 1

        source = None
        if function & BACnetNpduControls.SourceSpecified == BACnetNpduControls.SourceSpecified:
            a1 = buffer[offset] << 8
            offset = offset + 1
            a2 = buffer[offset] << 0
            offset = offset + 1
            source = BACnetAddress(type=BACnetAddressTypes.NONE, net=BACnetTool.int_to_ushort(a1 | a2), adr=None)
            adr_len = buffer[offset]
            offset = offset + 1
            if adr_len > 0:
                source.adr = bytearray(adr_len)
                for i in range(0, adr_len):
                    source.adr[i] = buffer[offset]
                    offset = offset + 1

        hop_count = 0
        if function & BACnetNpduControls.DestinationSpecified == BACnetNpduControls.DestinationSpecified:
            hop_count = buffer[offset]
            offset = offset + 1

        network_msg_type = BACnetNetworkMessageTypes.NETWORK_MESSAGE_WHO_IS_ROUTER_TO_NETWORK
        vendor_id = 0

        if function & BACnetNpduControls.NetworkLayerMessage == BACnetNpduControls.NetworkLayerMessage:
            network_msg_type = BACnetNetworkMessageTypes(buffer[offset])
            offset = offset + 1
            if network_msg_type >= 0x80:
                a1 = buffer[offset] << 8
                offset = offset + 1
                a2 = buffer[offset] << 0
                offset = offset + 1
                vendor_id = BACnetTool.int_to_ushort(a1 | a2)
            elif network_msg_type == BACnetNetworkMessageTypes.NETWORK_MESSAGE_WHO_IS_ROUTER_TO_NETWORK:
                offset = offset + 2

        if buffer[org_offset + 0] != NPDU.BACNET_PROTOCOL_VERSION:
            return (-1, function, destination, source, hop_count, network_msg_type, vendor_id)

        return (offset - org_offset, function, destination, source, hop_count, network_msg_type, vendor_id)

    @staticmethod
    def Encode(buffer: EncodeBuffer, function: BACnetNpduControls, destination: BACnetAddress, source: BACnetAddress=None, hop_count: int=0xFF):
        has_destination = destination != None and destination.net > 0
        has_source = source != None and source.net > 0 and source.net != 0xFFFF

        buffer.buffer[buffer.offset] = NPDU.BACNET_PROTOCOL_VERSION
        buffer.offset = buffer.offset + 1
        buffer.buffer[buffer.offset] = function | (BACnetNpduControls.DestinationSpecified if has_destination else 0) | (BACnetNpduControls.SourceSpecified if has_source else 0)
        buffer.offset = buffer.offset + 1

        if has_destination:
            buffer.buffer[buffer.offset] = BACnetTool.int_to_byte((destination.net & 0xFF00) >> 8)
            buffer.offset = buffer.offset + 1
            buffer.buffer[buffer.offset] = BACnetTool.int_to_byte((destination.net & 0x00FF) >> 0)
            buffer.offset = buffer.offset + 1

            if destination.net == 0xFFFF:
                buffer.buffer[buffer.offset] = 0
                buffer.offset = buffer.offset + 1
            else:
                buffer.buffer[buffer.offset] = len(destination.adr)
                buffer.offset = buffer.offset + 1
                for i in range(len(destination.adr)):
                    buffer.buffer[buffer.offset] = destination.adr[i]
                    buffer.offset = buffer.offset + 1

        if has_source:
            buffer.buffer[buffer.offset] = BACnetTool.int_to_byte((source.net & 0xFF00) >> 8)
            buffer.offset = buffer.offset + 1
            buffer.buffer[buffer.offset] = BACnetTool.int_to_byte((source.net & 0x00FF) >> 0)
            buffer.offset = buffer.offset + 1

            buffer.buffer[buffer.offset] = len(source.adr)
            buffer.offset = buffer.offset + 1
            for i in range(len(source.adr)):
                buffer.buffer[buffer.offset] = source.adr[i]
                buffer.offset = buffer.offset + 1

        if has_destination:
            buffer.buffer[buffer.offset] = hop_count
            buffer.offset = buffer.offset + 1

# edit 20210924
class MSTP():

    MSTP_PREAMBLE1 = 0x55
    MSTP_PREAMBLE2 = 0xFF
    MSTP_MAX_APDU = BACnetMaxAdpu.MAX_APDU480
    MSTP_HEADER_LENGTH = 8

    @staticmethod
    def Calc_Header(dataValue: int, crcValue: int):
        crc = BACnetTool.int_to_ushort(crcValue ^ dataValue)
        crc = BACnetTool.int_to_ushort(crc ^ (crc << 1) ^ (crc << 2) ^ (crc << 3) ^ (crc << 4) ^ (crc << 5) ^ (crc << 6) ^ (crc << 7))
        return BACnetTool.int_to_byte((crc & 0xfe) ^ ((crc >> 8) & 1))

    @staticmethod
    def CRC_Calc_Header(buffer: bytearray, offset: int, length: int):
        crc = 0xff
        for i in range(offset, offset+length):
            crc = MSTP.Calc_Header(buffer[i], crc)
        return BACnetTool.int_to_byte(~crc)

    @staticmethod
    def Calc_Data(dataValue: int, crcValue: int):
        crcLow = BACnetTool.int_to_ushort((crcValue & 0xff) ^ dataValue)
        return BACnetTool.int_to_ushort((crcValue >> 8) ^ (crcLow << 8) ^ (crcLow << 3) ^ (crcLow << 12) ^ (crcLow >> 4) ^ (crcLow & 0x0f) ^ ((crcLow & 0x0f) << 7))

    @staticmethod
    def CRC_Calc_Data(buffer: bytearray, offset: int, length: int):
        crc = 0xFFFF
        for i in range(offset, offset + length):
            crc = MSTP.Calc_Data(buffer[i], crc)
        return BACnetTool.int_to_ushort(~crc)

    @staticmethod
    def Decode(buffer: bytearray, offset: int, max_length: int):
        if max_length < MSTP.MSTP_HEADER_LENGTH:
            return (-1, BACnetMstpFrameTypes.FRAME_TYPE_BACNET_DATA_EXPECTING_REPLY, 0, 0, 0)

        frame_type = BACnetMstpFrameTypes(buffer[offset + 2])
        destination_address = buffer[offset + 3]
        source_address = buffer[offset + 4]
        msg_length = (buffer[offset + 5] << 8) | (buffer[offset + 6] << 0)
        crc_header = buffer[offset + 7]
        crc_data = 0

        if msg_length > 0:
            if (offset + 8 + msg_length + 1) >= len(buffer):
                return (-1, BACnetMstpFrameTypes.FRAME_TYPE_BACNET_DATA_EXPECTING_REPLY, 0, 0, 0)
            crc_data = BACnetTool.int_to_ushort((buffer[offset + 8 + msg_length + 1] << 8) | (buffer[offset + 8 + msg_length + 0] << 0))

        if buffer[offset + 0] != MSTP.MSTP_PREAMBLE1:
            return (-1, BACnetMstpFrameTypes.FRAME_TYPE_BACNET_DATA_EXPECTING_REPLY, 0, 0, 0)

        if buffer[offset + 1] != MSTP.MSTP_PREAMBLE2:
            return (-1, BACnetMstpFrameTypes.FRAME_TYPE_BACNET_DATA_EXPECTING_REPLY, 0, 0, 0)

        if MSTP.CRC_Calc_Header(buffer, offset + 2, 5) != crc_header:
            return (-1, BACnetMstpFrameTypes.FRAME_TYPE_BACNET_DATA_EXPECTING_REPLY, 0, 0, 0)

        if msg_length > 0 and max_length >= (MSTP.MSTP_HEADER_LENGTH + msg_length + 2) and MSTP.CRC_Calc_Data(buffer, offset + 8, msg_length) != crc_data:
            return (-1, BACnetMstpFrameTypes.FRAME_TYPE_BACNET_DATA_EXPECTING_REPLY, 0, 0, 0)

        return (8 + msg_length + (2 if msg_length > 0 else 0), frame_type, destination_address, source_address, msg_length)

    @staticmethod
    def Encode(buffer: bytearray, offset: int, frame_type: BACnetMstpFrameTypes, destination_address: int, source_address: int, msg_length: int):
        buffer[offset + 0] = MSTP.MSTP_PREAMBLE1
        buffer[offset + 1] = MSTP.MSTP_PREAMBLE2
        buffer[offset + 2] = frame_type
        buffer[offset + 3] = destination_address
        buffer[offset + 4] = source_address
        buffer[offset + 5] = BACnetTool.int_to_byte((msg_length & 0xFF00) >> 8)
        buffer[offset + 6] = BACnetTool.int_to_byte((msg_length & 0x00FF) >> 0)
        buffer[offset + 7] = MSTP.CRC_Calc_Header(buffer, offset + 2, 5)
        if msg_length > 0:
            data_crc = MSTP.CRC_Calc_Data(buffer, offset + 8, msg_length)
            buffer[offset + 8 + msg_length + 0] = BACnetTool.int_to_byte(data_crc & 0xFF)
            buffer[offset + 8 + msg_length + 1] = BACnetTool.int_to_byte(data_crc >> 8)

        return MSTP.MSTP_HEADER_LENGTH + msg_length +  (2 if msg_length > 0 else 0)

# edit 20210924-20210926
class Services():

    @staticmethod
    def EncodeIamBroadcast(buffer: EncodeBuffer, device_id: int, max_apdu: int, segmentation: BACnetSegmentations, vendor_id: int):
        ASN1.encode_application_object_id(buffer, BACnetObjectTypes.OBJECT_DEVICE, device_id)
        ASN1.encode_application_unsigned(buffer, max_apdu)
        ASN1.encode_application_enumerated(buffer, segmentation)
        ASN1.encode_application_unsigned(buffer, vendor_id)

    @staticmethod
    def DecodeIamBroadcast(buffer: bytearray, offset: int):
        length = 0
        apdu_len = 0
        org_offset = offset
        device_id = 0
        max_apdu = 0
        segmentation = BACnetSegmentations.SEGMENTATION_NONE
        vendor_id = 0
        object_id = BACnetObjectId()

        (length, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + apdu_len)
        apdu_len += length
        if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_OBJECT_ID:
            return (-1, device_id, max_apdu, segmentation, vendor_id)
        (length, object_id.type, object_id.instance) = ASN1.decode_object_id(buffer, offset + apdu_len)
        apdu_len += length
        if object_id.type != BACnetObjectTypes.OBJECT_DEVICE:
            return (-1, device_id, max_apdu, segmentation, vendor_id)
        device_id = object_id.instance

        (length, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + apdu_len)
        apdu_len += length
        if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT:
            return (-1, device_id, max_apdu, segmentation, vendor_id)
        (length, decoded_value) = ASN1.decode_unsigned(buffer, offset + apdu_len, len_value)
        apdu_len += length
        max_apdu = decoded_value

        (length, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + apdu_len)
        apdu_len += length
        if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_ENUMERATED:
            return (-1, device_id, max_apdu, segmentation, vendor_id)
        (length, decoded_value) = ASN1.decode_enumerated(buffer, offset + apdu_len, len_value)
        apdu_len += length
        if decoded_value > BACnetSegmentations.SEGMENTATION_NONE:
            return (-1, device_id, max_apdu, segmentation, vendor_id)
        segmentation = BACnetSegmentations(decoded_value)

        (length, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + apdu_len)
        apdu_len += length
        if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT:
            return (-1, device_id, max_apdu, segmentation, vendor_id)
        (length, decoded_value) = ASN1.decode_unsigned(buffer, offset + apdu_len, len_value)
        apdu_len += length
        if decoded_value > 0xFFFF:
            return (-1, device_id, max_apdu, segmentation, vendor_id)
        vendor_id = BACnetTool.int_to_ushort(decoded_value)
        return (offset - org_offset, device_id, max_apdu, segmentation, vendor_id)

    @staticmethod
    def EncodeIhaveBroadcast(buffer: EncodeBuffer, device_id: BACnetObjectId, object_id: BACnetObjectId, object_name: str):
        ASN1.encode_application_object_id(buffer, device_id.type, device_id.instance)
        ASN1.encode_application_object_id(buffer, object_id.type, object_id.instance)
        ASN1.encode_application_character_string(buffer, object_name)

    @staticmethod
    def EncodeWhoIsBroadcast(buffer: EncodeBuffer, low_limit: int, high_limit: int):
        if (low_limit >= 0) and (low_limit <= ASN1.BACNET_MAX_INSTANCE) and (high_limit >= 0) and (high_limit <= ASN1.BACNET_MAX_INSTANCE):
            ASN1.encode_context_unsigned(buffer, 0, low_limit)
            ASN1.encode_context_unsigned(buffer, 1, high_limit)

    @staticmethod
    def DecodeWhoIsBroadcast(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        low_limit = -1
        high_limit = -1
        if apdu_len <= 0:
            return (length, low_limit, high_limit)

        (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l
        if tag_number != 0:
            return (-1, low_limit, high_limit)

        if apdu_len > length:
            (l, decoded_value) = ASN1.decode_unsigned(buffer, offset + length, len_value)
            length += l

            if decoded_value <= ASN1.BACNET_MAX_INSTANCE:
                low_limit = decoded_value
            if apdu_len > length:
                (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l
                if tag_number != 1:
                    return (-1, low_limit, high_limit)

                if apdu_len > length:
                    (l, decoded_value) = ASN1.decode_unsigned(buffer, offset + length, len_value)
                    length += l

                    if decoded_value <= ASN1.BACNET_MAX_INSTANCE:
                        high_limit = decoded_value
                else:
                    return (-1, low_limit, high_limit)
            else:
                return (-1, low_limit, high_limit)
        else:
            return (-1, low_limit, high_limit)

        return (length, low_limit, high_limit)

    @staticmethod
    def DecodeWhoHasBroadcast(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        ObjId = BACnetObjectId(BACnetObjectTypes.OBJECT_BINARY_OUTPUT, 0x3FFFFF)
        ObjName = None
        low_limit = -1
        high_limit = -1

        (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l
        if tag_number == 0:
            (l, decoded_value) = ASN1.decode_unsigned(buffer, offset + length, len_value)
            length += l

            if decoded_value <= ASN1.BACNET_MAX_INSTANCE:
                low_limit = decoded_value

            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

        if tag_number == 1:
            (l, decoded_value) = ASN1.decode_unsigned(buffer, offset + length, len_value)
            length += l

            if decoded_value <= ASN1.BACNET_MAX_INSTANCE:
                high_limit = decoded_value

            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

        if tag_number == 2:
            (l, ObjType, ObjInst) = ASN1.decode_object_id(buffer, offset + length)
            length += l

            ObjId = BACnetObjectId(BACnetObjectTypes(ObjType), ObjInst)

        if tag_number == 3:
            (l, ObjName) = ASN1.decode_character_string(buffer, offset + length,  apdu_len - (offset + length), len_value)
            length += l

        return (length, low_limit, high_limit, ObjId, ObjName)

    @staticmethod
    def EncodeAlarmAcknowledge(buffer: EncodeBuffer, ackProcessIdentifier: int, eventObjectIdentifier: BACnetObjectId, eventStateAcked: int, ackSource: str, eventTimeStamp: BACnetGenericTime, ackTimeStamp: BACnetGenericTime):
        ASN1.encode_context_unsigned(buffer, 0, ackProcessIdentifier)
        ASN1.encode_context_object_id(buffer, 1, eventObjectIdentifier.type, eventObjectIdentifier.instance)
        ASN1.encode_context_enumerated(buffer, 2, eventStateAcked)
        ASN1.bacapp_encode_context_timestamp(buffer, 3, eventTimeStamp)
        ASN1.encode_context_character_string(buffer, 4, ackSource)
        ASN1.bacapp_encode_context_timestamp(buffer, 5, ackTimeStamp)

    @staticmethod
    def EncodeAtomicReadFile(buffer: EncodeBuffer, is_stream: bool, object_id: BACnetObjectId, position: int, count: int):
        ASN1.encode_application_object_id(buffer, object_id.type, object_id.instance)
        if is_stream is True:
            ASN1.encode_opening_tag(buffer, 0)
            ASN1.encode_application_signed(buffer, position)
            ASN1.encode_application_unsigned(buffer, count)
            ASN1.encode_closing_tag(buffer, 0)
        else:
            ASN1.encode_opening_tag(buffer, 1)
            ASN1.encode_application_signed(buffer, position)
            ASN1.encode_application_unsigned(buffer, count)
            ASN1.encode_closing_tag(buffer, 1)

    @staticmethod
    def DecodeAtomicReadFile(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        is_stream = True
        object_id = BACnetObjectId()
        position = -1
        count = 0

        (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l

        if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_OBJECT_ID:
            return (-1, is_stream, object_id, position, count)

        (l, object_id.type, object_id.instance) = ASN1.decode_object_id(buffer, offset + length)
        length += l

        if ASN1.decode_is_opening_tag_number(buffer, offset + length, 0):
            is_stream = True
            length += 1

            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += tag_len

            if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_SIGNED_INT:
                return (-1, is_stream, object_id, position, count)

            (l, position) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
            length += l

            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += tag_len
            if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT:
                return (-1, is_stream, object_id, position, count)

            (l, count) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
            length += l
            if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 0):
                return (-1, is_stream, object_id, position, count)
            length += 1
        elif ASN1.decode_is_opening_tag_number(buffer, offset + length, 1):
            is_stream = False
            length += 1

            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += tag_len

            if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_SIGNED_INT:
                return (-1, is_stream, object_id, position, count)

            (l, position) = ASN1.decode_signed(buffer, offset + length, len_value_type)
            length += l

            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += tag_len
            if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT:
                return (-1, is_stream, object_id, position, count)

            (l, count) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
            length += l
            if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 1):
                return (-1, is_stream, object_id, position, count)
            length += 1
        else:
            return (-1, is_stream, object_id, position, count)
        return (length, is_stream, object_id, position, count)

    @staticmethod
    def EncodeAtomicReadFileAcknowledge(buffer: EncodeBuffer, is_stream: bool, end_of_file: bool, position: int, block_count: int, blocks: list, counts: list):
        ASN1.encode_application_boolean(buffer, end_of_file)
        if is_stream is True:
            ASN1.encode_opening_tag(buffer, 0)
            ASN1.encode_application_signed(buffer, position)
            ASN1.encode_application_octet_string(buffer, blocks[0], 0, counts[0])
            ASN1.encode_closing_tag(buffer, 0)
        else:
            ASN1.encode_opening_tag(buffer, 1)
            ASN1.encode_application_signed(buffer, position)
            ASN1.encode_application_unsigned(buffer, block_count)
            for i in range(block_count):
                ASN1.encode_application_octet_string(buffer, blocks[i], 0, counts[i])
            ASN1.encode_closing_tag(buffer, 1)

    @staticmethod
    def EncodeAtomicWriteFile(buffer: EncodeBuffer, is_stream: bool, object_id: BACnetObjectId, position: int, block_count: int, blocks: list, counts: list):
        ASN1.encode_application_object_id(buffer, object_id.type, object_id.instance)
        if is_stream is True:
            ASN1.encode_opening_tag(buffer, 0)
            ASN1.encode_application_signed(buffer, position)
            ASN1.encode_application_octet_string(buffer, blocks[0], 0, counts[0])
            ASN1.encode_closing_tag(buffer, 0)
        else:
            ASN1.encode_opening_tag(buffer, 1)
            ASN1.encode_application_signed(buffer, position)
            ASN1.encode_application_unsigned(buffer, block_count)
            for i in range(block_count):
                ASN1.encode_application_octet_string(buffer, blocks[i], 0, counts[i])
            ASN1.encode_closing_tag(buffer, 1)

    @staticmethod
    def DecodeAtomicWriteFile(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        is_stream = True
        object_id = BACnetObjectId()
        position = -1
        block_count = 0
        blocks = None
        counts = None

        (length, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_OBJECT_ID:
            return (-1, is_stream, object_id, position, block_count, blocks, counts)

        (l, object_id.type, object_id.instance) = ASN1.decode_object_id(buffer, offset + length)
        length += l

        if ASN1.decode_is_opening_tag_number(buffer, offset + length, 0):
            is_stream = True
            length += 1

            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += tag_len

            if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_SIGNED_INT:
                return (-1, is_stream, object_id, position, block_count, blocks, counts)

            (l, position) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
            length += l

            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += tag_len
            if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_OCTET_STRING:
                return (-1, is_stream, object_id, position, block_count, blocks, counts)

            block_count = 1
            blocks = [bytearray(len_value_type)]
            counts = [len_value_type]

            (l, blocks[0]) = ASN1.decode_octet_string(buffer, offset + length, apdu_len, blocks[0], 0, len_value_type)
            length += l

            if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 0):
                return (-1, is_stream, object_id, position, block_count, blocks, counts)

            length += 1
        elif ASN1.decode_is_opening_tag_number(buffer, offset + length, 1):
            is_stream = False
            length += 1

            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += tag_len

            if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_SIGNED_INT:
                return (-1, is_stream, object_id, position, block_count, blocks, counts)

            (l, position) = ASN1.decode_signed(buffer, offset + length, len_value_type)
            length += l

            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += tag_len

            if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_UNSIGNED_INT:
                return (-1, is_stream, object_id, position, block_count, blocks, counts)

            (l, block_count) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
            length += l

            blocks = []
            counts = [block_count]

            for i in range(block_count):
                (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += tag_len
                blocks.append(bytearray(len_value_type))
                counts.append(len_value_type)
                if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_OCTET_STRING:
                    return (-1, is_stream, object_id, position, block_count, blocks, counts)

                (l, blocks[i]) = ASN1.decode_octet_string(buffer, offset + length, apdu_len, blocks[i], 0, len_value_type)
                length += l

            if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 1):
                return (-1, is_stream, object_id, position, block_count, blocks, counts)

            length += 1
        else:
            return (-1, is_stream, object_id, position, block_count, blocks, counts)
        return (length, is_stream, object_id, position, block_count, blocks, counts)

    @staticmethod
    def EncodeCreateProperty(buffer: EncodeBuffer, object_id: BACnetObjectId, value_list: list):
        ASN1.encode_opening_tag(buffer, 0)
        ASN1.encode_context_object_id(buffer, 1, object_id.type, object_id.instance)
        ASN1.encode_closing_tag(buffer, 0)

        if value_list is not None:
            ASN1.encode_opening_tag(buffer, 1)
            for p_value in value_list:
                ASN1.encode_context_enumerated(buffer, 0, p_value.property.propertyIdentifier)
                if p_value.property.propertyArrayIndex != ASN1.BACNET_ARRAY_ALL:
                    ASN1.encode_context_unsigned(buffer, 1, p_value.property.propertyArrayIndex)
                ASN1.encode_opening_tag(buffer, 2)
                for value in p_value.value:
                    ASN1.bacapp_encode_application_data(buffer, value)
                ASN1.encode_closing_tag(buffer, 2)

                if p_value.priority != ASN1.BACNET_NO_PRIORITY:
                    ASN1.encode_context_unsigned(buffer, 3, p_value.priority)

            ASN1.encode_closing_tag(buffer, 1)

    @staticmethod
    def EncodeAddListElement(buffer: EncodeBuffer, object_id: BACnetObjectId, property_id: int, array_index: int, value_list: list):
        ASN1.encode_context_object_id(buffer, 0, object_id.type, object_id.instance)
        ASN1.encode_context_enumerated(buffer, 1, property_id)

        if array_index != ASN1.BACNET_ARRAY_ALL:
            ASN1.encode_context_unsigned(buffer, 2, array_index)


        ASN1.encode_opening_tag(buffer, 3)
        for value in value_list:
            ASN1.bacapp_encode_application_data(buffer, value)

        ASN1.encode_closing_tag(buffer, 3)

    @staticmethod
    def EncodeAtomicWriteFileAcknowledge(buffer: EncodeBuffer, is_stream: bool, position: int):
        if is_stream is True:
            ASN1.encode_context_signed(buffer, 0, position)
        else:
            ASN1.encode_context_signed(buffer, 1, position)

    @staticmethod
    def EncodeCOVNotifyConfirmed(buffer: EncodeBuffer, subscriberProcessIdentifier: int, initiatingDeviceIdentifier: int, monitoredObjectIdentifier: BACnetObjectId, timeRemaining: int, values: list):
        ASN1.encode_context_unsigned(buffer, 0, subscriberProcessIdentifier)
        ASN1.encode_context_object_id(buffer, 1, BACnetObjectTypes.OBJECT_DEVICE, initiatingDeviceIdentifier)
        ASN1.encode_context_object_id(buffer, 2, monitoredObjectIdentifier.type, monitoredObjectIdentifier.instance)
        ASN1.encode_context_unsigned(buffer, 3, timeRemaining)
        ASN1.encode_opening_tag(buffer, 4)
        for value in values:
            ASN1.encode_context_enumerated(buffer, 0, value.property.propertyIdentifier)
            if value.property.propertyArrayIndex != ASN1.BACNET_ARRAY_ALL:
                ASN1.encode_context_unsigned(buffer, 1, value.property.propertyArrayIndex)

            ASN1.encode_opening_tag(buffer, 2)
            for v in value.value:
                ASN1.bacapp_encode_application_data(buffer, v)
            ASN1.encode_closing_tag(buffer, 2)

            if value.priority != ASN1.BACNET_NO_PRIORITY:
                ASN1.encode_context_unsigned(buffer, 3, value.priority)
        ASN1.encode_closing_tag(buffer, 4)

    @staticmethod
    def EncodeCOVNotifyUnconfirmed(buffer: EncodeBuffer, subscriberProcessIdentifier: int, initiatingDeviceIdentifier: int, monitoredObjectIdentifier: BACnetObjectId, timeRemaining: int, values: list):
        ASN1.encode_context_unsigned(buffer, 0, subscriberProcessIdentifier)
        ASN1.encode_context_object_id(buffer, 1, BACnetObjectTypes.OBJECT_DEVICE, initiatingDeviceIdentifier)
        ASN1.encode_context_object_id(buffer, 2, monitoredObjectIdentifier.type, monitoredObjectIdentifier.instance)
        ASN1.encode_context_unsigned(buffer, 3, timeRemaining)
        ASN1.encode_opening_tag(buffer, 4)
        for value in values:
            ASN1.encode_context_enumerated(buffer, 0, value.property.propertyIdentifier)
            if value.property.propertyArrayIndex != ASN1.BACNET_ARRAY_ALL:
                ASN1.encode_context_unsigned(buffer, 1, value.property.propertyArrayIndex)

            ASN1.encode_opening_tag(buffer, 2)
            for v in value.value:
                ASN1.bacapp_encode_application_data(buffer, v)
            ASN1.encode_closing_tag(buffer, 2)

            if value.priority != ASN1.BACNET_NO_PRIORITY:
                ASN1.encode_context_unsigned(buffer, 3, value.priority)
        ASN1.encode_closing_tag(buffer, 4)

    @staticmethod
    def EncodeSubscribeCOV(buffer: EncodeBuffer, subscriberProcessIdentifier: int, monitoredObjectIdentifier: BACnetObjectId, cancellationRequest: bool, issueConfirmedNotifications: bool, lifetime: int):
        ASN1.encode_context_unsigned(buffer, 0, subscriberProcessIdentifier)
        ASN1.encode_context_object_id(buffer, 1, monitoredObjectIdentifier.type, monitoredObjectIdentifier.instance)
        if not cancellationRequest:
            ASN1.encode_context_boolean(buffer, 2, issueConfirmedNotifications)
            ASN1.encode_context_unsigned(buffer, 3, lifetime)

    @staticmethod
    def DecodeSubscribeCOV(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        subscriberProcessIdentifier = 0
        monitoredObjectIdentifier = BACnetObjectId()
        cancellationRequest = False
        issueConfirmedNotifications = False
        lifetime = 0

        if ASN1.decode_is_context_tag(buffer, offset + length, 0):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, subscriberProcessIdentifier) = ASN1.decode_unsigned(buffer, offset + length, len_value)
            length += l
        else:
            return (-1, subscriberProcessIdentifier, monitoredObjectIdentifier, cancellationRequest, issueConfirmedNotifications, lifetime)

        if ASN1.decode_is_context_tag(buffer, offset + length, 1):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, monitoredObjectIdentifier.type, monitoredObjectIdentifier.instance) = ASN1.decode_object_id(buffer, offset + length)
            length += l
        else:
            return (-1, subscriberProcessIdentifier, monitoredObjectIdentifier, cancellationRequest, issueConfirmedNotifications, lifetime)

        if length < apdu_len:
            if ASN1.decode_is_context_tag(buffer, offset + length, 2):
                cancellationRequest = False
                (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l
                issueConfirmedNotifications = buffer[offset + length] > 0
                length += len_value
            else:
                cancellationRequest = True

            if ASN1.decode_is_context_tag(buffer, offset + length, 3):
                (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l

                (l, lifetime) = ASN1.decode_unsigned(buffer, offset + length, len_value)
                length += l
            else:
                lifetime = 0
        else:
            cancellationRequest = True

        return (length, subscriberProcessIdentifier, monitoredObjectIdentifier, cancellationRequest, issueConfirmedNotifications, lifetime)

    @staticmethod
    def EncodeSubscribeProperty(buffer: EncodeBuffer, subscriberProcessIdentifier: int, monitoredObjectIdentifier: BACnetObjectId, cancellationRequest: bool, issueConfirmedNotifications: bool, lifetime: int, monitoredProperty: BACnetPropertyReference, covIncrementPresent: bool, covIncrement: float):
        ASN1.encode_context_unsigned(buffer, 0, subscriberProcessIdentifier)
        ASN1.encode_context_object_id(buffer, 1, monitoredObjectIdentifier.type, monitoredObjectIdentifier.instance)
        if not cancellationRequest:
            ASN1.encode_context_boolean(buffer, 2, issueConfirmedNotifications)
            ASN1.encode_context_unsigned(buffer, 3, lifetime)

        ASN1.encode_opening_tag(buffer, 4)
        ASN1.encode_context_enumerated(buffer, 0, monitoredProperty.propertyIdentifier)
        if monitoredProperty.propertyArrayIndex != ASN1.BACNET_ARRAY_ALL:
            ASN1.encode_context_unsigned(buffer, 1, monitoredProperty.propertyArrayIndex)
        ASN1.encode_closing_tag(buffer, 4)

        if covIncrementPresent:
            ASN1.encode_context_real(buffer, 5, covIncrement)

    @staticmethod
    def DecodeSubscribeProperty(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        subscriberProcessIdentifier = 0
        monitoredObjectIdentifier = BACnetObjectId()
        cancellationRequest = False
        issueConfirmedNotifications = False
        lifetime = 0
        covIncrement = 0
        monitoredProperty = BACnetPropertyReference()

        if ASN1.decode_is_context_tag(buffer, offset + length, 0):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, subscriberProcessIdentifier) = ASN1.decode_unsigned(buffer, offset + length, len_value)
            length += l
        else:
            return (-1, subscriberProcessIdentifier, monitoredObjectIdentifier, monitoredProperty, cancellationRequest, issueConfirmedNotifications, lifetime, covIncrement)

        if ASN1.decode_is_context_tag(buffer, offset + length, 1):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, monitoredObjectIdentifier.type, monitoredObjectIdentifier.instance) = ASN1.decode_object_id(buffer, offset + length)
            length += l
        else:
            return (-1, subscriberProcessIdentifier, monitoredObjectIdentifier, monitoredProperty, cancellationRequest, issueConfirmedNotifications, lifetime, covIncrement)

        if ASN1.decode_is_context_tag(buffer, offset + length, 2):
            cancellationRequest = False
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            issueConfirmedNotifications = buffer[offset + length] > 0
            length += 1
        else:
            cancellationRequest = True

        if ASN1.decode_is_context_tag(buffer, offset + length, 3):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, lifetime) = ASN1.decode_unsigned(buffer, offset + length, len_value)
            length += l
        else:
            lifetime = 0

        if not ASN1.decode_is_opening_tag_number(buffer, offset + length, 4):
            return (-1, subscriberProcessIdentifier, monitoredObjectIdentifier, monitoredProperty, cancellationRequest, issueConfirmedNotifications, lifetime, covIncrement)

        length += 1

        if ASN1.decode_is_context_tag(buffer, offset + length, 0):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, decoded_value) = ASN1.decode_enumerated(buffer, offset + length, len_value)
            length += l

            monitoredProperty.propertyIdentifier = decoded_value
        else:
            return (-1, subscriberProcessIdentifier, monitoredObjectIdentifier, monitoredProperty, cancellationRequest, issueConfirmedNotifications, lifetime, covIncrement)

        if ASN1.decode_is_context_tag(buffer, offset + length, 1):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, decoded_value) = ASN1.decode_unsigned(buffer, offset + length, len_value)
            length += l

            monitoredProperty.propertyArrayIndex = decoded_value
        else:
            monitoredProperty.propertyArrayIndex = ASN1.BACNET_ARRAY_ALL

        if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 4):
            return (-1, subscriberProcessIdentifier, monitoredObjectIdentifier, monitoredProperty, cancellationRequest, issueConfirmedNotifications, lifetime, covIncrement)

        length += 1

        if length < apdu_len and ASN1.decode_is_context_tag(buffer, offset + length, 5):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, covIncrement) = ASN1.decode_real(buffer, offset + length)
            length += l

        return (length, subscriberProcessIdentifier, monitoredObjectIdentifier, monitoredProperty, cancellationRequest, issueConfirmedNotifications, lifetime, covIncrement)

    @staticmethod
    def DecodeEventNotifyData(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        EventData = BACnetEventNotificationData()

        if ASN1.decode_is_context_tag(buffer, offset + length, 0):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, EventData.processIdentifier) = ASN1.decode_unsigned(buffer, offset + length, len_value)
            length += l
        else:
            return (-1, EventData)

        if ASN1.decode_is_context_tag(buffer, offset + length, 1):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, EventData.initiatingObjectIdentifier.type, EventData.initiatingObjectIdentifier.instance) = ASN1.decode_object_id(buffer, offset + length)
            length += l
        else:
            return (-1, EventData)

        if ASN1.decode_is_context_tag(buffer, offset + length, 2):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, EventData.eventObjectIdentifier.type, EventData.eventObjectIdentifier.instance) = ASN1.decode_object_id(buffer, offset + length)
            length += l
        else:
            return (-1, EventData)

        if ASN1.decode_is_context_tag(buffer, offset + length, 3):
            length += 2

            (l, date) = ASN1.decode_application_date(buffer, offset + length)
            length += l

            (l, time) = ASN1.decode_application_time(buffer, offset + length)
            length += l

            EventData.timeStamp.Time = datetime(date.year, date.month, date.day, time.hour, time.minute, time.second, time.microsecond)
            length += 2
        else:
            return (-1, EventData)

        if ASN1.decode_is_context_tag(buffer, offset + length, 4):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, EventData.notificationClass) = ASN1.decode_unsigned(buffer, offset + length, len_value)
            length += l
        else:
            return (-1, EventData)

        if ASN1.decode_is_context_tag(buffer, offset + length, 5):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, priority) = ASN1.decode_unsigned(buffer, offset + length, len_value)
            length += l

            if priority > 0xFF:
                return (-1, EventData)

            EventData.priority = priority
        else:
            return (-1, EventData)

        if ASN1.decode_is_context_tag(buffer, offset + length, 6):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, eventType) = ASN1.decode_enumerated(buffer, offset + length, len_value)
            length += l

            EventData.eventType = BACnetEventTypes(eventType)
        else:
            return (-1, EventData)

        if ASN1.decode_is_context_tag(buffer, offset + length, 7):
            (l, EventData.messageText) = ASN1.decode_context_character_string(buffer, offset + length, 20000, 7)

        if ASN1.decode_is_context_tag(buffer, offset + length, 8):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, notifyType) = ASN1.decode_enumerated(buffer, offset + length, len_value)
            length += l

            EventData.notifyType = BACnetNotifyTypes(notifyType)
        else:
            return (-1, EventData)

        if EventData.notifyType == BACnetNotifyTypes.NOTIFY_ALARM or EventData.notifyType == BACnetNotifyTypes.NOTIFY_EVENT:
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, val) = ASN1.decode_unsigned(buffer, offset + length, len_value)
            length += l
            EventData.ackRequired = bool(val)

            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, fromstate) = ASN1.decode_enumerated(buffer, offset + length, len_value)
            length += l
            EventData.fromState = BACnetEventStates(fromstate)

        if ASN1.decode_is_context_tag(buffer, offset + length, 11):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, toState) = ASN1.decode_enumerated(buffer, offset + length, len_value)
            length += l

            EventData.toState = BACnetEventStates(toState)
        else:
            return (-1, EventData)

        return (length, EventData)

    @staticmethod
    def EncodeEventNotifyData(buffer: EncodeBuffer, data: BACnetEventNotificationData):
        ASN1.encode_context_unsigned(buffer, 0, data.processIdentifier)
        ASN1.encode_context_object_id(buffer, 1, data.initiatingObjectIdentifier.type, data.initiatingObjectIdentifier.instance)
        ASN1.encode_context_object_id(buffer, 2, data.eventObjectIdentifier.type, data.eventObjectIdentifier.instance)
        ASN1.bacapp_encode_context_timestamp(buffer, 3, data.timeStamp)
        ASN1.encode_context_unsigned(buffer, 4, data.notificationClass)
        ASN1.encode_context_unsigned(buffer, 5, data.priority)
        ASN1.encode_context_enumerated(buffer, 6, data.eventType)
        if data.messageText is not None and len(data.messageText) > 0:
            ASN1.encode_context_character_string(buffer, 7, data.messageText)
        ASN1.encode_context_enumerated(buffer, 8, data.notifyType)
        if data.notifyType == BACnetNotifyTypes.NOTIFY_ALARM or data.notifyType == BACnetNotifyTypes.NOTIFY_EVENT:
            ASN1.encode_context_boolean(buffer, 9, data.ackRequired)
        ASN1.encode_context_enumerated(buffer, 11, data.toState)
        if data.notifyType == BACnetNotifyTypes.NOTIFY_ALARM or data.notifyType == BACnetNotifyTypes.NOTIFY_EVENT:
            ASN1.encode_opening_tag(buffer, 12)
            if data.eventType == BACnetEventTypes.EVENT_CHANGE_OF_BITSTRING:
                ASN1.encode_opening_tag(buffer, 0)
                ASN1.encode_context_bitstring(buffer, 0, data.changeOfBitstring_referencedBitString)
                ASN1.encode_context_bitstring(buffer, 1, data.changeOfBitstring_statusFlags)
                ASN1.encode_closing_tag(buffer, 0)
            elif data.eventType == BACnetEventTypes.EVENT_CHANGE_OF_STATE:
                ASN1.encode_opening_tag(buffer, 1)
                ASN1.encode_opening_tag(buffer, 0)
                ASN1.bacapp_encode_property_state(buffer, data.changeOfState_newState)
                ASN1.encode_closing_tag(buffer, 0)
                ASN1.encode_context_bitstring(buffer, 1, data.changeOfState_statusFlags)
                ASN1.encode_closing_tag(buffer, 1)
            elif data.eventType == BACnetEventTypes.EVENT_CHANGE_OF_VALUE:
                ASN1.encode_opening_tag(buffer, 2)
                ASN1.encode_opening_tag(buffer, 0)
                if data.changeOfValue_tag == BACnetCOVTypes.CHANGE_OF_VALUE_REAL:
                    ASN1.encode_context_real(buffer, 1, data.changeOfValue_changeValue)
                elif data.changeOfValue_tag == BACnetCOVTypes.CHANGE_OF_VALUE_BITS:
                    ASN1.encode_context_bitstring(buffer, 0, data.changeOfValue_changedBits)
                ASN1.encode_closing_tag(buffer, 0)
                ASN1.encode_context_bitstring(buffer, 1, data.changeOfValue_statusFlags)
                ASN1.encode_closing_tag(buffer, 2)
            elif data.eventType == BACnetEventTypes.EVENT_FLOATING_LIMIT:
                ASN1.encode_opening_tag(buffer, 4)
                ASN1.encode_context_real(buffer, 0, data.floatingLimit_referenceValue)
                ASN1.encode_context_bitstring(buffer, 1, data.floatingLimit_statusFlags)
                ASN1.encode_context_real(buffer, 2, data.floatingLimit_setPointValue)
                ASN1.encode_context_real(buffer, 3, data.floatingLimit_errorLimit)
                ASN1.encode_closing_tag(buffer, 4)
            elif data.eventType == BACnetEventTypes.EVENT_OUT_OF_RANGE:
                ASN1.encode_opening_tag(buffer, 5)
                ASN1.encode_context_real(buffer, 0, data.outOfRange_exceedingValue)
                ASN1.encode_context_bitstring(buffer, 1, data.outOfRange_statusFlags)
                ASN1.encode_context_real(buffer, 2, data.outOfRange_deadband)
                ASN1.encode_context_real(buffer, 3, data.outOfRange_exceededLimit)
                ASN1.encode_closing_tag(buffer, 5)
            elif data.eventType == BACnetEventTypes.EVENT_CHANGE_OF_LIFE_SAFETY:
                ASN1.encode_opening_tag(buffer, 8)
                ASN1.encode_context_enumerated(buffer, 0, data.changeOfLifeSafety_newState)
                ASN1.encode_context_enumerated(buffer, 1, data.changeOfLifeSafety_newMode)
                ASN1.encode_context_bitstring(buffer, 2, data.changeOfLifeSafety_statusFlags)
                ASN1.encode_context_enumerated(buffer, 3, data.changeOfLifeSafety_operationExpected)
                ASN1.encode_closing_tag(buffer, 8)
            elif data.eventType == BACnetEventTypes.EVENT_BUFFER_READY:
                ASN1.encode_opening_tag(buffer, 10)
                ASN1.bacapp_encode_context_device_obj_property_ref(buffer, 0, data.bufferReady_bufferProperty)
                ASN1.encode_context_unsigned(buffer, 1, data.bufferReady_previousNotification)
                ASN1.encode_context_unsigned(buffer, 2, data.bufferReady_currentNotification)
                ASN1.encode_closing_tag(buffer, 10)
            elif data.eventType == BACnetEventTypes.EVENT_UNSIGNED_RANGE:
                ASN1.encode_opening_tag(buffer, 11)
                ASN1.encode_context_unsigned(buffer, 0, data.unsignedRange_exceedingValue)
                ASN1.encode_context_bitstring(buffer, 1, data.unsignedRange_statusFlags)
                ASN1.encode_context_unsigned(buffer, 2, data.unsignedRange_exceededLimit)
                ASN1.encode_closing_tag(buffer, 11)
            elif data.eventType == BACnetEventTypes.EVENT_EXTENDED or data.eventType == BACnetEventTypes.EVENT_COMMAND_FAILURE:
                raise Exception('NotImplemented')
            ASN1.encode_closing_tag(buffer, 12)

    @staticmethod
    def EncodeEventNotifyUnconfirmed(buffer: EncodeBuffer, data: BACnetEventNotificationData):
        Services.EncodeEventNotifyData(buffer, data)

    @staticmethod
    def DecodeAlarmSummaryOrEvent(buffer: bytearray, offset: int, apdu_len: int, GetEvent: bool, Alarms: list):
        length = 0

        if GetEvent:
            length += 1

        while apdu_len - 2 - length > 0:
            value = BACnetGetEventInformationData()
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, value.objectIdentifier.type, value.objectIdentifier.instance) = ASN1.decode_object_id(buffer, offset + length)
            length += l

            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, tmp) = ASN1.decode_enumerated(buffer, offset + length, len_value)
            length += l
            value.eventState = BACnetEventStates(tmp)

            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, value.acknowledgedTransitions) = ASN1.decode_bitstring(buffer, offset + length, len_value)
            length += l

            if GetEvent:
                length += 1
                value.eventTimeStamps = []

                for i in range(3):
                    (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                    length += l

                    if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_NULL:
                        (l, date) = ASN1.decode_application_date(buffer, offset + length)
                        length += l

                        (l, time) = ASN1.decode_application_time(buffer, offset + length)
                        length += l

                        dt = datetime(date.year, date.month, date.day, time.hour, time.minute, time.second, time.microsecond)
                        value.eventTimeStamps.append(BACnetGenericTime(dt, BACnetTimestampTags.TIME_STAMP_DATETIME))
                        length += 1
                    else:
                        length += len_value

                length += 1

                (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l

                (l, tmp) = ASN1.decode_enumerated(buffer, offset + length, len_value)
                length += l
                value.notifyType = BACnetNotifyTypes(tmp)

                (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l

                (l, value.eventEnable) = ASN1.decode_bitstring(buffer, offset + length, len_value)
                length += l

                length += 1

                value.eventPriorities = []
                for i in range(3):
                    (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                    length += l

                    (l, eventPriorities) = ASN1.decode_unsigned(buffer, offset + length, len_value)
                    length += l

                    value.eventPriorities.append(eventPriorities)

                length += 1

            Alarms.append(value)

        if GetEvent:
            MoreEvent = (buffer[apdu_len - 1] == 1)
        else:
            MoreEvent = False

        return (length, Alarms, MoreEvent)

    @staticmethod
    def EncodeDeviceCommunicationControl(buffer: EncodeBuffer, timeDuration: int, enable_disable: int, password: str):
        if timeDuration > 0:
            ASN1.encode_context_unsigned(buffer, 0, timeDuration)
        ASN1.encode_context_enumerated(buffer, 1, enable_disable)
        if password is not None and len(password) > 0:
            ASN1.encode_context_character_string(buffer, 2, password)

    @staticmethod
    def DecodeDeviceCommunicationControl(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        timeDuration = 0
        enable_disable = 0
        password = ""

        if ASN1.decode_is_context_tag(buffer, offset + length, 0):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, timeDuration) = ASN1.decode_unsigned(buffer, offset + length, len_value)
            length += l

        if ASN1.decode_is_context_tag(buffer, offset + length, 1):
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, enable_disable) = ASN1.decode_unsigned(buffer, offset + length, len_value)
            length += l

        if length < apdu_len:
            if not ASN1.decode_is_context_tag(buffer, offset + length, 2):
                return (-1, timeDuration, enable_disable, password)

            (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, password) = ASN1.decode_character_string(buffer, offset + length, apdu_len - (offset + length), len_value_type)
            length += l

        return (length, timeDuration, enable_disable, password)

    @staticmethod
    def EncodeReinitializeDevice(buffer: EncodeBuffer, state: BACnetReinitializedStates, password: str):
        ASN1.encode_context_enumerated(buffer, 0, state)
        if password is not None and len(password) > 0:
            ASN1.encode_context_character_string(buffer, 1, password)

    @staticmethod
    def DecodeReinitializeDevice(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        state = BACnetReinitializedStates.BACNET_REINIT_IDLE
        password = ""

        if not ASN1.decode_is_context_tag(buffer, offset + length, 0):
            return (-1, state, password)

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l

        (l, value) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
        length += l
        state = BACnetReinitializedStates(value)

        if length < apdu_len:
            if not ASN1.decode_is_context_tag(buffer, offset + length, 1):
                return (-1, state, password)

            (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, password) = ASN1.decode_character_string(buffer, offset + length, apdu_len - (offset + length), len_value_type)
            length += l

        return (length, state, password)

    @staticmethod
    def EncodeReadRange(buffer: EncodeBuffer, object_id: BACnetObjectId, property_id: int, arrayIndex: int, requestType: BACnetReadRangeRequestTypes, position: int, time: datetime, count: int):
        ASN1.encode_context_object_id(buffer, 0, object_id.type, object_id.instance)
        ASN1.encode_context_enumerated(buffer, 1, property_id)
        if arrayIndex != ASN1.BACNET_ARRAY_ALL:
            ASN1.encode_context_unsigned(buffer, 2, arrayIndex)

        if requestType == BACnetReadRangeRequestTypes.RR_BY_POSITION:
            ASN1.encode_opening_tag(buffer, 3)
            ASN1.encode_application_unsigned(buffer, position)
            ASN1.encode_application_signed(buffer, count)
            ASN1.encode_closing_tag(buffer, 3)
        elif requestType == BACnetReadRangeRequestTypes.RR_BY_SEQUENCE:
            ASN1.encode_opening_tag(buffer, 6)
            ASN1.encode_application_unsigned(buffer, position)
            ASN1.encode_application_signed(buffer, count)
            ASN1.encode_closing_tag(buffer, 6)
        elif requestType == BACnetReadRangeRequestTypes.RR_BY_TIME:
            ASN1.encode_opening_tag(buffer, 7)
            ASN1.encode_application_date(buffer, time)
            ASN1.encode_application_time(buffer, time)
            ASN1.encode_application_signed(buffer, count)
            ASN1.encode_closing_tag(buffer, 7)

    @staticmethod
    def DecodeReadRange(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        object_id = BACnetObjectId()
        property = BACnetPropertyReference()
        requestType = BACnetReadRangeRequestTypes.RR_READ_ALL
        position = 0
        time = datetime(1, 1, 1)
        count = -1

        if not ASN1.decode_is_context_tag(buffer, offset + length, 0):
            return (-1, object_id, property, requestType, position, time, count)

        length += 1
        (l, object_id.type, object_id.instance) = ASN1.decode_object_id(buffer, offset + length)
        length += l

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l

        if tag_number != 1:
            return (-1, object_id, property, requestType, position, time, count)

        (l, property.propertyIdentifier) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
        length += l

        if length < apdu_len and ASN1.decode_is_context_tag(buffer, offset + length, 0):
            (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            (l, property.propertyArrayIndex) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
            length += l
        else:
            property.propertyArrayIndex = ASN1.BACNET_ARRAY_ALL

        if length < apdu_len:
            (l, tag_number) = ASN1.decode_tag_number(buffer, offset + length)
            length += l

            if tag_number == 3:
                requestType = BACnetReadRangeRequestTypes.RR_BY_POSITION
                (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l

                (l, position) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
                length += l

                (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l

                (l, count) = ASN1.decode_signed(buffer, offset + length, len_value_type)
                length += l
            elif tag_number == 6:
                requestType = BACnetReadRangeRequestTypes.RR_BY_SEQUENCE
                (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l

                (l, position) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
                length += l

                (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l

                (l, count) = ASN1.decode_signed(buffer, offset + length, len_value_type)
                length += l
            elif tag_number == 7:
                requestType = BACnetReadRangeRequestTypes.RR_BY_TIME
                (l, date) = ASN1.decode_application_date(buffer, offset + length)
                length += l

                (l, time) = ASN1.decode_application_time(buffer, offset + length)
                length += l

                time = datetime(date.year, date.month, date.day, time.hour, time.minute, time.second, time.microsecond)

                (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l

                (l, count) = ASN1.decode_signed(buffer, offset + length, len_value_type)
                length += l
            else:
                return (-1, object_id, property, requestType, position, time, count)

            (l, tag_number) = ASN1.decode_tag_number(buffer, offset + length)
            length += l

        return (length, object_id, property, requestType, position, time, count)

    @staticmethod
    def EncodeReadRangeAcknowledge(buffer: EncodeBuffer, object_id: BACnetObjectId, property_id: int, arrayIndex: int, ResultFlags: BACnetBitString, ItemCount: int, application_data: bytearray, requestType: BACnetReadRangeRequestTypes, FirstSequence: int):
        ASN1.encode_context_object_id(buffer, 0, object_id.type, object_id.instance)
        ASN1.encode_context_enumerated(buffer, 1, property_id)

        if arrayIndex != ASN1.BACNET_ARRAY_ALL:
            ASN1.encode_context_unsigned(buffer, 2, arrayIndex)

        ASN1.encode_context_bitstring(buffer, 3, ResultFlags)
        ASN1.encode_context_unsigned(buffer, 4, ItemCount)
        ASN1.encode_opening_tag(buffer, 5)

        if ItemCount != 0 and requestType != BACnetReadRangeRequestTypes.RR_BY_POSITION and requestType != BACnetReadRangeRequestTypes.RR_READ_ALL:
            ASN1.encode_context_unsigned(buffer, 6, FirstSequence)

    @staticmethod
    def DecodeReadRangeAcknowledge(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        RangeBuffer = None
        object_id = BACnetObjectId()
        property = BACnetPropertyReference()
        ResultFlag = BACnetBitString()

        if not ASN1.decode_is_context_tag(buffer, offset + length, 0):
            return (-1, RangeBuffer)

        length += 1
        (l, object_id.type, object_id.instance) = ASN1.decode_object_id(buffer, offset + length)
        length += l

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l

        if tag_number != 1:
            return (0, RangeBuffer)

        (l, property.propertyIdentifier) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
        length += l

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l
        if tag_number == 2 and length < apdu_len:
            (l, property.propertyArrayIndex) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
            length += l
        else:
            (l, ResultFlag) = ASN1.decode_bitstring(buffer, offset + length, 2)
            length += l

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l
        (l, ItemCount) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
        length += l

        if not ASN1.decode_is_opening_tag(buffer, offset + length):
            return (0, RangeBuffer)
        length += 1

        RangeBuffer = bytearray(len(buffer) - offset - length - 1)
        RangeBuffer[0: len(RangeBuffer)] = buffer[offset+length: offset+length+len(RangeBuffer)]
        return (ItemCount, RangeBuffer)

    @staticmethod
    def EncodeReadProperty(buffer: EncodeBuffer, object_id: BACnetObjectId, property_id: int, array_index: int=ASN1.BACNET_ARRAY_ALL):
        if object_id.type <= ASN1.BACNET_MAX_OBJECT:
            ASN1.encode_context_object_id(buffer, 0, object_id.type, object_id.instance)
        if property_id <= BACnetPropertyIds.MAX_BACNET_PROPERTY_ID:
            ASN1.encode_context_enumerated(buffer, 1, property_id)
        if array_index != ASN1.BACNET_ARRAY_ALL:
            ASN1.encode_context_unsigned(buffer, 2, array_index)

    @staticmethod
    def DecodeAtomicWriteFileAcknowledge(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        is_stream = False
        position = 0

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l
        if tag_number == 0:
            is_stream = True
            (l, position) = ASN1.decode_signed(buffer, offset + length, len_value_type)
            length += l
        elif tag_number == 2:
            is_stream = False
            (l, position) = ASN1.decode_signed(buffer, offset + length, len_value_type)
            length += l
        else:
            return (-1, is_stream, position)

        return (length, is_stream, position)

    @staticmethod
    def DecodeAtomicReadFileAcknowledge(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        end_of_file = False
        is_stream = False
        position = -1
        count = 0
        target_buffer = None
        target_offset = -1

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l
        if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_BOOLEAN:
            return (-1, end_of_file, is_stream, position, count, target_buffer, target_offset)

        end_of_file = len_value_type > 0
        if ASN1.decode_is_opening_tag_number(buffer, offset + length, 0):
            is_stream = True
            length += 1

            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += tag_len
            if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_SIGNED_INT:
                return (-1, end_of_file, is_stream, position, count, target_buffer, target_offset)

            (l, position) = ASN1.decode_signed(buffer, offset + length, len_value_type)
            length += l

            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += tag_len
            if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_OCTET_STRING:
                return (-1, end_of_file, is_stream, position, count, target_buffer, target_offset)

            target_buffer = buffer
            target_offset = offset + length
            count = len_value_type
            length += count

            if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 0):
                return (-1, end_of_file, is_stream, position, count, target_buffer, target_offset)

            length += 1
        elif ASN1.decode_is_opening_tag_number(buffer, offset + length, 1):
            is_stream = False
            raise Exception('Non stream File transfers are not supported')
        else:
            return (-1, end_of_file, is_stream, position, count, target_buffer, target_offset)

        return (length, end_of_file, is_stream, position, count, target_buffer, target_offset)

    @staticmethod
    def DecodeReadProperty(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        object_id = BACnetObjectId()
        property = BACnetPropertyReference()

        if apdu_len < 7:
            return (-1, object_id, property)

        if not ASN1.decode_is_context_tag(buffer, offset + length, 0):
            return (-2, object_id, property)

        length += 1
        (l, object_id.type, object_id.instance) = ASN1.decode_object_id(buffer, offset + length)
        length += l

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l
        if tag_number != 1:
            return (-2, object_id, property)

        (l, property.propertyIdentifier) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
        length += l

        if length < apdu_len:
            (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l
            if tag_number == 2 and length < apdu_len:
                (l, property.propertyArrayIndex) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
                length += l
            else:
                return (-2, object_id, property)
        else:
            property.propertyArrayIndex = ASN1.BACNET_ARRAY_ALL

        if length < apdu_len:
            return (-3, object_id, property)

        return (length, object_id, property)

    @staticmethod
    def EncodeReadPropertyAcknowledge(buffer: EncodeBuffer, object_id: BACnetObjectId, property_id: int, array_index: int, value_list: list):
        ASN1.encode_context_object_id(buffer, 0, object_id.type, object_id.instance)
        ASN1.encode_context_enumerated(buffer, 1, property_id)
        if array_index != ASN1.BACNET_ARRAY_ALL:
            ASN1.encode_context_unsigned(buffer, 2, array_index)
        ASN1.encode_opening_tag(buffer, 3)
        for value in value_list:
            ASN1.bacapp_encode_application_data(buffer, value)
        ASN1.encode_closing_tag(buffer, 3)

    @staticmethod
    def DecodeReadPropertyAcknowledge(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        object_id = BACnetObjectId()
        property = BACnetPropertyReference()
        value_list = []

        if not ASN1.decode_is_context_tag(buffer, offset, 0):
            return (-1, object_id, property, value_list)

        length = 1
        (l, object_id.type, object_id.instance) = ASN1.decode_object_id(buffer, offset + length)
        length += l

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l
        if tag_number != 1:
            return (-1, object_id, property, value_list)

        (l, property.propertyIdentifier) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
        length += l

        (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        if tag_number == 2:
            length += tag_len
            (l, property.propertyArrayIndex) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
            length += l
        else:
            property.propertyArrayIndex = ASN1.BACNET_ARRAY_ALL

        if ASN1.decode_is_opening_tag_number(buffer, offset + length, 3):
            length += 1

            while apdu_len - length > 1:
                (tag_len, value) = ASN1.bacapp_decode_application_data(buffer, offset + length, apdu_len + offset, object_id.type, BACnetPropertyIds(property.propertyIdentifier))
                if tag_len < 0:
                    return (-1, object_id, property, value_list)
                length += tag_len
                value_list.append(value)
        else:
            return (-1, object_id, property, value_list)

        if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 3):
            return (-1, object_id, property, value_list)

        length += 1

        return (length, object_id, property, value_list)

    @staticmethod
    def EncodeReadPropertyMultiple1(buffer: EncodeBuffer, properties: list):
        for value in properties:
            ASN1.encode_read_access_specification(buffer, value)

    @staticmethod
    def EncodeReadPropertyMultiple(buffer: EncodeBuffer, object_id: BACnetObjectId, properties: list):
        Services.EncodeReadPropertyMultiple1(buffer, [BACnetReadAccessSpecification(object_id, properties)])

    @staticmethod
    def DecodeReadPropertyMultiple(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        values = []
        properties = None

        while apdu_len - length > 0:
            (tmp, value) = ASN1.decode_read_access_specification(buffer, offset + length, apdu_len - length)
            if tmp < 0:
                return (-1, properties)
            length += tmp
            values.append(value)

        properties = values
        return (length, properties)

    @staticmethod
    def EncodeReadPropertyMultipleAcknowledge(buffer: EncodeBuffer, values: list):
        for value in values:
            ASN1.encode_read_access_result(buffer, value)

    @staticmethod
    def DecodeReadPropertyMultipleAcknowledge(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        _values = []
        values = None

        while apdu_len - length > 0:
            (tmp, value) = ASN1.decode_read_access_result(buffer, offset + length, apdu_len - length)
            if tmp < 0:
                return (-1, values)
            length += tmp
            _values.append(value)

        values = _values
        return (length, values)

    @staticmethod
    def EncodeWriteProperty(buffer: EncodeBuffer, object_id: BACnetObjectId, property_id: int, array_index: int, priority: int, value_list: list):
        ASN1.encode_context_object_id(buffer, 0, object_id.type, object_id.instance)
        ASN1.encode_context_enumerated(buffer, 1, property_id)

        if array_index != ASN1.BACNET_ARRAY_ALL:
            ASN1.encode_context_unsigned(buffer, 2, array_index)

        ASN1.encode_opening_tag(buffer, 3)
        for value in value_list:
            ASN1.bacapp_encode_application_data(buffer, value)
        ASN1.encode_closing_tag(buffer, 3)

        if priority != ASN1.BACNET_NO_PRIORITY:
            ASN1.encode_context_unsigned(buffer, 4, priority)

    @staticmethod
    def DecodeCOVNotifyUnconfirmed(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        subscriberProcessIdentifier = 0
        initiatingDeviceIdentifier = BACnetObjectId()
        monitoredObjectIdentifier = BACnetObjectId()
        timeRemaining = 0
        values = None

        if ASN1.decode_is_context_tag(buffer, offset + length, 0):
            (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l
            (l, subscriberProcessIdentifier) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
            length += l
        else:
            return (-1, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values)

        if ASN1.decode_is_context_tag(buffer, offset + length, 1):
            (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l
            (l, initiatingDeviceIdentifier.type, initiatingDeviceIdentifier.instance) = ASN1.decode_object_id(buffer, offset + length)
            length += l
        else:
            return (-1, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values)

        if ASN1.decode_is_context_tag(buffer, offset + length, 2):
            (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l
            (l, monitoredObjectIdentifier.type, monitoredObjectIdentifier.instance) = ASN1.decode_object_id(buffer, offset + length)
            length += l
        else:
            return (-1, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values)

        if ASN1.decode_is_context_tag(buffer, offset + length, 3):
            (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l
            (l, timeRemaining) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
            length += l
        else:
            return (-1, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values)

        if not ASN1.decode_is_context_tag(buffer, offset + length, 4):
            return (-1, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values)

        length += 1
        _values = []
        while not ASN1.decode_is_closing_tag_number(buffer, offset + length, 4):
            new_entry = BACnetPropertyValue()

            if ASN1.decode_is_context_tag(buffer, offset + length, 0):
                (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l
                (l, new_entry.property.propertyIdentifier) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
                length += l
            else:
                return (-1, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values)

            if ASN1.decode_is_context_tag(buffer, offset + length, 1):
                (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l
                (l, new_entry.property.propertyArrayIndex) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
                length += l
            else:
                return (-1, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values)

            if not ASN1.decode_is_context_tag(buffer, offset + length, 2):
                return (-1, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values)

            length += 1
            b_values = []
            while not ASN1.decode_is_closing_tag_number(buffer, offset + length, 2):
                (tmp, b_value) = ASN1.bacapp_decode_application_data(buffer, offset + length, apdu_len + offset,monitoredObjectIdentifier.type, BACnetPropertyIds(new_entry.property.propertyIdentifier))
                if tmp < 0:
                    return (-1, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values)
                length += tmp
                b_values.append(b_value)
            new_entry.value = b_values

            length += 1
            if ASN1.decode_is_context_tag(buffer, offset + length, 3):
                (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l
                (l, new_entry.priority) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
                length += l
            else:
                new_entry.priority = ASN1.BACNET_NO_PRIORITY

            _values.append(new_entry)
        values = _values

        return (length, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values)

    @staticmethod
    def DecodeWriteProperty(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        object_id = BACnetObjectId()
        value = BACnetPropertyValue()

        if not ASN1.decode_is_context_tag(buffer, offset + length, 0):
            return (-1, object_id, value)

        length += 1
        (l, object_id.type, object_id.instance) = ASN1.decode_object_id(buffer, offset + length)
        length += l

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l
        if tag_number != 1:
            return (-1, object_id, value)

        (l, property.propertyIdentifier) = ASN1.decode_enumerated(buffer, offset + length, len_value_type)
        length += l

        (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        if tag_number == 2:
            length += tag_len
            (l, property.propertyArrayIndex) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
            length += l
        else:
            value.property.propertyArrayIndex = ASN1.BACNET_ARRAY_ALL

        if not ASN1.decode_is_opening_tag_number(buffer, offset + length, 3):
            return (-1, object_id, value)
        length += 1

        _value_list = []

        if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 3):
            return (-1, object_id, value)
        length += 1

        value.priority = ASN1.BACNET_MAX_PRIORITY

        if length < apdu_len:
            (tag_len, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            if tag_number == 4:
                length += tag_len
                (l, unsigned_value) = ASN1.decode_unsigned(buffer, offset + length, len_value_type)
                length += l
                if unsigned_value >= ASN1.BACNET_MIN_PRIORITY and unsigned_value <= ASN1.BACNET_MAX_PRIORITY:
                    value.priority = unsigned_value
                else:
                    return (-1, object_id, value)

        return (length, object_id, value)

    @staticmethod
    def EncodeWriteObjectMultipleV1(buffer: EncodeBuffer, object_id: BACnetObjectId, value_list: list):
        ASN1.encode_context_object_id(buffer, 0, object_id.type, object_id.instance)
        ASN1.encode_opening_tag(buffer, 1)
        for p_value in value_list:
            ASN1.encode_context_enumerated(buffer, 0, p_value.property.propertyIdentifier)

            if p_value.property.propertyArrayIndex != ASN1.BACNET_ARRAY_ALL:
                ASN1.encode_context_unsigned(buffer, 1, p_value.property.propertyArrayIndex)

            ASN1.encode_opening_tag(buffer, 2)
            for value in p_value.value:
                ASN1.bacapp_encode_application_data(buffer, value)
            ASN1.encode_closing_tag(buffer, 2)

            if p_value.priority != ASN1.BACNET_NO_PRIORITY:
                ASN1.encode_context_unsigned(buffer, 3, p_value.priority)
        ASN1.encode_closing_tag(buffer, 1)

    @staticmethod
    def EncodeWriteObjectMultiple(buffer: EncodeBuffer, value_list: list):
        for r_value in value_list:
            Services.EncodeWriteObjectMultipleV1(buffer, r_value.objectIdentifier, r_value.values)

    @staticmethod
    def DecodeCreateObject(buffer: bytearray, offset: int, apdu_len: int):
        length = 0

        object_id = BACnetObjectId()
        values_refs = None

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l

        if tag_number == 0 and apdu_len > length:
            apdu_len -= length
            if apdu_len >= 4:
                (l, object_id.type, object_id.instance) = ASN1.decode_context_object_id(buffer, offset + length, 1)
                length += l
            else:
                return (-1, object_id, values_refs)
        else:
            return (-1, object_id, values_refs)

        if ASN1.decode_is_closing_tag(buffer, offset + length):
            length += 1

        if len(buffer) == offset + length:
            return (-1, object_id, values_refs)

        if not ASN1.decode_is_opening_tag_number(buffer, offset + length, 1):
            return (-1, object_id, values_refs)
        length += 1

        _values = []
        while apdu_len - length > 1:
            new_entry = BACnetPropertyValue()

            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l
            if tag_number == 0:
                (l, property_id) = ASN1.decode_enumerated(buffer, offset + length, len_value)
                length += l
            else:
                return (-1, object_id, values_refs)

            ulVal = ASN1.BACNET_ARRAY_ALL
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l
            if tag_number == 1:
                (l, ulVal) = ASN1.decode_enumerated(buffer, offset + length, len_value)
                length += l

                (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l
            new_entry.property = BACnetPropertyReference(id=property_id, array_index=ulVal)

            if tag_number == 2 and ASN1.decode_is_opening_tag(buffer, offset + length - 1):
                values = []
                while not ASN1.decode_is_closing_tag(buffer, offset + length):
                    (l, value) = ASN1.bacapp_decode_application_data(buffer, offset + length, apdu_len + offset, object_id.type, BACnetPropertyIds(property_id))
                    if l <= 0:
                        return (-1, object_id, values_refs)
                    length += l
                    values.append(value)
                length += 1
                new_entry.value = values
            else:
                return (-1, object_id, values_refs)

            _values.append(new_entry)

        if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 1):
            return (-1, object_id, values_refs)

        length += 1

        values_refs = _values
        return (-1, object_id, values_refs)

    @staticmethod
    def DecodeDeleteObject(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        object_id = BACnetObjectId()

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset)

        if tag_number != 12:
            return (-1, object_id)

        length = 1
        (l, object_id.type, object_id.instance) = ASN1.decode_object_id(buffer, offset + length)
        length += l

        if length == apdu_len:
            return (length, object_id)
        else:
            return (-1, object_id)

    @staticmethod
    def EncodeCreateObjectAcknowledge(buffer: EncodeBuffer, object_id: BACnetObjectId):
        ASN1.encode_application_object_id(buffer, object_id.type, object_id.instance)

    @staticmethod
    def DecodeWritePropertyMultiple(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        object_id = BACnetObjectId()
        values_refs = None

        (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
        length += l
        if tag_number == 0 and apdu_len > length:
            apdu_len -= length
            if apdu_len >= 4:
                (l, object_id.type, object_id.instance) = ASN1.decode_object_id(buffer, offset + length)
                length += l
            else:
                return (-1, object_id, values_refs)
        else:
            return (-1, object_id, values_refs)

        if not ASN1.decode_is_opening_tag_number(buffer, offset + length, 1):
            return (-1, object_id, values_refs)
        length += 1

        _values = []
        while apdu_len - length > 1:
            new_entry = BACnetPropertyValue()

            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l
            if tag_number == 0:
                (l, property_id) = ASN1.decode_enumerated(buffer, offset + length, len_value)
                length += l
            else:
                return (-1, object_id, values_refs)

            ulVal = ASN1.BACNET_ARRAY_ALL
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l
            if tag_number == 1:
                (l, ulVal) = ASN1.decode_enumerated(buffer, offset + length, len_value)
                length += l

                (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l
            new_entry.property = BACnetPropertyReference(property_id, ulVal)

            if tag_number == 2 and ASN1.decode_is_opening_tag(buffer, offset + length - 1):
                values = []
                while not ASN1.decode_is_closing_tag(buffer, offset + length):
                    (l, value) = ASN1.bacapp_decode_application_data(buffer, offset + length, apdu_len + offset, object_id.type, BACnetPropertyIds(property_id))
                    if l <= 0:
                        return (-1, object_id, values_refs)
                    length += l
                    values.append(value)
                length += 1
                new_entry.value = values
            else:
                return (-1, object_id, values_refs)

            ulVal = ASN1.BACNET_NO_PRIORITY
            (l, tag_number, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l

            if tag_number == 3:
                (l, ulVal) = ASN1.decode_unsigned(buffer, offset + length, len_value)
                length += l
            else:
                length -= 1
            new_entry.priority = ulVal

            _values.append(new_entry)

        if not ASN1.decode_is_closing_tag_number(buffer, offset + length, 1):
            return (-1, object_id, values_refs)

        length += 1

        values_refs = _values
        return (-1, object_id, values_refs)

    @staticmethod
    def EncodeTimeSync(buffer: EncodeBuffer, time: datetime):
        ASN1.encode_application_date(buffer, time)
        ASN1.encode_application_time(buffer, time)

    @staticmethod
    def DecodeTimeSync(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        dateTime = datetime(1, 1, 1)

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset)

        if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_DATE:
            return (-1, dateTime)

        (l, d_date) = ASN1.decode_date(buffer, offset + length)
        length += l

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset)

        if tag_number != BACnetApplicationTags.BACNET_APPLICATION_TAG_TIME:
            return (-1, dateTime)

        (l, t_date) = ASN1.decode_bacnet_time(buffer, offset + length)
        length += l

        dateTime = datetime(d_date.year, d_date.month, d_date.day, t_date.hour, t_date.minute, t_date.second, t_date.microsecond)
        return (length, dateTime)

    @staticmethod
    def EncodeError(buffer: EncodeBuffer, error_class: BACnetErrorClasses, error_code: BACnetErrorCodes):
        ASN1.encode_application_enumerated(buffer, error_class)
        ASN1.encode_application_enumerated(buffer, error_code)

    @staticmethod
    def DecodeError(buffer: bytearray, offset: int, apdu_len: int):
        length = 0
        org_offset = offset

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset)
        offset += l
        (l, tmp) = ASN1.decode_enumerated(buffer, offset, len_value_type)
        offset += l
        error_class = BACnetErrorClasses(tmp)

        (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset)
        offset += l
        (l, tmp) = ASN1.decode_enumerated(buffer, offset, len_value_type)
        offset += l
        error_code = BACnetErrorCodes(tmp)

        return (offset-org_offset, error_class, error_code)

    @staticmethod
    def EncodeLogRecord(buffer: EncodeBuffer, record: BACnetLogRecord):
        ASN1.encode_opening_tag(buffer, 0)
        ASN1.encode_application_date(buffer, record.timestamp)
        ASN1.encode_application_time(buffer, record.timestamp)
        ASN1.encode_closing_tag(buffer, 0)

        if record.type != BACnetTrendLogValueType.TL_TYPE_NULL:
            if record.type == BACnetTrendLogValueType.TL_TYPE_ERROR:
                ASN1.encode_opening_tag(buffer, 1)
                ASN1.encode_opening_tag(buffer, 8)
                err: BACnetError = record.GetValue(BACnetError)
                Services.EncodeError(buffer, err.error_class, err.error_code)
                ASN1.encode_closing_tag(buffer, 8)
                ASN1.encode_closing_tag(buffer, 1)
                return

            ASN1.encode_opening_tag(buffer, 1)
            tmp1 = EncodeBuffer()
            if record.type == BACnetTrendLogValueType.TL_TYPE_ANY:
                raise Exception('NotImplemented')
            elif record.type == BACnetTrendLogValueType.TL_TYPE_BITS:
                ASN1.encode_bitstring(tmp1, record.GetValue(BACnetBitString))
            elif record.type == BACnetTrendLogValueType.TL_TYPE_BOOL:
                tmp1.AddByte(1 if record.GetValue(bool) else 0)
            elif record.type == BACnetTrendLogValueType.TL_TYPE_DELTA:
                ASN1.encode_bacnet_real(tmp1, record.GetValue(float))
            elif record.type == BACnetTrendLogValueType.TL_TYPE_ENUM:
                ASN1.encode_application_enumerated(tmp1, record.GetValue(int))
            elif record.type == BACnetTrendLogValueType.TL_TYPE_REAL:
                ASN1.encode_bacnet_real(tmp1, record.GetValue(float))
            elif record.type == BACnetTrendLogValueType.TL_TYPE_SIGN:
                ASN1.encode_bacnet_signed(tmp1, record.GetValue(int))
            elif record.type == BACnetTrendLogValueType.TL_TYPE_STATUS:
                ASN1.encode_bitstring(tmp1, record.GetValue(BACnetBitString))
            elif record.type == BACnetTrendLogValueType.TL_TYPE_UNSIGN:
                ASN1.encode_bacnet_unsigned(tmp1, record.GetValue(int))

            ASN1.encode_tag(buffer, record.type, False, tmp1.offset)
            buffer.Add(tmp1.buffer, tmp1.offset)
            ASN1.encode_closing_tag(buffer, 1)

        if record.statusFlags.bits_used > 0:
            ASN1.encode_opening_tag(buffer, 2)
            ASN1.encode_application_bitstring(buffer, record.statusFlags)
            ASN1.encode_closing_tag(buffer, 2)

    @staticmethod
    def DecodeLogRecord(buffer: bytearray, offset: int, length1: int, n_curves: int):
        length = 0
        records = []

        (l, tag_number) = ASN1.decode_tag_number(buffer, offset + length)
        length += l

        if tag_number != 0:
            return (-1, records)

        (l, date) = ASN1.decode_application_date(buffer, offset + length)
        length += l
        (l, time) = ASN1.decode_application_time(buffer, offset + length)
        length += l

        dt = datetime(date.year, date.month, date.day, time.hour, time.minute, time.second, time.microsecond)

        if not ASN1.decode_is_closing_tag(buffer, offset + length):
            return (-1, records)

        length += 1

        (l, tag_number) = ASN1.decode_tag_number(buffer, offset + length)
        length += l
        if tag_number != 1:
            return (-1, records)

        ContextTagType = 0
        for CurveNumber in range(n_curves):
            records.append(BACnetLogRecord())
            (l, ContextTagType, len_value) = ASN1.decode_tag_number_and_value(buffer, offset + length)
            length += l
            records[CurveNumber].timestamp = dt
            records[CurveNumber].type = BACnetTrendLogValueType(ContextTagType)

            if ContextTagType == BACnetTrendLogValueType.TL_TYPE_STATUS:
                (l, sval) = ASN1.decode_bitstring(buffer, offset + length, len_value)
                length += l
                records[CurveNumber].Value = sval
            elif ContextTagType == BACnetTrendLogValueType.TL_TYPE_BOOL:
                records[CurveNumber].Value = True if buffer[offset + length] > 0 else False
                length += 1
            elif ContextTagType == BACnetTrendLogValueType.TL_TYPE_REAL:
                (l, rval) = ASN1.decode_real(buffer, offset + length)
                length += l
                records[CurveNumber].Value = rval
            elif ContextTagType == BACnetTrendLogValueType.TL_TYPE_ENUM:
                (l, evalue) = ASN1.decode_enumerated(buffer, offset + length, len_value)
                length += l
                records[CurveNumber].Value = evalue
            elif ContextTagType == BACnetTrendLogValueType.TL_TYPE_SIGN:
                (l, ival) = ASN1.decode_signed(buffer, offset + length, len_value)
                length += l
                records[CurveNumber].Value = ival
            elif ContextTagType == BACnetTrendLogValueType.TL_TYPE_UNSIGN:
                (l, uinval) = ASN1.decode_unsigned(buffer, offset + length, len_value)
                length += l
                records[CurveNumber].Value = uinval
            elif ContextTagType == BACnetTrendLogValueType.TL_TYPE_ERROR:
                (l, Errclass, Errcode) = Services.DecodeError(buffer, offset + length, length1)
                length += l
                records[CurveNumber].Value = BACnetError(Errclass, Errcode)
                length += 1
            elif ContextTagType == BACnetTrendLogValueType.TL_TYPE_NULL:
                length += 1
                records[CurveNumber].Value = None
            elif ContextTagType == BACnetTrendLogValueType.TL_TYPE_DELTA:
                (l, dval) = ASN1.decode_real(buffer, offset + length)
                length += l
                records[CurveNumber].Value = dval
            elif ContextTagType == BACnetTrendLogValueType.TL_TYPE_NULL:
                raise Exception('NotImplemente')
            elif ContextTagType == BACnetTrendLogValueType.TL_TYPE_BITS:
                (l, bval) = ASN1.decode_bitstring(buffer, offset + length, len_value)
                length += l
                records[CurveNumber].Value = bval
            else:
                return (0, records)

        if not ASN1.decode_is_closing_tag(buffer, offset + length):
            return (-1, records)
        length += 1

        if length < length1:
            (l, tag_number) = ASN1.decode_tag_number(buffer, offset + length)
            if tag_number == 2:
                length += 1

                (l, StatusFlags) = ASN1.decode_bitstring(buffer, offset + length, 2)
                length += l

                for CurveNumber in range(n_curves):
                    records[CurveNumber].statusFlags = StatusFlags

        return (length, records)

##BACnetTransport############

class BVLC:
    MyBBMDTransport = None
    BroadcastAdd: str = ''
    BBMD_FD_ServiceActivated: bool = False
    BVLL_TYPE_BACNET_IP: int = 0x81
    BVLC_HEADER_LENGTH: int = 4
    BVLC_MAX_APDU: BACnetMaxAdpu = BACnetMaxAdpu.MAX_APDU1476
    ForeignDevices: list = []
    BBMDs: list = []
    AutorizedFDR: list = []
    MessageReceived: Callable = None
    lockDevice = threading.Lock()
    lockBBMD = threading.Lock()

    def __init__(self, Transport):
        self.MyBBMDTransport = Transport
        BroadcastAdd = self.MyBBMDTransport.GetBroadcastAddress().__str__().split(':')[0]

    def FDList(self):
        sb = ''
        with self.lockDevice:
            for item in self.ForeignDevices:
                if item(1) > datetime.now():
                    self.ForeignDevices.remove(item)
            for client in self.ForeignDevices:
                sb = f"{sb}{client[0].Address}:{client[0].Port};"
        return sb

    def AddFDRAutorisationRule(self, IpRule):
        self.AutorizedFDR.append(IpRule)

    def AddBBMDPeer(self, BBMD: tuple, Mask: str):
        self.BBMD_FD_ServiceActivated = True
        if BBMD is not None:
            with self.lockBBMD:
                self.BBMDs.append((BBMD, Mask))

    def RegisterForeignDevice(self, sender, TTL: int):
        with self.lockDevice:
            for item in self.ForeignDevices:
                if item(0).Equals(sender):
                    self.ForeignDevices.remove(item)

            Expiration = datetime.now() + timedelta(TTL + 30)
            if len(self.AutorizedFDR) == 0:
                self.ForeignDevices.append((sender, Expiration))
            else:
                for r in self.AutorizedFDR:
                    if r.match(sender.Address.__str__()):
                        self.ForeignDevices.append((sender, Expiration))

    def SendToFDs(self, buffer: bytearray, msg_length: int, EPsender=None):
        with self.lockDevice:
            for item in self.ForeignDevices:
                if item(1) > datetime.now():
                    self.ForeignDevices.remove(item)
            for client in self.ForeignDevices:
                if not client[0].Equals(EPsender):
                    self.MyBBMDTransport.Send(buffer, msg_length, client[1])

    def BBMDSentAdd(self, BBMD: tuple, Mask: str):
        bm = Mask.encode('utf-8')
        bip = BBMD[0].encode('utf-8')

        for i in range(len(bm)):
            bip[i] = BACnetTool.int_to_byte(bip[i] | (~bm[i]))

        return (bip, BBMD[1])

    def SendToBBMDs(self, buffer: bytearray, msg_length: int):
        with self.BBMDs:
            for e in self.BBMDs:
                endpoint = self.BBMDSentAdd(e[0], e[1])
                self.MyBBMDTransport.Send(buffer, msg_length, endpoint)

    def First4BytesHeaderEncode(self, b: bytearray, function: BACnetBvlcFunctions, msg_length: int):
        b[0] = BVLC.BVLL_TYPE_BACNET_IP
        b[1] = function
        b[2] = BACnetTool.int_to_byte(((msg_length) & 0xFF00) >> 8)
        b[3] = BACnetTool.int_to_byte(((msg_length) & 0x00FF) >> 0)

    #TODO
    def Forward_NPDU(self, buffer: bytearray, msg_length: int, ToGlobalBroadcast: bool, EPsender):
        b = bytearray(msg_length + 6)
        b[6:] = buffer[0: msg_length]
        self.First4BytesHeaderEncode(b, BACnetBvlcFunctions.BVLC_FORWARDED_NPDU, msg_length + 6)
        BacSender: BACnetAddress = BACnetIpUdpProtocolTransport.ConvertBACnetAddress(EPsender)
        for i in range(len(BacSender.adr)):
            b[4 + i] = BacSender.adr[i]
        self.SendToBBMDs(b, msg_length + 6)
        self.SendToFDs(b, msg_length + 6, EPsender)
        if ToGlobalBroadcast is ToGlobalBroadcast:
            self.MyBBMDTransport.Send(b, msg_length + 6, (self.BroadcastAdd, self.MyBBMDTransport.SharedPort))

    def SendResult(self, sender, ResultCode: BACnetBvlcResults):
        b = bytearray(6)
        self.First4BytesHeaderEncode(b, BACnetBvlcFunctions.BVLC_RESULT, 6)
        b[4] = BACnetTool.int_to_byte((BACnetTool.int_to_ushort(ResultCode) & 0xFF00) >> 8)
        b[5] = BACnetTool.int_to_byte(BACnetTool.int_to_ushort(ResultCode) & 0xFF)
        self.MyBBMDTransport.Send(b, 6, sender)

    def SendRegisterAsForeignDevice(self, BBMD, TTL: int):
        b = bytearray(6)
        self.First4BytesHeaderEncode(b, BACnetBvlcFunctions.BVLC_REGISTER_FOREIGN_DEVICE, 6)
        b[4] = BACnetTool.int_to_byte((TTL & 0xFF00) >> 8)
        b[5] = BACnetTool.int_to_byte(TTL & 0xFF)
        self.MyBBMDTransport.Send(b, 6, BBMD)

    def SendReadBroadCastTable(self, BBMD):
        b = bytearray(4)
        self.First4BytesHeaderEncode(b, BACnetBvlcFunctions.BVLC_READ_BROADCAST_DIST_TABLE, 4)
        self.MyBBMDTransport.Send(b, 4, BBMD)

    def SendReadFDRTable(self, BBMD):
        b = bytearray(4)
        self.First4BytesHeaderEncode(b, BACnetBvlcFunctions.BVLC_READ_FOREIGN_DEVICE_TABLE, 4)
        self.MyBBMDTransport.Send(b, 4, BBMD)

    def SendWriteBroadCastTable(self, BBMD, Entries: list):
        b = bytearray(4 + 10 * len(Entries))
        self.First4BytesHeaderEncode(b, BACnetBvlcFunctions.BVLC_WRITE_BROADCAST_DISTRIBUTION_TABLE, 4 + 10 * len(Entries))
        for i in range(len(Entries)):
            b[4 + i * 10: 8 + i * 10] = Entries[i][0].Address.GetAddressBytes()[0: 4]
            b[8 + i * 10] = BACnetTool.int_to_byte(Entries[i][0].Port >> 8)
            b[9 + i * 10] = BACnetTool.int_to_byte(Entries[i][0].Port & 0xFF)
            b[10 + i * 10: 14 + i * 10] = Entries[i][1].Address.GetAddressBytes()[0: 4]

        self.MyBBMDTransport.Send(b, 4 + 10 * len(Entries), BBMD)

    def SendDeleteForeignDeviceEntry(self, BBMD, Fdevice):
        b = bytearray(4 + 6)
        self.First4BytesHeaderEncode(b, BACnetBvlcFunctions.BVLC_READ_FOREIGN_DEVICE_TABLE, 4 + 6)
        b[4:8] = Fdevice.Address.GetAddressBytes()[0:4]
        b[8] = BACnetTool.int_to_byte(Fdevice.Port >> 8)
        b[9] = BACnetTool.int_to_byte(Fdevice.Port & 0xFF)
        self.MyBBMDTransport.Send(b, 4 + 6, BBMD)

    def SendRemoteWhois(self, buffer: bytearray, BBMD, msg_length: int):
        self.Encode(buffer, 0, BACnetBvlcFunctions.BVLC_DISTRIBUTE_BROADCAST_TO_NETWORK, msg_length)
        self.MyBBMDTransport.Send(buffer, msg_length, BBMD)

    def Encode(self, buffer: bytearray, offset: int, function: BACnetBvlcFunctions, msg_length: int):
        self.First4BytesHeaderEncode(buffer, function, msg_length)
        if self.BBMD_FD_ServiceActivated is True and function == BACnetBvlcFunctions.BVLC_ORIGINAL_BROADCAST_NPDU:
            me = self.MyBBMDTransport.LocalEndPoint
            if me.Address.__str__() != '0.0.0.0':
                self.Forward_NPDU(buffer, msg_length, False, me)
        return 4

    def Decode(self, buffer: bytearray, offset: int, sender):
        function = buffer[1]
        msg_length = (buffer[2] << 8) | (buffer[3] << 0)
        if buffer[0] != BVLC.BVLL_TYPE_BACNET_IP or msg_length != len(buffer):
            return (-1, function, msg_length)

        if function == BACnetBvlcFunctions.BVLC_RESULT:
            ResultCode = (buffer[4] << 8) + buffer[5]
            if self.MessageReceived is not None:
                self.MessageReceived(sender, function, BACnetBvlcResults(ResultCode), None)
            return (0, function, msg_length)
        elif function == BACnetBvlcFunctions.BVLC_ORIGINAL_UNICAST_NPDU:
            return (4, function, msg_length)
        elif function == BACnetBvlcFunctions.BVLC_ORIGINAL_BROADCAST_NPDU:
            if self.BBMD_FD_ServiceActivated is True:
                self.Forward_NPDU(buffer, msg_length, False, sender)
            return (4, function, msg_length)
        elif function == BACnetBvlcFunctions.BVLC_FORWARDED_NPDU:
            if self.BBMD_FD_ServiceActivated is True and msg_length >= 10:
                with self.BBMDs:
                    for item in self.BBMDs:
                        if item(0).Address.Equals(sender.Address):
                            del item
                            self.SendToFDs(buffer, msg_length)

                            self.MyBBMDTransport.Send(buffer, msg_length, (self.BroadcastAdd, self.MyBBMDTransport.SharedPort))
            return (10, function, msg_length)
        elif function == BACnetBvlcFunctions.BVLC_DISTRIBUTE_BROADCAST_TO_NETWORK:
            if self.BBMD_FD_ServiceActivated is True:
                with self.ForeignDevices:
                    for item in self.ForeignDevices:
                        if item(0).Equals(sender):
                            self.Forward_NPDU(buffer, msg_length, True, sender)
                        else:
                            self.SendResult(sender, BACnetBvlcResults.BVLC_RESULT_DISTRIBUTE_BROADCAST_TO_NETWORK_NAK)
            return (0, function, msg_length)
        elif function == BACnetBvlcFunctions.BVLC_REGISTER_FOREIGN_DEVICE:
            if self.BBMD_FD_ServiceActivated is True and msg_length == 6:
                TTL = (buffer[4] << 8) + buffer[5]
                self.RegisterForeignDevice(sender, TTL)
                self.SendResult(sender, BACnetBvlcResults.BVLC_RESULT_SUCCESSFUL_COMPLETION)
            return (0, function, msg_length)
        elif function == BACnetBvlcFunctions.BVLC_READ_FOREIGN_DEVICE_TABLE:
            self.SendResult(sender, BACnetBvlcResults.BVLC_RESULT_READ_FOREIGN_DEVICE_TABLE_NAK)
            return (0, function, msg_length)
        elif function == BACnetBvlcFunctions.BVLC_DELETE_FOREIGN_DEVICE_TABLE_ENTRY:
            self.SendResult(sender, BACnetBvlcResults.BVLC_RESULT_DELETE_FOREIGN_DEVICE_TABLE_ENTRY_NAK)
            return (0, function, msg_length)
        elif function == BACnetBvlcFunctions.BVLC_READ_BROADCAST_DIST_TABLE:
            self.SendResult(sender, BACnetBvlcResults.BVLC_RESULT_READ_BROADCAST_DISTRIBUTION_TABLE_NAK)
            return (0, function, msg_length)
        elif function == BACnetBvlcFunctions.BVLC_WRITE_BROADCAST_DISTRIBUTION_TABLE or function == BACnetBvlcFunctions.BVLC_READ_BROADCAST_DIST_TABLE_ACK:
            NbEntries = int((msg_length - 4) / 10)
            Entries = []
            for i in range(0, NbEntries):
                add = struct.unpack('I', buffer[4 + i * 10:8 + i * 10])[0]
                buffer_port = bytearray(2)
                for j in range(2):
                    buffer_port[j] = buffer[8 + i * 10 + 1 - j]
                port = struct.unpack('I', buffer_port)[0]

                Mask = buffer[10 + i * 10:14 + i * 10]
                entry = (add, port)
                Entries.append(entry)

            if self.MessageReceived is not None and function == BACnetBvlcFunctions.BVLC_READ_BROADCAST_DIST_TABLE_ACK:
                self.MessageReceived(sender, function, BACnetBvlcResults.BVLC_RESULT_SUCCESSFUL_COMPLETION, Entries)

            if function == BACnetBvlcFunctions.BVLC_WRITE_BROADCAST_DISTRIBUTION_TABLE:
                self.SendResult(sender, BACnetBvlcResults.BVLC_RESULT_WRITE_BROADCAST_DISTRIBUTION_TABLE_NAK)

            return (0, function, msg_length)
        elif function == BACnetBvlcFunctions.BVLC_READ_FOREIGN_DEVICE_TABLE_ACK:
            NbEntries = int((msg_length - 4) / 10)
            Entries = []

            for i in range(0, NbEntries):
                add = struct.unpack('I', buffer[4 + i * 10:8 + i * 10])[0]

                buffer_port = bytearray(2)
                for j in range(2):
                    buffer_port[j] = buffer[8 + i * 10 + 1 - j]
                port = struct.unpack('I', buffer_port)[0]

                for j in range(2):
                    buffer_port[j] = buffer[10 + i * 10 + 1 - j]
                TTL = struct.unpack('I', buffer_port)[0]

                for j in range(2):
                    buffer_port[j] = buffer[12 + i * 10 + 1 - j]
                RemainTTL = struct.unpack('I', buffer_port)[0]

                entry = (add, port)
                Entries.append(entry)

            if self.MessageReceived is not None:
                self.MessageReceived(sender, function, BACnetBvlcResults.BVLC_RESULT_SUCCESSFUL_COMPLETION, Entries)

            return (0, function, msg_length)
        else:
            return (-1, None, None)

class BACnetTransport():

    def __init__(self):
        self.events: dict = {}
        self.transport = None
        self.MaxInfoFrames: int = 0

    def Dispose(self):
        pass

    def GetType(self) -> BACnetAddressTypes:
        return BACnetAddressTypes.NONE

    def Send(self, buffer: bytearray, data_length: int, ep: tuple):
        pass

    def Send1(self, buffer: bytearray, offset: int, data_length: int, address: BACnetAddress, wait_for_transmission: bool, timeout: float):
        pass

    def GetBroadcastAddress(self) -> BACnetAddress:
        pass

    def Start(self):
        pass

    def WaitForAllTransmits(self, timeout: float):
        pass

    def MaxBufferLength(self) -> int:
        pass

    def HeaderLength(self) -> int:
        pass

    def MaxAdpuLength(self) -> BACnetMaxAdpu:
        pass

    def Equals(self, obj):
        pass

class BACnetIpUdpProtocolTransport(BACnetTransport, asyncio.Protocol):
    m_port: int = 0
    m_shared_conn: Any = None
    m_exclusive_conn: Any = None
    bvlc: BVLC = None
    m_exclusive_port: bool = False
    m_dont_fragment: bool = False
    m_max_payload: int = 0
    m_local_endpoint: str = ''
    headerlength: int = BVLC.BVLC_HEADER_LENGTH

    def __init__(self, port: int, use_exclusive_port: bool = False, dont_fragment: bool = False, max_payload: int = 1472, local_endpoint_ip: str = "192.168.0.44"):
        super().__init__()
        self.m_port = port
        self.m_max_payload = max_payload
        self.m_exclusive_port = use_exclusive_port
        self.m_dont_fragment = dont_fragment
        self.m_local_endpoint = local_endpoint_ip
        self.transport = None

        self.loop = asyncio.get_event_loop()

    def __str__(self):
        return f"Udp:{self.m_port}"

    def __hash__(self):
        return hash(self.m_port)

    def __eq__(self, other):
        return self.Equals(other)

    def Equals(self, obj):
        if not isinstance(obj, BACnetIpUdpProtocolTransport):
            return False
        return obj.m_port == self.m_port

    def GetType(self) -> BACnetAddressTypes:
        return BACnetAddressTypes.IP

    def Open(self):
        coro = self.loop.create_datagram_endpoint(lambda: self, local_addr=(self.m_local_endpoint, self.m_port))
        self.loop.run_until_complete(coro)
        self.loop.run_forever()

    def Dispose(self):
        if self.transport is not None:
            self.transport.close()
        self.loop.stop()
        time.sleep(2)

    def GetBroadcastAddress(self) -> BACnetAddress:
        broadcast = "255.255.255.255"
        for interface, data in net_if_addrs().items():
            for addr in data:
                if addr.family == 2:
                    if addr.address == self.m_local_endpoint:
                        broadcast = IPv4Network(addr.address + '/' + addr.netmask, False).broadcast_address
                        break
        bc = self.ConvertBACnetAddress((broadcast, self.m_port))
        bc.net = 0xFFFF
        return bc

    def MaxBufferLength(self) -> int:
        return self.m_max_payload

    def Start(self):

        self.bvlc = BVLC(self)

        threading.Thread(target=self.Open, daemon=False).start()

    def WaitForAllTransmits(self, timeout: float):
        return True

    def connection_made(self, transport):
        self.transport = transport
        sock = transport.get_extra_info("socket")
        sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        sock.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)

    def connection_lost(self, exc):
        self.transport = None

    def datagram_received(self, data, addr):
        asyncio.ensure_future(self.OnReceiveData(data, addr))

    @staticmethod
    def ConvertBACnetAddress(addr: tuple):
        tmp1 = bytes(map(int, addr[0].split('.')))   #addr[0].encode('utf-8')
        tmp2 = addr[1].to_bytes(2, byteorder='big', signed=False)    #从低地址向高地址进行的
        return BACnetAddress(type=BACnetAddressTypes.IP, net=0, adr=tmp1+tmp2)

    def Convert(self, addr: BACnetAddress):
        ip_address = f"{addr.adr[0]}.{addr.adr[1]}.{addr.adr[2]}.{addr.adr[3]}"
        port = BACnetTool.int_to_ushort((addr.adr[4] << 8) | (addr.adr[5] << 0))
        return (ip_address, port)

    def SendRegisterAsForeignDevice(self, BBMD, TTL: int):
        self.bvlc.SendRegisterAsForeignDevice(BBMD, TTL)
        return True

    def SendRemoteWhois(self, buffer, BBMD, msg_length):
        self.bvlc.SendRemoteWhois(buffer, BBMD, msg_length)
        return True

    def MaxAdpuLength(self) -> BACnetMaxAdpu:
        return self.bvlc.BVLC_MAX_APDU

    def HeaderLength(self):
        return self.bvlc.BVLC_HEADER_LENGTH

    def MaxInfoFrames(self):
        return 0xff

    def Send(self, buffer: bytearray, data_length: int, ep: tuple):
        return self.transport.sendto(buffer, ep)

    def Send1(self, buffer: bytearray, offset: int, data_length: int, address: BACnetAddress, wait_for_transmission: bool, timeout: float):
        full_length = data_length + self.headerlength
        self.bvlc.Encode(buffer, offset - self.bvlc.BVLC_HEADER_LENGTH, BACnetBvlcFunctions.BVLC_ORIGINAL_BROADCAST_NPDU if address.net == 0xFFFF else BACnetBvlcFunctions.BVLC_ORIGINAL_UNICAST_NPDU, full_length)
        ep = self.Convert(address)
        self.transport.sendto(buffer[:full_length], ep)
        return full_length

    async def OnReceiveData(self, local_buffer, address):
        rx = len(local_buffer)
        if rx < self.bvlc.BVLC_HEADER_LENGTH or rx == 0:
            return    #.debug("recieved garbage")
        else:
            (HEADER_LENGTH, function, msg_length) = self.bvlc.Decode(local_buffer, 0, address)
            if HEADER_LENGTH == -1:
                print('Unknow BVLC Header')
                return

            if function == BACnetBvlcFunctions.BVLC_RESULT:
                print('Receive Register as Foreign Device Response')

            remote_address = self.ConvertBACnetAddress(address)

            if function == BACnetBvlcFunctions.BVLC_FORWARDED_NPDU:
                ip = int(local_buffer[7] << 24) + int(local_buffer[6] << 16) + int(local_buffer[5] << 8) + int(local_buffer[4])
                port = int(local_buffer[8] << 8) + local_buffer[9]
                ep = (ip, port)
                remote_address = self.ConvertBACnetAddress(ep)

            if HEADER_LENGTH != -1:
                if function == BACnetBvlcFunctions.BVLC_RESULT:
                    print("Receive Register as Foreign Device Response")
                if function == BACnetBvlcFunctions.BVLC_FORWARDED_NPDU:
                    print("BVLC_FORWARDED_NPDU do something!!")
                if function == BACnetBvlcFunctions.BVLC_ORIGINAL_UNICAST_NPDU or function == BACnetBvlcFunctions.BVLC_ORIGINAL_BROADCAST_NPDU or function == BACnetBvlcFunctions.BVLC_FORWARDED_NPDU:
                    if 'MessageRecieved' in self.events.keys() and rx > HEADER_LENGTH:
                        self.events['MessageRecieved'](self, local_buffer, HEADER_LENGTH, rx - HEADER_LENGTH, remote_address)

class BACnetSerialTransport():

    def __init__(self):
        pass

    def Read(self, buffer: bytearray, offset: int, length: int, timeout_ms: int) -> int:
        pass

    def Write(self, buffer: bytearray, offset: int, length: int):
        pass

    def Open(self):
        pass

    def Dispose(self):
        pass

    def BytesToRead(self):
        return 0

class BACnetSerialPortTransport(BACnetSerialTransport):

    def __init__(self, port_name: str, baud_rate: int):
        super().__init__()
        self.m_port_name = port_name
        self.m_baud_rate = baud_rate
        self.m_port = serial.Serial(self.m_port_name, self.m_baud_rate)
        self.out_put = False

    def __str__(self):
        return self.m_port_name

    def __hash__(self):
        return hash(self.m_port_name)

    def __eq__(self, other):
        return self.Equals(other)

    def Equals(self, obj):
        if obj is None:
            return False
        elif not isinstance(obj, BACnetSerialPortTransport):
            return False
        return self.m_port_name == obj.m_port_name

    def Open(self):
        if self.m_port is None:
            return
        if not self.m_port.isOpen():
            self.m_port.open()

    def Close(self):
        if self.m_port is None:
            return
        if self.m_port.isOpen():
            self.m_port.close()

    def Read(self, buffer: bytearray, offset: int, length: int, timeout: float) -> int:
        if self.m_port is None:
            return 0
        self.m_port.timeout = timeout/1000
        try:
            buffer_recv = self.m_port.read(length)
            rx = len(buffer_recv)
            if rx != length:
                return -BACnetMstpProtocolTransport.ETIMEDOUT
            buffer[offset: offset + length] = buffer_recv
        except Exception as e:
            return -1
        return rx

    def Write(self, buffer: bytearray, offset: int, length: int, out_put: bool=False):
        if self.m_port is None:
            return
        self.out_put = out_put
        self.m_port.write(buffer[offset: offset + length])

    def Dispose(self):
        self.Close()

    def BytesToRead(self):
        if self.m_port is None:
            return 0
        return self.m_port.inWaiting()

class MessageFrame:

    def __init__(self, frame_type: BACnetMstpFrameTypes, destination_address: int, data: Any, data_length: int, send_flag: bool=False):
        self.frame_type = frame_type
        self.destination_address = destination_address
        self.data = data
        self.data_length = data_length
        self.send_mutex = threading.Event()
        self.send_flag = send_flag

class BACnetMstpProtocolTransport(BACnetTransport):
    T_FRAME_ABORT: float = 95 # (60 bit times - 100 ms) At 9600 baud, 60 bit times would be about 6.25 ms  Tframe_abort = 1 + ((1000 * 60) / 9600)
    T_NO_TOKEN: float = 500
    T_REPLY_TIMEOUT: float = 295  # 255ms - 300ms
    T_USAGE_TIMEOUT: float = 95  # 20ms - 100ms
    T_REPLY_DELAY: float = 250 # 250ms
    ETIMEDOUT: int = 110
    TR_OFF: int = 30  #29 bit times < Troff < 40 bit times.

    #
    def __init__(self, port_name: str, baud_rate: int, source_address: int=-1, max_master: int=127, max_info_frames: int=1):
        super().__init__()

        self.m_max_info_frames = max_info_frames
        self.m_TS = source_address
        self.m_max_master = max_master
        self.m_local_buffer = bytearray(self.MaxBufferLength())
        self.m_port = BACnetSerialPortTransport(port_name, baud_rate)

        self.m_NS: int = 0
        self.m_PS: int = 0
        self.m_local_offset: int = 0
        self.m_transmit_thread: threading = None
        self.m_frame_count: int = 0
        self.m_token_count: int = 0
        self.m_max_poll: int = 0
        self.m_sole_master: bool = False
        self.m_retry_token: int = 1
        self.m_reply_source: int = 0
        self.m_is_running: bool = False
        self.m_reply_mutex: threading.Event = threading.Event()
        self.m_reply: Optional[MessageFrame] = None
        self.m_send_queue: List[MessageFrame] = []
        self.m_send_queue_lock = threading.Lock()

    def __str__(self):
        return self.m_port.__str__()

    def __hash__(self):
        return hash(self.m_port)

    def __eq__(self, other):
        return self.Equals(other)

    def Equals(self, obj):
        if obj is None:
            return False
        elif not isinstance(obj, BACnetMstpProtocolTransport):
            return False
        return self.m_port.Equals(obj.m_port)

    def GetType(self) -> BACnetAddressTypes:
        return BACnetAddressTypes.MSTP

    def Dispose(self):
        if self.m_port is not None:
            try:
                self.m_port.Dispose()
            except:
                pass
            self.m_port = None

    def MaxBufferLength(self):
        return 502

    def HeaderLength(self):
        return MSTP.MSTP_HEADER_LENGTH

    def IsRunning(self):
        return self.m_is_running

    def MaxAdpuLength(self):
        return MSTP.MSTP_MAX_APDU

    def Start_SpyMode(self):
        if self.m_port is None:
            return
        self.m_port.Open()

        threading.Thread(target=self.mstp_thread_sniffer, daemon=False).start()


    def mstp_thread_sniffer(self):
        while True:
            try:
                (status, frame_type, destination_address, source_address, msg_length) = self.GetNextMessage(self.T_NO_TOKEN)
                if status == GetMessageStatus.ConnectionClose:
                    self.m_port = None
                    return
                elif status == GetMessageStatus.Good:
                    if 'RawMessageRecieved' in self.events.keys():
                        length = msg_length + MSTP.MSTP_HEADER_LENGTH + (2 if msg_length > 0 else 0)

                        packet = bytearray(length)
                        packet[0:length] = self.m_local_buffer[0:length]
                        self.events['RawMessageRecieved'](packet, 0, length)

                    self.RemoveCurrentMessage(msg_length)
            except Exception as e:
                self.m_port = None

    def Start(self):
        if self.m_port is None:
            return
        self.m_port.Open()

        self.m_transmit_thread = threading.Thread(name='MSTP Thread', target=self.mstp_thread, daemon=False)
        self.m_transmit_thread.start()

    def QueueFrame(self, frame_type: BACnetMstpFrameTypes, destination_address: int):
        with self.m_send_queue_lock:
            self.m_send_queue.append(MessageFrame(frame_type, destination_address, None, 0))

    def SendFrame(self, frame_type: BACnetMstpFrameTypes, destination_address: int):
        self.SendFrame1(MessageFrame(frame_type, destination_address, None, 0))

    def SendFrame1(self, frame: MessageFrame):
        if self.m_TS == -1 or self.m_port is None:
            return

        if frame.data is None or len(frame.data) == 0:
            tmp_transmit_buffer = bytearray(MSTP.MSTP_HEADER_LENGTH)
            tx = MSTP.Encode(tmp_transmit_buffer, 0, frame.frame_type, frame.destination_address, self.m_TS, 0)
            self.m_port.Write(tmp_transmit_buffer, 0, tx)
        else:
            tx = MSTP.Encode(frame.data, 0, frame.frame_type, frame.destination_address, self.m_TS, frame.data_length)
            self.m_port.Write(frame.data, 0, tx, frame.send_flag)
        frame.send_mutex.set()

    def RemoveCurrentMessage(self, msg_length: int):
        full_msg_length = MSTP.MSTP_HEADER_LENGTH + msg_length + (2 if msg_length > 0 else 0)
        if self.m_local_offset > full_msg_length:
            self.m_local_buffer[0: 0 + self.m_local_offset - full_msg_length] = self.m_local_buffer[full_msg_length: full_msg_length + self.m_local_offset - full_msg_length]
        self.m_local_offset -= full_msg_length

    def PollForMaster(self):
        while True:
            self.SendFrame(BACnetMstpFrameTypes.FRAME_TYPE_POLL_FOR_MASTER, self.m_PS)

            (status, frame_type, destination_address, source_address, msg_length) = self.GetNextMessage(self.T_USAGE_TIMEOUT)
            if status == GetMessageStatus.Good:
                try:
                    if frame_type ==  BACnetMstpFrameTypes.FRAME_TYPE_REPLY_TO_POLL_FOR_MASTER and destination_address == self.m_TS:
                        self.m_sole_master = False
                        self.m_NS = source_address
                        self.m_PS = self.m_TS
                        self.m_token_count = 0
                        return StateChanges.ReceivedReplyToPFM
                    else:
                        return StateChanges.ReceivedUnexpectedFrame
                finally:
                    self.RemoveCurrentMessage(msg_length)
            else:
                if self.m_sole_master:
                    self.m_frame_count = 0
                    return StateChanges.SoleMaster
                else:
                    if self.m_NS != self.m_TS:
                        return StateChanges.DoneWithPFM
                    else:
                        if (self.m_PS +1) % (self.m_max_master + 1) != self.m_TS:
                            self.m_PS = BACnetTool.int_to_byte((self.m_PS + 1) % (self.m_max_master + 1))
                            continue
                        else:
                            self.m_sole_master = True
                            self.m_frame_count = 0
                            return StateChanges.DeclareSoleMaster

    def DoneWithToken(self):
        if self.m_frame_count < self.m_max_info_frames:
            return StateChanges.SendAnotherFrame
        elif not self.m_sole_master and self.m_NS == self.m_TS:
            self.m_PS = BACnetTool.int_to_byte((self.m_TS + 1) % (self.m_max_master + 1))
            return StateChanges.NextStationUnknown
        elif self.m_token_count < self.m_max_poll - 1:
            self.m_token_count += 1
            if self.m_sole_master and self.m_NS != (self.m_TS + 1) % (self.m_max_master + 1):
                self.m_frame_count = 0
                return StateChanges.SoleMaster
            else:
                return StateChanges.SendToken
        elif (self.m_PS + 1) % (self.m_max_master + 1) == self.m_NS:
            if not self.m_sole_master:
                self.m_PS = self.m_TS
                self.m_token_count = 1
                return StateChanges.ResetMaintenancePFM
            else:
                self.m_PS = BACnetTool.int_to_byte((self.m_NS + 1) % (self.m_max_master + 1))
                self.m_NS = self.m_TS
                self.m_token_count = 1
                return StateChanges.SoleMasterRestartMaintenancePFM
        else:
            self.m_PS = BACnetTool.int_to_byte((self.m_PS + 1) % (self.m_max_master + 1))
            return StateChanges.SendMaintenancePFM

    def WaitForReply(self):
        (status, frame_type, destination_address, source_address, msg_length) = self.GetNextMessage(self.T_REPLY_TIMEOUT)
        if status == GetMessageStatus.Good:
            try:
                if destination_address == self.m_TS and (frame_type == BACnetMstpFrameTypes.FRAME_TYPE_TEST_RESPONSE or frame_type == BACnetMstpFrameTypes.FRAME_TYPE_BACNET_DATA_NOT_EXPECTING_REPLY):
                   if 'MessageRecieved' in self.events.keys() and frame_type != BACnetMstpFrameTypes.FRAME_TYPE_TEST_RESPONSE:
                        remote_address = BACnetAddress(type=BACnetAddressTypes.MSTP, net=0, adr=source_address.to_bytes(1, byteorder='big', signed=False))
                        try:
                            self.events['MessageRecieved'](self, self.m_local_buffer.copy(), MSTP.MSTP_HEADER_LENGTH, msg_length, remote_address)
                        except Exception as e:
                            print(f"Exception in MessageRecieved event: {e.__str__()}")
                   return StateChanges.ReceivedReply
                elif frame_type == BACnetMstpFrameTypes.FRAME_TYPE_REPLY_POSTPONED:
                    return StateChanges.ReceivedPostpone
                else:
                    return StateChanges.ReceivedUnexpectedFrame
            finally:
                self.RemoveCurrentMessage(msg_length)
        elif status == GetMessageStatus.Timeout:
            self.m_frame_count = self.m_max_info_frames
            return StateChanges.ReplyTimeOut
        else:
            return StateChanges.InvalidFrame

    def UseToken(self):
        if len(self.m_send_queue) == 0:
            self.m_frame_count = self.m_max_info_frames
            return StateChanges.NothingToSend
        else:
            with self.m_send_queue_lock:
                message_frame: MessageFrame = self.m_send_queue.pop(0)
            self.SendFrame1(message_frame)
            self.m_frame_count += 1
            if message_frame.frame_type == BACnetMstpFrameTypes.FRAME_TYPE_BACNET_DATA_EXPECTING_REPLY or message_frame.frame_type == BACnetMstpFrameTypes.FRAME_TYPE_TEST_REQUEST:
                return StateChanges.SendAndWait
            else:
                return StateChanges.SendNoWait

    def PassToken(self):
        for i in range(self.m_retry_token+1):
            self.SendFrame(BACnetMstpFrameTypes.FRAME_TYPE_TOKEN, self.m_NS)
            (status, frame_type, destination_address, source_address, msg_length) = self.GetNextMessage(self.T_USAGE_TIMEOUT)
            if status == GetMessageStatus.Good or status == GetMessageStatus.DecodeError:
                return StateChanges.SawTokenUser

        self.m_PS = BACnetTool.int_to_byte((self.m_NS + 1) % (self.m_max_master + 1))
        self.m_NS = self.m_TS
        self.m_token_count = 0
        return StateChanges.FindNewSuccessor

    def Idle(self):
        no_token_timeout = self.T_NO_TOKEN + 10 * self.m_TS
        while self.m_port is not None:
            (status, frame_type, destination_address, source_address, msg_length) = self.GetNextMessage(no_token_timeout)
            if status == GetMessageStatus.Good:
                try:
                    if destination_address == self.m_TS or destination_address == 0xFF:
                        if frame_type == BACnetMstpFrameTypes.FRAME_TYPE_POLL_FOR_MASTER:
                            if destination_address == 0xFF:
                                self.QueueFrame(BACnetMstpFrameTypes.FRAME_TYPE_REPLY_TO_POLL_FOR_MASTER, source_address)
                            else:
                                self.SendFrame(BACnetMstpFrameTypes.FRAME_TYPE_REPLY_TO_POLL_FOR_MASTER, source_address)
                        elif frame_type == BACnetMstpFrameTypes.FRAME_TYPE_TOKEN:
                            if destination_address != 0xFF:
                                self.m_frame_count = 0
                                self.m_sole_master = False
                                return StateChanges.ReceivedToken
                        elif frame_type == BACnetMstpFrameTypes.FRAME_TYPE_TEST_REQUEST:
                            if destination_address == 0xFF:
                                self.QueueFrame(BACnetMstpFrameTypes.FRAME_TYPE_TEST_RESPONSE, source_address)
                            else:
                                self.SendFrame(BACnetMstpFrameTypes.FRAME_TYPE_TEST_RESPONSE, source_address)
                        elif frame_type == BACnetMstpFrameTypes.FRAME_TYPE_BACNET_DATA_NOT_EXPECTING_REPLY or frame_type == BACnetMstpFrameTypes.FRAME_TYPE_BACNET_DATA_EXPECTING_REPLY:
                            if 'MessageRecieved' in self.events.keys():
                                remote_address = BACnetAddress(type=BACnetAddressTypes.MSTP, net=0, adr=source_address.to_bytes(1, byteorder='big', signed=False))
                                try:
                                    self.events['MessageRecieved'](self, self.m_local_buffer.copy(), MSTP.MSTP_HEADER_LENGTH, msg_length, remote_address)
                                except Exception as e:
                                    print(f"Exception in MessageRecieved event: {e.__str__()}")

                            if frame_type == BACnetMstpFrameTypes.FRAME_TYPE_BACNET_DATA_EXPECTING_REPLY:
                                self.m_reply_source = source_address
                                self.m_reply = None
                                self.m_reply_mutex.clear()
                                return StateChanges.ReceivedDataNeedingReply
                finally:
                    self.RemoveCurrentMessage(msg_length)
            elif status == GetMessageStatus.Timeout:
                self.m_PS = BACnetTool.int_to_byte((self.m_TS + 1) % (self.m_max_master + 1))
                self.m_NS = self.m_TS
                self.m_token_count = 0
                return StateChanges.GenerateToken
            elif status == GetMessageStatus.ConnectionClose:
                print(f"No connection")
            elif status == GetMessageStatus.ConnectionError:
                print(f"Connection Error")
            else:
                print(f"Garbage")
        return StateChanges.Reset

    def AnswerDataRequest(self):
        if self.m_reply_mutex.wait(self.T_REPLY_DELAY/1000):
            self.SendFrame1(self.m_reply)
            with self.m_send_queue_lock:
                self.m_send_queue.remove(self.m_reply)
            return StateChanges.Reply
        else:
            self.SendFrame(BACnetMstpFrameTypes.FRAME_TYPE_REPLY_POSTPONED, self.m_reply_source)
            return StateChanges.DeferredReply

    def Initialize(self):
        self.m_token_count = self.m_max_poll
        self.m_frame_count = 0
        self.m_sole_master = False
        self.m_NS = self.m_TS
        self.m_PS = self.m_TS
        return StateChanges.DoneInitializing

    def mstp_thread(self):
        try:
            state_change = StateChanges.Reset
            while self.m_port is not None:
                if state_change == StateChanges.Reset:
                    state_change = self.Initialize()
                elif state_change == StateChanges.DoneInitializing or state_change == StateChanges.ReceivedUnexpectedFrame or state_change == StateChanges.Reply or state_change == StateChanges.DeferredReply or state_change == StateChanges.SawTokenUser:
                    state_change = self.Idle()
                elif state_change == StateChanges.GenerateToken or state_change == StateChanges.FindNewSuccessor or state_change == StateChanges.SendMaintenancePFM or state_change == StateChanges.SoleMasterRestartMaintenancePFM or state_change == StateChanges.NextStationUnknown:
                    state_change = self.PollForMaster()
                elif state_change == StateChanges.DoneWithPFM or state_change == StateChanges.ResetMaintenancePFM or state_change == StateChanges.ReceivedReplyToPFM or state_change == StateChanges.SendToken:
                    state_change = self.PassToken()
                elif state_change == StateChanges.ReceivedDataNeedingReply:
                    state_change = self.AnswerDataRequest()
                elif state_change == StateChanges.ReceivedToken or state_change == StateChanges.SoleMaster or state_change == StateChanges.DeclareSoleMaster or state_change == StateChanges.SendAnotherFrame:
                    state_change = self.UseToken()
                elif state_change == StateChanges.NothingToSend or state_change == StateChanges.SendNoWait or state_change == StateChanges.ReplyTimeOut or state_change == StateChanges.InvalidFrame or state_change == StateChanges.ReceivedReply or state_change == StateChanges.ReceivedPostpone:
                    state_change = self.DoneWithToken()
                elif state_change == StateChanges.SendAndWait:
                    state_change = self.WaitForReply()
            print("MSTP thread is closing down")
        except Exception as e:
            print(f"Exception in MSTP thread: {e.__str__()}")
        self.m_is_running = False

    def RemoveGarbage(self):
        for i in range(self.m_local_offset - 1):
            if self.m_local_buffer[i] == MSTP.MSTP_PREAMBLE1 and self.m_local_buffer[i + 1] == MSTP.MSTP_PREAMBLE2:
                if i > 0:
                    self.m_local_buffer[0: 0 + self.m_local_offset - i] = self.m_local_buffer[i: i + self.m_local_offset - i]
                    self.m_local_offset -= i
                return

        if self.m_local_offset > 0 and self.m_local_buffer[self.m_local_offset - 1] == MSTP.MSTP_PREAMBLE1:
            if self.m_local_offset != 1:
                self.m_local_buffer[0] = MSTP.MSTP_PREAMBLE1
                self.m_local_offset = 1
            return

        if self.m_local_offset > 0:
            self.m_local_offset = 0

    def GetNextMessage(self, timeout_ms: float):

        frame_type = BACnetMstpFrameTypes.FRAME_TYPE_TOKEN
        destination_address: int = 0
        source_address: int = 0
        msg_length: int = 0

        while self.m_local_offset < MSTP.MSTP_HEADER_LENGTH:
            if self.m_port is None:
                return (GetMessageStatus.ConnectionClose, frame_type, destination_address, source_address, msg_length)

            if self.m_local_offset > 0:
                timeout = self.T_FRAME_ABORT
            else:
                timeout = timeout_ms
            rx = self.m_port.Read(self.m_local_buffer, self.m_local_offset, MSTP.MSTP_HEADER_LENGTH - self.m_local_offset, timeout)
            if rx == -self.ETIMEDOUT:
                status = GetMessageStatus.Timeout if self.m_local_offset == 0 else GetMessageStatus.SubTimeout
                self.m_local_buffer[0] = 0xFF
                self.RemoveGarbage()
                return (status, frame_type, destination_address, source_address, msg_length)
            elif rx < 0:
                self.m_local_buffer[0] = 0xFF
                self.RemoveGarbage()
                return (GetMessageStatus.ConnectionError, frame_type, destination_address, source_address, msg_length)
            elif rx == 0:
                self.m_local_buffer[0] = 0xFF
                self.RemoveGarbage()
                return (GetMessageStatus.ConnectionClose, frame_type, destination_address, source_address, msg_length)
            self.m_local_offset += rx
            self.RemoveGarbage()

        (l, frame_type, destination_address, source_address, msg_length) = MSTP.Decode(self.m_local_buffer, 0, self.m_local_offset)

        if l < 0:
            self.m_local_buffer[0] = 0xFF
            self.RemoveGarbage()
            return (GetMessageStatus.DecodeError, frame_type, destination_address, source_address, msg_length)

        full_msg_length = msg_length + MSTP.MSTP_HEADER_LENGTH + (2 if msg_length > 0 else 0)
        if msg_length > self.MaxBufferLength():
            self.m_local_buffer[0] = 0xFF
            self.RemoveGarbage()
            return (GetMessageStatus.DecodeError, frame_type, destination_address, source_address, msg_length)

        if msg_length > 0:
            timeout = self.T_FRAME_ABORT
            while self.m_local_offset < full_msg_length:
                rx = self.m_port.Read(self.m_local_buffer, self.m_local_offset, full_msg_length - self.m_local_offset, timeout)

                if rx == -self.ETIMEDOUT:
                    status = GetMessageStatus.Timeout if self.m_local_offset == 0 else GetMessageStatus.SubTimeout
                    self.m_local_buffer[0] = 0xFF
                    self.RemoveGarbage()
                    return (status, frame_type, destination_address, source_address, msg_length)
                elif rx < 0:
                    self.m_local_buffer[0] = 0xFF
                    self.RemoveGarbage()
                    return (GetMessageStatus.ConnectionError, frame_type, destination_address, source_address, msg_length)
                elif rx == 0:
                    self.m_local_buffer[0] = 0xFF
                    self.RemoveGarbage()
                    return (GetMessageStatus.ConnectionClose, frame_type, destination_address, source_address, msg_length)
                self.m_local_offset += rx

            (l, frame_type, destination_address, source_address, msg_length) = MSTP.Decode(self.m_local_buffer, 0, self.m_local_offset)
            if l < 0:
                self.m_local_buffer[0] = 0xFF
                self.RemoveGarbage()
                return (GetMessageStatus.DecodeError, frame_type, destination_address, source_address, msg_length)

        if 'FrameRecieved' in self.events.keys():
            _frame_type = frame_type
            _destination_address = destination_address
            _source_address = source_address
            _msg_length = msg_length
            threading.Thread(target=self.events['FrameRecieved'], args=(self, _frame_type, _destination_address, _source_address, _msg_length), daemon=False).start()

        return (GetMessageStatus.Good, frame_type, destination_address, source_address, msg_length)

    def Send1(self, buffer: bytearray, offset: int, data_length: int, address: BACnetAddress, wait_for_transmission: bool, timeout: float):
        if self.m_TS == -1:
            raise Exception(f"Source address must be set up before sending messages")

        function = NPDU.DecodeFunction(buffer, offset)
        frame_type = BACnetMstpFrameTypes.FRAME_TYPE_BACNET_DATA_EXPECTING_REPLY if (function & BACnetNpduControls.ExpectingReply) == BACnetNpduControls.ExpectingReply else BACnetMstpFrameTypes.FRAME_TYPE_BACNET_DATA_NOT_EXPECTING_REPLY
        copy = bytearray(data_length + MSTP.MSTP_HEADER_LENGTH + 2)
        copy[MSTP.MSTP_HEADER_LENGTH: MSTP.MSTP_HEADER_LENGTH + data_length] = buffer[offset: offset+data_length]
        f = MessageFrame(frame_type, address.adr[0], copy, data_length, True)
        with self.m_send_queue_lock:
            self.m_send_queue.append(f)
        if self.m_reply is None:
            self.m_reply = f
            self.m_reply_mutex.set()

        if wait_for_transmission:
            if not f.send_mutex.wait(timeout/1000):
                return -self.ETIMEDOUT

        return data_length

    def WaitForAllTransmits(self, timeout: float):
        while len(self.m_send_queue) > 0:
            with self.m_send_queue_lock:
                ev = self.m_send_queue[0].send_mutex
            if ev.wait(timeout/1000):
                return False
        return True

    def GetBroadcastAddress(self) -> BACnetAddress:
        return BACnetAddress(type=BACnetAddressTypes.MSTP, net=0xFFFF, adr=bytearray(int(0xFF).to_bytes(1, byteorder='little', signed=False)))

##BACnetClient################################
class LastSegmentACK:

    def __init__(self):
        self.invoke_id: int = 0
        self.sequence_number: int = 0
        self.window_size: int = 0
        self.adr: BACnetAddress = BACnetAddress()
        self.m_wait: threading.Event = threading.Event()
        self.m_lockObject = threading.Lock()

    def Set(self, adr: BACnetAddress, invoke_id: int, sequence_number: int, window_size: int):
        self.adr = adr
        self.invoke_id = invoke_id
        self.sequence_number = sequence_number
        self.window_size = window_size
        self.m_wait.set()

    def Wait(self, adr: BACnetAddress, invoke_id: int, timeout: int):
        self.m_lockObject.locked()
        while not adr.Equals(self.adr) or self.invoke_id != invoke_id:
            self.m_wait.clear()
            self.m_lockObject.release()
            if not self.m_wait.wait(timeout):
                return False
            self.m_lockObject.locked()
        self.m_lockObject.release()
        self.adr = None
        return True

class Segmentation:

    def __init__(self):
        self.buffer: Optional[EncodeBuffer] = None
        self.sequence_number: int = 0
        self.window_size: int = 0
        self.max_segments: int = 0

class BACnetAsyncResult:

    def __init__(self, comm, adr: BACnetAddress, invoke_id: int, transmit_buffer: bytearray, transmit_length: int, wait_for_transmit: bool, transmit_timeout: int):
        self.m_transmit_timeout = transmit_timeout
        self.m_adr = adr
        self.m_wait_for_transmit = wait_for_transmit
        self.m_transmit_buffer = transmit_buffer
        self.m_transmit_length = transmit_length
        self.AsyncWaitHandle = threading.Event()
        self.m_comm = comm
        self.m_wait_invoke_id = invoke_id
        self.m_error: Optional[Exception] = None
        self.m_result: Optional[bytearray] = None
        self.Segmented: bool = False

        self.m_comm.events['OnComplexAck'] = self.m_comm_OnComplexAck
        self.m_comm.events['OnError'] = self.m_comm_OnError
        self.m_comm.events['OnAbort'] = self.m_comm_OnAbort
        self.m_comm.events['OnSimpleAck'] = self.m_comm_OnSimpleAck
        self.m_comm.events['OnSegment'] = self.m_comm_OnSegment

    def Resend(self):
        try:
            if self.m_comm.m_client.Send1(self.m_transmit_buffer, self.m_comm.m_client.HeaderLength(), self.m_transmit_length, self.m_adr, self.m_wait_for_transmit, self.m_transmit_timeout) < 0:
                self.m_error = Exception("Write Timeout")
                self.CompletedSynchronously = True
                self.AsyncWaitHandle.set()
        except Exception as e:
            self.m_error = Exception(f"Write Exception: {e.__str__()}")
            self.CompletedSynchronously = True
            self.AsyncWaitHandle.set()

    def m_comm_OnSegment(self, sender, adr: BACnetAddress, type: BACnetPduTypes, service: BACnetConfirmedServices, invoke_id: int, max_segments: BACnetMaxSegments, max_adpu: BACnetMaxAdpu, sequence_number: int, first: bool, more_follows: bool, buffer: bytearray, offset: int, length: int):
        if invoke_id == self.m_wait_invoke_id:
            self.Segmented = True
            self.AsyncWaitHandle.set()

    def m_comm_OnSimpleAck(self, sender, adr: BACnetAddress, type: BACnetPduTypes, service: BACnetConfirmedServices, invoke_id: int, buffer: bytearray, offset: int, length: int):
        if invoke_id == self.m_wait_invoke_id:
            self.AsyncWaitHandle.set()

    def m_comm_OnAbort(self, sender, adr: BACnetAddress, type: BACnetPduTypes, invoke_id: int, reason: int, buffer: bytearray, offset: int, length: int):
        if invoke_id == self.m_wait_invoke_id:
            self.m_error = Exception(f"Abort from device: {reason}")
            self.CompletedSynchronously = True
            self.AsyncWaitHandle.set()

    def m_comm_OnError(self, sender, adr: BACnetAddress, type: BACnetPduTypes, service: BACnetConfirmedServices, invoke_id: int, error_class: BACnetErrorClasses, error_code: BACnetErrorCodes, buffer: bytearray, offset: int, length: int):
        if invoke_id == self.m_wait_invoke_id:
            self.m_error = Exception(f"Error from device: {error_class} - {error_code}")
            self.CompletedSynchronously = True
            self.AsyncWaitHandle.set()

    def m_comm_OnComplexAck(self, sender, adr: BACnetAddress, type: BACnetPduTypes, service: BACnetConfirmedServices, invoke_id: int, buffer: bytearray, offset: int, length: int):
        if invoke_id == self.m_wait_invoke_id:
            self.Segmented = False
            self.m_result = bytearray(length)
            if length > 0:
                self.m_result[0: length] = buffer[offset: offset + length]
            self.AsyncWaitHandle.set()

    def wait(self, timeout: int):
        while True:
            if not self.AsyncWaitHandle.wait(timeout):
                return False
            if self.Segmented:
                self.AsyncWaitHandle.clear()
            else:
                return True

    def Dispose(self):
        if self.m_comm is None:
            return

        if 'OnComplexAck' in self.m_comm.events:
            del self.m_comm.events['OnComplexAck']

        if 'OnError' in self.m_comm.events:
            del self.m_comm.events['OnError']

        if 'OnAbort' in self.m_comm.events:
            del self.m_comm.events['OnAbort']

        if 'OnSimpleAck' in self.m_comm.events:
            del self.m_comm.events['OnSimpleAck']

        if 'OnSegment' in self.m_comm.events:
            del self.m_comm.events['OnSegment']

        self.m_comm = None

class BACnetClient:

    def __init__(self, transport: BACnetTransport, timeout: int=1, retries: int=3):
        self.m_client = transport
        self.m_timeout = timeout
        self.m_retries = max(1, retries)
        self.m_transmit_timeout: int = 30000  # ms
        self.m_invoke_id: int = 0
        self.m_max_segments: BACnetMaxSegments = BACnetMaxSegments.MAX_SEG0
        self.m_last_sequence_number: int = 0
        self.m_proposed_window_size: int = 0
        self.m_default_segmentation_handling: bool = False
        self.m_segments: list = []
        self.m_last_segment_ack = LastSegmentACK()
        self.m_force_window_size: bool = False
        self.m_writepriority: int = 0
        self.raw_buffer: Optional[bytearray] = None
        self.raw_offset: int = 0
        self.raw_length: int = 0
        self.events: dict = {}
        self.DefaultSegmentationHandling = True

    def __str__(self):
        return self.m_client.__str__()

    def __hash__(self):
        return self.m_client.__hash__()

    def __eq__(self, other):
        return self.Equals(other)

    def __del__(self):
        self.Dispose()

    def Equals(self, obj):
        if not isinstance(obj, BACnetClient):
            return False
        return self.m_client.Equals(obj.m_client)

    def GetEncodeBuffer(self, start_offset: int):
        return EncodeBuffer(bytearray(self.m_client.MaxBufferLength()), start_offset)

    def unchecked(self, value: int):
        value = value + 1
        if value > 255:
            return 0
        return value

    def Start(self):
        self.m_client.Start()
        self.m_client.events['MessageRecieved'] = self.OnRecieve

    def SetHandleLog(self, HandleLog: Callable):
        self.events['HandleLog'] = HandleLog

    def ProcessConfirmedServiceRequest(self, adr: BACnetAddress, type: BACnetPduTypes, service:BACnetConfirmedServices, max_segments: BACnetMaxSegments, max_adpu: BACnetMaxAdpu, invoke_id: int, buffer: bytearray, offset: int, length: int):
        try:
            self.raw_buffer = buffer
            self.raw_length = length
            self.raw_offset = offset

            if 'OnConfirmedServiceRequest' in self.events.keys():
                self.events['OnConfirmedServiceRequest'](self, adr, type, service, max_segments, max_adpu, invoke_id, buffer, offset, length)

            if (type & BACnetPduTypes.SEGMENTED_RESPONSE_ACCEPTED) == 0:
                max_segments = BACnetMaxSegments.MAX_SEG0

            if service == BACnetConfirmedServices.SERVICE_CONFIRMED_READ_PROPERTY and 'OnReadPropertyRequest' in self.events.keys():
                (Ths_Reject_Reason, object_id, property) = Services.DecodeReadProperty(buffer, offset, length)
                if Ths_Reject_Reason >= 0:
                    self.events['OnReadPropertyRequest'](self, adr, invoke_id, object_id, property, max_segments)
                else:
                    if Ths_Reject_Reason == -1:
                        self.SendConfirmedServiceReject(adr, invoke_id, BACnetRejectReasons.REJECT_REASON_MISSING_REQUIRED_PARAMETER)
                    elif Ths_Reject_Reason == -2:
                        self.SendConfirmedServiceReject(adr, invoke_id, BACnetRejectReasons.REJECT_REASON_INVALID_TAG)
                    elif Ths_Reject_Reason == -3:
                        self.SendConfirmedServiceReject(adr, invoke_id, BACnetRejectReasons.REJECT_REASON_TOO_MANY_ARGUMENTS)
            elif service == BACnetConfirmedServices.SERVICE_CONFIRMED_WRITE_PROPERTY and 'OnWritePropertyRequest' in self.events.keys():
                (l, object_id, value) = Services.DecodeWriteProperty(buffer, offset, length)
                if l >= 0:
                    self.events['OnWritePropertyRequest'](self, adr, invoke_id, object_id, value, max_segments)
                else:
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            elif service == BACnetConfirmedServices.SERVICE_CONFIRMED_READ_PROP_MULTIPLE and 'OnReadPropertyMultipleRequest' in self.events.keys():
                (l, properties) = Services.DecodeReadPropertyMultiple(buffer, offset, length)
                if l >= 0:
                    self.events['OnReadPropertyMultipleRequest'](self, adr, invoke_id, properties, max_segments)
                else:
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            elif service == BACnetConfirmedServices.SERVICE_CONFIRMED_WRITE_PROP_MULTIPLE and 'OnWritePropertyMultipleRequest' in self.events.keys():
                (l, object_id, values) = Services.DecodeWritePropertyMultiple(buffer, offset, length)
                if l >= 0:
                    self.events['OnWritePropertyMultipleRequest'](self, adr, invoke_id, object_id, values, max_segments)
                else:
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            elif service == BACnetConfirmedServices.SERVICE_CONFIRMED_COV_NOTIFICATION and 'OnCOVNotification' in self.events.keys():
                (l, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values) = Services.DecodeCOVNotifyUnconfirmed(buffer, offset, length)
                if l >= 0:
                    self.events['OnCOVNotification'](self, adr, invoke_id, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, True, values, max_segments)
                else:
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            elif service == BACnetConfirmedServices.SERVICE_CONFIRMED_ATOMIC_WRITE_FILE and 'OnAtomicWriteFileRequest' in self.events.keys():
                (l, is_stream, object_id, position, block_count, blocks, counts) = Services.DecodeAtomicWriteFile(buffer, offset, length)
                if l >= 0:
                    self.events['OnAtomicWriteFileRequest'](self, adr, invoke_id, is_stream, object_id, position, block_count, blocks, counts, max_segments)
                else:
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            elif service == BACnetConfirmedServices.SERVICE_CONFIRMED_ATOMIC_READ_FILE and 'OnAtomicReadFileRequest' in self.events.keys():
                (l, is_stream, object_id, position, count) = Services.DecodeAtomicReadFile(buffer, offset, length)
                if l >= 0:
                    self.events['OnAtomicReadFileRequest'](self, adr, invoke_id, is_stream, object_id, position, count, max_segments)
                else:
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES,BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            elif service == BACnetConfirmedServices.SERVICE_CONFIRMED_SUBSCRIBE_COV and 'OnSubscribeCOV' in self.events.keys():
                (l, subscriberProcessIdentifier, monitoredObjectIdentifier, cancellationRequest, issueConfirmedNotifications, lifetime) = Services.DecodeSubscribeCOV(buffer, offset, length)
                if l >= 0:
                    self.events['OnSubscribeCOV'](self, adr, invoke_id, subscriberProcessIdentifier, monitoredObjectIdentifier, cancellationRequest, issueConfirmedNotifications, lifetime, max_segments)
                else:
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            elif service == BACnetConfirmedServices.SERVICE_CONFIRMED_SUBSCRIBE_COV_PROPERTY and 'OnSubscribeCOVProperty' in self.events.keys():
                (l, subscriberProcessIdentifier, monitoredObjectIdentifier, monitoredProperty, cancellationRequest, issueConfirmedNotifications, lifetime, covIncrement) = Services.DecodeSubscribeProperty(buffer, offset, length)
                if l >= 0:
                    self.events['OnSubscribeCOVProperty'](self, adr, invoke_id, subscriberProcessIdentifier, monitoredObjectIdentifier, monitoredProperty, cancellationRequest, issueConfirmedNotifications, lifetime, covIncrement, max_segments)
                else:
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            elif service == BACnetConfirmedServices.SERVICE_CONFIRMED_DEVICE_COMMUNICATION_CONTROL and 'OnDeviceCommunicationControl' in self.events.keys():
                (l, timeDuration, enable_disable, password) = Services.DecodeDeviceCommunicationControl(buffer, offset, length)
                if l >= 0:
                    self.events['OnDeviceCommunicationControl'](self, adr, invoke_id, timeDuration, enable_disable, password, max_segments)
                else:
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            elif service == BACnetConfirmedServices.SERVICE_CONFIRMED_REINITIALIZE_DEVICE and 'OnReinitializedDevice' in self.events.keys():
                (l, state, password) = Services.DecodeReinitializeDevice(buffer, offset, length)
                if l >= 0:
                    self.events['OnReinitializedDevice'](self, adr, invoke_id, state, password, max_segments)
                else:
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            elif service == BACnetConfirmedServices.SERVICE_CONFIRMED_EVENT_NOTIFICATION and 'OnEventNotify' in self.events.keys():
                (l, EventData) = Services.DecodeEventNotifyData(buffer, offset, length)
                if l >= 0:
                    self.events['OnEventNotify'](self, adr, invoke_id, EventData, True)
                else:
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            elif service == BACnetConfirmedServices.SERVICE_CONFIRMED_READ_RANGE and 'OnReadRange' in self.events.keys():
                (l, objectId, property, requestType, position, time, count) = Services.DecodeReadRange(buffer, offset, length)
                if l >= 0:
                    self.events['OnReadRange'](self, adr, invoke_id, objectId, property, requestType, position, time, count, max_segments)
                else:
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            elif service == BACnetConfirmedServices.SERVICE_CONFIRMED_CREATE_OBJECT and 'OnCreateObjectRequest' in self.events.keys():
                (l, object_id, values) = Services.DecodeReadRange(buffer, offset, length)
                if l >= 0:
                    self.events['OnCreateObjectRequest'](self, adr, invoke_id, object_id, values, max_segments)
                else:
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            elif service == BACnetConfirmedServices.SERVICE_CONFIRMED_DELETE_OBJECT and 'OnDeleteObjectRequest' in self.events.keys():
                (l, object_id) = Services.DecodeDeleteObject(buffer, offset, length)
                if l >= 0:
                    self.events['OnDeleteObjectRequest'](self, adr, invoke_id, object_id, max_segments)
                else:
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
            else:
                self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)
        except Exception as e:
            self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_OTHER)

    def ProcessUnconfirmedServiceRequest(self, adr: BACnetAddress, type: BACnetPduTypes, service: BACnetUnconfirmedServices, buffer: bytearray, offset: int, length: int):
        try:
            if 'OnUnconfirmedServiceRequest' in self.events.keys():
                self.events['OnUnconfirmedServiceRequest'](self, adr, type, service, buffer, offset, length)

            if service == BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_I_AM and 'OnIam' in self.events.keys():
                (l, device_id, max_adpu, segmentation, vendor_id) = Services.DecodeIamBroadcast(buffer, offset)
                if l >= 0:
                    self.events['OnIam'](self, adr, device_id, max_adpu, segmentation, vendor_id)
                else:
                    if 'HandleLog' in self.events.keys():
                        self.events['HandleLog'](f"ERROR: Couldn't decode IamBroadcast")
            elif service == BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_WHO_IS and 'OnWhoIs' in self.events.keys():
                (l, low_limit, high_limit) = Services.DecodeWhoIsBroadcast(buffer, offset, length)
                if l >= 0:
                    self.events['OnWhoIs'](self, adr, low_limit, high_limit)
                else:
                    if 'HandleLog' in self.events.keys():
                        self.events['HandleLog'](f"ERROR: Couldn't decode WhoIsBroadcast")
            elif service == BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_WHO_HAS and 'OnWhoHas' in self.events.keys():
                (l, low_limit, high_limit, ObjId, ObjName) = Services.DecodeWhoHasBroadcast(buffer, offset, length)
                if l >= 0:
                    self.events['OnWhoHas'](self, adr, low_limit, high_limit, ObjId, ObjName)
                else:
                    if 'HandleLog' in self.events.keys():
                        self.events['HandleLog'](f"ERROR: Couldn't decode WhoHasBroadcast")
            elif service == BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_COV_NOTIFICATION and 'OnCOVNotification' in self.events.keys():
                (l, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values) = Services.DecodeCOVNotifyUnconfirmed(buffer, offset, length)
                if l >= 0:
                    self.events['OnCOVNotification'](self, adr, 0, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, False, values, BACnetMaxSegments.MAX_SEG0)
                else:
                    if 'HandleLog' in self.events.keys():
                        self.events['HandleLog'](f"ERROR: Couldn't decode COVNotifyUnconfirmed")
            elif service == BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_TIME_SYNCHRONIZATION and 'OnTimeSynchronize' in self.events.keys():
                (l, dateTime) = Services.DecodeTimeSync(buffer, offset, length)
                if l >= 0:
                    self.events['OnTimeSynchronize'](self, adr, dateTime, False)
                else:
                    if 'HandleLog' in self.events.keys():
                        self.events['HandleLog'](f"ERROR: Couldn't decode TimeSynchronize")
            elif service == BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_UTC_TIME_SYNCHRONIZATION and 'OnTimeSynchronize' in self.events.keys():
                (l, dateTime) = Services.DecodeTimeSync(buffer, offset, length)
                if l >= 0:
                    self.events['OnTimeSynchronize'](self, adr, dateTime, False)
                else:
                    if 'HandleLog' in self.events.keys():
                        self.events['HandleLog'](f"ERROR: Couldn't decode TimeSynchronize")
            elif service == BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_EVENT_NOTIFICATION and 'OnEventNotify' in self.events.keys():
                (l, EventData) = Services.DecodeEventNotifyData(buffer, offset, length)
                if l >= 0:
                    self.events['OnEventNotify'](self, adr, 0, EventData, False)
                else:
                    if 'HandleLog' in self.events.keys():
                        self.events['HandleLog'](f"ERROR: Couldn't decode Event/Alarm Notification")
            else:
                if 'HandleLog' in self.events.keys():
                    self.events['HandleLog'](f"ERROR: Unconfirmed service not handled: {service.__str__()}")
        except Exception as e:
            if 'HandleLog' in self.events.keys():
                self.events['HandleLog'](f"ERROR: Error in ProcessUnconfirmedServiceRequest: {e.__str__()}")

    def ProcessSimpleAck(self, adr: BACnetAddress, type: BACnetPduTypes, service:BACnetConfirmedServices, invoke_id: int, buffer: bytearray, offset: int, length: int):
        try:
            if 'OnSimpleAck' in self.events.keys():
                self.events['OnSimpleAck'](self, adr, type, service, invoke_id, buffer, offset, length)
        except Exception as e:
            if 'HandleLog' in self.events.keys():
                self.events['HandleLog'](f"ERROR: Error in ProcessSimpleAck: {e.__str__()}")

    def ProcessComplexAck(self, adr: BACnetAddress, type: BACnetPduTypes, service:BACnetConfirmedServices, invoke_id: int, buffer: bytearray, offset: int, length: int):
        try:
            if 'OnComplexAck' in self.events.keys():
                self.events['OnComplexAck'](self, adr, type, service, invoke_id, buffer, offset, length)
        except Exception as e:
            if 'HandleLog' in self.events.keys():
                self.events['HandleLog'](f"ERROR: Error in ProcessComplexAck: {e.__str__()}")

    def ProcessError(self, adr: BACnetAddress, type: BACnetPduTypes, service:BACnetConfirmedServices, invoke_id: int, buffer: bytearray, offset: int, length: int):
        try:
            (l, error_class, error_code) = Services.DecodeError(buffer, offset, length)
            if l < 0:
                if 'HandleLog' in self.events.keys():
                    self.events['HandleLog'](f"ERROR: Couldn't decode Error")

            if 'OnError' in self.events.keys():
                self.events['OnError'](self, adr, type, service, invoke_id, error_class, error_code, buffer, offset, length)
        except Exception as e:
            if 'HandleLog' in self.events.keys():
                self.events['HandleLog'](f"ERROR: Error in ProcessError: {e.__str__()}")

    def ProcessAbort(self, adr: BACnetAddress, type: BACnetPduTypes, invoke_id: int, reason: int, buffer: bytearray, offset: int, length: int):
        try:
            if 'OnAbort' in self.events.keys():
                self.events['OnAbort'](self, adr, type, invoke_id, reason, buffer, offset, length)
        except Exception as e:
            if 'HandleLog' in self.events.keys():
                self.events['HandleLog'](f"ERROR: Error in ProcessAbort: {e.__str__()}")

    def ProcessSegmentAck(self, adr: BACnetAddress, type: BACnetPduTypes, original_invoke_id: int, sequence_number: int, actual_window_size: int, buffer: bytearray, offset: int, length: int):
        try:
            if 'OnSegmentAck' in self.events.keys():
                self.events['OnSegmentAck'](self, adr, type, original_invoke_id, sequence_number, actual_window_size, buffer, offset, length)
        except Exception as e:
            if 'HandleLog' in self.events.keys():
                self.events['HandleLog'](f"ERROR: Error in ProcessSegmentAck: {e.__str__()}")

    def ProcessSegment(self, adr: BACnetAddress, type: BACnetPduTypes, service: BACnetConfirmedServices, invoke_id: int, max_segments: BACnetMaxSegments, max_adpu: BACnetMaxAdpu, server: bool, sequence_number: int, proposed_window_number: int, buffer: bytearray, offset: int, length: int):
        first = False
        if sequence_number == 0 and self.m_last_sequence_number == 0:
            first = True
        else:
            if sequence_number != self.m_last_sequence_number + 1:
                self.SegmentAckResponse(adr, True, server, invoke_id, self.m_last_sequence_number, proposed_window_number)
                return
        self.m_last_sequence_number = sequence_number

        more_follows = (type & BACnetPduTypes.MORE_FOLLOWS) == BACnetPduTypes.MORE_FOLLOWS
        if not more_follows:
            self.m_last_sequence_number = 0

        if sequence_number % proposed_window_number == 0 or not more_follows:
            if self.m_force_window_size:
                proposed_window_number = self.m_proposed_window_size
            self.SegmentAckResponse(adr, False, server, invoke_id, sequence_number, proposed_window_number)

        if 'OnSegment' in self.events.keys():
            self.events['OnSegment'](self, adr, type, service, invoke_id, max_segments, max_adpu, sequence_number, first, more_follows, buffer, offset, length)

        if self.m_default_segmentation_handling:
            self.PerformDefaultSegmentHandling(self, adr, type, service, invoke_id, max_segments, max_adpu, sequence_number, first, more_follows, buffer, offset, length)

    def AssembleSegments(self):
        count = 0
        for arr in self.m_segments:
            count += len(arr)
        ret = bytearray(count)
        count = 0
        for arr in self.m_segments:
            ret[count: count+ len(arr)] = arr[0:len(arr)]
            count += len(arr)
        return ret

    def PerformDefaultSegmentHandling(self, sender, adr: BACnetAddress, type: BACnetPduTypes, service: BACnetConfirmedServices, invoke_id: int, max_segments: BACnetMaxSegments, max_adpu: BACnetMaxAdpu, sequence_number: int, first: bool, more_follows: bool, buffer: bytearray, offset: int, length: int):
        if first:
            self.m_segments.clear()
            type &= ~BACnetPduTypes.SEGMENTED_MESSAGE
            adpu_header_len = 3
            if (type & BACnetPduTypes.PDU_TYPE_MASK) == BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST:
                adpu_header_len = 4
            copy = bytearray(length + adpu_header_len)
            copy[adpu_header_len: adpu_header_len + length] = buffer[offset: offset+length]
            if (type & BACnetPduTypes.PDU_TYPE_MASK) == BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST:
                APDU.EncodeConfirmedServiceRequest(EncodeBuffer(copy, 0), type, service, max_segments, max_adpu, invoke_id, 0, 0)
            else:
                APDU.EncodeComplexAck(EncodeBuffer(copy, 0), type, service, invoke_id, 0, 0)
            self.m_segments.append(copy)
        else:
            copy = bytearray(length)
            copy[0: length] = buffer[offset: offset + length]
            self.m_segments.append(copy)

        if not more_follows:
            apdu_buffer = self.AssembleSegments()
            self.m_segments.clear()

            self.ProcessApdu(adr, type, apdu_buffer, 0, len(apdu_buffer))

    def ProcessApdu(self, adr: BACnetAddress, type: BACnetPduTypes, buffer: bytearray, offset: int, length: int):
        if type & BACnetPduTypes.PDU_TYPE_MASK == BACnetPduTypes.PDU_TYPE_UNCONFIRMED_SERVICE_REQUEST:
            (apdu_header_len, type, service) = APDU.DecodeUnconfirmedServiceRequest(buffer, offset)
            offset += apdu_header_len
            length -= apdu_header_len
            self.ProcessUnconfirmedServiceRequest(adr, type, service, buffer, offset, length)
        elif type & BACnetPduTypes.PDU_TYPE_MASK == BACnetPduTypes.PDU_TYPE_SIMPLE_ACK:
            (apdu_header_len, type, service, invoke_id) = APDU.DecodeSimpleAck(buffer, offset)
            offset += apdu_header_len
            length -= apdu_header_len
            self.ProcessSimpleAck(adr, type, service, invoke_id, buffer, offset, length)
        elif type & BACnetPduTypes.PDU_TYPE_MASK == BACnetPduTypes.PDU_TYPE_COMPLEX_ACK:
            (apdu_header_len, type, service, invoke_id, sequence_number, proposed_window_number) = APDU.DecodeComplexAck(buffer, offset)
            offset += apdu_header_len
            length -= apdu_header_len
            if type & BACnetPduTypes.SEGMENTED_MESSAGE == 0:
                self.ProcessComplexAck(adr, type, service, invoke_id, buffer, offset, length)
            else:
                self.ProcessSegment(adr, type, service, invoke_id, BACnetMaxSegments.MAX_SEG0, BACnetMaxAdpu.MAX_APDU50, False, sequence_number, proposed_window_number, buffer, offset, length)
        elif type & BACnetPduTypes.PDU_TYPE_MASK == BACnetPduTypes.PDU_TYPE_SEGMENT_ACK:
            (apdu_header_len, type, original_invoke_id, sequence_number, actual_window_size) = APDU.DecodeSegmentAck(buffer, offset)
            offset += apdu_header_len
            length -= apdu_header_len
            self.m_last_segment_ack.Set(adr, original_invoke_id, sequence_number, actual_window_size)
            self.ProcessSegmentAck(adr, type, original_invoke_id, sequence_number, actual_window_size, buffer, offset, length)
        elif type & BACnetPduTypes.PDU_TYPE_MASK == BACnetPduTypes.PDU_TYPE_ERROR:
            (apdu_header_len, type, service, invoke_id) = APDU.DecodeSimpleAck(buffer, offset)
            offset += apdu_header_len
            length -= apdu_header_len
            self.ProcessError(adr, type, service, invoke_id, buffer, offset, length)
        elif type & BACnetPduTypes.PDU_TYPE_MASK == BACnetPduTypes.PDU_TYPE_REJECT or type & BACnetPduTypes.PDU_TYPE_MASK == BACnetPduTypes.PDU_TYPE_ABORT:
            (apdu_header_len, type, invoke_id, reason) = APDU.DecodeSimpleAck(buffer, offset)
            offset += apdu_header_len
            length -= apdu_header_len
            self.ProcessAbort(adr, type, invoke_id, reason, buffer, offset, length)
        elif type & BACnetPduTypes.PDU_TYPE_MASK == BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST:
            (apdu_header_len, type, service, max_segments, max_adpu, invoke_id, sequence_number, proposed_window_number) = APDU.DecodeConfirmedServiceRequest(buffer, offset)
            offset += apdu_header_len
            length -= apdu_header_len
            if type & BACnetPduTypes.SEGMENTED_MESSAGE == 0:
                self.ProcessConfirmedServiceRequest(adr, type, service, max_segments, max_adpu, invoke_id, buffer, offset, length)
            else:
                self.ProcessSegment(adr, type, service, invoke_id, max_segments, max_adpu, True, sequence_number, proposed_window_number, buffer, offset, length)
        else:
            if 'HandleLog' in self.events.keys():
                self.events['HandleLog'](f"Warning: Something else arrived: {type}")

    def OnRecieve(self, sender: BACnetTransport, buffer: bytearray, offset: int, msg_length: int, remote_address: BACnetAddress):
        try:
            if self.m_client is None:
                return

            if msg_length > 0:
                (npdu_len, npdu_function, destination, source, hop_count, nmt, vendor_id) = NPDU.Decode(buffer, offset)
                remote_address.RoutedSource = source
                if (npdu_function & BACnetNpduControls.NetworkLayerMessage) == BACnetNpduControls.NetworkLayerMessage:
                    if 'HandleLog' in self.events.keys():
                        self.events['HandleLog'](f"ERROR: Network Layer message received")
                    return

                if npdu_len >0:
                    offset += npdu_len
                    msg_length -= npdu_len

                    if msg_length > 0:
                        apdu_type = APDU.GetDecodedType(buffer, offset)
                        self.ProcessApdu(remote_address, apdu_type, buffer, offset, msg_length)
        except Exception as e:
            if 'HandleLog' in self.events.keys():
                self.events['HandleLog'](f"ERROR: Error in OnRecieve: {e.__str__()}")

    def RegisterAsForeignDevice(self, BBMD_IP: str, TTL: int, Port: int = 0xbac0):
        sent = False
        try:
            ep = (BBMD_IP, Port)
            if isinstance(self.m_client, BACnetIpUdpProtocolTransport):
                sent = self.m_client.SendRegisterAsForeignDevice(ep, TTL)

            if sent is False:
                if 'HandleLog' in self.events.keys():
                    self.events['HandleLog'](f"Warning: The given address do not match with the IP version")
            else:
                if 'HandleLog' in self.events.keys():
                    self.events['HandleLog'](f"INFO : Sending Register as a Foreign Device ...")
        except Exception as e:
            if 'HandleLog' in self.events.keys():
                self.events['HandleLog'](f"ERROR: Error on RegisterAsForeignDevice (Wrong Transport, not IP ?){e.__str__()}")

    def RemoteWhoIs(self, BBMD_IP: str, Port: int=0xbac0, low_limit: int=-1, high_limit: int=-1):
        try:
            ep = (BBMD_IP, Port)
            b = self.GetEncodeBuffer(self.m_client.HeaderLength())
            broadcast = self.m_client.GetBroadcastAddress()
            NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage, broadcast)
            APDU.EncodeUnconfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_UNCONFIRMED_SERVICE_REQUEST, BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_WHO_IS)
            Services.EncodeWhoIsBroadcast(b, low_limit, high_limit)

            sent = False
            if isinstance(self.m_client, BACnetIpUdpProtocolTransport):
                sent = self.m_client.SendRemoteWhois(b.buffer, ep, b.offset)

            if sent is False:
                if 'HandleLog' in self.events.keys():
                    self.events['HandleLog'](f"Warning: The given address do not match with the IP version")
            else:
                if 'HandleLog' in self.events.keys():
                    self.events['HandleLog'](f"INFO : Sending Remote Whois ...")
        except Exception as e:
            if 'HandleLog' in self.events.keys():
                self.events['HandleLog'](f"ERROR: Error on Sending Whois to remote BBMD (Wrong Transport, not IP ?){e.__str__()}")

    def WhoIs(self, low_limit: int=-1, high_limit: int=-1, _receiver: BACnetAddress=None):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending WhoIs ... ")
        b = self.GetEncodeBuffer(self.m_client.HeaderLength())

        if _receiver is not None:
            receiver = _receiver
        else:
            receiver = self.m_client.GetBroadcastAddress()

        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage, receiver)
        APDU.EncodeUnconfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_UNCONFIRMED_SERVICE_REQUEST, BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_WHO_IS)
        Services.EncodeWhoIsBroadcast(b, low_limit, high_limit)
        self.m_client.Send1(b.buffer, self.m_client.HeaderLength(), b.offset - self.m_client.HeaderLength(), receiver, False, 0)

    def Iam(self, device_id: int, segmentation: BACnetSegmentations=BACnetSegmentations.SEGMENTATION_BOTH, vendor_id: int=260, source: BACnetAddress=None):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending Iam({device_id.__str__()}-{segmentation.__str__()}) ... ")
        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        broadcast = self.m_client.GetBroadcastAddress()
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage, broadcast, source)
        APDU.EncodeUnconfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_UNCONFIRMED_SERVICE_REQUEST, BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_I_AM)
        Services.EncodeIamBroadcast(b, device_id, self.GetMaxApdu(), segmentation, vendor_id)
        self.m_client.Send1(b.buffer, self.m_client.HeaderLength(), b.offset - self.m_client.HeaderLength(), broadcast, False, 0)

    def IHave(self, device_id: BACnetObjectId, ObjId: BACnetObjectId, ObjName: str):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending IHave({device_id.__str__()}-{ObjId.__str__()}) ... ")
        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        broadcast = self.m_client.GetBroadcastAddress()
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage, broadcast)
        APDU.EncodeUnconfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_UNCONFIRMED_SERVICE_REQUEST, BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_I_HAVE)
        Services.EncodeIhaveBroadcast(b, device_id, ObjId, ObjName)
        self.m_client.Send1(b.buffer, self.m_client.HeaderLength(), b.offset - self.m_client.HeaderLength(), broadcast, False, 0)

    def SendUnconfirmedEventNotification(self, adr: BACnetAddress, eventData: BACnetEventNotificationData):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending Event Notification({adr.__str__()}) ... ")
        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage, adr.RoutedSource)
        APDU.EncodeUnconfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_UNCONFIRMED_SERVICE_REQUEST, BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_EVENT_NOTIFICATION)
        Services.EncodeEventNotifyUnconfirmed(b, eventData)
        self.m_client.Send1(b.buffer, self.m_client.HeaderLength(), b.offset - self.m_client.HeaderLength(), adr, False, 0)

    def SendConfirmedServiceReject(self, adr: BACnetAddress, invoke_id: int, reason: BACnetRejectReasons):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending Service reject({adr.__str__()}-{invoke_id.__str__()}) ... ")
        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage, adr.RoutedSource)
        APDU.EncodeError(b, BACnetPduTypes.PDU_TYPE_REJECT, BACnetConfirmedServices(reason), invoke_id)
        self.m_client.Send1(b.buffer, self.m_client.HeaderLength(), b.offset - self.m_client.HeaderLength(), adr, False, 0)

    def SynchronizeTime(self, adr: BACnetAddress, dateTime: datetime, utc: bool):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending Time Synchronize({adr.__str__()}-{dateTime.__str__()}) ... ")
        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage, adr.RoutedSource)
        if not utc:
            APDU.EncodeUnconfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_UNCONFIRMED_SERVICE_REQUEST, BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_TIME_SYNCHRONIZATION)
        else:
            APDU.EncodeUnconfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_UNCONFIRMED_SERVICE_REQUEST, BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_UTC_TIME_SYNCHRONIZATION)
        Services.EncodeTimeSync(b, dateTime)
        self.m_client.Send1(b.buffer, self.m_client.HeaderLength(), b.offset - self.m_client.HeaderLength(), adr, False, 0)

    def GetMaxApdu(self):
        max_apdu = 0
        if self.m_client.MaxAdpuLength() == BACnetMaxAdpu.MAX_APDU1476:
            max_apdu = 1476
        elif self.m_client.MaxAdpuLength() == BACnetMaxAdpu.MAX_APDU1024:
            max_apdu = 1024
        elif self.m_client.MaxAdpuLength() == BACnetMaxAdpu.MAX_APDU480:
            max_apdu = 480
        elif self.m_client.MaxAdpuLength() == BACnetMaxAdpu.MAX_APDU206:
            max_apdu = 206
        elif self.m_client.MaxAdpuLength() == BACnetMaxAdpu.MAX_APDU128:
            max_apdu = 128
        elif self.m_client.MaxAdpuLength() == BACnetMaxAdpu.MAX_APDU50:
            max_apdu = 50
        else:
            raise Exception(f"NotImplemented")
        max_npdu_header_length = 4
        return min(max_apdu, self.m_client.MaxBufferLength() - self.m_client.HeaderLength() - max_npdu_header_length)

    def GetFileBufferMaxSize(self):
        return self.GetMaxApdu() - 18

    def WriteFileRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, position: int, count: int, file_buffer: bytearray, invoke_id: int=0):
        result = self.BeginWriteFileRequest(adr, object_id, position, count, file_buffer, True, invoke_id)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                (position, ex) = self.EndWriteFileRequest(result)
                if ex is not None:
                    raise ex
                else:
                    return True
            if r < self.m_retries - 1:
                result.Resend()
        return False

    def BeginWriteFileRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, position: int, count: int, file_buffer: bytearray, wait_for_transmit: bool, invoke_id: int=0):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending AtomicWriteFileRequest({adr.__str__()}-{object_id.__str__()}-{position.__str__()}) ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST | (BACnetPduTypes.SEGMENTED_RESPONSE_ACCEPTED if self.m_max_segments != BACnetMaxSegments.MAX_SEG0 else 0), BACnetConfirmedServices.SERVICE_CONFIRMED_ATOMIC_WRITE_FILE, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)
        Services.EncodeAtomicWriteFile(b, True, object_id, position, 1, [file_buffer], [count])

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndWriteFileRequest(self, result: BACnetAsyncResult):
        res = result #(BACnetAsyncResult)result
        ex = res.m_error
        if ex is None and not res.wait(self.m_timeout):
            ex = Exception("Wait Timeout")

        if ex is None:
            (l, is_stream, position) = Services.DecodeAtomicWriteFileAcknowledge(res.m_result, 0, len(res.m_result))
            if l < 0:
                ex = Exception("Decode")
        else:
            position = -1
        res.Dispose()
        return (position, ex)

    def BeginReadFileRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, position: int, count: int, wait_for_transmit: bool, invoke_id: int=0):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending AtomicReadFileRequest({adr.__str__()}-{object_id.__str__()}-{position.__str__()}) ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)
        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST | (BACnetPduTypes.SEGMENTED_RESPONSE_ACCEPTED if self.m_max_segments != BACnetMaxSegments.MAX_SEG0 else 0), BACnetConfirmedServices.SERVICE_CONFIRMED_ATOMIC_READ_FILE, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)
        Services.EncodeAtomicReadFile(b, True, object_id, position, count)

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndReadFileRequest(self, result: BACnetAsyncResult):
        res = result
        ex = res.m_error
        if ex is None and not res.wait(self.m_timeout):
            ex = Exception("Wait Timeout")

        if ex is None:
            (l, end_of_file, is_stream, position, count, file_buffer, file_buffer_offset) = Services.DecodeAtomicReadFileAcknowledge(res.m_result, 0, len(res.m_result))
            if l < 0:
                ex = Exception("Decode")
        else:
            count = 0
            end_of_file = True
            position = -1
            file_buffer_offset = -1
            file_buffer = bytearray(0)
        res.Dispose()
        return (count, position, end_of_file, file_buffer, file_buffer_offset, ex)

    def ReadFileRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, position: int, count: int, invoke_id: int=0):

        result = self.BeginReadFileRequest(adr, object_id, position, count, True, invoke_id)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                (count, position, end_of_file, file_buffer, file_buffer_offset, ex) = self.EndReadFileRequest(result)
                if ex is not None:
                    raise ex
                else:
                    return (True, position, count, end_of_file, file_buffer, file_buffer_offset, invoke_id)
            if r < self.m_retries - 1:
                result.Resend()

        position = -1
        count = 0
        file_buffer = None
        end_of_file = True
        file_buffer_offset = -1
        return (False, position, count, end_of_file, file_buffer, file_buffer_offset, invoke_id)

    def BeginReadRangeRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, idxBegin: int, Quantity: int, wait_for_transmit: bool, invoke_id: int=0):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending ReadRangeRequest({adr.__str__()}-{object_id.__str__()}-{idxBegin.__str__()}) ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST | (BACnetPduTypes.SEGMENTED_RESPONSE_ACCEPTED if self.m_max_segments != BACnetMaxSegments.MAX_SEG0 else 0), BACnetConfirmedServices.SERVICE_CONFIRMED_READ_RANGE, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)
        Services.EncodeReadRange(b, object_id, BACnetPropertyIds.PROP_LOG_BUFFER, ASN1.BACNET_ARRAY_ALL, BACnetReadRangeRequestTypes.RR_BY_POSITION, idxBegin, datetime.now(), Quantity)

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndReadRangeRequest(self, result: BACnetAsyncResult):
        res = result
        ItemCount = 0
        trendbuffer = None

        ex = res.m_error
        if ex is None and not res.wait(40*1000):
            ex = Exception("Wait Timeout")

        if ex is None:
            (ItemCount, trendbuffer) = Services.DecodeReadRangeAcknowledge(res.m_result, 0, len(res.m_result))
            if ItemCount == 0:
                ex = Exception("Decode")
        res.Dispose()
        return (trendbuffer, ItemCount, ex)

    def ReadRangeRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, idxBegin: int, Quantity: int, invoke_id: int=0):
        Range = None
        result = self.BeginReadRangeRequest(adr, object_id, idxBegin, Quantity, True, invoke_id)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                (Range, Quantity, ex) = self.EndReadRangeRequest(result)
                if ex is not None:
                    raise ex
                else:
                    return (True, Quantity, Range)
            if r < self.m_retries - 1:
                result.Resend()

        return (False, Quantity, Range)

    def SubscribeCOVRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, subscribe_id: int, cancel: bool, issue_confirmed_notifications: bool, lifetime: int, invoke_id: int=0):
        result = self.BeginSubscribeCOVRequest(adr, object_id, subscribe_id, cancel, issue_confirmed_notifications, lifetime, True, invoke_id)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                ex = self.EndSubscribeCOVRequest(result)
                if ex is not None:
                    raise ex
                else:
                    return True
            if r < self.m_retries - 1:
                result.Resend()

        return False

    def BeginSubscribeCOVRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, subscribe_id: int, cancel: bool, issue_confirmed_notifications: bool, lifetime: int, wait_for_transmit: bool, invoke_id: int=0):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending SubscribeCOVRequest({adr.__str__()}-{object_id.__str__()}-{subscribe_id.__str__()}) ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST | (BACnetPduTypes.SEGMENTED_RESPONSE_ACCEPTED if self.m_max_segments != BACnetMaxSegments.MAX_SEG0 else 0), BACnetConfirmedServices.SERVICE_CONFIRMED_SUBSCRIBE_COV, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)
        Services.EncodeSubscribeCOV(b, subscribe_id, object_id, cancel, issue_confirmed_notifications, lifetime)

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndSubscribeCOVRequest(self, result: BACnetAsyncResult):
        res = result

        ex = res.m_error
        if ex is None and not res.wait(self.m_timeout):
            ex = Exception("Wait Timeout")

        res.Dispose()
        return ex

    def ReadPropertyRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, property_id: BACnetPropertyIds, invoke_id: int=0, array_index: int=ASN1.BACNET_ARRAY_ALL):
        result = self.BeginReadPropertyRequest(adr, object_id, property_id, True, invoke_id, array_index)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                (value_list, ex) = self.EndReadPropertyRequest(result)
                if ex is not None:
                    raise ex
                else:
                    return (True, value_list)
            if r < self.m_retries - 1:
                result.Resend()

        return (False, None)

    def BeginReadPropertyRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, property_id: BACnetPropertyIds, wait_for_transmit: bool, invoke_id: int=0, array_index: int=ASN1.BACNET_ARRAY_ALL):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending ReadPropertyRequest({adr.__str__()}-{object_id.__str__()}-{property_id.__str__()}) ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST | (BACnetPduTypes.SEGMENTED_RESPONSE_ACCEPTED if self.m_max_segments != BACnetMaxSegments.MAX_SEG0 else 0), BACnetConfirmedServices.SERVICE_CONFIRMED_READ_PROPERTY, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)
        Services.EncodeReadProperty(b, object_id, property_id, array_index)
        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndReadPropertyRequest(self, result: BACnetAsyncResult):
        res = result

        ex = res.m_error
        if ex is None and not res.wait(self.m_timeout):
            ex = Exception("Wait Timeout")

        if ex is None:
            (l, response_object_id, response_property, value_list) = Services.DecodeReadPropertyAcknowledge(res.m_result, 0, len(res.m_result))
            if l < 0:
                ex = Exception("Decode")
        else:
            value_list = None

        res.Dispose()
        return (value_list, ex)

    def WritePropertyRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, property_id: BACnetPropertyIds, value_list: list, invoke_id: int=0):
        result = self.BeginWritePropertyRequest(adr, object_id, property_id, value_list, True, invoke_id)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                ex = self.EndWritePropertyRequest(result)
                if ex is not None:
                    raise ex
                else:
                    return True
            if r < self.m_retries - 1:
                result.Resend()

        return False

    def BeginWritePropertyRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, property_id: BACnetPropertyIds, value_list: list, wait_for_transmit: bool, invoke_id: int=0):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending WritePropertyRequest({adr.__str__()}-{object_id.__str__()}-{property_id.__str__()}) ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST , BACnetConfirmedServices.SERVICE_CONFIRMED_WRITE_PROPERTY, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)
        Services.EncodeWriteProperty(b, object_id, property_id, ASN1.BACNET_ARRAY_ALL, self.m_writepriority, value_list)

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndWritePropertyRequest(self, result: BACnetAsyncResult):
        res = result

        ex = res.m_error
        if ex is None and not res.wait(self.m_timeout):
            ex = Exception("Wait Timeout")

        res.Dispose()
        return ex

    def ReadPropertyMultipleRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, property_id_and_array_index: list, invoke_id: int=0):
        result = self.BeginReadPropertyMultipleRequest(adr, object_id, property_id_and_array_index, True, invoke_id)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                (values, ex) = self.EndReadPropertyMultipleRequest(result)
                if ex is not None:
                    raise ex
                else:
                    return (True, values)
            if r < self.m_retries - 1:
                result.Resend()

        return (False, None)

    def BeginReadPropertyMultipleRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, property_id_and_array_index: list, wait_for_transmit: bool, invoke_id: int=0):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending ReadPropertyMultipleRequest({adr.__str__()}-{object_id.__str__()}) ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST | (BACnetPduTypes.SEGMENTED_RESPONSE_ACCEPTED if self.m_max_segments != BACnetMaxSegments.MAX_SEG0 else 0), BACnetConfirmedServices.SERVICE_CONFIRMED_READ_PROP_MULTIPLE, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)
        Services.EncodeReadPropertyMultiple(b, object_id, property_id_and_array_index)

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def ReadPropertyMultipleRequest1(self, adr: BACnetAddress, properties: list, invoke_id: int=0):
        result = self.BeginReadPropertyMultipleRequest1(adr, properties, True, invoke_id)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                (values, ex) = self.EndReadPropertyMultipleRequest(result)
                if ex is not None:
                    raise ex
                else:
                    return (True, values)
            if r < self.m_retries - 1:
                result.Resend()

        return (False, None)

    def BeginReadPropertyMultipleRequest1(self, adr: BACnetAddress, properties: list, wait_for_transmit: bool, invoke_id: int=0):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending ReadPropertyMultipleRequest1({adr.__str__()}-{[p.__str__() for p in properties]})  ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST | (BACnetPduTypes.SEGMENTED_RESPONSE_ACCEPTED if self.m_max_segments != BACnetMaxSegments.MAX_SEG0 else 0), BACnetConfirmedServices.SERVICE_CONFIRMED_READ_PROP_MULTIPLE, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)
        Services.EncodeReadPropertyMultiple1(b, properties)

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndReadPropertyMultipleRequest(self, result: BACnetAsyncResult):
        res = result

        ex = res.m_error
        if ex is None and not res.wait(self.m_timeout):
            ex = Exception("Wait Timeout")

        if ex is None:
            (l, values) = Services.DecodeReadPropertyMultipleAcknowledge(res.m_result, 0, len(res.m_result))
            if l < 0:
                ex = Exception("Decode")
        else:
            values = None

        res.Dispose()
        return (values, ex)

    def CreateObjectRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, value_list: list, invoke_id: int=0):
        result = self.BeginCreateObjectRequest(adr, object_id, value_list, True, invoke_id)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                ex = self.EndCreateObjectRequest(result)
                if ex is not None:
                    raise ex
                else:
                    return True
            if r < self.m_retries - 1:
                result.Resend()

        return False

    def BeginCreateObjectRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, value_list: list, wait_for_transmit: bool, invoke_id: int=0):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending CreateObjectRequest({adr.__str__()}-{object_id.__str__()}) ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST | (BACnetPduTypes.SEGMENTED_RESPONSE_ACCEPTED if self.m_max_segments != BACnetMaxSegments.MAX_SEG0 else 0), BACnetConfirmedServices.SERVICE_CONFIRMED_CREATE_OBJECT, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)
        Services.EncodeCreateProperty(b, object_id, value_list)

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndCreateObjectRequest(self, result: BACnetAsyncResult):
        res = result

        ex = res.m_error
        if ex is None and not res.wait(self.m_timeout):
            ex = Exception("Wait Timeout")

        res.Dispose()
        return ex

    def DeleteObjectRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, invoke_id: int=0):
        result = self.BeginDeleteObjectRequest(adr, object_id, True, invoke_id)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                ex = self.EndDeleteObjectRequest(result)
                if ex is not None:
                    raise ex
                else:
                    return True
            if r < self.m_retries - 1:
                result.Resend()

        return False

    def BeginDeleteObjectRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, wait_for_transmit: bool, invoke_id: int=0):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending DeleteObjectRequest({adr.__str__()}-{object_id.__str__()}) ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST, BACnetConfirmedServices.SERVICE_CONFIRMED_DELETE_OBJECT, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)
        ASN1.encode_application_object_id(b, object_id.type, object_id.instance)

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndDeleteObjectRequest(self, result: BACnetAsyncResult):
        res = result

        ex = res.m_error
        if ex is None and not res.wait(self.m_timeout):
            ex = Exception("Wait Timeout")

        res.Dispose()
        return ex

    def RawEncodedDecodedPropertyConfirmedRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, property_id: BACnetPropertyIds, service_id: BACnetConfirmedServices, InOutBuffer: bytearray, invoke_id: int=0):
        result = self.BeginRawEncodedDecodedPropertyConfirmedRequest(adr, object_id, property_id, service_id, InOutBuffer, True, invoke_id)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                (InOutBuffer, ex) = self.EndRawEncodedDecodedPropertyConfirmedRequest(result, service_id)
                if ex is not None:
                    raise ex
                else:
                    return True
            if r < self.m_retries - 1:
                result.Resend()

        return False

    def BeginRawEncodedDecodedPropertyConfirmedRequest(self, adr: BACnetAddress, object_id: BACnetObjectId, property_id: BACnetPropertyIds, service_id: BACnetConfirmedServices, InOutBuffer: bytearray, wait_for_transmit: bool, invoke_id: int=0):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending RawEncodedRequest({adr.__str__()}-{object_id.__str__()}-{property_id.__str__()}-{service_id.__str__()}) ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST | (BACnetPduTypes.SEGMENTED_RESPONSE_ACCEPTED if self.m_max_segments != BACnetMaxSegments.MAX_SEG0 else 0), service_id, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)

        ASN1.encode_context_object_id(b, 0, object_id.type, object_id.instance)
        ASN1.encode_context_enumerated(b, 1, property_id)

        if InOutBuffer is not None:
            b.Add(InOutBuffer, len(InOutBuffer))

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndRawEncodedDecodedPropertyConfirmedRequest(self, result: BACnetAsyncResult, service_id: BACnetConfirmedServices):
        res = result

        ex = res.m_error
        if ex is None and not res.wait(self.m_timeout):
            ex = Exception("Wait Timeout")

        InOutBuffer = None
        if ex is None:
            if service_id == BACnetConfirmedServices.SERVICE_CONFIRMED_READ_PROPERTY:
                buffer = res.m_result
                ex = Exception("Decode")

                offset = 0
                length = 0
                if not ASN1.decode_is_context_tag(buffer, offset, 0):
                    return

                length = 1
                response_object_id = BACnetObjectId()
                (l, response_object_id.type, response_object_id.instance) = ASN1.decode_object_id(buffer, offset + length)
                length += l

                (l, tag_number, len_value_type) = ASN1.decode_tag_number_and_value(buffer, offset + length)
                length += l

                InOutBuffer = bytearray(len(buffer) - length)
                InOutBuffer[0: len(InOutBuffer)] = buffer[length: length + len(InOutBuffer)]
                ex = None
        res.Dispose()
        return (InOutBuffer, ex)

    def DeviceCommunicationControlRequest(self, adr: BACnetAddress, timeDuration: int, enable_disable: int, password: str, invoke_id: int=0):
        result = self.BeginDeviceCommunicationControlRequest(adr, timeDuration, enable_disable, password, True, invoke_id)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                ex = self.EndDeviceCommunicationControlRequest(result)
                if ex is not None:
                    return False
                else:
                    return True
            if r < self.m_retries - 1:
                result.Resend()

        return False

    def BeginDeviceCommunicationControlRequest(self, adr: BACnetAddress, timeDuration: int, enable_disable: int, password: str, wait_for_transmit: bool, invoke_id: int=0):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending DeviceCommunicationControlRequest({adr.__str__()}) ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST, BACnetConfirmedServices.SERVICE_CONFIRMED_DEVICE_COMMUNICATION_CONTROL, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)
        Services.EncodeDeviceCommunicationControl(b, timeDuration, enable_disable, password)

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndDeviceCommunicationControlRequest(self, result: BACnetAsyncResult):
        res = result

        ex = res.m_error
        if ex is None and not res.wait(self.m_timeout):
            ex = Exception("Wait Timeout")

        res.Dispose()
        return ex

    def GetAlarmSummaryOrEventRequest(self, adr: BACnetAddress, GetEvent: bool, Alarms: list, invoke_id: int=0):
        result = self.BeginGetAlarmSummaryOrEventRequest(adr, GetEvent, Alarms, True, invoke_id)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                (Alarms, MoreEvent, ex) = self.EndGetAlarmSummaryOrEventRequest(result, GetEvent, Alarms)
                if ex is not None:
                    return (False, Alarms)
                else:
                    if MoreEvent is True:
                        return self.GetAlarmSummaryOrEventRequest(adr, GetEvent, Alarms)
                    return (True, Alarms)

            if r < self.m_retries - 1:
                result.Resend()

        return (False, Alarms)

    def BeginGetAlarmSummaryOrEventRequest(self, adr: BACnetAddress, GetEvent: bool, Alarms: list, wait_for_transmit: bool, invoke_id: int=0, array_index: int=ASN1.BACNET_ARRAY_ALL):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending Alarm summary request({adr.__str__()}) ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        if GetEvent is False:
            APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST | (BACnetPduTypes.SEGMENTED_RESPONSE_ACCEPTED if self.m_max_segments != BACnetMaxSegments.MAX_SEG0 else 0), BACnetConfirmedServices.SERVICE_CONFIRMED_GET_ALARM_SUMMARY, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)
        else:
            APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST | (BACnetPduTypes.SEGMENTED_RESPONSE_ACCEPTED if self.m_max_segments != BACnetMaxSegments.MAX_SEG0 else 0), BACnetConfirmedServices.SERVICE_CONFIRMED_GET_EVENT_INFORMATION, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)

        if len(Alarms) != 0:
            ASN1.encode_context_object_id(b, 0, Alarms[len(Alarms) - 1].objectIdentifier.type, Alarms[len(Alarms) - 1].objectIdentifier.instance)

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndGetAlarmSummaryOrEventRequest(self, result: BACnetAsyncResult, GetEvent: bool, Alarms: list):
        res = result

        ex = res.m_error
        if ex is None and not res.wait(self.m_timeout):
            ex = Exception("Wait Timeout")

        if ex is None:
            (l, Alarms, MoreEvent) = Services.DecodeAlarmSummaryOrEvent(res.m_result, 0, len(res.m_result), GetEvent, Alarms)
            if l < 0:
                ex = Exception("Decode")
        else:
            ex = Exception("Service not available")

        res.Dispose()
        return (Alarms, GetEvent, ex)

    def AlarmAcknowledgement(self, adr: BACnetAddress, objid: BACnetObjectId, eventState: BACnetEventStates, AckText: str, evTimeStamp: BACnetGenericTime, ackTimeStamp: BACnetGenericTime, invoke_id: int=0):
        result = self.BeginAlarmAcknowledgement(adr, objid, eventState, AckText, evTimeStamp, ackTimeStamp, True, invoke_id)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                ex = self.EndAlarmAcknowledgement(result)
                if ex is not None:
                    return False
                else:
                    return True

            if r < self.m_retries - 1:
                result.Resend()

        return False

    def BeginAlarmAcknowledgement(self, adr: BACnetAddress, objid: BACnetObjectId, eventState: BACnetEventStates, AckText: str, evTimeStamp: BACnetGenericTime, ackTimeStamp: BACnetGenericTime, wait_for_transmit: bool, invoke_id: int=0, array_index: int=ASN1.BACNET_ARRAY_ALL):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending AlarmAcknowledgement({adr.__str__()}-{objid.__str__()}) ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST, BACnetConfirmedServices.SERVICE_CONFIRMED_ACKNOWLEDGE_ALARM, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)

        Services.EncodeAlarmAcknowledge(b, 57, objid, eventState, AckText, evTimeStamp, ackTimeStamp)

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndAlarmAcknowledgement(self, result: BACnetAsyncResult):
        res = result

        ex = res.m_error
        if ex is None and not res.wait(self.m_timeout):
            ex = Exception("Wait Timeout")

        res.Dispose()
        return ex

    def ReinitializeRequest(self, adr: BACnetAddress, state: BACnetReinitializedStates, password: str, invoke_id: int=0):
        result = self.BeginReinitializeRequest(adr, state, password, True, invoke_id)
        for r in range(self.m_retries):
            if result.wait(self.m_timeout):
                ex = self.EndReinitializeRequest(result)
                if ex is not None:
                    return False
                else:
                    return True

            if r < self.m_retries - 1:
                result.Resend()

        return False

    def BeginReinitializeRequest(self, adr: BACnetAddress, state: BACnetReinitializedStates, password: str, wait_for_transmit: bool, invoke_id: int=0):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending ReinitializeRequest({adr.__str__()}-{state.__str__()}) ... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST, BACnetConfirmedServices.SERVICE_CONFIRMED_REINITIALIZE_DEVICE, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)
        Services.EncodeReinitializeDevice(b, state, password)

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndReinitializeRequest(self, result: BACnetAsyncResult):
        res = result

        ex = res.m_error
        if ex is None and not res.wait(self.m_timeout):
            ex = Exception("Wait Timeout")

        res.Dispose()
        return ex

    def Notify(self, adr: BACnetAddress, subscriberProcessIdentifier: int, initiatingDeviceIdentifier: int, monitoredObjectIdentifier: BACnetObjectId, timeRemaining: int, issueConfirmedNotifications: bool, values: list):
        if not issueConfirmedNotifications:
            if 'HandleLog' in self.events.keys():
                self.events['HandleLog'](f"INFO : Sending Notify (unconfirmed) ({adr.__str__()}-{subscriberProcessIdentifier.__str__()}-{initiatingDeviceIdentifier.__str__()}-{monitoredObjectIdentifier.__str__()})... ")
            b = self.GetEncodeBuffer(self.m_client.HeaderLength())
            NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage, adr.RoutedSource)
            APDU.EncodeUnconfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_UNCONFIRMED_SERVICE_REQUEST, BACnetUnconfirmedServices.SERVICE_UNCONFIRMED_COV_NOTIFICATION)
            Services.EncodeCOVNotifyUnconfirmed(b, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values)

            sendbytes = self.m_client.Send1(b.buffer, self.m_client.HeaderLength(), b.offset - self.m_client.HeaderLength(), adr, False, 0)
            if sendbytes == b.offset:
                return True
            else:
                return False
        else:
            result = self.BeginConfirmedNotify(adr, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values, True)
            for r in range(self.m_retries):
                if result.wait(self.m_timeout):
                    ex = self.EndConfirmedNotify(result)
                    if ex is not None:
                        raise ex
                    else:
                        return True

                if r < self.m_retries - 1:
                    result.Resend()
        return False

    def BeginConfirmedNotify(self, adr: BACnetAddress, subscriberProcessIdentifier: int, initiatingDeviceIdentifier: int, monitoredObjectIdentifier: BACnetObjectId, timeRemaining: int, values: list, wait_for_transmit: bool, invoke_id: int=0):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending Notify (confirmed) ({adr.__str__()}-{subscriberProcessIdentifier.__str__()}-{initiatingDeviceIdentifier.__str__()}-{monitoredObjectIdentifier.__str__()})... ")
        if invoke_id == 0:
            invoke_id = self.m_invoke_id
            self.m_invoke_id = self.unchecked(self.m_invoke_id)

        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage | BACnetNpduControls.ExpectingReply, adr.RoutedSource)
        APDU.EncodeConfirmedServiceRequest(b, BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST, BACnetConfirmedServices.SERVICE_CONFIRMED_COV_NOTIFICATION, self.m_max_segments, self.m_client.MaxAdpuLength(), invoke_id)
        Services.EncodeCOVNotifyConfirmed(b, subscriberProcessIdentifier, initiatingDeviceIdentifier, monitoredObjectIdentifier, timeRemaining, values)

        ret = BACnetAsyncResult(self, adr, invoke_id, b.buffer, b.offset - self.m_client.HeaderLength(), wait_for_transmit, self.m_transmit_timeout)
        ret.Resend()

        return ret

    def EndConfirmedNotify(self, result: BACnetAsyncResult):
        res = result

        ex = res.m_error
        if ex is None and not res.wait(self.m_timeout):
            ex = Exception("Wait Timeout")

        res.Dispose()
        return ex

    def GetSegmentsCount(self, max_segments: BACnetMaxSegments) -> int:
        if max_segments == BACnetMaxSegments.MAX_SEG0:
            return 0
        elif max_segments == BACnetMaxSegments.MAX_SEG2:
            return 2
        elif max_segments == BACnetMaxSegments.MAX_SEG4:
            return 4
        elif max_segments == BACnetMaxSegments.MAX_SEG8:
            return 8
        elif max_segments == BACnetMaxSegments.MAX_SEG16:
            return 16
        elif max_segments == BACnetMaxSegments.MAX_SEG32:
            return 32
        elif max_segments == BACnetMaxSegments.MAX_SEG64:
            return 64
        elif max_segments == BACnetMaxSegments.MAX_SEG65:
            return 0xFF
        else:
            raise Exception("Not an option")

    def GetSegmentsCount1(self, max_segments: int) -> BACnetMaxSegments:
        if max_segments == 0:
            return BACnetMaxSegments.MAX_SEG0
        elif max_segments <= 2:
            return BACnetMaxSegments.MAX_SEG2
        elif max_segments <= 4:
            return BACnetMaxSegments.MAX_SEG4
        elif max_segments <= 8:
            return BACnetMaxSegments.MAX_SEG8
        elif max_segments <= 16:
            return BACnetMaxSegments.MAX_SEG16
        elif max_segments <= 32:
            return BACnetMaxSegments.MAX_SEG32
        elif max_segments <= 64:
            return BACnetMaxSegments.MAX_SEG64
        else:
            return BACnetMaxSegments.MAX_SEG65

    def GetSegmentBuffer(self, max_segments: BACnetMaxSegments):
        if max_segments == BACnetMaxSegments.MAX_SEG0:
            return None
        ret = Segmentation()
        ret.buffer = self. GetEncodeBuffer(self.m_client.HeaderLength())
        ret.max_segments = self.GetSegmentsCount(max_segments)
        ret.window_size = self.m_proposed_window_size
        return ret

    def EncodeSegmentHeader(self, adr: BACnetAddress, invoke_id: int, segmentation: Segmentation, service: BACnetConfirmedServices, more_follows: bool):
        is_segmented = False
        if segmentation is None:
            buffer = self.GetEncodeBuffer(self.m_client.HeaderLength())
        else:
            buffer = segmentation.buffer
            is_segmented = segmentation.sequence_number > 0 | more_follows
        buffer.Reset(self.m_client.HeaderLength())

        NPDU.Encode(buffer, BACnetNpduControls.PriorityNormalMessage, adr.RoutedSource)

        buffer.max_offset = buffer.offset + self.GetMaxApdu()
        apdu_header = APDU.EncodeComplexAck(buffer, BACnetPduTypes.PDU_TYPE_COMPLEX_ACK | (BACnetPduTypes.SEGMENTED_MESSAGE | BACnetPduTypes.SERVER) if is_segmented else 0 | (BACnetPduTypes.MORE_FOLLOWS if more_follows else 0), service, invoke_id, segmentation.sequence_number if segmentation is not None else 0, segmentation.window_size if segmentation is not None else 0)
        buffer.min_limit = (self.GetMaxApdu() - apdu_header) * (segmentation.sequence_number if segmentation is not None else 0)
        return buffer

    def EncodeSegment(self, adr: BACnetAddress, invoke_id: int, segmentation: Segmentation, service: BACnetConfirmedServices, apdu_content_encode):
        buffer = self.EncodeSegmentHeader(adr, invoke_id, segmentation, service, False)
        apdu_content_encode(buffer)

        more_follows = (buffer.result & EncodeResult.NotEnoughBuffer) > 0
        if segmentation is not None and more_follows:
            self.EncodeSegmentHeader(adr, invoke_id, segmentation, service, True)
            apdu_content_encode(buffer)
            return (True, buffer)
        elif more_follows:
            return (True, buffer)
        else:
            return (segmentation.sequence_number > 0 if segmentation is not None else False, buffer)

    def HandleSegmentationResponseThread(self, adr: BACnetAddress, invoke_id: int, segmentation: Segmentation, transmit):
        old_max_info_frames = self.m_client.MaxInfoFrames
        self.m_client.MaxInfoFrames = segmentation.window_size

        while True:
            more_follows = (segmentation.buffer.result & EncodeResult.NotEnoughBuffer) > 0
            if (segmentation.sequence_number - 1) % segmentation.window_size == 0 or not more_follows:
                if not self.WaitForAllTransmits(self.m_transmit_timeout):
                    if 'HandleLog' in self.events.keys():
                        self.events['HandleLog']('Warning: Transmit timeout')
                    break
                current_number = segmentation.sequence_number
                if not self.WaitForSegmentAck(adr, invoke_id, segmentation, self.m_timeout):
                    if 'HandleLog' in self.events.keys():
                        self.events['HandleLog']("Warning: Didn't get segmentACK")
                    break
                if segmentation.sequence_number != current_number:
                    if 'HandleLog' in self.events.keys():
                        self.events['HandleLog']('Warning: Oh, a retransmit')
                    more_follows = True
            else:
                current_number = segmentation.sequence_number
                self.WaitForSegmentAck(adr, invoke_id, segmentation, 0)
                if segmentation.sequence_number != current_number:
                    if 'HandleLog' in self.events.keys():
                        self.events['HandleLog']('Warning: Oh, a retransmit')
                    more_follows = True

            if more_follows is True:
                transmit(segmentation)
            else:
                break

        self.m_client.MaxInfoFrames = old_max_info_frames

    def HandleSegmentationResponse(self, adr: BACnetAddress, invoke_id: int, segmentation: Segmentation, transmit):
        transmit(segmentation)

        if segmentation is None or segmentation.buffer.result == EncodeResult.Good:
            return

        threading.Thread(target=self.HandleSegmentationResponseThread, args=(adr, invoke_id, segmentation, transmit), daemon=True).start()

    def SendComplexAck(self, adr: BACnetAddress, invoke_id: int, segmentation: Segmentation, service: BACnetConfirmedServices, apdu_content_encode):

        (l, buffer) = self.EncodeSegment(adr, invoke_id, segmentation, service, apdu_content_encode)
        if l:
            if segmentation is None:
                if 'HandleLog' in self.events.keys():
                    self.events['HandleLog']('INFO : Segmenation denied')
                self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_APDU_TOO_LONG)
                buffer.result = EncodeResult.Good
                return

            if segmentation.sequence_number == 0:
                if segmentation.max_segments != 0xFF and segmentation.buffer.offset > (segmentation.max_segments * (self.GetMaxApdu() - 5)):
                    if 'HandleLog' in self.events.keys():
                        self.events['HandleLog']('INFO : Too much segmenation')
                    self.ErrorResponse(adr, service, invoke_id, BACnetErrorClasses.ERROR_CLASS_SERVICES, BACnetErrorCodes.ERROR_CODE_ABORT_APDU_TOO_LONG)
                    buffer.result = EncodeResult.Good
                    return
                else:
                    if 'HandleLog' in self.events.keys():
                        self.events['HandleLog']('INFO : Segmentation required')

            segmentation.sequence_number += 1

        self.m_client.Send1(buffer.buffer, self.m_client.HeaderLength(), buffer.GetLength() - self.m_client.HeaderLength(), adr, False, 0)

    def ReadPropertyResponse(self, adr: BACnetAddress, invoke_id: int, segmentation: Segmentation, object_id: BACnetObjectId, property: BACnetPropertyReference, value):
        self.HandleSegmentationResponse(adr, invoke_id, segmentation, self.SendComplexAck(adr, invoke_id, segmentation, BACnetConfirmedServices.SERVICE_CONFIRMED_READ_PROPERTY, Services.EncodeReadPropertyAcknowledge(EncodeBuffer(), object_id, property.propertyIdentifier, property.propertyArrayIndex, value)))

    def CreateObjectResponse(self, adr: BACnetAddress, invoke_id: int, segmentation: Segmentation, object_id: BACnetObjectId):
        self.SendComplexAck(adr, invoke_id, segmentation, BACnetConfirmedServices.SERVICE_CONFIRMED_CREATE_OBJECT, Services.EncodeCreateObjectAcknowledge(EncodeBuffer(), object_id))

    def ReadPropertyMultipleResponse(self, adr: BACnetAddress, invoke_id: int, segmentation: Segmentation, values: list):
        self.HandleSegmentationResponse(adr, invoke_id, segmentation, self.SendComplexAck(adr, invoke_id, segmentation, BACnetConfirmedServices.SERVICE_CONFIRMED_READ_PROP_MULTIPLE, Services.EncodeReadPropertyMultipleAcknowledge(EncodeBuffer(), values)))

    def ReadRangeResponse(self, adr: BACnetAddress, invoke_id: int, segmentation: Segmentation, object_id: BACnetObjectId, property: BACnetPropertyReference, status: BACnetResultFlags, item_count: int, application_data: bytearray, request_type: BACnetReadRangeRequestTypes, first_sequence_no: int):
        self.HandleSegmentationResponse(adr, invoke_id, segmentation, self.SendComplexAck(adr, invoke_id, segmentation, BACnetConfirmedServices.SERVICE_CONFIRMED_READ_RANGE, Services.EncodeReadRangeAcknowledge(EncodeBuffer(), object_id, property.propertyIdentifier, property.propertyArrayIndex, BACnetBitString().ConvertFromInt(status), item_count, application_data, request_type, first_sequence_no)))

    def ReadFileResponse(self, adr: BACnetAddress, invoke_id: int, segmentation: Segmentation, position: int, count: int, end_of_file: bool, file_buffer: bytearray):
        self.HandleSegmentationResponse(adr, invoke_id, segmentation, self.SendComplexAck(adr, invoke_id, segmentation, BACnetConfirmedServices.SERVICE_CONFIRMED_READ_PROPERTY, Services.EncodeAtomicReadFileAcknowledge(EncodeBuffer(), True, end_of_file, position, 1, [file_buffer], [count])))

    def WriteFileResponse(self, adr: BACnetAddress, invoke_id: int, segmentation: Segmentation, position: int):
        self.SendComplexAck(adr, invoke_id, segmentation, BACnetConfirmedServices.SERVICE_CONFIRMED_ATOMIC_WRITE_FILE, Services.EncodeAtomicWriteFileAcknowledge(EncodeBuffer(), True, position))

    def ErrorResponse(self, adr: BACnetAddress, service: BACnetConfirmedServices, invoke_id: int, error_class: BACnetErrorClasses, error_code: BACnetErrorCodes):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending ErrorResponse ({adr.__str__()}-{service.__str__()}-{invoke_id})... ")
        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage, adr.RoutedSource)
        APDU.EncodeError(b, BACnetPduTypes.PDU_TYPE_ERROR, service, invoke_id)
        Services.EncodeError(b, error_class, error_code)
        self.m_client.Send1(b.buffer, self.m_client.HeaderLength(), b.offset - self.m_client.HeaderLength(), adr, False, 0)

    def SimpleAckResponse(self, adr: BACnetAddress, service: BACnetConfirmedServices, invoke_id: int):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending SimpleAckResponse ({adr.__str__()}-{service.__str__()}-{invoke_id})... ")
        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage, adr.RoutedSource)
        APDU.EncodeError(b, BACnetPduTypes.PDU_TYPE_SIMPLE_ACK, service, invoke_id)
        self.m_client.Send1(b.buffer, self.m_client.HeaderLength(), b.offset - self.m_client.HeaderLength(), adr, False, 0)

    def SegmentAckResponse(self, adr: BACnetAddress, negative: bool, server: bool, original_invoke_id: int, sequence_number: int, actual_window_size: int):
        if 'HandleLog' in self.events.keys():
            self.events['HandleLog'](f"INFO : Sending SimpleAckResponse ... ({adr.__str__()}-{original_invoke_id})")
        b = self.GetEncodeBuffer(self.m_client.HeaderLength())
        NPDU.Encode(b, BACnetNpduControls.PriorityNormalMessage, adr.RoutedSource)
        APDU.EncodeSegmentAck(b, BACnetPduTypes.PDU_TYPE_SEGMENT_ACK | BACnetPduTypes.NEGATIVE_ACK if negative else 0 | BACnetPduTypes.SERVER if server else 0, original_invoke_id, sequence_number, actual_window_size)
        self.m_client.Send1(b.buffer, self.m_client.HeaderLength(), b.offset - self.m_client.HeaderLength(), adr, False, 0)

    def WaitForAllTransmits(self, timeout: int):
        self.m_client.WaitForAllTransmits(timeout)

    def WaitForSegmentAck(self, adr: BACnetAddress, invoke_id: int, segmentation: Segmentation, timeout: int):
        signaled = self.m_last_segment_ack.Wait(adr, invoke_id, timeout)
        if signaled:
            segmentation.sequence_number = ((self.m_last_segment_ack.sequence_number + 1) % 256)
            segmentation.window_size = self.m_last_segment_ack.window_size
        return signaled

    def Dispose(self):
        self.m_client.Dispose()
        del self.m_client
        self.m_client = None

    '''
    def ReadPropertyRequest(self,device_identifier:BACnetObjectIdentifier = None, adr:BACnetAddress = None, rq: ReadProperty_Request = None):
        #fixme as async and retries and await answer, not correct yet!!!
        task = asyncio.ensure_future(self.BeginReadPropertyRequest(device_identifier, adr, rq))
        return task

    async def BeginReadPropertyRequest(self,device_identifier:BACnetObjectIdentifier = None, adr:BACnetAddress = None, rq: ReadProperty_Request = None):

        logging.info("Sending ReadPropertyRequest")
        npdu = NPDU(destination=BACnetAddress(net_type=BACnetNetworkType.IPV4, address=device_identifier,
                                              network_number=adr.network_number))
        npdu.control.data_expecting_reply = True
        npdu.control.NetworkPriority.Normal_Message = True

        apdu = APDU(pdu_type=BACnetPduTypes.PDU_TYPE_CONFIRMED_SERVICE_REQUEST,
                    service_choice=BACnetConfirmedServiceChoice.READ_PROPERTY,
                    segmented_response_accepted=False,
                    max_segments_accepted=BACnetSegmentation.NO_SEGMENTATION,
                    max_apdu_length_accepted=BACnetMaxAdpu.MAX_APDU1476,
                    invoke_id=self._m_invoke_id
                    )

        buffer = npdu.encode() + apdu.encode() + rq.ASN1encode()
        result = BACnetResult(self, adr, self._m_invoke_id, buffer, len(buffer), False, 0)
        self._m_invoke_id += 1
        if self._m_invoke_id > 255:
            self._m_invoke_id = 0

        result.send()
        await result.Done()
        print(result.result)
        return result.result
    '''

class BACnetDeviceLine:

    def __init__(self, comm: BACnetClient):
        self.Line = comm
        self.Devices: dict = {}
        self.mstp_sources_seen: set = set()
        self.mstp_pfm_destinations_seen: set = set()
        self.mstp_sources_seen_lock = threading.Lock()

##BACnetApplication###########################
class BACnetApplication:

    def __init__(self):
        self.default_port: int = 0xBAC0
        self.default_proposed_window_size: int = 20
        self.default_retries: int = 3
        self.default_timeout: float = 1
        self.default_segments_max: int = 65
        self.default_id: int = 12345

        self.m_devices: dict = {}
        self.m_subscription_list: dict = {}
        self.m_next_subscription_id = 0
        self.Subscriptions_IssueConfirmedNotifies = False
        self.Subscriptions_Lifetime = 120

        self.call_back_func = None

    def SetHandleLog(self, HandleLog: Callable):
        self.call_back_func = HandleLog
        for sender in self.m_devices.keys():
            sender.SetHandleLog(HandleLog)

    def OnWhoIs(self, sender : BACnetClient, adr: BACnetAddress, low_limit: int, high_limit: int):
        if low_limit != -1 and self.default_id < low_limit:
            return
        elif high_limit != -1 and self.default_id > high_limit:
            return
        #sender.Iam(self.default_id, BACnetSegmentations.SEGMENTATION_BOTH, 61440)

    def OnIam(self, sender : BACnetClient, adr: BACnetAddress, device_id: int, max_apdu: int, segmentation: BACnetSegmentations, vendor_id: int):
        if self.call_back_func is not None:
            self.call_back_func(f"INFO : Receive IAM {device_id} {adr.__str__()}")
        new_entry = (adr, device_id)
        if sender not in self.m_devices.keys():
            return
        if new_entry not in self.m_devices[sender].Devices.keys():
            self.m_devices[sender].Devices[new_entry] = BACnetDevice(**{'adr': adr, 'device_id': device_id, 'max_apdu': max_apdu, 'vendor_id': vendor_id, 'segmentation': segmentation})
        else:
            self.m_devices[sender].Devices[new_entry].Update(**{'adr': adr, 'device_id': device_id, 'max_apdu': max_apdu, 'vendor_id': vendor_id, 'segmentation': segmentation})

    # TODO
    def OnReadPropertyRequest(self, sender: BACnetClient, adr: BACnetAddress, invoke_id: int, object_id: BACnetObjectId, property: BACnetPropertyReference, max_segments: BACnetMaxSegments):
        if self.call_back_func is not None:
            self.call_back_func(f"INFO : OnReadPropertyRequest {adr.__str__()}-{invoke_id}-{object_id.__str__()}")

    #TODO
    def OnReadPropertyMultipleRequest(self, sender: BACnetClient, adr: BACnetAddress, invoke_id: int, properties: list, max_segments: BACnetMaxSegments):
        if self.call_back_func is not None:
            self.call_back_func(f"INFO : OnReadPropertyMultipleRequest {adr.__str__()}-{invoke_id}")

    # TODO
    def OnEventNotify(self, sender: BACnetClient, adr: BACnetAddress, invoke_id: int, EventData: BACnetEventNotificationData, need_confirm: bool):
        if self.call_back_func is not None:
            self.call_back_func(f"INFO : OnEventNotify {adr.__str__()}-{invoke_id}")

    # TODO
    def OnCOVNotification(self, sender: BACnetClient, adr: BACnetAddress, invoke_id: int, subscriberProcessIdentifier: int, initiatingDeviceIdentifier: BACnetObjectId, monitoredObjectIdentifier: BACnetObjectId, timeRemaining: int, need_confirm: bool, values: list, max_segments: BACnetMaxSegments):
        if self.call_back_func is not None:
            self.call_back_func(f"INFO : OnCOVNotification {adr.__str__()}-{invoke_id}")
        sub_key = f"{adr.__str__()}:{initiatingDeviceIdentifier.instance}:{subscriberProcessIdentifier}"
        if sub_key in self.m_subscription_list.keys():
            self.m_subscription_list[sub_key].UpdateProperty(values)

        if need_confirm:
            sender.SimpleAckResponse(adr, BACnetConfirmedServices.SERVICE_CONFIRMED_COV_NOTIFICATION, invoke_id)

    def MSTP_FrameRecieved(self, sender: BACnetMstpProtocolTransport, frame_type: BACnetMstpFrameTypes, destination_address: int, source_address: int, msg_length: int):
        device_line = None
        for l in self.m_devices.values():
            if l.Line.m_client == sender:
                device_line = l
                break

        if device_line is None:
            return

        with device_line.mstp_sources_seen_lock:
            if source_address not in device_line.mstp_sources_seen:
                device_line.mstp_sources_seen.add(source_address)
                adr = BACnetAddress(BACnetAddressTypes.MSTP, 0, bytearray(BACnetTool.int_to_bytes(source_address)))
                device_id = 0xFFFFFFFF
                new_entry = (adr, device_id)

                if new_entry not in device_line.Devices.keys():
                    device_line.Devices[new_entry] = BACnetDevice(**{'adr': adr, 'device_id': device_id})
                else:
                    device_line.Devices[new_entry].Update(**{'adr': adr, 'device_id': device_id})

            if frame_type == BACnetMstpFrameTypes.FRAME_TYPE_POLL_FOR_MASTER and destination_address not in device_line.mstp_pfm_destinations_seen and sender.m_TS != destination_address:
                device_line.mstp_pfm_destinations_seen.add(destination_address)

    ##API###########
    def AddBACnetClient(self, type: BACnetAddressTypes, **kwargs):
        comm: Optional[BACnetClient] = None
        if type == BACnetAddressTypes.IP:
            comm = self.CreateBacnetIpUdpClient(kwargs.get('ip'), kwargs.get('port', self.default_port))
        elif type == BACnetAddressTypes.MSTP:
            comm = self.CreateBacnetMstpClient(kwargs.get('port_name'), kwargs.get('baud_rate'), kwargs.get('source_address'))

        self.default_id = kwargs.get('identifier', self.default_retries)

        if comm is not None:
            comm.m_proposed_window_size = kwargs.get('proposed_window_size', self.default_proposed_window_size)
            comm.m_retries = kwargs.get('retries', self.default_retries)
            comm.m_timeout = kwargs.get('timeout', self.default_timeout)
            comm.m_max_segments = comm.GetSegmentsCount1(kwargs.get('segments_max', self.default_segments_max))

            comm.events['OnWhoIs'] = self.OnWhoIs
            comm.events['OnIam'] = self.OnIam
            comm.events['OnReadPropertyRequest'] = self.OnReadPropertyRequest
            comm.events['OnCOVNotification'] = self.OnCOVNotification
            comm.events['OnEventNotify'] = self.OnEventNotify

            if kwargs.get('whois') is True and (type == BACnetAddressTypes.IP or isinstance(comm.m_client, BACnetMstpProtocolTransport)):
                self.QueueUserWorkItem(self.SendWhoIs, (comm,))

            if isinstance(comm.m_client, BACnetMstpProtocolTransport):
                comm.m_client.events['FrameRecieved'] = self.MSTP_FrameRecieved

            comm.Start()

            time.sleep(2)

        return comm

    def CreateBacnetIpUdpClient(self, ip: str, port: int=0xBAC0) -> BACnetClient:
        comm = BACnetClient(BACnetIpUdpProtocolTransport(port, False, False, 1472, ip))
        self.m_devices[comm] = BACnetDeviceLine(comm)
        return comm

    def CreateBacnetMstpClient(self, port_name: str, baud_rate: int, source_address: int) -> BACnetClient:
        comm = BACnetClient(BACnetMstpProtocolTransport(port_name, baud_rate, source_address))
        self.m_devices[comm] = BACnetDeviceLine(comm)
        return comm

    def QueueUserWorkItem(self, target, args):
        threading.Thread(target=target, args=args, daemon=True).start()

    def SendWhoIs(self, comm: BACnetClient):
        for i in range(comm.m_retries):
            comm.WhoIs()
            time.sleep(comm.m_timeout)

    #
    def GetDeviceLine(self, comm: BACnetClient):
        return self.m_devices.get(comm, None)

    #192.168.1.184:47808/150 192.168.1.184:47808:6:2/150 1
    def ConvertBACnetAddress(self, address: str) -> Optional[BACnetAddress]:
        if len(address) > 0:
            if address.find(':') > 0 or address.find('.') > 0:
                address_info = address.split('/')[0].split(':')
                if len(address_info) == 2:
                    tmp1 = bytes(map(int, address_info[0].split('.')))  # addr[0].encode('utf-8')
                    tmp2 = int(address_info[1]).to_bytes(2, byteorder='big', signed=False)  # 从低地址向高地址进行的
                    return BACnetAddress(type=BACnetAddressTypes.IP, net=0, adr=bytearray(tmp1 + tmp2))
                elif len(address_info) == 4:  # 192.168.1.184:47808:6:2/150
                    tmp1 = bytes(map(int, address_info[0].split('.')))  # addr[0].encode('utf-8')
                    tmp2 = int(address_info[1]).to_bytes(2, byteorder='big', signed=False)  # 从低地址向高地址进行的
                    bc = BACnetAddress(type=BACnetAddressTypes.IP, net=0, adr=bytearray(tmp1 + tmp2))
                    bc.RoutedSource = BACnetAddress(type=BACnetAddressTypes.NONE, net=int(address_info[2]), adr=bytearray(BACnetTool.int_to_bytes(int(address_info[3]))))
                    return bc
            else:   # 1
                return BACnetAddress(type=BACnetAddressTypes.MSTP, net=0, adr=bytearray(BACnetTool.int_to_bytes(int(address))))
        return None

    # 读取设备点结构
    def ReadBACnetObjects(self, comm: BACnetClient, adr: BACnetAddress, device_id: int):
        (result, objects) = comm.ReadPropertyRequest(adr, BACnetObjectId(BACnetObjectTypes.OBJECT_DEVICE, device_id), BACnetPropertyIds.PROP_OBJECT_LIST)
        if result:
            return self.SortBACnetObjects(objects)
        return []

    # 读取设备点属性
    def ReadBACnetObjectProperty(self, comm: BACnetClient, adr: BACnetAddress, object_id: BACnetObjectId):
        try:
            properties = []
            properties.append(BACnetPropertyReference(BACnetPropertyIds.PROP_ALL, ASN1.BACNET_ARRAY_ALL))

            (r, multi_value_list) = comm.ReadPropertyMultipleRequest(adr, object_id, properties)  # 批量读取
            if not r:
                if self.call_back_func is not None:
                    self.call_back_func(f"Warning: Couldn't fetch properties({adr.__str__()}-{object_id.__str__()})")
            else:
                return multi_value_list
        except Exception as e:
            if self.call_back_func is not None:
                self.call_back_func(f"Warning: Couldn't perform ReadPropertyMultiple({adr.__str__()}-{object_id.__str__()})")

            try:
                (r, multi_value_list) = self.ReadAllPropertiesBySingle(comm, adr, object_id)
                if not r:
                    if self.call_back_func is not None:
                        self.call_back_func(f"Warning: Couldn't fetch properties({adr.__str__()}-{object_id.__str__()})")
                else:
                    return multi_value_list
            except Exception as e:
                if self.call_back_func is not None:
                    self.call_back_func(f"Warning: Error during read({adr.__str__()}-{object_id.__str__()}): {e.__str__()}")

        return []

    # 读取设备点值
    def ReadBACnetObjectValues(self, comm: BACnetClient, adr: Optional[Union[str, BACnetAddress]], object_list: Optional[List[BACnetObject]]):
        try:
            if isinstance(adr, str):
                adr = self.ConvertBACnetAddress(adr)
                if adr is None:
                    raise Exception(f"ERROR: Invalid adr")
            properties = []
            properties.append(BACnetPropertyReference(BACnetPropertyIds.PROP_PRESENT_VALUE, ASN1.BACNET_ARRAY_ALL))

            propToRead = []
            propToReadO = []
            for object in object_list:
                if object.IsValid() is True:
                    propToRead.append(BACnetReadAccessSpecification(object.object_id, properties))
                    propToReadO.append(object)

            (r, multi_value_list) = comm.ReadPropertyMultipleRequest1(adr, propToRead)  # 批量读取
            if not r:
                if self.call_back_func is not None:
                    self.call_back_func(f"Warning: Couldn't read values({adr.__str__()}-{[object_id.__str__() for object_id in object_list]})")
            else:
                if len(multi_value_list) == len(propToReadO):
                    for i in range(len(multi_value_list)):
                        propToReadO[i].UpdateProperty(multi_value_list[i].values)
                else:
                    if self.call_back_func is not None:
                        self.call_back_func(f'ERROR: Value size not match({adr.__str__()}-{[object_id.__str__() for object_id in object_list]})')

        except Exception as e:
            if self.call_back_func is not None:
                self.call_back_func(f"Warning: Couldn't perform ReadPropertyMultiple({adr.__str__()}-{[object_id.__str__() for object_id in object_list]})")

    # 订阅Bacnet变化事件
    def SubscribeBacnetCOV(self, comm: BACnetClient, adr: BACnetAddress, device_id: int, object_id: BACnetObjectId):
        self.m_next_subscription_id = self.m_next_subscription_id + 1
        sub_key = f"{adr.__str__()}:{device_id}:{self.m_next_subscription_id}"
        self.m_subscription_list[sub_key] = object_id

        SubscribeOK = False
        try:
            SubscribeOK = comm.SubscribeCOVRequest(adr, object_id, self.m_next_subscription_id, False, self.Subscriptions_IssueConfirmedNotifies, self.Subscriptions_Lifetime)
        except:
            if self.call_back_func is not None:
                self.call_back_func(f"ERROR: SubscribeBacnetCOV({adr.__str__()}-{object_id.__str__()}) Fail")
        return SubscribeOK

    # 读取设备点
    def ReadAllBacnet(self, comm: BACnetClient):
        if comm in self.m_devices.keys():
            device_l: BACnetDeviceLine = self.m_devices[comm]
            for (adr, device_id), device in device_l.Devices.items():
                object_list = self.ReadBACnetObjects(comm, adr, device_id)
                for object in object_list:
                    bacnet_object = BACnetObject(object)
                    device.object_list.append(bacnet_object)
                    if bacnet_object.IsValid() is True:
                        objects_properties = self.ReadBACnetObjectProperty(comm, adr, object)
                        if len(objects_properties) > 0:
                            bacnet_object.UpdateProperty(objects_properties[0].values)
            return device_l
        else:
            return None

    def SortBACnetObjects(self, RawList: list):
        SortedList = []
        for value in RawList:
            if isinstance(value.Value, BACnetObjectId):
                SortedList.append(value.Value)
            elif isinstance(value.Value, BACnetDeviceObjectReference):
                SortedList.append(value.Value.objectIdentifier)
        SortedList.sort()
        return SortedList

    def ReadProperty(self, comm: BACnetClient, adr: BACnetAddress, object_id: BACnetObjectId, property_id: BACnetPropertyIds, array_index: int=ASN1.BACNET_ARRAY_ALL):
        new_entry = BACnetPropertyValue()
        new_entry.property = BACnetPropertyReference(int(property_id), array_index)
        values = []
        try:
            (r, value) = comm.ReadPropertyRequest(adr, object_id, property_id, 0, array_index)
            if not r:
                return (False, values)
        except Exception as e:
            return (False, values)
        new_entry.value = value
        values.append(new_entry)
        return (True, values)

    def ReadAllPropertiesBySingle(self, comm: BACnetClient, adr: BACnetAddress, object_id: BACnetObjectId):
        return (False, None)

    def Close(self):
        for sender in self.m_devices.keys():
            sender.Dispose()

class BACnetEngine(threading.Thread):

    def __init__(self, address: str, identifier: int, segmentation: int, vendor_identifier: int,  read_limit: int, cmd_interval: float, timeout: int, point_dict: dict, call_back_func=None):
        threading.Thread.__init__(self)

        # param#########
        self.address = address
        self.identifier = identifier
        self.segmentation = segmentation
        self.vendor_identifier = vendor_identifier
        self.read_limit = read_limit

        self.cmd_interval = cmd_interval
        self.timeout = timeout
        self.point_dict = point_dict
        self.call_back_func = call_back_func
        ##############

        self.bacnet_client = None
        self.bacnet_comm = None
        self.bacnet_type = BACnetAddressTypes.NONE
        self.bacnet_params = {}

    def __del__(self):
        try:
            self.exit()
        except Exception as e:
            raise e

    def exit(self):
        try:
            self.bacnet_thread_exit = True
            self.close_bacnet_client()
        except Exception as e:
            raise e

    def reset(self):
        self.bacnet_client = None

    # get local bacnet ip (192.168.1.0/24)
    def _get_local_bacnet_ip(self, ip: str, net: str):
        ip_start = ip_address(str(ip_network(ip, False)).split('/')[0])
        ip_end = ip_network(ip, False).broadcast_address

        addrs = net_if_addrs().items()
        for k, v in addrs:
            for item in v:
                if item[0] == 2:
                    item_ip = item[1]
                    if ':' not in item_ip:
                        item_ip = ip_address(item_ip)
                        if ip_start <= item_ip < ip_end:
                            return '%s' % item_ip
        raise Exception(f'bacnet bind ip fail({ip})')

    # change to local ip
    def _change_bacnet_address(self, bacnet_address):
        bacnet_type = BACnetAddressTypes.NONE
        bacnet_params = {}
        if len(bacnet_address) > 0:
            address_list = bacnet_address.split(':')
            if len(address_list) >= 2:  # IP:Port
                bacnet_type = BACnetAddressTypes.IP
                [ip, port] = address_list
                ip_list = ip.split('/')
                if len(ip_list) >= 2:
                    bacnet_params.update({'ip': self._get_local_bacnet_ip(ip, ip_list[1]), 'port': int(str(port))})
                else:
                    bacnet_params.update({'ip': ip, 'port': int(str(port))})
            else:  # COM/Baud
                bacnet_type = BACnetAddressTypes.MSTP
                ip = address_list[0]
                ip_list = ip.split('/')
                if len(ip_list) >= 2:
                    bacnet_params.update({'port_name': ip_list[0], 'baud_rate': int(str(ip_list[1]))})
        return bacnet_type, bacnet_params

    def get_bacnet_client(self):
        try:
            if self.bacnet_client == None:
                bacnet_type, bacnet_params = self._change_bacnet_address(self.address)
                bacnet_params.update({'source_address': self.identifier, 'retries': 2, 'timeout': self.timeout, 'whois': False, 'proposed_window_size': self.read_limit, 'segments_max': self.segmentation, 'identifier': self.identifier, 'vendor_identifier': self.vendor_identifier})
                if bacnet_type == BACnetAddressTypes.IP or bacnet_type == BACnetAddressTypes.MSTP:
                    self.bacnet_client = BACnetApplication()
                    self.bacnet_comm = self.bacnet_client.AddBACnetClient(bacnet_type, **bacnet_params)
        except Exception as e:
            raise e
        return self.bacnet_client

    def close_bacnet_client(self):
        try:
            if self.bacnet_client:
                self.bacnet_client.Close()
        except Exception as e:
            print('ERROR: close_bacnet_client(%s)' % e.__str__())
        self.reset()

    def run(self):
        pass

    #
    def read_values(self, dict_point):
        result_dict = {}
        try:
            if self.get_bacnet_client():
                bacnet_point_dict = {}
                for point_name in dict_point.keys():
                    read_key = dict_point[point_name].get_point_device_address
                    if read_key not in bacnet_point_dict.keys():
                        bacnet_point_dict[read_key] = []
                    bacnet_point_dict[read_key].append(dict_point[point_name])
                for read_key in bacnet_point_dict.keys():

                    points = []
                    for point in bacnet_point_dict[read_key]:
                        points.append(BACnetObject(BACnetObjectId(point.get_point_type, point.get_point_address), point.get_point_name))

                    self.bacnet_client.ReadBACnetObjectValues(self.bacnet_comm, read_key, points)

                    for point in points:
                        if point.present_value is not None:
                            result_dict[point.object_name.__str__()] = point.present_value
                #result_dict.update(self._read_bacnet_values_v1(bacnet_point_dict))
        except Exception as e:
            raise e
        return result_dict

    def scrap_value(self):
        result_dict = {}
        try:
            if self.get_bacnet_client():
                if self.add_group is False:
                    result_dict = self.read_values(self.point_dict)
                    self.add_group = True
                else:
                    values = self.bacnet_client.read()

        except Exception as e:
            raise e
        return result_dict

    def write_values(self, dict_point):
        result_dict = {}
        try:
            if self.get_bacnet_client():
                set_values = []
                point_tags: dict = {}
                for point_name, point_value in dict_point.items():
                    if point_name in self.point_dict.keys():
                        point_tag = self.point_dict[point_name].get_point_tag
                        if point_tag not in point_tags.keys():
                            point_tags[point_tag] = []
                        if point_name not in point_tags[point_tag]:
                            point_tags[point_tag].append(point_name)
                        set_values.append((point_tag, point_value))

                if len(set_values) > 0:
                    values = self.bacnet_client.write(set_values)
                    for tag, status in values:
                        names = point_tags.get(tag, [])
                        for name in names:
                            if status == 'Success':
                                result_dict[name] = True
        except Exception as e:
            raise e
        return result_dict

    # search###
    def _format_name_v1(self, device_address: str, device_id: str, type: str, address: str):
        return '%s_%s_%s_%s' % (device_address.replace('.','_').replace(':','_'), device_id, type, address)

    def search_points(self, call_back_func):
        search_point = {}
        try:
            if self.get_bacnet_client():
                self.bacnet_client.SetHandleLog(call_back_func)
                self.bacnet_client.SendWhoIs(self.bacnet_comm)
                device_l = self.bacnet_client.ReadAllBacnet(self.bacnet_comm)
                for (adr, device_id), device in device_l.Devices.items():
                    for object in device.object_list:
                        address = adr.__str__()
                        bacnet_point_name = self._format_name_v1(str(address), str(device_id), str(object.object_id.type), str(object.object_id.instance))
                        description = object.description.__str__() if not isinstance(object.description, BACnetError) else ''
                        present_value = object.present_value.__str__() if not isinstance(object.present_value, BACnetError) else ''
                        object_name = object.object_name.__str__() if not isinstance(object.object_name, BACnetError) else ''

                        bacnet_point_property = {'point_name': bacnet_point_name,
                                                 'index': 0, 'point_writable': 'True',
                                                 'device_address': '%s/%s' % (str(address), str(device_id)),
                                                 'point_type': str(object.object_id.type),
                                                 'point_property': 'presentValue',
                                                 'point_address': str(object.object_id.instance), 'description': description, 'present_value': present_value, 'object_name': object_name}

                        search_point[bacnet_point_name] = bacnet_point_property
        except Exception as e:
            raise e
        return search_point

    def ping_target(self):
        if self.get_bacnet_client():
            return self.bacnet_client.ping()
        return False

# BacnetPoint
class BacnetPoint(BasePoint):

    # point_writable, point_name, point_device_address, point_type, point_property, point_address, point_description
    def __init__(self, point_writable: bool, point_name: str, point_device_address: str, point_type: str,
                 point_property: str, point_address: int, point_description: str = ''):
        super().__init__('bacnet', point_writable, point_name, point_description)

        self.point_device_address = point_device_address
        self.point_type = point_type
        self.point_property = point_property
        self.point_address = point_address

    # get point_device_address
    @property
    def get_point_device_address(self):
        return self.point_device_address

    # get point_type
    @property
    def get_point_type(self):
        return int(self.point_type)

    # get point_property
    @property
    def get_point_property(self):
        return self.point_property

    # get point_address
    @property
    def get_point_address(self):
        return self.point_address

#Bacnet  Driver
class BacnetDriver(BaseDriver):

    def __init__(self, dict_config: dict, dict_point: PointTableDict):
        super().__init__(dict_config, dict_point)

        ##########
        self.bacnet_application = None
        self.bacnet_multi_read = 20         # bacnet Number of batches read
        self.bacnet_cmd_interval = 0.3      # bacnet cmd interval
        self.bacnet_time_out = 5            #timeout
        self.enable = True
        self.bacnet_busy = False

        ##server#######
        self.bacnet_server_ip = ''                          #bacnet server ip(Bind network card)
        self.bacnet_server_identifier = '555'              #bacnet server identifier
        self.bacnet_server_name = 'BacnetDriver'      #bacnet server name
        self.bacnet_server_segmentation = 65  #bacnet server segmentation
        self.bacnet_server_vendor_identifier = 15           #bacnet server vendor
        self.bacnet_server_apdu_timeout = 5000              #bacnet server timeout
        self.bacnet_server_apdu_retties = 0                 #bacnet server apdu retries
        self.bacnet_support_property = ['analogInput', 'analogOutput', 'analogValue', 'binaryInput', 'binaryOutput',
                                        'binaryValue', 'multiStateInput', 'multiStateOutput', 'multiStateValue']

        ##########

        self.bacnet_device_list = []
        self.dict_bacnet_point = {}
        self.callback_fun = None

        self.configure()

    # exit
    def __del__(self):
        super().__del__()

        self.close()

    #close
    def close(self):
        if self.bacnet_application:
            self.bacnet_application.exit()
            del self.bacnet_application
        self.bacnet_application = None

    #Parse configuration
    def configure(self):

        #config
        self.bacnet_server_ip = str(self.dict_config.get("address", ""))  # bacnet server ip(Bind network card)
        self.bacnet_server_identifier = int(str(self.dict_config.get("identifier", "555")))  # bacnet server identifier
        self.bacnet_server_name = str(self.dict_config.get("name", 'BacnetDriver'))  # bacnet server name
        self.bacnet_server_segmentation = int(str(self.dict_config.get("segmentation", 65)))  # bacnet server segmentation
        self.bacnet_server_vendor_identifier = int(str(self.dict_config.get("vendor_identifier", "15")))  # bacnet server vendor
        self.bacnet_multi_read = int(str(self.dict_config.get("multi_read", "20")))
        self.bacnet_cmd_interval = float(str(self.dict_config.get("cmd_interval", "0.3")))
        self.bacnet_time_out = int(str(self.dict_config.get("timeout", "5")))  # timeout
        self.enable = str(self.dict_config.get("enabled", "true")).lower() == 'true'  # enable

        #csv
        for point_name in self.dict_point.keys():
            self.dict_bacnet_point[point_name] = BacnetPoint(bool(str(self.dict_point[point_name]['point_writable'])),str(self.dict_point[point_name]['point_name']),str(self.dict_point[point_name]['device_address']) , str(self.dict_point[point_name]['point_type']),str(self.dict_point[point_name]['point_property']),int(str(self.dict_point[point_name]['point_address'])),str(self.dict_point[point_name]['description']))
        
        # create_engine
        close_delay = 0
        if self.bacnet_application != None:
            self.bacnet_application.exit()
            del self.bacnet_application
            self.bacnet_application = None
            close_delay = 2

        self.bacnet_application = BACnetEngine(self.bacnet_server_ip, self.bacnet_server_identifier, self.bacnet_server_segmentation, self.bacnet_server_vendor_identifier, self.bacnet_multi_read, self.bacnet_cmd_interval, self.bacnet_time_out, self.dict_bacnet_point)
        self.bacnet_application.start()

        time.sleep(close_delay)  # delay on closed socket(This prevents a fast reconnect by the client)
        
    # ping server
    def ping_target(self):
        return True

    def get_points(self, dict_point: dict = {}) -> dict:
        result_dict = {}
        if self.bacnet_busy is True:
            return result_dict
        else:
            self.bacnet_busy = True
            try:
                if len(dict_point) == 0:
                    result_dict = self._scrap_points()
                else:
                    result_dict = self._get_points(dict_point)
            finally:
                self.bacnet_busy = False
        return result_dict

    def _get_points(self, dict_point: dict) -> dict:
        result_dict = {}
        try:
            dict_bacnet_point = {}
            for point_name in dict_point.keys():
                if point_name in self.dict_bacnet_point.keys():
                    dict_bacnet_point[point_name] = self.dict_bacnet_point[point_name]

            if len(dict_bacnet_point) > 0 and self.bacnet_application:
                result_dict = self.bacnet_application.read_values(dict_bacnet_point)
        except Exception as e:
            raise e
        return result_dict

    #scrap
    def _scrap_points(self) -> dict:
        result_dict = {}
        try:
            if len(self.dict_bacnet_point) > 0 and self.bacnet_application:
                result_dict = self.bacnet_application.read_values(self.dict_bacnet_point)
        except Exception as e:
            raise e
        return result_dict

    # set
    def set_points(self, dict_point: dict) -> dict:
        set_result_dict = {}
        if self.bacnet_busy is True:
            return set_result_dict
        else:
            self.bacnet_busy = True
            try:
                dict_bacnet_point = {}
                for point_name in dict_point.keys():
                    if point_name in self.dict_bacnet_point.keys():
                        dict_bacnet_point[point_name] = dict_point[point_name]

                if len(dict_bacnet_point) > 0 and self.bacnet_application:
                    set_result_dict = self.bacnet_application.write_values(dict_bacnet_point)              
            finally:
                self.bacnet_busy = False
        return set_result_dict

    # reset config
    def reset_config(self, dict_config: dict):
        super().reset_config(dict_config)
        self.configure()

    # reset point
    def reset_point(self, dict_point: dict):
        super().reset_point(dict_point)
        self.configure()

    # search
    def search_points(self, call_back_func=None) -> dict:
        search_point = {}
        self.callback_fun = call_back_func
        
        return self.bacnet_application.search_points(call_back_func)