"""wfp_market table and view."""

from sqlalchemy import Float, ForeignKey, String, select
from sqlalchemy.orm import Mapped, mapped_column, relationship

from hapi_schema.db_admin1 import DBAdmin1
from hapi_schema.db_admin2 import DBAdmin2
from hapi_schema.db_location import DBLocation
from hapi_schema.utils.base import Base
from hapi_schema.utils.constraints import latlon_constraint
from hapi_schema.utils.view_params import ViewParams
from hapi_schema.views import get_admin2_case


class DBWFPMarket(Base):
    __tablename__ = "wfp_market"
    __table_args__ = (latlon_constraint(),)

    code: Mapped[str] = mapped_column(String(32), primary_key=True)
    admin2_ref: Mapped[int] = mapped_column(
        ForeignKey("admin2.id", onupdate="CASCADE"),
        nullable=False,
    )
    provider_admin1_name: Mapped[str] = mapped_column(
        String(512), nullable=False, index=True
    )
    provider_admin2_name: Mapped[str] = mapped_column(
        String(512), nullable=False, index=True
    )
    name: Mapped[str] = mapped_column(String(512), nullable=False, index=True)
    lat: Mapped[float] = mapped_column(Float, nullable=True, index=True)
    lon: Mapped[float] = mapped_column(Float, nullable=True, index=True)

    admin2 = relationship(DBAdmin2)


view_params_wfp_market = ViewParams(
    name="wfp_market_view",
    metadata=Base.metadata,
    selectable=select(
        *DBWFPMarket.__table__.columns,
        DBLocation.code.label("location_code"),
        DBLocation.name.label("location_name"),
        DBLocation.has_hrp.label("has_hrp"),
        DBLocation.in_gho.label("in_gho"),
        DBAdmin1.code.label("admin1_code"),
        DBAdmin1.name.label("admin1_name"),
        DBAdmin1.is_unspecified.label("admin1_is_unspecified"),
        DBAdmin1.location_ref.label("location_ref"),
        DBAdmin2.code.label("admin2_code"),
        DBAdmin2.name.label("admin2_name"),
        DBAdmin2.is_unspecified.label("admin2_is_unspecified"),
        DBAdmin2.admin1_ref.label("admin1_ref"),
        get_admin2_case(DBWFPMarket),
    ).select_from(
        # Join market to admin2 to admin1 to loc
        DBWFPMarket.__table__.join(
            DBAdmin2.__table__,
            DBWFPMarket.admin2_ref == DBAdmin2.id,
            isouter=True,
        )
        .join(
            DBAdmin1.__table__,
            DBAdmin2.admin1_ref == DBAdmin1.id,
            isouter=True,
        )
        .join(
            DBLocation.__table__,
            DBAdmin1.location_ref == DBLocation.id,
            isouter=True,
        )
    ),
)
