import hashlib
import http
import secrets
from typing import Self

import msgspec
import sqlparse
from django.conf import settings
from django.contrib.contenttypes.models import ContentType
from django.core.management import call_command
from django.core.validators import RegexValidator
from django.db import connections, models
from django.db.models.query import QuerySet
from django.urls import reverse
from django.utils import timezone

from .. import events
from .router import activated_site

DJANGO_MIGRATIONS_SQL = """
    CREATE TABLE IF NOT EXISTS {}.django_migrations (
        id bigint GENERATED BY DEFAULT AS IDENTITY,
        app varchar(255) NOT NULL,
        name varchar(255) NOT NULL,
        applied timestamptz NOT NULL,
        PRIMARY KEY (id)
    )
"""


class Site(models.Model):
    name = models.CharField(max_length=200)
    slug = models.SlugField(max_length=200, unique=True)
    schema_name = models.CharField(
        max_length=63,
        unique=True,
        validators=[
            RegexValidator(r"[a-z][a-z0-9_]+"),
        ],
    )
    date_created = models.DateTimeField(auto_now_add=True)

    # Not exactly QuerySets, but close enough.
    members: QuerySet["SiteMember"]
    nodes: QuerySet["Node"]
    keys: QuerySet["SiteKey"]
    logs: QuerySet["Log"]
    errors: QuerySet["Error"]
    metrics: QuerySet["Metric"]
    requests: QuerySet["Request"]

    @property
    def queries(self):
        # The generated related_name is `querys` -- yuck.
        return Query.objects.filter(site=self)

    class Meta:
        db_table = "site"

    def __str__(self):
        return self.name

    def get_absolute_url(self):
        return reverse("site-overview", kwargs={"slug": self.slug})

    def activated(self):
        return activated_site(self)

    def ensure_schema(self):
        if (
            not settings.VARANUS_USE_SCHEMAS
            or connections[settings.VARANUS_DB_ALIAS].vendor != "postgresql"
        ):
            return
        with connections[settings.VARANUS_DB_ALIAS].cursor() as cursor:
            cursor.execute(f"CREATE SCHEMA IF NOT EXISTS {self.schema_name}")
            cursor.execute(DJANGO_MIGRATIONS_SQL.format(self.schema_name))
        with activated_site(self):
            call_command("migrate", database=settings.VARANUS_DB_ALIAS, verbosity=0)

    def save(self, **kwargs):
        super().save(**kwargs)
        self.ensure_schema()


class SiteMember(models.Model):
    site = models.ForeignKey(Site, on_delete=models.CASCADE, related_name="members")
    user = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name="memberships",
    )
    is_admin = models.BooleanField(default=False)
    date_created = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = "site_member"
        constraints = [
            models.UniqueConstraint("site", "user", name="unique_site_user"),
        ]

    def __str__(self):
        return f"{self.user} ({self.site})"


class SiteKey(models.Model):
    site = models.ForeignKey(Site, on_delete=models.CASCADE, related_name="keys")
    access_key = models.CharField(
        max_length=100,
        unique=True,
        default=secrets.token_urlsafe,
    )
    active = models.BooleanField(default=True)
    date_created = models.DateTimeField(auto_now_add=True)
    date_expires = models.DateField()

    class Meta:
        db_table = "site_key"

    def __str__(self):
        return f"{self.access_key} ({self.site})"

    def get_ingest_url(self, request):
        return "{scheme}://{key}@{host}".format(
            scheme=request.scheme,
            key=self.access_key,
            host=request.get_host(),
        )


class EnvironmentModel(models.Model):
    site = models.ForeignKey(
        Site,
        on_delete=models.DO_NOTHING,
        related_name="%(class)ss",
        db_constraint=False,
    )
    environment = models.CharField(max_length=200, blank=True)

    class Meta:
        abstract = True


class Node(EnvironmentModel):
    name = models.CharField(max_length=200)
    platform = models.CharField(max_length=200)
    python_version = models.CharField(max_length=100)
    first_seen = models.DateTimeField(auto_now_add=True)
    last_seen = models.DateTimeField(auto_now=True)

    packages: QuerySet["NodePackage"]
    updates: QuerySet["NodeUpdate"]

    class Meta:
        db_table = "node"
        constraints = [
            models.UniqueConstraint(
                "site", "environment", "name", name="unique_site_node"
            ),
        ]

    def __str__(self):
        return f"{self.environment}@{self.name}"

    def get_absolute_url(self):
        return reverse(
            "node-details",
            kwargs={
                "slug": self.site.slug,
                "pk": self.pk,
            },
        )

    @classmethod
    def update(cls, info: events.NodeInfo, site: Site, environment: str = ""):
        # Update or create the Node instance.
        node, created = site.nodes.update_or_create(
            name=info.name,
            environment=environment,
            defaults={
                "platform": info.platform,
                "python_version": info.python_version,
            },
        )

        # Now figure out which packages were installed, updated, or removed.
        existing: dict[str, NodePackage] = {p.package: p for p in node.packages.all()}
        new = [
            NodePackage(node=node, package=p, version=v)
            for p, v in info.packages.items()
            if p not in existing
        ]
        changed = [
            p.set_version(info.packages[name])
            for name, p in existing.items()
            if name in info.packages and info.packages[name] != p.version
        ]
        deleted = {p for p in existing if p not in info.packages}

        if new:
            node.packages.bulk_create(new)
        if changed:
            node.packages.bulk_update(changed, ["version"])
        if deleted:
            node.packages.filter(package__in=deleted).delete()

        node_update = None
        if new or changed or deleted:
            node_update = node.updates.create(
                installed={p.package: p.version for p in new},
                updated={p.package: [p.old_version, p.version] for p in changed},
                removed=list(deleted),
            )

        return node, created, node_update


class NodePackage(models.Model):
    node = models.ForeignKey(Node, on_delete=models.CASCADE, related_name="packages")
    package = models.CharField(max_length=200)
    version = models.CharField(max_length=50)

    old_version: str

    class Meta:
        db_table = "node_package"
        ordering = ["package"]
        constraints = [
            models.UniqueConstraint("node", "package", name="unique_node_package"),
        ]

    def __str__(self):
        return f"{self.package} ({self.version})"

    def set_version(self, new_version: str) -> Self:
        setattr(self, "old_version", self.version)
        self.version = new_version
        return self


class NodeUpdate(models.Model):
    node = models.ForeignKey(Node, on_delete=models.CASCADE, related_name="updates")
    installed = models.JSONField(default=dict)
    updated = models.JSONField(default=dict)
    removed = models.JSONField(default=list)
    timestamp = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = "node_update"
        ordering = ["-timestamp"]

    def __str__(self):
        return f"{self.node} - {self.timestamp}"

    def to_dict(self):
        return {
            "timestamp": self.timestamp.isoformat(),
            "installed": self.installed,
            "updated": self.updated,
            "removed": self.removed,
        }


class EventModel(EnvironmentModel):
    event_id = models.UUIDField(db_index=True)
    timestamp = models.DateTimeField(default=timezone.now)
    tags = models.JSONField(default=dict, blank=True)
    node = models.ForeignKey(
        Node,
        on_delete=models.SET_NULL,
        related_name="%(class)ss",
        null=True,
        blank=True,
    )

    class Meta:
        abstract = True
        ordering = ["-timestamp"]

    @classmethod
    def from_event(cls, event: events.Event, **extra):
        raise NotImplementedError()

    def get_absolute_url(self):
        ct = ContentType.objects.get_for_model(self)
        return reverse(
            "event-details",
            kwargs={
                "slug": self.site.slug,
                "event_id": self.event_id,
                "model": ct.model,
                "pk": self.pk,
            },
        )


class FingerprintModel(models.Model):
    fingerprint = models.CharField(max_length=64, db_index=True)

    class Meta:
        abstract = True

    def fingerprint_parts(self):
        raise NotImplementedError()

    def save(self, **kwargs):
        if not self.fingerprint:
            self.fingerprint = hashlib.sha256(
                ":".join(
                    str(p).strip().lower() for p in self.fingerprint_parts() if p
                ).encode("utf-8")
            ).hexdigest()
        super().save(**kwargs)


class Request(EventModel):
    host = models.CharField(max_length=253)
    path = models.TextField()
    method = models.CharField(max_length=20)
    status = models.IntegerField()
    headers = models.JSONField(default=dict, blank=True)
    size = models.IntegerField(null=True, blank=True)
    ip = models.GenericIPAddressField(null=True, blank=True)
    user = models.CharField(max_length=200, blank=True)

    class Meta:
        db_table = "request"
        ordering = ["-timestamp"]

    @classmethod
    def from_event(cls, event: events.Request, **extra):
        return cls.objects.create(
            timestamp=event.timestamp,
            tags=event.tags,
            host=event.host,
            path=event.path,
            method=event.method,
            status=event.status,
            size=event.size,
            ip=event.ip,
            user=event.user or "",
            **extra,
        )

    def __str__(self):
        return f"{self.method} {self.host}{self.path}"

    @property
    def http_status(self):
        return http.HTTPStatus(self.status)


class Context(EventModel):
    name = models.CharField(max_length=200)
    elapsed_ms = models.IntegerField()
    request = models.OneToOneField(
        Request,
        on_delete=models.CASCADE,
        related_name="context",
        null=True,
        blank=True,
    )
    parent = models.ForeignKey(
        "self",
        on_delete=models.SET_NULL,
        related_name="subcontexts",
        null=True,
        blank=True,
    )

    logs: QuerySet["Log"]
    errors: QuerySet["Error"]
    metrics: QuerySet["Metric"]

    @property
    def queries(self):
        # The generated related_name is `querys` -- yuck.
        return Query.objects.filter(context=self)

    class Meta:
        db_table = "context"
        ordering = ["-timestamp"]

    def __str__(self):
        return self.name

    @classmethod
    def from_event(cls, event: events.Context, **extra):
        context = cls.objects.create(
            timestamp=event.timestamp,
            tags=event.tags,
            name=event.name,
            elapsed_ms=event.elapsed_ms,
            request=(
                Request.from_event(event.request, **extra) if event.request else None
            ),
            **extra,
        )
        for err in event.errors:
            Error.from_event(err, context=context, **extra)
        for log in event.logs:
            Log.from_event(log, context=context, **extra)
        for q in event.queries:
            Query.from_event(q, context=context, **extra)
        for m in event.metrics:
            Metric.from_event(m, context=context, **extra)
        for ctx in event.subcontexts:
            Context.from_event(
                ctx,
                parent=context,
                **extra,
            )
        return context


class ContextualModel(EventModel):
    context = models.ForeignKey(
        Context,
        on_delete=models.CASCADE,
        related_name="%(class)ss",
    )

    class Meta:
        abstract = True


class Error(ContextualModel, FingerprintModel):
    kind = models.CharField(max_length=200)
    module = models.CharField(max_length=200)
    message = models.TextField()
    lines = models.JSONField(default=list, blank=True)

    class Meta:
        db_table = "error"
        ordering = ["-timestamp"]

    @classmethod
    def from_event(cls, event: events.Error, **extra):
        return cls.objects.create(
            timestamp=event.timestamp,
            tags=event.tags,
            kind=event.kind,
            module=event.module,
            message=event.message,
            lines=[msgspec.structs.asdict(line) for line in event.lines],
            **extra,
        )

    def __str__(self):
        return f"{self.kind}({self.message})"

    def fingerprint_parts(self):
        yield self.kind
        yield self.message
        for line in self.lines:
            yield line["module"]
            yield line["function"]


class Log(ContextualModel, FingerprintModel):
    class Level(models.IntegerChoices):
        CRITICAL = 50, "CRITICAL"
        ERROR = 40, "ERROR"
        WARNING = 30, "WARNING"
        INFO = 20, "INFO"
        DEBUG = 10, "DEBUG"
        NOTSET = 0, "NOTSET"

    message = models.TextField()
    name = models.CharField(max_length=200, blank=True)
    level = models.IntegerField(null=True, blank=True, choices=Level)
    file = models.TextField(blank=True)
    lineno = models.IntegerField(null=True, blank=True)
    error = models.ForeignKey(
        Error,
        on_delete=models.SET_NULL,
        related_name="logs",
        blank=True,
        null=True,
    )

    class Meta:
        db_table = "log"
        ordering = ["-timestamp"]

    @classmethod
    def from_event(cls, event: events.Log, **extra):
        return cls.objects.create(
            timestamp=event.timestamp,
            tags=event.tags,
            message=event.message,
            name=event.name or "",
            level=event.level,
            file=event.file or "",
            lineno=event.lineno,
            error=Error.from_event(event.error, **extra) if event.error else None,
            **extra,
        )

    def fingerprint_parts(self):
        return (self.message, self.name, self.level, self.file, self.lineno)


class Metric(ContextualModel):
    name = models.CharField(max_length=100)
    agg_count = models.IntegerField(default=0)
    agg_sum = models.FloatField(default=0.0)
    agg_avg = models.FloatField(default=0.0)
    agg_min = models.FloatField(default=0.0)
    agg_max = models.FloatField(default=0.0)

    class Meta:
        db_table = "metric"
        ordering = ["-timestamp"]

    @classmethod
    def from_event(cls, event: events.Metric, **extra):
        return cls.objects.create(
            timestamp=event.timestamp,
            tags=event.tags,
            name=event.name,
            agg_count=event.agg_count,
            agg_sum=event.agg_sum,
            agg_avg=event.agg_avg,
            agg_min=event.agg_min,
            agg_max=event.agg_max,
            **extra,
        )


class Query(ContextualModel, FingerprintModel):
    sql = models.TextField()
    params = models.JSONField(default=list)
    db = models.CharField(max_length=200, default="default")
    elapsed_ms = models.IntegerField()
    success = models.BooleanField(default=True)
    command = models.CharField(max_length=20, blank=True)

    class Meta:
        db_table = "query"
        verbose_name_plural = "queries"
        ordering = ["-timestamp"]

    @classmethod
    def from_event(cls, event: events.Query, **extra):
        return cls.objects.create(
            timestamp=event.timestamp,
            tags=event.tags,
            sql=event.sql,
            params=event.params,
            db=event.db,
            elapsed_ms=event.elapsed_ms,
            success=event.success,
            **extra,
        )

    @property
    def sql_summary(self):
        stripped = " ".join(self.sql.split())
        if len(stripped) <= 80:
            return stripped
        return stripped[:79] + "…"

    @property
    def sql_formatted(self):
        return sqlparse.format(self.sql, reindent=True, keyword_case="upper")

    def fingerprint_parts(self):
        return self.sql.split()

    def save(self, **kwargs):
        if self.sql and not self.command:
            stmt = sqlparse.parse(self.sql)[0]
            self.command = stmt.get_type()
        super().save(**kwargs)
