import functools
import operator
from typing import Any, ClassVar

from django.db.models import Count, Q, QuerySet

from .base import SearchField, StringValues


class Choice:
    id: str
    value: str
    label: str
    count: int

    def __init__(self, field: SearchField, value: Any, count: int):
        self.value = self.stringify(value)
        if callable(field.choice_label):
            self.label = field.choice_label(value)
        else:
            self.label = self.value or field.empty_label
        self.id = field.id + "_" + self.value.replace(" ", "")
        self.count = count

    @classmethod
    def stringify(cls, value):
        return str(value) if value is not None else ""


class Facet(SearchField):
    choice_class: ClassVar[type[Choice]] = Choice

    def get_choices(self, queryset):
        choice_qs = queryset.values_list(self.field_name).annotate(num=Count("*"))
        match self.order:
            case "value":
                choice_qs = choice_qs.order_by(self.field_name)
            case "count":
                choice_qs = choice_qs.order_by("-num")
        return [self.choice_class(self, c, num) for c, num in choice_qs]


class MultiFacet(Facet):
    template_name = "search/multifacet.html"

    def apply(self, queryset: QuerySet, field_data: StringValues) -> QuerySet:
        filters = []
        if selected := set(field_data.get("s", [])):
            if "" in selected:
                selected.discard("")
                filters.append(Q(**{f"{self.field_name}__isnull": True}))
            if selected:
                filters.append(Q(**{f"{self.field_name}__in": selected}))
        if filters:
            queryset = queryset.filter(functools.reduce(operator.or_, filters))
        return queryset

    def get_context(self, queryset: QuerySet, field_data: StringValues) -> dict:
        return {
            **super().get_context(queryset, field_data),
            "choices": self.get_choices(queryset),
            "selected": field_data.get("s", []),
        }
