from typing import ClassVar

from django.contrib.contenttypes.models import ContentType
from django.db.models import Count, OuterRef, QuerySet, Subquery
from django.shortcuts import get_object_or_404
from django.utils.functional import Promise

from ..models import Context, Log
from ..search import DateRange, MultiFacet, Search
from .base import SiteView


def level_name(level) -> str | Promise:
    try:
        return Log.Level(level).label
    except ValueError:
        return str(level)


class Overview(SiteView):
    template_name = "site/overview.html"

    def get_context(self):
        nodes = self.site.nodes.annotate(
            last_event=Subquery(
                Context.objects.filter(node_id=OuterRef("id"))
                .order_by("-timestamp")
                .values("timestamp")[:1]
            ),
            num_packages=Count("packages"),
        )
        return {
            "host": self.request.get_host(),
            "scheme": self.request.scheme,
            "nodes": nodes,
        }


class SearchView(SiteView):
    search_class: ClassVar[type[Search]]

    def get_queryset(self) -> QuerySet:
        raise NotImplementedError()

    def get_context(self):
        search = self.search_class(self.get_queryset(), self.request.GET)
        return {
            "search": search,
            "results": search.queryset()[:50],
        }


class Logs(SearchView):
    template_name = "site/logs.html"

    class search_class(Search):
        timeframe = DateRange(field_name="timestamp")
        name = MultiFacet()
        level = MultiFacet(choice_label=level_name)
        environment = MultiFacet()
        node__name = MultiFacet()

    def get_queryset(self) -> QuerySet:
        return self.site.logs.select_related("context")


class Errors(SearchView):
    template_name = "site/errors.html"

    class search_class(Search):
        timeframe = DateRange(field_name="timestamp")
        kind = MultiFacet()
        module = MultiFacet()
        environment = MultiFacet()
        node__name = MultiFacet()

    def get_queryset(self) -> QuerySet:
        return self.site.errors.select_related("context")


class Requests(SearchView):
    template_name = "site/requests.html"

    class search_class(Search):
        timeframe = DateRange(field_name="timestamp")
        status = MultiFacet()
        environment = MultiFacet()
        node__name = MultiFacet()

    def get_queryset(self) -> QuerySet:
        return self.site.requests.select_related("context")


class Queries(SearchView):
    template_name = "site/queries.html"

    class search_class(Search):
        timeframe = DateRange(field_name="timestamp")
        type = MultiFacet(field_name="command")
        db = MultiFacet()
        environment = MultiFacet()
        node__name = MultiFacet()

    def get_queryset(self) -> QuerySet:
        return self.site.queries.select_related("context")


class Metrics(SearchView):
    template_name = "site/metrics.html"

    class search_class(Search):
        timeframe = DateRange(field_name="timestamp")
        name = MultiFacet()
        environment = MultiFacet()
        node__name = MultiFacet()

    def get_queryset(self) -> QuerySet:
        return self.site.metrics.select_related("context")


class Details(SiteView):
    @property
    def template_name(self):
        model = self.kwargs["model"]
        return f"site/details/{model}.html"

    def get_context(self):
        ct = ContentType.objects.get_by_natural_key("varanus", self.kwargs["model"])
        obj = ct.get_object_for_this_type(pk=self.kwargs["pk"])
        return {
            "object": obj,
            "context": obj.context,
        }


class NodeDetails(SiteView):
    template_name = "site/details/node.html"

    def get_context(self):
        node = get_object_or_404(self.site.nodes, pk=self.kwargs["pk"])
        return {
            "node": node,
        }


class NodeEnvironments(SiteView):
    template_name = "site/details/node_environments.html"

    def get_context(self):
        return {
            "name": self.kwargs["name"],
            "nodes": self.site.nodes.filter(name__iexact=self.kwargs["name"]),
        }


class EnvironmentNodes(SiteView):
    template_name = "site/details/environment_nodes.html"

    def get_context(self):
        return {
            "environment": self.kwargs["environment"],
            "nodes": self.site.nodes.filter(
                environment__iexact=self.kwargs["environment"]
            ),
        }
