# PyDepM — Python Dependency Manager

[![Python Version](https://img.shields.io/pypi/pyversions/pydm.svg)](https://pypi.org/project/pydm/)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)

PyDepM is a modern Python project manager that simplifies dependency management, script execution, and project construction with an intuitive command-line interface.

## Key Features

- 🚀 Easy project initialization with `pydep init`
- 📦 Dependency management with `pypackage.json`
- 📝 Project configuration with `pyproject.toml` (PEP 621) for module-type projects
- 📦 Build standalone executables with PyInstaller using `pydep build` for application projects and `pydep build` generates pyproject.toml and build with python -m build for module projects
- 🔧 Run scripts with `pydep run`.
- 🌍 Global mode for system-wide tool installation
- ⚡ Fast and efficient dependency resolution
- 🔍 Security audit for dependencies
- 📊 Outdated dependencies visualization
- 🎛️ Advanced `pyproject.toml` customization for module projects


## Requirements

- Python 3.9 or higher
- pip (included with Python)

## Installation

```bash
pip install pydepm
```

## Quick Start Guide

### Create a New Project

```bash
# Initialize a new application project
pydep init my-app --type app

# Or initialize a new module project
pydep init my-module --type module

# Navigate to the project directory
cd my-app
```

### Advanced pyproject.toml Configuration (Module Projects Only)

For module-type projects, you can customize the generated `pyproject.toml` in two ways:

1. **Structured Format**: Define sections using JSON keys in `[section]` format
2. **Raw Format**: Provide direct TOML content in the `_raw` field

Example `pypackage.json` with custom pyproject configuration:

```json
{
  "type": "module",
  "name": "my-module",
  "version": "0.1.0",
  "description": "This is a simple module autogenerated with PyDM.",
  "authors": [
    {
      "name": "PyDM User"
    }
  ],
  "license": "MIT",
  "dependencies": {},
  "scripts": {
    "start": "python -m my-module"
  },
  "cli": {
    "my-module": {
      "name": "my-module",
      "target": "my-module.main:main"
    }
  },
  "pyproject": {
    "[project.scripts]": {
      "my-module": "my-module.main:main"
    },
    "[tool.black]": {
      "line-length": 100,
      "target-version": ["py39"]
    },
    "[tool.mypy]": {
      "python_version": "3.9",
      "strict": true
    },
    "_raw": "# Direct TOML content here\n[tool.poetry]\nname = \"my-package\"\nversion = \"0.1.0\"\ndescription = \"My package description\"\n"
  }
}
```

When you run `pydep convert --to toml`, this will generate a `pyproject.toml` file with all the custom configurations merged together.

### Manage Dependencies

```bash
# Add dependencies
pydep add requests pytest

# Add development dependencies
pydep add --dev black flake8

# Install dependencies
pydep install

# Update dependencies
pydep update

# Remove dependencies
pydep remove nombre-del-paquete

# Build
pydep build
```

### Ejecutar scripts

Add scripts to `pypackage.json`:
```json
{
  "scripts": {
    "iniciar": "python main.py",
    "probar": "pytest",
    "formatear": "black ."
  }
}
```

Run scripts with:
```bash
pydep run iniciar
pydep run probar
pydep run formatear
```

### Build standalone executables (only for application projects)

Add the compilation configuration to `pypackage.json`:
```json
{
  "executable": {
    "target": "main.py",
    "parameters": ["--onefile"],
    "output": "dist/ejecutable"
  }
}
```

Build the executable:
```bash
pydep build
```

## Project structure

### Application projects (`type: "app"` in pypackage.json)
- Simple project structure
- Focus on creating standalone executables
- Without `pyproject.toml` by default (can be enabled with `"pyprojectUse": true`)
- Ideal for final user applications

### Module projects (`type: "module"` in pypackage.json)
- Python package structure
- Includes `pyproject.toml` for package metadata
- Supports entry points and console scripts
- Ideal for reusable libraries and packages

## Commands

### Initialization and configuration
- `pydep init [NAME] --type {app|module}`: Creates a new project
- `pydep install [-e|--editable] [--global]`: Installs project dependencies
- `pydep build`: Build the project

### Dependency management
- `pydep add [PACKAGE]... [--dev] [--global]`: Adds packages to the project
- `pydep remove [PACKAGE]... [--global]`: Removes packages from the project
- `pydep update [PACKAGE]... [--global]`: Updates packages
- `pydep outdated`: Shows outdated packages
- `pydep audit [--json] [--extended]`: Audits security vulnerabilities
- `pydep list`: Lista paquetes instalados
- `pydep why PAQUETE`: Muestra por qué está instalado un paquete

### Execution
- `pydep run SCRIPT`: Ejecuta un script definido en pypackage.json
- `pydep version`: Muestra la versión de PyDM

### Utility pydepx
PyDM includes a utility `pydepx` to run Python tools with enriched output:

```bash
# Run black with improved output
pydepx black .

# Run a module as a script
pydepx -m http.server 8000

# Run tests with pytest
pydepx -m pytest tests/
```

## pypackage.json

The `pypackage.json` file is the main configuration file of PyDM. Here is a complete example:

```json
{
  "type": "app",
  "name": "my-project",
  "version": "0.1.0",
  "description": "A description of my project",
  "authors": [{"name": "Your Name"}],
  "license": "MIT",
  "dependencies": {
    "requests": "^2.31.0"
  },
  "devDependencies": {
    "pytest": "^8.0.0",
    "black": "^24.0.0"
  },
  "scripts": {
    "start": "python main.py",
    "test": "pytest"
  },
  "executable": {
    "target": "main.py",
    "parameters": ["--onefile"],
    "output": "dist/ejecutable"
  },
  "useGlobalDeps": false,
  "pyprojectUse": false
}
```

## Advanced usage

### Global mode
You can install packages globally with the `-g|--global` flag:

```bash
# Install a package globally
pydep add -g black

# List .venv or global packages
pydep list
```

### Environment variables
PyDepM respects the following environment variables:
- `PYDM_VERBOSE`: Activates verbose mode (equivalent to `--logs`)
- `PYDM_GLOBAL_DEPS`: Uses global dependencies (equivalent to `--global`)

## Contributing

Contributions are welcome! Please read our contribution guide before submitting pull requests.

## License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.

## Acknowledgments

- To the Python community for their incredible ecosystem
- Projects that inspire us: npm, pip, pipenv, poetry
- To all contributors who help improve PyDM
