"""Spec generator for extracting API specifications from backend frameworks."""

import importlib.util
import sys
from pathlib import Path
from typing import Dict, Any, Optional
import json
import subprocess

from .config import ConciliateConfig


class SpecGeneratorError(Exception):
    """Exception raised when spec generation fails."""
    pass


class SpecGenerator:
    """Generates API specifications from backend code."""
    
    def __init__(self, config: ConciliateConfig):
        self.config = config
        self.backend_path = Path(config.backend_path)
    
    def generate(self) -> Dict[str, Any]:
        """
        Generate OpenAPI spec from backend.
        
        Returns:
            OpenAPI specification as dictionary
        
        Raises:
            SpecGeneratorError: If spec generation fails
        """
        framework = self.config.framework.lower()
        
        if framework == "fastapi":
            return self._generate_fastapi_spec()
        elif self.config.custom_spec_command:
            return self._generate_from_command()
        else:
            raise SpecGeneratorError(
                f"Unsupported framework: {framework}. "
                "Please specify a custom_spec_command in .conciliate.yaml"
            )
    
    def _generate_fastapi_spec(self) -> Dict[str, Any]:
        """
        Extract OpenAPI spec from FastAPI application.
        
        Returns:
            OpenAPI specification dictionary
        """
        # Find main.py or app.py in backend directory
        main_file = self._find_fastapi_app()
        
        if not main_file:
            raise SpecGeneratorError(
                "Could not find FastAPI app. Looking for main.py or app.py "
                f"in {self.backend_path}"
            )
        
        # Try to import and extract spec
        try:
            spec = self._import_and_extract_spec(main_file)
            return spec
        except Exception as e:
            raise SpecGeneratorError(f"Failed to extract FastAPI spec: {e}")
    
    def _find_fastapi_app(self) -> Optional[Path]:
        """Find the main FastAPI application file."""
        candidates = ["main.py", "app.py", "api.py", "server.py"]
        
        for candidate in candidates:
            app_file = self.backend_path / candidate
            if app_file.exists():
                return app_file
        
        # Search recursively
        for py_file in self.backend_path.rglob("*.py"):
            if py_file.name in candidates:
                return py_file
        
        return None
    
    def _import_and_extract_spec(self, app_file: Path) -> Dict[str, Any]:
        """
        Import FastAPI app and extract OpenAPI schema.
        
        Args:
            app_file: Path to the FastAPI application file
        
        Returns:
            OpenAPI specification dictionary
        """
        # Add backend directory to Python path
        backend_path_str = str(self.backend_path.resolve())
        if backend_path_str not in sys.path:
            sys.path.insert(0, backend_path_str)
        
        # Import the module
        module_name = app_file.stem
        spec = importlib.util.spec_from_file_location(module_name, app_file)
        
        if spec is None or spec.loader is None:
            raise SpecGeneratorError(f"Could not load module from {app_file}")
        
        module = importlib.util.module_from_spec(spec)
        sys.modules[module_name] = module
        spec.loader.exec_module(module)
        
        # Find FastAPI app instance
        app = None
        for attr_name in dir(module):
            if attr_name.startswith("_"):
                continue
            attr = getattr(module, attr_name)
            # Check if it's a FastAPI instance (not the class itself)
            # FastAPI instances have openapi method and routes attribute
            if (hasattr(attr, "openapi") and 
                hasattr(attr, "routes") and 
                callable(getattr(attr, "openapi", None)) and
                not isinstance(attr, type)):  # Ensure it's not a class
                app = attr
                break
        
        if app is None:
            raise SpecGeneratorError(f"No FastAPI app instance found in {app_file}")
        
        # Extract OpenAPI schema
        openapi_schema = app.openapi()
        
        return openapi_schema
    
    def _generate_from_command(self) -> Dict[str, Any]:
        """
        Generate spec using custom command.
        
        Returns:
            OpenAPI specification dictionary
        """
        if not self.config.custom_spec_command:
            raise SpecGeneratorError("No custom spec command configured")
        
        try:
            result = subprocess.run(
                self.config.custom_spec_command,
                shell=True,
                cwd=self.backend_path,
                capture_output=True,
                text=True,
                timeout=30,
            )
            
            if result.returncode != 0:
                raise SpecGeneratorError(
                    f"Custom command failed: {result.stderr}"
                )
            
            # Parse JSON output
            spec = json.loads(result.stdout)
            return spec
            
        except subprocess.TimeoutExpired:
            raise SpecGeneratorError("Custom command timed out")
        except json.JSONDecodeError as e:
            raise SpecGeneratorError(f"Invalid JSON output from custom command: {e}")
        except Exception as e:
            raise SpecGeneratorError(f"Failed to run custom command: {e}")


def generate_spec(config: ConciliateConfig) -> Dict[str, Any]:
    """
    Convenience function to generate API spec.
    
    Args:
        config: Conciliate configuration
    
    Returns:
        OpenAPI specification dictionary
    """
    generator = SpecGenerator(config)
    return generator.generate()
