# mm-pos

`mm-pos` is an open-source Point of Sale (POS) library built to replace outdated and expensive systems many restaurants still rely on. It started as a project for my family-owned restaurant and is designed to provide a modern, flexible, and affordable foundation for handling orders, payments, receipts, tables, inventory, and reports.

The library focuses on being reusable and framework-agnostic. It can be integrated into web applications, desktop software, or custom automation workflows. Developers can build complete POS systems on top of `mm-pos` without having to implement the core business logic themselves.

## Features

- **Menu Management** – create and categorize menu items
- **Orders** – link orders to menu items, tables, and staff
- **Payments** – support multiple payment methods per order
- **Users and Roles** – waiter, cashier, admin with role-based permissions
- **Table Management** – open, close, and merge tables
- **Inventory Tracking** – optional stock deduction tied to menu items
- **Receipts** – generate itemized receipts for orders
- **Reports** – daily totals, top-selling items, payment breakdowns
- **Database Utilities** – simple setup and initialization helpers

 ## Installation

Install the package from PyPI:

```bash
pip install mm-pos
```

## Requirements
- Python 3.9+
- Dependencies listed in requirements.txt

To install all dependencies for development:

```bash
pip install -r requirements.txt
```

## Quick Start

### 1. Initialize a database

```python
from mm_pos.db import init_db, MenuItemDB

# In-memory SQLite database for testing
Session = init_db("sqlite:///:memory:")
session = Session()
```

### 2. Create a menu item

```python
burger = MenuItemDB(name="Burger", price=9.99, category="Food")
session.add(burger)
session.commit()
```

### 3. Create a user

```python
from mm_pos.db import UserDB

alice = UserDB(name="Alice", role="waiter")
alice.set_pin("1234")  # securely hash the PIN
session.add(alice)
session.commit()
```

### 4. Create an order with items

```python
from mm_pos.db import OrderDB, OrderItemDB

order = OrderDB(table_number=1, user=alice)
session.add(order)
session.commit()

session.add(OrderItemDB(order_id=order.id, menu_item_id=burger.id, qty=2))
session.commit()
```

### 5. Record a payment

```python
from mm_pos.db import PaymentDB

payment = PaymentDB(order=order, method="cash", amount_given=25.0, user=alice)
session.add(payment)
session.commit()
```

### 6. Generate a receipt

```python
from mm_pos.receipt import Receipt

receipt = Receipt(order)
print(receipt.generate_text())
```

## Next Steps

- Explore `InventoryManager` for stock management
- Use `TableManager` for managing tables in a restaurant
- Generate reports with the `Reports` class
- Integrate with FastAPI to expose your POS system as a web service
