/******/ (() => { // webpackBootstrap
/******/ 	"use strict";

;// ./private/js/cms.texteditor.js
/* eslint-env es6 */
/* jshint esversion: 6 */
/* global document, window, console */

class CmsTextEditor {
    constructor (el, options, save_callback) {
        this.el = el;
        this.plugin_identifier = this.find_plugin_identifier();
        const id_split = this.plugin_identifier.split('-');
        this.plugin_id = parseInt(id_split[id_split.length-1]);
        this.options = options;
        this.events = {};
        this.save = save_callback;
        this.init();
    }

    destroy () {
        this.el.removeEventListener('focus', this._focus.bind(this));
        this.el.removeEventListener('blur', this._blur.bind(this));
        this.el.removeEventListener('input', this._change);
        this.el.removeEventListener('keydown', this._key_down);
        this.el.removeEventListener('paste', this._paste);
        this.el.setAttribute('contenteditable', 'false');
    }

    init () {
        this.el.setAttribute('contenteditable', 'plaintext-only');
        if (!this.el.isContentEditable) {
            this.el.setAttribute('contenteditable', 'true');
            this.options.enforcePlaintext = true;

        }
        this.el.setAttribute('spellcheck', this.options.spellcheck || 'false');
        this.el.addEventListener('input', this._change);
        this.el.addEventListener('focus', this._focus.bind(this));
        this.el.addEventListener('blur', this._blur.bind(this));
        this.el.addEventListener('keydown', this._key_down);
        if (this.options.enforcePlaintext) {
            this.el.addEventListener('paste', this._paste);
        }
    }

    _key_down (e) {
        if (e.key === 'Enter') {
            e.preventDefault();
            e.target.blur();
        }
        if (e.key === 'Escape') {
            e.preventDefault();
            if (e.target.dataset.undo) {
                e.target.innerText = e.target.dataset.undo;
                e.target.dataset.changed = false;
            }
            e.target.blur();
        }
    }

    _focus (e) {
        this.options.undo = this.el.innerText;
    }

    _blur (e) {
        this.save(e.target, (el, response) => {
            setTimeout(() => {
                if (e.target.dataset.changed === 'true') {
                    e.target.innerText = this.options.undo;
                    e.target.dataset.changed = 'false';
                    e.target.focus();
                }
            }, 100);
        });
    }

    _paste (e) {
        // Upon past only take the plain text
        e.preventDefault();
        let text = e.clipboardData.getData('text/plain');
        if (text) {
            const [start, end] = [e.target.selectionStart, this.el.selectionEnd];
            e.target.setRangeText(text, start, end, 'select');
        }
    }

    _change (e) {
        e.target.dataset.changed = 'true';
    }

    find_plugin_identifier () {
        const header = 'cms-plugin-';

        for (let cls of this.el.classList) {
            if (cls.startsWith(header)) {
                let items = cls.substring(header.length).split('-');
                if (items.length === 4 && items[items.length-1] == parseInt(items[items.length-1])) {
                    return items.join('-');
                }
            }
        }
        return null;
    }
}



;// ./private/js/cms.linkfield.js
/* eslint-env es11 */
/* jshint esversion: 11 */
/* global document, window, console */


class LinkField {
    constructor(element, options) {
        const hasFocus = element.contains(document.activeElement);

        this.options = options;
        this.urlElement = element;
        this.form = element.closest("form");
        this.selectElement = this.form?.querySelector(`input[name="${this.urlElement.name + '_select'}"]`);
        this.dropdownIsOpen = false;
        this.boundCloseDropdown = this.closeDropdown.bind(this);
        if (this.selectElement) {
            this.urlElement.setAttribute('type', 'hidden');  // Two input types?
            this.selectElement.setAttribute('type', 'hidden');  // Make hidden and add common input
            this.createInput(hasFocus);
            this.registerEvents();
        }
        this.populateField();
    }

    createInput(hasFocus = false) {
        this.inputElement = document.createElement('input');
        this.inputElement.setAttribute('type', 'text');
        this.inputElement.setAttribute('autocomplete', 'off');
        this.inputElement.setAttribute('spellcheck', 'false');
        this.inputElement.setAttribute('autocorrect', 'off');
        this.inputElement.setAttribute('autocapitalize', 'off');
        this.inputElement.setAttribute('placeholder', this.urlElement.getAttribute('placeholder') ||'');
        this.inputElement.className = this.urlElement.className;
        this.inputElement.classList.add('cms-linkfield-input');

        this.wrapper = document.createElement('div');
        this.wrapper.classList.add('cms-linkfield-wrapper');
        this.urlElement.insertAdjacentElement('afterend', this.wrapper);
        this.urlElement.setAttribute('type', 'hidden');
        this.dropdown = document.createElement('div');
        this.dropdown.classList.add('cms-linkfield-dropdown');
        if (this.form.style.zIndex) {
            this.dropdown.style.zIndex = this.form.style.zIndex + 1;
        }
        this.wrapper.appendChild(this.inputElement);
        this.wrapper.appendChild(this.dropdown);
        if (hasFocus) {
            this.inputElement.focus();
        }
    }

    populateField() {
        if (this.selectElement) {
            if (this.selectElement.value) {
                this.handleChange();
                this.inputElement.classList.add('cms-linkfield-selected');
            } else if (this.urlElement.value) {
                this.inputElement.value = this.urlElement.value;
                this.inputElement.classList.remove('cms-linkfield-selected');
            } else {
                this.inputElement.value = '';
                this.search();  // Trigger search to populate dropdown
                this.closeDropdown();  // Close dropdown if it was open
                this.inputElement.classList.remove('cms-linkfield-selected');
            }
            if (this.selectElement.getAttribute('data-value')) {
                this.inputElement.value = this.selectElement.getAttribute('data-value');
                this.inputElement.classList.add('cms-linkfield-selected');
            }
            if (this.selectElement.getAttribute('data-href')) {
                this.urlElement.value = this.selectElement.getAttribute('data-href');
                this.inputElement.classList.add('cms-linkfield-selected');
            }
            this.dropdown.innerHTML = '';  // CSS hides dropdown when empty
        }
    }

    registerEvents() {
        this.inputElement.addEventListener('input', this.handleInput.bind(this));
        this.inputElement.addEventListener('click', event => {
            if (this.dropdownIsOpen) {
                this.closeDropdown();
            } else {
                this.search();
            }
        });
        // Allow focus-triggered dropdown for keyboard accessibility
        this.inputElement.addEventListener('focus', event => {
            if (!this.dropdownIsOpen) {
                this.search();
            }
        });
        // Keyboard navigation (open dropdown on ArrowDown)
        this.inputElement.addEventListener('keydown', event => {
            if ((event.key === 'ArrowDown' || event.key === 'Down') && !this.dropdownIsOpen) {
                event.preventDefault();  // Prevent cursor movement
                event.stopPropagation();  // Prevent closing the input
                this.search();
            }
            if ((event.key === 'Escape' || event.key === 'Esc') && this.dropdownIsOpen) {
                event.preventDefault();  // Prevent closing the input
                event.stopPropagation();  // Prevent closing the dropdown
                this.closeDropdown();
            }
        });
        this.urlElement.addEventListener('input', event => {
            this.inputElement.value = event.target.value || '';
            this.inputElement.classList.remove('cms-linkfield-selected');
            // this.selectElement.value = '';
        });
        this.selectElement.addEventListener('input', event => this.handleChange(event));
        this.dropdown.addEventListener('click', this.handleSelection.bind(this));
        this.intersection = new IntersectionObserver((entries, observer) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    this.updateSearch();
                    observer.disconnect();
                }
            });
        });
    }

    handleInput(event) {
        // User typed something into the field -> no predefined value selected
        this.selectElement.value = '';
        this.urlElement.value = this.inputElement.value;
        this.inputElement.classList.remove('cms-linkfield-selected');
        this.search();
       }

    showResults(response, page = 1) {
        let currentSection;  // Keep track of the current section so that paginated data can be added
        if (page === 1) {
            // First page clears the dropdown
            this.dropdown.innerHTML = '';
            currentSection = '';
        } else {
            // Remove the more link
            const more = this.dropdown.querySelector('.cms-linkfield-more');
            currentSection = more?.dataset.group;
            more?.remove();
        }
        response.results.forEach(result => currentSection = this._addResult(result, currentSection));
        if (response?.pagination?.more) {
            const more = document.createElement('div');
            more.classList.add('cms-linkfield-more');
            more.setAttribute('data-page', page + 1);
            more.setAttribute('data-group', currentSection);
            more.textContent = '...';
            this.dropdown.appendChild(more);
            this.intersection.observe(more);
        }
    }

    _addResult(result, currentSection = '') {
        const item = document.createElement('div');
        item.textContent = result.text;
        if (result.id) {
            item.classList.add('cms-linkfield-option');
            item.setAttribute('data-value', result.id);
            item.setAttribute('data-href', result.url || '#');
            item.setAttribute('data-text', result.verbose || result.text);
        }
        if (result.children && result.children.length > 0) {
            item.classList.add('cms-linkfield-parent');
            if (result.text !== currentSection) {
                this.dropdown.appendChild(item);
                currentSection = result.text;
            }
            result.children.forEach(child => {
                this._addResult(child);
            });
        } else {
            this.dropdown.appendChild(item);
        }
        return currentSection;
    }

    handleSelection(event) {
        event.stopPropagation();
        event.preventDefault();
        if (event.target.classList.contains('cms-linkfield-option')) {
            const value = event.target.getAttribute('data-text') || event.target.textContent;
            this.inputElement.value = value.trim();
            this.inputElement.classList.add('cms-linkfield-selected');
            this.urlElement.value = event.target.getAttribute('data-href');
            this.selectElement.value = event.target.getAttribute('data-value');
            this.inputElement.blur();
            this.closeDropdown(event);
        }
        // this.dropdown.innerHTML = '';  // CSS hides dropdown when empty
    }

    openDropdown(event) {
        if (!this.dropdownIsOpen) {
            this.dropdownIsOpen = true;
            this.dropdown.classList.add('open');
            document.addEventListener('click', this.boundCloseDropdown);
        }
    }

    closeDropdown(event) {
        if (!event || !this.wrapper.contains(event.target) || this.dropdown.contains(event.target)) {
            this.dropdownIsOpen = false;
            this.dropdown.classList.remove('open');
            document.removeEventListener('click', this.boundCloseDropdown);
        }
    }

    toggleDropdown() {
       this.dropdownIsOpen ? this.closeDropdown() : this.openDropdown();
    }

    handleChange(event) {
        if (this.selectElement.value && this.options.url) {
            fetch(this.options.url + (this.options.url.includes('?') ? '&g=' : '?g=') + encodeURIComponent(this.selectElement.value))
                .then(response => response.json())
                .then(data => {
                    this.inputElement.value = data.text;
                    this.inputElement.classList.add('cms-linkfield-selected');
                    this.urlElement.value = data.url || '#';
                });
        }
    }

    search(page = 1) {
        this.openDropdown();
        const searchText = this.inputElement.value.toLowerCase();
        this.fetchData(searchText, page).then(response => {
            this.showResults(response, page);
        }).catch (error => {
           this.dropdown.innerHTML = `<div class="cms-linkfield-error">${error.message}</div>`;
        });
    }

    updateSearch() {
        const more = this.dropdown.querySelector('.cms-linkfield-more');
        if (more) {
            this.search(parseInt(more.getAttribute('data-page')));
        }
    }

    fetchData(searchText, page ) {
        if (this.options.url) {
            const url = this.options.url + (this.options.url.includes('?') ? '&' : '?') +
                `q=${encodeURIComponent(searchText)}${page > 1 ? '&page=' + page : ''}`;
            return fetch(url)
                .then(response => response.json());
        }
        return new Promise(resolve => {
            resolve({results: []});
        });
    }
}

/* harmony default export */ const cms_linkfield = (LinkField);

;// ./private/js/cms.dialog.js
/* eslint-env es6 */
/* jshint esversion: 6 */
/* global document, window, console */


class CmsDialog {
    /**
     * Constructor for creating an instance of the class showing a django CMS modal in a
     * modal HTML dialog element to show a plugin admin form in an iframe.
     *
     * The django CMS modal is resizable (thanks to CSS) and movable. It cannot be
     * minimized or maximized. It blocks all other input on the page until it is closed.
     *
     * The dialog element is attached to the <div id="cms-top"> at the beginning of a page
     * with a toolbar. The dialog is removed when it is closed.
     *
     * @param {Element} el - The editor element to be associated with the instance.
     * @param {Function} saveSuccess - The success callback function to be invoked upon save.
     * @param {Function} cancel - The callback function to be invoked upon cancellation.
     */
    constructor(el, saveSuccess, cancel) {
        this.el = el;
        this.saveSuccess = saveSuccess;
        this.cancel = cancel;
        this.close = this._close.bind(this);
    }

    /**
     * Create a plugin modal dialog.
     * @return {HTMLIFrameElement} - The newly created iframe element.
     */
    pluginDialog() {
        this.dialog = document.createElement("dialog");
        this.dialog.classList.add("cms-dialog");
        this.dialog.dataset.editor = this.el.id;
        this.dialog.innerHTML = `
            <div class="cms-modal-head">
                <span class="cms-modal-title">
                    <span class="cms-modal-title-prefix"></span>
                    <span class="cms-modal-title-suffix"></span>
                    <span class="cms-modal-close cms-icon cms-icon-close"></span>
                </span>
            </div>
            <div class="cms-modal-body">
            </div>
            <div class="cms-modal-foot">
                <div class="cms-modal-buttons">
                <div class="cms-modal-buttons-inner">
                    <div class="cms-modal-item-buttons"><a href="#" class="cms-btn cms-btn-action default">Save</a></div>
                    <div class="cms-modal-item-buttons"><a href="#" class="cms-btn cms-btn-close-action">Cancel</a></div>
                </div>
            </div>
        `;

        (window.parent || window).document.querySelector('div.cms').prepend(this.dialog);
        const settings = window.CMS_Editor.getSettings(this.el);
        this.dialog.querySelector(".cms-modal-title-suffix").textContent = settings.lang.edit;
        this.dialog.querySelector(".cms-modal-title-prefix").textContent = settings.lang.toolbar;
        this.dialog.querySelector('.cms-modal-title').addEventListener('mousedown', (event) => {
            this.dragDialog(event);
        });
        this.dialog.querySelector('.cms-modal-title').addEventListener('touchstart', (event) => {
            this.swipeDialog(event);
        });
        const closeEvent = (event) => {
            event.stopPropagation();
            event.preventDefault();
            this.close();
            if (this.cancel) {
                this.cancel(event);
            }
        };
        this.dialog.addEventListener("close", (event) => closeEvent(event));
        this.dialog
            .querySelector(".cms-btn-close-action")
            .addEventListener('click', (event) => closeEvent(event));
        this.dialog
            .querySelector(".cms-modal-close")
            .addEventListener('click', (event) => closeEvent(event));
        this.dialog.addEventListener('keydown', (event) => {
            if (event.key === 'Escape') {
                // Don't propagate the event to the CMS modal, or it will close, too
                event.stopPropagation();
            }
        });
        const iframe = document.createElement('iframe');
        this.dialog.querySelector(".cms-modal-body").append(iframe);
        return iframe;
    }

    /**
     * Opens the edit dialog for a specific plugin.
     *
     * @param {string} pluginId - The ID of the plugin to be edited.
     *
     * @return {void}
     */
    editDialog(pluginId) {
        const iframe = this.pluginDialog();

        window.CMS_Editor.editPluginForm(pluginId, iframe, this.el,
            (el, content, heading, submitrow) => this._dialogLoaded(el, content, heading, submitrow),
            (el, content, data) => this._dialogSaved(el, content, data));
    }

    /**
     * Dialog to add a plugin.
     *
     * @param {string} pluginType - The type of the plugin.
     * @param {string} selectionText - The selected text (will be copied into the input field with
     *                                 the class "js-prepopulate-selected-text").
     *
     * @return {void}
     */
    addDialog(pluginType, selectionText) {
        const iframe = this.pluginDialog();

        window.CMS_Editor.addPluginForm(pluginType, iframe, this.el,
            (el, content, heading, submitrow) =>
                this._dialogLoaded(el, content, heading, submitrow, selectionText),
            (el, content, data) => this._dialogSaved(el, content, data));
    }

    _dialogLoaded(el, content, heading, submitrow, selectionText) {
        if (submitrow) {
            this.dialog.querySelector('.cms-btn-action.default')
                .addEventListener('click', (event) => {
                    event.stopPropagation();
                    event.preventDefault();
                    submitrow.closest('form').submit();
                });
        } else {
            this.dialog.querySelector('.cms-btn-action.default').style.display = 'none';
        }
        if (heading) {
            this.dialog.querySelector(".cms-modal-title-suffix").textContent = heading.textContent;
        }
        this.open();
        // If form is big (i.e., scrollbars appear), make the dialog bigger
        if (content.documentElement.scrollHeight > content.documentElement.clientHeight) {
            this.dialog.style.height = "60%";
            this.dialog.style.width = "80%";
            content.documentElement.scrollTop = 0;
        }

        // Prefill marked input fields with selected text
        selectionText = selectionText || '';
        if (selectionText.length > 0) {
            let fillInput = content.querySelector('.js-ckeditor-use-selected-text,.js-prepopulate-selected-text') ||
                content.querySelector('#id_name');
            if (fillInput) {  // Does such a field exist?
                if (!(fillInput.value.trim())) {
                    fillInput.value = selectionText;  // Prefill the field only if it is empty
                    fillInput.focus();
                }
            }
        }
    }

    _dialogSaved(el, content, data) {
        this.saveSuccess(data);
        this.close();
    }

    /**
     * Opens the dialog by showing it in a modal state.
     *
     * @return {void}
     */
    open() {
        this.dialog.showModal();
    }

    /**
     * Removes the dialog
     *
     * @method close
     * @memberof ClassName
     * @returns {void}
     */
    _close() {
        this.dialog.removeEventListener("close", this.close);
        this.dialog.remove();
    }

    /**
     * Allows dragging the dialog based on the user's mouse movements.
     *
     * @param {Event} event - The mouse event that triggers the drag.
     */
    dragDialog(event) {
        if (event.which !== 1) {
            return;
        }
        event.preventDefault();
        const firstX = event.pageX;
        const firstY = event.pageY;
        const initialX = parseInt(getComputedStyle(this.dialog).left);
        const initialY = parseInt(getComputedStyle(this.dialog).top);

        const dragIt = (e) => {
            this.dialog.style.left = initialX + e.pageX - firstX + 'px';
            this.dialog.style.top = initialY + e.pageY - firstY + 'px';
        };
        const Window = window.parent || window;
        Window.addEventListener('mousemove', dragIt, false);
        Window.addEventListener('mouseup', (e) => {
            Window.removeEventListener('mousemove', dragIt, false);
        }, false);
    }

    /**
     * Allows dragging the dialog based on the user's touch movements.
     *
     * @param {Event} event - The touch event that triggers the drag.
     */
    swipeDialog(event) {
        event.preventDefault();

        const firstX = event.pageX;
        const firstY = event.pageY;
        const initialX = parseInt(getComputedStyle(this.dialog).left);
        const initialY = parseInt(getComputedStyle(this.dialog).top);

        const swipeIt = (e) => {
            const contact = e.touches;
            this.dialog.style.left = initialX + contact[0].pageX - firstX + 'px';
            this.dialog.style.top = initialY + contact[0].pageY - firstY + 'px';
        };

        const Window = window.parent || window;
        Window.addEventListener('touchmove', swipeIt, false);
        Window.addEventListener('touchend', (e) => {
            Window.removeEventListener('touchmove', swipeIt, false);
        }, false);
    }
}


/**
 * Represents an editor form, e.g. to enter a link address, or a pop-up toolbar.
 *
 * The form is contained in a (non-modal) dialog element which is attached to the editor's div wrapper.
 *
 *
 * @constructor
 * @param {*} el - The element to attach the form to.
 * @param {function} saveSuccess - The callback function to be called when form is successfully submitted.
 * @param {function} cancel - The callback function to be called when form is cancelled.
 */
class CmsForm {
     constructor(el, saveSuccess, cancel) {
        this.el = el;
        this.saveSuccess = saveSuccess;
        this.cancel = cancel;
        this.close = this._close.bind(this);
    }

    formDialog(form, options) {
        this.dialog = document.createElement("dialog");
        this.dialog.classList.add("cms-form-dialog");
        if (options.toolbar) {
            this.dialog.innerHTML = form;
        } else {
            this.dialog.innerHTML = `
                <form class="cms-form">
                    <div class="cms-form-inputs">${form}</div>
                    <div class="cms-form-buttons">
                        <span class="submit"><svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-check-lg" viewBox="0 0 16 16">
                            <path d="M12.736 3.97a.733.733 0 0 1 1.047 0c.286.289.29.756.01 1.05L7.88 12.01a.733.733 0 0 1-1.065.02L3.217 8.384a.757.757 0 0 1 0-1.06.733.733 0 0 1 1.047 0l3.052 3.093 5.4-6.425z"/>
                            </svg></span>
                        <span class="cancel"><svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-x-lg" viewBox="0 0 16 16">
                            <path d="M2.146 2.854a.5.5 0 1 1 .708-.708L8 7.293l5.146-5.147a.5.5 0 0 1 .708.708L8.707 8l5.147 5.146a.5.5 0 0 1-.708.708L8 8.707l-5.146 5.147a.5.5 0 0 1-.708-.708L7.293 8z"/>
                            </svg></span>
                    </div>
                </form>`;
        }

        if (options && options.x && options.y) {
            const el_pos = this.el.getBoundingClientRect();
            if (options.x > window.innerWidth / 2) {
                this.dialog.classList.add("right");
                this.dialog.style.right = (el_pos.x + el_pos.width - options.x - 24 - 10) + 'px';
            } else {
                this.dialog.style.left = (options.x - el_pos.x - 24) + 'px';
            }
            this.dialog.style.top = (options.y - el_pos.y + 5) + 'px';
            this.dialog.style.transform = 'none';
        }

        // Add the dialog to the inline editor
        this.el.prepend(this.dialog);
        this.dialog.addEventListener("close", (event) => {
            event.stopPropagation();
            this.close();
        });
        document.addEventListener("click", this.close);
        if (this.dialog.querySelector('.cancel')) {
            this.dialog.querySelector(".cancel")
                .addEventListener('click',  () => this.close() );
        }
        this.dialog.addEventListener('keydown', (event) => {
            if (event.key === 'Escape') {
                // Don't propagate the event to the CMS modal, or it will close, too
                event.stopPropagation();
                event.preventDefault();
                this.close();
            }
        });
        const formElement = this.dialog.querySelector('form');
        if (formElement) {
            formElement.addEventListener('submit', (event) => {
                event.preventDefault();
                this.submit();
            });
            this.dialog.querySelector(".submit").addEventListener('click', (event) => {
                event.preventDefault();
                if (this.dialog.querySelector('form').reportValidity()) {
                    this.submit();
                }
            });
        }
        return formElement || this.dialog;
    }

    open() {
        this.dialog.show();
        const firstInput = this.dialog.querySelector('input');
        if (firstInput) {
            firstInput.focus();
        }
    }

    _close(event) {
        if (!event || !this.dialog.contains(event.target)) {
            // Do only close if the click is outside the dialog
            document.removeEventListener("click", this.close);
            this.dialog.removeEventListener("close", this.close);
            if (this.cancel) {
                this.cancel(event);
            }
            this.dialog.remove();
        }
    }

    submit() {
        const data = new FormData(this.dialog.querySelector('form'));
        this.dialog.remove();
        this.saveSuccess(data);
    }
}


/**
 * Converts a given form array to HTML representation.
 *
 * @param {Array} formArray - The array containing form elements.
 * @returns {string} - The HTML form representation.
 */
function formToHtml(formArray) {
    'use strict';
    let form = '';

    formArray.forEach((element) => {
        const id = Math.random().toString(36).slice(2, 9);
        switch(element.type) {
            case 'text':
            case 'url':
                if (element.label) {
                    form += `<label for="id${id}">${element.label}</label>`;
                }
                form += `<input type="${element.type}" id="id${id}"
                    placeholder="${element.placeholder || ''}" name="${element.name}"
                    class="${element.class || ''}"
                    value="${element.value || ''}"${element.required ? ' required' : ''}>`;
                break;
            case 'select':
                if (element.label) {
                    form += `<label for="id${id}">${element.label}</label>`;
                }
                form += `<select id="id${id}" name="${element.name}"${element.required ? ' required' : ''} class="${element.class || ''}">`;
                element.options.forEach((option) => {
                    form += `<option value="${option.value}"${option.value === element.value ? ' selected' : ''}>${option.label}</option>`;
                });
                form += '</select>';
                break;
            case 'hidden':
                form += `<input type="hidden" id="id${id}" name="${element.name}" value="${element.value || ''}">`;
                break;
            case 'hr':
                form += '<hr>';
                break;
            case 'link':
                form += `<a href="${element.url}" target="_blank">${element.label}</a>`;
                break;
            case 'section':
                const content = formToHtml(element.content);
                form += `<details><summary>${element.label}</summary>${content}</details>`;
                break;
        }
    });
    return form + '<input type="submit" hidden />';
}


/**
 * Populates a given HTML form with values from a form object.
 *
 * @param {HTMLFormElement} htmlForm - The HTML form element to populate.
 * @param {object} attributes - The attributes to populate the form with.
 * @param {Array} formObject - The form object containing input values.
 */
function populateForm(htmlForm,  attributes, formObject) {
    'use strict';
    if (attributes && formObject) {
        for (const input of formObject) {
            let value;
            if (input.name in attributes) {
                value = attributes[input.name] || '';
            } else {
                value = input.value || '';
            }
            const field = htmlForm.querySelector(`[name="${input.name}"]`);
            if (field && field.value !== value) {
                field.value = value;
                if (field.getAttribute('type') === 'hidden') {
                    // Trigger change event for hidden fields
                    field.dispatchEvent(new Event('input', {bubbles: true, cancelable: true}));
                }
            }
        }
    }
}






;// ./private/js/cms.editor.js
/* eslint-env es11 */
/* jshint esversion: 11 */
/* global window, document, fetch, IntersectionObserver, URLSearchParams, console */





// #############################################################################
// CMS Editor
// #############################################################################

class CMSEditor {

    // CMS Editor: constructor
    // Initialize the editor object
    constructor() {
        this._global_settings = {};
        this._editor_settings = {};
        this._generic_editors = {};
        this._admin_selector = 'textarea.CMS_Editor';
        this._admin_add_row_selector = 'body.change-form .add-row a';
        this._inline_admin_selector = 'body.change-form .form-row';
        this.API = {
            LinkField: cms_linkfield,
            CmsDialog: CmsDialog,
            CmsTextEditor: CmsTextEditor,
        };

        document.addEventListener('DOMContentLoaded', () => {
            // Get the CMS object from the parent window
            if (window.CMS !== undefined && window.CMS.config !== undefined) {
                this.mainWindow = window;
                this.CMS = window.CMS;
            } else {
                this.mainWindow = window.parent;
                this.CMS = window.parent.CMS;
            }

            if (this.CMS) {
                // Only needs to happen on the main window.
                this.CMS.$(window).on('cms-content-refresh', () => {
                    this._resetInlineEditors();
                });
            }

            if (document.querySelector(this._inline_admin_selector + '.empty-form')) {
                // Marker for inline admin form: do **not** initialize empty form templates
                this._admin_selector = this._inline_admin_selector + ':not(.empty-form) ' + this._admin_selector;
            }
            this.initAll();
        }, { once: true });
    }

    // CMS Editor: init_all
    // Initialize all editors on the page
    initAll () {
        // Get global options from script element
        try {
            this._global_settings = JSON.parse(document.getElementById('cms-editor-cfg').textContent);
        } catch (e) {
            this._global_settings = {};
        }

        // All textareas with class CMS_Editor: typically on admin site
        document.querySelectorAll(this._admin_selector).forEach(
            (el) => this.init(el), this
        );
        // Register all plugins on the page for inline editing
        this.initInlineEditors();

        // Listen to the add row click for inline admin in a change form
        if (this._admin_add_row_selector) {
            setTimeout(() => {
                for (const el of document.querySelectorAll(this._admin_add_row_selector)) {
                    el.addEventListener('click', (event) => {
                        setTimeout(() => {
                            document.querySelectorAll(this._admin_selector).forEach(
                                (el) => this.init(el), this
                            );
                        }, 0);
                    });
                }
            }, 0);
        }
    }

    // CMS Editor: init
    // Initialize a single editor
    init (el) {
        if (!el.id) {
            el.id = "cms-edit-" + Math.random().toString(36).slice(2, 9);
        }
        // Create editor
        if (!el.dataset.cmsType || el.dataset.cmsType === 'TextPlugin' || el.dataset.cmsType === 'HTMLField') {
            this._createRTE(el);
        } else if (el.dataset.cmsType === 'CharField') {
            // Creat simple generic text editor
            this._generic_editors[el.id] = new CmsTextEditor(el, {
                    spellcheck: el.dataset.spellcheck || 'false',
                },
                (el) => this.saveData(el)
            );
        }
    }

    // CMS Editor: initInlineEditors
    // Register all plugins on the page for inline editing
    // This is called from init_all
    initInlineEditors() {
        if (this.CMS === undefined || this.CMS._plugins === undefined) {
            // Check the CMS frontend for plugins
            // no plugins -> no inline editors
            return;
        }

        this.observer = this.observer || new IntersectionObserver( (entries) => {
            entries.forEach((entry) => {
                if (entry.isIntersecting) {
                    this.observer.unobserve(entry.target);  // Only init once
                    this.init(entry.target);
                }
            }, this);
        }, {
            root: null,
            threshold: 0.05
        });
        this.observer.disconnect();

        let generic_inline_fields = document.getElementById('cms-generic-inline-fields') || {};
        if (generic_inline_fields) {
            generic_inline_fields = JSON.parse(generic_inline_fields.textContent || '{}');
        }

        this.CMS._plugins.forEach(function (plugin) {
            if (plugin[1].type === 'plugin' || plugin[1].type === 'generic') {
                // Either plugin or frontend editable element
                const url = plugin[1].urls.edit_plugin;
                const id = plugin[1].plugin_id;
                let wrapper;

                if (plugin[1].type === 'plugin' && plugin[1].plugin_type === 'TextPlugin') {
                    // Text plugin
                    const elements = document.querySelectorAll('.cms-plugin.' + plugin[0]);
                    wrapper = this._initInlineRichText(elements, url, plugin[0]);
                    if (wrapper) {
                        wrapper.dataset.cmsPluginId = id;
                        wrapper.dataset.cmsType = 'TextPlugin';
                    }
                } else if (plugin[1].type === 'generic') {
                    // Frontend editable element
                    const edit_fields = new URL(url.replace('&amp;', '&'), 'https://random-base.org')
                        .searchParams.get('edit_fields');  // Get the edit_fields parameter from the URL
                    if (edit_fields && edit_fields.indexOf(',') === -1 && edit_fields !== 'changelist') {
                        // Single field
                        const generic_class = plugin[0].split('-');
                        const search_key = `${generic_class[2]}-${generic_class[3]}-${edit_fields}`;
                        if (generic_inline_fields[search_key]) {
                            // Inline editable?
                            wrapper = this._initInlineRichText(document.querySelectorAll(`.${plugin[0]}`), url, plugin[0]);
                            if (wrapper) {
                                wrapper.dataset.cmsCsrfToken = this.CMS.config.csrf;
                                wrapper.dataset.cmsField = edit_fields;
                                wrapper.dataset.cmsType = (
                                    generic_inline_fields[search_key] === 'HTMLFormField' ?
                                        'HTMLField' : generic_inline_fields[search_key]
                                );
                                wrapper.dataset.settings = 'cms-cfg-htmlfield-inline';
                            }
                        }
                    }
                }

                if (wrapper) {
                    // Catch CMS single click event to highlight the plugin
                    // Catch CMS double click event if present, since double click is needed by Editor
                    if (!Array.from(this.observer.root?.children || []).includes(wrapper)) {
                        // Only add to the observer if not already observed (e.g., if the page only was update partially)
                        this.observer.observe(wrapper);
                        if (this.CMS) {
                            // Remove django CMS core's double click event handler which opens an edit dialog
                            this.CMS.$(wrapper).off('dblclick.cms.plugin')
                                .on('dblclick.cms-editor', function (event) {
                                    event.stopPropagation();
                                });
                            wrapper.addEventListener('focusin', () => {
                                this._highlightTextplugin(id);
                            }, true);
                            // Prevent tooltip on hover
                            this.CMS.$(wrapper).off('pointerover.cms.plugin pointerout.cms.plugin')
                                .on('pointerover.cms-editor', function (event) {
                                    window.CMS.API.Tooltip.displayToggle(false, event.target, '', id);
                                    event.stopPropagation();
                                });
                        }
                    }
                }
            }
        }, this);

        window.addEventListener('beforeunload', (event) =>  {
            if (document.querySelector('.cms-editor-inline-wrapper[data-changed="true"]')) {
                event.preventDefault();
                event.returnValue = true;
                return 'Do you really want to leave this page?';
            }
        });
    }

    _createRTE(el) {
        const settings = this.getSettings(el);
        // Element options overwrite
        settings.options = Object.assign({},
            settings.options || {},
            JSON.parse(el.dataset.options || '{}')
        );

        // Add event listener to delete data on modal cancel
        if (settings.revert_on_cancel) {
            const CMS = this.CMS;
            const csrf = CMS.config?.csrf || document.querySelector('input[name="csrfmiddlewaretoken"]').value;
            CMS.API.Helpers.addEventListener(
                'modal-close.text-plugin.text-plugin-' + settings.plugin_id,
                function(e, opts) {
                    if (!settings.revert_on_cancel || !settings.cancel_plugin_url) {
                        return;
                    }
                    CMS.$.ajax({
                        method: 'POST',
                        url: settings.cancel_plugin_url,
                        data: {
                            token: settings.action_token,
                            csrfmiddlewaretoken: csrf
                        },
                    }).done(function () {
                        CMS.API.Helpers.removeEventListener(
                            'modal-close.text-plugin.text-plugin-' + settings.plugin_id
                        );
                        opts.instance.close();
                    }).fail(function (res) {
                        CMS.API.Messages.open({
                            message: res.responseText + ' | ' + res.status + ' ' + res.statusText,
                            delay: 0,
                            error: true
                        });
                    });

                }
            );
        }
        const inModal = !!document.querySelector(
            '.app-djangocms_text.model-text.change-form #' + el.id
        );
        // Get content: json > textarea > innerHTML
        let content;

        if (el.dataset.json) {
            content = JSON.parse(el.dataset.json);
        } else  {
            content = el.innerHTML;
        }
        if (el.tagName === 'TEXTAREA') {
            el.visible = false;
            content = el.value;
            // el = el.insertAdjacentElement('afterend', document.createElement('div'));
        }

        window.cms_editor_plugin.create(
            el,
            inModal,
            content, settings,
            el.tagName !== 'TEXTAREA' ? () => this.saveData(el) : () => {
            }
        );
    }

    /**
     * Retrieves the settings for the given editor.
     * If the element is a string, it will be treated as an element's ID.
     * Reads settings from a json script element.
     *
     * @param {string|HTMLElement} el - The element or element's ID to retrieve the settings for.
     *
     * @return {Object} - The settings object for the element.
     */
    getSettings(el) {
        if (typeof el === "string") {
            if (this._editor_settings[el]) {
                return this._editor_settings[el];
            }
            el = document.getElementById(el);
        }
        const settings_el = (
            document.getElementById(el.dataset.settings) ||
            document.getElementById('cms-cfg-' + el.dataset.cmsPluginId)
        );
        if (settings_el) {
            this._editor_settings[el.id] = Object.assign(
                {},
                this._global_settings,
                JSON.parse(settings_el.textContent || '{}')
            );
        } else {
            this._editor_settings[el.id] = Object.assign(
                {},
                this._global_settings,
            );
        }
        if (!this._editor_settings[el.id].plugin_language) {
            // If the plugin change form contains a language meta tag, use it
            const lang = document.querySelector('meta[name="plugin_language"]');

            if (lang && lang.dataset.pluginLanguage) {
                this._editor_settings[el.id].plugin_language = lang.dataset.pluginLanguage;
            } else {
                // If the grouper admin is used, use the language from the grouper admin form
                const lang = document.querySelector('input[name="content__language"]');
                if (lang && lang.value) {
                    this._editor_settings[el.id].plugin_language = lang.value;
                }
            }
        }
        return this._editor_settings[el.id];
    }

    /**
     * Retrieves the list of installed plugins. (Returns empty list of no editor has been initialized.)
     *
     * @returns {Array} - The list of installed plugins.
     */
    getInstalledPlugins() {
        if (this._editor_settings) {
            return this.getSettings(Object.keys(this._editor_settings)[0]).installed_plugins || [];
        }
        return [];
    }

    // CMS Editor: destroy
    destroyAll() {
        this.destroyRTE();
        this.destroyGenericEditor();
    }

    destroyRTE() {
        for (const el of Object.keys(this._editor_settings)) {
            const element = document.getElementById(el);
            window.cms_editor_plugin.destroyEditor(el);
        }
        this._editor_settings = {};
    }

    // CMS Editor: destroyGenericEditor
    destroyGenericEditor (el) {
        if (el in this._generic_editors) {
            this._generic_editors[el].destroy();
            delete this._generic_editors[el];
            this._generic_editors.pop(el);
        }
    }

    saveData(el, action) {
        if (el && el.dataset.changed === "true") {
            const html = window.cms_editor_plugin.getHTML(el),
                json = window.cms_editor_plugin.getJSON(el);

            let url = el.dataset.cmsEditUrl;
            let csrf = el.dataset.cmsCsrfToken;
            let field = el.dataset.cmsField;
            if (this.CMS) {
                this.CMS.API.Toolbar.showLoader();
                url = this.CMS.API.Helpers.updateUrlWithPath(url);
                csrf = this.CMS.config.csrf;
            }

            let data = {
                csrfmiddlewaretoken: csrf,
                _save: 'Save'
            };
            if (field) {
                // FormField data
                data[field] = el.dataset.cmsType === 'HTMLField' ? html : el.textContent ;
            } else {
                // Plugin data
                data.body = html;
                data.json = JSON.stringify(json) || '';
            }

            fetch(url, {
                method: 'POST',
                body: new URLSearchParams(data),
            })
                .then(response => {
                        if (action !== undefined) {
                            action(el, response);
                        }
                        if (this.CMS) {
                            this.CMS.API.Toolbar.hideLoader();
                        }
                        return response.text();
                }).then(body => {
                    // If the edited field does not force a reload, read the CMS databridge values from the response,
                    // either directly or from a script tag or from the response using regex.
                    // This depends on the exact format django CMS core returns it. This will need to be adjusted
                    // if the format changes.
                    // Fallback solution is to reload the page as djagocms-text-ckeditor used to do.
                    const dom = document.createElement('html');
                    dom.innerHTML = body;
                    const success_element = dom.querySelectorAll('div.messagelist > div.success').length > 0;
                    if (!success_element) {
                        el.dataset.changed = 'true';
                        // Collect messages
                        const domMessages = dom.querySelectorAll(
                            `.field-${field ? field : 'body'} ul.errorlist > li`
                        );
                        let messages = [];
                        domMessages.forEach((message) => {
                            messages.push(message.textContent);
                        });
                        const domField = dom.querySelectorAll(
                            `.field-${field ? field : 'body'} label`
                        );
                        if (messages.length === 0) {
                            // Maybe CMS message from error.html?
                            const errorDescription = dom.querySelector('form fieldset .description');
                            if (errorDescription) {
                                messages.push(errorDescription.textContent);
                            }
                        }
                        if (messages.length > 0 && this.CMS) {
                            this.CMS.API.Toolbar.hideLoader();
                            this.CMS.API.Messages.open({
                                message: (domField.length > 0 ? domField[0].textContent : '') + messages.join(', '),
                                error: true,
                                delay: -1,
                            });
                        }
                        return;  // No databridge to evaluate
                    }
                    if (this.CMS) {
                        // Success:
                        // Remove an error message from a previous save attempt
                        this.CMS.API.Messages.close();
                        // Show messages if any
                        const settings = this.getSettings(el);
                        if (settings.messages_url) {
                            fetch(settings.messages_url)
                                .then(response => response.json())
                                .then(messages => {
                                    let error = "success", message_text = "";
                                    for (let message of messages.messages) {
                                        if (message.level_tag === "error") {
                                            error = "error";
                                        }
                                        message_text += `<p>${message.message}</p>`;
                                    }
                                    if (message_text.length > 0) {
                                        this.CMS.API.Messages.open({
                                            message: message_text,
                                            error: error === "error",
                                        });
                                    }
                                });
                        }

                    }
                    el.dataset.changed = 'false';
                    this.processDataBridge(dom);
                    if (!this.CMS.API.Helpers.dataBridge) {
                        // No databridge found
                        this.CMS.API.Helpers.reloadBrowser('REFRESH_PAGE');
                        return;
                    }

                    if (this.CMS.settings.version.startsWith('3.')) {
                        /* Reflect dirty flag in django CMS < 4 */
                        try {
                            /* For some reason, in v3 this fails if the structure board is not open */
                            this.CMS.API.StructureBoard.handleEditPlugin(this.CMS.API.Helpers.dataBridge);
                        } catch (e) {
                            console.error(e);
                        }
                        this._loadToolbar();
                    } else {
                        this.CMS.API.StructureBoard.handleEditPlugin(this.CMS.API.Helpers.dataBridge);
                    }
                })
                .catch(error => {
                    el.dataset.changed = 'true';
                    if (this.CMS) {
                        this.CMS.API.Toolbar.hideLoader();
                        this.CMS.API.Messages.open({
                            message: error.message,
                            error: true,
                            delay: -1,
                        });
                    }
                    window.console.error(error.message);
                    window.console.log(error.stack);
                });
        }
    }

    processDataBridge(dom) {
        const script = dom.querySelector('script#data-bridge');

        if (script && script.textContent.length > 2) {
            this.CMS.API.Helpers.dataBridge = JSON.parse(script.textContent);
        } else {
            const html = dom.innerHTML;
            const regex1 = /^\s*Window\.CMS\.API\.Helpers\.dataBridge\s=\s(.*?);$/gmu.exec(html);
            const regex2 = /^\s*Window\.CMS\.API\.Helpers\.dataBridge\.structure\s=\s(.*?);$/gmu.exec(html);

            if (regex1 && regex2 && this.CMS) {
                this.CMS.API.Helpers.dataBridge = JSON.parse(regex1[1]);
                this.CMS.API.Helpers.dataBridge.structure = JSON.parse(regex2[1]);
            } else {
                // No databridge found
                this.CMS.API.Helpers.dataBridge = null;
            }
        }
        // Additional content for the page disrupts inline editing and needs to be removed
        if (this.CMS.API.Helpers.dataBridge?.structure?.content) {
            delete this.CMS.API.Helpers.dataBridge.structure.content;
        }
    }

    // CMS Editor: addPluginForm
    // Get form for a new child plugin
    addPluginForm (plugin_type, iframe, el , onLoad, onSave) {
        const settings = this.getSettings(el);
        const data = {
            placeholder_id: settings.placeholder_id,
            plugin_type: plugin_type,
            plugin_parent: settings.plugin_id,
            plugin_language: settings.plugin_language,
            plugin_position: settings.plugin_position + 1,
            cms_path: window.parent.location.pathname,
            cms_history: 0,
        };
        const url = `${settings.add_plugin_url}?${new URLSearchParams(data).toString()}`;
        return this.loadPluginForm(url, iframe, el, onLoad, onSave);
    }

    // CMS Editor: addPluginForm
    // Get form for a new child plugin
    editPluginForm (plugin_id, iframe, el, onLoad, onSave) {
        let url = el.dataset.cmsEditUrl || window.location.href;
        url = url.replace(/\/edit-plugin\/\d+/, '/edit-plugin/' + plugin_id);
        const data = {
            '_popup': 1,
            cms_path: window.parent.location.pathname,
            cms_history: 0,
        };
        url = `${url}?${new URLSearchParams(data).toString()}`;
        return this.loadPluginForm(url, iframe, el, onLoad, onSave);
    }

    loadPluginForm (url, iframe, el, onLoad, onSave) {
        iframe.addEventListener('load', () => {
            const form = iframe.contentDocument;
            const heading = form.querySelector('#content h1');
            const submitrow = form.querySelector('.submit-row');

            // Remove submit button and heading
            if (submitrow) {
                submitrow.style.display = 'none';
            }
            if (heading) {
                heading.style.display = 'none';
            }

            //
            let saveSuccess = !!form.querySelector('div.messagelist div.success');
            if (!saveSuccess) {
                saveSuccess =
                    !!form.querySelector('.dashboard #content-main') &&
                    !form.querySelector('.messagelist .error');
            }
            if (saveSuccess) {
                // Mark document and child as changed
                el.dataset.changed = 'true';
                // Hook into the django CMS dataBridge to get the details of the newly created or saved
                // plugin. For new plugins we need their id to get the content.

                this.processDataBridge(form.body);
                // Needed to update StructureBoard
                if (onSave && this.CMS.API.Helpers.dataBridge) {
                    onSave(el, form, this.CMS.API.Helpers.dataBridge);
                }
                //  Do callback
            } else if (onLoad) {
                onLoad(el, form, heading, submitrow);
            }
            // Editor-specific dialog setup goes into the callback
        });
        iframe.setAttribute('src', url);

    }

    // CMS Editor: requestPluginMarkup
    // Get HTML markup for a child plugin
    requestPluginMarkup (plugin_id, el) {
        const settings = this.getSettings(el);
        const data = {
            plugin: plugin_id,
            token: settings.action_token,
        };

        const url = `${settings.render_plugin_url}?${new URLSearchParams(data).toString()}`;

        return fetch(url, {method: 'GET'})
            .then(response => {
                if (response.status === 200) {
                    return response.text();
                }
                else if (response.status === 204)
                {
                    return null;
                }
            });
     }

    // CMS Editor: resetInlineEditors
    _resetInlineEditors () {
        this.initInlineEditors();
    }

    // CMS Editor: loadToolbar
    // Load the toolbar after saving for update
    _loadToolbar () {
        const $ = this.CMS.$;
        this.CMS.API.StructureBoard._loadToolbar()
            .done((newToolbar) => {
                this.CMS.API.Toolbar._refreshMarkup($(newToolbar).find('.cms-toolbar'));
            })
            .fail(() => this.CMS.API.Helpers.reloadBrowser());
    }

    _highlightTextplugin (pluginId) {
        const HIGHLIGHT_TIMEOUT = 100;

        if (this.CMS) {
            const $ = this.CMS.$;
            const draggable = $('.cms-draggable-' + pluginId);
            const doc = $(document);
            const currentExpandmode = doc.data('expandmode');


            // expand necessary parents
            doc.data('expandmode', false);
            draggable
                .parents('.cms-draggable')
                .find('> .cms-dragitem-collapsable:not(".cms-dragitem-expanded") > .cms-dragitem-text')
                .each((i, el) => {
                    $(el).triggerHandler(this.CMS.Plugin.click);
                });
            if (draggable.length > 0) {  // Expanded elements available
                setTimeout(function () {
                    doc.data('expandmode', currentExpandmode);
                });
                setTimeout( () => {
                    this.CMS.Plugin._highlightPluginStructure(draggable.find('.cms-dragitem:first'),
                        {successTimeout: 200, delay: 2000, seeThrough: true});
                }, HIGHLIGHT_TIMEOUT);
            }
        }
    }

    _initInlineRichText(elements, url, cls) {
        let wrapper;

        if (elements.length > 0) {
            if (elements.length === 1 && (
                elements[0].tagName === 'DIV' || // Single wrapping div
                elements[0].tagName === 'CMS-PLUGIN' ||  // Single wrapping cms-plugin tag
                elements[0].classList.contains('cms-editor-inline-wrapper')  // already wrapped
            )) {
                // already wrapped?
                wrapper = elements[0];
                wrapper.classList.add('cms-editor-inline-wrapper');
            } else {  // no, wrap now!
                wrapper = document.createElement('div');
                wrapper.classList.add('cms-editor-inline-wrapper', 'wrapped');
                wrapper.classList.add('cms-plugin', cls, 'cms-plugin-start', 'cms-plugin-end');
                wrapper = this._wrapAll(elements, wrapper, cls);
            }
            wrapper.dataset.cmsEditUrl = url;
            return wrapper;
        }
        // No elements found
        return undefined;
    }

    // Wrap wrapper around nodes
    // Just pass a collection of nodes, and a wrapper element
    _wrapAll (nodes, wrapper, cls) {
        // Cache the current parent and previous sibling of the first node.
        const parent = nodes[0].parentNode;
        const previousSibling = nodes[0].previousSibling;

        // Place each node in wrapper.
        for (const node of nodes) {
            // Remove class markers
            node.classList.remove('cms-plugin', cls, 'cms-plugin-start', 'cms-plugin-end');
            // ... and add to wrapper
            wrapper.appendChild(node);
        }

        // Place the wrapper just after the cached previousSibling,
        // or if that is null, just before the first child.
        const nextSibling = previousSibling ? previousSibling.nextSibling : parent.firstChild;
        parent.insertBefore(wrapper, nextSibling);

        return wrapper;
    }
}


// Create global editor object
window.CMS_Editor = window.CMS_Editor || new CMSEditor();


/******/ })()
;
//# sourceMappingURL=bundle.editor.min.js.map