import enum
from collections.abc import Awaitable
from collections.abc import Coroutine
from dataclasses import dataclass
from pathlib import Path
from typing import IO
from typing import Protocol
from typing import final

from pydantic import BaseModel

from goose.config import EcosystemConfig
from goose.config import EnvironmentConfig
from goose.executable_unit import ExecutableUnit
from goose.manifest import LockManifest


class InitialStage(enum.Enum):
    bootstrapped = "bootstrapped"
    frozen = "frozen"


class SyncedStage(enum.Enum):
    synced = "synced"


class SyncedState(BaseModel):
    stage: SyncedStage
    checksum: str
    ecosystem: EcosystemConfig
    bootstrapped_version: str


class InitialState(BaseModel):
    stage: InitialStage
    ecosystem: EcosystemConfig
    bootstrapped_version: str


class UninitializedState: ...


type State = SyncedState | InitialState | UninitializedState


class Bootstrap(Protocol):
    def __call__(
        self,
        *,
        config: EnvironmentConfig,
        env_path: Path,
        manifest: LockManifest | None,
    ) -> Awaitable[InitialState]: ...


class Freeze(Protocol):
    def __call__(
        self,
        *,
        config: EnvironmentConfig,
        env_path: Path,
        lock_files_path: Path,
    ) -> Awaitable[tuple[InitialState, LockManifest]]: ...


class Sync(Protocol):
    def __call__(
        self,
        *,
        config: EnvironmentConfig,
        env_path: Path,
        lock_files_path: Path,
        manifest: LockManifest,
    ) -> Awaitable[SyncedState]: ...


class RunResult(enum.Enum):
    ok = enum.auto()
    error = enum.auto()
    modified = enum.auto()


class Run(Protocol):
    def __call__(
        self,
        *,
        config: EnvironmentConfig,
        env_path: Path,
        unit: ExecutableUnit,
        buffer: IO[str],
    ) -> Coroutine[None, None, RunResult]: ...


@final
@dataclass(frozen=True, slots=True, kw_only=True)
class Backend:
    ecosystem: str
    bootstrap: Bootstrap
    """
    - Create environment if it does not exist.
    - Install basic dependencies to enable freezing.
    """
    freeze: Freeze
    """
    - Update lock file.
    """
    sync: Sync
    """
    - Install missing dependencies.
    - Uninstall obsolete dependencies.
    """
    run: Run
