import asyncio
import io
import os
import shutil
import sys
from collections.abc import Mapping
from pathlib import Path
from typing import Final
from typing import assert_never
from typing import final

from pydantic import RootModel

from goose.git.status import get_git_status

from .backend.base import InitialState
from .backend.base import RunResult
from .backend.base import State
from .backend.base import SyncedState
from .backend.base import UninitializedState
from .backend.index import load_backend
from .config import Config
from .config import EnvironmentConfig
from .config import EnvironmentId
from .executable_unit import ExecutableUnit
from .manifest import EnvironmentState
from .manifest import LockFileState
from .manifest import check_environment_state
from .manifest import check_lock_files
from .manifest import read_manifest
from .manifest import write_manifest


class NeedsFreeze(Exception): ...


_PersistedState = RootModel[SyncedState | InitialState]


def read_state(env_dir: Path) -> State:
    state_file = env_dir / "goose-state.json"
    if not state_file.exists():
        return UninitializedState()
    return _PersistedState.model_validate_json(state_file.read_bytes()).root


def write_state(env_dir: Path, state: SyncedState | InitialState) -> None:
    state_file = env_dir / "goose-state.json"
    with state_file.open("w") as fd:
        print(state.model_dump_json(), file=fd)


@final
class Environment:
    def __init__(
        self,
        config: EnvironmentConfig,
        path: Path,
        lock_files_path: Path,
        discovered_state: State,
    ) -> None:
        self.config: Final = config
        self._backend: Final = load_backend(config.ecosystem.language)
        self._path: Final = path
        self.lock_files_path: Final = lock_files_path / config.id
        # This is read initially from file-system, so we don't entirely trust
        # it. Each stage does some additional steps to verify we are in sync.
        self.state = discovered_state

    def __repr__(self) -> str:
        return f"Environment(id={self.config.id}, ecosystem={self._backend.ecosystem})"

    def check_should_teardown(self) -> bool:
        if isinstance(self.state, UninitializedState):
            return False
        environment_state = check_environment_state(
            lock_files_path=self.lock_files_path,
            config=self.config,
            state_ecosystem=self.state.ecosystem,
            bootstrapped_version=self.state.bootstrapped_version,
        )

        if (
            environment_state is EnvironmentState.config_state_mismatch
            or environment_state is EnvironmentState.manifest_state_mismatch
        ):
            return True
        elif (
            environment_state is EnvironmentState.matching
            or environment_state is EnvironmentState.config_manifest_mismatch
        ):
            return False
        else:
            assert_never(environment_state)

    def check_should_bootstrap(self) -> bool:
        if isinstance(self.state, UninitializedState):
            return True
        if not self._path.exists():
            print("State mismatch: environment does not exist")
            return False
        return False

    def check_should_freeze(self) -> bool:
        # Check if current lock files are up-to-date with dependencies
        # configured for the environment.
        state = check_lock_files(
            lock_files_path=self.lock_files_path,
            state_checksum=None,
            config=self.config,
        )

        if (
            state is LockFileState.missing_lock_file
            or state is LockFileState.manifest_lock_file_mismatch
            or state is LockFileState.config_manifest_mismatch
        ):
            return True
        elif (
            state is LockFileState.matching
            # Mismatch between state and manifest needs _sync_ needs to run, but
            # does not indicate lock files are out of sync with configured
            # dependencies. So no need to run freeze again.
            or state is LockFileState.state_manifest_mismatch
        ):
            return False
        else:
            assert_never(state)

    def check_should_sync(self) -> bool:
        if not isinstance(self.state, SyncedState):
            return True

        state = check_lock_files(
            lock_files_path=self.lock_files_path,
            state_checksum=self.state.checksum,
            config=self.config,
        )

        if state is LockFileState.matching:
            return False
        elif state is LockFileState.missing_lock_file:
            print(
                f"[{self.config.id}] Expected lock file is missing.",
                file=sys.stderr,
            )
            return True
        elif state is LockFileState.state_manifest_mismatch:
            print(
                f"[{self.config.id}] Environment state does not match manifest.",
                file=sys.stderr,
            )
            return True
        elif state is LockFileState.manifest_lock_file_mismatch:
            raise RuntimeError(
                "Manifest does not match lock file, needs freezing. "
                "This should not normally occur, as freezing is always "
                "checked before syncing."
            )
        elif state is LockFileState.config_manifest_mismatch:
            raise RuntimeError(
                "Manifest does not match configuration, needs freezing. "
                "This should not normally occur, as freezing is always "
                "checked before syncing."
            )
        else:
            assert_never(state)

    async def teardown(self) -> None:
        await asyncio.to_thread(shutil.rmtree, self._path)
        self.state = UninitializedState()

    async def bootstrap(self) -> None:
        try:
            manifest = read_manifest(self.lock_files_path)
        except FileNotFoundError:
            manifest = None

        self.state = await self._backend.bootstrap(
            env_path=self._path,
            config=self.config,
            manifest=manifest,
        )
        write_state(self._path, self.state)

    async def freeze(self) -> None:
        self.lock_files_path.mkdir(exist_ok=True)
        self.state, manifest = await self._backend.freeze(
            env_path=self._path,
            config=self.config,
            lock_files_path=self.lock_files_path,
        )
        write_manifest(self.lock_files_path, manifest)
        write_state(self._path, self.state)
        os.sync()

    async def sync(self) -> None:
        manifest = read_manifest(self.lock_files_path)
        self.state = await self._backend.sync(
            env_path=self._path,
            config=self.config,
            lock_files_path=self.lock_files_path,
            manifest=manifest,
        )
        write_state(self._path, self.state)

    async def run(self, unit: ExecutableUnit, verbose: bool) -> RunResult:
        buffer = io.StringIO()
        coroutine = self._backend.run(
            env_path=self._path,
            config=self.config,
            unit=unit,
            buffer=buffer,
        )

        async def _run() -> RunResult:
            # We don't track modifications for read-only hooks.
            if unit.hook.read_only:
                return await coroutine

            status_prior = await get_git_status(unit.targets)
            result = await coroutine
            if result is RunResult.error:
                return result
            status_post = await get_git_status(unit.targets)
            if status_prior != status_post:
                return RunResult.modified
            return result

        result = await _run()

        if verbose or result is not RunResult.ok:
            value = buffer.getvalue().rstrip()
            if value:
                print(value, file=sys.stderr)

        return result


def build_environments(
    config: Config,
    env_dir: Path,
    lock_files_path: Path,
) -> Mapping[EnvironmentId, Environment]:
    environments = {}
    for cfg in config.environments:
        path = env_dir / cfg.id
        environments[cfg.id] = Environment(
            config=cfg,
            path=path,
            lock_files_path=lock_files_path,
            discovered_state=read_state(path),
        )
    return environments


async def prepare_environment(
    environment: Environment,
    upgrade: bool = False,
    verbose: bool = False,
) -> None:
    log_prefix = f"[{environment.config.id}] "

    if environment.check_should_teardown():
        print(
            f"{log_prefix}Environment needs rebuilding, tearing down ...",
            file=sys.stderr,
        )
        await environment.teardown()
        print(
            f"{log_prefix}Environment deleted.",
            file=sys.stderr,
        )

    if environment.check_should_bootstrap():
        print(f"{log_prefix}Bootstrapping environment ...", file=sys.stderr)
        await environment.bootstrap()
        print(f"{log_prefix}Bootstrapping done.", file=sys.stderr)
    elif verbose:
        print(
            f"{log_prefix}Found previously bootstrapped environment.",
            file=sys.stderr,
        )

    if upgrade:
        print(f"{log_prefix}Freezing dependencies ...", file=sys.stderr)
        await environment.freeze()
        print(f"{log_prefix}Freezing done.")
    elif environment.check_should_freeze():
        print(f"{log_prefix}Missing lock files.", file=sys.stderr)
        raise NeedsFreeze
    elif verbose:
        print(f"{log_prefix}Found existing lock files up-to-date.", file=sys.stderr)

    if environment.check_should_sync():
        print(f"{log_prefix}Syncing dependencies ...", file=sys.stderr)
        await environment.sync()
        print(f"{log_prefix}Syncing done.", file=sys.stderr)
    elif verbose:
        print(
            f"{log_prefix}Found dependencies up-to-date.",
            file=sys.stderr,
        )
