"""Test bulk operations integration with DataFlow."""

import sys
from pathlib import Path
from unittest.mock import AsyncMock, Mock, patch

import pytest

# Add parent directory to path
sys.path.insert(0, str(Path(__file__).parent.parent))

from dataflow import DataFlow

from kailash.runtime.local import LocalRuntime
from kailash.workflow.builder import WorkflowBuilder


@pytest.mark.asyncio
async def test_dataflow_bulk_create():
    """Test that DataFlow generates bulk create nodes correctly."""
    # Use PostgreSQL for testing - AsyncSQLDatabaseNode is PostgreSQL-only
    db = DataFlow("postgresql://test_user:test_password@localhost:5434/kailash_test")
    runtime = LocalRuntime()

    @db.model
    class Product:
        name: str
        price: float
        category: str
        in_stock: bool = True

    # Verify bulk nodes were generated
    assert "bulk_create" in db._model_nodes["Product"]
    assert "bulk_update" in db._model_nodes["Product"]
    assert "bulk_delete" in db._model_nodes["Product"]

    # Test bulk create with mocked adapter
    workflow = WorkflowBuilder()

    # Mock the adapter to avoid actual database operations
    mock_adapter = AsyncMock()
    mock_adapter.execute = AsyncMock(return_value=3)  # 3 records inserted

    # Import the generated node
    # Nodes are dynamically generated by DataFlow
    # ProductBulkCreateNode will be available after model registration

    # Create test workflow with mocked adapter
    with patch.object(ProductBulkCreateNode, "_get_adapter", return_value=mock_adapter):
        workflow.add_node(
            "ProductBulkCreateNode",
            "bulk_insert",
            {
                "records": [
                    {"name": "Product 1", "price": 10.99, "category": "electronics"},
                    {"name": "Product 2", "price": 20.99, "category": "electronics"},
                    {"name": "Product 3", "price": 30.99, "category": "electronics"},
                ]
            },
        )

        results, _ = await runtime.execute_async(workflow.build())

        assert results["bulk_insert"]["status"] == "success"
        assert results["bulk_insert"]["successful_records"] == 3
        assert results["bulk_insert"]["total_records"] == 3


@pytest.mark.asyncio
async def test_dataflow_node_registration():
    """Test that DataFlow nodes are properly registered with NodeRegistry."""
    db = DataFlow("postgresql://test_user:test_password@localhost:5434/kailash_test")

    @db.model
    class User:
        name: str
        email: str

    # Test that nodes are registered and can be accessed
    assert "bulk_create" in db._model_nodes["User"]
    assert "bulk_update" in db._model_nodes["User"]
    assert "bulk_delete" in db._model_nodes["User"]

    # Test import via module
    # Nodes are dynamically generated by DataFlow
    # UserBulkCreateNode will be available after model registration

    assert UserBulkCreateNode is not None

    # Verify node has correct configuration
    node = UserBulkCreateNode()
    assert hasattr(node, "model_name")
    assert node.model_name == "User"
    assert hasattr(node, "operation")
    assert node.operation == "bulk_create"


if __name__ == "__main__":
    import asyncio

    async def run_all_tests():
        await test_dataflow_bulk_create()
        await test_dataflow_node_registration()
        print("All tests passed!")

    asyncio.run(run_all_tests())
