
# ProductCategorization.com Python Client

[![PyPI version](https://badge.fury.io/py/productcategorization.svg)](https://pypi.org/project/productcategorization/)
[![API Docs](https://www.productcategorization.com/api.php)](https://www.productcategorization.com/product_categorization_api.php)

---

## Overview

The `productcategorization` Python package provides seamless access to one of the world's most advanced product categorization APIs, powering e-commerce classification for unicorn startups, multinational enterprises, retail analytics platforms, adTech innovators, and online merchants. Whether you operate an e-commerce storefront, marketplace, or SaaS platform, this package allows you to integrate AI-powered categorization directly into your Python applications, unlocking world-class product, URL, and image classification using industry-standard taxonomies.

## Key Features

* **Ultra-Accurate Product Categorization**
  Classify product titles, descriptions, and URLs using:

  * **Google Shopping Taxonomy:** Over 5,500 hierarchical categories for granular and up-to-date mapping.
  * **Shopify Taxonomy:** Leverage the latest Shopify category structure with \~11,000 fine-grained categories.
  * **Amazon and Other Standard Taxonomies:** Flexibility for diverse retail needs.
  * **Custom Taxonomies:** Tailor classifiers to your unique vertical or proprietary taxonomy.

* **Multi-Modal Classification**

  * **Text**: Classify any product-related string.
  * **URL**: Categorize products directly from their web pages.
  * **Image**: Obtain Shopify categories and attribute extraction directly from images (using AI vision).

* **Buyer Persona Enrichment**
  Every classification returns relevant buyer personas—select from a proprietary library of over 1,800 personas to enrich your analytics, personalization, or marketing automations.
  Confidence scores and expanded context available.

* **High Scalability and Reliability**
  Robust API supporting high throughput (rate limits adjustable upon request), with credit-based billing for predictable scaling.

* **Plug-and-Play Python Integration**
  Simple, modern, and extensible Python API client.
  See [Quickstart](#quickstart) for usage examples.

---

## Table of Contents

* [Getting Started](#getting-started)
* [Authentication](#authentication)
* [API Usage](#api-usage)

  * [Text Categorization](#text-categorization)
  * [URL Categorization](#url-categorization)
  * [Image Categorization](#image-categorization)
* [Advanced Options](#advanced-options)

  * [Buyer Personas and Confidence Scores](#buyer-personas-and-confidence-scores)
  * [Context Expansion](#context-expansion)
* [Error Handling](#error-handling)
* [Best Practices](#best-practices)
* [Integration Examples](#integration-examples)
* [Contact & Support](#contact--support)
* [Related Services](#related-services)
* [References](#references)

---

## Getting Started

Install the package via PyPI:

```bash
pip install productcategorization
```

Or add it to your `requirements.txt` for automatic deployment.

---

## Authentication

All API access is secured by a personal API key.
To obtain your API key:

1. Sign up and purchase a subscription at [www.productcategorization.com](https://www.productcategorization.com/pricing.php
3. Provide the API key in every request (see examples).

> **Note:** Never share your API key publicly. Store it securely as an environment variable or in your configuration files.

---

## API Usage

### Text Categorization

Classify any product text (title, description, or keyword) in a single line:

```python
from productcategorization import ProductCategorizationAPI

api = ProductCategorizationAPI(api_key="your_api_key")
result = api.categorize_text("Fluorescent Highlighters 3pc Yellow")
print(result)
```

**Sample Response:**

```json
{
    "total_credits": 100044,
    "remaining_credits": 33075,
    "language": "en",
    "classification": "Office Supplies > Office Instruments > Writing & Drawing Instruments",
    "buyer_personas": [
        "Business Professional", "Office Professional", "Administrative Coordinator", ...
    ],
    "buyer_personas_confidence_selection": {
        "Office Professional": 0.9,
        "Business Professional": 0.8,
        ...
    },
    "ID": "977",
    "status": 200
}
```

**Parameters:**

* `query` (str): Product text for categorization.
* `confidence` (optional, int): Set to `1` to include confidence scores for each persona.
* `expand_context` (optional, int): Set to `1` to auto-generate expanded context for short/ambiguous texts.

---

### URL Categorization

You can also classify products by URL, leveraging our AI’s ability to extract relevant text and metadata:

```python
result = api.categorize_url("https://www.apple.com")
print(result)
```

**Sample Python (requests):**

```python
import requests

payload = {'query': 'www.apple.com', 'api_key': 'your_api_key', 'data_type': 'url'}
response = requests.post("https://www.productcategorization.com/api/iab/iab_web_content_filtering_url.php", data=payload)
print(response.json())
```

---

### Image Categorization

Classify products using image URLs or local image files (Shopify Taxonomy + attribute extraction):

```python
result = api.categorize_image(image_url="https://images.com/product.jpg", text="Product title")
print(result)
```

**Example Function:**

```python
import requests
import io

def call_api(image_url, text, api_key):
    api_endpoint = 'https://www.productcategorization.com/api/ecommerce/ecommerce_shopify_image.php'
    response = requests.get(image_url)
    if response.status_code != 200:
        return {'error': 'Failed to download image'}
    image_file = io.BytesIO(response.content)
    data = {'ip': '0', 'api_key': api_key, 'login': '0', 'text': text}
    files = {'image': ('image.jpg', image_file, 'image/jpeg')}
    response = requests.post(api_endpoint, data=data, files=files)
    return response.json()
```

---

## Advanced Options

### Buyer Personas and Confidence Scores

Our AI delivers a unique set of buyer personas for every product—ideal for market analysis, targeted marketing, or persona-based analytics.
Enable confidence scoring to obtain relevance weights for each persona:

```python
result = api.categorize_text("Eco-Friendly Notebook", confidence=1)
print(result["buyer_personas_confidence_selection"])
```

### Context Expansion

For short or ambiguous inputs, enable `expand_context=1` to let our AI generate an enhanced description for improved classification accuracy:

```python
result = api.categorize_text("3pc Yellow Highlighters", expand_context=1)
print(result["expanded_context"])
```

---

## Error Handling

All API responses include a `status` code for programmatic error handling:

| Status | Meaning                                  |
| ------ | ---------------------------------------- |
| 200    | Request was successful                   |
| 400    | Request malformed (check parameters)     |
| 401    | Invalid API key (check or purchase key)  |
| 403    | Quota exhausted (upgrade or add credits) |

Example error handling in Python:

```python
if result["status"] != 200:
    print(f"API Error: {result.get('message', 'Unknown error')}")
```

---

## Best Practices

* **Monitor Remaining Credits:** Every response includes `total_credits` and `remaining_credits`. Plan your usage to avoid interruptions.
* **Respect Rate Limits:** Default is 60 requests per minute. Contact support for higher needs.
* **Secure Your API Key:** Do not embed directly in code if publishing open-source.
* **Use Context Expansion When Needed:** For short/ambiguous product titles, enable `expand_context`.
* **Batch Requests:** For large datasets, consider batching requests and handling quota gracefully.

---

## Integration Examples

### Python Example

```python
from productcategorization import ProductCategorizationAPI

api = ProductCategorizationAPI(api_key="your_api_key")
result = api.categorize_text("Fluorescent Highlighters 3pc Yellow")
print(result["classification"])
```

### JavaScript Example

```javascript
const apiBaseUrl = "https://www.productcategorization.com/api/ecommerce/ecommerce_category6_get.php?";
const apiKey = "your_api_key";
const queryText = "Fluorescent Highlighters 3pc Yellow";
const encodedQueryText = encodeURIComponent(queryText);
const finalUrl = `${apiBaseUrl}query=${encodedQueryText}&api_key=${apiKey}`;

fetch(finalUrl)
  .then(response => response.json())
  .then(data => console.log(data));
```

### Ruby Example

```ruby
require 'uri'
require 'net/http'

api_base_url = "https://www.productcategorization.com/api/ecommerce/ecommerce_category6_get.php"
api_key = "your_api_key"
query_text = "Fluorescent Highlighters 3pc Yellow"

encoded_query = URI.encode_www_form_component(query_text)
url = URI("#{api_base_url}?query=#{encoded_query}&api_key=#{api_key}")

response = Net::HTTP.get(url)
puts response
```

### C# Example

```csharp
using System;
using System.Net.Http;
using System.Threading.Tasks;

class Program {
    static async Task Main(string[] args) {
        var apiBaseUrl = "https://www.productcategorization.com/api/ecommerce/ecommerce_category6_get.php?";
        var apiKey = "your_api_key";
        var queryText = "Fluorescent Highlighters 3pc Yellow";
        var encodedQueryText = Uri.EscapeDataString(queryText);
        var finalUrl = $"{apiBaseUrl}query={encodedQueryText}&api_key={apiKey}";

        using (HttpClient client = new HttpClient()) {
            var response = await client.GetStringAsync(finalUrl);
            Console.WriteLine(response);
        }
    }
}
```

---

## Contact & Support

Need a higher rate limit, a custom classifier, or additional support?
Visit [Contact](https://www.productcategorization.com/contact), or email support via your account dashboard.

---

## Related Services

Leverage our broader suite of AI-powered APIs to cover every aspect of your business’s data intelligence and privacy needs:

* **[Comment Moderation API](https://www.contentmoderationapi.net) – comment moderation api:**
  Safeguard your community, app, or platform with industry-leading AI moderation for comments and user-generated content. Detect profanity, hate speech, spam, and toxicity in real time.

* **[Live Video Anonymization](https://www.anonymizationapi.com) – live video anonymization:**
  Protect privacy with automatic anonymization of faces and sensitive objects in live video streams, supporting GDPR compliance and safeguarding user identities.

* **[Text Redaction API](https://www.redactionapi.net) – text redaction api:**
  Redact personal data, financial information, or any sensitive fields from documents at scale using our high-precision redaction API.

* **[Company Enrichment Data](https://www.companydataapi.com) – company enrichment data:**
  Instantly enhance your CRM, sales, or analytics platform with up-to-date company profiles, firmographics, and contact data.

* **[Domain Categorization Data](https://www.urlcategorizationdatabase.com) – domain categorization data:**
  Access the world’s largest database of categorized domains for cybersecurity, web filtering, and content safety.

* **[AI Contract Analysis](https://www.aicontractreviewtool.com) – ai contract analysis:**
  Revolutionize contract review workflows with advanced AI-driven contract analysis, risk detection, and compliance assessment.

Our APIs integrate seamlessly with your product workflows, providing reliable, scalable, and secure endpoints for your business logic.

---

## References & Further Reading

For best-in-class taxonomy, AI, and categorization research, explore:

* [Stanford AI Lab](https://ai.stanford.edu)
* [MIT CSAIL](https://www.csail.mit.edu)
* [Berkeley AI Research](https://bair.berkeley.edu)
* [Oxford Internet Institute](https://www.oii.ox.ac.uk)
* [UCL Centre for Artificial Intelligence](https://www.ucl.ac.uk/ai)
* [Google AI Blog](https://ai.googleblog.com/)
* [Microsoft Research](https://www.microsoft.com/en-us/research/)
* [arXiv Machine Learning](https://arxiv.org/list/cs.LG/recent)

For taxonomy standards and e-commerce data:

* [Google Shopping Taxonomy](https://support.google.com/merchants/answer/6324436)
* [Shopify Product Taxonomy](https://github.com/Shopify/product-taxonomy)

---

## License

This library is distributed under the MIT License.

---

## Disclaimer

This project is unaffiliated with Google, Shopify, or Amazon.
All trademarks are property of their respective owners.

---

# `__init__.py` Example

```python
import requests

class ProductCategorizationAPI:
    def __init__(self, api_key):
        self.api_key = api_key
        self.base_url = "https://www.productcategorization.com/api/"
    
    def categorize_text(self, text, confidence=0, expand_context=0):
        params = {
            "query": text,
            "api_key": self.api_key,
            "confidence": str(confidence),
            "expand_context": str(expand_context)
        }
        response = requests.get(self.base_url + "ecommerce/ecommerce_category6_get.php", params=params)
        return response.json()

    def categorize_url(self, url):
        payload = {
            'query': url,
            'api_key': self.api_key,
            'data_type': 'url'
        }
        headers = {
            'Content-Type': 'application/x-www-form-urlencoded'
        }
        response = requests.post(self.base_url + "iab/iab_web_content_filtering_url.php", data=payload, headers=headers)
        return response.json()

    def categorize_image(self, image_url, text="", ip="0", login="0"):
        # Download image to memory
        image_response = requests.get(image_url)
        if image_response.status_code != 200:
            return {'error': 'Failed to download image'}
        import io
        image_file = io.BytesIO(image_response.content)
        data = {
            'ip': ip,
            'api_key': self.api_key,
            'login': login,
            'text': text
        }
        files = {
            'image': ('image.jpg', image_file, 'image/jpeg')
        }
        response = requests.post(self.base_url + "ecommerce/ecommerce_shopify_image.php", data=data, files=files)
        return response.json()
```

---

