"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.HyperledgerFabricIdentity = void 0;
const child_process_1 = require("child_process");
const path = require("path");
const cdk = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const customresources = require("aws-cdk-lib/custom-resources");
const constructs = require("constructs");
const utilities = require("./utilities");
/**
 * Creates custom resources to enroll admin and register user
 * identities with the CA using the fabric-ca-client SDK.
 * Admin identity is enrolled by default. User identities are
 * registered and enrolled, if provided.
 */
class HyperledgerFabricIdentity extends constructs.Construct {
    constructor(scope, id) {
        super(scope, id);
        // Collect metadata on the stack
        const partition = cdk.Stack.of(this).partition;
        const region = cdk.Stack.of(this).region;
        // Retrieve the S3 Bucket and key that contains the TLS cert file
        const tlsBucketData = utilities.getTlsBucket(region);
        const adminPasswordArn = scope.adminPasswordSecret.secretArn;
        const adminPrivateKeyArn = scope.adminPrivateKeySecret.secretArn;
        const adminSignedCertArn = scope.adminSignedCertSecret.secretArn;
        const caEndpoint = scope.caEndpoint;
        const client = scope.client;
        const memberName = scope.memberName;
        // Role for the custom resource lambda functions
        const customResourceRole = new iam.Role(this, 'CustomResourceRole', { assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com') });
        // Policies for the custom resource lambda to enroll and register users
        customResourceRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        customResourceRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        customResourceRole.addToPolicy(new iam.PolicyStatement({
            actions: ['s3:GetObject', 'secretsmanager:CreateSecret', 'secretsmanager:GetSecretValue', 'secretsmanager:PutSecretValue'],
            resources: [
                `arn:${partition}:s3:::${tlsBucketData.bucketName}/*`,
                adminPasswordArn,
                adminPrivateKeyArn,
                adminSignedCertArn,
            ],
        }));
        const lambdaDirectory = '../lambdas/fabric';
        // Have to use docker local bundling as esbuild doesn't resolve the path on the fabric-proto package
        const codeProp = {
            bundling: {
                image: lambda.Runtime.NODEJS_14_X.bundlingImage,
                command: [
                    'bash', '-c', 'cp -a . /asset-output',
                    'npm install --prefix /asset-output',
                ],
                local: {
                    tryBundle(outputDir) {
                        try {
                            child_process_1.execSync('npm --version');
                            child_process_1.execSync(`cp -a ${path.join(__dirname, lambdaDirectory)}/. ${outputDir}`);
                            child_process_1.execSync(`npm install --prefix ${outputDir}`);
                        }
                        catch {
                            return false;
                        }
                        return true;
                    },
                },
            },
        };
        // Lambda function to enroll the admin and import credentials to secrets manager.
        const adminFunction = new lambda.Function(this, 'AdminFunction', {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'enroll-admin.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, lambdaDirectory), codeProp),
            environment: {
                ADMIN_PASSWORD_ARN: adminPasswordArn,
                CA_ENDPOINT: caEndpoint,
                PRIVATE_KEY_ARN: adminPrivateKeyArn,
                SIGNED_CERT_ARN: adminSignedCertArn,
                TLS_CERT_BUCKET: tlsBucketData.bucketName,
                TLS_CERT_KEY: tlsBucketData.key,
            },
            role: customResourceRole,
            vpc: client.vpc,
            vpcSubnets: client.vpc.selectSubnets(),
            timeout: cdk.Duration.minutes(1),
        });
        // Port range to access the Network
        const ledgerPortRange = ec2.Port.tcpRange(utilities.STARTING_PORT, utilities.ENDING_PORT);
        // Add access to the lambda for the Network ports
        client.vpcEndpoint.connections.allowFrom(adminFunction, ledgerPortRange);
        // Custom Resource provider
        this.adminProvider = new customresources.Provider(this, 'AdminProvider', {
            onEventHandler: adminFunction,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        // Lambda function to register and enroll users and
        // import credentials to secrets manager.
        const userFunction = new lambda.Function(scope, 'UserFunction', {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'register-user.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, lambdaDirectory), codeProp),
            environment: {
                CA_ENDPOINT: caEndpoint,
                MEMBER_NAME: memberName,
                PRIVATE_KEY_ARN: adminPrivateKeyArn,
                SIGNED_CERT_ARN: adminSignedCertArn,
                TLS_CERT_BUCKET: tlsBucketData.bucketName,
                TLS_CERT_KEY: tlsBucketData.key,
            },
            role: customResourceRole,
            vpc: client.vpc,
            vpcSubnets: client.vpc.selectSubnets(),
            timeout: cdk.Duration.minutes(1),
        });
        // Add access to the lambda for the Network ports
        scope.client.vpcEndpoint.connections.allowFrom(userFunction, ledgerPortRange);
        // Custom Resource provider
        HyperledgerFabricIdentity.userProvider = new customresources.Provider(scope, 'UserProvider', {
            onEventHandler: userFunction,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        // Populate the custom role static variable
        HyperledgerFabricIdentity.customRole = customResourceRole;
    }
}
exports.HyperledgerFabricIdentity = HyperledgerFabricIdentity;
//# sourceMappingURL=data:application/json;base64,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