"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateString = exports.validateInteger = exports.validateAvailabilityZone = exports.validateRegion = exports.getTlsBucket = exports.ENDING_PORT = exports.STARTING_PORT = exports.SUPPORTED_AVAILABILITY_ZONES = exports.SUPPORTED_REGIONS = void 0;
/**
 * Regions where Managed Blockchain is supported, for details see
 * https://aws.amazon.com/managed-blockchain/pricing/hyperledger/
 */
exports.SUPPORTED_REGIONS = [
    'ap-northeast-1',
    'ap-northeast-2',
    'ap-southeast-1',
    'eu-west-1',
    'eu-west-2',
    'us-east-1',
];
/**
 * Map of supported regions to their availability zones
 */
exports.SUPPORTED_AVAILABILITY_ZONES = {
    'ap-northeast-1': [
        'ap-northeast-1a',
        'ap-northeast-1b',
        'ap-northeast-1c',
    ],
    'ap-northeast-2': [
        'ap-northeast-2a',
        'ap-northeast-2b',
        'ap-northeast-2c',
        'ap-northeast-2d',
    ],
    'ap-southeast-1': [
        'ap-southeast-1a',
        'ap-southeast-1b',
        'ap-southeast-1c',
    ],
    'eu-west-1': [
        'eu-west-1a',
        'eu-west-1b',
        'eu-west-1c',
    ],
    'eu-west-2': [
        'eu-west-2a',
        'eu-west-2b',
        'eu-west-2c',
    ],
    'us-east-1': [
        'us-east-1a',
        'us-east-1b',
        'us-east-1c',
        'us-east-1d',
        'us-east-1e',
        'us-east-1f',
    ],
};
/**
 * Starting port of the Network port range
 */
exports.STARTING_PORT = 30001;
/**
 * Ending port of the Network port range
 */
exports.ENDING_PORT = 30004;
/**
 * Returns the S3 Bucket and key that contains the TLS cert file
 */
function getTlsBucket(region) {
    return { bucketName: `${region}.managedblockchain`, key: 'etc/managedblockchain-tls-chain.pem' };
}
exports.getTlsBucket = getTlsBucket;
/**
 * Throw an error if provided region is not in the supported list
 */
function validateRegion(region) {
    if (!exports.SUPPORTED_REGIONS.includes(region)) {
        const regionList = exports.SUPPORTED_REGIONS.join(', ');
        throw new Error(`Managed Blockchain is only available in the following regions: ${regionList}.`);
    }
}
exports.validateRegion = validateRegion;
/**
 * Throw an error if provided availability zone is not in the supported list
 */
function validateAvailabilityZone(region, availabilityZone) {
    const availabililtyZonesForRegion = exports.SUPPORTED_AVAILABILITY_ZONES[region];
    if (typeof availabilityZone === 'undefined' || !availabililtyZonesForRegion.includes(availabilityZone)) {
        const availabilityZoneList = availabililtyZonesForRegion.join(', ');
        throw new Error(`Managed Blockchain in ${region} is only available in the following availability zones: ${availabilityZoneList}.`);
    }
}
exports.validateAvailabilityZone = validateAvailabilityZone;
/**
 * Throw an error if provided number is not an integer, or not with the given range (inclusive)
 */
function validateInteger(value, min, max) {
    if (!Number.isInteger(value))
        return false;
    if (value < min || value > max)
        return false;
    return true;
}
exports.validateInteger = validateInteger;
/**
 * Throw an error if provided string has length with a given range (inclusive),
 * and optionally matches a provided regular expression pattern
 */
function validateString(value, min, max, regexp) {
    if (value.length < min || value.length > max)
        return false;
    if (typeof regexp !== 'undefined' && !value.match(regexp))
        return false;
    return true;
}
exports.validateString = validateString;
//# sourceMappingURL=data:application/json;base64,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