
# ****************************************************************************
# Copyright 2023 Technology Innovation Institute
# 
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
# ****************************************************************************


from claasp.cipher import Cipher
from claasp.name_mappings import INPUT_KEY, INPUT_PLAINTEXT

PARAMETERS_CONFIGURATION_LIST = [{'number_of_rounds': 11}]


class BEA1BlockCipher(Cipher):
    """
    Return a cipher object of BEA-1 Block Cipher.

    INPUT:

    - ``number_of_rounds`` -- **integer** (default: `11`); number of rounds of the cipher

    EXAMPLES::

        sage: from claasp.ciphers.block_ciphers.bea1_block_cipher import BEA1BlockCipher
        sage: bea = BEA1BlockCipher()
        sage: key = 0x8cdd0f3459fb721e798655298d5c1
        sage: plaintext = 0x47a57eff5d6475a68916
        sage: ciphertext = 0x439d5298656eccc67dee
        sage: bea.evaluate([key, plaintext]) == ciphertext
        True
    """

    def __init__(self, number_of_rounds=11):

        self.KEY_BLOCK_SIZE = 120
        self.CIPHER_BLOCK_SIZE = 80
        self.SBOX_BIT_SIZE = 10
        self.NROUNDS = number_of_rounds

        super().__init__(family_name='bea1_block_cipher', 
                         cipher_type='block_cipher', 
                         cipher_inputs=[INPUT_KEY, INPUT_PLAINTEXT],
                         cipher_inputs_bit_size=[self.KEY_BLOCK_SIZE, self.CIPHER_BLOCK_SIZE],
                         cipher_output_bit_size=self.CIPHER_BLOCK_SIZE)
        
        self.sboxes = {
            0: [
                0x0BA, 0x026, 0x0A0, 0x1E1, 0x183, 0x3DB, 0x1A4, 0x083, 0x110, 0x350, 0x085, 0x2E5, 0x3B4, 0x195, 0x359, 0x2E6,
                0x33A, 0x26B, 0x209, 0x217, 0x1CE, 0x2E3, 0x0C0, 0x136, 0x129, 0x0C8, 0x3D6, 0x054, 0x040, 0x3F2, 0x09F, 0x322,
                0x11B, 0x07F, 0x139, 0x07D, 0x2CF, 0x02A, 0x268, 0x227, 0x246, 0x1C5, 0x12B, 0x3B6, 0x16C, 0x20D, 0x1E7, 0x35B,
                0x313, 0x0CD, 0x11E, 0x1E6, 0x117, 0x355, 0x182, 0x0E6, 0x094, 0x1B9, 0x19C, 0x28C, 0x2B9, 0x336, 0x0AF, 0x19D,
                0x2BC, 0x1A9, 0x31B, 0x02E, 0x282, 0x2AE, 0x272, 0x2E9, 0x3AA, 0x1DD, 0x013, 0x2D3, 0x30F, 0x35A, 0x159, 0x1BB,
                0x11C, 0x12A, 0x248, 0x3C7, 0x28B, 0x191, 0x025, 0x173, 0x018, 0x38D, 0x1A1, 0x185, 0x007, 0x156, 0x378, 0x312,
                0x0C9, 0x143, 0x05D, 0x3FA, 0x038, 0x3DE, 0x081, 0x0F9, 0x2D1, 0x3FB, 0x1C7, 0x3E0, 0x1DC, 0x16A, 0x2D8, 0x23F,
                0x030, 0x1EB, 0x3AF, 0x311, 0x36D, 0x3BD, 0x3C9, 0x348, 0x261, 0x1AF, 0x071, 0x3EE, 0x3BA, 0x3AB, 0x1B8, 0x3CA,
                0x22B, 0x118, 0x279, 0x0F6, 0x3FF, 0x122, 0x1B2, 0x360, 0x1D6, 0x1B6, 0x3D4, 0x3BB, 0x3B3, 0x0EA, 0x097, 0x308,
                0x3A9, 0x086, 0x0AE, 0x15A, 0x253, 0x058, 0x0BB, 0x3D5, 0x01D, 0x1A3, 0x23E, 0x053, 0x35D, 0x277, 0x384, 0x0E2,
                0x233, 0x2B8, 0x2AF, 0x0D0, 0x1B1, 0x105, 0x0B3, 0x215, 0x2A2, 0x27F, 0x2DB, 0x17E, 0x12C, 0x3A2, 0x18E, 0x2AC,
                0x321, 0x09C, 0x294, 0x04C, 0x036, 0x2F1, 0x3D2, 0x18D, 0x188, 0x349, 0x128, 0x069, 0x198, 0x2F4, 0x3DC, 0x370,
                0x138, 0x324, 0x23C, 0x1FD, 0x082, 0x247, 0x005, 0x0A3, 0x0F0, 0x273, 0x152, 0x17B, 0x1A0, 0x1C8, 0x04E, 0x34C,
                0x12F, 0x0CC, 0x075, 0x10E, 0x290, 0x021, 0x1AE, 0x211, 0x3E6, 0x17A, 0x276, 0x289, 0x3B5, 0x123, 0x01F, 0x048,
                0x201, 0x08F, 0x29A, 0x002, 0x179, 0x32E, 0x120, 0x1AC, 0x1E3, 0x109, 0x079, 0x37C, 0x297, 0x096, 0x12D, 0x323,
                0x165, 0x0AC, 0x18B, 0x0AB, 0x1FF, 0x230, 0x25B, 0x3D3, 0x111, 0x07E, 0x21C, 0x1BE, 0x187, 0x30E, 0x34A, 0x318,
                0x269, 0x343, 0x29F, 0x395, 0x1AD, 0x1D2, 0x023, 0x2DE, 0x1B3, 0x35E, 0x2D7, 0x044, 0x206, 0x3F1, 0x310, 0x0A7,
                0x287, 0x3C3, 0x2A5, 0x213, 0x3E4, 0x3DA, 0x0FD, 0x140, 0x38E, 0x2C2, 0x154, 0x254, 0x15F, 0x02C, 0x1FB, 0x1ED,
                0x1C6, 0x051, 0x062, 0x090, 0x214, 0x14B, 0x190, 0x15D, 0x0A1, 0x186, 0x032, 0x0B9, 0x1DA, 0x239, 0x3D1, 0x383,
                0x331, 0x06D, 0x02D, 0x009, 0x2FC, 0x3AD, 0x2AA, 0x363, 0x1EF, 0x38F, 0x39A, 0x2DC, 0x3BF, 0x106, 0x39B, 0x31F,
                0x03E, 0x0DE, 0x1BC, 0x067, 0x0CF, 0x155, 0x2CE, 0x240, 0x05E, 0x0E8, 0x0C4, 0x149, 0x08C, 0x3E5, 0x2A1, 0x150,
                0x1D1, 0x228, 0x3DF, 0x0E0, 0x3F6, 0x193, 0x19B, 0x27D, 0x2B0, 0x35C, 0x0E3, 0x171, 0x180, 0x022, 0x00E, 0x358,
                0x161, 0x0EE, 0x365, 0x15B, 0x0C3, 0x2CD, 0x3E1, 0x06C, 0x119, 0x283, 0x0F1, 0x3B9, 0x212, 0x226, 0x076, 0x382,
                0x38C, 0x1D3, 0x15C, 0x0B2, 0x22C, 0x314, 0x056, 0x216, 0x364, 0x3DD, 0x1E9, 0x020, 0x176, 0x389, 0x2F2, 0x073,
                0x06F, 0x27E, 0x027, 0x14E, 0x177, 0x26D, 0x1BA, 0x0EC, 0x25A, 0x194, 0x3C6, 0x2F9, 0x221, 0x0E1, 0x3F4, 0x0B7,
                0x14F, 0x293, 0x144, 0x0FB, 0x2F0, 0x3ED, 0x0F4, 0x1CC, 0x0C6, 0x065, 0x028, 0x315, 0x3E2, 0x2DD, 0x274, 0x0FA,
                0x0D3, 0x041, 0x080, 0x2C5, 0x072, 0x08D, 0x339, 0x2A3, 0x1F1, 0x1DF, 0x2F5, 0x267, 0x015, 0x0B1, 0x275, 0x21B,
                0x091, 0x03F, 0x259, 0x18F, 0x1C3, 0x27B, 0x319, 0x153, 0x0D2, 0x0BD, 0x2D0, 0x064, 0x000, 0x379, 0x2F6, 0x2A9,
                0x142, 0x0F5, 0x3EF, 0x03B, 0x3F8, 0x344, 0x3BC, 0x265, 0x0E7, 0x334, 0x238, 0x08E, 0x347, 0x174, 0x18C, 0x162,
                0x112, 0x1D0, 0x01C, 0x292, 0x2C0, 0x0E9, 0x2B6, 0x301, 0x0C1, 0x30D, 0x369, 0x1C0, 0x1E4, 0x1F7, 0x08A, 0x2FA,
                0x3CB, 0x34D, 0x2BE, 0x28F, 0x09A, 0x39D, 0x232, 0x262, 0x333, 0x2F8, 0x397, 0x2C4, 0x06E, 0x27A, 0x317, 0x017,
                0x327, 0x26C, 0x325, 0x167, 0x05B, 0x36C, 0x362, 0x004, 0x3F7, 0x0F7, 0x20B, 0x22D, 0x222, 0x2D2, 0x0CA, 0x196,
                0x33F, 0x3B2, 0x17D, 0x302, 0x146, 0x170, 0x367, 0x18A, 0x1DE, 0x0B5, 0x099, 0x3BE, 0x2C1, 0x0BC, 0x2A0, 0x01B,
                0x11D, 0x010, 0x342, 0x169, 0x366, 0x2EC, 0x088, 0x361, 0x291, 0x131, 0x2FF, 0x199, 0x1CA, 0x3B0, 0x00D, 0x24F,
                0x2B7, 0x063, 0x3EB, 0x281, 0x0A5, 0x070, 0x1CB, 0x07B, 0x270, 0x2CC, 0x398, 0x32B, 0x1C1, 0x396, 0x278, 0x39E,
                0x160, 0x0FF, 0x1A2, 0x0D4, 0x024, 0x24B, 0x178, 0x1BD, 0x326, 0x2EF, 0x28D, 0x392, 0x21F, 0x24A, 0x10B, 0x042,
                0x141, 0x256, 0x229, 0x218, 0x0EB, 0x260, 0x145, 0x050, 0x035, 0x0E5, 0x300, 0x3AE, 0x1E2, 0x34E, 0x223, 0x20A,
                0x164, 0x02F, 0x0C5, 0x210, 0x1A6, 0x258, 0x3F5, 0x32D, 0x1B4, 0x2EA, 0x1C4, 0x3D0, 0x381, 0x371, 0x2D9, 0x101,
                0x3C8, 0x3F3, 0x1F2, 0x10F, 0x0D1, 0x1BF, 0x2D6, 0x320, 0x390, 0x25E, 0x249, 0x341, 0x33B, 0x203, 0x087, 0x23A,
                0x09E, 0x095, 0x2C8, 0x3A6, 0x0F2, 0x263, 0x108, 0x307, 0x3E8, 0x3C4, 0x2BB, 0x14C, 0x36E, 0x13E, 0x2C9, 0x376,
                0x014, 0x00F, 0x0DA, 0x133, 0x163, 0x05C, 0x0AA, 0x1E5, 0x019, 0x37D, 0x043, 0x1FC, 0x184, 0x07A, 0x3FE, 0x03D,
                0x0FE, 0x25F, 0x26E, 0x3B7, 0x135, 0x2E8, 0x3B1, 0x1B7, 0x012, 0x2CA, 0x0C2, 0x113, 0x001, 0x271, 0x1D8, 0x01A,
                0x16F, 0x1C9, 0x0AD, 0x236, 0x299, 0x3CF, 0x3EC, 0x24E, 0x3F0, 0x1D4, 0x3CC, 0x2BF, 0x2C3, 0x338, 0x1B5, 0x25C,
                0x181, 0x052, 0x243, 0x1F3, 0x11F, 0x2EE, 0x332, 0x32C, 0x034, 0x3A8, 0x2B4, 0x34F, 0x031, 0x305, 0x006, 0x124,
                0x13F, 0x13C, 0x19E, 0x3A0, 0x17C, 0x2E2, 0x3CE, 0x345, 0x3CD, 0x0EF, 0x205, 0x31A, 0x23D, 0x06B, 0x059, 0x19F,
                0x1E0, 0x3D8, 0x3F9, 0x103, 0x337, 0x0DB, 0x14D, 0x353, 0x127, 0x0CE, 0x385, 0x114, 0x107, 0x3D7, 0x057, 0x288,
                0x04F, 0x2B2, 0x2CB, 0x039, 0x234, 0x2B5, 0x2E1, 0x32A, 0x2FB, 0x115, 0x116, 0x37B, 0x3A5, 0x092, 0x373, 0x17F,
                0x21E, 0x2AB, 0x37F, 0x2FD, 0x2ED, 0x2BA, 0x1EA, 0x125, 0x208, 0x16E, 0x33C, 0x0A9, 0x2F3, 0x3C2, 0x3C1, 0x21D,
                0x11A, 0x0A4, 0x3EA, 0x047, 0x157, 0x25D, 0x1D9, 0x10A, 0x16D, 0x20E, 0x098, 0x2B1, 0x340, 0x22E, 0x241, 0x078,
                0x1F5, 0x0ED, 0x31E, 0x298, 0x3A7, 0x30C, 0x1CF, 0x05F, 0x351, 0x0E4, 0x335, 0x046, 0x151, 0x24C, 0x1EE, 0x235,
                0x12E, 0x2A6, 0x1A5, 0x061, 0x3A1, 0x29C, 0x011, 0x066, 0x093, 0x03A, 0x38A, 0x1F8, 0x1F0, 0x084, 0x134, 0x356,
                0x225, 0x20C, 0x3D9, 0x2E4, 0x0A8, 0x0BE, 0x1FE, 0x0FC, 0x0C7, 0x377, 0x2F7, 0x07C, 0x074, 0x045, 0x1E8, 0x05A,
                0x36B, 0x36F, 0x37E, 0x375, 0x04D, 0x1FA, 0x257, 0x13B, 0x089, 0x220, 0x399, 0x00B, 0x158, 0x2D5, 0x068, 0x280,
                0x357, 0x0DD, 0x0BF, 0x1B0, 0x2A7, 0x23B, 0x255, 0x3FC, 0x00A, 0x330, 0x2A4, 0x200, 0x016, 0x008, 0x126, 0x0A6,
                0x09B, 0x37A, 0x284, 0x2D4, 0x0F3, 0x28E, 0x237, 0x31D, 0x0DF, 0x368, 0x386, 0x060, 0x374, 0x31C, 0x033, 0x26A,
                0x100, 0x394, 0x1F9, 0x04B, 0x391, 0x39F, 0x30B, 0x00C, 0x077, 0x2EB, 0x3E3, 0x231, 0x29B, 0x049, 0x202, 0x224,
                0x132, 0x2DA, 0x2A8, 0x286, 0x06A, 0x189, 0x130, 0x13D, 0x1EC, 0x29D, 0x104, 0x387, 0x32F, 0x316, 0x207, 0x137,
                0x0DC, 0x02B, 0x1D7, 0x21A, 0x354, 0x39C, 0x0B6, 0x329, 0x285, 0x3A4, 0x0D9, 0x245, 0x2B3, 0x0D6, 0x33E, 0x252,
                0x0CB, 0x1DB, 0x172, 0x296, 0x192, 0x04A, 0x244, 0x250, 0x1F6, 0x2AD, 0x2C6, 0x346, 0x09D, 0x388, 0x328, 0x3A3,
                0x2C7, 0x3E7, 0x29E, 0x3C0, 0x0D5, 0x22A, 0x1F4, 0x168, 0x3FD, 0x242, 0x102, 0x3C5, 0x0F8, 0x251, 0x264, 0x2DF,
                0x27C, 0x029, 0x003, 0x38B, 0x10C, 0x380, 0x10D, 0x295, 0x303, 0x197, 0x1CD, 0x219, 0x13A, 0x306, 0x166, 0x304,
                0x175, 0x19A, 0x0D8, 0x28A, 0x0A2, 0x26F, 0x3B8, 0x1C2, 0x148, 0x30A, 0x0B8, 0x24D, 0x1A7, 0x121, 0x15E, 0x372,
                0x0B4, 0x266, 0x22F, 0x2FE, 0x0B0, 0x055, 0x01E, 0x3AC, 0x14A, 0x2E0, 0x34B, 0x1D5, 0x3E9, 0x393, 0x2E7, 0x037,
                0x20F, 0x0D7, 0x1A8, 0x1AB, 0x16B, 0x36A, 0x352, 0x204, 0x2BD, 0x08B, 0x147, 0x1AA, 0x35F, 0x03C, 0x309, 0x33D
            ],
            1: [
                0x021, 0x09B, 0x37A, 0x3AB, 0x0DF, 0x016, 0x1FE, 0x004, 0x07C, 0x3BE, 0x141, 0x397, 0x300, 0x185, 0x00C, 0x1A7,
                0x2FA, 0x3AA, 0x235, 0x0B9, 0x003, 0x3CF, 0x14A, 0x18F, 0x356, 0x363, 0x173, 0x2E4, 0x168, 0x0CF, 0x373, 0x379,
                0x2CA, 0x326, 0x16B, 0x393, 0x283, 0x2E0, 0x2B9, 0x3E9, 0x12F, 0x247, 0x3D8, 0x07B, 0x288, 0x146, 0x30F, 0x267,
                0x15C, 0x01F, 0x22C, 0x0F8, 0x10F, 0x35D, 0x367, 0x343, 0x1EC, 0x047, 0x008, 0x062, 0x2CF, 0x3D6, 0x36B, 0x148,
                0x0B4, 0x2E3, 0x25E, 0x234, 0x0D2, 0x1F8, 0x184, 0x2FF, 0x2EB, 0x2BB, 0x3A1, 0x34F, 0x312, 0x10B, 0x2EA, 0x04D,
                0x1B1, 0x2FE, 0x084, 0x229, 0x216, 0x337, 0x0D4, 0x08D, 0x21F, 0x035, 0x164, 0x32A, 0x1AA, 0x182, 0x24B, 0x1BF,
                0x245, 0x257, 0x01E, 0x34E, 0x375, 0x197, 0x292, 0x1DD, 0x14D, 0x190, 0x27E, 0x13D, 0x137, 0x3A3, 0x228, 0x392,
                0x010, 0x34C, 0x389, 0x114, 0x3B9, 0x28B, 0x325, 0x210, 0x1E7, 0x30B, 0x388, 0x1A1, 0x094, 0x088, 0x038, 0x1C2,
                0x305, 0x38E, 0x112, 0x0AA, 0x01B, 0x260, 0x3C1, 0x104, 0x30E, 0x3D4, 0x0EF, 0x079, 0x347, 0x382, 0x22E, 0x09D,
                0x1E6, 0x087, 0x278, 0x20D, 0x25B, 0x060, 0x215, 0x2C6, 0x3E0, 0x055, 0x3F9, 0x179, 0x252, 0x1B5, 0x105, 0x368,
                0x029, 0x1E9, 0x2C4, 0x2C5, 0x037, 0x233, 0x204, 0x133, 0x3BD, 0x20B, 0x37D, 0x1AE, 0x03D, 0x116, 0x1B2, 0x2F3,
                0x266, 0x333, 0x08F, 0x050, 0x1B9, 0x328, 0x26F, 0x1EA, 0x1A9, 0x0E6, 0x291, 0x2ED, 0x05E, 0x162, 0x1EE, 0x362,
                0x15B, 0x351, 0x20F, 0x17D, 0x08B, 0x2D5, 0x259, 0x271, 0x14F, 0x2F5, 0x011, 0x3E7, 0x14B, 0x391, 0x248, 0x0B2,
                0x119, 0x3CD, 0x160, 0x23E, 0x06A, 0x0D0, 0x3C3, 0x01C, 0x171, 0x3D3, 0x349, 0x061, 0x16F, 0x0FB, 0x1DF, 0x342,
                0x082, 0x068, 0x218, 0x2E9, 0x3B3, 0x225, 0x2F9, 0x230, 0x020, 0x223, 0x151, 0x0C5, 0x2A9, 0x0FE, 0x096, 0x045,
                0x0F2, 0x0DA, 0x03A, 0x015, 0x049, 0x370, 0x14C, 0x255, 0x369, 0x193, 0x38A, 0x20E, 0x0B1, 0x3A6, 0x039, 0x387,
                0x24C, 0x030, 0x315, 0x3CA, 0x0A1, 0x0C6, 0x02C, 0x203, 0x107, 0x115, 0x3FE, 0x244, 0x26C, 0x264, 0x1C6, 0x1C9,
                0x123, 0x090, 0x36F, 0x28F, 0x1A3, 0x19D, 0x0BE, 0x317, 0x19B, 0x25C, 0x117, 0x0ED, 0x395, 0x0BF, 0x37E, 0x3E4,
                0x04C, 0x3FB, 0x103, 0x2E6, 0x3C8, 0x11E, 0x3D1, 0x279, 0x316, 0x38C, 0x277, 0x286, 0x081, 0x074, 0x213, 0x1F7,
                0x3C5, 0x095, 0x2FC, 0x09F, 0x2B5, 0x332, 0x05C, 0x31F, 0x324, 0x09E, 0x2DD, 0x3FC, 0x19F, 0x111, 0x2A7, 0x2B0,
                0x091, 0x329, 0x106, 0x10E, 0x012, 0x273, 0x2EC, 0x341, 0x080, 0x174, 0x2DB, 0x1C7, 0x102, 0x2D3, 0x2EE, 0x1B0,
                0x03F, 0x2D4, 0x364, 0x131, 0x0A6, 0x275, 0x00A, 0x386, 0x052, 0x3DC, 0x339, 0x11A, 0x211, 0x02A, 0x27F, 0x0DD,
                0x318, 0x27B, 0x17B, 0x2D7, 0x1E4, 0x285, 0x0AC, 0x269, 0x3F4, 0x1EF, 0x093, 0x3BB, 0x307, 0x08E, 0x3B0, 0x0EB,
                0x209, 0x2CB, 0x0BB, 0x3A5, 0x129, 0x1CA, 0x027, 0x028, 0x3E6, 0x064, 0x221, 0x125, 0x159, 0x2B7, 0x0F9, 0x37C,
                0x054, 0x32D, 0x3F6, 0x031, 0x053, 0x29F, 0x23C, 0x2A1, 0x0D9, 0x237, 0x336, 0x232, 0x1B3, 0x1C1, 0x380, 0x2C1,
                0x1DA, 0x360, 0x30C, 0x265, 0x34A, 0x17F, 0x296, 0x3E1, 0x20C, 0x0A2, 0x1F6, 0x207, 0x0F1, 0x040, 0x1D5, 0x026,
                0x200, 0x121, 0x134, 0x2AB, 0x2FB, 0x272, 0x0D7, 0x07E, 0x001, 0x262, 0x27A, 0x1FF, 0x299, 0x3EB, 0x1FA, 0x39F,
                0x253, 0x006, 0x128, 0x36E, 0x14E, 0x289, 0x0F6, 0x3A8, 0x3D2, 0x261, 0x178, 0x3E5, 0x2C0, 0x0B7, 0x303, 0x181,
                0x097, 0x22A, 0x32E, 0x166, 0x306, 0x0FC, 0x139, 0x138, 0x3BF, 0x1AC, 0x1FD, 0x29B, 0x0AF, 0x041, 0x2CC, 0x0CA,
                0x23B, 0x1F2, 0x25D, 0x0EC, 0x314, 0x20A, 0x03C, 0x120, 0x3C6, 0x0C0, 0x158, 0x28C, 0x3E8, 0x21E, 0x06E, 0x263,
                0x0C4, 0x085, 0x1BD, 0x051, 0x3E2, 0x153, 0x013, 0x0F3, 0x2B6, 0x1A8, 0x17C, 0x2DC, 0x2C7, 0x3B7, 0x33C, 0x29E,
                0x0B5, 0x27C, 0x3F2, 0x398, 0x194, 0x099, 0x0A9, 0x320, 0x35A, 0x366, 0x2C2, 0x05D, 0x1F9, 0x226, 0x098, 0x04E,
                0x05A, 0x3AC, 0x33E, 0x0E8, 0x0A7, 0x186, 0x1D8, 0x17E, 0x126, 0x32B, 0x110, 0x05F, 0x1A5, 0x390, 0x3CE, 0x1FC,
                0x11F, 0x019, 0x3D5, 0x13C, 0x2BD, 0x251, 0x355, 0x065, 0x1F5, 0x3DF, 0x152, 0x07A, 0x086, 0x1B6, 0x308, 0x188,
                0x0DC, 0x124, 0x15F, 0x075, 0x2E7, 0x39E, 0x046, 0x302, 0x32C, 0x2CE, 0x3CC, 0x3AF, 0x208, 0x066, 0x394, 0x12B,
                0x06D, 0x371, 0x2AF, 0x12A, 0x378, 0x319, 0x24D, 0x1D7, 0x37F, 0x3A2, 0x21D, 0x157, 0x31A, 0x3FF, 0x3B2, 0x2DA,
                0x071, 0x31B, 0x256, 0x3F3, 0x33D, 0x280, 0x144, 0x08C, 0x21C, 0x058, 0x1CD, 0x2D6, 0x165, 0x3A0, 0x077, 0x354,
                0x022, 0x32F, 0x359, 0x2BC, 0x374, 0x1EB, 0x30A, 0x192, 0x1CF, 0x1BA, 0x06B, 0x0A0, 0x177, 0x183, 0x28E, 0x2A8,
                0x29C, 0x130, 0x323, 0x122, 0x331, 0x201, 0x3B1, 0x0BC, 0x25A, 0x0D8, 0x34B, 0x11B, 0x24F, 0x2E8, 0x1F1, 0x3F5,
                0x31C, 0x254, 0x346, 0x376, 0x11C, 0x000, 0x243, 0x0C8, 0x381, 0x0E9, 0x22D, 0x01A, 0x161, 0x3D0, 0x07F, 0x1E0,
                0x295, 0x175, 0x04F, 0x3C4, 0x1AF, 0x2A2, 0x191, 0x2F7, 0x34D, 0x36C, 0x2E2, 0x3D7, 0x0F7, 0x18B, 0x0F5, 0x2F6,
                0x0C1, 0x30D, 0x025, 0x1F3, 0x01D, 0x1D3, 0x06C, 0x13B, 0x109, 0x2DF, 0x38B, 0x2E5, 0x18C, 0x0E1, 0x231, 0x10D,
                0x36D, 0x3DB, 0x377, 0x1DB, 0x16D, 0x09C, 0x024, 0x242, 0x072, 0x39B, 0x31D, 0x2C9, 0x149, 0x0F0, 0x089, 0x0A3,
                0x0EA, 0x057, 0x250, 0x2CD, 0x38F, 0x2A0, 0x0B3, 0x169, 0x12D, 0x309, 0x2D8, 0x2AD, 0x358, 0x3F1, 0x1C8, 0x043,
                0x268, 0x2A3, 0x1D6, 0x28A, 0x3EC, 0x18D, 0x2AA, 0x02F, 0x1DE, 0x3C7, 0x0D3, 0x274, 0x147, 0x219, 0x02D, 0x2B2,
                0x0CC, 0x13F, 0x383, 0x3DA, 0x3ED, 0x26A, 0x0AE, 0x1DC, 0x301, 0x2A4, 0x350, 0x2F2, 0x0AB, 0x2A6, 0x39A, 0x014,
                0x2D2, 0x352, 0x108, 0x0E3, 0x270, 0x3E3, 0x02E, 0x29D, 0x1BE, 0x06F, 0x002, 0x059, 0x0A4, 0x198, 0x23A, 0x044,
                0x0CB, 0x258, 0x348, 0x39C, 0x176, 0x2B4, 0x007, 0x3C2, 0x33F, 0x217, 0x287, 0x073, 0x238, 0x15E, 0x03B, 0x167,
                0x2B8, 0x2D0, 0x340, 0x0F4, 0x0BD, 0x2F0, 0x353, 0x100, 0x18A, 0x29A, 0x399, 0x246, 0x1CB, 0x02B, 0x1A2, 0x2E1,
                0x3F0, 0x212, 0x1B7, 0x032, 0x281, 0x357, 0x3AD, 0x048, 0x322, 0x3A9, 0x3B6, 0x33A, 0x196, 0x1BB, 0x1FB, 0x19A,
                0x1E2, 0x0AD, 0x101, 0x033, 0x22F, 0x227, 0x0B6, 0x345, 0x0C2, 0x220, 0x07D, 0x298, 0x3EF, 0x0B8, 0x2F1, 0x0DE,
                0x304, 0x0E4, 0x202, 0x0D1, 0x21B, 0x005, 0x12C, 0x0EE, 0x13A, 0x0C7, 0x092, 0x00D, 0x05B, 0x009, 0x37B, 0x365,
                0x0DB, 0x2AC, 0x27D, 0x39D, 0x3A7, 0x214, 0x338, 0x1AD, 0x335, 0x2DE, 0x1D9, 0x1E5, 0x1C0, 0x3DE, 0x140, 0x24A,
                0x2B3, 0x26B, 0x1F0, 0x3C0, 0x3A4, 0x04A, 0x0A8, 0x2C3, 0x0BA, 0x078, 0x1D4, 0x1E3, 0x16A, 0x145, 0x170, 0x2C8,
                0x00B, 0x35B, 0x1AB, 0x127, 0x2BF, 0x16E, 0x2BE, 0x241, 0x1E1, 0x063, 0x334, 0x2B1, 0x136, 0x3EE, 0x3B8, 0x1C5,
                0x23D, 0x2D1, 0x042, 0x372, 0x3BA, 0x1ED, 0x0FA, 0x327, 0x0C9, 0x018, 0x1C3, 0x396, 0x3F8, 0x26E, 0x1BC, 0x187,
                0x034, 0x3FD, 0x310, 0x118, 0x1D1, 0x076, 0x22B, 0x143, 0x38D, 0x33B, 0x0E5, 0x0D5, 0x3B4, 0x199, 0x3C9, 0x3B5,
                0x0E2, 0x195, 0x10A, 0x284, 0x156, 0x150, 0x11D, 0x155, 0x3DD, 0x15D, 0x0CD, 0x163, 0x1A0, 0x0C3, 0x10C, 0x35C,
                0x180, 0x1A6, 0x321, 0x00E, 0x276, 0x03E, 0x25F, 0x0D6, 0x189, 0x206, 0x1D0, 0x1CC, 0x26D, 0x205, 0x17A, 0x3FA,
                0x35E, 0x036, 0x35F, 0x2F8, 0x067, 0x2BA, 0x2A5, 0x16C, 0x3D9, 0x2FD, 0x297, 0x18E, 0x113, 0x0FD, 0x313, 0x0E7,
                0x15A, 0x1B8, 0x08A, 0x239, 0x04B, 0x384, 0x083, 0x385, 0x2F4, 0x19C, 0x12E, 0x017, 0x3BC, 0x224, 0x135, 0x290,
                0x09A, 0x311, 0x240, 0x13E, 0x0A5, 0x24E, 0x069, 0x3CB, 0x0FF, 0x236, 0x36A, 0x1A4, 0x344, 0x3AE, 0x1E8, 0x31E,
                0x132, 0x23F, 0x222, 0x070, 0x2AE, 0x3EA, 0x249, 0x023, 0x293, 0x0B0, 0x330, 0x21A, 0x28D, 0x1CE, 0x154, 0x172,
                0x1F4, 0x056, 0x00F, 0x2EF, 0x361, 0x1D2, 0x0E0, 0x1C4, 0x19E, 0x282, 0x1B4, 0x3F7, 0x294, 0x142, 0x2D9, 0x0CE
            ],
            2: [
                0x12E, 0x38B, 0x18E, 0x131, 0x039, 0x10D, 0x2DE, 0x246, 0x286, 0x2BE, 0x315, 0x384, 0x21D, 0x142, 0x06D, 0x0CA,
                0x2A2, 0x2CE, 0x264, 0x085, 0x374, 0x3BB, 0x3B9, 0x1B7, 0x3E6, 0x3BC, 0x207, 0x002, 0x392, 0x1B5, 0x0BA, 0x318,
                0x39C, 0x2EE, 0x1DA, 0x125, 0x019, 0x063, 0x27E, 0x126, 0x19A, 0x082, 0x305, 0x0E3, 0x206, 0x0A0, 0x009, 0x3C6,
                0x100, 0x3F3, 0x2AD, 0x199, 0x102, 0x108, 0x1DB, 0x2F0, 0x310, 0x245, 0x0A8, 0x116, 0x022, 0x3C1, 0x028, 0x332,
                0x1E1, 0x2E7, 0x0DA, 0x2B7, 0x0CB, 0x07C, 0x2A0, 0x240, 0x150, 0x165, 0x258, 0x2C8, 0x0C4, 0x334, 0x36B, 0x2D1,
                0x1E0, 0x138, 0x39A, 0x0FF, 0x1A7, 0x10C, 0x353, 0x19B, 0x171, 0x038, 0x3BD, 0x000, 0x3A2, 0x1B8, 0x282, 0x2EB,
                0x1D4, 0x3D4, 0x20F, 0x23C, 0x0D7, 0x154, 0x012, 0x0DF, 0x3A8, 0x237, 0x09E, 0x155, 0x2E2, 0x189, 0x2F2, 0x136,
                0x1B4, 0x381, 0x273, 0x123, 0x052, 0x12C, 0x158, 0x033, 0x2D2, 0x3D3, 0x23B, 0x3B8, 0x2F7, 0x160, 0x341, 0x124,
                0x337, 0x1E6, 0x3BE, 0x327, 0x1D3, 0x045, 0x2E4, 0x107, 0x1C2, 0x263, 0x2A4, 0x2CF, 0x244, 0x196, 0x36A, 0x16D,
                0x0A1, 0x2C3, 0x004, 0x049, 0x209, 0x3A3, 0x221, 0x361, 0x01E, 0x1B0, 0x05D, 0x319, 0x21B, 0x249, 0x2B2, 0x399,
                0x198, 0x26A, 0x080, 0x1B1, 0x340, 0x28A, 0x33C, 0x316, 0x0FC, 0x37F, 0x1A8, 0x134, 0x17F, 0x3DF, 0x34F, 0x3E5,
                0x2D9, 0x32A, 0x34A, 0x1D1, 0x09D, 0x3FB, 0x0BE, 0x3EA, 0x383, 0x036, 0x3B6, 0x222, 0x22E, 0x2B6, 0x3A6, 0x0FA,
                0x1C1, 0x0B2, 0x113, 0x3E8, 0x129, 0x34C, 0x153, 0x333, 0x07E, 0x01F, 0x01D, 0x213, 0x299, 0x0F8, 0x130, 0x1B9,
                0x182, 0x0A2, 0x1A1, 0x3D5, 0x119, 0x10F, 0x24C, 0x020, 0x097, 0x3F0, 0x280, 0x112, 0x04C, 0x14D, 0x1EB, 0x307,
                0x386, 0x0AE, 0x322, 0x2FE, 0x0C5, 0x3D7, 0x1AF, 0x345, 0x05B, 0x3F5, 0x110, 0x1C8, 0x03C, 0x1C5, 0x35A, 0x0C0,
                0x3F1, 0x15B, 0x338, 0x1CB, 0x0F4, 0x2B4, 0x00F, 0x3A1, 0x242, 0x03D, 0x29D, 0x1B3, 0x003, 0x114, 0x3FA, 0x313,
                0x35F, 0x217, 0x261, 0x2C1, 0x15D, 0x28F, 0x390, 0x1C9, 0x1DD, 0x3C7, 0x14F, 0x11D, 0x066, 0x04D, 0x03B, 0x0E9,
                0x2BA, 0x2FD, 0x347, 0x191, 0x044, 0x0B8, 0x194, 0x148, 0x256, 0x360, 0x326, 0x257, 0x1AE, 0x396, 0x09B, 0x2CD,
                0x1E7, 0x3CD, 0x1FF, 0x269, 0x040, 0x3E7, 0x08A, 0x216, 0x0C9, 0x33B, 0x3D9, 0x1BC, 0x2B1, 0x325, 0x11B, 0x16F,
                0x053, 0x22A, 0x186, 0x180, 0x27D, 0x11F, 0x2A9, 0x13E, 0x3E1, 0x0D4, 0x24E, 0x1D2, 0x2FC, 0x3C9, 0x1FB, 0x31A,
                0x3DE, 0x1D7, 0x025, 0x372, 0x339, 0x2C7, 0x2ED, 0x25F, 0x0A7, 0x098, 0x2EF, 0x247, 0x0E8, 0x2D3, 0x105, 0x09F,
                0x2CC, 0x36D, 0x31F, 0x24B, 0x1D8, 0x241, 0x068, 0x211, 0x2AF, 0x0AA, 0x355, 0x35C, 0x026, 0x2BD, 0x238, 0x0EF,
                0x35B, 0x233, 0x05A, 0x1BE, 0x291, 0x368, 0x137, 0x035, 0x298, 0x140, 0x26B, 0x1E4, 0x379, 0x07F, 0x3EB, 0x164,
                0x20B, 0x12D, 0x375, 0x1BF, 0x12F, 0x1AA, 0x18B, 0x268, 0x3F4, 0x364, 0x0F7, 0x1CC, 0x0B9, 0x3C5, 0x060, 0x19D,
                0x22B, 0x17C, 0x11C, 0x0B1, 0x23A, 0x3B4, 0x05F, 0x2F5, 0x219, 0x224, 0x0E5, 0x042, 0x06F, 0x39D, 0x218, 0x023,
                0x1DE, 0x177, 0x190, 0x395, 0x274, 0x359, 0x0E2, 0x2E9, 0x397, 0x0F1, 0x010, 0x099, 0x17D, 0x08E, 0x314, 0x317,
                0x0DC, 0x03F, 0x1AC, 0x1A6, 0x132, 0x152, 0x195, 0x3AD, 0x3E9, 0x3C2, 0x1BB, 0x0F0, 0x0CD, 0x074, 0x178, 0x174,
                0x184, 0x3E0, 0x389, 0x2FB, 0x1A9, 0x0B7, 0x250, 0x27B, 0x06C, 0x13B, 0x0FB, 0x296, 0x297, 0x30F, 0x350, 0x14E,
                0x007, 0x10E, 0x19C, 0x055, 0x351, 0x034, 0x175, 0x103, 0x272, 0x02D, 0x2C0, 0x21C, 0x047, 0x20D, 0x0EA, 0x29B,
                0x13F, 0x1DF, 0x162, 0x376, 0x0BF, 0x1CA, 0x3EC, 0x2B9, 0x3FE, 0x388, 0x133, 0x0A9, 0x33A, 0x304, 0x1FE, 0x059,
                0x13D, 0x0BD, 0x294, 0x02B, 0x127, 0x1E8, 0x275, 0x07B, 0x14C, 0x018, 0x031, 0x1C6, 0x0A3, 0x0EC, 0x27C, 0x087,
                0x38D, 0x3B0, 0x284, 0x1FA, 0x1F5, 0x00A, 0x3E2, 0x02E, 0x228, 0x285, 0x34B, 0x311, 0x075, 0x2F1, 0x1C4, 0x094,
                0x3FF, 0x202, 0x27F, 0x2F9, 0x30D, 0x135, 0x33F, 0x301, 0x3D8, 0x2C6, 0x3D2, 0x309, 0x057, 0x073, 0x1F1, 0x289,
                0x3B5, 0x3CE, 0x111, 0x0B4, 0x20E, 0x1BA, 0x1F7, 0x24A, 0x394, 0x157, 0x366, 0x336, 0x39B, 0x017, 0x25C, 0x3C4,
                0x1EC, 0x2BC, 0x144, 0x1E9, 0x193, 0x16A, 0x33D, 0x344, 0x295, 0x079, 0x027, 0x2D4, 0x38A, 0x17A, 0x292, 0x0AC,
                0x0F2, 0x35E, 0x1EF, 0x0BB, 0x106, 0x071, 0x2DA, 0x3F7, 0x084, 0x037, 0x2AB, 0x330, 0x0B0, 0x2DB, 0x07A, 0x22D,
                0x00C, 0x149, 0x0AF, 0x290, 0x2E0, 0x122, 0x283, 0x32E, 0x3AE, 0x3C3, 0x1D9, 0x2E5, 0x37B, 0x0BC, 0x265, 0x32D,
                0x089, 0x2CB, 0x115, 0x081, 0x18A, 0x255, 0x05C, 0x1A3, 0x287, 0x0D0, 0x276, 0x32C, 0x0C3, 0x30B, 0x226, 0x1C0,
                0x2F3, 0x0A5, 0x121, 0x2AA, 0x210, 0x091, 0x208, 0x3EE, 0x230, 0x320, 0x385, 0x28E, 0x21E, 0x3F9, 0x11E, 0x05E,
                0x159, 0x281, 0x0C8, 0x37C, 0x0DD, 0x188, 0x04F, 0x26E, 0x33E, 0x2F8, 0x3A0, 0x3B3, 0x3C8, 0x227, 0x1A2, 0x3AA,
                0x302, 0x36E, 0x38F, 0x19E, 0x212, 0x13C, 0x24D, 0x0B3, 0x141, 0x3EF, 0x1CE, 0x262, 0x145, 0x362, 0x346, 0x176,
                0x1E3, 0x14B, 0x3A9, 0x3DD, 0x093, 0x3F8, 0x070, 0x0D3, 0x1EA, 0x3BA, 0x248, 0x146, 0x201, 0x243, 0x1F6, 0x205,
                0x1CD, 0x20A, 0x2F6, 0x00E, 0x267, 0x26D, 0x2A7, 0x1FC, 0x2D0, 0x0D1, 0x38E, 0x006, 0x30A, 0x3E3, 0x2C5, 0x28B,
                0x2D5, 0x3A7, 0x1D5, 0x3A4, 0x101, 0x2D7, 0x34E, 0x2B5, 0x072, 0x26C, 0x090, 0x1F8, 0x1F9, 0x3AF, 0x1F0, 0x0C2,
                0x2C2, 0x21A, 0x06A, 0x0AB, 0x1EE, 0x109, 0x16B, 0x15E, 0x161, 0x38C, 0x156, 0x271, 0x279, 0x369, 0x342, 0x1D6,
                0x01A, 0x016, 0x352, 0x173, 0x34D, 0x354, 0x181, 0x185, 0x1A5, 0x23F, 0x16C, 0x030, 0x215, 0x1C3, 0x2EA, 0x0CF,
                0x0DE, 0x078, 0x18D, 0x01B, 0x117, 0x393, 0x3F2, 0x39E, 0x37D, 0x1BD, 0x24F, 0x18C, 0x29A, 0x0A4, 0x08D, 0x187,
                0x015, 0x065, 0x0C1, 0x251, 0x00D, 0x348, 0x014, 0x21F, 0x001, 0x008, 0x2CA, 0x321, 0x1B2, 0x1B6, 0x043, 0x147,
                0x223, 0x0B6, 0x054, 0x1AB, 0x0FD, 0x373, 0x31E, 0x323, 0x20C, 0x151, 0x10B, 0x288, 0x2DF, 0x041, 0x349, 0x2E3,
                0x32F, 0x0ED, 0x277, 0x179, 0x278, 0x3F6, 0x23E, 0x252, 0x077, 0x04A, 0x120, 0x200, 0x308, 0x300, 0x312, 0x04B,
                0x1ED, 0x048, 0x30C, 0x183, 0x0D2, 0x39F, 0x3B7, 0x0AD, 0x3FD, 0x204, 0x050, 0x1C7, 0x197, 0x3BF, 0x046, 0x04E,
                0x1F3, 0x343, 0x051, 0x1F2, 0x169, 0x266, 0x25A, 0x26F, 0x0F3, 0x2B0, 0x095, 0x17B, 0x31B, 0x0F6, 0x0E6, 0x2DC,
                0x225, 0x36C, 0x377, 0x253, 0x058, 0x0E1, 0x021, 0x31C, 0x3D6, 0x1AD, 0x167, 0x2AC, 0x06B, 0x23D, 0x398, 0x032,
                0x35D, 0x2FA, 0x00B, 0x391, 0x239, 0x0F5, 0x335, 0x02C, 0x083, 0x143, 0x02A, 0x29E, 0x36F, 0x214, 0x104, 0x14A,
                0x0E4, 0x096, 0x19F, 0x2E1, 0x1FD, 0x30E, 0x28D, 0x07D, 0x11A, 0x0EE, 0x0EB, 0x370, 0x358, 0x1DC, 0x163, 0x056,
                0x367, 0x03A, 0x2D6, 0x363, 0x229, 0x3DA, 0x08B, 0x1E5, 0x270, 0x2E8, 0x2FF, 0x168, 0x10A, 0x2AE, 0x170, 0x28C,
                0x2A3, 0x08C, 0x1CF, 0x076, 0x3D1, 0x32B, 0x2EC, 0x2A5, 0x2C9, 0x2A6, 0x29C, 0x3ED, 0x09C, 0x0CC, 0x2A8, 0x203,
                0x0FE, 0x293, 0x29F, 0x2F4, 0x0E7, 0x232, 0x0CE, 0x3AB, 0x13A, 0x011, 0x3DB, 0x220, 0x0D8, 0x1F4, 0x22F, 0x236,
                0x062, 0x1A4, 0x27A, 0x128, 0x329, 0x324, 0x067, 0x365, 0x024, 0x2B8, 0x3E4, 0x0D6, 0x3AC, 0x3A5, 0x172, 0x306,
                0x16E, 0x0DB, 0x3C0, 0x25B, 0x088, 0x2D8, 0x303, 0x380, 0x259, 0x2A1, 0x1E2, 0x0B5, 0x02F, 0x029, 0x356, 0x2BF,
                0x2C4, 0x03E, 0x2BB, 0x3B1, 0x17E, 0x3CC, 0x01C, 0x25E, 0x2B3, 0x069, 0x0A6, 0x15C, 0x0D5, 0x3DC, 0x118, 0x2E6,
                0x2DD, 0x235, 0x18F, 0x371, 0x064, 0x260, 0x0C7, 0x0E0, 0x0C6, 0x1D0, 0x254, 0x0D9, 0x37A, 0x387, 0x3CB, 0x234,
                0x3D0, 0x15F, 0x3B2, 0x08F, 0x15A, 0x013, 0x331, 0x328, 0x06E, 0x25D, 0x0F9, 0x092, 0x166, 0x378, 0x31D, 0x139,
                0x005, 0x09A, 0x12B, 0x061, 0x231, 0x1A0, 0x3CF, 0x3CA, 0x382, 0x192, 0x086, 0x357, 0x22C, 0x12A, 0x3FC, 0x37E
            ],
            3: [
                0x200, 0x084, 0x1B5, 0x30A, 0x25A, 0x151, 0x174, 0x3F9, 0x113, 0x3B4, 0x35B, 0x291, 0x332, 0x170, 0x021, 0x31E,
                0x00E, 0x2FC, 0x023, 0x0B0, 0x3A9, 0x259, 0x2BC, 0x378, 0x031, 0x050, 0x0D0, 0x1FF, 0x26C, 0x0D5, 0x214, 0x23E,
                0x1AB, 0x0AB, 0x3AC, 0x036, 0x0E2, 0x2F6, 0x07A, 0x0EA, 0x2CB, 0x0FE, 0x24E, 0x280, 0x138, 0x073, 0x219, 0x3EA,
                0x2E2, 0x27C, 0x032, 0x162, 0x285, 0x13C, 0x0B6, 0x1ED, 0x0B3, 0x2F5, 0x2C6, 0x34B, 0x335, 0x1EF, 0x26E, 0x37A,
                0x273, 0x17E, 0x30F, 0x2E7, 0x14B, 0x3BC, 0x1CE, 0x039, 0x315, 0x01A, 0x144, 0x1C4, 0x20A, 0x17B, 0x362, 0x10D,
                0x235, 0x1D9, 0x2F9, 0x0A4, 0x052, 0x0E3, 0x0BD, 0x061, 0x02C, 0x140, 0x0E1, 0x156, 0x10E, 0x250, 0x288, 0x1BE,
                0x07C, 0x2B8, 0x05D, 0x242, 0x192, 0x0A8, 0x3B0, 0x0DB, 0x129, 0x2AF, 0x063, 0x3AF, 0x3D1, 0x0C8, 0x0A6, 0x029,
                0x2B9, 0x3B8, 0x0D2, 0x078, 0x2A2, 0x06E, 0x2CF, 0x3CF, 0x0EF, 0x0E7, 0x019, 0x1F1, 0x07E, 0x1BB, 0x2C7, 0x251,
                0x36A, 0x2CA, 0x076, 0x216, 0x2E5, 0x0E6, 0x1DD, 0x2FE, 0x390, 0x277, 0x1D2, 0x394, 0x2C5, 0x022, 0x05A, 0x396,
                0x0F4, 0x265, 0x0FD, 0x150, 0x057, 0x111, 0x2EC, 0x29C, 0x3DF, 0x11F, 0x13A, 0x158, 0x388, 0x1D3, 0x3C8, 0x386,
                0x38B, 0x279, 0x064, 0x1A4, 0x028, 0x22F, 0x1D5, 0x352, 0x2C8, 0x257, 0x3C4, 0x355, 0x104, 0x322, 0x2C1, 0x382,
                0x1DF, 0x1A9, 0x137, 0x3DC, 0x015, 0x096, 0x2AA, 0x2A4, 0x3F6, 0x1A3, 0x3DA, 0x086, 0x2E8, 0x343, 0x233, 0x11A,
                0x0A5, 0x38D, 0x328, 0x348, 0x292, 0x132, 0x3F4, 0x059, 0x31C, 0x1AC, 0x1C6, 0x3BF, 0x1C2, 0x36D, 0x1D8, 0x0ED,
                0x191, 0x3D3, 0x3D4, 0x3DE, 0x0E8, 0x373, 0x034, 0x23C, 0x224, 0x3C5, 0x11E, 0x393, 0x00B, 0x308, 0x2DA, 0x00F,
                0x209, 0x230, 0x19D, 0x184, 0x1B8, 0x339, 0x360, 0x2D7, 0x011, 0x305, 0x17A, 0x324, 0x344, 0x128, 0x3F0, 0x0F3,
                0x317, 0x0B4, 0x08D, 0x18E, 0x035, 0x0C9, 0x345, 0x0D3, 0x37D, 0x3CA, 0x284, 0x3EF, 0x00D, 0x197, 0x36B, 0x06B,
                0x08B, 0x10B, 0x18A, 0x218, 0x046, 0x32A, 0x2CC, 0x0AE, 0x254, 0x3FC, 0x066, 0x246, 0x24D, 0x232, 0x0A2, 0x145,
                0x2DB, 0x199, 0x37F, 0x1E1, 0x392, 0x3F3, 0x1C8, 0x1CD, 0x136, 0x2D0, 0x325, 0x27B, 0x068, 0x1F5, 0x077, 0x22D,
                0x12F, 0x2F4, 0x0B2, 0x2E9, 0x3CC, 0x296, 0x2EA, 0x116, 0x30D, 0x276, 0x02D, 0x266, 0x09C, 0x25E, 0x157, 0x195,
                0x3A1, 0x3F2, 0x3D7, 0x130, 0x258, 0x227, 0x0D4, 0x26B, 0x027, 0x1EA, 0x379, 0x329, 0x179, 0x2D5, 0x0C4, 0x09F,
                0x39E, 0x09E, 0x1FD, 0x15B, 0x126, 0x2B3, 0x15E, 0x012, 0x21A, 0x372, 0x356, 0x154, 0x042, 0x017, 0x217, 0x19B,
                0x1F8, 0x261, 0x3ED, 0x14A, 0x1FB, 0x110, 0x037, 0x1B7, 0x079, 0x045, 0x3C9, 0x0EE, 0x2B6, 0x107, 0x3CB, 0x302,
                0x19E, 0x21D, 0x1E5, 0x205, 0x25F, 0x3BD, 0x196, 0x198, 0x337, 0x069, 0x32E, 0x0DF, 0x3EE, 0x201, 0x0BC, 0x3C2,
                0x01D, 0x37B, 0x3C0, 0x0A3, 0x22E, 0x123, 0x2A9, 0x0DE, 0x2A7, 0x2FF, 0x3A5, 0x05B, 0x38F, 0x047, 0x1B4, 0x350,
                0x0CB, 0x0A1, 0x29D, 0x1FC, 0x024, 0x29B, 0x3A3, 0x2A1, 0x3BE, 0x215, 0x09A, 0x37E, 0x2A0, 0x0C2, 0x377, 0x0B1,
                0x149, 0x33B, 0x323, 0x365, 0x3D6, 0x2E3, 0x082, 0x35A, 0x38C, 0x0D7, 0x134, 0x3D0, 0x36E, 0x336, 0x334, 0x1F6,
                0x1DC, 0x3B2, 0x2B1, 0x213, 0x3F1, 0x1FA, 0x380, 0x06C, 0x020, 0x211, 0x033, 0x28D, 0x0DA, 0x34C, 0x20C, 0x1A8,
                0x28F, 0x369, 0x349, 0x3F5, 0x2FB, 0x1CF, 0x383, 0x387, 0x35E, 0x08F, 0x29A, 0x135, 0x3A7, 0x2B0, 0x346, 0x30E,
                0x163, 0x33C, 0x32F, 0x093, 0x0FA, 0x125, 0x244, 0x226, 0x1C1, 0x1EE, 0x1A2, 0x252, 0x1E9, 0x3A0, 0x146, 0x3D8,
                0x148, 0x353, 0x0FF, 0x37C, 0x09D, 0x2C0, 0x268, 0x048, 0x117, 0x1E3, 0x2A6, 0x003, 0x11B, 0x0AD, 0x1D7, 0x313,
                0x072, 0x18D, 0x297, 0x39A, 0x0C7, 0x12D, 0x016, 0x222, 0x056, 0x1CB, 0x287, 0x095, 0x366, 0x293, 0x3E0, 0x354,
                0x13E, 0x299, 0x190, 0x10F, 0x25B, 0x183, 0x080, 0x1B6, 0x361, 0x3AA, 0x3E1, 0x318, 0x2BA, 0x15C, 0x0D8, 0x1DB,
                0x342, 0x2EE, 0x1AE, 0x04F, 0x1A7, 0x2CD, 0x2F8, 0x03A, 0x06A, 0x0BA, 0x188, 0x090, 0x2B7, 0x1E4, 0x16F, 0x0C5,
                0x1B0, 0x2D2, 0x1CC, 0x3B9, 0x267, 0x153, 0x24B, 0x1E7, 0x20B, 0x0FC, 0x2E6, 0x0F7, 0x3BB, 0x376, 0x1C7, 0x0D9,
                0x00C, 0x271, 0x0AA, 0x1C5, 0x357, 0x1E8, 0x01E, 0x3FE, 0x081, 0x245, 0x314, 0x294, 0x164, 0x13F, 0x212, 0x340,
                0x141, 0x1B9, 0x120, 0x02E, 0x34F, 0x0E4, 0x092, 0x26A, 0x171, 0x249, 0x22B, 0x206, 0x0C0, 0x001, 0x0A0, 0x23F,
                0x02B, 0x2BB, 0x06F, 0x05E, 0x275, 0x20E, 0x3B3, 0x12A, 0x28A, 0x100, 0x2AC, 0x22A, 0x263, 0x0F9, 0x1C0, 0x21B,
                0x203, 0x303, 0x35C, 0x295, 0x088, 0x008, 0x3E5, 0x0DD, 0x307, 0x105, 0x121, 0x185, 0x0A7, 0x3EC, 0x11C, 0x347,
                0x094, 0x39D, 0x1B2, 0x02A, 0x3B1, 0x204, 0x114, 0x312, 0x167, 0x131, 0x304, 0x290, 0x231, 0x3E7, 0x2D3, 0x3D2,
                0x2C2, 0x32C, 0x3E6, 0x04A, 0x009, 0x10C, 0x327, 0x1BD, 0x2D1, 0x1BA, 0x2FD, 0x35D, 0x253, 0x2EF, 0x282, 0x3D9,
                0x338, 0x14F, 0x1B1, 0x28B, 0x330, 0x2F0, 0x18C, 0x175, 0x12E, 0x169, 0x2D9, 0x223, 0x2F3, 0x255, 0x0C3, 0x13D,
                0x398, 0x15F, 0x16D, 0x2DC, 0x2BD, 0x0FB, 0x3FA, 0x2ED, 0x147, 0x161, 0x01B, 0x04B, 0x17D, 0x28C, 0x058, 0x3C1,
                0x1A6, 0x21F, 0x1DA, 0x0E9, 0x124, 0x2BF, 0x39C, 0x005, 0x054, 0x35F, 0x143, 0x3CE, 0x19A, 0x043, 0x36F, 0x1F3,
                0x0CF, 0x286, 0x18B, 0x243, 0x006, 0x106, 0x333, 0x152, 0x1BF, 0x3FF, 0x3B7, 0x1EC, 0x30B, 0x098, 0x08E, 0x1D1,
                0x089, 0x3CD, 0x1F0, 0x210, 0x2EB, 0x309, 0x2F7, 0x13B, 0x20D, 0x3AD, 0x02F, 0x0EC, 0x11D, 0x06D, 0x3A8, 0x38E,
                0x311, 0x1E6, 0x3FB, 0x0AF, 0x2E1, 0x12B, 0x220, 0x03D, 0x0F1, 0x2FA, 0x208, 0x16C, 0x28E, 0x181, 0x33A, 0x119,
                0x109, 0x10A, 0x0CA, 0x2A5, 0x010, 0x31F, 0x3BA, 0x0B7, 0x3B5, 0x2DD, 0x193, 0x2AD, 0x283, 0x085, 0x00A, 0x32B,
                0x2A8, 0x3AB, 0x1D0, 0x2F1, 0x0EB, 0x2DF, 0x298, 0x1DE, 0x065, 0x17C, 0x18F, 0x364, 0x33E, 0x0B8, 0x2CE, 0x1F2,
                0x289, 0x142, 0x2B2, 0x0D6, 0x3E2, 0x24C, 0x101, 0x24A, 0x39B, 0x09B, 0x097, 0x1A0, 0x229, 0x375, 0x320, 0x062,
                0x33D, 0x118, 0x3EB, 0x03C, 0x15A, 0x281, 0x1A1, 0x207, 0x3C7, 0x331, 0x319, 0x27A, 0x127, 0x34E, 0x07B, 0x239,
                0x23A, 0x300, 0x0B5, 0x01C, 0x2B5, 0x1E0, 0x39F, 0x180, 0x321, 0x133, 0x26F, 0x371, 0x1B3, 0x363, 0x26D, 0x0F5,
                0x0C6, 0x165, 0x0F6, 0x19C, 0x070, 0x0E0, 0x367, 0x002, 0x247, 0x389, 0x053, 0x27F, 0x2D6, 0x2B4, 0x3DB, 0x29E,
                0x30C, 0x1AD, 0x0E5, 0x22C, 0x15D, 0x2E0, 0x013, 0x236, 0x2A3, 0x228, 0x0BB, 0x14D, 0x018, 0x278, 0x155, 0x1C9,
                0x178, 0x0CD, 0x370, 0x07D, 0x3A6, 0x23B, 0x049, 0x2D4, 0x397, 0x0BF, 0x1BC, 0x21E, 0x399, 0x3F8, 0x0AC, 0x004,
                0x34A, 0x055, 0x04D, 0x14C, 0x33F, 0x1F7, 0x301, 0x05C, 0x3A2, 0x112, 0x2DE, 0x075, 0x3F7, 0x391, 0x040, 0x326,
                0x2D8, 0x000, 0x0DC, 0x0D1, 0x041, 0x14E, 0x067, 0x160, 0x166, 0x168, 0x051, 0x0A9, 0x31B, 0x0CC, 0x16E, 0x172,
                0x1CA, 0x2C4, 0x3C3, 0x1E2, 0x03F, 0x173, 0x27E, 0x08A, 0x25D, 0x1F9, 0x014, 0x17F, 0x044, 0x16B, 0x2F2, 0x176,
                0x31D, 0x3E9, 0x108, 0x139, 0x2C9, 0x04C, 0x187, 0x071, 0x29F, 0x381, 0x316, 0x038, 0x0CE, 0x2C3, 0x34D, 0x1AA,
                0x122, 0x225, 0x1C3, 0x04E, 0x368, 0x351, 0x202, 0x38A, 0x102, 0x189, 0x194, 0x306, 0x026, 0x0F0, 0x248, 0x08C,
                0x05F, 0x19F, 0x2BE, 0x270, 0x060, 0x083, 0x186, 0x3DD, 0x2AE, 0x0C1, 0x23D, 0x272, 0x241, 0x0F8, 0x1EB, 0x01F,
                0x374, 0x177, 0x3E4, 0x358, 0x1FE, 0x099, 0x2AB, 0x1D4, 0x3A4, 0x310, 0x030, 0x1AF, 0x1F4, 0x0BE, 0x074, 0x16A,
                0x274, 0x1D6, 0x21C, 0x3FD, 0x3C6, 0x238, 0x234, 0x262, 0x3D5, 0x31A, 0x395, 0x27D, 0x3E8, 0x240, 0x1A5, 0x087,
                0x32D, 0x359, 0x341, 0x25C, 0x0B9, 0x115, 0x237, 0x0F2, 0x2E4, 0x12C, 0x103, 0x025, 0x20F, 0x260, 0x3AE, 0x269,
                0x07F, 0x03B, 0x03E, 0x007, 0x182, 0x159, 0x091, 0x3B6, 0x3E3, 0x384, 0x264, 0x385, 0x36C, 0x256, 0x221, 0x24F
            ]
        }

        self.mix_columns_matrix = [
            [0, 1, 0, 0, 1, 1, 1, 1, 1, 0, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 1, 0, 1, 1, 1, 1, 0, 0],
            [1, 1, 0, 1, 0, 1, 1, 1, 0, 0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0],
            [1, 1, 0, 0, 1, 0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 1, 1, 1, 0, 1, 0],
            [1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 1],
            [1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0],
            [1, 0, 1, 1, 0, 1, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 1],
            [1, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 1, 0, 0, 0, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 1, 1, 1, 0, 0, 1, 1, 1],
            [1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 0, 1, 1, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 1, 0, 1, 0, 0, 1, 1, 1],
            [1, 1, 0, 1, 0, 0, 0, 1, 0, 0, 1, 1, 1, 0, 0, 1, 0, 1, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 1, 0, 0, 1, 0, 1],
            [0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 1, 0, 1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 1, 0, 1, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 1],
            [0, 0, 1, 1, 1, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 1],
            [0, 1, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 1, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 0, 1],
            [0, 1, 0, 1, 0, 1, 1, 1, 1, 0, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 1, 1, 1],
            [0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 1, 0, 1, 1, 1, 0, 0, 1, 0, 0, 1, 0, 1, 0, 1, 1],
            [1, 1, 1, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 1, 1, 0, 1, 0, 1, 1],
            [0, 1, 1, 1, 1, 1, 0, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 1],
            [0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 1, 0, 1, 1, 1, 1, 1, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 0, 1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 0],
            [0, 1, 0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 1, 0, 1, 0, 1, 1, 0, 0, 1, 0, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0],
            [0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 1, 0, 1, 1, 0, 1, 1, 0, 1, 1],
            [1, 0, 0, 0, 1, 1, 0, 1, 1, 1, 1, 0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0],
            [0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 1, 1, 1, 0, 1, 0, 1, 1, 0, 0, 1, 0, 1, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0, 1, 1, 0, 1],
            [1, 0, 1, 1, 0, 1, 0, 1, 0, 1, 1, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 1],
            [0, 0, 1, 0, 0, 1, 1, 0, 1, 0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, 0, 1, 1, 0, 1, 1, 0, 1, 1, 0, 1, 0, 0, 1, 0, 1, 1],
            [1, 0, 0, 1, 1, 0, 1, 0, 0, 1, 1, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 1, 1, 0, 1],
            [0, 1, 1, 0, 1, 1, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 1, 1, 1, 0, 1, 0, 1, 1, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0],
            [0, 1, 1, 0, 0, 0, 1, 0, 0, 1, 1, 0, 1, 0, 1, 0, 1, 0, 1, 1, 0, 1, 1, 0, 1, 0, 0, 1, 1, 0, 1, 1, 1, 0, 0, 1, 1, 1, 0, 1],
            [0, 0, 1, 1, 0, 1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 0],
            [0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 1, 1, 1],
            [1, 1, 1, 0, 0, 0, 1, 0, 1, 1, 0, 1, 1, 0, 1, 0, 0, 1, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0],
            [0, 0, 0, 1, 1, 1, 0, 1, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 1, 0, 0, 0, 1, 1, 0, 1, 1, 1, 0, 1, 0, 0, 1],
            [0, 0, 1, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1],
            [1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 1, 1, 1, 1, 0, 1, 0, 0, 1, 0, 1, 0, 1, 1, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0],
            [0, 1, 1, 1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 1, 0, 1, 1, 0, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 0, 1, 1, 1, 1, 1, 0],
            [0, 1, 0, 1, 1, 0, 0, 1, 0, 1, 1, 1, 1, 0, 1, 1, 1, 0, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 1, 1, 0, 0, 1, 1, 1, 1, 0, 1, 1, 1],
            [1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 1, 1, 0, 1, 0, 0, 1, 0, 0, 1, 1, 1, 1, 0, 1, 1, 0, 0, 0],
            [1, 0, 0, 1, 0, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 1, 1, 1],
            [1, 1, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 0, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1],
            [1, 0, 0, 0, 1, 0, 1, 1, 0, 1, 0, 1, 1, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 1, 0, 1, 1],
            [0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 1, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1],
            [1, 0, 1, 0, 0, 0, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 0, 0, 1, 0, 1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 0, 0, 1, 0, 0, 1, 1, 1, 0]
        ]
        
        self.add_round()

        _zero = self.add_constant_component(self.SBOX_BIT_SIZE, 0);

        key_state = [
            self.add_XOR_component(
                [INPUT_KEY if j < (self.KEY_BLOCK_SIZE // self.SBOX_BIT_SIZE) else _zero.id, _zero.id],
                [
                    [i for i in range(j * self.SBOX_BIT_SIZE, (j + 1) * self.SBOX_BIT_SIZE)] if (j < (self.KEY_BLOCK_SIZE // self.SBOX_BIT_SIZE)) else [i for i in range(self.SBOX_BIT_SIZE)],
                    [i for i in range(self.SBOX_BIT_SIZE)]
                 ],
                self.SBOX_BIT_SIZE)
            for j in range(2 * self.KEY_BLOCK_SIZE // self.SBOX_BIT_SIZE)
        ]

        for i in range(7):
            x = self.add_linear_layer_component(
                [key_state[(12*i + 8) % 24].id, key_state[(12*i + 9) % 24].id, key_state[(12*i + 10) % 24].id, key_state[(12*i + 11) % 24].id],
                [[j for j in range(self.SBOX_BIT_SIZE)]] * 4,
                self.SBOX_BIT_SIZE * 4,
                self.mix_columns_matrix)
            x = [
                self.add_SBOX_component(
                    [x.id],
                    [[j * self.SBOX_BIT_SIZE + k for k in range(self.SBOX_BIT_SIZE)]],
                    self.SBOX_BIT_SIZE,
                    self.sboxes[j]
                ) for j in range(4)
            ]
            xor_constant = self.add_constant_component(
                self.SBOX_BIT_SIZE,
                pow(3, i, 2**10)
            )
            x[0] = self.add_XOR_component(
                [x[0].id, xor_constant.id],
                [[k for k in range(self.SBOX_BIT_SIZE)]] * 2,
                self.SBOX_BIT_SIZE
            )

            for j in range(4):
                key_state[(12*i + 12) % 24 + j] = self.add_XOR_component(
                    [key_state[(12*i) % 24 + j].id, x[j].id],
                    [[k for k in range(self.SBOX_BIT_SIZE)]] * 2,
                    self.SBOX_BIT_SIZE)
            for j in range(4):
                key_state[(12*i + 16) % 24 + j] = self.add_XOR_component(
                    [key_state[(12*i + 4) % 24 + j].id, key_state[(12*i + 12) % 24 + j].id],
                    [[k for k in range(self.SBOX_BIT_SIZE)]] * 2,
                    self.SBOX_BIT_SIZE)
            for j in range(4):
                key_state[(12*i + 20) % 24 + j] = self.add_XOR_component(
                    [key_state[(12*i + 8) % 24 + j].id, key_state[(12*i + 16) % 24 + j].id],
                    [[k for k in range(self.SBOX_BIT_SIZE)]] * 2,
                    self.SBOX_BIT_SIZE)

        cipher_state = INPUT_PLAINTEXT
        for round_number in range(self.NROUNDS):
            if round_number > 0:
                self.add_round()

            # add key[round_number]
            cipher_state = self.xor_round_key(round_number, key_state, cipher_state)

            # sbox
            cipher_state = [
                self.add_SBOX_component(
                    [cipher_state[i].id],
                    [[j for j in range(self.SBOX_BIT_SIZE)]],
                    self.SBOX_BIT_SIZE,
                    self.sboxes[i%4]
                )
                for i in range(8)
            ]

            if round_number != self.NROUNDS - 1:
                # mix columns + shift rows
                mx1 = self.add_linear_layer_component(
                    [cipher_state[0].id, cipher_state[5].id, cipher_state[2].id, cipher_state[7].id],
                    [[j for j in range(self.SBOX_BIT_SIZE)]] * 4,
                    self.SBOX_BIT_SIZE * 4,
                    self.mix_columns_matrix
                )
                mx2 = self.add_linear_layer_component(
                    [cipher_state[4].id, cipher_state[1].id, cipher_state[6].id, cipher_state[3].id],
                    [[j for j in range(self.SBOX_BIT_SIZE)]] * 4,
                    self.SBOX_BIT_SIZE * 4,
                    self.mix_columns_matrix
                )
                cipher_state = [
                    self.add_XOR_component(
                        [mx1.id, _zero.id],
                        [
                            [j for j in range(i * self.SBOX_BIT_SIZE, (i + 1) * self.SBOX_BIT_SIZE)],
                            [j for j in range(self.SBOX_BIT_SIZE)]
                        ],
                        self.SBOX_BIT_SIZE
                    )
                    for i in range(4)
                ]
                cipher_state += [
                    self.add_XOR_component(
                        [mx2.id, _zero.id],
                        [
                            [j for j in range(i * self.SBOX_BIT_SIZE, (i + 1) * self.SBOX_BIT_SIZE)],
                            [j for j in range(self.SBOX_BIT_SIZE)]
                        ],
                        self.SBOX_BIT_SIZE
                    )
                    for i in range(4)
                ]

                self.add_intermediate_output_component(
                    [mx1.id, mx2.id],
                    [[j for j in range(self.SBOX_BIT_SIZE * 4)]] * 2,
                    self.SBOX_BIT_SIZE * 8,
                    'round_output'
                )
            else:
                # shift rows
                cipher_state = [
                    cipher_state[0], cipher_state[5], cipher_state[2], cipher_state[7],
                    cipher_state[4], cipher_state[1], cipher_state[6], cipher_state[3]
                ]
                # add last key
                cipher_state = self.xor_round_key(self.NROUNDS, key_state, cipher_state)
                self.add_cipher_output_component(
                    [cipher_state[i].id for i in range(8)],
                    [[j for j in range(self.SBOX_BIT_SIZE)]] * 8,
                    self.SBOX_BIT_SIZE * 8
                )


    def xor_round_key(self, round_number, key_state, cipher_state):
        key = [key_state[(8*round_number + i) % 24] for i in range(8)]

        self.add_round_key_output_component(
            [key[i].id for i in range(8)],
            [[j for j in range(self.SBOX_BIT_SIZE)]] * 8,
            self.SBOX_BIT_SIZE * 8
        )

        if type(cipher_state) is str:
            # cipher_state == INPUT_PLAINTEXT
            return [
                self.add_XOR_component(
                    [cipher_state, key[i].id],
                    [
                        [j for j in range(i * self.SBOX_BIT_SIZE, (i + 1) * self.SBOX_BIT_SIZE)],
                        [j for j in range(self.SBOX_BIT_SIZE)]
                    ],
                    self.SBOX_BIT_SIZE
                )
                for i in range(8)
            ]
        else:
            return [
                self.add_XOR_component(
                    [cipher_state[i].id, key[i].id],
                    [[j for j in range(self.SBOX_BIT_SIZE)]] * 2,
                    self.SBOX_BIT_SIZE
                )
                for i in range(8)
            ]
