# ****************************************************************************
# Copyright 2023 Technology Innovation Institute
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
# ****************************************************************************


from claasp.cipher import Cipher
from claasp.DTOs.component_state import ComponentState
from claasp.name_mappings import INPUT_STATE

PARAMETERS_CONFIGURATION_LIST = [{'key_bit_size': 128, 'lfsr_state_bit_size': 128, 'fsm_bit_size': 4,
                                  'keystream_bit_len': 125}]

LFSR_DESCR = [
    [
        [25, [[0], [5], [13], [17]]],  # Register_1: len=25, feedback poly = s0+ s5 + s13 + s17
        [31, [[25], [32], [40], [44]]],  # Register_2: len=31, feedback poly = s25+ s32 + s40 + s44
        [33, [[56], [61], [65], [85]]],  # Register_3: len=33, feedback poly = s56+ s61 + s65 + s85
        [39, [[89], [92], [100], [124]]]  # Register_1: len=39, feedback poly = s89+ s92 + s100 + s124
    ],
    1  # Registers' word-size=1-bit
]


class BluetoothStreamCipherE0(Cipher):
    """
        Return a cipher object of bluetooth encryption/decryption keystream generator E0.

        For the detail, one can follow https://eprint.iacr.org/2022/016.pdf

        The Bluetooth E0 algorithm comprises two distinct stages. In the first stage, an
        initial 132-bit state is generated by utilizing four inputs: the 128-bit key, the
        48-bit Bluetooth address, and the 26-bit master counter. The second stage is
        dedicated to the generation of the keystream, which is produced using the 132-bit
        input state from the first stage. This class handles only the key generation part of
        the algorithm.

        INPUT:
        - ``keystream_bit_len`` -- **integer** (default: `125`);
        - ``key_bit_size (lfsr_state_size)`` --  128-bit;
        - ``finite_state_machine_bit_size`` --   4-bit;
        - ``input_bit_size`` --   132-bit (128-bit lfsr_input_state.append(4-bit fsm_input_state));


        EXAMPLES:

        sage: from claasp.ciphers.stream_ciphers.bluetooth_stream_cipher_e0 import BluetoothStreamCipherE0
        sage: e0 = BluetoothStreamCipherE0(keystream_bit_len=125)
        sage: fsm=0xb
        sage: key =0x25ac1ea08e1ec131e0a1780f7a2a42bb
        sage: input= int(hex(key<<4|fsm),16) #key.append(fsm)
        sage: keystream=0x8cd29cc32668b90ee2312924376f1b4
        sage: e0.evaluate([input])==keystream
        True

        sage: fsm=0xd
        sage: key =0xe22f92fff8c245c49d10359a02f1e555
        sage: input= int(hex(key<<4|fsm),16) #key.append(fsm)
        sage: keystream=0x1198636720bac54986d1ab5a494866c9
        sage: e0.evaluate([input])==keystream
        True


    """

    def __init__(self, key_bit_size=128, lfsr_state_bit_size=128, fsm_bit_size=4, keystream_bit_len=125):
        self.fsm_bit_size = fsm_bit_size
        self.lfsr_state_bit_size = lfsr_state_bit_size
        self.key_bit_size = key_bit_size

        super().__init__(family_name="bluetooth_E0_stream_cipher",
                         cipher_type="stream_cipher",
                         cipher_inputs=[INPUT_STATE],
                         cipher_inputs_bit_size=[key_bit_size + fsm_bit_size],
                         cipher_output_bit_size=keystream_bit_len)

        input_state = ComponentState([INPUT_STATE], [list(range(self.key_bit_size))])
        fsm_id = [input_state.id[0], input_state.id[0], input_state.id[0], input_state.id[0]]
        fsm_pos = [[_] for _ in range(self.key_bit_size, self.key_bit_size+self.fsm_bit_size)]
        lfsr_state = input_state.id[0]
        keystream = []

        for clock_number in range(self._get_len_of_keystream(keystream_bit_len)):
            self.add_round()
            keystream = self.e0_keystream(lfsr_state, fsm_id, fsm_pos, clock_number, keystream)
            fsm_id, fsm_pos = self.e0_nonlinear_function(lfsr_state, fsm_id, fsm_pos)
            lfsr_state = self.add_FSR_component([lfsr_state], [list(range(self.lfsr_state_bit_size))],
                                                self.lfsr_state_bit_size, LFSR_DESCR).id

        self.add_cipher_output_component([keystream], [list(range(self._get_len_of_keystream(keystream_bit_len)))],
                                         self._get_len_of_keystream(keystream_bit_len))

    def e0_nonlinear_function(self, lfsr_state, fsm_id, fsm_pos):
        x_id = y_id = z_id = u_id = lfsr_state
        x_pos = [1]
        y_pos = [32]
        z_pos = [57]
        u_pos = [96]

        y0 = self.add_XOR_component([x_id, y_id, z_id, u_id], [x_pos, y_pos, z_pos, u_pos], 1).id

        y1_0 = self.add_XOR_component([y_id, z_id, u_id], [y_pos, z_pos, u_pos], 1).id
        y1_0 = self.add_AND_component([x_id, y1_0], [x_pos, [0]], 1).id
        y1_1 = self.add_XOR_component([z_id, u_id], [z_pos, u_pos], 1).id
        y1_1 = self.add_AND_component([y_id, y1_1], [y_pos, [0]], 1).id
        y1_2 = self.add_AND_component([z_id, u_id], [z_pos, u_pos], 1).id
        y1 = self.add_XOR_component([y1_0, y1_1, y1_2], [[0], [0], [0]], 1).id

        y2 = self.add_AND_component([x_id, y_id, z_id, u_id], [x_pos, y_pos, z_pos, u_pos], 1).id

        t0_0 = self.add_AND_component([y0, fsm_id[2]], [[0], fsm_pos[2]], 1).id
        t0 = self.add_XOR_component([t0_0, y1, fsm_id[3], fsm_id[2], fsm_id[1], fsm_id[0]],
                                    [[0], [0], fsm_pos[3], fsm_pos[2], fsm_pos[1], fsm_pos[0]],
                                    1).id

        t1_0 = self.add_AND_component([y1, fsm_id[3]], [[0], fsm_pos[3]], 1).id
        t1_1 = self.add_AND_component([y0, fsm_id[2], fsm_id[3]], [[0], fsm_pos[2], fsm_pos[3]], 1).id
        t1_2 = self.add_AND_component([y1, y0, fsm_id[2]], [[0], [0], fsm_pos[2]], 1).id
        t1 = self.add_XOR_component([y2, t1_0, t1_1, t1_2, fsm_id[3], fsm_id[0]],
                                    [[0], [0], [0], [0], fsm_pos[3], fsm_pos[0]], 1).id

        fsm_id[0] = fsm_id[2]
        fsm_id[1] = fsm_id[3]
        fsm_id[2] = t0
        fsm_id[3] = t1
        fsm_pos[0] = fsm_pos[2]
        fsm_pos[1] = fsm_pos[3]
        fsm_pos[2] = [0]
        fsm_pos[3] = [0]
        return fsm_id, fsm_pos

    def _get_len_of_keystream(self, keystream_bit_len):
        if keystream_bit_len is not None:
            return keystream_bit_len
        configuration_keystream_bit_len = None
        for parameters in PARAMETERS_CONFIGURATION_LIST:
            if parameters['fsm_bit_size'] == self.fsm_bit_size and parameters['key_bit_size'] == self.key_bit_size \
                    and parameters['lfsr_state_bit_size'] == self.lfsr_state_bit_size:
                configuration_keystream_bit_len = parameters['keystream_bit_len']
                break
        if configuration_keystream_bit_len is None:
            raise ValueError("No available len of keystream for the given parameters.")
        return configuration_keystream_bit_len

    def e0_keystream(self, lfsr_state, fsm_id, fsm_pos, clock_number, ks):
        ks_id = [lfsr_state, lfsr_state, lfsr_state, lfsr_state, fsm_id[2]]
        ks_pos = [[1], [32], [57], [96], fsm_pos[2]]
        z = self.add_XOR_component(ks_id, ks_pos, 1).id
        if clock_number == 0:
            ks = self.add_round_output_component([z], [list(range(1))], 1).id
        else:
            ks = self.add_round_output_component([ks, z], [list(range(clock_number)), [0]], clock_number + 1).id
        return ks
