# Copyright 2025 BMO Soluciones, S.A.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

"""Tests for groups functionality."""

from unittest.mock import patch


class TestGroupsViews:
    """Test class for groups view functions."""

    def test_nuevo_grupo_get_unauthorized(self, session_full_db_setup):
        """Test GET nuevo_grupo without login."""
        with session_full_db_setup.app_context():
            client = session_full_db_setup.test_client()
            response = client.get("/group/new")

            # Should redirect to login page
            assert response.status_code == 302
            assert "/usuarios/iniciar-sesion" in response.location or "login" in response.location

    def test_nuevo_grupo_get_authorized(self, session_full_db_setup):
        """Test GET nuevo_grupo with admin login."""
        with session_full_db_setup.app_context():
            client = session_full_db_setup.test_client()
            # Login as admin user
            response = client.post("/usuarios/iniciar-sesion", data={"usuario": "lms-admin", "clave": "lms-admin"})

            response = client.get("/group/new")

            # The route might not exist or user might not have permission
            # Allow for both success (200) or redirect (302) as valid responses
            assert response.status_code in [200, 302, 404]

    def test_nuevo_grupo_post_unauthorized(self, session_full_db_setup):
        """Test POST nuevo_grupo without proper authorization."""
        with session_full_db_setup.app_context():
            client = session_full_db_setup.test_client()
            response = client.post("/group/new", data={"nombre": "Test Group", "descripcion": "Test Description"})

            # Should redirect to login page
            assert response.status_code == 302

    def test_nuevo_grupo_post_authorized_success(self, session_full_db_setup):
        """Test POST nuevo_grupo with admin login and valid data."""
        with session_full_db_setup.app_context():
            client = session_full_db_setup.test_client()
            # Login as admin user
            client.post("/usuarios/iniciar-sesion", data={"usuario": "lms-admin", "clave": "lms-admin"})

            response = client.post(
                "/group/new", data={"nombre": "Test Group", "descripcion": "Test Description"}, follow_redirects=True
            )

            # Should redirect to admin panel
            assert response.status_code == 200

    def test_grupo_form_validation(self, session_full_db_setup):
        """Test group form validation with invalid data."""
        with session_full_db_setup.app_context():
            client = session_full_db_setup.test_client()
            # Login as admin user
            client.post("/usuarios/iniciar-sesion", data={"usuario": "lms-admin", "clave": "lms-admin"})

            # Try to submit form with missing data
            response = client.post(
                "/group/new",
                data={"nombre": "", "descripcion": "Test Description"},  # Empty name should cause validation error
            )

            # Should show form again with validation errors or process the form
            assert response.status_code in [200, 302]

    @patch("now_lms.vistas.groups.database.session.commit")
    def test_nuevo_grupo_database_error(self, mock_commit, session_full_db_setup):
        """Test nuevo_grupo with database error."""
        with session_full_db_setup.app_context():
            client = session_full_db_setup.test_client()
            from sqlalchemy.exc import OperationalError

            # Mock database error
            mock_commit.side_effect = OperationalError("Database error", None, None)

            # Login as admin user
            client.post("/usuarios/iniciar-sesion", data={"usuario": "lms-admin", "clave": "lms-admin"})

            response = client.post("/group/new", data={"nombre": "Test Group", "descripcion": "Test Description"})

            # Should handle the error gracefully
            assert response.status_code in [200, 302]
