# Copyright 2025 BMO Soluciones, S.A.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

"""Basic tests for mail functionality to improve coverage."""

import os
from types import SimpleNamespace
from unittest.mock import patch

from now_lms.mail import _load_mail_config_from_env


class TestMailFunctions:
    """Test basic mail configuration functions."""

    @patch.dict(
        os.environ,
        {
            "MAIL_SERVER": "smtp.example.com",
            "MAIL_PORT": "587",
            "MAIL_USERNAME": "test@example.com",
            "MAIL_PASSWORD": "password123",
            "MAIL_USE_TLS": "True",
            "MAIL_USE_SSL": "False",
            "MAIL_DEFAULT_SENDER": "noreply@example.com",
        },
    )
    def test_load_mail_config_from_env_complete(self):
        """Test loading complete mail configuration from environment variables."""
        config = _load_mail_config_from_env()

        assert config.mail_configured is True
        assert config.MAIL_SERVER == "smtp.example.com"
        assert config.MAIL_PORT == "587"
        assert config.MAIL_USERNAME == "test@example.com"
        assert config.MAIL_PASSWORD == "password123"
        # Note: The capitalize() + match pattern means "True"/"False" strings don't convert to boolean
        assert config.MAIL_USE_TLS == "True"
        assert config.MAIL_USE_SSL == "False"
        assert config.MAIL_DEFAULT_SENDER == "noreply@example.com"

    @patch.dict(
        os.environ,
        {
            "MAIL_SERVER": "smtp.example.com",
            "MAIL_PORT": "465",
            "MAIL_USERNAME": "test@example.com",
            "MAIL_PASSWORD": "password123",
            "MAIL_USE_TLS": "FALSE",  # Will become "False" after capitalize, doesn't match "FALSE" pattern
            "MAIL_USE_SSL": "TRUE",  # Will become "True" after capitalize, doesn't match "TRUE" pattern
        },
    )
    def test_load_mail_config_from_env_ssl_enabled(self):
        """Test loading mail configuration with SSL enabled."""
        config = _load_mail_config_from_env()

        assert config.mail_configured is True
        # Bug: The capitalize() + uppercase pattern matching means these remain strings
        assert config.MAIL_USE_TLS == "False"
        assert config.MAIL_USE_SSL == "True"

    @patch.dict(
        os.environ,
        {
            "MAIL_SERVER": "smtp.example.com",
            "MAIL_PORT": "587",
            "MAIL_USERNAME": "test@example.com",
            # Missing MAIL_PASSWORD
        },
    )
    def test_load_mail_config_from_env_incomplete(self):
        """Test loading incomplete mail configuration from environment variables."""
        config = _load_mail_config_from_env()

        assert config.mail_configured is False
        assert config.MAIL_SERVER == "smtp.example.com"
        assert config.MAIL_PORT == "587"
        assert config.MAIL_USERNAME == "test@example.com"
        assert config.MAIL_PASSWORD is None

    @patch.dict(os.environ, {}, clear=True)
    def test_load_mail_config_from_env_empty(self):
        """Test loading mail configuration with no environment variables."""
        config = _load_mail_config_from_env()

        assert config.mail_configured is False
        assert config.MAIL_SERVER is None
        assert config.MAIL_PORT is None
        assert config.MAIL_USERNAME is None
        assert config.MAIL_PASSWORD is None
        assert config.MAIL_USE_TLS == "False"  # Default "False" string, not boolean
        assert config.MAIL_USE_SSL == "False"  # Default "False" string, not boolean
        assert config.MAIL_DEFAULT_SENDER is None

    @patch.dict(
        os.environ,
        {
            "MAIL_SERVER": "smtp.example.com",
            "MAIL_PORT": "587",
            "MAIL_USERNAME": "test@example.com",
            "MAIL_PASSWORD": "password123",
            "MAIL_USE_TLS": "false",  # lowercase - will become "False" after capitalize
            "MAIL_USE_SSL": "true",  # lowercase - will become "True" after capitalize
        },
    )
    def test_load_mail_config_from_env_case_insensitive(self):
        """Test that boolean conversion handles different cases."""
        config = _load_mail_config_from_env()

        # The capitalize() converts to "False"/"True" but match only handles "FALSE"/"TRUE"
        assert config.MAIL_USE_TLS == "False"  # String, not boolean
        assert config.MAIL_USE_SSL == "True"  # String, not boolean

    def test_load_mail_config_from_env_unreachable_boolean_paths(self):
        """Test that demonstrates the unreachable boolean conversion paths."""
        # This test documents a bug in the original code:
        # The capitalize() method converts "FALSE" -> "False" and "TRUE" -> "True"
        # But the match statement checks for "FALSE" and "TRUE" (uppercase)
        # This means the boolean conversion is never reached

        import os
        from unittest.mock import patch

        # Try to reach the boolean conversion paths
        with patch.dict(os.environ, {"MAIL_USE_TLS": "FALSE", "MAIL_USE_SSL": "TRUE"}):
            config = _load_mail_config_from_env()
            # These should be boolean per the intended logic, but are strings due to the bug
            assert isinstance(config.MAIL_USE_TLS, str)
            assert isinstance(config.MAIL_USE_SSL, str)
            assert config.MAIL_USE_TLS == "False"
            assert config.MAIL_USE_SSL == "True"

    @patch.dict(
        os.environ,
        {
            "MAIL_SERVER": "smtp.example.com",
            "MAIL_PORT": "587",
            "MAIL_USERNAME": "test@example.com",
            "MAIL_PASSWORD": "password123",
            "MAIL_USE_TLS": "invalid_value",
            "MAIL_USE_SSL": "another_invalid",
        },
    )
    def test_load_mail_config_from_env_invalid_boolean_values(self):
        """Test handling of invalid boolean values."""
        config = _load_mail_config_from_env()

        # Invalid values should not match TRUE/FALSE and remain as strings
        # The match statement doesn't cover invalid cases, so they remain strings
        assert config.MAIL_USE_TLS == "Invalid_value"  # capitalized but not boolean
        assert config.MAIL_USE_SSL == "Another_invalid"  # capitalized but not boolean

    def test_mail_config_returns_simplenamespace(self):
        """Test that mail config function returns SimpleNamespace object."""
        config = _load_mail_config_from_env()

        assert isinstance(config, SimpleNamespace)
        assert hasattr(config, "mail_configured")
        assert hasattr(config, "MAIL_SERVER")
        assert hasattr(config, "MAIL_PORT")
        assert hasattr(config, "MAIL_USERNAME")
        assert hasattr(config, "MAIL_PASSWORD")
        assert hasattr(config, "MAIL_USE_TLS")
        assert hasattr(config, "MAIL_USE_SSL")
        assert hasattr(config, "MAIL_DEFAULT_SENDER")
